/*  crossing.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef CROSSING_H_
#define CROSSING_H_

#include <ginac/ginac.h>
#include <list>
#include <string>
#include "combinatorics.h"
#include "yamlconfigurable.h"

namespace YAML {
class Node;
class Emitter;
}

namespace Reduze {

class INT;
class Identity;
class IntegralFamily;
class Sector;
class LinearCombination;
class Kinematics;

/// permutation of external momenta
class Crossing: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("crossing");
		s.set_short_description("A permutation of external momenta.");
		s.set_long_description(""//
					"A crossing is defined by a permutation of integers"
					" that represents the permutation of external momenta,"
					" combined list of incoming and outgoing momenta in"
					" that order. The integers must be in the range [1, N],"
					" where N is the number of external momenta. The permuted"
					" external momenta must have the same (squared) mass."
					" Crossings of mixed incoming and outgoing momenta are allowed.");
		s.add_option("permutation", true, "permutation", ""//
					"Permutation of the external momenta in cycle notation.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	/// create identity Crossing for a kinematics
	Crossing(const Kinematics* kin);

	/// construct a Crossing for given permutation and kinematics
	Crossing(const Kinematics* kin, const Permutation& perm);

	virtual ~Crossing() {
	}

	/// returns the inverse crossing
	Crossing inverse() const;
	/// product crossing: c1*c2 is the result of first applying c2 and then c1
	static Crossing compose(const Crossing& c1, const Crossing& c2);

	// getters

	const Kinematics* kinematics() const {
		return kinematics_;
	}

	/// name of the crossing, used as suffix for crossed IntegralFamily names
	inline const std::string& name() const {
		return name_;
	}
	inline const Permutation& permutation() const {
		return permutation_;
	}
	/// rules to substitute external momenta
	inline const GiNaC::exmap& rules_momenta() const {
		return rules_momenta_;
	}
	/// rules to substitute kinematic invariants
	inline const GiNaC::exmap& rules_invariants() const {
		return rules_invariants_;
	}

	/// returns name of the integral family with this crossing applied
	std::string name_for_crossed_family(const IntegralFamily*) const;
	/// returns crossed family
	const IntegralFamily* transform(const IntegralFamily*) const;
	/// returns crossed sector
	Sector transform(const Sector&) const;
	/// returns crossed INT
	INT transform(const INT&) const;
	/// returns crossed linear combination (no reconstruction of symbol to replace by one)
	/** if transform_to_minimal_equiv is true then all crossings of the integrals
	 ** are further replaced by the minimal equivalent crossing */
	LinearCombination transform(const LinearCombination& lc,
			bool transform_to_minimal_equiv = false) const;
	/// returns crossed identity
	/** if transform_to_minimal_equiv is true then all crossings of the integrals
	 ** are further replaced by the minimal equivalent crossing */
	Identity
	transform(const Identity& id, bool transform_to_minimal_equiv = false) const;
	/// applies all transformation rules to the expression
	GiNaC::ex transform(const GiNaC::ex&) const;

	/// returns the source INT and the Crossing the INT was constructed from
	/** for an uncrossed INT, the INT itself and the identity is returned */
	static std::pair<INT, Crossing> uncross(const INT&);

	/// returns the source INT and the Crossing the INT was constructed from
	/** for an uncrossed INT, the INT itself and the identity is returned */
	static std::pair<const IntegralFamily*, Crossing> uncross(const IntegralFamily*);

	/// returns the integral where its crossing is replaced by the minimal equivalent crossing
	static INT transform_to_minimal_equivalent(const INT& i);

	/// whether both Crossings give same trafo of kinematic invariants
	bool is_equivalent(const Crossing& other) const;
	/// whether the crossing is the identity
	inline bool is_identity() const {
		return permutation_.is_identity();
	}
	/// whether the crossing leaves the kinematic invariants unchanged
	inline bool is_equivalent_to_identity() const {
		return is_equivalent_to_identity_;
	}
	/// returns sector with minimal equivalent crossed integral family
	static Sector to_minimal_crossing(const Sector&);
	/// returns integral with minimal equivalent crossed integral family
	static INT to_minimal_crossing(const INT&);


	// operators

	/// crossings  equivalent to the identity are preferred
	bool operator<(const Crossing& other) const;
	bool operator==(const Crossing& other) const;
	bool operator!=(const Crossing& other) const;

	virtual void print(YAML::Emitter&) const;
	virtual void read(const YAML::Node&);
	friend class YAMLProxy<Crossing> ;

private:
	/// a crossing is always bound to a kinematic
	Crossing() :
		kinematics_(0) {
	}

	/// the kinematic
	const Kinematics* kinematics_;
	/// permutation of external legs 1, ..., n
	Permutation permutation_;

	/// name of the crossing, determined from the cycle rep. of the permutation
	std::string name_;
	/// rules to substitute external momenta
	GiNaC::exmap rules_momenta_;
	/// rules to substitute kinematic invariants
	GiNaC::exmap rules_invariants_;

	/// whether the crossing leaves the kinematic invariants untchanged
	bool is_equivalent_to_identity_;
	/// whether the symbol to replace by one appears in lhs of kinematic rules
	bool has_symb2one_in_lhs_;
};

inline void operator>>(const YAML::Node& n, Crossing& c) {
	c.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const Crossing& c) {
	c.print(ye);
	return ye;
}

/// ordered crossings of a kinematics
class OrderedCrossings {
public:
	// construction of default empty crossings
	OrderedCrossings(const Kinematics* kin);
	virtual ~OrderedCrossings() {
	}

	/// construct the crossings from the leg permutations
	void construct_crossings();

	virtual void print(YAML::Emitter&) const;
	virtual void read(const YAML::Node&);

	/// returns all possible ordered crossing of external legs by their leg permutations (identity crossing NOT included)
	const std::list<Crossing>& ordered_crossings() const {
		return ordered_crossings_;
	}
	/// minimal equivalent crossings by crossing (identity included)
	const std::map<Crossing, Crossing>& equivalent_crossings() const {
		return equivalent_crossings_;
	}
	/// minimal crossing for each equivalence class (identity included)
	const std::set<Crossing>& equivalent_crossing_classes() const {
		return equivalent_crossing_classes_;
	}
	/// whether the crossing is included
	bool has(const Crossing& crossing) const;

	/// log some info
	void print_info() const;

private:
	const Kinematics* kinematics_;
	/// ordered crossings without the identity crossing
	std::list<Crossing> ordered_crossings_;
	/// minimal equivalent crossings by crossing (identity included)
	std::map<Crossing, Crossing> equivalent_crossings_;
	/// equivalence classes of crossings
	std::set<Crossing> equivalent_crossing_classes_;
	/// crossings by name
	std::map<std::string, Crossing> crossings_by_name_;

private:
	OrderedCrossings() :
			kinematics_(0) {
	}
	// setup remaining members from ordered_crossings_
	void init();
};

inline void operator>>(const YAML::Node& n, OrderedCrossings& c) {
	c.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const OrderedCrossings& c) {
	c.print(ye);
	return ye;
}

}

#endif /* CROSSING_H_ */
