// ------------------------------------
//
// translatePythia.cc
// Author: Lynn Garren
//
// translate an ID number to or from the standard numbering scheme and Pythia
// use static maps
//
//  The maps are initialized if and only if the public functions are called.
//  Because the maps are static, the initialization happens only once.
//
//  The user NEVER calls PythiaPDTMapInit()
//  We use a data table (struct SList) so that compile time is not impacted.
//
//  public functions:
//        int translatePythiatoPDT( const int id )
//        int translatePDTtoPythia( const int id )
//        PythiaPDTMap const & getPythiaPDTMap()
//        PDTPythiaMap const & getPDTPythiaMap()
//
// ------------------------------------

#include <map>
#include <utility>	// make_pair

#include "HepPID/Version.hh"
#include "HepPID/ParticleIDTranslations.hh"
#include "HepPID/ParticleIDMethods.hh"
#include "HepPID/ParticleName.hh"

namespace HepPID {

 typedef  std::map< int, int >  PythiaPDTMap;
 typedef  std::map< int, int >  PDTPythiaMap;

namespace {	// PythiaPDTMapInit is private

 PythiaPDTMap const & getPythiaPDTMap();
 PDTPythiaMap const & getPDTPythiaMap();

PythiaPDTMap const & PythiaPDTMapInit()
{

  static PythiaPDTMap  m;

  static const struct {
      int hid;	// Pythia
      int pid;	// PDT
  } SList[] = {
    {            1,            1 },
    {           -1,           -1 },
    {            2,            2 },
    {           -2,           -2 },
    {            3,            3 },
    {           -3,           -3 },
    {            4,            4 },
    {           -4,           -4 },
    {            5,            5 },
    {           -5,           -5 },
    {            6,            6 },
    {           -6,           -6 },
    {            7,            7 },
    {           -7,           -7 },
    {            8,            8 },
    {           -8,           -8 },
    {           11,           11 },
    {          -11,          -11 },
    {           12,           12 },
    {          -12,          -12 },
    {           13,           13 },
    {          -13,          -13 },
    {           14,           14 },
    {          -14,          -14 },
    {           15,           15 },
    {          -15,          -15 },
    {           16,           16 },
    {          -16,          -16 },
    {           17,           17 },
    {          -17,          -17 },
    {           18,           18 },
    {          -18,          -18 },
    {           21,           21 },
    {           22,           22 },
    {           23,           23 },
    {           24,           24 },
    {          -24,          -24 },
    {           25,           25 },
    {           32,           32 },
    {           33,           33 },
    {           34,           34 },
    {          -34,          -34 },
    {           35,           35 },
    {           36,           36 },
    {           37,           37 },
    {          -37,          -37 },
    {           39,           39 },
    {           41,           41 },
    {          -41,          -41 },
    {           42,           42 },
    {          -42,          -42 },
    {           81,           81 },
    {           82,           82 },
    {          -82,          -82 },
    {           83,           83 },
    {           84,           84 },
    {          -84,          -84 },
    {           85,           85 },
    {          -85,          -85 },
    {           88,           88 },
    {           90,           90 },
    {           91,           91 },
    {           92,           92 },
    {           93,           93 },
    {           94,           94 },
    {           95,           95 },
    {           96,           96 },
    {           97,           97 },
    {           98,           98 },
    {           99,           99 },
    {          110,          110 },
    {          990,          990 },
    {          111,          111 },
    {          113,          113 },
    {          115,          115 },
    {          130,          130 },
    {          211,          211 },
    {         -211,         -211 },
    {          213,          213 },
    {         -213,         -213 },
    {          215,          215 },
    {         -215,         -215 },
    {          221,          221 },
    {          223,          223 },
    {          225,          225 },
    {          310,          310 },
    {          311,          311 },
    {         -311,         -311 },
    {          313,          313 },
    {         -313,         -313 },
    {          315,          315 },
    {         -315,         -315 },
    {          321,          321 },
    {         -321,         -321 },
    {          323,          323 },
    {         -323,         -323 },
    {          325,          325 },
    {         -325,         -325 },
    {          331,          331 },
    {          333,          333 },
    {          335,          335 },
    {          411,          411 },
    {         -411,         -411 },
    {          413,          413 },
    {         -413,         -413 },
    {          415,          415 },
    {         -415,         -415 },
    {          421,          421 },
    {         -421,         -421 },
    {          423,          423 },
    {         -423,         -423 },
    {          425,          425 },
    {         -425,         -425 },
    {          431,          431 },
    {         -431,         -431 },
    {          433,          433 },
    {         -433,         -433 },
    {          435,          435 },
    {         -435,         -435 },
    {          441,          441 },
    {          443,          443 },
    {          445,          445 },
    {          511,          511 },
    {         -511,         -511 },
    {          513,          513 },
    {         -513,         -513 },
    {          515,          515 },
    {         -515,         -515 },
    {          521,          521 },
    {         -521,         -521 },
    {          523,          523 },
    {         -523,         -523 },
    {          525,          525 },
    {         -525,         -525 },
    {          531,          531 },
    {         -531,         -531 },
    {          533,          533 },
    {         -533,         -533 },
    {          535,          535 },
    {         -535,         -535 },
    {          541,          541 },
    {         -541,         -541 },
    {          543,          543 },
    {         -543,         -543 },
    {          545,          545 },
    {         -545,         -545 },
    {          551,          551 },
    {          553,          553 },
    {          555,          555 },
    {         1103,         1103 },
    {        -1103,        -1103 },
    {         1114,         1114 },
    {        -1114,        -1114 },
    {         2101,         2101 },
    {        -2101,        -2101 },
    {         2103,         2103 },
    {        -2103,        -2103 },
    {         2112,         2112 },
    {        -2112,        -2112 },
    {         2114,         2114 },
    {        -2114,        -2114 },
    {         2203,         2203 },
    {        -2203,        -2203 },
    {         2212,         2212 },
    {        -2212,        -2212 },
    {         2214,         2214 },
    {        -2214,        -2214 },
    {         2224,         2224 },
    {        -2224,        -2224 },
    {         3101,         3101 },
    {        -3101,        -3101 },
    {         3103,         3103 },
    {        -3103,        -3103 },
    {         3112,         3112 },
    {        -3112,        -3112 },
    {         3114,         3114 },
    {        -3114,        -3114 },
    {         3122,         3122 },
    {        -3122,        -3122 },
    {         3201,         3201 },
    {        -3201,        -3201 },
    {         3203,         3203 },
    {        -3203,        -3203 },
    {         3212,         3212 },
    {        -3212,        -3212 },
    {         3214,         3214 },
    {        -3214,        -3214 },
    {         3222,         3222 },
    {        -3222,        -3222 },
    {         3224,         3224 },
    {        -3224,        -3224 },
    {         3303,         3303 },
    {        -3303,        -3303 },
    {         3312,         3312 },
    {        -3312,        -3312 },
    {         3314,         3314 },
    {        -3314,        -3314 },
    {         3322,         3322 },
    {        -3322,        -3322 },
    {         3324,         3324 },
    {        -3324,        -3324 },
    {         3334,         3334 },
    {        -3334,        -3334 },
    {         4101,         4101 },
    {        -4101,        -4101 },
    {         4103,         4103 },
    {        -4103,        -4103 },
    {         4112,         4112 },
    {        -4112,        -4112 },
    {         4114,         4114 },
    {        -4114,        -4114 },
    {         4122,         4122 },
    {        -4122,        -4122 },
    {         4132,         4132 },
    {        -4132,        -4132 },
    {         4201,         4201 },
    {        -4201,        -4201 },
    {         4203,         4203 },
    {        -4203,        -4203 },
    {         4212,         4212 },
    {        -4212,        -4212 },
    {         4214,         4214 },
    {        -4214,        -4214 },
    {         4222,         4222 },
    {        -4222,        -4222 },
    {         4224,         4224 },
    {        -4224,        -4224 },
    {         4232,         4232 },
    {        -4232,        -4232 },
    {         4301,         4301 },
    {        -4301,        -4301 },
    {         4303,         4303 },
    {        -4303,        -4303 },
    {         4312,         4312 },
    {        -4312,        -4312 },
    {         4314,         4314 },
    {        -4314,        -4314 },
    {         4322,         4322 },
    {        -4322,        -4322 },
    {         4324,         4324 },
    {        -4324,        -4324 },
    {         4332,         4332 },
    {        -4332,        -4332 },
    {         4334,         4334 },
    {        -4334,        -4334 },
    {         4403,         4403 },
    {        -4403,        -4403 },
    {         4412,         4412 },
    {        -4412,        -4412 },
    {         4414,         4414 },
    {        -4414,        -4414 },
    {         4422,         4422 },
    {        -4422,        -4422 },
    {         4424,         4424 },
    {        -4424,        -4424 },
    {         4432,         4432 },
    {        -4432,        -4432 },
    {         4434,         4434 },
    {        -4434,        -4434 },
    {         4444,         4444 },
    {        -4444,        -4444 },
    {         5101,         5101 },
    {        -5101,        -5101 },
    {         5103,         5103 },
    {        -5103,        -5103 },
    {         5112,         5112 },
    {        -5112,        -5112 },
    {         5114,         5114 },
    {        -5114,        -5114 },
    {         5122,         5122 },
    {        -5122,        -5122 },
    {         5132,         5132 },
    {        -5132,        -5132 },
    {         5142,         5142 },
    {        -5142,        -5142 },
    {         5201,         5201 },
    {        -5201,        -5201 },
    {         5203,         5203 },
    {        -5203,        -5203 },
    {         5212,         5212 },
    {        -5212,        -5212 },
    {         5214,         5214 },
    {        -5214,        -5214 },
    {         5222,         5222 },
    {        -5222,        -5222 },
    {         5224,         5224 },
    {        -5224,        -5224 },
    {         5232,         5232 },
    {        -5232,        -5232 },
    {         5242,         5242 },
    {        -5242,        -5242 },
    {         5301,         5301 },
    {        -5301,        -5301 },
    {         5303,         5303 },
    {        -5303,        -5303 },
    {         5312,         5312 },
    {        -5312,        -5312 },
    {         5314,         5314 },
    {        -5314,        -5314 },
    {         5322,         5322 },
    {        -5322,        -5322 },
    {         5324,         5324 },
    {        -5324,        -5324 },
    {         5332,         5332 },
    {        -5332,        -5332 },
    {         5334,         5334 },
    {        -5334,        -5334 },
    {         5342,         5342 },
    {        -5342,        -5342 },
    {         5401,         5401 },
    {        -5401,        -5401 },
    {         5403,         5403 },
    {        -5403,        -5403 },
    {         5412,         5412 },
    {        -5412,        -5412 },
    {         5414,         5414 },
    {        -5414,        -5414 },
    {         5422,         5422 },
    {        -5422,        -5422 },
    {         5424,         5424 },
    {        -5424,        -5424 },
    {         5432,         5432 },
    {        -5432,        -5432 },
    {         5434,         5434 },
    {        -5434,        -5434 },
    {         5442,         5442 },
    {        -5442,        -5442 },
    {         5444,         5444 },
    {        -5444,        -5444 },
    {         5503,         5503 },
    {        -5503,        -5503 },
    {         5512,         5512 },
    {        -5512,        -5512 },
    {         5514,         5514 },
    {        -5514,        -5514 },
    {         5522,         5522 },
    {        -5522,        -5522 },
    {         5524,         5524 },
    {        -5524,        -5524 },
    {         5532,         5532 },
    {        -5532,        -5532 },
    {         5534,         5534 },
    {        -5534,        -5534 },
    {         5542,         5542 },
    {        -5542,        -5542 },
    {         5544,         5544 },
    {        -5544,        -5544 },
    {         5554,         5554 },
    {        -5554,        -5554 },
    {        10111,      9000111 },
    {        10113,        10113 },
    {        10211,      9000211 },
    {       -10211,     -9000211 },
    {        10213,        10213 },
    {       -10213,       -10213 },
    {        10221,      9010221 },
    {        10223,        10223 },
    {        10311,        10311 },
    {       -10311,       -10311 },
    {        10313,        10313 },
    {       -10313,       -10313 },
    {        10321,        10321 },
    {       -10321,       -10321 },
    {        10323,        10323 },
    {       -10323,       -10323 },
    {        10331,        10221 },
    {        10333,        10333 },
    {        10411,        10411 },
    {       -10411,       -10411 },
    {        10413,        10413 },
    {       -10413,       -10413 },
    {        10421,        10421 },
    {       -10421,       -10421 },
    {        10423,        10423 },
    {       -10423,       -10423 },
    {        10431,        10431 },
    {       -10431,       -10431 },
    {        10433,        10433 },
    {       -10433,       -10433 },
    {        10441,        10441 },
    {        10443,        10443 },
    {        10511,        10511 },
    {       -10511,       -10511 },
    {        10513,        10513 },
    {       -10513,       -10513 },
    {        10521,        10521 },
    {       -10521,       -10521 },
    {        10523,        10523 },
    {       -10523,       -10523 },
    {        10531,        10531 },
    {       -10531,       -10531 },
    {        10533,        10533 },
    {       -10533,       -10533 },
    {        10541,        10541 },
    {       -10541,       -10541 },
    {        10543,        10543 },
    {       -10543,       -10543 },
    {        10551,        10551 },
    {        10553,        10553 },
    {        20113,        20113 },
    {        20213,        20213 },
    {       -20213,       -20213 },
    {        20223,        20223 },
    {        20313,        20313 },
    {       -20313,       -20313 },
    {        20323,        20323 },
    {       -20323,       -20323 },
    {        20333,        20333 },
    {        20413,        20413 },
    {       -20413,       -20413 },
    {        20423,        20423 },
    {       -20423,       -20423 },
    {        20433,        20433 },
    {       -20433,       -20433 },
    {        20443,        20443 },
    {        20513,        20513 },
    {       -20513,       -20513 },
    {        20523,        20523 },
    {       -20523,       -20523 },
    {        20533,        20533 },
    {       -20533,       -20533 },
    {        20543,        20543 },
    {       -20543,       -20543 },
    {        20553,        20553 },
    {       100443,       100443 },
    {       100553,       100553 },
    {      1000001,      1000001 },
    {     -1000001,     -1000001 },
    {      1000002,      1000002 },
    {     -1000002,     -1000002 },
    {      1000003,      1000003 },
    {     -1000003,     -1000003 },
    {      1000004,      1000004 },
    {     -1000004,     -1000004 },
    {      1000005,      1000005 },
    {     -1000005,     -1000005 },
    {      1000006,      1000006 },
    {     -1000006,     -1000006 },
    {      1000011,      1000011 },
    {     -1000011,     -1000011 },
    {      1000012,      1000012 },
    {     -1000012,     -1000012 },
    {      1000013,      1000013 },
    {     -1000013,     -1000013 },
    {      1000014,      1000014 },
    {     -1000014,     -1000014 },
    {      1000015,      1000015 },
    {     -1000015,     -1000015 },
    {      1000016,      1000016 },
    {     -1000016,     -1000016 },
    {      1000021,      1000021 },
    {      1000022,      1000022 },
    {      1000023,      1000023 },
    {      1000024,      1000024 },
    {     -1000024,     -1000024 },
    {      1000025,      1000025 },
    {      1000035,      1000035 },
    {      1000037,      1000037 },
    {     -1000037,     -1000037 },
    {      1000039,      1000039 },
    {      2000001,      2000001 },
    {     -2000001,     -2000001 },
    {      2000002,      2000002 },
    {     -2000002,     -2000002 },
    {      2000003,      2000003 },
    {     -2000003,     -2000003 },
    {      2000004,      2000004 },
    {     -2000004,     -2000004 },
    {      2000005,      2000005 },
    {     -2000005,     -2000005 },
    {      2000006,      2000006 },
    {     -2000006,     -2000006 },
    {      2000011,      2000011 },
    {     -2000011,     -2000011 },
    {      2000012,      2000012 },
    {     -2000012,     -2000012 },
    {      2000013,      2000013 },
    {     -2000013,     -2000013 },
    {      2000014,      2000014 },
    {     -2000014,     -2000014 },
    {      2000015,      2000015 },
    {     -2000015,     -2000015 },
    {      2000016,      2000016 },
    {     -2000016,     -2000016 },
    {      3000111,      3000111 },
    {      3000211,      3000211 },
    {     -3000211,     -3000211 },
    {      3000115,      3000115 },
    {      3000215,      3000215 },
    {     -3000215,     -3000215 },
    {      3000221,      3000221 },
    {      3000331,      3100221 },
    {      3000113,      3000113 },
    {      3000213,      3000213 },
    {     -3000213,     -3000213 },
    {      3000223,      3000223 },
    {      3100021,      3100021 },
    {      3100111,      3060111 },
    {      3200111,      3160111 },
    {      3100113,      3130113 },
    {      3200113,      3140113 },
    {      3300113,      3150113 },
    {      3400113,      3160113 },
    {      4000001,      4000001 },
    {     -4000001,     -4000001 },
    {      4000002,      4000002 },
    {     -4000002,     -4000002 },
    {      4000011,      4000011 },
    {     -4000011,     -4000011 },
    {      4000012,      4000012 },
    {     -4000012,     -4000012 },
    {      5000039,      4000039 },
    {      5100001,      5100001 },    
    {      5100002,      5100002 },    
    {      5100003,      5100003 },    
    {      5100004,      5100004 },    
    {      5100005,      5100005 },    
    {      5100006,      5100006 },    
    {      5100011,      5100011 },    
    {      5100012,      5100012 },    
    {      5100013,      5100013 },    
    {      5100014,      5100014 },    
    {      5100015,      5100015 },    
    {      5100016,      5100016 },    
    {      5100021,      5100021 },    
    {      5100022,      5100022 },    
    {      5100023,      5100023 },    
    {      5100024,      5100024 },    
    {      6100001,      6100001 },    
    {      6100002,      6100002 },    
    {      6100003,      6100003 },    
    {      6100004,      6100004 },    
    {      6100005,      6100005 },    
    {      6100006,      6100006 },    
    {      6100011,      6100011 },    
    {      6100012,      6100012 },    
    {      6100013,      6100013 },    
    {      6100014,      6100014 },    
    {      6100015,      6100015 },    
    {      6100016,      6100016 },    
    {      9900012,      9900012 },
    {      9900014,      9900014 },
    {      9900016,      9900016 },
    {      9900023,      9900023 },
    {      9900024,      9900024 },
    {     -9900024,     -9900024 },
    {      9900041,      9900041 },
    {     -9900041,     -9900041 },
    {      9900042,      9900042 },
    {     -9900042,     -9900042 },
    {      9900110,      9910113 },
    {      9900210,      9910211 },
    {     -9900210,     -9910211 },
    {      9900220,      9910223 },
    {      9900330,      9910333 },
    {      9900440,      9910443 },
    {      9902110,      9912112 },
    {     -9902110,     -9912112 },
    {      9902210,      9912212 },
    {     -9902210,     -9912212 },
    {      9900441,      9900441 },     
    {      9910441,      9910441 },     
    {      9900443,      9900443 },     
    {      9900551,      9900551 },     
    {      9910551,      9910551 },     
    {      9900553,      9900553 },    
    {   1000010020,   1000010020 },
    {  -1000010020,  -1000010020 },
    {   1000010030,   1000010030 },
    {  -1000010030,  -1000010030 },
    {   1000020030,   1000020030 },
    {  -1000020030,  -1000020030 },
    {   1000020040,   1000020040 },
    {  -1000020040,  -1000020040 }
  };

  int listSize = sizeof(SList)/sizeof(SList[0]);
  for( int k=0; k!=listSize; ++k) {
      m.insert( std::make_pair( SList[k].hid, SList[k].pid) );
  }
  return m;
}  // PythiaPDTMapInit()

PDTPythiaMap const & PDTPythiaMapInit()
{
    static PDTPythiaMap m;
    static PythiaPDTMap const & hmap = getPythiaPDTMap();
    
    for(PythiaPDTMap::const_iterator cit=hmap.begin(), mend=hmap.end(); cit!=mend; ++cit ) {
	m.insert( std::make_pair( cit->second, cit->first ));
    }
    return m;
}
  
//
// getPythiaPDTMap is the ONLY function allowed to call PythiaPDTMapInit
//
PythiaPDTMap const & getPythiaPDTMap()
{
  static PythiaPDTMap const & hmap = PythiaPDTMapInit();
  return hmap;
}  // getPythiaPDTMap()

//
// getPDTPythiaMap is the ONLY function allowed to call PythiaPDTMapInit
//
PDTPythiaMap const & getPDTPythiaMap()
{
  static PDTPythiaMap const & hmap = PDTPythiaMapInit();
  return hmap;
}  // getPDTPythiaMap()

} // unnamed namespace
  
int translatePythiatoPDT( const int id )
{
    static PythiaPDTMap const & hmap = getPythiaPDTMap();

    PythiaPDTMap::const_iterator const cit = hmap.find( id );
    // found it in the map
    if ( cit != hmap.end() ) { return cit->second; }
    // check to see if someone has defined a valid particle type
    // that isn't in the map
    if( isValid(id) ) { return id; }
    return 0;
}

int translatePDTtoPythia( const int id )
{
    static PDTPythiaMap const & pmap = getPDTPythiaMap();

    PDTPythiaMap::const_iterator const cit = pmap.find( id );
    // found it in the map
    if ( cit != pmap.end() ) { return cit->second; }
    // check to see if someone has defined a valid particle type
    // that isn't in the map
    if( isValid(id) ) { return id; }
    return 0;
}

void writePythiaTranslationLine( int i, std::ostream & os  )
{
    // only write map entries
    static PythiaPDTMap const & hmap = getPythiaPDTMap();

    PythiaPDTMap::const_iterator const cit = hmap.find( i );
    // found it in the map
    if ( cit != hmap.end() ) { 
        int id = cit->second;
	os << " Pythia number: " ;
	os.width(10);
	os << i << "  HepPID number: " ;
	os.width(10);
	os << id << "  " << particleName(id) << std::endl;
	// check reverse translation
	int iback =  translatePDTtoPythia(id);
	if(iback != i) {
	    os << " WARNING: " << id << " translates back to " 
	       << iback << " not to " << i << std::endl;
	}
    }
    return;
}  // writePythiaTranslationLine()

void  writePythiaTranslation( std::ostream & os )
{
    writeVersion( os );
    os << "     HepPID Particle List" << std::endl;
    os << std::endl;

    int id, i, j, q1, q2, q3, l, m, n;
    // special cases
    for( id=1; id<101; ++id) {
        writePythiaTranslationLine(  id, os );
        writePythiaTranslationLine( -id, os );
    }
    for( i=11; i<1000; ++i) {
        id = i*10;
        writePythiaTranslationLine(  id, os );
        writePythiaTranslationLine( -id, os );
    }
    // SUSY
    for( n=1; n<3; ++n) {
        for( i=1; i<40; ++i) {
	     id = 1000000*n+i;
             writePythiaTranslationLine(  id, os );
             writePythiaTranslationLine( -id, os );
	}
    }
    // technicolor
    for( q2=1; q2<4; ++q2) {
	for( q1=1; q1<4; ++q1) {
	    for( j=1; j<6; ++j) {
		for( m=0; m<5; ++m) {
		    id = 3000000+100000*m+100*q2+10*q1+j;
        	    writePythiaTranslationLine(  id, os );
        	    writePythiaTranslationLine( -id, os );
		}
	    }
	}
    }
    // excited particles
    for( n=4; n<6; ++n) {
        for( i=1; i<40; ++i) {
	    id = 1000000*n+i;
            writePythiaTranslationLine(  id, os );
            writePythiaTranslationLine( -id, os );
	}
    }
    // miscellaneous generator particles
    for( l=0; l<9; ++l) {
        for( q3=0; q3<10; ++q3) {
            for( q2=0; q2<10; ++q2) {
		for( q1=0; q1<10; ++q1) {
	            for( j=0; j<10; ++j) {
			id = 9900000+10000*l+1000*q3+100*q2+10*q1+j;
        		writePythiaTranslationLine(  id, os );
        		writePythiaTranslationLine( -id, os );
		    }
		}
	    }
	}
    }
    // diquark
    for( i=11; i<100; ++i) {
        for( j=0; j<10; ++j) {
            id = 100*i+j;
            writePythiaTranslationLine(  id, os );
            writePythiaTranslationLine( -id, os );
	}
    }
    // mesons
    for( q2=1; q2<10; ++q2) {
	for( q1=1; q1<10; ++q1) {
	    for( j=1; j<10; ++j) {
		for( m=0; m<9; ++m) {
		    for( l=0; l<10; ++l) {
			id = 100000*m+10000*l+100*q2+10*q1+j;
        		writePythiaTranslationLine(  id, os );
        		writePythiaTranslationLine( -id, os );
			id = 9000000+100000*m+10000*l+100*q2+10*q1+j;
        		writePythiaTranslationLine(  id, os );
        		writePythiaTranslationLine( -id, os );
		    }
		}
	    }
	}
    }
    // baryons
    for( q3=1; q3<10; ++q3) {
	for( q2=1; q2<10; ++q2) {
	    for( q1=1; q1<10; ++q1) {
		for( j=1; j<10; ++j) {
		    id = 1000*q3+100*q2+10*q1+j;
        	    writePythiaTranslationLine(  id, os );
        	    writePythiaTranslationLine( -id, os );
		}
	    }
	}
    }
    // pentaquarks
    for( l=1; l<9; ++l ) {
        for ( m=1; m<9; ++m ) {
	    for( q3=1; q3<9; ++q3) {
		for( q2=1; q2<9; ++q2) {
		    for( q1=1; q1<9; ++q1) {
			id = 9*1000000+l*100000+m*10000+1000*q3+100*q2+10*q1+2;
        		writePythiaTranslationLine(  id, os );
        		writePythiaTranslationLine( -id, os );
		    }
		}
	    }
        }
    }
    return;
}  // writePythiaTranslation()

}	// HepPID
