/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 SHA-1 + Twofish";
static const u64   KERN_TYPE      = 14613; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$sha1$twofish$xts-plain64$256$151351$13141718422ca869ae072d33b700c8c43a1796294bb74240d7aa60f03f322d1a$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$3fe4ef25ca5034797a45f8f69aa8f11848e24d0b6df0c8ce1938f151f432fa60860fdd7c23515fa05cdfbb1ad3da37d979f3d0f619d4725ea3c141915e49f429050b28ef8ecae19a656f198a787c4c25ac0e891d9c2e4941f3352229dbc0cd834786a7af7e8798b8053af17d043fd3bc0e986bad4ed48d3244e4de5b3ffb3b3198808212505fd1698a4a5be229c67ebb92e3e28df046f140772204cbe9bbf4f9c511c9cda80d303884559be0d348ef1e70a28ee5845e524c13bae3fc317ebc6c62b3e0c2738885b51850264a7a5e4d3a4141acd6050977af26ac5a57db5d837d5a798a8d06ef5282c729589b3c60e6c062758e6920f64ef77e3305e49e625c5e224e905447903bb197e212e49ba5069bae90639d4c418c24f27e7fb4ed4b060f2183801bd94eacec96b31858c21d481fcfa1a5970361d2f2ef73aa6ea566d44a83a42cfdb1014da9695c381902ba7dac73343b2b5dd6d852b93b6043b020105c99ba6a7852bb74288b7d90532ee11924662fc14a8996336fb4754f5ec3c0d77632ebcf43a14cda2a6c24899e850647e1372ab083f71e0e5086334fa700b99503a10f5ad47e5717ad6492863400170e017b0726fa0f9f1ac298121be057e98b520daa7b3629164d99ff69ad0f5db736c4e435d0e66a8c1991d7d45457aecb7fb6ece5900720dfdafa9bb16420517d91ea74685997da6baad439cc058f6ac098f2";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$sha1$twofish$";

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

u32 module_pw_max (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  // this overrides the reductions of PW_MAX in case optimized kernel is selected
  // IOW, even in optimized kernel mode it support length 256

  const u32 pw_max = PW_MAX;

  return pw_max;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 21;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len_min[2] = 3;
  token.len_max[2] = 3;
  token.attr[2]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len_min[4] = LUKS_SALT_HEX_LEN;
  token.len_max[4] = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len_min[6] = LUKS_CT_HEX_LEN;
  token.len_max[6] = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_SHA1;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_TWOFISH;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];

  if (strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  luks_t *luks = (luks_t *) esalt_buf;

  // cipher mode

  char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char salt_buf[LUKS_SALT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char af_buf[LUKS_AF_MAX_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char ct_buf[LUKS_CT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = module_pw_max;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = MODULE_DEFAULT;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
