// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "Background_image.h"




// *******************
// *** CONSTRUCTOR ***
// *******************

Mafferyew::Background_image::Background_image(const std::string& image_file)
{
    SDL_Surface* image = IMG_Load( image_file.c_str() );
    
    const int screen_width = image->w;
    const int screen_height = image->h;
    
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
    const Uint32 rmask = 0xff0000;
    const Uint32 gmask = 0x00ff00;
    const Uint32 bmask = 0x0000ff;
#else
    const Uint32 rmask = 0x0000ff;
    const Uint32 gmask = 0x00ff00;
    const Uint32 bmask = 0xff0000;
#endif
    
    GLint max_tex_size;
    glGetIntegerv( GL_MAX_TEXTURE_SIZE, &max_tex_size );
    //std::cout << "max_tex_size=" << max_tex_size << std::endl;
    
    int y_marker = 0;
    do
    {
        int x_marker = 0;
        do
        {
            SDL_Rect r = { 0 };
            r.x = x_marker;
            r.y = y_marker;
            
            if ( x_marker + max_tex_size > screen_width )
            {
                r.w = screen_width - x_marker;
                r.w = Useful_gl::power_of_two(r.w);
            }
            else
            {
                r.w = max_tex_size;
            } // if ... else
            
            if ( y_marker + max_tex_size > screen_height )
            {
                r.h = screen_height - y_marker;
                r.h = Useful_gl::power_of_two(r.h);
            }
            else
            {
                r.h = max_tex_size;
            } // if ... else
            
            SDL_Surface* conv = SDL_CreateRGBSurface(
                    SDL_SWSURFACE,
                    r.w, r.h,
                    image->format->BitsPerPixel,   // should be 24 (no alpha)
                    rmask, gmask, bmask, 0);    // alpha mask unused
            SDL_BlitSurface(image, &r, conv, 0);
            
            GLuint tex;
            glGenTextures(1, &tex);
            glBindTexture(GL_TEXTURE_2D, tex);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
            glTexImage2D(GL_TEXTURE_2D, 
                    0, // level - 0 unless multiple resolutions 
                    conv->format->BytesPerPixel, 
                    conv->w, conv->h, 
                    0, // border 
                    GL_RGB, 
                    GL_UNSIGNED_BYTE, 
                    conv->pixels);
            
            textures.push_back(tex);
            rects.push_back(r);
            
            SDL_FreeSurface(conv);
            
            x_marker += max_tex_size;
            
        } while (x_marker < screen_width);
        
        y_marker += max_tex_size;
        
    } while (y_marker < screen_height);

    SDL_FreeSurface(image);
    
} // Mafferyew::Background_image::Background_image


// ******************
// *** DESTRUCTOR ***
// ******************

Mafferyew::Background_image::~Background_image()
{
    std::vector<GLuint>::iterator iter = textures.begin();
    while ( iter != textures.end() )
    {
        glDeleteTextures( 1, &(*iter) );
        ++iter;
    } // while
    
} // Mafferyew::Background_image::~Background_image




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************

void Mafferyew::Background_image::draw() const
{
    for (unsigned int i = 0; i < textures.size(); ++i)
    {
        glBindTexture(GL_TEXTURE_2D, textures[i]);
        glBegin(GL_QUADS);
            glTexCoord2i(0, 0);
            glVertex3f(rects[i].x, rects[i].y, 0.0);
        
            glTexCoord2i(1, 0);
            glVertex3f(rects[i].x + rects[i].w, rects[i].y, 0.0);
        
            glTexCoord2i(1, 1);
            glVertex3f(rects[i].x + rects[i].w, rects[i].y + rects[i].h, 0.0);
        
            glTexCoord2i(0, 1);
            glVertex3f(rects[i].x, rects[i].y + rects[i].h, 0.0);
        glEnd();
    } // for
    
} // Mafferyew::Background_image::draw

// **************************************************
