/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

// OptUrTree.cpp: implementation of the OptUrTree class.
//
//////////////////////////////////////////////////////////////////////

#include <math.h>
#include <iostream>
#include <time.h>
#include <stdio.h>

#include "opturtree.h"
#include "brent.h"
#include "brarr.h"
#include "ali.h"
#include "inndarr.h"
#include "model.h"
#include "rate.h"
#include "ptnls.h"
#include "outstream.h"
#include "urtree.h"

#ifdef PARALLEL
#include <mpi.h>
#endif

extern int nto[NUM_CHAR];
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

/*
int OptUrTree::curNInNd_;
int OptUrTree::curNExNd_;
int OptUrTree::curNBr_;
int OptUrTree::maxNInNd_;
int OptUrTree::maxNExNd_;
int OptUrTree::maxNBr_;
 
int OptUrTree::outGrpNdNo_;
int OptUrTree::outGrpBrNo_;
int OptUrTree::artInRootNo_;
 
int OptUrTree::actBrNo_;
int OptUrTree::isBrLenEsted_;
int OptUrTree::nCmpNegLogLiBr_ = 0;
int OptUrTree::nCmpLi_ = 0;
int OptUrTree::nCmpLogLi_ = 0;
 
int OptUrTree::nCmpExLogLi_ = 0;
int OptUrTree::nCmpInLogLi_ = 0;
double OptUrTree::bestLogLi_;
Vec<int> OptUrTree::inBrNoLs_;
 
Mat<double> *OptUrTree::genDisMat_;
 
double OptUrTree::oriLenArr_[MAX_NUM_SEQ * 2];
double OptUrTree::canLenArr_[MAX_NUM_SEQ * 2];
double OptUrTree::optedLenArr_[MAX_NUM_SEQ * 2];
 
ExNdArr OptUrTree::exNdArr_;
InNdArr OptUrTree::inNdArr_;
LiBrArr OptUrTree::liBrArr_;
*/
OptUrTree opt_urtree;

ofstream nni_file;
ofstream nni_tree;
int nni_lh_iter = 0;
double global_bestlogli = -INFINITY;

//--------------------------------------------------------------------
//the constructor function
OptUrTree::OptUrTree() {                
	nCmpNegLogLiBr_ = 0;
	nCmpLi_ = 0;
	nCmpLogLi_ = 0;

	nCmpExLogLi_ = 0;
	nCmpInLogLi_ = 0;
	do_parallel_phase = 0;
}

#ifdef PARALLEL
void OptUrTree::turnOnParallelPhase(bool par_logli) {
	if (par_logli)
		do_parallel_phase = DO_PARALLEL_LOGLI_FUNC;
	else
		do_parallel_phase = DO_PARALLEL_OTHER_FUNC;
}

void OptUrTree::turnOffParallelPhase() {
	do_parallel_phase = 0;
}

bool OptUrTree::isParallelPhase(bool par_logli) {
	if (mpi_size <= 1 || do_parallel_phase == 0)
		return false;
	if (par_logli)
		return do_parallel_phase == DO_PARALLEL_LOGLI_FUNC;
	else
		return do_parallel_phase == DO_PARALLEL_OTHER_FUNC;
}
#endif // PARALLEL
 
//--------------------------------------------------------------------
//clean all data of this class
void OptUrTree::clean () {
	isBrLenEsted_ = 0;
	artInRootNo_ = -1;
	outGrpBrNo_ = -1;
}

//--------------------------------------------------------------------
//this function is used for constructing this class
void OptUrTree::doConstructor () {
	curNInNd_ = 0;
	curNExNd_ = 0;
	curNBr_ = 0;
	maxNInNd_ = 0;
	maxNExNd_ = 0;
	maxNBr_ = 0;

	outGrpNdNo_ = 0;
	outGrpBrNo_ = -1;
	actBrNo_ = -1;
	clean ();
	oriLenArr_ = NULL;
	canLenArr_ = NULL;
	optedLenArr_ = NULL;	
}

void OptUrTree::allocMemory() {
	int nseq2 = alignment.getNSeq() * 2;
	if (!oriLenArr_) {
		oriLenArr_ = new double[nseq2];
		canLenArr_ = new double[nseq2];
		optedLenArr_ = new double[nseq2];	
	}
}


//--------------------------------------------------------------------
//set the exNdArr for this class, construct the structure of this unrooted tree
void OptUrTree::setExNd (ExNdArr &exNdArr) {
	exNdArr_ = exNdArr;
	curNExNd_ = exNdArr.getCurNNd ();
	maxNExNd_ = exNdArr.getMaxNNd ();
}

//--------------------------------------------------------------------
//set the inNdArr for this class, construct the structure of this unrooted tree
void OptUrTree::setInNd (InNdArr &inNdArr) {
	inNdArr_ = inNdArr;
	curNInNd_ = inNdArr.getCurNNd ();
	maxNInNd_ = inNdArr.getMaxNNd ();
}

//--------------------------------------------------------------------
//set out group node no for this tree
void OptUrTree::setOutGrpNd (int outGrpNdNo) {
	outGrpNdNo_ = outGrpNdNo;
}

//--------------------------------------------------------------------
//set the brArr for this class, construct the structure of this unrooted tree
void OptUrTree::setBr (BrArr<double> &brArr) {
	liBrArr_.copyBase (brArr);
	curNBr_ = brArr.getCurNBr ();
	maxNBr_ = brArr.getMaxNBr ();
}

double OptUrTree::getBrLen (int brNo) {
	return liBrArr_[brNo].getLen ();
}

//--------------------------------------------------------------------
//get the brArr of this tree after optimizing
void OptUrTree::getBr (BrArr<double> &brArr) {
	for (int countBr_ = 0; countBr_ < maxNBr_; countBr_ ++) {
		brArr[countBr_] = liBrArr_[countBr_];
	}




}

//--------------------------------------------------------------------
//get the exNdArr of this tree after optimizing
void OptUrTree::getExNd (ExNdArr &exNdArr) {
	exNdArr = exNdArr_;
}

//--------------------------------------------------------------------
//get the inNdArr of this tree after optimizing
void OptUrTree::getInNd (InNdArr &inNdArr) {
	inNdArr = inNdArr_;
}

//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
/*begin local optimization
 
 
 
             *
             *
             *
             parNd
             *
           *   *
         *      *dBrNo_
       *         *
    sNd        dNd
    *
  *   *
 *     *chiBrNo_
*       *
*        chiNdNo
*/

void OptUrTree::changeNd (int parNdNo, int sNdNo, int chiNdNo, int dNdNo) {
	int dBrNo_ = inNdArr_[parNdNo].findComBr (dNdNo);
	int chiBrNo_ = inNdArr_[sNdNo].findComBr (chiNdNo);

	double dBrLen_ = liBrArr_[dBrNo_].getLen ();
	double chiBrLen_ = liBrArr_[chiBrNo_].getLen ();

	LiNd *tailChiLiNd_;
	tailChiLiNd_ = &liBrArr_[chiBrNo_].getTailLiNd ();

	LiNd *tailDLiNd_;
	tailDLiNd_ = &liBrArr_[dBrNo_].getTailLiNd ();


	liBrArr_[chiBrNo_].setTailLiNd (tailDLiNd_);
	liBrArr_[dBrNo_].setTailLiNd (tailChiLiNd_);


	liBrArr_[dBrNo_].setLen (chiBrLen_);

	inNdArr_[parNdNo].changeNeiNd (dNdNo, chiNdNo);
	if (isInNd (dNdNo) == 1) {
		inNdArr_[dNdNo].changeBr (dBrNo_, chiBrNo_);
		inNdArr_[dNdNo].changeNeiNd (parNdNo, sNdNo);
	} else {
		exNdArr_[dNdNo].changeBr (chiBrNo_);
		exNdArr_[dNdNo].changeInNd (sNdNo);
	}


	liBrArr_[chiBrNo_].setLen (dBrLen_);
	inNdArr_[sNdNo].changeNeiNd (chiNdNo, dNdNo);
	if (isInNd (chiNdNo) == 1) {
		inNdArr_[chiNdNo].changeBr (chiBrNo_, dBrNo_);
		inNdArr_[chiNdNo].changeNeiNd (sNdNo, parNdNo);
	} else {
		exNdArr_[chiNdNo].changeBr (dBrNo_);
		exNdArr_[chiNdNo].changeInNd (parNdNo);
	}

	Utl::swap (optedLenArr_[chiBrNo_], optedLenArr_[dBrNo_]);
	Utl::swap (oriLenArr_[chiBrNo_], oriLenArr_[dBrNo_]);
	Utl::swap (canLenArr_[chiBrNo_], canLenArr_[dBrNo_]);

	//draw ();
}


double OptUrTree::optLocalBr (int brNo) {
	liBrArr_[brNo].reCmpHeadLiNd ();
	liBrArr_[brNo].reCmpTailLiNd ();
	//  double oldLogLi_ = liBrArr_[brNo].cmpInLogLi ();

	// double oldBrLen_ = liBrArr_[brNo].getLen ();
	double newBrLen_ = optBr (brNo);
	liBrArr_[brNo].setLen (newBrLen_);

	double newLogLi_ = liBrArr_[brNo].cmpLogLi ();
	return newLogLi_;
}

/*
 
             *                                  *                          *
 
 *                      *                          *
    *                      *                          *
   parNd                    parNd                     parNd
    *                     *   *                    *     *
   *   *                  *      *                 *       *
   *      *               *         *               *         *
   *        *             *            *            *           *
  sNd         dNd         sNd           lChi       sNd          RChi
  *                       *   *                    *    *
  *   *                  *     *                  *       *
LChi  RChi             dNd     RChi             LNd      dChi
*/

void OptUrTree::draw () {
	UrTree tree_;
	tree_.getTree ();
	Mat<char> image_;
	tree_.createImage (NON_BR_LEN, 0, image_);
	OutStream::write (image_, std::cout);
}

//==================================================================
void OptUrTree::createScoreBr () {
	// added BQM
	score_sum = 0.0;
	original_logli = cmpLogLi();
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++) {
		oriLenArr_[brNo_] = liBrArr_[brNo_].getLen ();
		if (liBrArr_[brNo_].isIn () == 1) {
#ifdef PARALLEL
			if (stopSignalCome())
				return;
#endif
			//      draw ();
			int parNdNo_ = liBrArr_[brNo_].getHeadNd ();
			int sNdNo_ = liBrArr_[brNo_].getTailNd ();
			int lChiNdNo_, rChiNdNo_;
			inNdArr_[sNdNo_].get2RemNeiNd (parNdNo_, lChiNdNo_, rChiNdNo_);
			int dNdNo_ = inNdArr_[parNdNo_].getSibNd (sNdNo_);
			/*
			      liBrArr_[brNo_].cmpHeadLiNd ();
			      liBrArr_[brNo_].cmpTailLiNd ();
			      double oriLogLi_ = liBrArr_[brNo_].cmpInLogLiOneRate ();
			*/
			double logLi0_ = optLocalBr (brNo_);
			double optedBrLen_ = liBrArr_[brNo_].getLen ();

			optedLenArr_[brNo_] = optedBrLen_;

			/*
			      if (oriLogLi_ > logLi0_) {
			        logLi0_ = optLocalBr (brNo_);
			      }
			*/
			changeNd (parNdNo_, sNdNo_, lChiNdNo_, dNdNo_);
			double logLi1_ = optLocalBr (brNo_);
			double brLen1_ = liBrArr_[brNo_].getLen ();
			changeNd (parNdNo_, sNdNo_, dNdNo_, lChiNdNo_);

			changeNd (parNdNo_, sNdNo_, rChiNdNo_, dNdNo_);
			double logLi2_ = optLocalBr (brNo_);
			double brLen2_ = liBrArr_[brNo_].getLen ();
			changeNd (parNdNo_, sNdNo_, dNdNo_, rChiNdNo_);
			/*
			      if (logLi2_ > logLi0_ || logLi1_ > logLi0_) {
			        std::cout << brNo_ << " " << logLi0_ << " " << logLi1_ - logLi0_ << " " << logLi2_ - logLi0_ << endl;
			     }
			*/
			liBrArr_[brNo_].setLen (oriLenArr_[brNo_]);
			liBrArr_[brNo_].reCmpHeadLiNd ();
			liBrArr_[brNo_].reCmpTailLiNd ();

			if (logLi1_ > logLi2_) {
				liBrArr_[brNo_].score_ = logLi1_ - logLi0_;
				canLenArr_[brNo_] = brLen1_;
				liBrArr_[brNo_].isSwapLChi_ = 1;
			} else {
				liBrArr_[brNo_].score_ = logLi2_ - logLi0_;
				canLenArr_[brNo_] = brLen2_;
				liBrArr_[brNo_].isSwapLChi_ = 0;
			}
			double best_logli = (logLi0_ > logLi1_) ? logLi0_ : logLi1_;
			if (logLi2_ > best_logli) best_logli = logLi2_;
			score_sum += best_logli - original_logli;
			//      draw ();
		} else {
			optedLenArr_[brNo_] = oriLenArr_[brNo_];
		}
	}
}

#ifdef PARALLEL

struct BranchScoreInfo {
	double score, len, opt_len;
};

//==================================================================
/**
	Parallel version of create score for every branch, used for NNI
*/
void OptUrTree::par_createScoreBr () {

 	// added BQM
	score_sum = 0.0;
	original_logli = cmpLogLi();
	
	//cout << mpi_myrank << " parallel" << endl;
	int total_work = maxNBr_ - maxNExNd_; // number of internal branches
	int work_load = (int)ceil( (double)total_work / mpi_size );
	int work_start = work_load * mpi_myrank;
	int global_count = -1;
	
	BranchScoreInfo *global_branch_info = new BranchScoreInfo[work_load * mpi_size];
	
	BranchScoreInfo * branch_info = new BranchScoreInfo[work_load];
	
	int count = 0;
	
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_++) {
		oriLenArr_[brNo_] = liBrArr_[brNo_].getLen ();
		if (liBrArr_[brNo_].isIn () == 1) {
			global_count++;
			if (global_count < work_start || count >= work_load)
				continue;
			//      draw ();
			int parNdNo_ = liBrArr_[brNo_].getHeadNd ();
			int sNdNo_ = liBrArr_[brNo_].getTailNd ();
			int lChiNdNo_, rChiNdNo_;
			inNdArr_[sNdNo_].get2RemNeiNd (parNdNo_, lChiNdNo_, rChiNdNo_);
			int dNdNo_ = inNdArr_[parNdNo_].getSibNd (sNdNo_);
			/*
			      liBrArr_[brNo_].cmpHeadLiNd ();
			      liBrArr_[brNo_].cmpTailLiNd ();
			      double oriLogLi_ = liBrArr_[brNo_].cmpInLogLiOneRate ();
			*/
			double logLi0_ = optLocalBr (brNo_);
			double optedBrLen_ = liBrArr_[brNo_].getLen ();

			branch_info[count].opt_len = optedBrLen_;

			/*
			      if (oriLogLi_ > logLi0_) {
			        logLi0_ = optLocalBr (brNo_);
			      }
			*/
			changeNd (parNdNo_, sNdNo_, lChiNdNo_, dNdNo_);
			double logLi1_ = optLocalBr (brNo_);
			double brLen1_ = liBrArr_[brNo_].getLen ();
			changeNd (parNdNo_, sNdNo_, dNdNo_, lChiNdNo_);

			changeNd (parNdNo_, sNdNo_, rChiNdNo_, dNdNo_);
			double logLi2_ = optLocalBr (brNo_);
			double brLen2_ = liBrArr_[brNo_].getLen ();
			changeNd (parNdNo_, sNdNo_, dNdNo_, rChiNdNo_);
			/*
			      if (logLi2_ > logLi0_ || logLi1_ > logLi0_) {
			        std::cout << brNo_ << " " << logLi0_ << " " << logLi1_ - logLi0_ << " " << logLi2_ - logLi0_ << endl;
			     }
			*/
			liBrArr_[brNo_].setLen (oriLenArr_[brNo_]);
			liBrArr_[brNo_].reCmpHeadLiNd ();
			liBrArr_[brNo_].reCmpTailLiNd ();

			if (logLi1_ > logLi2_) {
				branch_info[count].score = logLi1_ - logLi0_;
				branch_info[count].len = brLen1_;
				//is_left[count] = true;
			} else {
				branch_info[count].score = logLi2_ - logLi0_;
				branch_info[count].len = -brLen2_;
				//is_left[count] = false;
			}
			//      draw ();
			count++;
		} else {
			optedLenArr_[brNo_] = oriLenArr_[brNo_];
		}
			
	}
	
	// broadcast to all processes
	MPI_Allgather(branch_info, work_load * 3, MPI_DOUBLE, global_branch_info, work_load * 3, MPI_DOUBLE, MPI_COMM_WORLD);
	
	// copy data
	count = -1;
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_++)
		if (liBrArr_[brNo_].isIn () == 1) {
			count++;
			liBrArr_[brNo_].score_ = global_branch_info[count].score;
			canLenArr_[brNo_] = fabs(global_branch_info[count].len);
			optedLenArr_[brNo_] = global_branch_info[count].opt_len;
			liBrArr_[brNo_].isSwapLChi_ = (global_branch_info[count].len > 0.0);
		}
		
	delete branch_info;
	delete global_branch_info;
}

#endif // PARALLEL

//==================================================================
void OptUrTree::sortScoreBr () {
	inBrNoLs_.set (maxNExNd_ - 3, 0);
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (liBrArr_[brNo_].isIn () == 1)
			inBrNoLs_ += brNo_;

	int nInBr_ = inBrNoLs_.getSize ();
	
	/*
	for (int count_ = 0; count_ < nInBr_; count_ ++)
		for (int count2_ = count_ + 1; count2_ < nInBr_; count2_ ++) {

			int brNo_ = inBrNoLs_[count_];
			int brNo2_ = inBrNoLs_[count2_];
			if (liBrArr_[brNo_].score_ < liBrArr_[brNo2_].score_)
				Utl::swap (inBrNoLs_[count_], inBrNoLs_[count2_]);
		}
	*/
	sortScoreBr(0, nInBr_ - 1);
}

//==================================================================
/**
	sort branch scores by Quick Sort
*/

void OptUrTree::sortScoreBr (int left, int right) {
	int ori_left = left, ori_right = right;
	int middle = (left + right) / 2;
	double middle_value = liBrArr_[inBrNoLs_[middle]].score_;
	do {
		while (liBrArr_[inBrNoLs_[left]].score_ > middle_value) left++;
		while (liBrArr_[inBrNoLs_[right]].score_ < middle_value) right--;
		if (left <= right) {
			Utl::swap (inBrNoLs_[left], inBrNoLs_[right]);
			left++;
			right--;
		}
	} while (left <= right);
	if (ori_left < right) sortScoreBr(ori_left, right);
	if (left < ori_right) sortScoreBr(left, ori_right);
}

//==================================================================
int OptUrTree::findNPosSwapBr (int maxNSwapBr) {
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++) {
		liBrArr_[brNo_].isCan_ = 1;
		liBrArr_[brNo_].isSwap_ = 0;
	}

	int nPosSwapBr_ = 0;
	int nInBr_ = inBrNoLs_.getSize ();

	int time_, count_;
	for (time_ = 0; time_ < maxNSwapBr; time_ ++) {

		for (count_ = 0; count_ < nInBr_; count_ ++) {
			int brNo_ = inBrNoLs_[count_];

			if (liBrArr_[brNo_].score_ <= ZERO)
				return nPosSwapBr_;

			if (liBrArr_[brNo_].isCan_ == 1) {

				liBrArr_[brNo_].isSwap_ = 1;
				liBrArr_[brNo_].isCan_ = 0;
				nPosSwapBr_ ++;

				int headNdNo_ = liBrArr_[brNo_].getHeadNd ();
				int tailNdNo_ = liBrArr_[brNo_].getTailNd ();

				int parBrNo0_, parBrNo1_;
				inNdArr_[headNdNo_].get2RemBr (brNo_, parBrNo0_, parBrNo1_);
				liBrArr_[parBrNo0_].isCan_ = 0;
				liBrArr_[parBrNo1_].isCan_ = 0;


				int chiBrNo0_, chiBrNo1_;
				inNdArr_[tailNdNo_].get2RemBr (brNo_, chiBrNo0_, chiBrNo1_);
				liBrArr_[chiBrNo0_].isCan_ = 0;
				liBrArr_[chiBrNo1_].isCan_ = 0;

				break;
			}
		}

		if (count_ == nInBr_)
			return nPosSwapBr_;
	}

	return nPosSwapBr_;
}

void OptUrTree::trySwapBr (double lamda) {
	int brNo_;
	for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (liBrArr_[brNo_].isSwap_ == 1) {
			int parNdNo_ = liBrArr_[brNo_].getHeadNd ();
			int sNdNo_ = liBrArr_[brNo_].getTailNd ();
			int lChiNdNo_, rChiNdNo_;
			inNdArr_[sNdNo_].get2RemNeiNd (parNdNo_, lChiNdNo_, rChiNdNo_);
			int dNdNo_ = inNdArr_[parNdNo_].getSibNd (sNdNo_);

			if (liBrArr_[brNo_].isSwapLChi_ == 1) {
				//         std::cout << parNdNo_ << "  " << sNdNo_ << " " << lChiNdNo_ << " " << dNdNo_ << endl;
				changeNd (parNdNo_, sNdNo_, lChiNdNo_, dNdNo_);
			} else {
				//         std::cout << parNdNo_ << "  " << sNdNo_ << " " << rChiNdNo_ << " " << dNdNo_ << endl;
				changeNd (parNdNo_, sNdNo_, rChiNdNo_, dNdNo_);
			}

			double newLen_ = canLenArr_[brNo_];
			liBrArr_[brNo_].setLen (newLen_);
		}

	for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (liBrArr_[brNo_].isSwap_ == 0) {
			double oriLen_ = oriLenArr_[brNo_];
			double optedLen_ = optedLenArr_[brNo_];
			double newLen_ = oriLen_ + lamda * (optedLen_ - oriLen_);
			liBrArr_[brNo_].setLen (newLen_);
		}

}

void OptUrTree::swapBrBack () {
	int brNo_;
	for (brNo_ = maxNBr_ - 1; brNo_ >= 0; brNo_ --) {
		if (liBrArr_[brNo_].isSwap_ == 1) {
			int parNdNo_ = liBrArr_[brNo_].getHeadNd ();
			int sNdNo_ = liBrArr_[brNo_].getTailNd ();
			int lChiNdNo_, rChiNdNo_;
			inNdArr_[sNdNo_].get2RemNeiNd (parNdNo_, lChiNdNo_, rChiNdNo_);
			int dNdNo_ = inNdArr_[parNdNo_].getSibNd (sNdNo_);

			if (liBrArr_[brNo_].isSwapLChi_ == 1)
				changeNd (parNdNo_, sNdNo_, lChiNdNo_, dNdNo_);
			else
				changeNd (parNdNo_, sNdNo_, rChiNdNo_, dNdNo_);
		}
	}

	for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		liBrArr_[brNo_].setLen (oriLenArr_[brNo_]);
	//  std::cout << "finish swaping branches back !" << endl;
}

double OptUrTree::cmpSwapLogLi () {
	cmpLiNd ();
	double logLi_ = liBrArr_[outGrpBrNo_].cmpLogLi ();
	cmpHeadLiNd ();
	return logLi_;
}

double OptUrTree::optSwapLogLi () {
	cmpLiNd ();
	double logLi_ = optBranches (1);
	cmpHeadLiNd ();
	return logLi_;
}


//#define DEBUG_SWAPBR

int OptUrTree::swapBr () {
#ifdef PARALLEL
/*
	if ( isParallelPhase(!PARALLEL_LOGLI) )
		par_createScoreBr();
	else
*/
		createScoreBr();
#else
	createScoreBr ();
#endif // PARALLEL
	
#ifdef PARALLEL
	if (stopSignalCome())
		return 0;
#endif
	sortScoreBr ();

	int nPosSwapBr_ = findNPosSwapBr (maxNBr_);
	if (nPosSwapBr_ == 0) {
#ifdef DEBUG_SWAPBR	
		cout << endl;
#endif 
		
		return 0;
	}

#ifdef DEBUG_SWAPBR	
	cout.precision(10);
	cout << "Old logli = " << original_logli;
	upper_bound_logli = original_logli + score_sum * (nPosSwapBr_ + 1);
	cout << " predicted = " << upper_bound_logli << endl;
#endif 

	double lamda_ = 0.75;
	double curLogLi_ = original_logli, newLogLi_ ;

	int nSwapBr_ = static_cast<int> (nPosSwapBr_ * lamda_);
	if (nSwapBr_ == 0 && nPosSwapBr_ > 0)
		nSwapBr_ = 1;

#ifdef DEBUG_SWAPBR	
	cout << "score = " << score_sum << " nswap = " << nPosSwapBr_;
#endif 

	do {
#ifdef PARALLEL
		if (stopSignalCome())
			break;
#endif // PARALLEL
		//cout << "nSwapBr= " << nSwapBr_ << endl;
		findNPosSwapBr (nSwapBr_);
		trySwapBr (lamda_);
		newLogLi_ = optSwapLogLi ();
		if (newLogLi_ < curLogLi_) {
			swapBrBack ();
			lamda_ /= 2.0;
			nSwapBr_ = static_cast<int> (nPosSwapBr_ * lamda_);
			if (nSwapBr_ == 0)
				nSwapBr_ = 1;
		} else
			break;
	} while (nSwapBr_ >= 1);

#ifdef DEBUG_SWAPBR	
	cout << " optimized = " << newLogLi_ << endl;
#endif 

	return nSwapBr_;
}


void OptUrTree::updateBestLogLi (double logLi) {
	//  std::cout << bestLogLi_ << " / " << logLi << endl;
	if (logLi > bestLogLi_ + ZERO) {
		bestLogLi_ = logLi;
		//    std::cout << "current best logli = " << bestLogLi_ << endl;
	}
}

double OptUrTree::optTopology (bool &topo_changed) {
	//  std::cout << "start log likelihood " << bestLogLi_ << endl;
	int nSwapBr_ = 0;
	topo_changed = false;
	
	int count = 0;

	if (bestLogLi_ > global_bestlogli) global_bestlogli = bestLogLi_;
	double first_logli;

	if (nni_lh_iter) {
		nni_file << nni_lh_iter << "\t" << 0 << "\t" << global_bestlogli << "\t0"<< endl;
		
		first_logli = optBranches();
		nni_file << nni_lh_iter << "\t" << count+1 << "\t" << first_logli << "\t0" << endl;
		//nni_file.flush();
	}

	do {
		count++;
		//cout << count << " " << endl;
		nSwapBr_ = swapBr ();
		if (nSwapBr_ != 0)
			topo_changed = true;
#ifdef PARALLEL
		if (stopSignalCome())
			break;
#endif
		double newLogLi_ = cmpLogLi ();
		updateBestLogLi (newLogLi_);
		if (nni_lh_iter) {
			nni_file << nni_lh_iter << "\t" << count+1 << "\t" << newLogLi_ << "\t" << nSwapBr_ << endl;

			nni_tree << nSwapBr_ << endl;
			int nInBr_ = inBrNoLs_.getSize ();
			int count_;

			for (count_ = 0; count_ < nInBr_; count_ ++) {
				int brNo_ = inBrNoLs_[count_];
				if (liBrArr_[brNo_].isSwap_ == 1) {
					nni_tree << first_logli + liBrArr_[brNo_].score_ << endl;
				}
			}
			first_logli = newLogLi_;
			//nni_file << newLogLi_ << " ";
			//nni_file.flush();
		}
		//   std::cout << nSwapBr_ << " / " << newLogLi_ << endl;
	} while (nSwapBr_ != 0);

	if (bestLogLi_ > global_bestlogli) global_bestlogli = bestLogLi_;
	nni_lh_iter++;

	/*
	if (nni_lh_iter) {
		nni_file << endl;
	}
	*/
	return bestLogLi_;
}


//end of local optimization
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************
//****************************************************************************

//--------------------------------------------------------------------
//assign values for all liInNd_ of this tree
void OptUrTree::cmpLiNd () {
	int brNo_;
	for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++) {
		if (liBrArr_[brNo_].getId () != -1)
			liBrArr_[brNo_].openLiNd ();
	}

	liBrArr_[outGrpBrNo_].cmpTailLiNd ();
	liBrArr_[outGrpBrNo_].cmpHeadLiNd ();
}




//--------------------------------------------------------------------
//assign values for all liInNd_ of this tree
void OptUrTree::cmpHeadLiNd () {
	int brNo_;
	for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++) {
		if (liBrArr_[brNo_].getId () != -1)
			liBrArr_[brNo_].openHeadLiNd ();
	}
	liBrArr_[outGrpBrNo_].cmpHeadLiNd ();
}


//--------------------------------------------------------------------
//cmp the log likelihood of this tree
double OptUrTree::cmpLogLi () {
	double logLi_ = liBrArr_[outGrpBrNo_].cmpLogLi ();
	return logLi_;
}


//--------------------------------------------------------------------
/*
cmp the negative log likelihood of this tree when only the active br changes
Use it for optimizing by brent method
*/
double OptUrTree::cmpNegLogLi (double brLen) {
	nCmpNegLogLiBr_ ++;
	liBrArr_[actBrNo_].setLen (brLen);
	double logLi_;

	logLi_ = liBrArr_[actBrNo_].cmpLogLi  ();

	return -logLi_;
}

//--------------------------------------------------------------------
/*
cmp the negative log likelihood of this tree when only the active br changes
Use it for optimizing by brent method
*/
double OptUrTree::cmpLogLiDerivatives (double brLen, double &logli_derv1, double &logli_derv2, bool calc_logli) {
	nCmpNegLogLiBr_ ++;
	liBrArr_[actBrNo_].setLen (brLen);

	return liBrArr_[actBrNo_].cmpLogLiDerivatives  (logli_derv1, logli_derv2, calc_logli);

}


double brent_time = 0.0;
double newton_time, old_time_ = 0.0;

//#define DEBUG_INFO

//--------------------------------------------------------------------
//opt this br
double OptUrTree::optBr (int brNo) {
	actBrNo_ = brNo;
	double len_ = liBrArr_[brNo].getLen ();
	if (len_ <= MIN_BR_LEN)
		len_ = MIN_BR_LEN;
	if (len_ >= MAX_BR_LEN)
		len_ = MAX_BR_LEN;


	double newLen_ = -1.0;

	//printf("br %d\n", brNo);

#ifdef PARALLEL
	double start_time = MPI_Wtime();
#endif // PARALLEL

#ifdef DEBUG_INFO
	newLen_ = optOneDim(MIN_BR_LEN, len_, MAX_BR_LEN,
	                    EPS_OPT_UR_TREE_BR, &fx_, &f2x_);
#endif // DEBUG_INFO

#ifdef PARALLEL
	double end_time = MPI_Wtime();
	brent_time += end_time - start_time;
#endif // PARALLEL

#ifdef DEBUG_INFO
	cout << "Brent: " << newLen_ << " logli=" << -fx_;
#endif // DEBUG_INFO

#ifdef PARALLEL
#ifdef DEBUG_INFO
	cout << " time=" << brent_time;
#endif // DEBUG_INFO
#endif // PARALLEL

	double logli;

#ifdef PARALLEL
	start_time = MPI_Wtime();
#endif // PARALLEL

	
	double fx_, f2x_;
	
	
	if (mymodel.pam_brent == 2)
		newLen_ = optOneDim(MIN_BR_LEN, len_, MAX_BR_LEN, EPS_OPT_UR_TREE_BR, &fx_, &f2x_);
	  else 
		newLen_ = doNewtonRaphson(len_, logli);
	
	
#ifdef PARALLEL
	end_time = MPI_Wtime();
	newton_time += end_time-start_time;
#endif // PARALLEL

#ifdef DEBUG_INFO
	cout << " vs. Newton: " << newLen_ << " logli=" << logli;
#endif // DEBUG_INFO

#ifdef PARALLEL
#ifdef DEBUG_INFO
	cout << " time=" << newton_time;
	cout << endl;
#endif // DEBUG_INFO
#endif // PARALLEL

	/*
	#ifndef DEBUG_INFO
		if (newton_time > old_time_ + 1.0) {
			cout << "Brent time: " << brent_time << "  Newton time: " << newton_time << endl;
			old_time_ = newton_time;
		}
	#endif
	*/
	return newLen_;
}


//--------------------------------------------------------------------
//opt the node. Thus, optimize all chidrent branches of this tree
void OptUrTree::optBranches (int ndNo, int &nChangedBr) {
	if (isExNd (ndNo) == 1) { //the case of an external node
		if (ndNo == outGrpNdNo_) {//the case of out group node no
			double oldLen_ = liBrArr_[outGrpBrNo_].getLen();
			double newLen_ = optBr(outGrpBrNo_);
			liBrArr_[outGrpBrNo_].setLen (newLen_);
			if (fabs (newLen_ - oldLen_) > BR_DIF_ERROR)

				nChangedBr ++;
			optBranches (artInRootNo_, nChangedBr);
		}
	} else { //the case of internal node
		Vec<int> *neiNdNoLs_;
		neiNdNoLs_ = &inNdArr_[ndNo].getNeiNd ();

		Vec<int> *brNoLs_;

		brNoLs_ = &inNdArr_[ndNo].getBr ();

		for (int count_ = 1; count_ < neiNdNoLs_->getSize (); count_ ++) {
			int chiNdNo_ = (*neiNdNoLs_)[count_];
			int chiBrNo_ = (*brNoLs_)[count_];

			double oldLen_ = liBrArr_[chiBrNo_].getLen();

			liBrArr_[chiBrNo_].reCmpHeadLiNd ();
			double newLen_ = optBr(chiBrNo_);

			if (fabs (newLen_ - oldLen_) > BR_DIF_ERROR)
				nChangedBr ++;

			liBrArr_[chiBrNo_].setLen (newLen_);

			optBranches (chiNdNo_, nChangedBr);
		}

		int parBrNo_ = (*brNoLs_)[0];
		liBrArr_[parBrNo_].reCmpTailLiNd ();
	}
}



//--------------------------------------------------------------------
//opt the tree. Thus, optimize all branches of this tree
double OptUrTree::optBranches (int nIt) {
	//cout << "Optimize branches of tree... ";
	//do not optimize again
	double oldLogLi_ = -INFINITIVE;
	double newLogLi_ = -INFINITIVE;
	oldLogLi_ = cmpLogLi ();
	int nChangedBr_;


	nCmpExLogLi_ = 0;


	nCmpInLogLi_ = 0;

	double *bestBrLenArr_ = new double[alignment.getNSeq() * 2];
	for (int time_ = 0; time_ < nIt; time_++) {
		nChangedBr_ = 0;

		int brNo_;
		for (brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
			bestBrLenArr_[brNo_] = liBrArr_[brNo_].getLen ();

#ifdef TIME_MONITOR
		double start_time = MPI_Wtime();
#endif

		optBranches (outGrpNdNo_, nChangedBr_);

#ifdef TIME_MONITOR
		double end_time = MPI_Wtime();
		start_time = MPI_Wtime();
#endif

		newLogLi_ = cmpLogLi ();

		if (nChangedBr_ == 0)
			break;
		else
			oldLogLi_ = newLogLi_;
	}

	bestLogLi_ = newLogLi_;

	delete bestBrLenArr_;

	return bestLogLi_;
}


//--------------------------------------------------------------------
//check whether or not it is a external node
int OptUrTree::isExNd (int ndNo) {
	if (ndNo < maxNExNd_)
		return 1;
	else
		return 0;
}

//--------------------------------------------------------------------

//check whether or not it is a external node

int OptUrTree::isInNd (int ndNo) {
	if (ndNo >= maxNExNd_)
		return 1;
	else
		return 0;
}

//--------------------------------------------------------------------
//write all information of this optimal unrooted tree into the screaten
void OptUrTree:: write () {
	OutStream::write (&exNdArr_, &inNdArr_, &liBrArr_, std::cout);

}

//--------------------------------------------------------------------
//create pattern for all internal node of this tree
void OptUrTree::createPtnLiNd() {}

//--------------------------------------------------------------------
//create external descendant nodes for all internal nodes of this tree
void OptUrTree::createExDesLiNd () {
	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (liBrArr_[brNo_].getId () != -1) {
			liBrArr_[brNo_].createExDesLiNd ();
			//      OutStream::write (liBrArr_[brNo_].getGreLiNd (), std::cout);
			//    OutStream::write (liBrArr_[brNo_].getSmaLiNd (), std::cout);
		}
}

//--------------------------------------------------------------------
//everything is inited here
void OptUrTree::init () {
	clean ();
	outGrpBrNo_ = exNdArr_[outGrpNdNo_].getBr ();
	artInRootNo_ = exNdArr_[outGrpNdNo_].getInNd ();

	genDisMat_ = alignment.getGenDis ();
	estBrLen ();
	//  OutStream::write (liBrArr_, std::cout);


	createExDesLiNd ();
	//  double startTime_ = clock ();
	//  createPtnLiNd ();
	//  std::cout << (clock () - startTime_) / CLOCKS_PER_SEC << endl;
	cmpLiNd ();
}


//--------------------------------------------------------------------
//return the maximum log likelihood of this tree.
double OptUrTree::getLogLi () {
	double logLi_ = cmpLogLi ();
	return logLi_;
}

//--------------------------------------------------------------------
/*create the list of all exnternal nodes which belong to the same one root
it is the function of least square estimate*/
void OptUrTree::createGrpExNd (const int parGrpNo, const int grpNo, Vec<int> &exNdNoLsGrp_) {
	exNdNoLsGrp_.clean ();
	//the case, parNo is an external node
	if (isExNd (parGrpNo) == 1) {
		Vec<int> *exNdNoLs_;
		exNdNoLs_ = &exNdArr_.getNdNo ();
		int nExNd_ = exNdNoLs_->getSize ();


		exNdNoLsGrp_.setLimit (nExNd_);
		for (int count_ = 0; count_ < nExNd_; count_ ++)
			if ( (*exNdNoLs_)[count_] != parGrpNo)
				exNdNoLsGrp_ += (exNdNoLs_)[count_];

	} else { //the case, parNo is an internal node
		inNdArr_.getExDesNd (parGrpNo, grpNo, exNdNoLsGrp_);
	}
}

//--------------------------------------------------------------------
/*
 
it is the function of least square estimate
 
 
compute the distance between two groups, group 1 and group 2.
group1 is a subtree whose the root is grpNo1
group2 is a subtree whose the root is grpNo2
parentNo is the parent of grpNo1, and grpNo2
*/
double OptUrTree::cmpGrpDis (const int parGrpNo, const int grpNo1, const int grpNo2) {
	Vec<int> exNdNoLsGrp1_;
	createGrpExNd (parGrpNo, grpNo1, exNdNoLsGrp1_);

	Vec<int> exNdNoLsGrp2_;

	createGrpExNd (parGrpNo, grpNo2, exNdNoLsGrp2_);

	double grpDis_ = 0.0;
	for (int count1_ = 0; count1_ < exNdNoLsGrp1_.getSize (); count1_ ++)
		for (int count2_ = 0; count2_ < exNdNoLsGrp2_.getSize (); count2_ ++) {
			int exNdNo1_ = exNdNoLsGrp1_[count1_];
			int exNdNo2_ = exNdNoLsGrp2_[count2_];
			grpDis_ += (*genDisMat_)[ exNdNo1_][exNdNo2_];
		}
	return grpDis_;


}

//--------------------------------------------------------------------
/*
it is the function of least square estimate
 
compute the distance between two groups, group 1 and group 2.
group1 is a subtree whose the root is grpNo1
group2 is a subtree whose the root is grpNo2
parentNo is the parent of grpNo1, and grpNo2
*/
double OptUrTree::cmpGrpDis (Vec<int> &exNdNoLsGrp1, Vec<int> &exNdNoLsGrp2) {
	double grpDis_ = 0.0;
	for (int count1_ = 0; count1_ < exNdNoLsGrp1.getSize (); count1_ ++) {

		int exNdNo1_ = exNdNoLsGrp1[count1_];
		for (int count2_ = 0; count2_ < exNdNoLsGrp2.getSize (); count2_ ++) {
			int exNdNo2_ = exNdNoLsGrp2[count2_];
			grpDis_ += (*genDisMat_)[ exNdNo1_][exNdNo2_];
		}
	}
	return grpDis_;
}

//--------------------------------------------------------------------
/*it is the function of least square estimate
compute the distance from exNdNo to exNdNoLsGrp_ */
double OptUrTree::cmpGrpDis (int exNdNo, Vec<int> &exNdNoLsGrp_) {
	double grpDis_ = 0.0;
	for (int count_ = 0; count_ < exNdNoLsGrp_.getSize (); count_ ++)
		grpDis_ += (*genDisMat_)[ exNdNo ][ exNdNoLsGrp_[count_] ];
	return grpDis_;
}




//--------------------------------------------------------------------
/*it is the function of least square estimate
   exNdNo
     *
     *
   inNdNo
  *     *
 *       *
B         C
 
using Rzhetsky and Nei method to estimate the branch of this tree as
a starting point for maximum likelihood*/
double OptUrTree::estExBrLen (Br<double> &exBr) {
	int inNdNo_ = exBr.getGreNd ();
	int exNdNo_ = exBr.getSmaNd ();
	int grpNoB_, grpNoC_;
	inNdArr_[inNdNo_].get2RemNeiNd (exNdNo_, grpNoB_, grpNoC_);

	Vec<int> exNdNoLsGrpB_;
	createGrpExNd (inNdNo_, grpNoB_, exNdNoLsGrpB_);
	//  OutStream::write (exNdNoLsGrpB_, 1);


	Vec<int> exNdNoLsGrpC_;
	createGrpExNd (inNdNo_, grpNoC_, exNdNoLsGrpC_);
	//  OutStream::write (exNdNoLsGrpC_, 1);

	int mB_ = exNdNoLsGrpB_.getSize ();
	int mC_ = exNdNoLsGrpC_.getSize ();

	double aveDisAB_ = cmpGrpDis (exNdNo_, exNdNoLsGrpB_) / mB_;
	double aveDisAC_ = cmpGrpDis (exNdNo_, exNdNoLsGrpC_) / mC_;
	double aveDisBC_ = cmpGrpDis (exNdNoLsGrpB_, exNdNoLsGrpC_) / (mB_ * mC_);
	double estExBrLen_ = ( aveDisAB_ + aveDisAC_ - aveDisBC_) / 2.0;

	if (estExBrLen_ < MIN_BR_LEN)
		estExBrLen_ = MIN_BR_LEN;

	exBr.setLen (estExBrLen_);
	return estExBrLen_;
}


//--------------------------------------------------------------------
/*it is the function of least square estimate
A                           C
  *                        *
     inNdNoAB_******inNdNoDC_
  *                         *
B                            D
using Rzhetsky and Nei method to estimate the branch of this tree as
a starting point for maximum likelihood*/
double OptUrTree::estInBrLen (Br<double> &inBr) {
	int inNdNoAB_, inNdNoCD_;
	inBr.getNd (inNdNoAB_, inNdNoCD_);

	int grpNoA_, grpNoB_, grpNoC_, grpNoD_;
	inNdArr_[inNdNoAB_].get2RemNeiNd (inNdNoCD_, grpNoA_, grpNoB_);
	inNdArr_[inNdNoCD_].get2RemNeiNd (inNdNoAB_, grpNoC_, grpNoD_);

	Vec<int> exNdNoLsGrpA_, exNdNoLsGrpB_, exNdNoLsGrpC_, exNdNoLsGrpD_;
	createGrpExNd (inNdNoAB_, grpNoA_, exNdNoLsGrpA_);
	createGrpExNd (inNdNoAB_, grpNoB_, exNdNoLsGrpB_);
	createGrpExNd (inNdNoCD_, grpNoC_, exNdNoLsGrpC_);
	createGrpExNd (inNdNoCD_, grpNoD_, exNdNoLsGrpD_);


	int mA_ = exNdNoLsGrpA_.getSize ();
	int mB_ = exNdNoLsGrpB_.getSize ();
	int mC_ = exNdNoLsGrpC_.getSize ();
	int mD_ = exNdNoLsGrpD_.getSize ();


	double gamma_ = static_cast<double> (mB_ * mC_ + mA_ * mD_) / ( ( mA_ + mB_)  * (mC_ + mD_) ) ;

	double aveDisAB_ = cmpGrpDis (exNdNoLsGrpA_, exNdNoLsGrpB_) / (mA_ * mB_);

	double aveDisAC_ = cmpGrpDis (exNdNoLsGrpA_, exNdNoLsGrpC_) / (mA_ * mC_);
	double aveDisAD_ = cmpGrpDis (exNdNoLsGrpA_, exNdNoLsGrpD_) / (mA_ * mD_);
	double aveDisBC_ = cmpGrpDis (exNdNoLsGrpB_, exNdNoLsGrpC_) / (mB_ * mC_);
	double aveDisBD_ = cmpGrpDis (exNdNoLsGrpB_, exNdNoLsGrpD_) / (mB_ * mD_);
	double aveDisCD_ = cmpGrpDis (exNdNoLsGrpC_, exNdNoLsGrpD_) / (mC_ * mD_);

	double estInBrLen_ =  0.5 * ( gamma_ * ( aveDisAC_ + aveDisBD_ ) +
	                              (1.0 - gamma_) * ( aveDisBC_ + aveDisAD_ )

	                              - aveDisAB_ - aveDisCD_ );
	if (estInBrLen_ < MIN_BR_LEN)

		estInBrLen_ = MIN_BR_LEN;

	inBr.setLen (estInBrLen_);
	return estInBrLen_;

}


//--------------------------------------------------------------------
/*it is the function of least square estimate
using Rzhetsky and Nei method to estimate the branches of this tree as
a starting point for maximum likelihood*/
void OptUrTree::estBrLen () {
	//do not estimate the branhes len again of the same tree
	if (isBrLenEsted_ == 1)
		return;
	//  double start_ = clock ();

	for (int brNo_ = 0; brNo_ < maxNBr_; brNo_ ++)
		if (liBrArr_[brNo_].getId () != -1  && liBrArr_[brNo_].getLen () < 0.0) {
			if (liBrArr_[brNo_].isEx () == 1)
				estExBrLen (liBrArr_[brNo_]);
			else
				estInBrLen (liBrArr_[brNo_]);
		}



	//  std::cout << "time " << (clock () - start_) / CLOCKS_PER_SEC << "  /  ";

	isBrLenEsted_ = 1;
}



//--------------------------------------------------------------------
//release the memory of this class
void OptUrTree::release () {
	liBrArr_.release ();


	exNdArr_.release ();
	inNdArr_.release ();
	if (optedLenArr_) delete optedLenArr_;	
	if (canLenArr_) delete canLenArr_;
	if (oriLenArr_) delete oriLenArr_;
	oriLenArr_ = NULL;
	canLenArr_ = NULL;
	optedLenArr_ = NULL;	

}

//--------------------------------------------------------------------
//the destructor function
OptUrTree::~OptUrTree () {
	release ();
	//	std::cout << "this is the destructor function of OptUrTree class " << endl;
}


/**
	Newton-Raphson optimization method
	@param guess the first guess of branch length
	@return the value which optimizes the log likelihood function
*/

#define MAXIT 100
#define MAX(x, y) ((x)>=(y)) ? (x) : (y)

double OptUrTree::doNewtonRaphson(double guess, double &logli) {
	// first and second derivatives
	double logli_derv1, logli_derv2;

	double sub_rate = mymodel.model_->total_num_subst;
	if (sub_rate == 0.0)
		sub_rate = 0.01;

	// evolutionary time and equivalent branch length
	double time_ = guess / sub_rate, brlen = guess;

	// take the first and second derivatives of log likelihood
	logli = cmpLogLiDerivatives(guess, logli_derv1, logli_derv2, true);

	double new_logli = logli, new_time;

	// minimum time
	double min_time = MIN_BR_LEN / sub_rate;

	// maximum time
	double max_time = MAX_BR_LEN / sub_rate;

	// tolerance, to determine the stop criterium
	double tol_time = BR_DIF_ERROR / sub_rate;
	int count = 0;
	do {
		count++;

		// get the step length, used to update the time
		//double abs_logli_derv2 = fabs(logli_derv2);
		
		//double step = - logli_derv1 / MAX(abs_logli_derv2, 1e-100);
		double step;
		
		if (logli_derv2 == 0.0)
			break;
		else  {
			step = logli_derv1 / logli_derv2;
			if (logli_derv2 > 0)
				step = -step; 
		}
		do {

			// update the new time
			new_time = time_ - step;

			// check the lower bound
			if (new_time < min_time) {
				new_time = min_time;
				step = time_ - new_time;
			}

			// check the upper bound
			if (new_time > max_time) {
				new_time = max_time;
				step = time_ - new_time;
			}

			// compute the new log likelihood
			new_logli = -cmpNegLogLi(new_time * sub_rate);
			if (new_logli < logli) {
				// if the new log likelihood is worse, reduce the step length by a factor of 2.0
				step /= 2.0;
			} else
				// otherwise, get a better log likelihood
				break;
		} while (fabs(new_time - time_) > tol_time);

		if (new_logli <= logli)	break;

		// stop if converge
		if (fabs(new_time - time_) <= tol_time) {
			time_ = new_time;
			logli = new_logli;
			brlen = time_ * sub_rate;
			break;
		}

		// update the time for next iteration
		time_ = new_time;
		logli = new_logli;
		brlen = time_ * sub_rate;
		// compute the first and second derivatives again
		cmpLogLiDerivatives(brlen, logli_derv1, logli_derv2, false);
	} while (count < MAXIT);


	if (count >= MAXIT && isMasterProc()) {
		cout << "Warning: Newton Raphson - reach maximum number of iterations!" << endl;
	}


	if (logli_derv2 >= 0) { // method does not converge, perform Brent method now

		//cout << " not local maximum! logli="<<logli << " derv1=" << logli_derv1 <<" derv2=" << logli_derv2 << " brlen=" << brlen << endl;
		double f2x_;

		brlen = optOneDim(MIN_BR_LEN, brlen, MAX_BR_LEN, BR_DIF_ERROR, &logli, &f2x_);
		logli = -logli;
		//logli = -cmpNegLogLi(brlen);

		//cout << " brent logli="<<logli << "  brlen=" << brlen << endl;
	}

	return brlen;

}


#undef MAXIT 
#undef MAX
