#
# txt2tags command line options tester (http://txt2tags.org)
# See also: ../run.py ../lib.py
#
# Note: The .t2t files are generated dynamicaly, based on 'tests' dict data
#

import os
import sys

sys.path.insert(0, "..")
import lib

del sys.path[0]

# text patterns to compose source files
EMPTY_HEADER = "\n"
FULL_HEADER = "Header 1\nHeader 2\nHeader 3\n"
SIMPLE_BODY = "Text.\n"
TITLED_BODY = "= Title 1 =\nText.\n== Title 2 ==\nText.\n"
EMAIL = "user@domain.com\n"
CONFIG_FILE_TXT = "%!target: html\n"
CSS_FILE_TXT = "p { color: blue; }\n"

# a nice postproc to rip off version information from output
VERSION_GOTCHA = "%!postproc: '(generated by txt2tags) [^ ]+' '\\1'\n"

# the registered tests
tests = [
    {
        "name": "arguments-missing",  # t2t
        "content": "",
        "cmdline": [""],
        "redir": ["> arguments-missing.out"],
        "extra": ["notarget", "noinfile"],
    },
    {
        "name": "infile-missing",  # t2t -t html
        "content": "",
        "cmdline": ["-t html"],
        "redir": ["> infile-missing.out"],
        "extra": ["notarget", "noinfile"],
    },
    {
        "name": "infile-empty",  # infile is empty
        "content": "",
        "cmdline": ["-t html"],
        "redir": ["> infile-empty.out"],
        "extra": ["notarget"],
    },
    {
        "name": "infile-not-found-1",  # infile -t html ERROR.t2t
        "content": "",
        "cmdline": ["-t html ERROR.t2t"],
        "redir": ["> infile-not-found-1.out"],
        "extra": ["notarget", "noinfile"],
    },
    {
        "name": "infile-not-found-2",  # infile -t html -i ERROR.t2t
        "content": "",
        "cmdline": ["-t html -i ERROR.t2t"],
        "redir": ["> infile-not-found-2.out"],
        "extra": ["notarget", "noinfile"],
    },
    {
        "name": "target-invalid",  # t2t --target ERROR
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--target ERROR"],
        "redir": ["> target-invalid.out"],
        "extra": ["notarget"],
    },
    {
        "name": "t-invalid",  # t2t --t ERROR
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-t ERROR"],
        "redir": ["> t-invalid.out"],
        "extra": ["notarget"],
    },
    {
        "name": "invalid-short-1",  # t2t -z
        "content": "",
        "cmdline": ["-z"],
        "redir": ["> invalid-short-1.out"],
        "extra": ["notarget", "noinfile"],
    },
    {
        "name": "invalid-long-1",  # t2t --zzzz
        "content": "",
        "cmdline": ["--zzzz"],
        "redir": ["> invalid-long-1.out"],
        "extra": ["notarget", "noinfile"],
    },
    {
        "name": "invalid-short-2",  # t2t -z infile.t2t
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-z"],
        "redir": ["> invalid-short-2.out"],
        "extra": ["notarget"],
    },
    {
        "name": "invalid-long-2",  # t2t --zzzz infile.t2t
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--zzzz"],
        "redir": ["> invalid-long-2.out"],
        "extra": ["notarget"],
    },
    {
        "name": "version",  # t2t --version
        "content": "",
        "cmdline": ["--version"],
        "redir": ["> version.out"],
        "extra": ["notarget", "noinfile"],
    },
    {
        "name": "V",  # t2t -V
        "content": "",
        "cmdline": ["-V"],
        "redir": ["> V.out"],
        "extra": ["notarget", "noinfile"],
    },
    {
        "name": "verbose-1",  # t2t --verbose infile.t2t
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet --verbose -o-"],
        "redir": ["> verbose-1.txt"],
    },
    {
        "name": "verbose-2",  # t2t --verbose*2 infile.t2t
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet --verbose --verbose -o-"],
        "redir": ["> verbose-2.txt"],
    },
    {
        "name": "verbose-3",  # t2t --verbose*3 infile.t2t
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet --verbose --verbose --verbose -o-"],
        "redir": ["> verbose-3.txt"],
    },
    {
        "name": "verbose-4",  # t2t --verbose*4 infile.t2t (same -vvv)
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet --verbose --verbose --verbose --verbose -o-"],
        "redir": ["> verbose-4.txt"],
    },
    {
        "name": "verbose-5",  # t2t --verbose*5 infile.t2t (same -vvv)
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet --verbose --verbose --verbose --verbose --verbose -o-"],
        "redir": ["> verbose-5.txt"],
    },
    {
        "name": "v-1",  # t2t -v infile.t2t
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -v -o-"],
        "redir": ["> v-1.txt"],
    },
    {
        "name": "v-2",  # t2t -v -v infile.t2t
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -v -v -o-"],
        "redir": ["> v-2.txt"],
    },
    {
        "name": "v-3",  # t2t -v -v -v infile.t2t
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -v -v -v -o-"],
        "redir": ["> v-3.txt"],
    },
    {
        "name": "v-4",  # t2t -v -v -v -v infile.t2t (same -vvv)
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -v -v -v -v -o-"],
        "redir": ["> v-4.txt"],
    },
    {
        "name": "v-5",  # t2t -v -v -v -v -v infile.t2t (same -vvv)
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -v -v -v -v -v -o-"],
        "redir": ["> v-5.txt"],
    },
    {
        "name": "vv",  # t2t -vv infile.t2t
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -vv -o-"],
        "redir": ["> vv.txt"],
    },
    {
        "name": "vvv",  # t2t -vvv infile.t2t
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -vvv -o-"],
        "redir": ["> vvv.txt"],
    },
    {
        "name": "vvvv",  # t2t -vvvv infile.t2t (same -vvv)
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -vvvv -o-"],
        "redir": ["> vvvv.txt"],
    },
    {
        "name": "vvvvv",  # t2t -vvvvv infile.t2t (same -vvv)
        "target": "txt",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -vvvvv -o-"],
        "redir": ["> vvvvv.txt"],
    },
    {
        "name": "q-verbose",  # t2t -q -vvv infile.t2t
        "target": "txt",
        "content": FULL_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet -q -vvv -o-"],
        "redir": ["> q-verbose.txt"],
    },
    {
        "name": "quiet-verbose",  # t2t --quiet -vvv infile.t2t
        "target": "txt",
        "content": FULL_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-quiet --quiet -vvv -o-"],
        "redir": ["> quiet-verbose.txt"],
    },
    {
        "name": "target",  # t2t --target html infile.t2t
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --target html"],
        "extra": ["notarget"],
    },
    {
        "name": "t",  # t2t -t html infile.t2t
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H -t html"],
        "extra": ["notarget"],
    },
    {
        "name": "infile",
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --infile"],
    },
    {
        "name": "no-infile-1",  # useless
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --no-infile"],
    },
    {
        "name": "no-infile-2",  # turning OFF
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --infile fake --no-infile"],
    },
    {
        "name": "no-infile-3",  # turning OFF multiple
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --infile fake1 --infile fake2 --no-infile"],
    },
    {
        "name": "i",
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H -i"],
    },
    {
        "name": "outfile-1",  # same name as default
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --outfile outfile-1.html"],
    },
    {
        "name": "outfile-2",  # different name
        "target": "foo",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H -t html --outfile outfile-2.foo"],
        "extra": ["notarget"],
    },
    {
        "name": "no-outfile-1",  # useless
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --no-outfile"],
    },
    {
        "name": "no-outfile-2",  # turning OFF
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --outfile fake --no-outfile"],
    },
    {
        "name": "o",  # same name as default
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H -o o.html"],
    },
    {
        "name": "enum-title-1",
        "target": "html",
        "content": EMPTY_HEADER + TITLED_BODY,
        "cmdline": ["-H --enum-title"],
    },
    {
        "name": "enum-title-2",  # with --toc
        "target": "html",
        "content": EMPTY_HEADER + TITLED_BODY,
        "cmdline": ["-H --toc --enum-title"],
    },
    {
        "name": "enum-title-3",  # no title to enumerate
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --enum-title"],
    },
    {
        "name": "no-enum-title-1",  # useless
        "target": "html",
        "content": EMPTY_HEADER + TITLED_BODY,
        "cmdline": ["-H --no-enum-title"],
    },
    {
        "name": "no-enum-title-2",  # turning OFF
        "target": "html",
        "content": EMPTY_HEADER + TITLED_BODY,
        "cmdline": ["-H --enum-title --no-enum-title"],
    },
    {
        "name": "n",
        "target": "html",
        "content": EMPTY_HEADER + TITLED_BODY,
        "cmdline": ["-H -n"],
    },
    {
        "name": "toc-1",
        "target": "html",
        "content": EMPTY_HEADER + TITLED_BODY,
        "cmdline": ["-H --toc"],
    },
    {
        "name": "toc-2",  # empty toc (no title)
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H --toc"],
    },
    {
        "name": "toc-3",  # empty body
        "target": "html",
        "content": EMPTY_HEADER,
        "cmdline": ["-H --toc"],
    },
    {
        "name": "no-toc-1",  # useless
        "target": "html",
        "content": EMPTY_HEADER + TITLED_BODY,
        "cmdline": ["-H --no-toc"],
    },
    {
        "name": "no-toc-2",  # turning OFF
        "target": "html",
        "content": EMPTY_HEADER + TITLED_BODY,
        "cmdline": ["-H --toc --no-toc"],
    },
    {
        "name": "headers-1",  # useless
        "target": "html",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--headers"],
    },
    {
        "name": "headers-2",  # turning OFF --no-headers
        "target": "html",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--no-headers --headers"],
    },
    {
        "name": "no-headers",
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["--no-headers"],
    },
    {
        "name": "H",
        "target": "html",
        "content": EMPTY_HEADER + SIMPLE_BODY,
        "cmdline": ["-H"],
    },
    {
        "name": "style-1",
        "target": "html",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--style", lib.CSS_FILE],
    },
    {
        "name": "style-2",  # multiple declaration
        "target": "html",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--style other.css --style", lib.CSS_FILE],
    },
    {
        "name": "style-3",  # LaTeX package
        "target": "tex",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--style mypackage"],
    },
    {
        "name": "style-4",  # LaTeX multiple package
        "target": "tex",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--style mypackage,otherpackage,another"],
    },
    {
        "name": "style-5",  # LaTeX module no .sty
        "target": "tex",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--style foo.sty --style bar.STY --style baz"],
    },
    {
        "name": "no-style-1",  # useless
        "target": "html",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--no-style"],
    },
    {
        "name": "no-style-2",  # turning OFF
        "target": "html",
        "content": EMPTY_HEADER + VERSION_GOTCHA + SIMPLE_BODY,
        "cmdline": ["--style fake.css --no-style"],
    },
    {
        "name": "no-targets",
        "content": EMPTY_HEADER + CONFIG_FILE_TXT + SIMPLE_BODY,
        "cmdline": ["-H -o- --targets --no-targets"],
        "redir": ["> no-targets.out"],
        "extra": ["notarget"],
    },
]


def run():
    for test in tests:
        infile = test["name"] + ".t2t"
        outfile = test["name"] + "." + (test.get("target") or "out")
        extra = test.get("extra") or []
        cmdline = test["cmdline"]
        if "noinfile" not in extra:
            cmdline = test["cmdline"] + [infile]
        if lib.initTest(test["name"], infile, outfile):
            # create the extra files (if needed for this test)
            if "config" in extra:
                lib.WriteFile(lib.CONFIG_FILE, CONFIG_FILE_TXT)
            if "css" in extra:
                lib.WriteFile(lib.CSS_FILE, CSS_FILE_TXT)
            # may I add the -t target automatically?
            if "notarget" not in extra:
                cmdline = ["-t", test["target"]] + cmdline
            # may I redirect the output to a file?
            if test.get("redir"):
                cmdline.extend(test["redir"])
            # always catch the error output
            cmdline.append("2>&1")
            # create the source file
            lib.WriteFile(infile, test["content"])
            # convert and check results
            lib.test(cmdline, outfile)
            # remove the trash
            os.remove(infile)
            if os.path.isfile(lib.CSS_FILE):
                os.remove(lib.CSS_FILE)
            if os.path.isfile(lib.CONFIG_FILE):
                os.remove(lib.CONFIG_FILE)
