/*
 * Copyright (C) 2014-2024 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <QAbstractItemView>
#include <QEvent>
#include <QKeyEvent>
#include <QTableWidget>

#include "src/views/table_space_selection_filter.h"

TableSpaceSelectionFilter::TableSpaceSelectionFilter(int checkCol,
    QObject *parent)
    : QObject(parent),
    m_checkCol(checkCol)
{
}

TableSpaceSelectionFilter::~TableSpaceSelectionFilter(void)
{
}

/*!
 * @brief Performs space bar selection on a QListView or QTableView.
 *
 * @param[in,out] av Non-null pointer to abstract item view.
 * @param[in]     ke Non-null pointer to key event.
 * @param[in]     checkCol Checkable column.
 * @return True when filter applied.
 */
static
bool viewFilter(QAbstractItemView *av, const QKeyEvent *ke, int checkCol)
{
	if (Q_UNLIKELY(Q_NULLPTR == av)) {
		Q_ASSERT(0);
		return false;
	}
	if (Q_UNLIKELY(Q_NULLPTR == ke)) {
		Q_ASSERT(0);
		return false;
	}

	switch (ke->key()) {
	case Qt::Key_Space:
		{
			QModelIndexList checkColIdxs(
			    av->selectionModel()->selectedRows(checkCol));

			if (checkColIdxs.isEmpty()) {
				return false;
			}

			bool checked = false;
			for (const QModelIndex &idx : checkColIdxs) {
				/* Get first user checkable value. */
				if (idx.flags() & Qt::ItemIsUserCheckable) {
					checked = idx.data(Qt::CheckStateRole) == Qt::Checked;
					break;
				}
			}

			for (const QModelIndex &idx : checkColIdxs) {
				if (idx.flags() & Qt::ItemIsUserCheckable) {
					av->model()->setData(idx,
					    checked ? Qt::Unchecked : Qt::Checked,
					    Qt::CheckStateRole);
				}
			}
		}
		return true;
		break;
	default:
		break;
	}

	return false;
}

/*!
 * @brief Performs space bar selection on a QTableWidget.
 *
 * @param[in,out] tw Non-null pointer to table widget.
 * @param[in]     ke Non-null pointer to key event.
 * @param[in]     checkCol Checkable column.
 * @return True when filter applied.
 */
static
bool widgetFilter(QTableWidget *tw, const QKeyEvent *ke, int checkCol)
{
	if (Q_UNLIKELY(Q_NULLPTR == tw)) {
		Q_ASSERT(0);
		return false;
	}
	if (Q_UNLIKELY(Q_NULLPTR == ke)) {
		Q_ASSERT(0);
		return false;
	}

	switch (ke->key()) {
	case Qt::Key_Space:
		{
			QList<QTableWidgetItem *> selectedItems =
			    tw->selectedItems();

			if (selectedItems.isEmpty()) {
				return false;
			}

			QTableWidgetItem *frstItem = tw->item(
			    selectedItems.first()->row(), checkCol);

			if (Q_UNLIKELY(Q_NULLPTR == frstItem)) {
				Q_ASSERT(0);
				return false;
			}

			bool checked = frstItem->checkState() == Qt::Checked;

			for (QTableWidgetItem *const item : selectedItems) {
				if (Q_UNLIKELY(Q_NULLPTR == item)) {
					continue;
				}
				QTableWidgetItem *checkItem =
				    tw->item(item->row(), checkCol);

				if (Q_UNLIKELY(Q_NULLPTR == checkItem)) {
					Q_ASSERT(0);
					continue;
				}
				checkItem->setCheckState(
				    checked ? Qt::Unchecked : Qt::Checked);
			}
		}
		return true;
		break;
	default:
		break;
	}

	return false;
}

bool TableSpaceSelectionFilter::eventFilter(QObject *object, QEvent *event)
{
	const QKeyEvent *ke = Q_NULLPTR;
	QAbstractItemView *av = Q_NULLPTR;
	QTableWidget *tw = Q_NULLPTR;
	if (event->type() == QEvent::KeyPress) {
		ke = (QKeyEvent *)event;
	}

	if (Q_NULLPTR != ke) {
		av = qobject_cast<QAbstractItemView *>(object);
		tw = qobject_cast<QTableWidget *>(object);
	}

	if (Q_NULLPTR != av) {
		if (viewFilter(av, ke, m_checkCol)) {
			return true;
		}
	} else if (Q_NULLPTR != tw) {
		if (widgetFilter(tw, ke, m_checkCol)) {
			return true;
		}
	}

	return QObject::eventFilter(object, event);
}
