
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: disc.c 2581 2007-07-22 13:49:23Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <stddef.h>
#include <stdio.h>
#include <mntent.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef HAVE_HAL
#include <libhal.h>
#include <libhal-storage.h>
#endif

#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#endif

#include "disc.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "meta_info.h"
#include "mutex.h"
#include "oxine.h"
#include "utils.h"

extern oxine_t *oxine;

#ifdef HAVE_HAL
static char *hal_drive_get_title (LibHalDrive * drive);
#endif


char *
volume_get_mountpoint (const char *device)
{
    /* First we have a look to see if the device is a softlink. If it is we
     * resolve it and recursively call ourself again. */
    char *target = resolve_softlink (device);
    if (strcmp (target, device) != 0) {
        char *mountpoint = volume_get_mountpoint (target);
        ho_free (target);
        return mountpoint;
    }
    else {
        ho_free (target);
    }

    char *mountpoint = NULL;

#ifdef HAVE_HAL
    LibHalContext *context = hal_get_context ();
    if (context) {
        LibHalVolume *volume = libhal_volume_from_device_file (context,
                                                               device);
        if (volume) {
            const char *mp = libhal_volume_get_mount_point (volume);
            mountpoint = mp ? ho_strdup (mp) : NULL;
            libhal_volume_free (volume);
        }
    }
#endif

    if (!mountpoint) {
        FILE *fstab = fopen ("/etc/fstab", "r");
        if (!fstab)
            return NULL;

        struct mntent *entry = getmntent (fstab);
        while (entry) {
            if (strcmp (device, entry->mnt_fsname) == 0) {
                mountpoint = ho_strdup (entry->mnt_dir);
                break;
            }
            entry = getmntent (fstab);
        }
        endmntent (fstab);
    }

    return mountpoint;
}


volume_type_t
volume_get_type (const char *device)
{
    /* First we have a look to see if the device is a softlink. If it is we
     * resolve it and recursively call ourself again. */
    char *target = resolve_softlink (device);
    if (strcmp (target, device) != 0) {
        volume_type_t type = volume_get_type (target);
        ho_free (target);
        return type;
    }
    else {
        ho_free (target);
    }

    volume_type_t type = VOLUME_TYPE_UNKNOWN;

#ifdef HAVE_HAL
    LibHalContext *context = hal_get_context ();
    char *udi = hal_volume_get_udi_from_device (device);

    if (!udi || !context)
        return type;

    if (libhal_device_get_property_bool
        (context, udi, "volume.disc.is_blank", NULL)) {
        type = VOLUME_TYPE_BLANK;
    }
    else if (libhal_device_get_property_bool
             (context, udi, "volume.disc.has_audio", NULL)) {
        type = VOLUME_TYPE_AUDIO_CD;
    }
    else if (libhal_device_get_property_bool
             (context, udi, "volume.disc.is_vcd", NULL)) {
        type = VOLUME_TYPE_VIDEO_CD;
    }
    else if (libhal_device_get_property_bool
             (context, udi, "volume.disc.is_svcd", NULL)) {
        type = VOLUME_TYPE_VIDEO_CD;
    }
    else if (libhal_device_get_property_bool
             (context, udi, "volume.disc.is_videodvd", NULL)) {
        type = VOLUME_TYPE_VIDEO_DVD;
    }
    else if (libhal_device_get_property_bool
             (context, udi, "volume.is_disc", NULL)) {
        type = VOLUME_TYPE_CDROM;
    }
    else if (hal_volume_is_mountable (device)) {
        type = VOLUME_TYPE_MOUNTABLE;
    }

    ho_free (udi);
#endif /* HAVE_HAL */

    return type;
}


char *
volume_get_title (const char *device)
{
    /* First we have a look to see if the device is a softlink. If it is we
     * resolve it and recursively call ourself again. */
    char *target = resolve_softlink (device);
    if (strcmp (target, device) != 0) {
        char *title = volume_get_title (target);
        ho_free (target);
        return title;
    }
    else {
        ho_free (target);
    }

    volume_type_t vol_type = volume_get_type (device);
    char *volume_title = NULL;
    char *drive_title = NULL;
    switch (vol_type) {
    case VOLUME_TYPE_AUDIO_CD:
        drive_title = ho_strdup (_("Audio CD"));
        {
            char *mrl = ho_strdup_printf ("cdda:/%s/1", device);
            char *album = meta_info_get (META_INFO_ALBUM, mrl);
            char *artist = meta_info_get (META_INFO_ARTIST, mrl);

            if (album && artist) {
                volume_title = ho_strdup_printf ("%s - %s", artist, album);
            }

            ho_free (mrl);
            ho_free (album);
            ho_free (artist);
        }
        break;
    case VOLUME_TYPE_VIDEO_CD:
        drive_title = ho_strdup (_("Video CD"));
        {
            char *mrl = ho_strdup_printf ("vcd:/%s", device);
            volume_title = meta_info_get (META_INFO_TITLE, mrl);

            ho_free (mrl);
        }
        break;
    case VOLUME_TYPE_VIDEO_DVD:
        drive_title = ho_strdup (_("Video DVD"));
        {
            char *mrl = ho_strdup_printf ("dvd:/%s", device);
            volume_title = meta_info_get (META_INFO_TITLE, mrl);

            ho_free (mrl);
        }
        break;
    case VOLUME_TYPE_MOUNTABLE:
    case VOLUME_TYPE_CDROM:
        break;
    case VOLUME_TYPE_BLANK:
        volume_title = ho_strdup (_("Blank Disc"));
        break;
    default:
        break;
    }

#ifdef HAVE_HAL
    LibHalVolume *volume = NULL;
    LibHalDrive *drive = NULL;
    LibHalContext *context = hal_get_context ();
    if (!context) {
        error (_("Failed to connect to HAL daemon!"));
        goto hal_out;
    }

    volume = libhal_volume_from_device_file (context, device);
    if (!volume) {
        error (_("Failed to get volume from device '%s'!"), device);
        goto hal_out;
    }
    const char *udi = libhal_volume_get_storage_device_udi (volume);
    const char *label = libhal_volume_get_label (volume);
    if (!volume_title && label) {
        volume_title = ho_strdup (label);
    }

    drive = libhal_drive_from_udi (context, udi);
    if (!drive) {
        error (_("Failed to get drive from device '%s'!"), device);
        goto hal_out;
    }
    if (!drive_title) {
        drive_title = hal_drive_get_title (drive);
    }

  hal_out:
    if (volume) {
        libhal_volume_free (volume);
    }
    if (drive) {
        libhal_drive_free (drive);
    }
#endif /* HAVE_HAL */

    if (!drive_title) {
        drive_title = ho_strdup (_("Drive"));
    }
    if (!volume_title) {
        volume_title = ho_strdup (_("Unknown Disc"));
    }

    char *title = ho_strdup_printf ("[%s: %s]", drive_title, volume_title);

    ho_free (drive_title);
    ho_free (volume_title);

    return title;
}


bool
volume_is_mounted (const char *device)
{
#ifdef HAVE_HAL
    return hal_volume_is_mounted (device);
#else
    FILE *f = fopen ("/proc/mounts", "r");
    if (!f) {
        error (_("Could not open '%s': %s!"),
               "/proc/mounts", strerror (errno));
        return false;
    }

    bool mounted = false;
    char line[256];
    while (fgets (line, 256, f) != NULL) {
        if (strstr (line, device) != NULL) {
            mounted = true;
            break;
        }
    }
    fclose (f);

    return mounted;
#endif /* !HAVE_HAL */
}


bool
volume_is_present (const char *device)
{
#ifdef HAVE_LINUX_CDROM_H
    int fd = open (device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not open '%s': %s!"), device, strerror (errno));
        return false;
    }

    bool present = false;
    int drive_status = ioctl (fd, CDROM_DRIVE_STATUS);
    switch (drive_status) {
    case CDS_NO_INFO:
        debug ("drive reports: no info");
        break;
    case CDS_NO_DISC:
        debug ("drive reports: no disc");
        break;
    case CDS_TRAY_OPEN:
        debug ("drive reports: tray open");
        break;
    case CDS_DRIVE_NOT_READY:
        debug ("drive reports: drive not ready");
        break;
    case CDS_DISC_OK:
        debug ("drive reports: disc ok");
        present = true;
        break;
    default:
        debug ("drive reports: unknown value");
        break;
    }
    close (fd);

    return present;
#else
    return false;
#endif
}


bool
volume_mount (const char *device)
{
    if (volume_is_mounted (device))
        return true;

    char *command = NULL;

#ifdef BIN_PMOUNT
    if (!command) {
        command = ho_strdup_printf ("%s %s", BIN_PMOUNT, device);
    }
#endif
#ifdef BIN_MOUNT
    if (!command) {
        command = ho_strdup_printf ("%s %s", BIN_MOUNT, device);
    }
#endif

    bool res = false;
    if (command) {
        if (!execute_shell (command, 0)) {
            error (_("Could not mount '%s': %s!"), device, _("Unknown"));
        }
        else {
            res = true;
        }
        ho_free (command);
    }

    return res;
}


bool
volume_umount (const char *device)
{
    if (!volume_is_mounted (device))
        return true;

    char *command = NULL;

#ifdef BIN_PMOUNT
    if (!command) {
        command = ho_strdup_printf ("%s %s", BIN_PUMOUNT, device);
    }
#endif
#ifdef BIN_UMOUNT
    if (!command) {
        command = ho_strdup_printf ("%s %s", BIN_UMOUNT, device);
    }
#endif

    bool res = false;
    if (command) {
        if (!execute_shell (command, 0)) {
            error (_("Could not unmount '%s': %s!"), device, _("Unknown"));
        }
        else {
            res = true;
        }
        ho_free (command);
    }

    return res;
}


bool
drive_is_ejected (const char *device)
{
#ifdef HAVE_LINUX_CDROM_H
    int fd = open (device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not open '%s': %s!"), device, strerror (errno));
        return false;
    }

    /** 
     * @bug This does not always work. Apparently ther is a bug in the SCSI
     * code of the 2.6.10 kernels that will return CDS_TRAY_OPEN if there is
     * no disc in the drive. We'll just wait and hope this gets fixed soon.
     */
    bool ejected = false;
    int drive_status = ioctl (fd, CDROM_DRIVE_STATUS);
    switch (drive_status) {
    case CDS_NO_INFO:
        debug ("drive reports: no info");
        break;
    case CDS_NO_DISC:
        debug ("drive reports: no disc");
        break;
    case CDS_TRAY_OPEN:
        debug ("drive reports: tray open");
        ejected = true;
        break;
    case CDS_DRIVE_NOT_READY:
        debug ("drive reports: drive not ready");
        break;
    case CDS_DISC_OK:
        debug ("drive reports: disc ok");
        break;
    default:
        debug ("drive reports: unknown value");
        break;
    }
    close (fd);

    return ejected;
#else
    return false;
#endif
}


bool
drive_inject (const char *device)
{
#if defined BIN_EJECT
    bool res = true;
    char *command = ho_strdup_printf ("%s -t %s", BIN_EJECT, device);

    if (!execute_shell (command, 0)) {
        error (_("Could not inject '%s': %s!"), device, _("Unknown"));
        res = false;
    }
    ho_free (command);

    return res;
#elif defined HAVE_LINUX_CDROM_H
    int fd = open (device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not inject '%s': %s!"), device, strerror (errno));
        return false;
    }

    if (ioctl (fd, CDROMCLOSETRAY) < 0) {
        error (_("Could not inject '%s': %s!"), device, strerror (errno));
        close (fd);
        return false;
    }

    close (fd);
    return true;
#else
    return false;
#endif
}


bool
drive_eject (const char *device)
{
#if defined BIN_EJECT
    bool res = true;
    char *command = ho_strdup_printf ("%s %s", BIN_EJECT, device);

    if (!execute_shell (command, 0)) {
        error (_("Could not eject '%s': %s!"), device, strerror (errno));
        res = false;
    }
    ho_free (command);

    return res;
#elif defined HAVE_LINUX_CDROM_H
    int fd = open (device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not eject '%s': %s!"), device, strerror (errno));
        return false;
    }

    if (ioctl (fd, CDROMEJECT) < 0) {
        error (_("Could not eject '%s': %s!"), device, strerror (errno));
        close (fd);
        return false;
    }

    close (fd);
    return true;
#else
    return false;
#endif
}


#ifdef HAVE_HAL
static char *
hal_drive_get_title (LibHalDrive * drive)
{
    LibHalDriveType type = libhal_drive_get_type (drive);

    char *title = NULL;
    switch (type) {
    case LIBHAL_DRIVE_TYPE_CDROM:
        title = ho_strdup (_("CDROM Drive"));
        break;
    case LIBHAL_DRIVE_TYPE_COMPACT_FLASH:
        title = ho_strdup (_("Compact Flash Drive"));
        break;
    case LIBHAL_DRIVE_TYPE_MEMORY_STICK:
        title = ho_strdup (_("Memory Stick Drive"));
        break;
    case LIBHAL_DRIVE_TYPE_SMART_MEDIA:
        title = ho_strdup (_("Smart Media Drive"));
        break;
    case LIBHAL_DRIVE_TYPE_SD_MMC:
        title = ho_strdup (_("SD/MMC Drive"));
        break;
    case LIBHAL_DRIVE_TYPE_CAMERA:
        {
            const char *vendor = libhal_drive_get_vendor (drive);
            const char *model = libhal_drive_get_model (drive);
            title = ho_strdup_printf ("%s%s%s%s%s",
                                      vendor != NULL ? vendor : "",
                                      vendor != NULL ? " " : "",
                                      model != NULL ? model : "",
                                      model != NULL ? " " : "",
                                      _("Digital Camera"));
        }
        break;
    case LIBHAL_DRIVE_TYPE_PORTABLE_AUDIO_PLAYER:
        {
            const char *vendor = libhal_drive_get_vendor (drive);
            const char *model = libhal_drive_get_model (drive);
            title = ho_strdup_printf ("%s%s%s%s%s",
                                      vendor != NULL ? vendor : "",
                                      vendor != NULL ? " " : "",
                                      model != NULL ? model : "",
                                      model != NULL ? " " : "",
                                      _("Music Player"));
        }
        break;
    case LIBHAL_DRIVE_TYPE_ZIP:
        title = ho_strdup (_("Zip Drive"));
        break;
    case LIBHAL_DRIVE_TYPE_JAZ:
        title = ho_strdup (_("Jaz Drive"));
        break;
    case LIBHAL_DRIVE_TYPE_FLASHKEY:
        title = ho_strdup (_("Pen Drive"));
        break;
    default:
        break;
    }

#if 0
    if (!title) {
        const char *vendor = libhal_drive_get_vendor (drive);
        const char *model = libhal_drive_get_model (drive);

        if (vendor == NULL || strlen (vendor) == 0) {
            if (model != NULL && strlen (model) > 0)
                title = ho_strdup (model);
        }
        else {
            if (model == NULL || strlen (model) == 0)
                title = ho_strdup (vendor);
            else {
                title = ho_strdup_printf ("%s %s", vendor, model);
            }
        }
    }
#endif

    if (!title) {
        if (libhal_drive_is_hotpluggable (drive)) {
            title = ho_strdup (_("External Drive"));
        }
        else {
            title = ho_strdup (_("Drive"));
        }
    }

    return title;
}
#endif /* HAVE_HAL */


char *
drive_get_title (const char *device)
{
    char *title = NULL;

    /* First we have a look to see if the device is a softlink. If it is we
     * resolve it and recursively call ourself again. */
    char *target = resolve_softlink (device);
    if (strcmp (target, device) != 0) {
        title = drive_get_title (target);
        ho_free (target);
        return title;
    }
    else {
        ho_free (target);
    }

#ifdef HAVE_HAL
    LibHalContext *context = hal_get_context ();
    if (!context) {
        error (_("Failed to connect to HAL daemon!"));
        goto out_hal;
    }

    LibHalDrive *drive = libhal_drive_from_device_file (context, device);
    if (!drive) {
        error (_("Failed to get drive from device '%s'!"), device);
        goto out_hal;
    }

    title = hal_drive_get_title (drive);

    libhal_drive_free (drive);
  out_hal:
#endif /* HAVE_HAL */

    if (!title) {
        title = ho_strdup (_("Drive"));
    }

    return title;
}
