/*
 * Copyright (c) 1990, 1991, 1992, 1993, 1994, 1995, 1996, 1997, 1998, 2000, 2009, 2010, 2012, 2013, 2016, 2019, 2021, 2023
 * The Regents of the University of California. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef lint
static const char rcsid[] __attribute__((unused)) =
    "@(#) $Id: report.c 1537 2023-09-05 17:50:56Z leres $ (LBL)";
#endif

/*
 * report - arpwatch report generating routines
 */

#include <sys/param.h>
#include <sys/types.h>				/* concession to AIX */
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>

#if __STDC__
struct mbuf;
struct rtentry;
#endif
#include <net/if.h>

#include <netinet/in.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <paths.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#ifdef TIME_WITH_SYS_TIME
#include <time.h>
#endif
#include <unistd.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "arpwatch.h"
#include "dns.h"
#include "ec.h"
#include "report.h"
#include "setsignal.h"
#include "util.h"

#define PLURAL(n) ((n) == 1 || (n) == -1 ? "" : "s")

static int cdepth;	/* number of outstanding children */

static char *fmtdate(time_t);
static char *fmtdelta(time_t);
void reaper(int);
static int32_t gmt2local(void);

static char *
fmtdelta(time_t t)
{
	char *cp;
	int minus;
	static char buf[132];

	minus = 0;
	if (t < 0) {
		t = -t;
		++minus;
	}
	if (t < 60) {
		cp = "second";
	} else if (t < 60 * 60) {
		t /= 60;
		cp = "minute";
	} else if (t < 24 * 60 * 60) {
		t /= (60 * 60);
		cp = "hour";
	} else {
		t /= (24 * 60 * 60);
		cp = "day";
	}
	if (minus)
		t = -t;
	(void)snprintf(buf, sizeof(buf), "%u %s%s",
	    (u_int32_t)t, cp, PLURAL(t));
	return(buf);
}

static char *dow[7] = {
	"Sunday",
	"Monday",
	"Tuesday",
	"Wednesday",
	"Thursday",
	"Friday",
	"Saturday"
};

static char *moy[12] = {
	"January",
	"February",
	"March",
	"April",
	"May",
	"June",
	"July",
	"August",
	"September",
	"October",
	"November",
	"December"
};

#define DOW(d) ((d) < 0 || (d) >= 7 ? "?" : dow[d])
#define MOY(m) ((m) < 0 || (m) >= 12 ? "?" : moy[(m)])

static char *
fmtdate(time_t t)
{
	struct tm *tm;
	int32_t mw;
	char ch;
	static int init = 0;
	static char zone[32], buf[132];

	if (t == 0)
		return("<no date>");

	if (!init) {
		mw = gmt2local() / 60;
		if (mw < 0) {
			ch = '-';
			mw = -mw;
		} else {
			ch = '+';
		}
		(void)snprintf(zone, sizeof(zone), "%c%02d%02d",
		    ch, mw / 60, mw % 60);
		++init;
	}

	tm = localtime(&t);
	(void)snprintf(buf, sizeof(buf), "%s, %s %d, %d %d:%02d:%02d %s",
	    DOW(tm->tm_wday),
	    MOY(tm->tm_mon),
	    tm->tm_mday,
	    tm->tm_year + 1900,
	    tm->tm_hour,
	    tm->tm_min,
	    tm->tm_sec,
	    zone);
	return(buf);
}

/*
 * Returns the difference between gmt and local time in seconds.
 * Use gmtime() and localtime() to keep things simple.
 */
static int32_t
gmt2local(void)
{
	int dt, dir;
	struct tm *gmt, *loc;
	time_t t;
	struct tm sgmt;

	t = time(NULL);
	gmt = &sgmt;
	*gmt = *gmtime(&t);
	loc = localtime(&t);
	dt = (loc->tm_hour - gmt->tm_hour) * 60 * 60 +
	    (loc->tm_min - gmt->tm_min) * 60;

	/*
	 * If the year or julian day is different, we span 00:00 GMT
	 * and must add or subtract a day. Check the year first to
	 * avoid problems when the julian day wraps.
	 */
	dir = loc->tm_year - gmt->tm_year;
	if (dir == 0)
		dir = loc->tm_yday - gmt->tm_yday;
	dt += dir * 24 * 60 * 60;

	return (dt);
}

void
reaper(int signo)
{
	pid_t pid;
	DECLWAITSTATUS status;

	for (;;) {
		pid = waitpid((pid_t)0, &status, WNOHANG);
		if ((int)pid < 0) {
			/* ptrace foo */
			if (errno == EINTR)
				continue;
			/* ECHILD means no one left */
			if (errno != ECHILD)
				lg(LOG_ERR, "reaper: %s", strerror(errno));
			break;
		}
		/* Already got everyone who was done */
		if (pid == 0)
			break;
		--cdepth;
		if (WEXITSTATUS(status))
			lg(LOG_DEBUG, "reaper: pid %d, exit status %d",
			    pid, WEXITSTATUS(status));
	}
}

void
report(const char *title, u_int32_t a, const u_char *e1, const u_char *e2,
    const time_t *t1p, const time_t *t2p)
{
	char *cp, *hn;
	int fd, pid;
	FILE *f;
	char tempfile[64], cpu[64], os[64];
	char *fmt = "%20s: %s\n";
	char *sendmail = PATH_SENDMAIL;
	char *unknown = "<unknown>";
	char buf[132];
	static int init = 0;

	/* No report until we're initialized */
	if (initializing)
		return;

	/* No mail for 0.0.0.0 if -z */
	if (zeroflag && a == 0) {
		dosyslog(LOG_NOTICE, title, a, e1, e2);
		return;
	}

	if (debug) {
		if (debug > 1) {
			dosyslog(LOG_NOTICE, title, a, e1, e2);
			return;
		}
		f = stdout;
		(void)putc('\n', f);
	} else {
		/* Setup child reaper if we haven't already */
		if (!init) {
			(void)setsignal(SIGCHLD, reaper);
			++init;
		}
		while (cdepth >= 3) {
			lg(LOG_ERR, "report: pausing (cdepth %d)", cdepth);
			pause();
		}

		/* Syslog this event too */
		dosyslog(LOG_NOTICE, title, a, e1, e2);

		/* Update child depth */
		++cdepth;

		/* Fork off child to send mail */
		pid = fork();
		if (pid) {
			/* Parent */
			if (pid < 0)
				lg(LOG_ERR, "report: fork() 1: %s",
				    strerror(errno));
			return;
		}

		/* Child */
		closelog();
		(void)strncpy(tempfile, "/tmp/arpwatch.XXXXXX",
		    sizeof(tempfile));
		tempfile[sizeof(tempfile) - 1] = '\0';
		if ((fd = mkstemp(tempfile)) < 0) {
			lg(LOG_ERR, "mkstemp(%s) %s",
			    tempfile, strerror(errno));
			exit(1);
		}
		if ((f = fdopen(fd, "w+")) == NULL) {
			lg(LOG_ERR, "child fdopen(%s): %s",
			    tempfile, strerror(errno));
			exit(1);
		}
		/* Cheap delete-on-close */
		if (unlink(tempfile) < 0)
			lg(LOG_ERR, "unlink(%s): %s",
			    tempfile, strerror(errno));
	}

	(void)fprintf(f, "From: %s\n", watchee);
	(void)fprintf(f, "To: %s\n", watcher);
	hn = gethname(a);
	if (hn != NULL && !isdigit(*hn))
		(void)fprintf(f, "Subject: %s (%s)\n", title, hn);
	else {
		(void)fprintf(f, "Subject: %s\n", title);
		hn = unknown;
	}
	(void)putc('\n', f);
	if (hn != NULL)
		(void)fprintf(f, fmt, "hostname", hn);
	(void)fprintf(f, fmt, "ip address", intoa(a));
	(void)fprintf(f, fmt, "ethernet address", e2str(e1));
	if ((cp = ec_find(e1)) == NULL)
		cp = unknown;
	(void)fprintf(f, fmt, "ethernet vendor", cp);
	if (hn != unknown && gethinfo(hn, cpu, sizeof(cpu), os, sizeof(os))) {
		(void)snprintf(buf, sizeof(buf), "%s %s", cpu, os);
		(void)fprintf(f, fmt, "dns cpu & os", buf);
	}
	if (e2) {
		(void)fprintf(f, fmt, "old ethernet address", e2str(e2));
		if ((cp = ec_find(e2)) == NULL)
			cp = unknown;
		(void)fprintf(f, fmt, "old ethernet vendor", cp);
	}
	if (t1p)
		(void)fprintf(f, fmt, "timestamp", fmtdate(*t1p));
	if (t2p)
		(void)fprintf(f, fmt, "previous timestamp", fmtdate(*t2p));
	if (t1p && t2p && *t1p && *t2p)
		(void)fprintf(f, fmt, "delta", fmtdelta(*t1p - *t2p));

	if (debug) {
		fflush(f);
		return;
	}

	(void)rewind(f);
	if (dup2(fileno(f), fileno(stdin)) < 0) {
		lg(LOG_ERR, "dup2: %s", strerror(errno));
		exit(1);
	}
	/* XXX Need to freopen()? */

	/*
	 * Open /dev/null as stdout and stderr so that sendmail 8.12.1 (and
	 * above ?) won't complain about missing file descriptors.
	 */
	fd = open(_PATH_DEVNULL, O_RDWR);
	if (fd < 0) {
		lg(LOG_ERR, "Cannot open %s: %s",
		    _PATH_DEVNULL, strerror(errno));
		exit(1);
	}
	if (dup2(fd, STDOUT_FILENO) < 0) {
		lg(LOG_ERR, "Cannot dup2 %s to stdout: %s",
		    _PATH_DEVNULL, strerror(errno));
		exit(1);
	}
	if (dup2(fd, STDERR_FILENO) < 0) {
		lg(LOG_ERR, "Cannot dup2 %s to stderr: %s",
		    _PATH_DEVNULL, strerror(errno));
		exit(1);
	}
	close(fd);

	/* Always Deliver interactively (pause when child depth gets large) */
	execl(sendmail, "sendmail", "-odi", watcher, NULL);
	lg(LOG_ERR, "execl: %s: %s", sendmail, strerror(errno));
	exit(1);
}
