#include <config.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <signal.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef __BEOS__
#include <kernel/OS.h>
#endif
#include "dial.h"
#include "buffer.h"
#include "sock.h"
#include "compat.h"

SOK *stream;

#ifdef __BEOS__
long keepalive_beos_entry(void *data);
#endif

int sock_close(void)
{
	return sclose(stream);
}

int recvline(char *buf, size_t maxlen)
{
	switch(srecv(stream, buf, maxlen)) {
	case SOK_FAILED:  fprintf(stderr, "Recv from socket failed\n");
			  return 1;
	case SOK_EOF:     fprintf(stderr, "Connection reset by server\n");
			  return 1;
	case SOK_TOOLONG: fprintf(stderr, "Garbage received from server\n");
			  /* Don't bother discarding, we don't want to continue
			   * if this happens. */
			  return 1;
	case SOK_TIMEOUT: fprintf(stderr, "recvline bug: timeout\n");
			  return 1; /* we have set an infinite timeout */
	}
	return 0; /* SOK_SUCCESS */
}

int sendline(const char *msg)
{
	switch(ssend(stream, msg)) {
	case SOK_FAILED:  fprintf(stderr, "Send to socket failed\n");
			  return 1;
	case SOK_TOOLONG: fprintf(stderr, "sendline bug: '%d' characters is "
					  "too long\n", strlen(msg));
			  return 1;
	case SOK_TIMEOUT: fprintf(stderr, "sendline bug: timeout\n");
			  return 1;
	}
	return 0; /* SOK_SUCCESS */
}
			  
int eventloop(char *buf, char *command)
{
	while (0==recvline(buf, MAXLINE)) {
		if (0!=strcmp(buf, "OKAY")) {
			puts(buf);
			fflush(stdout);
			if (*buf!='#') {
				return 1; /* server reported error */
			} if (0==strcmp(buf, "#Connected")) {
				if (command) {
					run(command);
					command = NULL; /* prevent DoS */
				}
			} else if (0==strcmp(buf, "#Connect failed")) {
				fprintf(stderr, "Server reported running "
						"commandon failed\n");
				return 1;
			}
		}
	}
	return 1;
}

int keepalive(volatile sig_atomic_t *pid, int keep_alive_secs)
{
	*pid = xfork();
	if (*pid==-1) {
		perror("Fork failed");
	       	return 1;
	} else if (*pid==0) {
		set_signals(SIG_DFL);
		unblock_all_signals();
		do {
			sleep(keep_alive_secs);
		} while (!sendline("ALIVE\n"));
		_exit(EXIT_FAILURE);
	}
	return 0;
}

#ifdef __BEOS__
/* BeOS doesn't inherit sockets over fork */
int keepalive_beos(int keep_alive_secs)
{
	thread_id tid;
	int *keeptmp = malloc(sizeof(keep_alive_secs));
	if (!keeptmp) {
		perror("Malloc failed");
		return 1;
	}
	*keeptmp = keep_alive_secs;
	tid = spawn_thread(keepalive_beos_entry, "keepalive", B_LOW_PRIORITY, 
			keeptmp);
	if (tid < B_NO_ERROR) {
		/* XXX: can use perror with spawn_thread? */
		perror("spawn_thread failed");
		return 1;
	}
	resume_thread(tid);
	return 0;
}

long keepalive_beos_entry(void *data)
{
	int keep_alive_secs = *(int *)data;
#if 0 /* XXX: Should we use this? */
	set_signals(SIG_DFL);
#endif
	do {
	       		sleep(keep_alive_secs);
       		} while (!sendline("ALIVE\n"));
	return 0;
}
#endif /* __BEOS__ */

int connect_to_host(int dest_port, char *server)
{
	struct sockaddr_in addr;
	struct hostent *host;
	int sok;

	if (NULL==(host = gethostbyname(server))) {
		fprintf(stderr, "Looking up '%s' failed\n", server);
		return 1;
	}
	addr.sin_family = AF_INET;
	addr.sin_port = htons(dest_port);
	addr.sin_addr.s_addr = inet_addr(inet_ntoa(*((struct in_addr *)
				host->h_addr)));
	memset(&addr.sin_zero, 0, sizeof(addr.sin_zero));
	if (-1==(sok = socket(AF_INET, SOCK_STREAM, 0))) {
		perror("Socket failed");
		return 1;
	}
	if (-1==connect(sok,(struct sockaddr *)&addr, sizeof(struct sockaddr))){
		perror("Connect failed");
		return 1;
	}
	stream = sdopen(sok);
	if (!stream) {
		fprintf(stderr, "Creating socket buffer failed\n");
		return 1;
	}
	return 0;
}
	
int recv_okay(char *buf)
{
	if (recvline(buf, MAXLINE))
		return RECV_ERR;
	if (0==strcmp(buf, "OKAY"))
		return RECV_STR_OKAY;
	puts(buf);
	if (*buf=='#')
		return RECV_STR_INFO;
	return RECV_STR_PROBLEM;
}
