/*
    Falling Block Game
    Copyright (C) 1999-2002 Jared Krinke <http://derajdezine.vze.com/>


    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    This application is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this distribution; if not, write to:
    Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA02111-1307  USA

    Jared Krinke

    Deraj DeZine
    http://derajdezine.vze.com/
*/

#include "fbg.h"
#include "glTGAImage.h"
#ifdef WIN32		// <GL/gl.h> and <GL/glu.h> require <windows.h> under Windows
#include <windows.h>
#endif
#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL/SDL.h>
#include <math.h>
#include <iostream>
#include <stdlib.h>
#include <physfs.h>

using namespace std;

#define FBG_TITLE "Falling Block Game"

bool fbgOpenGLRenderer::init(bool fs, int width, int height, int bpp) {
	// Initialize SDL/OpenGL
	if (SDL_Init(SDL_INIT_VIDEO|SDL_INIT_TIMER) < 0) {
		cerr << "Unable to initialize SDL: " << SDL_GetError() << endl;
		return false;
	}
	SDL_WM_SetCaption(FBG_TITLE, NULL);
	if (fs) SDL_ShowCursor(0);

	// User didn't specify bpp
	if (bpp < 0) bpp = SDL_GetVideoInfo()->vfmt->BitsPerPixel;

	// Set bits per channel
	const unsigned char bpc = bpp < 32 ? 4 : 5;
	SDL_GL_SetAttribute(SDL_GL_RED_SIZE, bpc);
	SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, bpc);
	SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, bpc);
	SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);

	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	if (SDL_SetVideoMode(width, height, bpp, (fs ? SDL_FULLSCREEN : 0) | SDL_OPENGL) == NULL) {
		cerr << "Unable to Create OpenGL Screen: " << SDL_GetError() << endl;
		return false;
	}

	glViewport(0, 0, width, height);

	// Setup OpenGL Viewport
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, GLfloat(4)/GLfloat(3), 300.0f, 2000.0f);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	loadTextures();
	buildLists();

	// Setup OpenGL options (lighting, texturing, etc.)
	glEnable(GL_TEXTURE_2D);

	glShadeModel(GL_FLAT);
	glClearColor(0.0f, 0.0f, 0.0f, 0.5f);

	glClearDepth(1.0f);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	GLfloat LightAmbient[] = {0.2f,0.2f,0.2f,1.0f};
	GLfloat LightDiffuse[] = {1.0f,1.0f,1.0f,1.0f};
	GLfloat LightPosition[] = {-2.0f,2.0f,0.0f,1.0f};

	glLightfv(GL_LIGHT1, GL_AMBIENT, LightAmbient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, LightDiffuse);
	glLightfv(GL_LIGHT1, GL_POSITION, LightPosition);
	glEnable(GL_LIGHT1);

	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glEnable(GL_LIGHTING);

	glEnable(GL_COLOR_MATERIAL);

	// Setup line gain speeds
	for (int row=0; row < 4; row++) {
		for (int col=0; col < 10; col++) lineZSpeeds[row][col] = float(rand()%100+50)/100*250;
	}

	return true;
}
void fbgOpenGLRenderer::exit() {
	killLists();
	// Quit SDL
	SDL_Quit();
}

void fbgOpenGLRenderer::loadTextures() {
	glTGAImage bgload,mlload,mbload,mtload,nbload,blocksload,digitsload, gameoverload;
	// Load Images
	if (
		   !bgload.loadTGAImage(game->loadThemeFile("bg.tga"))
		|| !mlload.loadTGAImage(game->loadThemeFile("ml.tga"))
		|| !mbload.loadTGAImage(game->loadThemeFile("mb.tga"))
		|| !mtload.loadTGAImage(game->loadThemeFile("mt.tga"))
		|| !nbload.loadTGAImage(game->loadThemeFile("nb.tga"))
		|| !nr.loadTGAImage(game->loadThemeFile("nr.tga"))
		|| !blocksload.loadTGAImage(game->loadThemeFile("blockmap.tga"))
		|| !digitsload.loadTGAImage(game->loadThemeFile("digits.tga"))
		|| !gameoverload.loadTGAImage(game->loadThemeFile("gameover.tga"))
	) {
		cout << "Could not load graphics" << endl;
		::exit(0);
	}

	// Setup Images
	bg = bgload.splitImageMap(256, 256, 12);
		for (int i=0; i < 12; i++) bg[i].makeClamped();

	mlload.cropImage(0, -24, 64, 768);
	ml = mlload.splitImageMap(64, 256, 3);
		for (int i=0; i < 3; i++) ml[i].makeClamped();
	mbload.cropImage(-24, 0, 512, 64);
	mb = mbload.splitImageMap(256, 64, 2);
		for (int i=0; i < 2; i++) mb[i].makeClamped();
	mtload.cropImage(-24, 0, 512, 64);
	mt = mtload.splitImageMap(256, 64, 2);
		for (int i=0; i < 2; i++) mt[i].makeClamped();

	nbload.cropImage(-32, 0, 512, 64);
	nb = nbload.splitImageMap(256, 64, 2);
		for (int i=0; i < 2; i++) nb[i].makeClamped();
	nr.cropImage(0, -32, 64, 256);
		nr.makeClamped();

	blocks = blocksload.splitImageMap(64, 64, 7);

	digits = digitsload.splitImageMap(64, 64, 10);
		for (int i=0; i < 10; i++) digits[i].makeClamped();

	gameover = gameoverload.splitImageMap(256, 256, 2);
		for (int i=0; i < 2; i++) gameover[i].makeClamped();
}
void fbgOpenGLRenderer::killTextures() {
	delete [] bg;
	delete [] ml;
	delete [] mb;
	delete [] mt;
	delete [] nb;
	delete [] digits;
	delete [] gameover;
}

void fbgOpenGLRenderer::buildLists() {
	cube = glGenLists(2);

	glNewList(cube, GL_COMPILE);
// Back
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glTexCoord2d(1,1);glVertex3f(0.0f, 0.0f, -40.0f);
		glTexCoord2d(0,1);glVertex3f(40.0f, 0.0f, -40.0f);
		glTexCoord2d(1,0);glVertex3f(0.0f, -40.0f, -40.0f);
		glTexCoord2d(0,0);glVertex3f(40.0f, -40.0f, -40.0f);
	glEnd();
// Bottom
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(0.0f, -1.0f, 0.0f);
		glTexCoord2d(1,1);glVertex3f(40.0f, -40.0f, 0.0f);
		glTexCoord2d(0,1);glVertex3f(0.0f, -40.0f, 0.0f);
		glTexCoord2d(1,0);glVertex3f(40.0f, -40.0f, -40.0f);
		glTexCoord2d(0,0);glVertex3f(0.0f, -40.0f, -40.0f);
	glEnd();
// Top
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glTexCoord2d(1,1);glVertex3f(40.0f, 0.0f, -40.0f);
		glTexCoord2d(0,1);glVertex3f(0.0f, 0.0f, -40.0f);
		glTexCoord2d(1,0);glVertex3f(40.0f, 0.0f, 0.0f);
		glTexCoord2d(0,0);glVertex3f(0.0f, 0.0f, 0.0f);
	glEnd();
// Left
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(-1.0f, 0.0f, 0.0f);
		glTexCoord2d(1,1);glVertex3f(0.0f, 0.0f, 0.0f);
		glTexCoord2d(0,1);glVertex3f(0.0f, 0.0f, -40.0f);
		glTexCoord2d(1,0);glVertex3f(0.0f, -40.0f, 0.0f);
		glTexCoord2d(0,0);glVertex3f(0.0f, -40.0f, -40.0f);
	glEnd();
// Right
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(1.0f, 0.0f, 0.0f);
		glTexCoord2d(1,1);glVertex3f(40.0f, 0.0f, -40.0f);
		glTexCoord2d(0,1);glVertex3f(40.0f, 0.0f, 0.0f);
		glTexCoord2d(1,0);glVertex3f(40.0f, -40.0f, -40.0f);
		glTexCoord2d(0,0);glVertex3f(40.0f, -40.0f, 0.0f);
	glEnd();
// Front
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glTexCoord2d(1,1);glVertex3f(40.0f, 0.0f, 0.0f);
		glTexCoord2d(0,1);glVertex3f(0.0f, 0.0f, 0.0f);
		glTexCoord2d(1,0);glVertex3f(40.0f, -40.0f, 0.0f);
		glTexCoord2d(0,0);glVertex3f(0.0f, -40.0f, 0.0f);
	glEnd();
	glEndList();

	cubeWithoutBack = cube+1;
	glNewList(cubeWithoutBack, GL_COMPILE);
// Bottom
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(0.0f, -1.0f, 0.0f);
		glTexCoord2d(1,1);glVertex3f(40.0f, -40.0f, 0.0f);
		glTexCoord2d(0,1);glVertex3f(0.0f, -40.0f, 0.0f);
		glTexCoord2d(1,0);glVertex3f(40.0f, -40.0f, -40.0f);
		glTexCoord2d(0,0);glVertex3f(0.0f, -40.0f, -40.0f);
	glEnd();
// Top
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glTexCoord2d(1,1);glVertex3f(40.0f, 0.0f, -40.0f);
		glTexCoord2d(0,1);glVertex3f(0.0f, 0.0f, -40.0f);
		glTexCoord2d(1,0);glVertex3f(40.0f, 0.0f, 0.0f);
		glTexCoord2d(0,0);glVertex3f(0.0f, 0.0f, 0.0f);
	glEnd();
// Left
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(-1.0f, 0.0f, 0.0f);
		glTexCoord2d(1,1);glVertex3f(0.0f, 0.0f, 0.0f);
		glTexCoord2d(0,1);glVertex3f(0.0f, 0.0f, -40.0f);
		glTexCoord2d(1,0);glVertex3f(0.0f, -40.0f, 0.0f);
		glTexCoord2d(0,0);glVertex3f(0.0f, -40.0f, -40.0f);
	glEnd();
// Right
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(1.0f, 0.0f, 0.0f);
		glTexCoord2d(1,1);glVertex3f(40.0f, 0.0f, -40.0f);
		glTexCoord2d(0,1);glVertex3f(40.0f, 0.0f, 0.0f);
		glTexCoord2d(1,0);glVertex3f(40.0f, -40.0f, -40.0f);
		glTexCoord2d(0,0);glVertex3f(40.0f, -40.0f, 0.0f);
	glEnd();
// Front
	glBegin(GL_TRIANGLE_STRIP);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glTexCoord2d(1,1);glVertex3f(40.0f, 0.0f, 0.0f);
		glTexCoord2d(0,1);glVertex3f(0.0f, 0.0f, 0.0f);
		glTexCoord2d(1,0);glVertex3f(40.0f, -40.0f, 0.0f);
		glTexCoord2d(0,0);glVertex3f(0.0f, -40.0f, 0.0f);
	glEnd();
	glEndList();

	digitsBase=glGenLists(10);
	for (int i=0; i<10; i++) {
		glNewList(digitsBase+i, GL_COMPILE);
		glBindTexture(GL_TEXTURE_2D, digits[i].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glNormal3f(32.0f,-32.0f,10.0f);
			glTexCoord2f(0.0f,1.0f);glVertex3f(0.0f, 64.0f, 0.0f);
			glTexCoord2f(0.0f,0.0f);glVertex3f(0.0f, 0.0f, 0.0f);
			glTexCoord2f(1.0f,1.0f);glVertex3f(64.0f, 64.0f, 0.0f);
			glTexCoord2f(1.0f,0.0f);glVertex3f(64.0f, 0.0f, 0.0f);
		glEnd();
		glEndList();
	}
}
void fbgOpenGLRenderer::killLists() {
	glDeleteLists(cube, 2);
	glDeleteLists(digitsBase, 10);
}
void fbgOpenGLRenderer::drawDigits(const string& str) {
	for (int i=0; i < str.size(); i++) {
		if (str[i] >= 48 && str[i] <= 57) glCallList(digitsBase+str[i]-48);
		glTranslatef(64.0f, 0.0f, 0.0f);
	}
	// Move back
	glTranslatef(-float(str.size()*64), 0.0f, 0.0f);
}

void fbgOpenGLRenderer::draw() {
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	// Makes the visible screen 0-1024x 0-(-768)y
	glLoadIdentity();
	glTranslatef(-512.0f,384.0f,-927.05801f);

// Main Face
	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glBindTexture(GL_TEXTURE_2D, bg[0].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f);glVertex3f(0.0f, 0.0f, 0.0f);
			glTexCoord2f(0.0f, 0.90625f);glVertex3f(0.0f, -24.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f);glVertex3f(256.0f, 0.0f, 0.0f);
			glTexCoord2f(1.0f, 0.90625f);glVertex3f(256.0f, -24.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[1].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(256.0f, 0.0f, 0.0f);
			glTexCoord2f(0.0f, 0.90625f); glVertex3f(256.0f, -24.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(512.0f, 0.0f, 0.0f);
			glTexCoord2f(1.0f, 0.90625f); glVertex3f(512.0f, -24.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[2].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(512.0f, 0.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(512.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(768.0f, 0.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(768.0f, -256.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[3].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(768.0f, 0.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(768.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(1024.0f, 0.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(1024.0f, -256.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[6].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(512.0f, -256.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(512.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(768.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(768.0f, -512.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[7].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(768.0f, -256.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(768.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(1024.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(1024.0f, -512.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[0].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.90625f); glVertex3f(0.0f, -24.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(0.0f, -256.0f, 0.0f);
			glTexCoord2f(0.09375f, 0.90625f); glVertex3f(24.0f, -24.0f, 0.0f);
			glTexCoord2f(0.09375f, 0.0f); glVertex3f(24.0f, -256.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[4].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0, 1); glVertex3f(0.0f, -256.0f, 0.0f);
			glTexCoord2f(0, 0); glVertex3f(0.0f, -512.0f, 0.0f);
			glTexCoord2f(0.09375f, 1); glVertex3f(24.0f, -256.0f, 0.0f);
			glTexCoord2f(0.09375f, 0); glVertex3f(24.0f, -512.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[8].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0, 1); glVertex3f(0.0f, -512.0f, 0.0f);
			glTexCoord2f(0, 0.09375f); glVertex3f(0.0f, -744.0f, 0.0f);
			glTexCoord2f(0.09375f, 1); glVertex3f(24.0f, -512.0f, 0.0f);
			glTexCoord2f(0.09375f, 0.09375f); glVertex3f(24.0f, -744.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[1].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.65625f, 0.90625f); glVertex3f(424.0f, -24.0f, 0.0f);
			glTexCoord2f(0.65625f, 0.0f); glVertex3f(424.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 0.90625f); glVertex3f(512.0f, -24.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(512.0f, -256.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[5].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.65625f, 1.0f); glVertex3f(424.0f, -256.0f, 0.0f);
			glTexCoord2f(0.65625f, 0.0f); glVertex3f(424.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(512.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(512.0f, -512.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[9].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.65625f, 1.0f); glVertex3f(424.0f, -512.0f, 0.0f);
			glTexCoord2f(0.65625f, 0.09375f); glVertex3f(424.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(512.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(512.0f, -744.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[8].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.09375f); glVertex3f(0.0f, -744.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(0.0f, -768.0f, 0.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(256.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(256.0f, -768.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[9].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.09375f); glVertex3f(256.0f, -744.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(256.0f, -768.0f, 0.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(512.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(512.0f, -768.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[10].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.09375f); glVertex3f(512.0f, -744.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(512.0f, -768.0f, 0.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(768.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(768.0f, -768.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[11].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.09375f); glVertex3f(768.0f, -744.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(768.0f, -768.0f, 0.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(1024.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(1024.0f, -768.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[10].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(512.0f, -512.0f, 0.0f);
			glTexCoord2f(0.0f, 0.09375f); glVertex3f(512.0f, -744.0f, 0.0f);
			glTexCoord2f(0.5625f, 1.0f); glVertex3f(656.0f, -512.0f, 0.0f);
			glTexCoord2f(0.5625f, 0.09375f); glVertex3f(656.0f, -744.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[11].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.3125f, 1.0f); glVertex3f(848.0f, -512.0f, 0.0f);
			glTexCoord2f(0.3125f, 0.09375f); glVertex3f(848.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(1024.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(1024.0f, -744.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[10].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.5625f, 1.0f); glVertex3f(656.0f, -512.0f, 0.0f);
			glTexCoord2f(0.5625f, 0.84375f); glVertex3f(656.0f, -552.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(768.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 0.84375f); glVertex3f(768.0f, -552.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[11].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(768.0f, -512.0f, 0.0f);
			glTexCoord2f(0.0f, 0.84375f); glVertex3f(768.0f, -552.0f, 0.0f);
			glTexCoord2f(0.3125f, 1.0f); glVertex3f(848.0f, -512.0f, 0.0f);
			glTexCoord2f(0.3125f, 0.84375f); glVertex3f(848.0f, -552.0f, 0.0f);
		glEnd();

	// Main Insets
	glBindTexture(GL_TEXTURE_2D, ml[0].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.90625f); glVertex3f(24.0f, -24.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(24.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 0.90625f); glVertex3f(24.0f, -24.0f, -64.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(24.0f, -256.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, ml[1].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(24.0f, -256.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(24.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(24.0f, -256.0f, -64.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(24.0f, -512.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, ml[2].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(24.0f, -512.0f, 0.0f);
			glTexCoord2f(0.0f, 0.09375f); glVertex3f(24.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(24.0f, -512.0f, -64.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(24.0f, -744.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, mb[0].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.09375f, 1.0f); glVertex3f(24.0f, -744.0f, -64.0f);
			glTexCoord2f(0.09375f, 0.0f); glVertex3f(24.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(256.0f, -744.0f, -64.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(256.0f, -744.0f, -0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, mb[1].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(256.0f, -744.0f, -64.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(256.0f, -744.0f, 0.0f);
			glTexCoord2f(0.65625f, 1.0f); glVertex3f(424.0f, -744.0f, -64.0f);
			glTexCoord2f(0.65625f, 0.0f); glVertex3f(424.0f, -744.0f, -0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, mt[0].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.09375f, 1.0f); glVertex3f(24.0f, -24.0f, 0.0f);
			glTexCoord2f(0.09375f, 0.0f); glVertex3f(24.0f, -24.0f, -64.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(256.0f, -24.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(256.0f, -24.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, mt[1].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(256.0f, -24.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(256.0f, -24.0f, -64.0f);
			glTexCoord2f(0.65625f, 1.0f); glVertex3f(424.0f, -24.0f, 0.0f);
			glTexCoord2f(0.65625f, 0.0f); glVertex3f(424.0f, -24.0f, -64.0f);
		glEnd();
	// Main Backing
	glBindTexture(GL_TEXTURE_2D, bg[0].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.09375f, 0.90625f); glVertex3f(24.0f, -24.0f, -64.0f);
			glTexCoord2f(0.09375f, 0.0f); glVertex3f(24.0f, -256.0f, -64.0f);
			glTexCoord2f(1.0f, 0.90625f); glVertex3f(256.0f, -24.0f, -64.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(256.0f, -256.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[1].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.90625f); glVertex3f(256.0f, -24.0f, -64.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(256.0f, -256.0f, -64.0f);
			glTexCoord2f(0.65625f, 0.90625f); glVertex3f(424.0f, -24.0f, -64.0f);
			glTexCoord2f(0.65625f, 0.0f); glVertex3f(424.0f, -256.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[4].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.09375f, 1.0f); glVertex3f(24.0f, -256.0f, -64.0f);
			glTexCoord2f(0.09375f, 0.0f); glVertex3f(24.0f, -512.0f, -64.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(256.0f, -256.0f, -64.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(256.0f, -512.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[5].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(256.0f, -256.0f, -64.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(256.0f, -512.0f, -64.0f);
			glTexCoord2f(0.65625f, 1.0f); glVertex3f(424.0f, -256.0f, -64.0f);
			glTexCoord2f(0.65625f, 0.0f); glVertex3f(424.0f, -512.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[8].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.09375f, 1.0f); glVertex3f(24.0f, -512.0f, -64.0f);
			glTexCoord2f(0.09375f, 0.09375f); glVertex3f(24.0f, -744.0f, -64.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(256.0f, -512.0f, -64.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(256.0f, -744.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[9].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(256.0f, -512.0f, -64.0f);
			glTexCoord2f(0.0f, 0.09375f); glVertex3f(256.0f, -744.0f, -64.0f);
			glTexCoord2f(0.65625f, 1.0f); glVertex3f(424.0f, -512.0f, -64.0f);
			glTexCoord2f(0.65625f, 0.09375f); glVertex3f(424.0f, -744.0f, -64.0f);
		glEnd();

	// Next Block Insets
	glBindTexture(GL_TEXTURE_2D, nb[0].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.5625f, 1.0f); glVertex3f(656.0f, -744.0f, -64.0f);
			glTexCoord2f(0.5625f, 0.0f); glVertex3f(656.0f, -744.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(768.0f, -744.0f, -64.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(768.0f, -744.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, nb[1].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(768.0f, -744.0f, -64.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(768.0f, -744.0f, 0.0f);
			glTexCoord2f(0.3125f, 1.0f); glVertex3f(848.0f, -744.0f, -64.0f);
			glTexCoord2f(0.3125f, 0.0f); glVertex3f(848.0f, -744.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, nr.getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.84375f); glVertex3f(848.0f, -552.0f, -64.0f);
			glTexCoord2f(0.0f, 0.09375f); glVertex3f(848.0f, -744.0f, -64.0f);
			glTexCoord2f(1.0f, 0.84375f); glVertex3f(848.0f, -552.0f, 0.0f);
			glTexCoord2f(1.0f, 0.09375f); glVertex3f(848.0f, -744.0f, 0.0f);
		glEnd();
	// Next Block Backing
	glBindTexture(GL_TEXTURE_2D, bg[10].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.5625f, 0.84375f); glVertex3f(656.0f, -552.0f, -64.0f);
			glTexCoord2f(0.5625f, 0.125f); glVertex3f(656.0f, -744.0f, -64.0f);
			glTexCoord2f(1.0f, 0.84375f); glVertex3f(768.0f, -552.0f, -64.0f);
			glTexCoord2f(1.0f, 0.125f); glVertex3f(768.0f, -744.0f, -64.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, bg[11].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 0.84375f); glVertex3f(768.0f, -552.0f, -64.0f);
			glTexCoord2f(0.0f, 0.125f); glVertex3f(768.0f, -744.0f, -64.0f);
			glTexCoord2f(0.3125f, 0.84375f); glVertex3f(848.0f, -552.0f, -64.0f);
			glTexCoord2f(0.3125f, 0.125f); glVertex3f(848.0f, -744.0f, -64.0f);
		glEnd();
	// Draw Level, Lines, Score
	char tmpString[9];

	glPushMatrix();
		glTranslatef(512.0f, -368.0f, 0.0f);
		sprintf(tmpString, "%02u", game->getLevel());
		drawDigits(tmpString);
	glPopMatrix();
	glPushMatrix();
		glTranslatef(800.0f, -368.0f, 0.0f);
		sprintf(tmpString, "%03u", game->getLines());
		drawDigits(tmpString);
	glPopMatrix();
	glPushMatrix();
		glTranslatef(528.0f, -480.0f, 0.0f);
		sprintf(tmpString, "%07u", game->getScore());
		drawDigits(tmpString);
	glPopMatrix();

	if (game->getLineGain()) drawLineGain();
	else {
		if (game->getState() == GAMEOVER) drawGameOver();
		else drawGameplay();
	}

	SDL_GL_SwapBuffers();
}
void fbgOpenGLRenderer::drawGameMatrix() {
	glPushMatrix();
	glTranslatef(24.0f, -24.0f, -24.0f);
	for (int row=0; row<18; row++) {
		for (int col=0; col < 10; col++) {
			if (game->getMatrix(row, col)) {
				glBindTexture(GL_TEXTURE_2D, blocks[game->getMatrix(row, col)-1].getID());
				glCallList(cubeWithoutBack);
			}
			glTranslatef(40.0f, 0.0f, 0.0f);
		}
		glTranslatef(-400.0f, -40.0f, 0.0f);	// Move to front of row
	}
	glPopMatrix();
}
void fbgOpenGLRenderer::drawNextBlock(fbgBlock* theBlock) {
	// Draw Next Block
	glPushMatrix();
	glTranslatef(48.0f+627.0f,-568.0f,0.0f);
	float tRot = game->getLight() ? 0.0f : sin(float(SDL_GetTicks() % 6000)/6000*2*3.1415926f)*15.0f;
	glTranslatef(80.0f, 0.0f, 0.0f); glRotatef(tRot, 0.0f, 1.0f, 0.0f); glTranslatef(-80.0f, 0.0f, 0.0f);	// Fancy rotate
	glTranslatef(float(theBlock->getWidth()%2)*(20.0f), (4.0f-theBlock->getHeight())/2*(-40.0f), 20.0f);
	glBindTexture(GL_TEXTURE_2D, blocks[theBlock->getIndex()].getID());
	for (int row=0; row<4; row++) {
		for (int col=0; col < 4; col++) {
			if (theBlock->getMatrix(row, col)) glCallList(cube+1);
			glTranslatef(40.0f, 0.0f, 0.0f);
		}
		glTranslatef(-160.0f, -40.0f, 0.0f);	// Move to front of row
	}
	glPopMatrix();
}

void fbgOpenGLRenderer::drawGameOver() {
	glEnable(GL_LIGHTING);
	drawGameMatrix();
	glDisable(GL_LIGHTING);
	glBindTexture(GL_TEXTURE_2D, gameover[0].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(0.0f, -256.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(0.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(256.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(256.0f, -512.0f, 0.0f);
		glEnd();
	glBindTexture(GL_TEXTURE_2D, gameover[1].getID());
		glBegin(GL_TRIANGLE_STRIP);
			glTexCoord2f(0.0f, 1.0f); glVertex3f(256.0f, -256.0f, 0.0f);
			glTexCoord2f(0.0f, 0.0f); glVertex3f(256.0f, -512.0f, 0.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex3f(512.0f, -256.0f, 0.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex3f(512.0f, -512.0f, 0.0f);
		glEnd();
}
void fbgOpenGLRenderer::drawGameplay() {
	// Draw Matrix
	glEnable(GL_LIGHTING);
	drawGameMatrix();
	// Current Block
	glPushMatrix();
	glTranslatef(24.0f+game->getCurBlock()->getPosX()*40.0f, -24.0f-game->getCurBlock()->getPosY()*40.0f+(game->getLight() ? 0.0f : game->getSmoothAdjust()), -24.0f);
	glBindTexture(GL_TEXTURE_2D, blocks[game->getCurBlock()->getIndex()].getID());
	for (int row=0; row<4; row++) {
		for (int col=0; col < 4; col++) {
			if (game->getCurBlock()->getMatrix(row, col)) glCallList(cube+1);
			glTranslatef(40.0f, 0.0f, 0.0f);
		}
		glTranslatef(-160.0f, -40.0f, 0.0f);
	}
	glPopMatrix();

	drawNextBlock(game->getBlockSet(game->getNextBlockIndex()));

	glDisable(GL_LIGHTING);
}

void fbgOpenGLRenderer::drawLineGain() {
	int time = Sint32(SDL_GetTicks() - Sint32(game->getLastLine()));
	float pct = float(time)/LINESPEED*2.0f;
	if (pct > 1.0f) pct = 1.0f;
	float pct2 = (float(time)/LINESPEED-0.5f)*2.0f;
	if (pct2 < 0.0f) pct2 = 0.0f;
	glEnable(GL_LIGHTING);

	// Draw non-transparent matrix
	char cd = 0;	// Current line from lineGain
	glTranslatef(24.0f, -24.0f-680.0f, -24.0f);
	for (int row=17; row >= 0; row--) {
		if (cd < 4 && row == game->getLineGain()->linePos[cd]) {
			glTranslatef(0.0f, 40.0f, 0.0f);
			cd++;
		}
		else {
			glTranslatef(0.0f, -40.0f*cd*pct2, 0.0f);
			for (int col=0; col < 10; col++) {
				if (game->getMatrix(row+cd, col)) {
					glBindTexture(GL_TEXTURE_2D, blocks[game->getMatrix(row+cd, col)-1].getID());
					glCallList(cubeWithoutBack);
				}
				glTranslatef(40.0f, 0.0f, 0.0f);
			}
			glTranslatef(0.0f, 40.0f*cd*pct2, 0.0f);
			glTranslatef(-400.0f, 40.0f, 0.0f);	// Move to front of row
		}
	}
	glTranslatef(-24.0f, 24.0f-40.0f, 24.0f);

	// Draw the transparent lines (possibly)
	if (pct < 1.0f) {
		cd = 0;
		glTranslatef(24.0f, -24.0f-680.0f, -24.0f);
		for (int row=17; row >= 0; row--) {
			if (cd < 4 && row == game->getLineGain()->linePos[cd]) {
				glEnable(GL_BLEND);
				glColor4f(1.0f, 1.0f, 1.0f, 1.0f-pct);
				glTranslatef(360.0f, 0.0f, 0.0f);	// Move to back of row
				for (int col=9; col >= 0; col--) {
					if (game->getLineGain()->lineMatrix[cd][col]) {
//						glTranslatef(0.0f, -180.0f*(pct*pct), lineZSpeeds[cd][col]*pct);
						glBindTexture(GL_TEXTURE_2D, blocks[game->getLineGain()->lineMatrix[cd][col]-1].getID());
						glCallList(cubeWithoutBack);
//						glTranslatef(0.0f, 180.0f*(pct*pct), -lineZSpeeds[cd][col]*pct);
					}
					glTranslatef(-40.0f, 0.0f, 0.0f);
				}
				glTranslatef(40.0f, 40.0f, 0.0f);
				glDisable(GL_BLEND);

				cd++;
			}
			else glTranslatef(0.0f, 40.0f, 0.0f);	// Move to front of row
		}
		glTranslatef(-24.0f, 24.0f-40.0f, 24.0f);
	}
	
	// If a level was gained
	if (game->getLineGain()->levelGained > 1) {
		char tmpString[3];
		glEnable(GL_BLEND);
		glColor4f(1.0f, 1.0f, 1.0f, 1.0f-pct);
		glTranslatef(512.0f, -368.0f, 400.0f*pct);
		sprintf(tmpString, "%02u", game->getLineGain()->levelGained);
		drawDigits(tmpString);
		glTranslatef(512.0f, -368.0f, -400.0f*pct);
		glDisable(GL_BLEND);
	}

	drawNextBlock(game->getCurBlock());
	glDisable(GL_LIGHTING);
}
