""" Mappings from Adobe glyph names to Unicode characters.

In some CMap tables, Adobe glyph names are used for specifying
Unicode characters instead of using decimal/hex character code.

The following data was taken by

  $ wget https://partners.adobe.com/public/developer/en/opentype/glyphlist.txt
  $ python tools/conv_glyphlist.py glyphlist.txt > glyphlist.py

"""

# ###################################################################################
# Copyright (c) 1997,1998,2002,2007 Adobe Systems Incorporated
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this documentation file to use, copy, publish, distribute,
# sublicense, and/or sell copies of the documentation, and to permit
# others to do the same, provided that:
# - No modification, editing or other alteration of this document is
# allowed; and
# - The above copyright notice and this permission notice shall be
# included in all copies of the documentation.
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this documentation file, to create their own derivative works
# from the content of this document to use, copy, publish, distribute,
# sublicense, and/or sell the derivative works, and to permit others to do
# the same, provided that the derived work is not represented as being a
# copy or version of this document.
#
# Adobe shall not be liable to any party for any loss of revenue or profit
# or for indirect, incidental, special, consequential, or other similar
# damages, whether based on tort (including without limitation negligence
# or strict liability), contract or other legal or equitable grounds even
# if Adobe has been advised or had reason to know of the possibility of
# such damages. The Adobe materials are provided on an "AS IS" basis.
# Adobe specifically disclaims all express, statutory, or implied
# warranties relating to the Adobe materials, including but not limited to
# those concerning merchantability or fitness for a particular purpose or
# non-infringement of any third party rights regarding the Adobe
# materials.
# ###################################################################################
# Name:          Adobe Glyph List
# Table version: 2.0
# Date:          September 20, 2002
#
# See http://partners.adobe.com/asn/developer/typeforum/unicodegn.html
#
# Format: Semicolon-delimited fields:
#            (1) glyph name
#            (2) Unicode scalar value


def convert_glyphlist(path: str) -> None:
    """Convert a glyph list into a python representation.

    See output below.
    """
    state = 0
    with open(path) as fileinput:
        for line in fileinput.readlines():
            line = line.strip()
            if not line or line.startswith("#"):
                if state == 1:
                    state = 2
                    print("}\n")
                print(line)
                continue
            if state == 0:
                print("\nglyphname2unicode = {")
                state = 1
            (name, x) = line.split(";")
            codes = x.split(" ")
            print(
                " {!r}: u'{}',".format(name, "".join("\\u%s" % code for code in codes))
            )


glyphname2unicode = {
    "A": "\u0041",
    "AE": "\u00C6",
    "AEacute": "\u01FC",
    "AEmacron": "\u01E2",
    "AEsmall": "\uF7E6",
    "Aacute": "\u00C1",
    "Aacutesmall": "\uF7E1",
    "Abreve": "\u0102",
    "Abreveacute": "\u1EAE",
    "Abrevecyrillic": "\u04D0",
    "Abrevedotbelow": "\u1EB6",
    "Abrevegrave": "\u1EB0",
    "Abrevehookabove": "\u1EB2",
    "Abrevetilde": "\u1EB4",
    "Acaron": "\u01CD",
    "Acircle": "\u24B6",
    "Acircumflex": "\u00C2",
    "Acircumflexacute": "\u1EA4",
    "Acircumflexdotbelow": "\u1EAC",
    "Acircumflexgrave": "\u1EA6",
    "Acircumflexhookabove": "\u1EA8",
    "Acircumflexsmall": "\uF7E2",
    "Acircumflextilde": "\u1EAA",
    "Acute": "\uF6C9",
    "Acutesmall": "\uF7B4",
    "Acyrillic": "\u0410",
    "Adblgrave": "\u0200",
    "Adieresis": "\u00C4",
    "Adieresiscyrillic": "\u04D2",
    "Adieresismacron": "\u01DE",
    "Adieresissmall": "\uF7E4",
    "Adotbelow": "\u1EA0",
    "Adotmacron": "\u01E0",
    "Agrave": "\u00C0",
    "Agravesmall": "\uF7E0",
    "Ahookabove": "\u1EA2",
    "Aiecyrillic": "\u04D4",
    "Ainvertedbreve": "\u0202",
    "Alpha": "\u0391",
    "Alphatonos": "\u0386",
    "Amacron": "\u0100",
    "Amonospace": "\uFF21",
    "Aogonek": "\u0104",
    "Aring": "\u00C5",
    "Aringacute": "\u01FA",
    "Aringbelow": "\u1E00",
    "Aringsmall": "\uF7E5",
    "Asmall": "\uF761",
    "Atilde": "\u00C3",
    "Atildesmall": "\uF7E3",
    "Aybarmenian": "\u0531",
    "B": "\u0042",
    "Bcircle": "\u24B7",
    "Bdotaccent": "\u1E02",
    "Bdotbelow": "\u1E04",
    "Becyrillic": "\u0411",
    "Benarmenian": "\u0532",
    "Beta": "\u0392",
    "Bhook": "\u0181",
    "Blinebelow": "\u1E06",
    "Bmonospace": "\uFF22",
    "Brevesmall": "\uF6F4",
    "Bsmall": "\uF762",
    "Btopbar": "\u0182",
    "C": "\u0043",
    "Caarmenian": "\u053E",
    "Cacute": "\u0106",
    "Caron": "\uF6CA",
    "Caronsmall": "\uF6F5",
    "Ccaron": "\u010C",
    "Ccedilla": "\u00C7",
    "Ccedillaacute": "\u1E08",
    "Ccedillasmall": "\uF7E7",
    "Ccircle": "\u24B8",
    "Ccircumflex": "\u0108",
    "Cdot": "\u010A",
    "Cdotaccent": "\u010A",
    "Cedillasmall": "\uF7B8",
    "Chaarmenian": "\u0549",
    "Cheabkhasiancyrillic": "\u04BC",
    "Checyrillic": "\u0427",
    "Chedescenderabkhasiancyrillic": "\u04BE",
    "Chedescendercyrillic": "\u04B6",
    "Chedieresiscyrillic": "\u04F4",
    "Cheharmenian": "\u0543",
    "Chekhakassiancyrillic": "\u04CB",
    "Cheverticalstrokecyrillic": "\u04B8",
    "Chi": "\u03A7",
    "Chook": "\u0187",
    "Circumflexsmall": "\uF6F6",
    "Cmonospace": "\uFF23",
    "Coarmenian": "\u0551",
    "Csmall": "\uF763",
    "D": "\u0044",
    "DZ": "\u01F1",
    "DZcaron": "\u01C4",
    "Daarmenian": "\u0534",
    "Dafrican": "\u0189",
    "Dcaron": "\u010E",
    "Dcedilla": "\u1E10",
    "Dcircle": "\u24B9",
    "Dcircumflexbelow": "\u1E12",
    "Dcroat": "\u0110",
    "Ddotaccent": "\u1E0A",
    "Ddotbelow": "\u1E0C",
    "Decyrillic": "\u0414",
    "Deicoptic": "\u03EE",
    "Delta": "\u2206",
    "Deltagreek": "\u0394",
    "Dhook": "\u018A",
    "Dieresis": "\uF6CB",
    "DieresisAcute": "\uF6CC",
    "DieresisGrave": "\uF6CD",
    "Dieresissmall": "\uF7A8",
    "Digammagreek": "\u03DC",
    "Djecyrillic": "\u0402",
    "Dlinebelow": "\u1E0E",
    "Dmonospace": "\uFF24",
    "Dotaccentsmall": "\uF6F7",
    "Dslash": "\u0110",
    "Dsmall": "\uF764",
    "Dtopbar": "\u018B",
    "Dz": "\u01F2",
    "Dzcaron": "\u01C5",
    "Dzeabkhasiancyrillic": "\u04E0",
    "Dzecyrillic": "\u0405",
    "Dzhecyrillic": "\u040F",
    "E": "\u0045",
    "Eacute": "\u00C9",
    "Eacutesmall": "\uF7E9",
    "Ebreve": "\u0114",
    "Ecaron": "\u011A",
    "Ecedillabreve": "\u1E1C",
    "Echarmenian": "\u0535",
    "Ecircle": "\u24BA",
    "Ecircumflex": "\u00CA",
    "Ecircumflexacute": "\u1EBE",
    "Ecircumflexbelow": "\u1E18",
    "Ecircumflexdotbelow": "\u1EC6",
    "Ecircumflexgrave": "\u1EC0",
    "Ecircumflexhookabove": "\u1EC2",
    "Ecircumflexsmall": "\uF7EA",
    "Ecircumflextilde": "\u1EC4",
    "Ecyrillic": "\u0404",
    "Edblgrave": "\u0204",
    "Edieresis": "\u00CB",
    "Edieresissmall": "\uF7EB",
    "Edot": "\u0116",
    "Edotaccent": "\u0116",
    "Edotbelow": "\u1EB8",
    "Efcyrillic": "\u0424",
    "Egrave": "\u00C8",
    "Egravesmall": "\uF7E8",
    "Eharmenian": "\u0537",
    "Ehookabove": "\u1EBA",
    "Eightroman": "\u2167",
    "Einvertedbreve": "\u0206",
    "Eiotifiedcyrillic": "\u0464",
    "Elcyrillic": "\u041B",
    "Elevenroman": "\u216A",
    "Emacron": "\u0112",
    "Emacronacute": "\u1E16",
    "Emacrongrave": "\u1E14",
    "Emcyrillic": "\u041C",
    "Emonospace": "\uFF25",
    "Encyrillic": "\u041D",
    "Endescendercyrillic": "\u04A2",
    "Eng": "\u014A",
    "Enghecyrillic": "\u04A4",
    "Enhookcyrillic": "\u04C7",
    "Eogonek": "\u0118",
    "Eopen": "\u0190",
    "Epsilon": "\u0395",
    "Epsilontonos": "\u0388",
    "Ercyrillic": "\u0420",
    "Ereversed": "\u018E",
    "Ereversedcyrillic": "\u042D",
    "Escyrillic": "\u0421",
    "Esdescendercyrillic": "\u04AA",
    "Esh": "\u01A9",
    "Esmall": "\uF765",
    "Eta": "\u0397",
    "Etarmenian": "\u0538",
    "Etatonos": "\u0389",
    "Eth": "\u00D0",
    "Ethsmall": "\uF7F0",
    "Etilde": "\u1EBC",
    "Etildebelow": "\u1E1A",
    "Euro": "\u20AC",
    "Ezh": "\u01B7",
    "Ezhcaron": "\u01EE",
    "Ezhreversed": "\u01B8",
    "F": "\u0046",
    "Fcircle": "\u24BB",
    "Fdotaccent": "\u1E1E",
    "Feharmenian": "\u0556",
    "Feicoptic": "\u03E4",
    "Fhook": "\u0191",
    "Fitacyrillic": "\u0472",
    "Fiveroman": "\u2164",
    "Fmonospace": "\uFF26",
    "Fourroman": "\u2163",
    "Fsmall": "\uF766",
    "G": "\u0047",
    "GBsquare": "\u3387",
    "Gacute": "\u01F4",
    "Gamma": "\u0393",
    "Gammaafrican": "\u0194",
    "Gangiacoptic": "\u03EA",
    "Gbreve": "\u011E",
    "Gcaron": "\u01E6",
    "Gcedilla": "\u0122",
    "Gcircle": "\u24BC",
    "Gcircumflex": "\u011C",
    "Gcommaaccent": "\u0122",
    "Gdot": "\u0120",
    "Gdotaccent": "\u0120",
    "Gecyrillic": "\u0413",
    "Ghadarmenian": "\u0542",
    "Ghemiddlehookcyrillic": "\u0494",
    "Ghestrokecyrillic": "\u0492",
    "Gheupturncyrillic": "\u0490",
    "Ghook": "\u0193",
    "Gimarmenian": "\u0533",
    "Gjecyrillic": "\u0403",
    "Gmacron": "\u1E20",
    "Gmonospace": "\uFF27",
    "Grave": "\uF6CE",
    "Gravesmall": "\uF760",
    "Gsmall": "\uF767",
    "Gsmallhook": "\u029B",
    "Gstroke": "\u01E4",
    "H": "\u0048",
    "H18533": "\u25CF",
    "H18543": "\u25AA",
    "H18551": "\u25AB",
    "H22073": "\u25A1",
    "HPsquare": "\u33CB",
    "Haabkhasiancyrillic": "\u04A8",
    "Hadescendercyrillic": "\u04B2",
    "Hardsigncyrillic": "\u042A",
    "Hbar": "\u0126",
    "Hbrevebelow": "\u1E2A",
    "Hcedilla": "\u1E28",
    "Hcircle": "\u24BD",
    "Hcircumflex": "\u0124",
    "Hdieresis": "\u1E26",
    "Hdotaccent": "\u1E22",
    "Hdotbelow": "\u1E24",
    "Hmonospace": "\uFF28",
    "Hoarmenian": "\u0540",
    "Horicoptic": "\u03E8",
    "Hsmall": "\uF768",
    "Hungarumlaut": "\uF6CF",
    "Hungarumlautsmall": "\uF6F8",
    "Hzsquare": "\u3390",
    "I": "\u0049",
    "IAcyrillic": "\u042F",
    "IJ": "\u0132",
    "IUcyrillic": "\u042E",
    "Iacute": "\u00CD",
    "Iacutesmall": "\uF7ED",
    "Ibreve": "\u012C",
    "Icaron": "\u01CF",
    "Icircle": "\u24BE",
    "Icircumflex": "\u00CE",
    "Icircumflexsmall": "\uF7EE",
    "Icyrillic": "\u0406",
    "Idblgrave": "\u0208",
    "Idieresis": "\u00CF",
    "Idieresisacute": "\u1E2E",
    "Idieresiscyrillic": "\u04E4",
    "Idieresissmall": "\uF7EF",
    "Idot": "\u0130",
    "Idotaccent": "\u0130",
    "Idotbelow": "\u1ECA",
    "Iebrevecyrillic": "\u04D6",
    "Iecyrillic": "\u0415",
    "Ifraktur": "\u2111",
    "Igrave": "\u00CC",
    "Igravesmall": "\uF7EC",
    "Ihookabove": "\u1EC8",
    "Iicyrillic": "\u0418",
    "Iinvertedbreve": "\u020A",
    "Iishortcyrillic": "\u0419",
    "Imacron": "\u012A",
    "Imacroncyrillic": "\u04E2",
    "Imonospace": "\uFF29",
    "Iniarmenian": "\u053B",
    "Iocyrillic": "\u0401",
    "Iogonek": "\u012E",
    "Iota": "\u0399",
    "Iotaafrican": "\u0196",
    "Iotadieresis": "\u03AA",
    "Iotatonos": "\u038A",
    "Ismall": "\uF769",
    "Istroke": "\u0197",
    "Itilde": "\u0128",
    "Itildebelow": "\u1E2C",
    "Izhitsacyrillic": "\u0474",
    "Izhitsadblgravecyrillic": "\u0476",
    "J": "\u004A",
    "Jaarmenian": "\u0541",
    "Jcircle": "\u24BF",
    "Jcircumflex": "\u0134",
    "Jecyrillic": "\u0408",
    "Jheharmenian": "\u054B",
    "Jmonospace": "\uFF2A",
    "Jsmall": "\uF76A",
    "K": "\u004B",
    "KBsquare": "\u3385",
    "KKsquare": "\u33CD",
    "Kabashkircyrillic": "\u04A0",
    "Kacute": "\u1E30",
    "Kacyrillic": "\u041A",
    "Kadescendercyrillic": "\u049A",
    "Kahookcyrillic": "\u04C3",
    "Kappa": "\u039A",
    "Kastrokecyrillic": "\u049E",
    "Kaverticalstrokecyrillic": "\u049C",
    "Kcaron": "\u01E8",
    "Kcedilla": "\u0136",
    "Kcircle": "\u24C0",
    "Kcommaaccent": "\u0136",
    "Kdotbelow": "\u1E32",
    "Keharmenian": "\u0554",
    "Kenarmenian": "\u053F",
    "Khacyrillic": "\u0425",
    "Kheicoptic": "\u03E6",
    "Khook": "\u0198",
    "Kjecyrillic": "\u040C",
    "Klinebelow": "\u1E34",
    "Kmonospace": "\uFF2B",
    "Koppacyrillic": "\u0480",
    "Koppagreek": "\u03DE",
    "Ksicyrillic": "\u046E",
    "Ksmall": "\uF76B",
    "L": "\u004C",
    "LJ": "\u01C7",
    "LL": "\uF6BF",
    "Lacute": "\u0139",
    "Lambda": "\u039B",
    "Lcaron": "\u013D",
    "Lcedilla": "\u013B",
    "Lcircle": "\u24C1",
    "Lcircumflexbelow": "\u1E3C",
    "Lcommaaccent": "\u013B",
    "Ldot": "\u013F",
    "Ldotaccent": "\u013F",
    "Ldotbelow": "\u1E36",
    "Ldotbelowmacron": "\u1E38",
    "Liwnarmenian": "\u053C",
    "Lj": "\u01C8",
    "Ljecyrillic": "\u0409",
    "Llinebelow": "\u1E3A",
    "Lmonospace": "\uFF2C",
    "Lslash": "\u0141",
    "Lslashsmall": "\uF6F9",
    "Lsmall": "\uF76C",
    "M": "\u004D",
    "MBsquare": "\u3386",
    "Macron": "\uF6D0",
    "Macronsmall": "\uF7AF",
    "Macute": "\u1E3E",
    "Mcircle": "\u24C2",
    "Mdotaccent": "\u1E40",
    "Mdotbelow": "\u1E42",
    "Menarmenian": "\u0544",
    "Mmonospace": "\uFF2D",
    "Msmall": "\uF76D",
    "Mturned": "\u019C",
    "Mu": "\u039C",
    "N": "\u004E",
    "NJ": "\u01CA",
    "Nacute": "\u0143",
    "Ncaron": "\u0147",
    "Ncedilla": "\u0145",
    "Ncircle": "\u24C3",
    "Ncircumflexbelow": "\u1E4A",
    "Ncommaaccent": "\u0145",
    "Ndotaccent": "\u1E44",
    "Ndotbelow": "\u1E46",
    "Nhookleft": "\u019D",
    "Nineroman": "\u2168",
    "Nj": "\u01CB",
    "Njecyrillic": "\u040A",
    "Nlinebelow": "\u1E48",
    "Nmonospace": "\uFF2E",
    "Nowarmenian": "\u0546",
    "Nsmall": "\uF76E",
    "Ntilde": "\u00D1",
    "Ntildesmall": "\uF7F1",
    "Nu": "\u039D",
    "O": "\u004F",
    "OE": "\u0152",
    "OEsmall": "\uF6FA",
    "Oacute": "\u00D3",
    "Oacutesmall": "\uF7F3",
    "Obarredcyrillic": "\u04E8",
    "Obarreddieresiscyrillic": "\u04EA",
    "Obreve": "\u014E",
    "Ocaron": "\u01D1",
    "Ocenteredtilde": "\u019F",
    "Ocircle": "\u24C4",
    "Ocircumflex": "\u00D4",
    "Ocircumflexacute": "\u1ED0",
    "Ocircumflexdotbelow": "\u1ED8",
    "Ocircumflexgrave": "\u1ED2",
    "Ocircumflexhookabove": "\u1ED4",
    "Ocircumflexsmall": "\uF7F4",
    "Ocircumflextilde": "\u1ED6",
    "Ocyrillic": "\u041E",
    "Odblacute": "\u0150",
    "Odblgrave": "\u020C",
    "Odieresis": "\u00D6",
    "Odieresiscyrillic": "\u04E6",
    "Odieresissmall": "\uF7F6",
    "Odotbelow": "\u1ECC",
    "Ogoneksmall": "\uF6FB",
    "Ograve": "\u00D2",
    "Ogravesmall": "\uF7F2",
    "Oharmenian": "\u0555",
    "Ohm": "\u2126",
    "Ohookabove": "\u1ECE",
    "Ohorn": "\u01A0",
    "Ohornacute": "\u1EDA",
    "Ohorndotbelow": "\u1EE2",
    "Ohorngrave": "\u1EDC",
    "Ohornhookabove": "\u1EDE",
    "Ohorntilde": "\u1EE0",
    "Ohungarumlaut": "\u0150",
    "Oi": "\u01A2",
    "Oinvertedbreve": "\u020E",
    "Omacron": "\u014C",
    "Omacronacute": "\u1E52",
    "Omacrongrave": "\u1E50",
    "Omega": "\u2126",
    "Omegacyrillic": "\u0460",
    "Omegagreek": "\u03A9",
    "Omegaroundcyrillic": "\u047A",
    "Omegatitlocyrillic": "\u047C",
    "Omegatonos": "\u038F",
    "Omicron": "\u039F",
    "Omicrontonos": "\u038C",
    "Omonospace": "\uFF2F",
    "Oneroman": "\u2160",
    "Oogonek": "\u01EA",
    "Oogonekmacron": "\u01EC",
    "Oopen": "\u0186",
    "Oslash": "\u00D8",
    "Oslashacute": "\u01FE",
    "Oslashsmall": "\uF7F8",
    "Osmall": "\uF76F",
    "Ostrokeacute": "\u01FE",
    "Otcyrillic": "\u047E",
    "Otilde": "\u00D5",
    "Otildeacute": "\u1E4C",
    "Otildedieresis": "\u1E4E",
    "Otildesmall": "\uF7F5",
    "P": "\u0050",
    "Pacute": "\u1E54",
    "Pcircle": "\u24C5",
    "Pdotaccent": "\u1E56",
    "Pecyrillic": "\u041F",
    "Peharmenian": "\u054A",
    "Pemiddlehookcyrillic": "\u04A6",
    "Phi": "\u03A6",
    "Phook": "\u01A4",
    "Pi": "\u03A0",
    "Piwrarmenian": "\u0553",
    "Pmonospace": "\uFF30",
    "Psi": "\u03A8",
    "Psicyrillic": "\u0470",
    "Psmall": "\uF770",
    "Q": "\u0051",
    "Qcircle": "\u24C6",
    "Qmonospace": "\uFF31",
    "Qsmall": "\uF771",
    "R": "\u0052",
    "Raarmenian": "\u054C",
    "Racute": "\u0154",
    "Rcaron": "\u0158",
    "Rcedilla": "\u0156",
    "Rcircle": "\u24C7",
    "Rcommaaccent": "\u0156",
    "Rdblgrave": "\u0210",
    "Rdotaccent": "\u1E58",
    "Rdotbelow": "\u1E5A",
    "Rdotbelowmacron": "\u1E5C",
    "Reharmenian": "\u0550",
    "Rfraktur": "\u211C",
    "Rho": "\u03A1",
    "Ringsmall": "\uF6FC",
    "Rinvertedbreve": "\u0212",
    "Rlinebelow": "\u1E5E",
    "Rmonospace": "\uFF32",
    "Rsmall": "\uF772",
    "Rsmallinverted": "\u0281",
    "Rsmallinvertedsuperior": "\u02B6",
    "S": "\u0053",
    "SF010000": "\u250C",
    "SF020000": "\u2514",
    "SF030000": "\u2510",
    "SF040000": "\u2518",
    "SF050000": "\u253C",
    "SF060000": "\u252C",
    "SF070000": "\u2534",
    "SF080000": "\u251C",
    "SF090000": "\u2524",
    "SF100000": "\u2500",
    "SF110000": "\u2502",
    "SF190000": "\u2561",
    "SF200000": "\u2562",
    "SF210000": "\u2556",
    "SF220000": "\u2555",
    "SF230000": "\u2563",
    "SF240000": "\u2551",
    "SF250000": "\u2557",
    "SF260000": "\u255D",
    "SF270000": "\u255C",
    "SF280000": "\u255B",
    "SF360000": "\u255E",
    "SF370000": "\u255F",
    "SF380000": "\u255A",
    "SF390000": "\u2554",
    "SF400000": "\u2569",
    "SF410000": "\u2566",
    "SF420000": "\u2560",
    "SF430000": "\u2550",
    "SF440000": "\u256C",
    "SF450000": "\u2567",
    "SF460000": "\u2568",
    "SF470000": "\u2564",
    "SF480000": "\u2565",
    "SF490000": "\u2559",
    "SF500000": "\u2558",
    "SF510000": "\u2552",
    "SF520000": "\u2553",
    "SF530000": "\u256B",
    "SF540000": "\u256A",
    "Sacute": "\u015A",
    "Sacutedotaccent": "\u1E64",
    "Sampigreek": "\u03E0",
    "Scaron": "\u0160",
    "Scarondotaccent": "\u1E66",
    "Scaronsmall": "\uF6FD",
    "Scedilla": "\u015E",
    "Schwa": "\u018F",
    "Schwacyrillic": "\u04D8",
    "Schwadieresiscyrillic": "\u04DA",
    "Scircle": "\u24C8",
    "Scircumflex": "\u015C",
    "Scommaaccent": "\u0218",
    "Sdotaccent": "\u1E60",
    "Sdotbelow": "\u1E62",
    "Sdotbelowdotaccent": "\u1E68",
    "Seharmenian": "\u054D",
    "Sevenroman": "\u2166",
    "Shaarmenian": "\u0547",
    "Shacyrillic": "\u0428",
    "Shchacyrillic": "\u0429",
    "Sheicoptic": "\u03E2",
    "Shhacyrillic": "\u04BA",
    "Shimacoptic": "\u03EC",
    "Sigma": "\u03A3",
    "Sixroman": "\u2165",
    "Smonospace": "\uFF33",
    "Softsigncyrillic": "\u042C",
    "Ssmall": "\uF773",
    "Stigmagreek": "\u03DA",
    "T": "\u0054",
    "Tau": "\u03A4",
    "Tbar": "\u0166",
    "Tcaron": "\u0164",
    "Tcedilla": "\u0162",
    "Tcircle": "\u24C9",
    "Tcircumflexbelow": "\u1E70",
    "Tcommaaccent": "\u0162",
    "Tdotaccent": "\u1E6A",
    "Tdotbelow": "\u1E6C",
    "Tecyrillic": "\u0422",
    "Tedescendercyrillic": "\u04AC",
    "Tenroman": "\u2169",
    "Tetsecyrillic": "\u04B4",
    "Theta": "\u0398",
    "Thook": "\u01AC",
    "Thorn": "\u00DE",
    "Thornsmall": "\uF7FE",
    "Threeroman": "\u2162",
    "Tildesmall": "\uF6FE",
    "Tiwnarmenian": "\u054F",
    "Tlinebelow": "\u1E6E",
    "Tmonospace": "\uFF34",
    "Toarmenian": "\u0539",
    "Tonefive": "\u01BC",
    "Tonesix": "\u0184",
    "Tonetwo": "\u01A7",
    "Tretroflexhook": "\u01AE",
    "Tsecyrillic": "\u0426",
    "Tshecyrillic": "\u040B",
    "Tsmall": "\uF774",
    "Twelveroman": "\u216B",
    "Tworoman": "\u2161",
    "U": "\u0055",
    "Uacute": "\u00DA",
    "Uacutesmall": "\uF7FA",
    "Ubreve": "\u016C",
    "Ucaron": "\u01D3",
    "Ucircle": "\u24CA",
    "Ucircumflex": "\u00DB",
    "Ucircumflexbelow": "\u1E76",
    "Ucircumflexsmall": "\uF7FB",
    "Ucyrillic": "\u0423",
    "Udblacute": "\u0170",
    "Udblgrave": "\u0214",
    "Udieresis": "\u00DC",
    "Udieresisacute": "\u01D7",
    "Udieresisbelow": "\u1E72",
    "Udieresiscaron": "\u01D9",
    "Udieresiscyrillic": "\u04F0",
    "Udieresisgrave": "\u01DB",
    "Udieresismacron": "\u01D5",
    "Udieresissmall": "\uF7FC",
    "Udotbelow": "\u1EE4",
    "Ugrave": "\u00D9",
    "Ugravesmall": "\uF7F9",
    "Uhookabove": "\u1EE6",
    "Uhorn": "\u01AF",
    "Uhornacute": "\u1EE8",
    "Uhorndotbelow": "\u1EF0",
    "Uhorngrave": "\u1EEA",
    "Uhornhookabove": "\u1EEC",
    "Uhorntilde": "\u1EEE",
    "Uhungarumlaut": "\u0170",
    "Uhungarumlautcyrillic": "\u04F2",
    "Uinvertedbreve": "\u0216",
    "Ukcyrillic": "\u0478",
    "Umacron": "\u016A",
    "Umacroncyrillic": "\u04EE",
    "Umacrondieresis": "\u1E7A",
    "Umonospace": "\uFF35",
    "Uogonek": "\u0172",
    "Upsilon": "\u03A5",
    "Upsilon1": "\u03D2",
    "Upsilonacutehooksymbolgreek": "\u03D3",
    "Upsilonafrican": "\u01B1",
    "Upsilondieresis": "\u03AB",
    "Upsilondieresishooksymbolgreek": "\u03D4",
    "Upsilonhooksymbol": "\u03D2",
    "Upsilontonos": "\u038E",
    "Uring": "\u016E",
    "Ushortcyrillic": "\u040E",
    "Usmall": "\uF775",
    "Ustraightcyrillic": "\u04AE",
    "Ustraightstrokecyrillic": "\u04B0",
    "Utilde": "\u0168",
    "Utildeacute": "\u1E78",
    "Utildebelow": "\u1E74",
    "V": "\u0056",
    "Vcircle": "\u24CB",
    "Vdotbelow": "\u1E7E",
    "Vecyrillic": "\u0412",
    "Vewarmenian": "\u054E",
    "Vhook": "\u01B2",
    "Vmonospace": "\uFF36",
    "Voarmenian": "\u0548",
    "Vsmall": "\uF776",
    "Vtilde": "\u1E7C",
    "W": "\u0057",
    "Wacute": "\u1E82",
    "Wcircle": "\u24CC",
    "Wcircumflex": "\u0174",
    "Wdieresis": "\u1E84",
    "Wdotaccent": "\u1E86",
    "Wdotbelow": "\u1E88",
    "Wgrave": "\u1E80",
    "Wmonospace": "\uFF37",
    "Wsmall": "\uF777",
    "X": "\u0058",
    "Xcircle": "\u24CD",
    "Xdieresis": "\u1E8C",
    "Xdotaccent": "\u1E8A",
    "Xeharmenian": "\u053D",
    "Xi": "\u039E",
    "Xmonospace": "\uFF38",
    "Xsmall": "\uF778",
    "Y": "\u0059",
    "Yacute": "\u00DD",
    "Yacutesmall": "\uF7FD",
    "Yatcyrillic": "\u0462",
    "Ycircle": "\u24CE",
    "Ycircumflex": "\u0176",
    "Ydieresis": "\u0178",
    "Ydieresissmall": "\uF7FF",
    "Ydotaccent": "\u1E8E",
    "Ydotbelow": "\u1EF4",
    "Yericyrillic": "\u042B",
    "Yerudieresiscyrillic": "\u04F8",
    "Ygrave": "\u1EF2",
    "Yhook": "\u01B3",
    "Yhookabove": "\u1EF6",
    "Yiarmenian": "\u0545",
    "Yicyrillic": "\u0407",
    "Yiwnarmenian": "\u0552",
    "Ymonospace": "\uFF39",
    "Ysmall": "\uF779",
    "Ytilde": "\u1EF8",
    "Yusbigcyrillic": "\u046A",
    "Yusbigiotifiedcyrillic": "\u046C",
    "Yuslittlecyrillic": "\u0466",
    "Yuslittleiotifiedcyrillic": "\u0468",
    "Z": "\u005A",
    "Zaarmenian": "\u0536",
    "Zacute": "\u0179",
    "Zcaron": "\u017D",
    "Zcaronsmall": "\uF6FF",
    "Zcircle": "\u24CF",
    "Zcircumflex": "\u1E90",
    "Zdot": "\u017B",
    "Zdotaccent": "\u017B",
    "Zdotbelow": "\u1E92",
    "Zecyrillic": "\u0417",
    "Zedescendercyrillic": "\u0498",
    "Zedieresiscyrillic": "\u04DE",
    "Zeta": "\u0396",
    "Zhearmenian": "\u053A",
    "Zhebrevecyrillic": "\u04C1",
    "Zhecyrillic": "\u0416",
    "Zhedescendercyrillic": "\u0496",
    "Zhedieresiscyrillic": "\u04DC",
    "Zlinebelow": "\u1E94",
    "Zmonospace": "\uFF3A",
    "Zsmall": "\uF77A",
    "Zstroke": "\u01B5",
    "a": "\u0061",
    "aabengali": "\u0986",
    "aacute": "\u00E1",
    "aadeva": "\u0906",
    "aagujarati": "\u0A86",
    "aagurmukhi": "\u0A06",
    "aamatragurmukhi": "\u0A3E",
    "aarusquare": "\u3303",
    "aavowelsignbengali": "\u09BE",
    "aavowelsigndeva": "\u093E",
    "aavowelsigngujarati": "\u0ABE",
    "abbreviationmarkarmenian": "\u055F",
    "abbreviationsigndeva": "\u0970",
    "abengali": "\u0985",
    "abopomofo": "\u311A",
    "abreve": "\u0103",
    "abreveacute": "\u1EAF",
    "abrevecyrillic": "\u04D1",
    "abrevedotbelow": "\u1EB7",
    "abrevegrave": "\u1EB1",
    "abrevehookabove": "\u1EB3",
    "abrevetilde": "\u1EB5",
    "acaron": "\u01CE",
    "acircle": "\u24D0",
    "acircumflex": "\u00E2",
    "acircumflexacute": "\u1EA5",
    "acircumflexdotbelow": "\u1EAD",
    "acircumflexgrave": "\u1EA7",
    "acircumflexhookabove": "\u1EA9",
    "acircumflextilde": "\u1EAB",
    "acute": "\u00B4",
    "acutebelowcmb": "\u0317",
    "acutecmb": "\u0301",
    "acutecomb": "\u0301",
    "acutedeva": "\u0954",
    "acutelowmod": "\u02CF",
    "acutetonecmb": "\u0341",
    "acyrillic": "\u0430",
    "adblgrave": "\u0201",
    "addakgurmukhi": "\u0A71",
    "adeva": "\u0905",
    "adieresis": "\u00E4",
    "adieresiscyrillic": "\u04D3",
    "adieresismacron": "\u01DF",
    "adotbelow": "\u1EA1",
    "adotmacron": "\u01E1",
    "ae": "\u00E6",
    "aeacute": "\u01FD",
    "aekorean": "\u3150",
    "aemacron": "\u01E3",
    "afii00208": "\u2015",
    "afii08941": "\u20A4",
    "afii10017": "\u0410",
    "afii10018": "\u0411",
    "afii10019": "\u0412",
    "afii10020": "\u0413",
    "afii10021": "\u0414",
    "afii10022": "\u0415",
    "afii10023": "\u0401",
    "afii10024": "\u0416",
    "afii10025": "\u0417",
    "afii10026": "\u0418",
    "afii10027": "\u0419",
    "afii10028": "\u041A",
    "afii10029": "\u041B",
    "afii10030": "\u041C",
    "afii10031": "\u041D",
    "afii10032": "\u041E",
    "afii10033": "\u041F",
    "afii10034": "\u0420",
    "afii10035": "\u0421",
    "afii10036": "\u0422",
    "afii10037": "\u0423",
    "afii10038": "\u0424",
    "afii10039": "\u0425",
    "afii10040": "\u0426",
    "afii10041": "\u0427",
    "afii10042": "\u0428",
    "afii10043": "\u0429",
    "afii10044": "\u042A",
    "afii10045": "\u042B",
    "afii10046": "\u042C",
    "afii10047": "\u042D",
    "afii10048": "\u042E",
    "afii10049": "\u042F",
    "afii10050": "\u0490",
    "afii10051": "\u0402",
    "afii10052": "\u0403",
    "afii10053": "\u0404",
    "afii10054": "\u0405",
    "afii10055": "\u0406",
    "afii10056": "\u0407",
    "afii10057": "\u0408",
    "afii10058": "\u0409",
    "afii10059": "\u040A",
    "afii10060": "\u040B",
    "afii10061": "\u040C",
    "afii10062": "\u040E",
    "afii10063": "\uF6C4",
    "afii10064": "\uF6C5",
    "afii10065": "\u0430",
    "afii10066": "\u0431",
    "afii10067": "\u0432",
    "afii10068": "\u0433",
    "afii10069": "\u0434",
    "afii10070": "\u0435",
    "afii10071": "\u0451",
    "afii10072": "\u0436",
    "afii10073": "\u0437",
    "afii10074": "\u0438",
    "afii10075": "\u0439",
    "afii10076": "\u043A",
    "afii10077": "\u043B",
    "afii10078": "\u043C",
    "afii10079": "\u043D",
    "afii10080": "\u043E",
    "afii10081": "\u043F",
    "afii10082": "\u0440",
    "afii10083": "\u0441",
    "afii10084": "\u0442",
    "afii10085": "\u0443",
    "afii10086": "\u0444",
    "afii10087": "\u0445",
    "afii10088": "\u0446",
    "afii10089": "\u0447",
    "afii10090": "\u0448",
    "afii10091": "\u0449",
    "afii10092": "\u044A",
    "afii10093": "\u044B",
    "afii10094": "\u044C",
    "afii10095": "\u044D",
    "afii10096": "\u044E",
    "afii10097": "\u044F",
    "afii10098": "\u0491",
    "afii10099": "\u0452",
    "afii10100": "\u0453",
    "afii10101": "\u0454",
    "afii10102": "\u0455",
    "afii10103": "\u0456",
    "afii10104": "\u0457",
    "afii10105": "\u0458",
    "afii10106": "\u0459",
    "afii10107": "\u045A",
    "afii10108": "\u045B",
    "afii10109": "\u045C",
    "afii10110": "\u045E",
    "afii10145": "\u040F",
    "afii10146": "\u0462",
    "afii10147": "\u0472",
    "afii10148": "\u0474",
    "afii10192": "\uF6C6",
    "afii10193": "\u045F",
    "afii10194": "\u0463",
    "afii10195": "\u0473",
    "afii10196": "\u0475",
    "afii10831": "\uF6C7",
    "afii10832": "\uF6C8",
    "afii10846": "\u04D9",
    "afii299": "\u200E",
    "afii300": "\u200F",
    "afii301": "\u200D",
    "afii57381": "\u066A",
    "afii57388": "\u060C",
    "afii57392": "\u0660",
    "afii57393": "\u0661",
    "afii57394": "\u0662",
    "afii57395": "\u0663",
    "afii57396": "\u0664",
    "afii57397": "\u0665",
    "afii57398": "\u0666",
    "afii57399": "\u0667",
    "afii57400": "\u0668",
    "afii57401": "\u0669",
    "afii57403": "\u061B",
    "afii57407": "\u061F",
    "afii57409": "\u0621",
    "afii57410": "\u0622",
    "afii57411": "\u0623",
    "afii57412": "\u0624",
    "afii57413": "\u0625",
    "afii57414": "\u0626",
    "afii57415": "\u0627",
    "afii57416": "\u0628",
    "afii57417": "\u0629",
    "afii57418": "\u062A",
    "afii57419": "\u062B",
    "afii57420": "\u062C",
    "afii57421": "\u062D",
    "afii57422": "\u062E",
    "afii57423": "\u062F",
    "afii57424": "\u0630",
    "afii57425": "\u0631",
    "afii57426": "\u0632",
    "afii57427": "\u0633",
    "afii57428": "\u0634",
    "afii57429": "\u0635",
    "afii57430": "\u0636",
    "afii57431": "\u0637",
    "afii57432": "\u0638",
    "afii57433": "\u0639",
    "afii57434": "\u063A",
    "afii57440": "\u0640",
    "afii57441": "\u0641",
    "afii57442": "\u0642",
    "afii57443": "\u0643",
    "afii57444": "\u0644",
    "afii57445": "\u0645",
    "afii57446": "\u0646",
    "afii57448": "\u0648",
    "afii57449": "\u0649",
    "afii57450": "\u064A",
    "afii57451": "\u064B",
    "afii57452": "\u064C",
    "afii57453": "\u064D",
    "afii57454": "\u064E",
    "afii57455": "\u064F",
    "afii57456": "\u0650",
    "afii57457": "\u0651",
    "afii57458": "\u0652",
    "afii57470": "\u0647",
    "afii57505": "\u06A4",
    "afii57506": "\u067E",
    "afii57507": "\u0686",
    "afii57508": "\u0698",
    "afii57509": "\u06AF",
    "afii57511": "\u0679",
    "afii57512": "\u0688",
    "afii57513": "\u0691",
    "afii57514": "\u06BA",
    "afii57519": "\u06D2",
    "afii57534": "\u06D5",
    "afii57636": "\u20AA",
    "afii57645": "\u05BE",
    "afii57658": "\u05C3",
    "afii57664": "\u05D0",
    "afii57665": "\u05D1",
    "afii57666": "\u05D2",
    "afii57667": "\u05D3",
    "afii57668": "\u05D4",
    "afii57669": "\u05D5",
    "afii57670": "\u05D6",
    "afii57671": "\u05D7",
    "afii57672": "\u05D8",
    "afii57673": "\u05D9",
    "afii57674": "\u05DA",
    "afii57675": "\u05DB",
    "afii57676": "\u05DC",
    "afii57677": "\u05DD",
    "afii57678": "\u05DE",
    "afii57679": "\u05DF",
    "afii57680": "\u05E0",
    "afii57681": "\u05E1",
    "afii57682": "\u05E2",
    "afii57683": "\u05E3",
    "afii57684": "\u05E4",
    "afii57685": "\u05E5",
    "afii57686": "\u05E6",
    "afii57687": "\u05E7",
    "afii57688": "\u05E8",
    "afii57689": "\u05E9",
    "afii57690": "\u05EA",
    "afii57694": "\uFB2A",
    "afii57695": "\uFB2B",
    "afii57700": "\uFB4B",
    "afii57705": "\uFB1F",
    "afii57716": "\u05F0",
    "afii57717": "\u05F1",
    "afii57718": "\u05F2",
    "afii57723": "\uFB35",
    "afii57793": "\u05B4",
    "afii57794": "\u05B5",
    "afii57795": "\u05B6",
    "afii57796": "\u05BB",
    "afii57797": "\u05B8",
    "afii57798": "\u05B7",
    "afii57799": "\u05B0",
    "afii57800": "\u05B2",
    "afii57801": "\u05B1",
    "afii57802": "\u05B3",
    "afii57803": "\u05C2",
    "afii57804": "\u05C1",
    "afii57806": "\u05B9",
    "afii57807": "\u05BC",
    "afii57839": "\u05BD",
    "afii57841": "\u05BF",
    "afii57842": "\u05C0",
    "afii57929": "\u02BC",
    "afii61248": "\u2105",
    "afii61289": "\u2113",
    "afii61352": "\u2116",
    "afii61573": "\u202C",
    "afii61574": "\u202D",
    "afii61575": "\u202E",
    "afii61664": "\u200C",
    "afii63167": "\u066D",
    "afii64937": "\u02BD",
    "agrave": "\u00E0",
    "agujarati": "\u0A85",
    "agurmukhi": "\u0A05",
    "ahiragana": "\u3042",
    "ahookabove": "\u1EA3",
    "aibengali": "\u0990",
    "aibopomofo": "\u311E",
    "aideva": "\u0910",
    "aiecyrillic": "\u04D5",
    "aigujarati": "\u0A90",
    "aigurmukhi": "\u0A10",
    "aimatragurmukhi": "\u0A48",
    "ainarabic": "\u0639",
    "ainfinalarabic": "\uFECA",
    "aininitialarabic": "\uFECB",
    "ainmedialarabic": "\uFECC",
    "ainvertedbreve": "\u0203",
    "aivowelsignbengali": "\u09C8",
    "aivowelsigndeva": "\u0948",
    "aivowelsigngujarati": "\u0AC8",
    "akatakana": "\u30A2",
    "akatakanahalfwidth": "\uFF71",
    "akorean": "\u314F",
    "alef": "\u05D0",
    "alefarabic": "\u0627",
    "alefdageshhebrew": "\uFB30",
    "aleffinalarabic": "\uFE8E",
    "alefhamzaabovearabic": "\u0623",
    "alefhamzaabovefinalarabic": "\uFE84",
    "alefhamzabelowarabic": "\u0625",
    "alefhamzabelowfinalarabic": "\uFE88",
    "alefhebrew": "\u05D0",
    "aleflamedhebrew": "\uFB4F",
    "alefmaddaabovearabic": "\u0622",
    "alefmaddaabovefinalarabic": "\uFE82",
    "alefmaksuraarabic": "\u0649",
    "alefmaksurafinalarabic": "\uFEF0",
    "alefmaksurainitialarabic": "\uFEF3",
    "alefmaksuramedialarabic": "\uFEF4",
    "alefpatahhebrew": "\uFB2E",
    "alefqamatshebrew": "\uFB2F",
    "aleph": "\u2135",
    "allequal": "\u224C",
    "alpha": "\u03B1",
    "alphatonos": "\u03AC",
    "amacron": "\u0101",
    "amonospace": "\uFF41",
    "ampersand": "\u0026",
    "ampersandmonospace": "\uFF06",
    "ampersandsmall": "\uF726",
    "amsquare": "\u33C2",
    "anbopomofo": "\u3122",
    "angbopomofo": "\u3124",
    "angkhankhuthai": "\u0E5A",
    "angle": "\u2220",
    "anglebracketleft": "\u3008",
    "anglebracketleftvertical": "\uFE3F",
    "anglebracketright": "\u3009",
    "anglebracketrightvertical": "\uFE40",
    "angleleft": "\u2329",
    "angleright": "\u232A",
    "angstrom": "\u212B",
    "anoteleia": "\u0387",
    "anudattadeva": "\u0952",
    "anusvarabengali": "\u0982",
    "anusvaradeva": "\u0902",
    "anusvaragujarati": "\u0A82",
    "aogonek": "\u0105",
    "apaatosquare": "\u3300",
    "aparen": "\u249C",
    "apostrophearmenian": "\u055A",
    "apostrophemod": "\u02BC",
    "apple": "\uF8FF",
    "approaches": "\u2250",
    "approxequal": "\u2248",
    "approxequalorimage": "\u2252",
    "approximatelyequal": "\u2245",
    "araeaekorean": "\u318E",
    "araeakorean": "\u318D",
    "arc": "\u2312",
    "arighthalfring": "\u1E9A",
    "aring": "\u00E5",
    "aringacute": "\u01FB",
    "aringbelow": "\u1E01",
    "arrowboth": "\u2194",
    "arrowdashdown": "\u21E3",
    "arrowdashleft": "\u21E0",
    "arrowdashright": "\u21E2",
    "arrowdashup": "\u21E1",
    "arrowdblboth": "\u21D4",
    "arrowdbldown": "\u21D3",
    "arrowdblleft": "\u21D0",
    "arrowdblright": "\u21D2",
    "arrowdblup": "\u21D1",
    "arrowdown": "\u2193",
    "arrowdownleft": "\u2199",
    "arrowdownright": "\u2198",
    "arrowdownwhite": "\u21E9",
    "arrowheaddownmod": "\u02C5",
    "arrowheadleftmod": "\u02C2",
    "arrowheadrightmod": "\u02C3",
    "arrowheadupmod": "\u02C4",
    "arrowhorizex": "\uF8E7",
    "arrowleft": "\u2190",
    "arrowleftdbl": "\u21D0",
    "arrowleftdblstroke": "\u21CD",
    "arrowleftoverright": "\u21C6",
    "arrowleftwhite": "\u21E6",
    "arrowright": "\u2192",
    "arrowrightdblstroke": "\u21CF",
    "arrowrightheavy": "\u279E",
    "arrowrightoverleft": "\u21C4",
    "arrowrightwhite": "\u21E8",
    "arrowtableft": "\u21E4",
    "arrowtabright": "\u21E5",
    "arrowup": "\u2191",
    "arrowupdn": "\u2195",
    "arrowupdnbse": "\u21A8",
    "arrowupdownbase": "\u21A8",
    "arrowupleft": "\u2196",
    "arrowupleftofdown": "\u21C5",
    "arrowupright": "\u2197",
    "arrowupwhite": "\u21E7",
    "arrowvertex": "\uF8E6",
    "asciicircum": "\u005E",
    "asciicircummonospace": "\uFF3E",
    "asciitilde": "\u007E",
    "asciitildemonospace": "\uFF5E",
    "ascript": "\u0251",
    "ascriptturned": "\u0252",
    "asmallhiragana": "\u3041",
    "asmallkatakana": "\u30A1",
    "asmallkatakanahalfwidth": "\uFF67",
    "asterisk": "\u002A",
    "asteriskaltonearabic": "\u066D",
    "asteriskarabic": "\u066D",
    "asteriskmath": "\u2217",
    "asteriskmonospace": "\uFF0A",
    "asterisksmall": "\uFE61",
    "asterism": "\u2042",
    "asuperior": "\uF6E9",
    "asymptoticallyequal": "\u2243",
    "at": "\u0040",
    "atilde": "\u00E3",
    "atmonospace": "\uFF20",
    "atsmall": "\uFE6B",
    "aturned": "\u0250",
    "aubengali": "\u0994",
    "aubopomofo": "\u3120",
    "audeva": "\u0914",
    "augujarati": "\u0A94",
    "augurmukhi": "\u0A14",
    "aulengthmarkbengali": "\u09D7",
    "aumatragurmukhi": "\u0A4C",
    "auvowelsignbengali": "\u09CC",
    "auvowelsigndeva": "\u094C",
    "auvowelsigngujarati": "\u0ACC",
    "avagrahadeva": "\u093D",
    "aybarmenian": "\u0561",
    "ayin": "\u05E2",
    "ayinaltonehebrew": "\uFB20",
    "ayinhebrew": "\u05E2",
    "b": "\u0062",
    "babengali": "\u09AC",
    "backslash": "\u005C",
    "backslashmonospace": "\uFF3C",
    "badeva": "\u092C",
    "bagujarati": "\u0AAC",
    "bagurmukhi": "\u0A2C",
    "bahiragana": "\u3070",
    "bahtthai": "\u0E3F",
    "bakatakana": "\u30D0",
    "bar": "\u007C",
    "barmonospace": "\uFF5C",
    "bbopomofo": "\u3105",
    "bcircle": "\u24D1",
    "bdotaccent": "\u1E03",
    "bdotbelow": "\u1E05",
    "beamedsixteenthnotes": "\u266C",
    "because": "\u2235",
    "becyrillic": "\u0431",
    "beharabic": "\u0628",
    "behfinalarabic": "\uFE90",
    "behinitialarabic": "\uFE91",
    "behiragana": "\u3079",
    "behmedialarabic": "\uFE92",
    "behmeeminitialarabic": "\uFC9F",
    "behmeemisolatedarabic": "\uFC08",
    "behnoonfinalarabic": "\uFC6D",
    "bekatakana": "\u30D9",
    "benarmenian": "\u0562",
    "bet": "\u05D1",
    "beta": "\u03B2",
    "betasymbolgreek": "\u03D0",
    "betdagesh": "\uFB31",
    "betdageshhebrew": "\uFB31",
    "bethebrew": "\u05D1",
    "betrafehebrew": "\uFB4C",
    "bhabengali": "\u09AD",
    "bhadeva": "\u092D",
    "bhagujarati": "\u0AAD",
    "bhagurmukhi": "\u0A2D",
    "bhook": "\u0253",
    "bihiragana": "\u3073",
    "bikatakana": "\u30D3",
    "bilabialclick": "\u0298",
    "bindigurmukhi": "\u0A02",
    "birusquare": "\u3331",
    "blackcircle": "\u25CF",
    "blackdiamond": "\u25C6",
    "blackdownpointingtriangle": "\u25BC",
    "blackleftpointingpointer": "\u25C4",
    "blackleftpointingtriangle": "\u25C0",
    "blacklenticularbracketleft": "\u3010",
    "blacklenticularbracketleftvertical": "\uFE3B",
    "blacklenticularbracketright": "\u3011",
    "blacklenticularbracketrightvertical": "\uFE3C",
    "blacklowerlefttriangle": "\u25E3",
    "blacklowerrighttriangle": "\u25E2",
    "blackrectangle": "\u25AC",
    "blackrightpointingpointer": "\u25BA",
    "blackrightpointingtriangle": "\u25B6",
    "blacksmallsquare": "\u25AA",
    "blacksmilingface": "\u263B",
    "blacksquare": "\u25A0",
    "blackstar": "\u2605",
    "blackupperlefttriangle": "\u25E4",
    "blackupperrighttriangle": "\u25E5",
    "blackuppointingsmalltriangle": "\u25B4",
    "blackuppointingtriangle": "\u25B2",
    "blank": "\u2423",
    "blinebelow": "\u1E07",
    "block": "\u2588",
    "bmonospace": "\uFF42",
    "bobaimaithai": "\u0E1A",
    "bohiragana": "\u307C",
    "bokatakana": "\u30DC",
    "bparen": "\u249D",
    "bqsquare": "\u33C3",
    "braceex": "\uF8F4",
    "braceleft": "\u007B",
    "braceleftbt": "\uF8F3",
    "braceleftmid": "\uF8F2",
    "braceleftmonospace": "\uFF5B",
    "braceleftsmall": "\uFE5B",
    "bracelefttp": "\uF8F1",
    "braceleftvertical": "\uFE37",
    "braceright": "\u007D",
    "bracerightbt": "\uF8FE",
    "bracerightmid": "\uF8FD",
    "bracerightmonospace": "\uFF5D",
    "bracerightsmall": "\uFE5C",
    "bracerighttp": "\uF8FC",
    "bracerightvertical": "\uFE38",
    "bracketleft": "\u005B",
    "bracketleftbt": "\uF8F0",
    "bracketleftex": "\uF8EF",
    "bracketleftmonospace": "\uFF3B",
    "bracketlefttp": "\uF8EE",
    "bracketright": "\u005D",
    "bracketrightbt": "\uF8FB",
    "bracketrightex": "\uF8FA",
    "bracketrightmonospace": "\uFF3D",
    "bracketrighttp": "\uF8F9",
    "breve": "\u02D8",
    "brevebelowcmb": "\u032E",
    "brevecmb": "\u0306",
    "breveinvertedbelowcmb": "\u032F",
    "breveinvertedcmb": "\u0311",
    "breveinverteddoublecmb": "\u0361",
    "bridgebelowcmb": "\u032A",
    "bridgeinvertedbelowcmb": "\u033A",
    "brokenbar": "\u00A6",
    "bstroke": "\u0180",
    "bsuperior": "\uF6EA",
    "btopbar": "\u0183",
    "buhiragana": "\u3076",
    "bukatakana": "\u30D6",
    "bullet": "\u2022",
    "bulletinverse": "\u25D8",
    "bulletoperator": "\u2219",
    "bullseye": "\u25CE",
    "c": "\u0063",
    "caarmenian": "\u056E",
    "cabengali": "\u099A",
    "cacute": "\u0107",
    "cadeva": "\u091A",
    "cagujarati": "\u0A9A",
    "cagurmukhi": "\u0A1A",
    "calsquare": "\u3388",
    "candrabindubengali": "\u0981",
    "candrabinducmb": "\u0310",
    "candrabindudeva": "\u0901",
    "candrabindugujarati": "\u0A81",
    "capslock": "\u21EA",
    "careof": "\u2105",
    "caron": "\u02C7",
    "caronbelowcmb": "\u032C",
    "caroncmb": "\u030C",
    "carriagereturn": "\u21B5",
    "cbopomofo": "\u3118",
    "ccaron": "\u010D",
    "ccedilla": "\u00E7",
    "ccedillaacute": "\u1E09",
    "ccircle": "\u24D2",
    "ccircumflex": "\u0109",
    "ccurl": "\u0255",
    "cdot": "\u010B",
    "cdotaccent": "\u010B",
    "cdsquare": "\u33C5",
    "cedilla": "\u00B8",
    "cedillacmb": "\u0327",
    "cent": "\u00A2",
    "centigrade": "\u2103",
    "centinferior": "\uF6DF",
    "centmonospace": "\uFFE0",
    "centoldstyle": "\uF7A2",
    "centsuperior": "\uF6E0",
    "chaarmenian": "\u0579",
    "chabengali": "\u099B",
    "chadeva": "\u091B",
    "chagujarati": "\u0A9B",
    "chagurmukhi": "\u0A1B",
    "chbopomofo": "\u3114",
    "cheabkhasiancyrillic": "\u04BD",
    "checkmark": "\u2713",
    "checyrillic": "\u0447",
    "chedescenderabkhasiancyrillic": "\u04BF",
    "chedescendercyrillic": "\u04B7",
    "chedieresiscyrillic": "\u04F5",
    "cheharmenian": "\u0573",
    "chekhakassiancyrillic": "\u04CC",
    "cheverticalstrokecyrillic": "\u04B9",
    "chi": "\u03C7",
    "chieuchacirclekorean": "\u3277",
    "chieuchaparenkorean": "\u3217",
    "chieuchcirclekorean": "\u3269",
    "chieuchkorean": "\u314A",
    "chieuchparenkorean": "\u3209",
    "chochangthai": "\u0E0A",
    "chochanthai": "\u0E08",
    "chochingthai": "\u0E09",
    "chochoethai": "\u0E0C",
    "chook": "\u0188",
    "cieucacirclekorean": "\u3276",
    "cieucaparenkorean": "\u3216",
    "cieuccirclekorean": "\u3268",
    "cieuckorean": "\u3148",
    "cieucparenkorean": "\u3208",
    "cieucuparenkorean": "\u321C",
    "circle": "\u25CB",
    "circlemultiply": "\u2297",
    "circleot": "\u2299",
    "circleplus": "\u2295",
    "circlepostalmark": "\u3036",
    "circlewithlefthalfblack": "\u25D0",
    "circlewithrighthalfblack": "\u25D1",
    "circumflex": "\u02C6",
    "circumflexbelowcmb": "\u032D",
    "circumflexcmb": "\u0302",
    "clear": "\u2327",
    "clickalveolar": "\u01C2",
    "clickdental": "\u01C0",
    "clicklateral": "\u01C1",
    "clickretroflex": "\u01C3",
    "club": "\u2663",
    "clubsuitblack": "\u2663",
    "clubsuitwhite": "\u2667",
    "cmcubedsquare": "\u33A4",
    "cmonospace": "\uFF43",
    "cmsquaredsquare": "\u33A0",
    "coarmenian": "\u0581",
    "colon": "\u003A",
    "colonmonetary": "\u20A1",
    "colonmonospace": "\uFF1A",
    "colonsign": "\u20A1",
    "colonsmall": "\uFE55",
    "colontriangularhalfmod": "\u02D1",
    "colontriangularmod": "\u02D0",
    "comma": "\u002C",
    "commaabovecmb": "\u0313",
    "commaaboverightcmb": "\u0315",
    "commaaccent": "\uF6C3",
    "commaarabic": "\u060C",
    "commaarmenian": "\u055D",
    "commainferior": "\uF6E1",
    "commamonospace": "\uFF0C",
    "commareversedabovecmb": "\u0314",
    "commareversedmod": "\u02BD",
    "commasmall": "\uFE50",
    "commasuperior": "\uF6E2",
    "commaturnedabovecmb": "\u0312",
    "commaturnedmod": "\u02BB",
    "compass": "\u263C",
    "congruent": "\u2245",
    "contourintegral": "\u222E",
    "control": "\u2303",
    "controlACK": "\u0006",
    "controlBEL": "\u0007",
    "controlBS": "\u0008",
    "controlCAN": "\u0018",
    "controlCR": "\u000D",
    "controlDC1": "\u0011",
    "controlDC2": "\u0012",
    "controlDC3": "\u0013",
    "controlDC4": "\u0014",
    "controlDEL": "\u007F",
    "controlDLE": "\u0010",
    "controlEM": "\u0019",
    "controlENQ": "\u0005",
    "controlEOT": "\u0004",
    "controlESC": "\u001B",
    "controlETB": "\u0017",
    "controlETX": "\u0003",
    "controlFF": "\u000C",
    "controlFS": "\u001C",
    "controlGS": "\u001D",
    "controlHT": "\u0009",
    "controlLF": "\u000A",
    "controlNAK": "\u0015",
    "controlRS": "\u001E",
    "controlSI": "\u000F",
    "controlSO": "\u000E",
    "controlSOT": "\u0002",
    "controlSTX": "\u0001",
    "controlSUB": "\u001A",
    "controlSYN": "\u0016",
    "controlUS": "\u001F",
    "controlVT": "\u000B",
    "copyright": "\u00A9",
    "copyrightsans": "\uF8E9",
    "copyrightserif": "\uF6D9",
    "cornerbracketleft": "\u300C",
    "cornerbracketlefthalfwidth": "\uFF62",
    "cornerbracketleftvertical": "\uFE41",
    "cornerbracketright": "\u300D",
    "cornerbracketrighthalfwidth": "\uFF63",
    "cornerbracketrightvertical": "\uFE42",
    "corporationsquare": "\u337F",
    "cosquare": "\u33C7",
    "coverkgsquare": "\u33C6",
    "cparen": "\u249E",
    "cruzeiro": "\u20A2",
    "cstretched": "\u0297",
    "curlyand": "\u22CF",
    "curlyor": "\u22CE",
    "currency": "\u00A4",
    "cyrBreve": "\uF6D1",
    "cyrFlex": "\uF6D2",
    "cyrbreve": "\uF6D4",
    "cyrflex": "\uF6D5",
    "d": "\u0064",
    "daarmenian": "\u0564",
    "dabengali": "\u09A6",
    "dadarabic": "\u0636",
    "dadeva": "\u0926",
    "dadfinalarabic": "\uFEBE",
    "dadinitialarabic": "\uFEBF",
    "dadmedialarabic": "\uFEC0",
    "dagesh": "\u05BC",
    "dageshhebrew": "\u05BC",
    "dagger": "\u2020",
    "daggerdbl": "\u2021",
    "dagujarati": "\u0AA6",
    "dagurmukhi": "\u0A26",
    "dahiragana": "\u3060",
    "dakatakana": "\u30C0",
    "dalarabic": "\u062F",
    "dalet": "\u05D3",
    "daletdagesh": "\uFB33",
    "daletdageshhebrew": "\uFB33",
    "dalethatafpatah": "\u05D3\u05B2",
    "dalethatafpatahhebrew": "\u05D3\u05B2",
    "dalethatafsegol": "\u05D3\u05B1",
    "dalethatafsegolhebrew": "\u05D3\u05B1",
    "dalethebrew": "\u05D3",
    "dalethiriq": "\u05D3\u05B4",
    "dalethiriqhebrew": "\u05D3\u05B4",
    "daletholam": "\u05D3\u05B9",
    "daletholamhebrew": "\u05D3\u05B9",
    "daletpatah": "\u05D3\u05B7",
    "daletpatahhebrew": "\u05D3\u05B7",
    "daletqamats": "\u05D3\u05B8",
    "daletqamatshebrew": "\u05D3\u05B8",
    "daletqubuts": "\u05D3\u05BB",
    "daletqubutshebrew": "\u05D3\u05BB",
    "daletsegol": "\u05D3\u05B6",
    "daletsegolhebrew": "\u05D3\u05B6",
    "daletsheva": "\u05D3\u05B0",
    "daletshevahebrew": "\u05D3\u05B0",
    "dalettsere": "\u05D3\u05B5",
    "dalettserehebrew": "\u05D3\u05B5",
    "dalfinalarabic": "\uFEAA",
    "dammaarabic": "\u064F",
    "dammalowarabic": "\u064F",
    "dammatanaltonearabic": "\u064C",
    "dammatanarabic": "\u064C",
    "danda": "\u0964",
    "dargahebrew": "\u05A7",
    "dargalefthebrew": "\u05A7",
    "dasiapneumatacyrilliccmb": "\u0485",
    "dblGrave": "\uF6D3",
    "dblanglebracketleft": "\u300A",
    "dblanglebracketleftvertical": "\uFE3D",
    "dblanglebracketright": "\u300B",
    "dblanglebracketrightvertical": "\uFE3E",
    "dblarchinvertedbelowcmb": "\u032B",
    "dblarrowleft": "\u21D4",
    "dblarrowright": "\u21D2",
    "dbldanda": "\u0965",
    "dblgrave": "\uF6D6",
    "dblgravecmb": "\u030F",
    "dblintegral": "\u222C",
    "dbllowline": "\u2017",
    "dbllowlinecmb": "\u0333",
    "dbloverlinecmb": "\u033F",
    "dblprimemod": "\u02BA",
    "dblverticalbar": "\u2016",
    "dblverticallineabovecmb": "\u030E",
    "dbopomofo": "\u3109",
    "dbsquare": "\u33C8",
    "dcaron": "\u010F",
    "dcedilla": "\u1E11",
    "dcircle": "\u24D3",
    "dcircumflexbelow": "\u1E13",
    "dcroat": "\u0111",
    "ddabengali": "\u09A1",
    "ddadeva": "\u0921",
    "ddagujarati": "\u0AA1",
    "ddagurmukhi": "\u0A21",
    "ddalarabic": "\u0688",
    "ddalfinalarabic": "\uFB89",
    "dddhadeva": "\u095C",
    "ddhabengali": "\u09A2",
    "ddhadeva": "\u0922",
    "ddhagujarati": "\u0AA2",
    "ddhagurmukhi": "\u0A22",
    "ddotaccent": "\u1E0B",
    "ddotbelow": "\u1E0D",
    "decimalseparatorarabic": "\u066B",
    "decimalseparatorpersian": "\u066B",
    "decyrillic": "\u0434",
    "degree": "\u00B0",
    "dehihebrew": "\u05AD",
    "dehiragana": "\u3067",
    "deicoptic": "\u03EF",
    "dekatakana": "\u30C7",
    "deleteleft": "\u232B",
    "deleteright": "\u2326",
    "delta": "\u03B4",
    "deltaturned": "\u018D",
    "denominatorminusonenumeratorbengali": "\u09F8",
    "dezh": "\u02A4",
    "dhabengali": "\u09A7",
    "dhadeva": "\u0927",
    "dhagujarati": "\u0AA7",
    "dhagurmukhi": "\u0A27",
    "dhook": "\u0257",
    "dialytikatonos": "\u0385",
    "dialytikatonoscmb": "\u0344",
    "diamond": "\u2666",
    "diamondsuitwhite": "\u2662",
    "dieresis": "\u00A8",
    "dieresisacute": "\uF6D7",
    "dieresisbelowcmb": "\u0324",
    "dieresiscmb": "\u0308",
    "dieresisgrave": "\uF6D8",
    "dieresistonos": "\u0385",
    "dihiragana": "\u3062",
    "dikatakana": "\u30C2",
    "dittomark": "\u3003",
    "divide": "\u00F7",
    "divides": "\u2223",
    "divisionslash": "\u2215",
    "djecyrillic": "\u0452",
    "dkshade": "\u2593",
    "dlinebelow": "\u1E0F",
    "dlsquare": "\u3397",
    "dmacron": "\u0111",
    "dmonospace": "\uFF44",
    "dnblock": "\u2584",
    "dochadathai": "\u0E0E",
    "dodekthai": "\u0E14",
    "dohiragana": "\u3069",
    "dokatakana": "\u30C9",
    "dollar": "\u0024",
    "dollarinferior": "\uF6E3",
    "dollarmonospace": "\uFF04",
    "dollaroldstyle": "\uF724",
    "dollarsmall": "\uFE69",
    "dollarsuperior": "\uF6E4",
    "dong": "\u20AB",
    "dorusquare": "\u3326",
    "dotaccent": "\u02D9",
    "dotaccentcmb": "\u0307",
    "dotbelowcmb": "\u0323",
    "dotbelowcomb": "\u0323",
    "dotkatakana": "\u30FB",
    "dotlessi": "\u0131",
    "dotlessj": "\uF6BE",
    "dotlessjstrokehook": "\u0284",
    "dotmath": "\u22C5",
    "dottedcircle": "\u25CC",
    "doubleyodpatah": "\uFB1F",
    "doubleyodpatahhebrew": "\uFB1F",
    "downtackbelowcmb": "\u031E",
    "downtackmod": "\u02D5",
    "dparen": "\u249F",
    "dsuperior": "\uF6EB",
    "dtail": "\u0256",
    "dtopbar": "\u018C",
    "duhiragana": "\u3065",
    "dukatakana": "\u30C5",
    "dz": "\u01F3",
    "dzaltone": "\u02A3",
    "dzcaron": "\u01C6",
    "dzcurl": "\u02A5",
    "dzeabkhasiancyrillic": "\u04E1",
    "dzecyrillic": "\u0455",
    "dzhecyrillic": "\u045F",
    "e": "\u0065",
    "eacute": "\u00E9",
    "earth": "\u2641",
    "ebengali": "\u098F",
    "ebopomofo": "\u311C",
    "ebreve": "\u0115",
    "ecandradeva": "\u090D",
    "ecandragujarati": "\u0A8D",
    "ecandravowelsigndeva": "\u0945",
    "ecandravowelsigngujarati": "\u0AC5",
    "ecaron": "\u011B",
    "ecedillabreve": "\u1E1D",
    "echarmenian": "\u0565",
    "echyiwnarmenian": "\u0587",
    "ecircle": "\u24D4",
    "ecircumflex": "\u00EA",
    "ecircumflexacute": "\u1EBF",
    "ecircumflexbelow": "\u1E19",
    "ecircumflexdotbelow": "\u1EC7",
    "ecircumflexgrave": "\u1EC1",
    "ecircumflexhookabove": "\u1EC3",
    "ecircumflextilde": "\u1EC5",
    "ecyrillic": "\u0454",
    "edblgrave": "\u0205",
    "edeva": "\u090F",
    "edieresis": "\u00EB",
    "edot": "\u0117",
    "edotaccent": "\u0117",
    "edotbelow": "\u1EB9",
    "eegurmukhi": "\u0A0F",
    "eematragurmukhi": "\u0A47",
    "efcyrillic": "\u0444",
    "egrave": "\u00E8",
    "egujarati": "\u0A8F",
    "eharmenian": "\u0567",
    "ehbopomofo": "\u311D",
    "ehiragana": "\u3048",
    "ehookabove": "\u1EBB",
    "eibopomofo": "\u311F",
    "eight": "\u0038",
    "eightarabic": "\u0668",
    "eightbengali": "\u09EE",
    "eightcircle": "\u2467",
    "eightcircleinversesansserif": "\u2791",
    "eightdeva": "\u096E",
    "eighteencircle": "\u2471",
    "eighteenparen": "\u2485",
    "eighteenperiod": "\u2499",
    "eightgujarati": "\u0AEE",
    "eightgurmukhi": "\u0A6E",
    "eighthackarabic": "\u0668",
    "eighthangzhou": "\u3028",
    "eighthnotebeamed": "\u266B",
    "eightideographicparen": "\u3227",
    "eightinferior": "\u2088",
    "eightmonospace": "\uFF18",
    "eightoldstyle": "\uF738",
    "eightparen": "\u247B",
    "eightperiod": "\u248F",
    "eightpersian": "\u06F8",
    "eightroman": "\u2177",
    "eightsuperior": "\u2078",
    "eightthai": "\u0E58",
    "einvertedbreve": "\u0207",
    "eiotifiedcyrillic": "\u0465",
    "ekatakana": "\u30A8",
    "ekatakanahalfwidth": "\uFF74",
    "ekonkargurmukhi": "\u0A74",
    "ekorean": "\u3154",
    "elcyrillic": "\u043B",
    "element": "\u2208",
    "elevencircle": "\u246A",
    "elevenparen": "\u247E",
    "elevenperiod": "\u2492",
    "elevenroman": "\u217A",
    "ellipsis": "\u2026",
    "ellipsisvertical": "\u22EE",
    "emacron": "\u0113",
    "emacronacute": "\u1E17",
    "emacrongrave": "\u1E15",
    "emcyrillic": "\u043C",
    "emdash": "\u2014",
    "emdashvertical": "\uFE31",
    "emonospace": "\uFF45",
    "emphasismarkarmenian": "\u055B",
    "emptyset": "\u2205",
    "enbopomofo": "\u3123",
    "encyrillic": "\u043D",
    "endash": "\u2013",
    "endashvertical": "\uFE32",
    "endescendercyrillic": "\u04A3",
    "eng": "\u014B",
    "engbopomofo": "\u3125",
    "enghecyrillic": "\u04A5",
    "enhookcyrillic": "\u04C8",
    "enspace": "\u2002",
    "eogonek": "\u0119",
    "eokorean": "\u3153",
    "eopen": "\u025B",
    "eopenclosed": "\u029A",
    "eopenreversed": "\u025C",
    "eopenreversedclosed": "\u025E",
    "eopenreversedhook": "\u025D",
    "eparen": "\u24A0",
    "epsilon": "\u03B5",
    "epsilontonos": "\u03AD",
    "equal": "\u003D",
    "equalmonospace": "\uFF1D",
    "equalsmall": "\uFE66",
    "equalsuperior": "\u207C",
    "equivalence": "\u2261",
    "erbopomofo": "\u3126",
    "ercyrillic": "\u0440",
    "ereversed": "\u0258",
    "ereversedcyrillic": "\u044D",
    "escyrillic": "\u0441",
    "esdescendercyrillic": "\u04AB",
    "esh": "\u0283",
    "eshcurl": "\u0286",
    "eshortdeva": "\u090E",
    "eshortvowelsigndeva": "\u0946",
    "eshreversedloop": "\u01AA",
    "eshsquatreversed": "\u0285",
    "esmallhiragana": "\u3047",
    "esmallkatakana": "\u30A7",
    "esmallkatakanahalfwidth": "\uFF6A",
    "estimated": "\u212E",
    "esuperior": "\uF6EC",
    "eta": "\u03B7",
    "etarmenian": "\u0568",
    "etatonos": "\u03AE",
    "eth": "\u00F0",
    "etilde": "\u1EBD",
    "etildebelow": "\u1E1B",
    "etnahtafoukhhebrew": "\u0591",
    "etnahtafoukhlefthebrew": "\u0591",
    "etnahtahebrew": "\u0591",
    "etnahtalefthebrew": "\u0591",
    "eturned": "\u01DD",
    "eukorean": "\u3161",
    "euro": "\u20AC",
    "evowelsignbengali": "\u09C7",
    "evowelsigndeva": "\u0947",
    "evowelsigngujarati": "\u0AC7",
    "exclam": "\u0021",
    "exclamarmenian": "\u055C",
    "exclamdbl": "\u203C",
    "exclamdown": "\u00A1",
    "exclamdownsmall": "\uF7A1",
    "exclammonospace": "\uFF01",
    "exclamsmall": "\uF721",
    "existential": "\u2203",
    "ezh": "\u0292",
    "ezhcaron": "\u01EF",
    "ezhcurl": "\u0293",
    "ezhreversed": "\u01B9",
    "ezhtail": "\u01BA",
    "f": "\u0066",
    "fadeva": "\u095E",
    "fagurmukhi": "\u0A5E",
    "fahrenheit": "\u2109",
    "fathaarabic": "\u064E",
    "fathalowarabic": "\u064E",
    "fathatanarabic": "\u064B",
    "fbopomofo": "\u3108",
    "fcircle": "\u24D5",
    "fdotaccent": "\u1E1F",
    "feharabic": "\u0641",
    "feharmenian": "\u0586",
    "fehfinalarabic": "\uFED2",
    "fehinitialarabic": "\uFED3",
    "fehmedialarabic": "\uFED4",
    "feicoptic": "\u03E5",
    "female": "\u2640",
    "ff": "\uFB00",
    "ffi": "\uFB03",
    "ffl": "\uFB04",
    "fi": "\uFB01",
    "fifteencircle": "\u246E",
    "fifteenparen": "\u2482",
    "fifteenperiod": "\u2496",
    "figuredash": "\u2012",
    "filledbox": "\u25A0",
    "filledrect": "\u25AC",
    "finalkaf": "\u05DA",
    "finalkafdagesh": "\uFB3A",
    "finalkafdageshhebrew": "\uFB3A",
    "finalkafhebrew": "\u05DA",
    "finalkafqamats": "\u05DA\u05B8",
    "finalkafqamatshebrew": "\u05DA\u05B8",
    "finalkafsheva": "\u05DA\u05B0",
    "finalkafshevahebrew": "\u05DA\u05B0",
    "finalmem": "\u05DD",
    "finalmemhebrew": "\u05DD",
    "finalnun": "\u05DF",
    "finalnunhebrew": "\u05DF",
    "finalpe": "\u05E3",
    "finalpehebrew": "\u05E3",
    "finaltsadi": "\u05E5",
    "finaltsadihebrew": "\u05E5",
    "firsttonechinese": "\u02C9",
    "fisheye": "\u25C9",
    "fitacyrillic": "\u0473",
    "five": "\u0035",
    "fivearabic": "\u0665",
    "fivebengali": "\u09EB",
    "fivecircle": "\u2464",
    "fivecircleinversesansserif": "\u278E",
    "fivedeva": "\u096B",
    "fiveeighths": "\u215D",
    "fivegujarati": "\u0AEB",
    "fivegurmukhi": "\u0A6B",
    "fivehackarabic": "\u0665",
    "fivehangzhou": "\u3025",
    "fiveideographicparen": "\u3224",
    "fiveinferior": "\u2085",
    "fivemonospace": "\uFF15",
    "fiveoldstyle": "\uF735",
    "fiveparen": "\u2478",
    "fiveperiod": "\u248C",
    "fivepersian": "\u06F5",
    "fiveroman": "\u2174",
    "fivesuperior": "\u2075",
    "fivethai": "\u0E55",
    "fl": "\uFB02",
    "florin": "\u0192",
    "fmonospace": "\uFF46",
    "fmsquare": "\u3399",
    "fofanthai": "\u0E1F",
    "fofathai": "\u0E1D",
    "fongmanthai": "\u0E4F",
    "forall": "\u2200",
    "four": "\u0034",
    "fourarabic": "\u0664",
    "fourbengali": "\u09EA",
    "fourcircle": "\u2463",
    "fourcircleinversesansserif": "\u278D",
    "fourdeva": "\u096A",
    "fourgujarati": "\u0AEA",
    "fourgurmukhi": "\u0A6A",
    "fourhackarabic": "\u0664",
    "fourhangzhou": "\u3024",
    "fourideographicparen": "\u3223",
    "fourinferior": "\u2084",
    "fourmonospace": "\uFF14",
    "fournumeratorbengali": "\u09F7",
    "fouroldstyle": "\uF734",
    "fourparen": "\u2477",
    "fourperiod": "\u248B",
    "fourpersian": "\u06F4",
    "fourroman": "\u2173",
    "foursuperior": "\u2074",
    "fourteencircle": "\u246D",
    "fourteenparen": "\u2481",
    "fourteenperiod": "\u2495",
    "fourthai": "\u0E54",
    "fourthtonechinese": "\u02CB",
    "fparen": "\u24A1",
    "fraction": "\u2044",
    "franc": "\u20A3",
    "g": "\u0067",
    "gabengali": "\u0997",
    "gacute": "\u01F5",
    "gadeva": "\u0917",
    "gafarabic": "\u06AF",
    "gaffinalarabic": "\uFB93",
    "gafinitialarabic": "\uFB94",
    "gafmedialarabic": "\uFB95",
    "gagujarati": "\u0A97",
    "gagurmukhi": "\u0A17",
    "gahiragana": "\u304C",
    "gakatakana": "\u30AC",
    "gamma": "\u03B3",
    "gammalatinsmall": "\u0263",
    "gammasuperior": "\u02E0",
    "gangiacoptic": "\u03EB",
    "gbopomofo": "\u310D",
    "gbreve": "\u011F",
    "gcaron": "\u01E7",
    "gcedilla": "\u0123",
    "gcircle": "\u24D6",
    "gcircumflex": "\u011D",
    "gcommaaccent": "\u0123",
    "gdot": "\u0121",
    "gdotaccent": "\u0121",
    "gecyrillic": "\u0433",
    "gehiragana": "\u3052",
    "gekatakana": "\u30B2",
    "geometricallyequal": "\u2251",
    "gereshaccenthebrew": "\u059C",
    "gereshhebrew": "\u05F3",
    "gereshmuqdamhebrew": "\u059D",
    "germandbls": "\u00DF",
    "gershayimaccenthebrew": "\u059E",
    "gershayimhebrew": "\u05F4",
    "getamark": "\u3013",
    "ghabengali": "\u0998",
    "ghadarmenian": "\u0572",
    "ghadeva": "\u0918",
    "ghagujarati": "\u0A98",
    "ghagurmukhi": "\u0A18",
    "ghainarabic": "\u063A",
    "ghainfinalarabic": "\uFECE",
    "ghaininitialarabic": "\uFECF",
    "ghainmedialarabic": "\uFED0",
    "ghemiddlehookcyrillic": "\u0495",
    "ghestrokecyrillic": "\u0493",
    "gheupturncyrillic": "\u0491",
    "ghhadeva": "\u095A",
    "ghhagurmukhi": "\u0A5A",
    "ghook": "\u0260",
    "ghzsquare": "\u3393",
    "gihiragana": "\u304E",
    "gikatakana": "\u30AE",
    "gimarmenian": "\u0563",
    "gimel": "\u05D2",
    "gimeldagesh": "\uFB32",
    "gimeldageshhebrew": "\uFB32",
    "gimelhebrew": "\u05D2",
    "gjecyrillic": "\u0453",
    "glottalinvertedstroke": "\u01BE",
    "glottalstop": "\u0294",
    "glottalstopinverted": "\u0296",
    "glottalstopmod": "\u02C0",
    "glottalstopreversed": "\u0295",
    "glottalstopreversedmod": "\u02C1",
    "glottalstopreversedsuperior": "\u02E4",
    "glottalstopstroke": "\u02A1",
    "glottalstopstrokereversed": "\u02A2",
    "gmacron": "\u1E21",
    "gmonospace": "\uFF47",
    "gohiragana": "\u3054",
    "gokatakana": "\u30B4",
    "gparen": "\u24A2",
    "gpasquare": "\u33AC",
    "gradient": "\u2207",
    "grave": "\u0060",
    "gravebelowcmb": "\u0316",
    "gravecmb": "\u0300",
    "gravecomb": "\u0300",
    "gravedeva": "\u0953",
    "gravelowmod": "\u02CE",
    "gravemonospace": "\uFF40",
    "gravetonecmb": "\u0340",
    "greater": "\u003E",
    "greaterequal": "\u2265",
    "greaterequalorless": "\u22DB",
    "greatermonospace": "\uFF1E",
    "greaterorequivalent": "\u2273",
    "greaterorless": "\u2277",
    "greateroverequal": "\u2267",
    "greatersmall": "\uFE65",
    "gscript": "\u0261",
    "gstroke": "\u01E5",
    "guhiragana": "\u3050",
    "guillemotleft": "\u00AB",
    "guillemotright": "\u00BB",
    "guilsinglleft": "\u2039",
    "guilsinglright": "\u203A",
    "gukatakana": "\u30B0",
    "guramusquare": "\u3318",
    "gysquare": "\u33C9",
    "h": "\u0068",
    "haabkhasiancyrillic": "\u04A9",
    "haaltonearabic": "\u06C1",
    "habengali": "\u09B9",
    "hadescendercyrillic": "\u04B3",
    "hadeva": "\u0939",
    "hagujarati": "\u0AB9",
    "hagurmukhi": "\u0A39",
    "haharabic": "\u062D",
    "hahfinalarabic": "\uFEA2",
    "hahinitialarabic": "\uFEA3",
    "hahiragana": "\u306F",
    "hahmedialarabic": "\uFEA4",
    "haitusquare": "\u332A",
    "hakatakana": "\u30CF",
    "hakatakanahalfwidth": "\uFF8A",
    "halantgurmukhi": "\u0A4D",
    "hamzaarabic": "\u0621",
    "hamzadammaarabic": "\u0621\u064F",
    "hamzadammatanarabic": "\u0621\u064C",
    "hamzafathaarabic": "\u0621\u064E",
    "hamzafathatanarabic": "\u0621\u064B",
    "hamzalowarabic": "\u0621",
    "hamzalowkasraarabic": "\u0621\u0650",
    "hamzalowkasratanarabic": "\u0621\u064D",
    "hamzasukunarabic": "\u0621\u0652",
    "hangulfiller": "\u3164",
    "hardsigncyrillic": "\u044A",
    "harpoonleftbarbup": "\u21BC",
    "harpoonrightbarbup": "\u21C0",
    "hasquare": "\u33CA",
    "hatafpatah": "\u05B2",
    "hatafpatah16": "\u05B2",
    "hatafpatah23": "\u05B2",
    "hatafpatah2f": "\u05B2",
    "hatafpatahhebrew": "\u05B2",
    "hatafpatahnarrowhebrew": "\u05B2",
    "hatafpatahquarterhebrew": "\u05B2",
    "hatafpatahwidehebrew": "\u05B2",
    "hatafqamats": "\u05B3",
    "hatafqamats1b": "\u05B3",
    "hatafqamats28": "\u05B3",
    "hatafqamats34": "\u05B3",
    "hatafqamatshebrew": "\u05B3",
    "hatafqamatsnarrowhebrew": "\u05B3",
    "hatafqamatsquarterhebrew": "\u05B3",
    "hatafqamatswidehebrew": "\u05B3",
    "hatafsegol": "\u05B1",
    "hatafsegol17": "\u05B1",
    "hatafsegol24": "\u05B1",
    "hatafsegol30": "\u05B1",
    "hatafsegolhebrew": "\u05B1",
    "hatafsegolnarrowhebrew": "\u05B1",
    "hatafsegolquarterhebrew": "\u05B1",
    "hatafsegolwidehebrew": "\u05B1",
    "hbar": "\u0127",
    "hbopomofo": "\u310F",
    "hbrevebelow": "\u1E2B",
    "hcedilla": "\u1E29",
    "hcircle": "\u24D7",
    "hcircumflex": "\u0125",
    "hdieresis": "\u1E27",
    "hdotaccent": "\u1E23",
    "hdotbelow": "\u1E25",
    "he": "\u05D4",
    "heart": "\u2665",
    "heartsuitblack": "\u2665",
    "heartsuitwhite": "\u2661",
    "hedagesh": "\uFB34",
    "hedageshhebrew": "\uFB34",
    "hehaltonearabic": "\u06C1",
    "heharabic": "\u0647",
    "hehebrew": "\u05D4",
    "hehfinalaltonearabic": "\uFBA7",
    "hehfinalalttwoarabic": "\uFEEA",
    "hehfinalarabic": "\uFEEA",
    "hehhamzaabovefinalarabic": "\uFBA5",
    "hehhamzaaboveisolatedarabic": "\uFBA4",
    "hehinitialaltonearabic": "\uFBA8",
    "hehinitialarabic": "\uFEEB",
    "hehiragana": "\u3078",
    "hehmedialaltonearabic": "\uFBA9",
    "hehmedialarabic": "\uFEEC",
    "heiseierasquare": "\u337B",
    "hekatakana": "\u30D8",
    "hekatakanahalfwidth": "\uFF8D",
    "hekutaarusquare": "\u3336",
    "henghook": "\u0267",
    "herutusquare": "\u3339",
    "het": "\u05D7",
    "hethebrew": "\u05D7",
    "hhook": "\u0266",
    "hhooksuperior": "\u02B1",
    "hieuhacirclekorean": "\u327B",
    "hieuhaparenkorean": "\u321B",
    "hieuhcirclekorean": "\u326D",
    "hieuhkorean": "\u314E",
    "hieuhparenkorean": "\u320D",
    "hihiragana": "\u3072",
    "hikatakana": "\u30D2",
    "hikatakanahalfwidth": "\uFF8B",
    "hiriq": "\u05B4",
    "hiriq14": "\u05B4",
    "hiriq21": "\u05B4",
    "hiriq2d": "\u05B4",
    "hiriqhebrew": "\u05B4",
    "hiriqnarrowhebrew": "\u05B4",
    "hiriqquarterhebrew": "\u05B4",
    "hiriqwidehebrew": "\u05B4",
    "hlinebelow": "\u1E96",
    "hmonospace": "\uFF48",
    "hoarmenian": "\u0570",
    "hohipthai": "\u0E2B",
    "hohiragana": "\u307B",
    "hokatakana": "\u30DB",
    "hokatakanahalfwidth": "\uFF8E",
    "holam": "\u05B9",
    "holam19": "\u05B9",
    "holam26": "\u05B9",
    "holam32": "\u05B9",
    "holamhebrew": "\u05B9",
    "holamnarrowhebrew": "\u05B9",
    "holamquarterhebrew": "\u05B9",
    "holamwidehebrew": "\u05B9",
    "honokhukthai": "\u0E2E",
    "hookabovecomb": "\u0309",
    "hookcmb": "\u0309",
    "hookpalatalizedbelowcmb": "\u0321",
    "hookretroflexbelowcmb": "\u0322",
    "hoonsquare": "\u3342",
    "horicoptic": "\u03E9",
    "horizontalbar": "\u2015",
    "horncmb": "\u031B",
    "hotsprings": "\u2668",
    "house": "\u2302",
    "hparen": "\u24A3",
    "hsuperior": "\u02B0",
    "hturned": "\u0265",
    "huhiragana": "\u3075",
    "huiitosquare": "\u3333",
    "hukatakana": "\u30D5",
    "hukatakanahalfwidth": "\uFF8C",
    "hungarumlaut": "\u02DD",
    "hungarumlautcmb": "\u030B",
    "hv": "\u0195",
    "hyphen": "\u002D",
    "hypheninferior": "\uF6E5",
    "hyphenmonospace": "\uFF0D",
    "hyphensmall": "\uFE63",
    "hyphensuperior": "\uF6E6",
    "hyphentwo": "\u2010",
    "i": "\u0069",
    "iacute": "\u00ED",
    "iacyrillic": "\u044F",
    "ibengali": "\u0987",
    "ibopomofo": "\u3127",
    "ibreve": "\u012D",
    "icaron": "\u01D0",
    "icircle": "\u24D8",
    "icircumflex": "\u00EE",
    "icyrillic": "\u0456",
    "idblgrave": "\u0209",
    "ideographearthcircle": "\u328F",
    "ideographfirecircle": "\u328B",
    "ideographicallianceparen": "\u323F",
    "ideographiccallparen": "\u323A",
    "ideographiccentrecircle": "\u32A5",
    "ideographicclose": "\u3006",
    "ideographiccomma": "\u3001",
    "ideographiccommaleft": "\uFF64",
    "ideographiccongratulationparen": "\u3237",
    "ideographiccorrectcircle": "\u32A3",
    "ideographicearthparen": "\u322F",
    "ideographicenterpriseparen": "\u323D",
    "ideographicexcellentcircle": "\u329D",
    "ideographicfestivalparen": "\u3240",
    "ideographicfinancialcircle": "\u3296",
    "ideographicfinancialparen": "\u3236",
    "ideographicfireparen": "\u322B",
    "ideographichaveparen": "\u3232",
    "ideographichighcircle": "\u32A4",
    "ideographiciterationmark": "\u3005",
    "ideographiclaborcircle": "\u3298",
    "ideographiclaborparen": "\u3238",
    "ideographicleftcircle": "\u32A7",
    "ideographiclowcircle": "\u32A6",
    "ideographicmedicinecircle": "\u32A9",
    "ideographicmetalparen": "\u322E",
    "ideographicmoonparen": "\u322A",
    "ideographicnameparen": "\u3234",
    "ideographicperiod": "\u3002",
    "ideographicprintcircle": "\u329E",
    "ideographicreachparen": "\u3243",
    "ideographicrepresentparen": "\u3239",
    "ideographicresourceparen": "\u323E",
    "ideographicrightcircle": "\u32A8",
    "ideographicsecretcircle": "\u3299",
    "ideographicselfparen": "\u3242",
    "ideographicsocietyparen": "\u3233",
    "ideographicspace": "\u3000",
    "ideographicspecialparen": "\u3235",
    "ideographicstockparen": "\u3231",
    "ideographicstudyparen": "\u323B",
    "ideographicsunparen": "\u3230",
    "ideographicsuperviseparen": "\u323C",
    "ideographicwaterparen": "\u322C",
    "ideographicwoodparen": "\u322D",
    "ideographiczero": "\u3007",
    "ideographmetalcircle": "\u328E",
    "ideographmooncircle": "\u328A",
    "ideographnamecircle": "\u3294",
    "ideographsuncircle": "\u3290",
    "ideographwatercircle": "\u328C",
    "ideographwoodcircle": "\u328D",
    "ideva": "\u0907",
    "idieresis": "\u00EF",
    "idieresisacute": "\u1E2F",
    "idieresiscyrillic": "\u04E5",
    "idotbelow": "\u1ECB",
    "iebrevecyrillic": "\u04D7",
    "iecyrillic": "\u0435",
    "ieungacirclekorean": "\u3275",
    "ieungaparenkorean": "\u3215",
    "ieungcirclekorean": "\u3267",
    "ieungkorean": "\u3147",
    "ieungparenkorean": "\u3207",
    "igrave": "\u00EC",
    "igujarati": "\u0A87",
    "igurmukhi": "\u0A07",
    "ihiragana": "\u3044",
    "ihookabove": "\u1EC9",
    "iibengali": "\u0988",
    "iicyrillic": "\u0438",
    "iideva": "\u0908",
    "iigujarati": "\u0A88",
    "iigurmukhi": "\u0A08",
    "iimatragurmukhi": "\u0A40",
    "iinvertedbreve": "\u020B",
    "iishortcyrillic": "\u0439",
    "iivowelsignbengali": "\u09C0",
    "iivowelsigndeva": "\u0940",
    "iivowelsigngujarati": "\u0AC0",
    "ij": "\u0133",
    "ikatakana": "\u30A4",
    "ikatakanahalfwidth": "\uFF72",
    "ikorean": "\u3163",
    "ilde": "\u02DC",
    "iluyhebrew": "\u05AC",
    "imacron": "\u012B",
    "imacroncyrillic": "\u04E3",
    "imageorapproximatelyequal": "\u2253",
    "imatragurmukhi": "\u0A3F",
    "imonospace": "\uFF49",
    "increment": "\u2206",
    "infinity": "\u221E",
    "iniarmenian": "\u056B",
    "integral": "\u222B",
    "integralbottom": "\u2321",
    "integralbt": "\u2321",
    "integralex": "\uF8F5",
    "integraltop": "\u2320",
    "integraltp": "\u2320",
    "intersection": "\u2229",
    "intisquare": "\u3305",
    "invbullet": "\u25D8",
    "invcircle": "\u25D9",
    "invsmileface": "\u263B",
    "iocyrillic": "\u0451",
    "iogonek": "\u012F",
    "iota": "\u03B9",
    "iotadieresis": "\u03CA",
    "iotadieresistonos": "\u0390",
    "iotalatin": "\u0269",
    "iotatonos": "\u03AF",
    "iparen": "\u24A4",
    "irigurmukhi": "\u0A72",
    "ismallhiragana": "\u3043",
    "ismallkatakana": "\u30A3",
    "ismallkatakanahalfwidth": "\uFF68",
    "issharbengali": "\u09FA",
    "istroke": "\u0268",
    "isuperior": "\uF6ED",
    "iterationhiragana": "\u309D",
    "iterationkatakana": "\u30FD",
    "itilde": "\u0129",
    "itildebelow": "\u1E2D",
    "iubopomofo": "\u3129",
    "iucyrillic": "\u044E",
    "ivowelsignbengali": "\u09BF",
    "ivowelsigndeva": "\u093F",
    "ivowelsigngujarati": "\u0ABF",
    "izhitsacyrillic": "\u0475",
    "izhitsadblgravecyrillic": "\u0477",
    "j": "\u006A",
    "jaarmenian": "\u0571",
    "jabengali": "\u099C",
    "jadeva": "\u091C",
    "jagujarati": "\u0A9C",
    "jagurmukhi": "\u0A1C",
    "jbopomofo": "\u3110",
    "jcaron": "\u01F0",
    "jcircle": "\u24D9",
    "jcircumflex": "\u0135",
    "jcrossedtail": "\u029D",
    "jdotlessstroke": "\u025F",
    "jecyrillic": "\u0458",
    "jeemarabic": "\u062C",
    "jeemfinalarabic": "\uFE9E",
    "jeeminitialarabic": "\uFE9F",
    "jeemmedialarabic": "\uFEA0",
    "jeharabic": "\u0698",
    "jehfinalarabic": "\uFB8B",
    "jhabengali": "\u099D",
    "jhadeva": "\u091D",
    "jhagujarati": "\u0A9D",
    "jhagurmukhi": "\u0A1D",
    "jheharmenian": "\u057B",
    "jis": "\u3004",
    "jmonospace": "\uFF4A",
    "jparen": "\u24A5",
    "jsuperior": "\u02B2",
    "k": "\u006B",
    "kabashkircyrillic": "\u04A1",
    "kabengali": "\u0995",
    "kacute": "\u1E31",
    "kacyrillic": "\u043A",
    "kadescendercyrillic": "\u049B",
    "kadeva": "\u0915",
    "kaf": "\u05DB",
    "kafarabic": "\u0643",
    "kafdagesh": "\uFB3B",
    "kafdageshhebrew": "\uFB3B",
    "kaffinalarabic": "\uFEDA",
    "kafhebrew": "\u05DB",
    "kafinitialarabic": "\uFEDB",
    "kafmedialarabic": "\uFEDC",
    "kafrafehebrew": "\uFB4D",
    "kagujarati": "\u0A95",
    "kagurmukhi": "\u0A15",
    "kahiragana": "\u304B",
    "kahookcyrillic": "\u04C4",
    "kakatakana": "\u30AB",
    "kakatakanahalfwidth": "\uFF76",
    "kappa": "\u03BA",
    "kappasymbolgreek": "\u03F0",
    "kapyeounmieumkorean": "\u3171",
    "kapyeounphieuphkorean": "\u3184",
    "kapyeounpieupkorean": "\u3178",
    "kapyeounssangpieupkorean": "\u3179",
    "karoriisquare": "\u330D",
    "kashidaautoarabic": "\u0640",
    "kashidaautonosidebearingarabic": "\u0640",
    "kasmallkatakana": "\u30F5",
    "kasquare": "\u3384",
    "kasraarabic": "\u0650",
    "kasratanarabic": "\u064D",
    "kastrokecyrillic": "\u049F",
    "katahiraprolongmarkhalfwidth": "\uFF70",
    "kaverticalstrokecyrillic": "\u049D",
    "kbopomofo": "\u310E",
    "kcalsquare": "\u3389",
    "kcaron": "\u01E9",
    "kcedilla": "\u0137",
    "kcircle": "\u24DA",
    "kcommaaccent": "\u0137",
    "kdotbelow": "\u1E33",
    "keharmenian": "\u0584",
    "kehiragana": "\u3051",
    "kekatakana": "\u30B1",
    "kekatakanahalfwidth": "\uFF79",
    "kenarmenian": "\u056F",
    "kesmallkatakana": "\u30F6",
    "kgreenlandic": "\u0138",
    "khabengali": "\u0996",
    "khacyrillic": "\u0445",
    "khadeva": "\u0916",
    "khagujarati": "\u0A96",
    "khagurmukhi": "\u0A16",
    "khaharabic": "\u062E",
    "khahfinalarabic": "\uFEA6",
    "khahinitialarabic": "\uFEA7",
    "khahmedialarabic": "\uFEA8",
    "kheicoptic": "\u03E7",
    "khhadeva": "\u0959",
    "khhagurmukhi": "\u0A59",
    "khieukhacirclekorean": "\u3278",
    "khieukhaparenkorean": "\u3218",
    "khieukhcirclekorean": "\u326A",
    "khieukhkorean": "\u314B",
    "khieukhparenkorean": "\u320A",
    "khokhaithai": "\u0E02",
    "khokhonthai": "\u0E05",
    "khokhuatthai": "\u0E03",
    "khokhwaithai": "\u0E04",
    "khomutthai": "\u0E5B",
    "khook": "\u0199",
    "khorakhangthai": "\u0E06",
    "khzsquare": "\u3391",
    "kihiragana": "\u304D",
    "kikatakana": "\u30AD",
    "kikatakanahalfwidth": "\uFF77",
    "kiroguramusquare": "\u3315",
    "kiromeetorusquare": "\u3316",
    "kirosquare": "\u3314",
    "kiyeokacirclekorean": "\u326E",
    "kiyeokaparenkorean": "\u320E",
    "kiyeokcirclekorean": "\u3260",
    "kiyeokkorean": "\u3131",
    "kiyeokparenkorean": "\u3200",
    "kiyeoksioskorean": "\u3133",
    "kjecyrillic": "\u045C",
    "klinebelow": "\u1E35",
    "klsquare": "\u3398",
    "kmcubedsquare": "\u33A6",
    "kmonospace": "\uFF4B",
    "kmsquaredsquare": "\u33A2",
    "kohiragana": "\u3053",
    "kohmsquare": "\u33C0",
    "kokaithai": "\u0E01",
    "kokatakana": "\u30B3",
    "kokatakanahalfwidth": "\uFF7A",
    "kooposquare": "\u331E",
    "koppacyrillic": "\u0481",
    "koreanstandardsymbol": "\u327F",
    "koroniscmb": "\u0343",
    "kparen": "\u24A6",
    "kpasquare": "\u33AA",
    "ksicyrillic": "\u046F",
    "ktsquare": "\u33CF",
    "kturned": "\u029E",
    "kuhiragana": "\u304F",
    "kukatakana": "\u30AF",
    "kukatakanahalfwidth": "\uFF78",
    "kvsquare": "\u33B8",
    "kwsquare": "\u33BE",
    "l": "\u006C",
    "labengali": "\u09B2",
    "lacute": "\u013A",
    "ladeva": "\u0932",
    "lagujarati": "\u0AB2",
    "lagurmukhi": "\u0A32",
    "lakkhangyaothai": "\u0E45",
    "lamaleffinalarabic": "\uFEFC",
    "lamalefhamzaabovefinalarabic": "\uFEF8",
    "lamalefhamzaaboveisolatedarabic": "\uFEF7",
    "lamalefhamzabelowfinalarabic": "\uFEFA",
    "lamalefhamzabelowisolatedarabic": "\uFEF9",
    "lamalefisolatedarabic": "\uFEFB",
    "lamalefmaddaabovefinalarabic": "\uFEF6",
    "lamalefmaddaaboveisolatedarabic": "\uFEF5",
    "lamarabic": "\u0644",
    "lambda": "\u03BB",
    "lambdastroke": "\u019B",
    "lamed": "\u05DC",
    "lameddagesh": "\uFB3C",
    "lameddageshhebrew": "\uFB3C",
    "lamedhebrew": "\u05DC",
    "lamedholam": "\u05DC\u05B9",
    "lamedholamdagesh": "\u05DC\u05B9\u05BC",
    "lamedholamdageshhebrew": "\u05DC\u05B9\u05BC",
    "lamedholamhebrew": "\u05DC\u05B9",
    "lamfinalarabic": "\uFEDE",
    "lamhahinitialarabic": "\uFCCA",
    "laminitialarabic": "\uFEDF",
    "lamjeeminitialarabic": "\uFCC9",
    "lamkhahinitialarabic": "\uFCCB",
    "lamlamhehisolatedarabic": "\uFDF2",
    "lammedialarabic": "\uFEE0",
    "lammeemhahinitialarabic": "\uFD88",
    "lammeeminitialarabic": "\uFCCC",
    "lammeemjeeminitialarabic": "\uFEDF\uFEE4\uFEA0",
    "lammeemkhahinitialarabic": "\uFEDF\uFEE4\uFEA8",
    "largecircle": "\u25EF",
    "lbar": "\u019A",
    "lbelt": "\u026C",
    "lbopomofo": "\u310C",
    "lcaron": "\u013E",
    "lcedilla": "\u013C",
    "lcircle": "\u24DB",
    "lcircumflexbelow": "\u1E3D",
    "lcommaaccent": "\u013C",
    "ldot": "\u0140",
    "ldotaccent": "\u0140",
    "ldotbelow": "\u1E37",
    "ldotbelowmacron": "\u1E39",
    "leftangleabovecmb": "\u031A",
    "lefttackbelowcmb": "\u0318",
    "less": "\u003C",
    "lessequal": "\u2264",
    "lessequalorgreater": "\u22DA",
    "lessmonospace": "\uFF1C",
    "lessorequivalent": "\u2272",
    "lessorgreater": "\u2276",
    "lessoverequal": "\u2266",
    "lesssmall": "\uFE64",
    "lezh": "\u026E",
    "lfblock": "\u258C",
    "lhookretroflex": "\u026D",
    "lira": "\u20A4",
    "liwnarmenian": "\u056C",
    "lj": "\u01C9",
    "ljecyrillic": "\u0459",
    "ll": "\uF6C0",
    "lladeva": "\u0933",
    "llagujarati": "\u0AB3",
    "llinebelow": "\u1E3B",
    "llladeva": "\u0934",
    "llvocalicbengali": "\u09E1",
    "llvocalicdeva": "\u0961",
    "llvocalicvowelsignbengali": "\u09E3",
    "llvocalicvowelsigndeva": "\u0963",
    "lmiddletilde": "\u026B",
    "lmonospace": "\uFF4C",
    "lmsquare": "\u33D0",
    "lochulathai": "\u0E2C",
    "logicaland": "\u2227",
    "logicalnot": "\u00AC",
    "logicalnotreversed": "\u2310",
    "logicalor": "\u2228",
    "lolingthai": "\u0E25",
    "longs": "\u017F",
    "lowlinecenterline": "\uFE4E",
    "lowlinecmb": "\u0332",
    "lowlinedashed": "\uFE4D",
    "lozenge": "\u25CA",
    "lparen": "\u24A7",
    "lslash": "\u0142",
    "lsquare": "\u2113",
    "lsuperior": "\uF6EE",
    "ltshade": "\u2591",
    "luthai": "\u0E26",
    "lvocalicbengali": "\u098C",
    "lvocalicdeva": "\u090C",
    "lvocalicvowelsignbengali": "\u09E2",
    "lvocalicvowelsigndeva": "\u0962",
    "lxsquare": "\u33D3",
    "m": "\u006D",
    "mabengali": "\u09AE",
    "macron": "\u00AF",
    "macronbelowcmb": "\u0331",
    "macroncmb": "\u0304",
    "macronlowmod": "\u02CD",
    "macronmonospace": "\uFFE3",
    "macute": "\u1E3F",
    "madeva": "\u092E",
    "magujarati": "\u0AAE",
    "magurmukhi": "\u0A2E",
    "mahapakhhebrew": "\u05A4",
    "mahapakhlefthebrew": "\u05A4",
    "mahiragana": "\u307E",
    "maichattawalowleftthai": "\uF895",
    "maichattawalowrightthai": "\uF894",
    "maichattawathai": "\u0E4B",
    "maichattawaupperleftthai": "\uF893",
    "maieklowleftthai": "\uF88C",
    "maieklowrightthai": "\uF88B",
    "maiekthai": "\u0E48",
    "maiekupperleftthai": "\uF88A",
    "maihanakatleftthai": "\uF884",
    "maihanakatthai": "\u0E31",
    "maitaikhuleftthai": "\uF889",
    "maitaikhuthai": "\u0E47",
    "maitholowleftthai": "\uF88F",
    "maitholowrightthai": "\uF88E",
    "maithothai": "\u0E49",
    "maithoupperleftthai": "\uF88D",
    "maitrilowleftthai": "\uF892",
    "maitrilowrightthai": "\uF891",
    "maitrithai": "\u0E4A",
    "maitriupperleftthai": "\uF890",
    "maiyamokthai": "\u0E46",
    "makatakana": "\u30DE",
    "makatakanahalfwidth": "\uFF8F",
    "male": "\u2642",
    "mansyonsquare": "\u3347",
    "maqafhebrew": "\u05BE",
    "mars": "\u2642",
    "masoracirclehebrew": "\u05AF",
    "masquare": "\u3383",
    "mbopomofo": "\u3107",
    "mbsquare": "\u33D4",
    "mcircle": "\u24DC",
    "mcubedsquare": "\u33A5",
    "mdotaccent": "\u1E41",
    "mdotbelow": "\u1E43",
    "meemarabic": "\u0645",
    "meemfinalarabic": "\uFEE2",
    "meeminitialarabic": "\uFEE3",
    "meemmedialarabic": "\uFEE4",
    "meemmeeminitialarabic": "\uFCD1",
    "meemmeemisolatedarabic": "\uFC48",
    "meetorusquare": "\u334D",
    "mehiragana": "\u3081",
    "meizierasquare": "\u337E",
    "mekatakana": "\u30E1",
    "mekatakanahalfwidth": "\uFF92",
    "mem": "\u05DE",
    "memdagesh": "\uFB3E",
    "memdageshhebrew": "\uFB3E",
    "memhebrew": "\u05DE",
    "menarmenian": "\u0574",
    "merkhahebrew": "\u05A5",
    "merkhakefulahebrew": "\u05A6",
    "merkhakefulalefthebrew": "\u05A6",
    "merkhalefthebrew": "\u05A5",
    "mhook": "\u0271",
    "mhzsquare": "\u3392",
    "middledotkatakanahalfwidth": "\uFF65",
    "middot": "\u00B7",
    "mieumacirclekorean": "\u3272",
    "mieumaparenkorean": "\u3212",
    "mieumcirclekorean": "\u3264",
    "mieumkorean": "\u3141",
    "mieumpansioskorean": "\u3170",
    "mieumparenkorean": "\u3204",
    "mieumpieupkorean": "\u316E",
    "mieumsioskorean": "\u316F",
    "mihiragana": "\u307F",
    "mikatakana": "\u30DF",
    "mikatakanahalfwidth": "\uFF90",
    "minus": "\u2212",
    "minusbelowcmb": "\u0320",
    "minuscircle": "\u2296",
    "minusmod": "\u02D7",
    "minusplus": "\u2213",
    "minute": "\u2032",
    "miribaarusquare": "\u334A",
    "mirisquare": "\u3349",
    "mlonglegturned": "\u0270",
    "mlsquare": "\u3396",
    "mmcubedsquare": "\u33A3",
    "mmonospace": "\uFF4D",
    "mmsquaredsquare": "\u339F",
    "mohiragana": "\u3082",
    "mohmsquare": "\u33C1",
    "mokatakana": "\u30E2",
    "mokatakanahalfwidth": "\uFF93",
    "molsquare": "\u33D6",
    "momathai": "\u0E21",
    "moverssquare": "\u33A7",
    "moverssquaredsquare": "\u33A8",
    "mparen": "\u24A8",
    "mpasquare": "\u33AB",
    "mssquare": "\u33B3",
    "msuperior": "\uF6EF",
    "mturned": "\u026F",
    "mu": "\u00B5",
    "mu1": "\u00B5",
    "muasquare": "\u3382",
    "muchgreater": "\u226B",
    "muchless": "\u226A",
    "mufsquare": "\u338C",
    "mugreek": "\u03BC",
    "mugsquare": "\u338D",
    "muhiragana": "\u3080",
    "mukatakana": "\u30E0",
    "mukatakanahalfwidth": "\uFF91",
    "mulsquare": "\u3395",
    "multiply": "\u00D7",
    "mumsquare": "\u339B",
    "munahhebrew": "\u05A3",
    "munahlefthebrew": "\u05A3",
    "musicalnote": "\u266A",
    "musicalnotedbl": "\u266B",
    "musicflatsign": "\u266D",
    "musicsharpsign": "\u266F",
    "mussquare": "\u33B2",
    "muvsquare": "\u33B6",
    "muwsquare": "\u33BC",
    "mvmegasquare": "\u33B9",
    "mvsquare": "\u33B7",
    "mwmegasquare": "\u33BF",
    "mwsquare": "\u33BD",
    "n": "\u006E",
    "nabengali": "\u09A8",
    "nabla": "\u2207",
    "nacute": "\u0144",
    "nadeva": "\u0928",
    "nagujarati": "\u0AA8",
    "nagurmukhi": "\u0A28",
    "nahiragana": "\u306A",
    "nakatakana": "\u30CA",
    "nakatakanahalfwidth": "\uFF85",
    "napostrophe": "\u0149",
    "nasquare": "\u3381",
    "nbopomofo": "\u310B",
    "nbspace": "\u00A0",
    "ncaron": "\u0148",
    "ncedilla": "\u0146",
    "ncircle": "\u24DD",
    "ncircumflexbelow": "\u1E4B",
    "ncommaaccent": "\u0146",
    "ndotaccent": "\u1E45",
    "ndotbelow": "\u1E47",
    "nehiragana": "\u306D",
    "nekatakana": "\u30CD",
    "nekatakanahalfwidth": "\uFF88",
    "newsheqelsign": "\u20AA",
    "nfsquare": "\u338B",
    "ngabengali": "\u0999",
    "ngadeva": "\u0919",
    "ngagujarati": "\u0A99",
    "ngagurmukhi": "\u0A19",
    "ngonguthai": "\u0E07",
    "nhiragana": "\u3093",
    "nhookleft": "\u0272",
    "nhookretroflex": "\u0273",
    "nieunacirclekorean": "\u326F",
    "nieunaparenkorean": "\u320F",
    "nieuncieuckorean": "\u3135",
    "nieuncirclekorean": "\u3261",
    "nieunhieuhkorean": "\u3136",
    "nieunkorean": "\u3134",
    "nieunpansioskorean": "\u3168",
    "nieunparenkorean": "\u3201",
    "nieunsioskorean": "\u3167",
    "nieuntikeutkorean": "\u3166",
    "nihiragana": "\u306B",
    "nikatakana": "\u30CB",
    "nikatakanahalfwidth": "\uFF86",
    "nikhahitleftthai": "\uF899",
    "nikhahitthai": "\u0E4D",
    "nine": "\u0039",
    "ninearabic": "\u0669",
    "ninebengali": "\u09EF",
    "ninecircle": "\u2468",
    "ninecircleinversesansserif": "\u2792",
    "ninedeva": "\u096F",
    "ninegujarati": "\u0AEF",
    "ninegurmukhi": "\u0A6F",
    "ninehackarabic": "\u0669",
    "ninehangzhou": "\u3029",
    "nineideographicparen": "\u3228",
    "nineinferior": "\u2089",
    "ninemonospace": "\uFF19",
    "nineoldstyle": "\uF739",
    "nineparen": "\u247C",
    "nineperiod": "\u2490",
    "ninepersian": "\u06F9",
    "nineroman": "\u2178",
    "ninesuperior": "\u2079",
    "nineteencircle": "\u2472",
    "nineteenparen": "\u2486",
    "nineteenperiod": "\u249A",
    "ninethai": "\u0E59",
    "nj": "\u01CC",
    "njecyrillic": "\u045A",
    "nkatakana": "\u30F3",
    "nkatakanahalfwidth": "\uFF9D",
    "nlegrightlong": "\u019E",
    "nlinebelow": "\u1E49",
    "nmonospace": "\uFF4E",
    "nmsquare": "\u339A",
    "nnabengali": "\u09A3",
    "nnadeva": "\u0923",
    "nnagujarati": "\u0AA3",
    "nnagurmukhi": "\u0A23",
    "nnnadeva": "\u0929",
    "nohiragana": "\u306E",
    "nokatakana": "\u30CE",
    "nokatakanahalfwidth": "\uFF89",
    "nonbreakingspace": "\u00A0",
    "nonenthai": "\u0E13",
    "nonuthai": "\u0E19",
    "noonarabic": "\u0646",
    "noonfinalarabic": "\uFEE6",
    "noonghunnaarabic": "\u06BA",
    "noonghunnafinalarabic": "\uFB9F",
    "noonhehinitialarabic": "\uFEE7\uFEEC",
    "nooninitialarabic": "\uFEE7",
    "noonjeeminitialarabic": "\uFCD2",
    "noonjeemisolatedarabic": "\uFC4B",
    "noonmedialarabic": "\uFEE8",
    "noonmeeminitialarabic": "\uFCD5",
    "noonmeemisolatedarabic": "\uFC4E",
    "noonnoonfinalarabic": "\uFC8D",
    "notcontains": "\u220C",
    "notelement": "\u2209",
    "notelementof": "\u2209",
    "notequal": "\u2260",
    "notgreater": "\u226F",
    "notgreaternorequal": "\u2271",
    "notgreaternorless": "\u2279",
    "notidentical": "\u2262",
    "notless": "\u226E",
    "notlessnorequal": "\u2270",
    "notparallel": "\u2226",
    "notprecedes": "\u2280",
    "notsubset": "\u2284",
    "notsucceeds": "\u2281",
    "notsuperset": "\u2285",
    "nowarmenian": "\u0576",
    "nparen": "\u24A9",
    "nssquare": "\u33B1",
    "nsuperior": "\u207F",
    "ntilde": "\u00F1",
    "nu": "\u03BD",
    "nuhiragana": "\u306C",
    "nukatakana": "\u30CC",
    "nukatakanahalfwidth": "\uFF87",
    "nuktabengali": "\u09BC",
    "nuktadeva": "\u093C",
    "nuktagujarati": "\u0ABC",
    "nuktagurmukhi": "\u0A3C",
    "numbersign": "\u0023",
    "numbersignmonospace": "\uFF03",
    "numbersignsmall": "\uFE5F",
    "numeralsigngreek": "\u0374",
    "numeralsignlowergreek": "\u0375",
    "numero": "\u2116",
    "nun": "\u05E0",
    "nundagesh": "\uFB40",
    "nundageshhebrew": "\uFB40",
    "nunhebrew": "\u05E0",
    "nvsquare": "\u33B5",
    "nwsquare": "\u33BB",
    "nyabengali": "\u099E",
    "nyadeva": "\u091E",
    "nyagujarati": "\u0A9E",
    "nyagurmukhi": "\u0A1E",
    "o": "\u006F",
    "oacute": "\u00F3",
    "oangthai": "\u0E2D",
    "obarred": "\u0275",
    "obarredcyrillic": "\u04E9",
    "obarreddieresiscyrillic": "\u04EB",
    "obengali": "\u0993",
    "obopomofo": "\u311B",
    "obreve": "\u014F",
    "ocandradeva": "\u0911",
    "ocandragujarati": "\u0A91",
    "ocandravowelsigndeva": "\u0949",
    "ocandravowelsigngujarati": "\u0AC9",
    "ocaron": "\u01D2",
    "ocircle": "\u24DE",
    "ocircumflex": "\u00F4",
    "ocircumflexacute": "\u1ED1",
    "ocircumflexdotbelow": "\u1ED9",
    "ocircumflexgrave": "\u1ED3",
    "ocircumflexhookabove": "\u1ED5",
    "ocircumflextilde": "\u1ED7",
    "ocyrillic": "\u043E",
    "odblacute": "\u0151",
    "odblgrave": "\u020D",
    "odeva": "\u0913",
    "odieresis": "\u00F6",
    "odieresiscyrillic": "\u04E7",
    "odotbelow": "\u1ECD",
    "oe": "\u0153",
    "oekorean": "\u315A",
    "ogonek": "\u02DB",
    "ogonekcmb": "\u0328",
    "ograve": "\u00F2",
    "ogujarati": "\u0A93",
    "oharmenian": "\u0585",
    "ohiragana": "\u304A",
    "ohookabove": "\u1ECF",
    "ohorn": "\u01A1",
    "ohornacute": "\u1EDB",
    "ohorndotbelow": "\u1EE3",
    "ohorngrave": "\u1EDD",
    "ohornhookabove": "\u1EDF",
    "ohorntilde": "\u1EE1",
    "ohungarumlaut": "\u0151",
    "oi": "\u01A3",
    "oinvertedbreve": "\u020F",
    "okatakana": "\u30AA",
    "okatakanahalfwidth": "\uFF75",
    "okorean": "\u3157",
    "olehebrew": "\u05AB",
    "omacron": "\u014D",
    "omacronacute": "\u1E53",
    "omacrongrave": "\u1E51",
    "omdeva": "\u0950",
    "omega": "\u03C9",
    "omega1": "\u03D6",
    "omegacyrillic": "\u0461",
    "omegalatinclosed": "\u0277",
    "omegaroundcyrillic": "\u047B",
    "omegatitlocyrillic": "\u047D",
    "omegatonos": "\u03CE",
    "omgujarati": "\u0AD0",
    "omicron": "\u03BF",
    "omicrontonos": "\u03CC",
    "omonospace": "\uFF4F",
    "one": "\u0031",
    "onearabic": "\u0661",
    "onebengali": "\u09E7",
    "onecircle": "\u2460",
    "onecircleinversesansserif": "\u278A",
    "onedeva": "\u0967",
    "onedotenleader": "\u2024",
    "oneeighth": "\u215B",
    "onefitted": "\uF6DC",
    "onegujarati": "\u0AE7",
    "onegurmukhi": "\u0A67",
    "onehackarabic": "\u0661",
    "onehalf": "\u00BD",
    "onehangzhou": "\u3021",
    "oneideographicparen": "\u3220",
    "oneinferior": "\u2081",
    "onemonospace": "\uFF11",
    "onenumeratorbengali": "\u09F4",
    "oneoldstyle": "\uF731",
    "oneparen": "\u2474",
    "oneperiod": "\u2488",
    "onepersian": "\u06F1",
    "onequarter": "\u00BC",
    "oneroman": "\u2170",
    "onesuperior": "\u00B9",
    "onethai": "\u0E51",
    "onethird": "\u2153",
    "oogonek": "\u01EB",
    "oogonekmacron": "\u01ED",
    "oogurmukhi": "\u0A13",
    "oomatragurmukhi": "\u0A4B",
    "oopen": "\u0254",
    "oparen": "\u24AA",
    "openbullet": "\u25E6",
    "option": "\u2325",
    "ordfeminine": "\u00AA",
    "ordmasculine": "\u00BA",
    "orthogonal": "\u221F",
    "oshortdeva": "\u0912",
    "oshortvowelsigndeva": "\u094A",
    "oslash": "\u00F8",
    "oslashacute": "\u01FF",
    "osmallhiragana": "\u3049",
    "osmallkatakana": "\u30A9",
    "osmallkatakanahalfwidth": "\uFF6B",
    "ostrokeacute": "\u01FF",
    "osuperior": "\uF6F0",
    "otcyrillic": "\u047F",
    "otilde": "\u00F5",
    "otildeacute": "\u1E4D",
    "otildedieresis": "\u1E4F",
    "oubopomofo": "\u3121",
    "overline": "\u203E",
    "overlinecenterline": "\uFE4A",
    "overlinecmb": "\u0305",
    "overlinedashed": "\uFE49",
    "overlinedblwavy": "\uFE4C",
    "overlinewavy": "\uFE4B",
    "overscore": "\u00AF",
    "ovowelsignbengali": "\u09CB",
    "ovowelsigndeva": "\u094B",
    "ovowelsigngujarati": "\u0ACB",
    "p": "\u0070",
    "paampssquare": "\u3380",
    "paasentosquare": "\u332B",
    "pabengali": "\u09AA",
    "pacute": "\u1E55",
    "padeva": "\u092A",
    "pagedown": "\u21DF",
    "pageup": "\u21DE",
    "pagujarati": "\u0AAA",
    "pagurmukhi": "\u0A2A",
    "pahiragana": "\u3071",
    "paiyannoithai": "\u0E2F",
    "pakatakana": "\u30D1",
    "palatalizationcyrilliccmb": "\u0484",
    "palochkacyrillic": "\u04C0",
    "pansioskorean": "\u317F",
    "paragraph": "\u00B6",
    "parallel": "\u2225",
    "parenleft": "\u0028",
    "parenleftaltonearabic": "\uFD3E",
    "parenleftbt": "\uF8ED",
    "parenleftex": "\uF8EC",
    "parenleftinferior": "\u208D",
    "parenleftmonospace": "\uFF08",
    "parenleftsmall": "\uFE59",
    "parenleftsuperior": "\u207D",
    "parenlefttp": "\uF8EB",
    "parenleftvertical": "\uFE35",
    "parenright": "\u0029",
    "parenrightaltonearabic": "\uFD3F",
    "parenrightbt": "\uF8F8",
    "parenrightex": "\uF8F7",
    "parenrightinferior": "\u208E",
    "parenrightmonospace": "\uFF09",
    "parenrightsmall": "\uFE5A",
    "parenrightsuperior": "\u207E",
    "parenrighttp": "\uF8F6",
    "parenrightvertical": "\uFE36",
    "partialdiff": "\u2202",
    "paseqhebrew": "\u05C0",
    "pashtahebrew": "\u0599",
    "pasquare": "\u33A9",
    "patah": "\u05B7",
    "patah11": "\u05B7",
    "patah1d": "\u05B7",
    "patah2a": "\u05B7",
    "patahhebrew": "\u05B7",
    "patahnarrowhebrew": "\u05B7",
    "patahquarterhebrew": "\u05B7",
    "patahwidehebrew": "\u05B7",
    "pazerhebrew": "\u05A1",
    "pbopomofo": "\u3106",
    "pcircle": "\u24DF",
    "pdotaccent": "\u1E57",
    "pe": "\u05E4",
    "pecyrillic": "\u043F",
    "pedagesh": "\uFB44",
    "pedageshhebrew": "\uFB44",
    "peezisquare": "\u333B",
    "pefinaldageshhebrew": "\uFB43",
    "peharabic": "\u067E",
    "peharmenian": "\u057A",
    "pehebrew": "\u05E4",
    "pehfinalarabic": "\uFB57",
    "pehinitialarabic": "\uFB58",
    "pehiragana": "\u307A",
    "pehmedialarabic": "\uFB59",
    "pekatakana": "\u30DA",
    "pemiddlehookcyrillic": "\u04A7",
    "perafehebrew": "\uFB4E",
    "percent": "\u0025",
    "percentarabic": "\u066A",
    "percentmonospace": "\uFF05",
    "percentsmall": "\uFE6A",
    "period": "\u002E",
    "periodarmenian": "\u0589",
    "periodcentered": "\u00B7",
    "periodhalfwidth": "\uFF61",
    "periodinferior": "\uF6E7",
    "periodmonospace": "\uFF0E",
    "periodsmall": "\uFE52",
    "periodsuperior": "\uF6E8",
    "perispomenigreekcmb": "\u0342",
    "perpendicular": "\u22A5",
    "perthousand": "\u2030",
    "peseta": "\u20A7",
    "pfsquare": "\u338A",
    "phabengali": "\u09AB",
    "phadeva": "\u092B",
    "phagujarati": "\u0AAB",
    "phagurmukhi": "\u0A2B",
    "phi": "\u03C6",
    "phi1": "\u03D5",
    "phieuphacirclekorean": "\u327A",
    "phieuphaparenkorean": "\u321A",
    "phieuphcirclekorean": "\u326C",
    "phieuphkorean": "\u314D",
    "phieuphparenkorean": "\u320C",
    "philatin": "\u0278",
    "phinthuthai": "\u0E3A",
    "phisymbolgreek": "\u03D5",
    "phook": "\u01A5",
    "phophanthai": "\u0E1E",
    "phophungthai": "\u0E1C",
    "phosamphaothai": "\u0E20",
    "pi": "\u03C0",
    "pieupacirclekorean": "\u3273",
    "pieupaparenkorean": "\u3213",
    "pieupcieuckorean": "\u3176",
    "pieupcirclekorean": "\u3265",
    "pieupkiyeokkorean": "\u3172",
    "pieupkorean": "\u3142",
    "pieupparenkorean": "\u3205",
    "pieupsioskiyeokkorean": "\u3174",
    "pieupsioskorean": "\u3144",
    "pieupsiostikeutkorean": "\u3175",
    "pieupthieuthkorean": "\u3177",
    "pieuptikeutkorean": "\u3173",
    "pihiragana": "\u3074",
    "pikatakana": "\u30D4",
    "pisymbolgreek": "\u03D6",
    "piwrarmenian": "\u0583",
    "plus": "\u002B",
    "plusbelowcmb": "\u031F",
    "pluscircle": "\u2295",
    "plusminus": "\u00B1",
    "plusmod": "\u02D6",
    "plusmonospace": "\uFF0B",
    "plussmall": "\uFE62",
    "plussuperior": "\u207A",
    "pmonospace": "\uFF50",
    "pmsquare": "\u33D8",
    "pohiragana": "\u307D",
    "pointingindexdownwhite": "\u261F",
    "pointingindexleftwhite": "\u261C",
    "pointingindexrightwhite": "\u261E",
    "pointingindexupwhite": "\u261D",
    "pokatakana": "\u30DD",
    "poplathai": "\u0E1B",
    "postalmark": "\u3012",
    "postalmarkface": "\u3020",
    "pparen": "\u24AB",
    "precedes": "\u227A",
    "prescription": "\u211E",
    "primemod": "\u02B9",
    "primereversed": "\u2035",
    "product": "\u220F",
    "projective": "\u2305",
    "prolongedkana": "\u30FC",
    "propellor": "\u2318",
    "propersubset": "\u2282",
    "propersuperset": "\u2283",
    "proportion": "\u2237",
    "proportional": "\u221D",
    "psi": "\u03C8",
    "psicyrillic": "\u0471",
    "psilipneumatacyrilliccmb": "\u0486",
    "pssquare": "\u33B0",
    "puhiragana": "\u3077",
    "pukatakana": "\u30D7",
    "pvsquare": "\u33B4",
    "pwsquare": "\u33BA",
    "q": "\u0071",
    "qadeva": "\u0958",
    "qadmahebrew": "\u05A8",
    "qafarabic": "\u0642",
    "qaffinalarabic": "\uFED6",
    "qafinitialarabic": "\uFED7",
    "qafmedialarabic": "\uFED8",
    "qamats": "\u05B8",
    "qamats10": "\u05B8",
    "qamats1a": "\u05B8",
    "qamats1c": "\u05B8",
    "qamats27": "\u05B8",
    "qamats29": "\u05B8",
    "qamats33": "\u05B8",
    "qamatsde": "\u05B8",
    "qamatshebrew": "\u05B8",
    "qamatsnarrowhebrew": "\u05B8",
    "qamatsqatanhebrew": "\u05B8",
    "qamatsqatannarrowhebrew": "\u05B8",
    "qamatsqatanquarterhebrew": "\u05B8",
    "qamatsqatanwidehebrew": "\u05B8",
    "qamatsquarterhebrew": "\u05B8",
    "qamatswidehebrew": "\u05B8",
    "qarneyparahebrew": "\u059F",
    "qbopomofo": "\u3111",
    "qcircle": "\u24E0",
    "qhook": "\u02A0",
    "qmonospace": "\uFF51",
    "qof": "\u05E7",
    "qofdagesh": "\uFB47",
    "qofdageshhebrew": "\uFB47",
    "qofhatafpatah": "\u05E7\u05B2",
    "qofhatafpatahhebrew": "\u05E7\u05B2",
    "qofhatafsegol": "\u05E7\u05B1",
    "qofhatafsegolhebrew": "\u05E7\u05B1",
    "qofhebrew": "\u05E7",
    "qofhiriq": "\u05E7\u05B4",
    "qofhiriqhebrew": "\u05E7\u05B4",
    "qofholam": "\u05E7\u05B9",
    "qofholamhebrew": "\u05E7\u05B9",
    "qofpatah": "\u05E7\u05B7",
    "qofpatahhebrew": "\u05E7\u05B7",
    "qofqamats": "\u05E7\u05B8",
    "qofqamatshebrew": "\u05E7\u05B8",
    "qofqubuts": "\u05E7\u05BB",
    "qofqubutshebrew": "\u05E7\u05BB",
    "qofsegol": "\u05E7\u05B6",
    "qofsegolhebrew": "\u05E7\u05B6",
    "qofsheva": "\u05E7\u05B0",
    "qofshevahebrew": "\u05E7\u05B0",
    "qoftsere": "\u05E7\u05B5",
    "qoftserehebrew": "\u05E7\u05B5",
    "qparen": "\u24AC",
    "quarternote": "\u2669",
    "qubuts": "\u05BB",
    "qubuts18": "\u05BB",
    "qubuts25": "\u05BB",
    "qubuts31": "\u05BB",
    "qubutshebrew": "\u05BB",
    "qubutsnarrowhebrew": "\u05BB",
    "qubutsquarterhebrew": "\u05BB",
    "qubutswidehebrew": "\u05BB",
    "question": "\u003F",
    "questionarabic": "\u061F",
    "questionarmenian": "\u055E",
    "questiondown": "\u00BF",
    "questiondownsmall": "\uF7BF",
    "questiongreek": "\u037E",
    "questionmonospace": "\uFF1F",
    "questionsmall": "\uF73F",
    "quotedbl": "\u0022",
    "quotedblbase": "\u201E",
    "quotedblleft": "\u201C",
    "quotedblmonospace": "\uFF02",
    "quotedblprime": "\u301E",
    "quotedblprimereversed": "\u301D",
    "quotedblright": "\u201D",
    "quoteleft": "\u2018",
    "quoteleftreversed": "\u201B",
    "quotereversed": "\u201B",
    "quoteright": "\u2019",
    "quoterightn": "\u0149",
    "quotesinglbase": "\u201A",
    "quotesingle": "\u0027",
    "quotesinglemonospace": "\uFF07",
    "r": "\u0072",
    "raarmenian": "\u057C",
    "rabengali": "\u09B0",
    "racute": "\u0155",
    "radeva": "\u0930",
    "radical": "\u221A",
    "radicalex": "\uF8E5",
    "radoverssquare": "\u33AE",
    "radoverssquaredsquare": "\u33AF",
    "radsquare": "\u33AD",
    "rafe": "\u05BF",
    "rafehebrew": "\u05BF",
    "ragujarati": "\u0AB0",
    "ragurmukhi": "\u0A30",
    "rahiragana": "\u3089",
    "rakatakana": "\u30E9",
    "rakatakanahalfwidth": "\uFF97",
    "ralowerdiagonalbengali": "\u09F1",
    "ramiddlediagonalbengali": "\u09F0",
    "ramshorn": "\u0264",
    "ratio": "\u2236",
    "rbopomofo": "\u3116",
    "rcaron": "\u0159",
    "rcedilla": "\u0157",
    "rcircle": "\u24E1",
    "rcommaaccent": "\u0157",
    "rdblgrave": "\u0211",
    "rdotaccent": "\u1E59",
    "rdotbelow": "\u1E5B",
    "rdotbelowmacron": "\u1E5D",
    "referencemark": "\u203B",
    "reflexsubset": "\u2286",
    "reflexsuperset": "\u2287",
    "registered": "\u00AE",
    "registersans": "\uF8E8",
    "registerserif": "\uF6DA",
    "reharabic": "\u0631",
    "reharmenian": "\u0580",
    "rehfinalarabic": "\uFEAE",
    "rehiragana": "\u308C",
    "rehyehaleflamarabic": "\u0631\uFEF3\uFE8E\u0644",
    "rekatakana": "\u30EC",
    "rekatakanahalfwidth": "\uFF9A",
    "resh": "\u05E8",
    "reshdageshhebrew": "\uFB48",
    "reshhatafpatah": "\u05E8\u05B2",
    "reshhatafpatahhebrew": "\u05E8\u05B2",
    "reshhatafsegol": "\u05E8\u05B1",
    "reshhatafsegolhebrew": "\u05E8\u05B1",
    "reshhebrew": "\u05E8",
    "reshhiriq": "\u05E8\u05B4",
    "reshhiriqhebrew": "\u05E8\u05B4",
    "reshholam": "\u05E8\u05B9",
    "reshholamhebrew": "\u05E8\u05B9",
    "reshpatah": "\u05E8\u05B7",
    "reshpatahhebrew": "\u05E8\u05B7",
    "reshqamats": "\u05E8\u05B8",
    "reshqamatshebrew": "\u05E8\u05B8",
    "reshqubuts": "\u05E8\u05BB",
    "reshqubutshebrew": "\u05E8\u05BB",
    "reshsegol": "\u05E8\u05B6",
    "reshsegolhebrew": "\u05E8\u05B6",
    "reshsheva": "\u05E8\u05B0",
    "reshshevahebrew": "\u05E8\u05B0",
    "reshtsere": "\u05E8\u05B5",
    "reshtserehebrew": "\u05E8\u05B5",
    "reversedtilde": "\u223D",
    "reviahebrew": "\u0597",
    "reviamugrashhebrew": "\u0597",
    "revlogicalnot": "\u2310",
    "rfishhook": "\u027E",
    "rfishhookreversed": "\u027F",
    "rhabengali": "\u09DD",
    "rhadeva": "\u095D",
    "rho": "\u03C1",
    "rhook": "\u027D",
    "rhookturned": "\u027B",
    "rhookturnedsuperior": "\u02B5",
    "rhosymbolgreek": "\u03F1",
    "rhotichookmod": "\u02DE",
    "rieulacirclekorean": "\u3271",
    "rieulaparenkorean": "\u3211",
    "rieulcirclekorean": "\u3263",
    "rieulhieuhkorean": "\u3140",
    "rieulkiyeokkorean": "\u313A",
    "rieulkiyeoksioskorean": "\u3169",
    "rieulkorean": "\u3139",
    "rieulmieumkorean": "\u313B",
    "rieulpansioskorean": "\u316C",
    "rieulparenkorean": "\u3203",
    "rieulphieuphkorean": "\u313F",
    "rieulpieupkorean": "\u313C",
    "rieulpieupsioskorean": "\u316B",
    "rieulsioskorean": "\u313D",
    "rieulthieuthkorean": "\u313E",
    "rieultikeutkorean": "\u316A",
    "rieulyeorinhieuhkorean": "\u316D",
    "rightangle": "\u221F",
    "righttackbelowcmb": "\u0319",
    "righttriangle": "\u22BF",
    "rihiragana": "\u308A",
    "rikatakana": "\u30EA",
    "rikatakanahalfwidth": "\uFF98",
    "ring": "\u02DA",
    "ringbelowcmb": "\u0325",
    "ringcmb": "\u030A",
    "ringhalfleft": "\u02BF",
    "ringhalfleftarmenian": "\u0559",
    "ringhalfleftbelowcmb": "\u031C",
    "ringhalfleftcentered": "\u02D3",
    "ringhalfright": "\u02BE",
    "ringhalfrightbelowcmb": "\u0339",
    "ringhalfrightcentered": "\u02D2",
    "rinvertedbreve": "\u0213",
    "rittorusquare": "\u3351",
    "rlinebelow": "\u1E5F",
    "rlongleg": "\u027C",
    "rlonglegturned": "\u027A",
    "rmonospace": "\uFF52",
    "rohiragana": "\u308D",
    "rokatakana": "\u30ED",
    "rokatakanahalfwidth": "\uFF9B",
    "roruathai": "\u0E23",
    "rparen": "\u24AD",
    "rrabengali": "\u09DC",
    "rradeva": "\u0931",
    "rragurmukhi": "\u0A5C",
    "rreharabic": "\u0691",
    "rrehfinalarabic": "\uFB8D",
    "rrvocalicbengali": "\u09E0",
    "rrvocalicdeva": "\u0960",
    "rrvocalicgujarati": "\u0AE0",
    "rrvocalicvowelsignbengali": "\u09C4",
    "rrvocalicvowelsigndeva": "\u0944",
    "rrvocalicvowelsigngujarati": "\u0AC4",
    "rsuperior": "\uF6F1",
    "rtblock": "\u2590",
    "rturned": "\u0279",
    "rturnedsuperior": "\u02B4",
    "ruhiragana": "\u308B",
    "rukatakana": "\u30EB",
    "rukatakanahalfwidth": "\uFF99",
    "rupeemarkbengali": "\u09F2",
    "rupeesignbengali": "\u09F3",
    "rupiah": "\uF6DD",
    "ruthai": "\u0E24",
    "rvocalicbengali": "\u098B",
    "rvocalicdeva": "\u090B",
    "rvocalicgujarati": "\u0A8B",
    "rvocalicvowelsignbengali": "\u09C3",
    "rvocalicvowelsigndeva": "\u0943",
    "rvocalicvowelsigngujarati": "\u0AC3",
    "s": "\u0073",
    "sabengali": "\u09B8",
    "sacute": "\u015B",
    "sacutedotaccent": "\u1E65",
    "sadarabic": "\u0635",
    "sadeva": "\u0938",
    "sadfinalarabic": "\uFEBA",
    "sadinitialarabic": "\uFEBB",
    "sadmedialarabic": "\uFEBC",
    "sagujarati": "\u0AB8",
    "sagurmukhi": "\u0A38",
    "sahiragana": "\u3055",
    "sakatakana": "\u30B5",
    "sakatakanahalfwidth": "\uFF7B",
    "sallallahoualayhewasallamarabic": "\uFDFA",
    "samekh": "\u05E1",
    "samekhdagesh": "\uFB41",
    "samekhdageshhebrew": "\uFB41",
    "samekhhebrew": "\u05E1",
    "saraaathai": "\u0E32",
    "saraaethai": "\u0E41",
    "saraaimaimalaithai": "\u0E44",
    "saraaimaimuanthai": "\u0E43",
    "saraamthai": "\u0E33",
    "saraathai": "\u0E30",
    "saraethai": "\u0E40",
    "saraiileftthai": "\uF886",
    "saraiithai": "\u0E35",
    "saraileftthai": "\uF885",
    "saraithai": "\u0E34",
    "saraothai": "\u0E42",
    "saraueeleftthai": "\uF888",
    "saraueethai": "\u0E37",
    "saraueleftthai": "\uF887",
    "sarauethai": "\u0E36",
    "sarauthai": "\u0E38",
    "sarauuthai": "\u0E39",
    "sbopomofo": "\u3119",
    "scaron": "\u0161",
    "scarondotaccent": "\u1E67",
    "scedilla": "\u015F",
    "schwa": "\u0259",
    "schwacyrillic": "\u04D9",
    "schwadieresiscyrillic": "\u04DB",
    "schwahook": "\u025A",
    "scircle": "\u24E2",
    "scircumflex": "\u015D",
    "scommaaccent": "\u0219",
    "sdotaccent": "\u1E61",
    "sdotbelow": "\u1E63",
    "sdotbelowdotaccent": "\u1E69",
    "seagullbelowcmb": "\u033C",
    "second": "\u2033",
    "secondtonechinese": "\u02CA",
    "section": "\u00A7",
    "seenarabic": "\u0633",
    "seenfinalarabic": "\uFEB2",
    "seeninitialarabic": "\uFEB3",
    "seenmedialarabic": "\uFEB4",
    "segol": "\u05B6",
    "segol13": "\u05B6",
    "segol1f": "\u05B6",
    "segol2c": "\u05B6",
    "segolhebrew": "\u05B6",
    "segolnarrowhebrew": "\u05B6",
    "segolquarterhebrew": "\u05B6",
    "segoltahebrew": "\u0592",
    "segolwidehebrew": "\u05B6",
    "seharmenian": "\u057D",
    "sehiragana": "\u305B",
    "sekatakana": "\u30BB",
    "sekatakanahalfwidth": "\uFF7E",
    "semicolon": "\u003B",
    "semicolonarabic": "\u061B",
    "semicolonmonospace": "\uFF1B",
    "semicolonsmall": "\uFE54",
    "semivoicedmarkkana": "\u309C",
    "semivoicedmarkkanahalfwidth": "\uFF9F",
    "sentisquare": "\u3322",
    "sentosquare": "\u3323",
    "seven": "\u0037",
    "sevenarabic": "\u0667",
    "sevenbengali": "\u09ED",
    "sevencircle": "\u2466",
    "sevencircleinversesansserif": "\u2790",
    "sevendeva": "\u096D",
    "seveneighths": "\u215E",
    "sevengujarati": "\u0AED",
    "sevengurmukhi": "\u0A6D",
    "sevenhackarabic": "\u0667",
    "sevenhangzhou": "\u3027",
    "sevenideographicparen": "\u3226",
    "seveninferior": "\u2087",
    "sevenmonospace": "\uFF17",
    "sevenoldstyle": "\uF737",
    "sevenparen": "\u247A",
    "sevenperiod": "\u248E",
    "sevenpersian": "\u06F7",
    "sevenroman": "\u2176",
    "sevensuperior": "\u2077",
    "seventeencircle": "\u2470",
    "seventeenparen": "\u2484",
    "seventeenperiod": "\u2498",
    "seventhai": "\u0E57",
    "sfthyphen": "\u00AD",
    "shaarmenian": "\u0577",
    "shabengali": "\u09B6",
    "shacyrillic": "\u0448",
    "shaddaarabic": "\u0651",
    "shaddadammaarabic": "\uFC61",
    "shaddadammatanarabic": "\uFC5E",
    "shaddafathaarabic": "\uFC60",
    "shaddafathatanarabic": "\u0651\u064B",
    "shaddakasraarabic": "\uFC62",
    "shaddakasratanarabic": "\uFC5F",
    "shade": "\u2592",
    "shadedark": "\u2593",
    "shadelight": "\u2591",
    "shademedium": "\u2592",
    "shadeva": "\u0936",
    "shagujarati": "\u0AB6",
    "shagurmukhi": "\u0A36",
    "shalshelethebrew": "\u0593",
    "shbopomofo": "\u3115",
    "shchacyrillic": "\u0449",
    "sheenarabic": "\u0634",
    "sheenfinalarabic": "\uFEB6",
    "sheeninitialarabic": "\uFEB7",
    "sheenmedialarabic": "\uFEB8",
    "sheicoptic": "\u03E3",
    "sheqel": "\u20AA",
    "sheqelhebrew": "\u20AA",
    "sheva": "\u05B0",
    "sheva115": "\u05B0",
    "sheva15": "\u05B0",
    "sheva22": "\u05B0",
    "sheva2e": "\u05B0",
    "shevahebrew": "\u05B0",
    "shevanarrowhebrew": "\u05B0",
    "shevaquarterhebrew": "\u05B0",
    "shevawidehebrew": "\u05B0",
    "shhacyrillic": "\u04BB",
    "shimacoptic": "\u03ED",
    "shin": "\u05E9",
    "shindagesh": "\uFB49",
    "shindageshhebrew": "\uFB49",
    "shindageshshindot": "\uFB2C",
    "shindageshshindothebrew": "\uFB2C",
    "shindageshsindot": "\uFB2D",
    "shindageshsindothebrew": "\uFB2D",
    "shindothebrew": "\u05C1",
    "shinhebrew": "\u05E9",
    "shinshindot": "\uFB2A",
    "shinshindothebrew": "\uFB2A",
    "shinsindot": "\uFB2B",
    "shinsindothebrew": "\uFB2B",
    "shook": "\u0282",
    "sigma": "\u03C3",
    "sigma1": "\u03C2",
    "sigmafinal": "\u03C2",
    "sigmalunatesymbolgreek": "\u03F2",
    "sihiragana": "\u3057",
    "sikatakana": "\u30B7",
    "sikatakanahalfwidth": "\uFF7C",
    "siluqhebrew": "\u05BD",
    "siluqlefthebrew": "\u05BD",
    "similar": "\u223C",
    "sindothebrew": "\u05C2",
    "siosacirclekorean": "\u3274",
    "siosaparenkorean": "\u3214",
    "sioscieuckorean": "\u317E",
    "sioscirclekorean": "\u3266",
    "sioskiyeokkorean": "\u317A",
    "sioskorean": "\u3145",
    "siosnieunkorean": "\u317B",
    "siosparenkorean": "\u3206",
    "siospieupkorean": "\u317D",
    "siostikeutkorean": "\u317C",
    "six": "\u0036",
    "sixarabic": "\u0666",
    "sixbengali": "\u09EC",
    "sixcircle": "\u2465",
    "sixcircleinversesansserif": "\u278F",
    "sixdeva": "\u096C",
    "sixgujarati": "\u0AEC",
    "sixgurmukhi": "\u0A6C",
    "sixhackarabic": "\u0666",
    "sixhangzhou": "\u3026",
    "sixideographicparen": "\u3225",
    "sixinferior": "\u2086",
    "sixmonospace": "\uFF16",
    "sixoldstyle": "\uF736",
    "sixparen": "\u2479",
    "sixperiod": "\u248D",
    "sixpersian": "\u06F6",
    "sixroman": "\u2175",
    "sixsuperior": "\u2076",
    "sixteencircle": "\u246F",
    "sixteencurrencydenominatorbengali": "\u09F9",
    "sixteenparen": "\u2483",
    "sixteenperiod": "\u2497",
    "sixthai": "\u0E56",
    "slash": "\u002F",
    "slashmonospace": "\uFF0F",
    "slong": "\u017F",
    "slongdotaccent": "\u1E9B",
    "smileface": "\u263A",
    "smonospace": "\uFF53",
    "sofpasuqhebrew": "\u05C3",
    "softhyphen": "\u00AD",
    "softsigncyrillic": "\u044C",
    "sohiragana": "\u305D",
    "sokatakana": "\u30BD",
    "sokatakanahalfwidth": "\uFF7F",
    "soliduslongoverlaycmb": "\u0338",
    "solidusshortoverlaycmb": "\u0337",
    "sorusithai": "\u0E29",
    "sosalathai": "\u0E28",
    "sosothai": "\u0E0B",
    "sosuathai": "\u0E2A",
    "space": "\u0020",
    "spacehackarabic": "\u0020",
    "spade": "\u2660",
    "spadesuitblack": "\u2660",
    "spadesuitwhite": "\u2664",
    "sparen": "\u24AE",
    "squarebelowcmb": "\u033B",
    "squarecc": "\u33C4",
    "squarecm": "\u339D",
    "squarediagonalcrosshatchfill": "\u25A9",
    "squarehorizontalfill": "\u25A4",
    "squarekg": "\u338F",
    "squarekm": "\u339E",
    "squarekmcapital": "\u33CE",
    "squareln": "\u33D1",
    "squarelog": "\u33D2",
    "squaremg": "\u338E",
    "squaremil": "\u33D5",
    "squaremm": "\u339C",
    "squaremsquared": "\u33A1",
    "squareorthogonalcrosshatchfill": "\u25A6",
    "squareupperlefttolowerrightfill": "\u25A7",
    "squareupperrighttolowerleftfill": "\u25A8",
    "squareverticalfill": "\u25A5",
    "squarewhitewithsmallblack": "\u25A3",
    "srsquare": "\u33DB",
    "ssabengali": "\u09B7",
    "ssadeva": "\u0937",
    "ssagujarati": "\u0AB7",
    "ssangcieuckorean": "\u3149",
    "ssanghieuhkorean": "\u3185",
    "ssangieungkorean": "\u3180",
    "ssangkiyeokkorean": "\u3132",
    "ssangnieunkorean": "\u3165",
    "ssangpieupkorean": "\u3143",
    "ssangsioskorean": "\u3146",
    "ssangtikeutkorean": "\u3138",
    "ssuperior": "\uF6F2",
    "sterling": "\u00A3",
    "sterlingmonospace": "\uFFE1",
    "strokelongoverlaycmb": "\u0336",
    "strokeshortoverlaycmb": "\u0335",
    "subset": "\u2282",
    "subsetnotequal": "\u228A",
    "subsetorequal": "\u2286",
    "succeeds": "\u227B",
    "suchthat": "\u220B",
    "suhiragana": "\u3059",
    "sukatakana": "\u30B9",
    "sukatakanahalfwidth": "\uFF7D",
    "sukunarabic": "\u0652",
    "summation": "\u2211",
    "sun": "\u263C",
    "superset": "\u2283",
    "supersetnotequal": "\u228B",
    "supersetorequal": "\u2287",
    "svsquare": "\u33DC",
    "syouwaerasquare": "\u337C",
    "t": "\u0074",
    "tabengali": "\u09A4",
    "tackdown": "\u22A4",
    "tackleft": "\u22A3",
    "tadeva": "\u0924",
    "tagujarati": "\u0AA4",
    "tagurmukhi": "\u0A24",
    "taharabic": "\u0637",
    "tahfinalarabic": "\uFEC2",
    "tahinitialarabic": "\uFEC3",
    "tahiragana": "\u305F",
    "tahmedialarabic": "\uFEC4",
    "taisyouerasquare": "\u337D",
    "takatakana": "\u30BF",
    "takatakanahalfwidth": "\uFF80",
    "tatweelarabic": "\u0640",
    "tau": "\u03C4",
    "tav": "\u05EA",
    "tavdages": "\uFB4A",
    "tavdagesh": "\uFB4A",
    "tavdageshhebrew": "\uFB4A",
    "tavhebrew": "\u05EA",
    "tbar": "\u0167",
    "tbopomofo": "\u310A",
    "tcaron": "\u0165",
    "tccurl": "\u02A8",
    "tcedilla": "\u0163",
    "tcheharabic": "\u0686",
    "tchehfinalarabic": "\uFB7B",
    "tchehinitialarabic": "\uFB7C",
    "tchehmedialarabic": "\uFB7D",
    "tchehmeeminitialarabic": "\uFB7C\uFEE4",
    "tcircle": "\u24E3",
    "tcircumflexbelow": "\u1E71",
    "tcommaaccent": "\u0163",
    "tdieresis": "\u1E97",
    "tdotaccent": "\u1E6B",
    "tdotbelow": "\u1E6D",
    "tecyrillic": "\u0442",
    "tedescendercyrillic": "\u04AD",
    "teharabic": "\u062A",
    "tehfinalarabic": "\uFE96",
    "tehhahinitialarabic": "\uFCA2",
    "tehhahisolatedarabic": "\uFC0C",
    "tehinitialarabic": "\uFE97",
    "tehiragana": "\u3066",
    "tehjeeminitialarabic": "\uFCA1",
    "tehjeemisolatedarabic": "\uFC0B",
    "tehmarbutaarabic": "\u0629",
    "tehmarbutafinalarabic": "\uFE94",
    "tehmedialarabic": "\uFE98",
    "tehmeeminitialarabic": "\uFCA4",
    "tehmeemisolatedarabic": "\uFC0E",
    "tehnoonfinalarabic": "\uFC73",
    "tekatakana": "\u30C6",
    "tekatakanahalfwidth": "\uFF83",
    "telephone": "\u2121",
    "telephoneblack": "\u260E",
    "telishagedolahebrew": "\u05A0",
    "telishaqetanahebrew": "\u05A9",
    "tencircle": "\u2469",
    "tenideographicparen": "\u3229",
    "tenparen": "\u247D",
    "tenperiod": "\u2491",
    "tenroman": "\u2179",
    "tesh": "\u02A7",
    "tet": "\u05D8",
    "tetdagesh": "\uFB38",
    "tetdageshhebrew": "\uFB38",
    "tethebrew": "\u05D8",
    "tetsecyrillic": "\u04B5",
    "tevirhebrew": "\u059B",
    "tevirlefthebrew": "\u059B",
    "thabengali": "\u09A5",
    "thadeva": "\u0925",
    "thagujarati": "\u0AA5",
    "thagurmukhi": "\u0A25",
    "thalarabic": "\u0630",
    "thalfinalarabic": "\uFEAC",
    "thanthakhatlowleftthai": "\uF898",
    "thanthakhatlowrightthai": "\uF897",
    "thanthakhatthai": "\u0E4C",
    "thanthakhatupperleftthai": "\uF896",
    "theharabic": "\u062B",
    "thehfinalarabic": "\uFE9A",
    "thehinitialarabic": "\uFE9B",
    "thehmedialarabic": "\uFE9C",
    "thereexists": "\u2203",
    "therefore": "\u2234",
    "theta": "\u03B8",
    "theta1": "\u03D1",
    "thetasymbolgreek": "\u03D1",
    "thieuthacirclekorean": "\u3279",
    "thieuthaparenkorean": "\u3219",
    "thieuthcirclekorean": "\u326B",
    "thieuthkorean": "\u314C",
    "thieuthparenkorean": "\u320B",
    "thirteencircle": "\u246C",
    "thirteenparen": "\u2480",
    "thirteenperiod": "\u2494",
    "thonangmonthothai": "\u0E11",
    "thook": "\u01AD",
    "thophuthaothai": "\u0E12",
    "thorn": "\u00FE",
    "thothahanthai": "\u0E17",
    "thothanthai": "\u0E10",
    "thothongthai": "\u0E18",
    "thothungthai": "\u0E16",
    "thousandcyrillic": "\u0482",
    "thousandsseparatorarabic": "\u066C",
    "thousandsseparatorpersian": "\u066C",
    "three": "\u0033",
    "threearabic": "\u0663",
    "threebengali": "\u09E9",
    "threecircle": "\u2462",
    "threecircleinversesansserif": "\u278C",
    "threedeva": "\u0969",
    "threeeighths": "\u215C",
    "threegujarati": "\u0AE9",
    "threegurmukhi": "\u0A69",
    "threehackarabic": "\u0663",
    "threehangzhou": "\u3023",
    "threeideographicparen": "\u3222",
    "threeinferior": "\u2083",
    "threemonospace": "\uFF13",
    "threenumeratorbengali": "\u09F6",
    "threeoldstyle": "\uF733",
    "threeparen": "\u2476",
    "threeperiod": "\u248A",
    "threepersian": "\u06F3",
    "threequarters": "\u00BE",
    "threequartersemdash": "\uF6DE",
    "threeroman": "\u2172",
    "threesuperior": "\u00B3",
    "threethai": "\u0E53",
    "thzsquare": "\u3394",
    "tihiragana": "\u3061",
    "tikatakana": "\u30C1",
    "tikatakanahalfwidth": "\uFF81",
    "tikeutacirclekorean": "\u3270",
    "tikeutaparenkorean": "\u3210",
    "tikeutcirclekorean": "\u3262",
    "tikeutkorean": "\u3137",
    "tikeutparenkorean": "\u3202",
    "tilde": "\u02DC",
    "tildebelowcmb": "\u0330",
    "tildecmb": "\u0303",
    "tildecomb": "\u0303",
    "tildedoublecmb": "\u0360",
    "tildeoperator": "\u223C",
    "tildeoverlaycmb": "\u0334",
    "tildeverticalcmb": "\u033E",
    "timescircle": "\u2297",
    "tipehahebrew": "\u0596",
    "tipehalefthebrew": "\u0596",
    "tippigurmukhi": "\u0A70",
    "titlocyrilliccmb": "\u0483",
    "tiwnarmenian": "\u057F",
    "tlinebelow": "\u1E6F",
    "tmonospace": "\uFF54",
    "toarmenian": "\u0569",
    "tohiragana": "\u3068",
    "tokatakana": "\u30C8",
    "tokatakanahalfwidth": "\uFF84",
    "tonebarextrahighmod": "\u02E5",
    "tonebarextralowmod": "\u02E9",
    "tonebarhighmod": "\u02E6",
    "tonebarlowmod": "\u02E8",
    "tonebarmidmod": "\u02E7",
    "tonefive": "\u01BD",
    "tonesix": "\u0185",
    "tonetwo": "\u01A8",
    "tonos": "\u0384",
    "tonsquare": "\u3327",
    "topatakthai": "\u0E0F",
    "tortoiseshellbracketleft": "\u3014",
    "tortoiseshellbracketleftsmall": "\uFE5D",
    "tortoiseshellbracketleftvertical": "\uFE39",
    "tortoiseshellbracketright": "\u3015",
    "tortoiseshellbracketrightsmall": "\uFE5E",
    "tortoiseshellbracketrightvertical": "\uFE3A",
    "totaothai": "\u0E15",
    "tpalatalhook": "\u01AB",
    "tparen": "\u24AF",
    "trademark": "\u2122",
    "trademarksans": "\uF8EA",
    "trademarkserif": "\uF6DB",
    "tretroflexhook": "\u0288",
    "triagdn": "\u25BC",
    "triaglf": "\u25C4",
    "triagrt": "\u25BA",
    "triagup": "\u25B2",
    "ts": "\u02A6",
    "tsadi": "\u05E6",
    "tsadidagesh": "\uFB46",
    "tsadidageshhebrew": "\uFB46",
    "tsadihebrew": "\u05E6",
    "tsecyrillic": "\u0446",
    "tsere": "\u05B5",
    "tsere12": "\u05B5",
    "tsere1e": "\u05B5",
    "tsere2b": "\u05B5",
    "tserehebrew": "\u05B5",
    "tserenarrowhebrew": "\u05B5",
    "tserequarterhebrew": "\u05B5",
    "tserewidehebrew": "\u05B5",
    "tshecyrillic": "\u045B",
    "tsuperior": "\uF6F3",
    "ttabengali": "\u099F",
    "ttadeva": "\u091F",
    "ttagujarati": "\u0A9F",
    "ttagurmukhi": "\u0A1F",
    "tteharabic": "\u0679",
    "ttehfinalarabic": "\uFB67",
    "ttehinitialarabic": "\uFB68",
    "ttehmedialarabic": "\uFB69",
    "tthabengali": "\u09A0",
    "tthadeva": "\u0920",
    "tthagujarati": "\u0AA0",
    "tthagurmukhi": "\u0A20",
    "tturned": "\u0287",
    "tuhiragana": "\u3064",
    "tukatakana": "\u30C4",
    "tukatakanahalfwidth": "\uFF82",
    "tusmallhiragana": "\u3063",
    "tusmallkatakana": "\u30C3",
    "tusmallkatakanahalfwidth": "\uFF6F",
    "twelvecircle": "\u246B",
    "twelveparen": "\u247F",
    "twelveperiod": "\u2493",
    "twelveroman": "\u217B",
    "twentycircle": "\u2473",
    "twentyhangzhou": "\u5344",
    "twentyparen": "\u2487",
    "twentyperiod": "\u249B",
    "two": "\u0032",
    "twoarabic": "\u0662",
    "twobengali": "\u09E8",
    "twocircle": "\u2461",
    "twocircleinversesansserif": "\u278B",
    "twodeva": "\u0968",
    "twodotenleader": "\u2025",
    "twodotleader": "\u2025",
    "twodotleadervertical": "\uFE30",
    "twogujarati": "\u0AE8",
    "twogurmukhi": "\u0A68",
    "twohackarabic": "\u0662",
    "twohangzhou": "\u3022",
    "twoideographicparen": "\u3221",
    "twoinferior": "\u2082",
    "twomonospace": "\uFF12",
    "twonumeratorbengali": "\u09F5",
    "twooldstyle": "\uF732",
    "twoparen": "\u2475",
    "twoperiod": "\u2489",
    "twopersian": "\u06F2",
    "tworoman": "\u2171",
    "twostroke": "\u01BB",
    "twosuperior": "\u00B2",
    "twothai": "\u0E52",
    "twothirds": "\u2154",
    "u": "\u0075",
    "uacute": "\u00FA",
    "ubar": "\u0289",
    "ubengali": "\u0989",
    "ubopomofo": "\u3128",
    "ubreve": "\u016D",
    "ucaron": "\u01D4",
    "ucircle": "\u24E4",
    "ucircumflex": "\u00FB",
    "ucircumflexbelow": "\u1E77",
    "ucyrillic": "\u0443",
    "udattadeva": "\u0951",
    "udblacute": "\u0171",
    "udblgrave": "\u0215",
    "udeva": "\u0909",
    "udieresis": "\u00FC",
    "udieresisacute": "\u01D8",
    "udieresisbelow": "\u1E73",
    "udieresiscaron": "\u01DA",
    "udieresiscyrillic": "\u04F1",
    "udieresisgrave": "\u01DC",
    "udieresismacron": "\u01D6",
    "udotbelow": "\u1EE5",
    "ugrave": "\u00F9",
    "ugujarati": "\u0A89",
    "ugurmukhi": "\u0A09",
    "uhiragana": "\u3046",
    "uhookabove": "\u1EE7",
    "uhorn": "\u01B0",
    "uhornacute": "\u1EE9",
    "uhorndotbelow": "\u1EF1",
    "uhorngrave": "\u1EEB",
    "uhornhookabove": "\u1EED",
    "uhorntilde": "\u1EEF",
    "uhungarumlaut": "\u0171",
    "uhungarumlautcyrillic": "\u04F3",
    "uinvertedbreve": "\u0217",
    "ukatakana": "\u30A6",
    "ukatakanahalfwidth": "\uFF73",
    "ukcyrillic": "\u0479",
    "ukorean": "\u315C",
    "umacron": "\u016B",
    "umacroncyrillic": "\u04EF",
    "umacrondieresis": "\u1E7B",
    "umatragurmukhi": "\u0A41",
    "umonospace": "\uFF55",
    "underscore": "\u005F",
    "underscoredbl": "\u2017",
    "underscoremonospace": "\uFF3F",
    "underscorevertical": "\uFE33",
    "underscorewavy": "\uFE4F",
    "union": "\u222A",
    "universal": "\u2200",
    "uogonek": "\u0173",
    "uparen": "\u24B0",
    "upblock": "\u2580",
    "upperdothebrew": "\u05C4",
    "upsilon": "\u03C5",
    "upsilondieresis": "\u03CB",
    "upsilondieresistonos": "\u03B0",
    "upsilonlatin": "\u028A",
    "upsilontonos": "\u03CD",
    "uptackbelowcmb": "\u031D",
    "uptackmod": "\u02D4",
    "uragurmukhi": "\u0A73",
    "uring": "\u016F",
    "ushortcyrillic": "\u045E",
    "usmallhiragana": "\u3045",
    "usmallkatakana": "\u30A5",
    "usmallkatakanahalfwidth": "\uFF69",
    "ustraightcyrillic": "\u04AF",
    "ustraightstrokecyrillic": "\u04B1",
    "utilde": "\u0169",
    "utildeacute": "\u1E79",
    "utildebelow": "\u1E75",
    "uubengali": "\u098A",
    "uudeva": "\u090A",
    "uugujarati": "\u0A8A",
    "uugurmukhi": "\u0A0A",
    "uumatragurmukhi": "\u0A42",
    "uuvowelsignbengali": "\u09C2",
    "uuvowelsigndeva": "\u0942",
    "uuvowelsigngujarati": "\u0AC2",
    "uvowelsignbengali": "\u09C1",
    "uvowelsigndeva": "\u0941",
    "uvowelsigngujarati": "\u0AC1",
    "v": "\u0076",
    "vadeva": "\u0935",
    "vagujarati": "\u0AB5",
    "vagurmukhi": "\u0A35",
    "vakatakana": "\u30F7",
    "vav": "\u05D5",
    "vavdagesh": "\uFB35",
    "vavdagesh65": "\uFB35",
    "vavdageshhebrew": "\uFB35",
    "vavhebrew": "\u05D5",
    "vavholam": "\uFB4B",
    "vavholamhebrew": "\uFB4B",
    "vavvavhebrew": "\u05F0",
    "vavyodhebrew": "\u05F1",
    "vcircle": "\u24E5",
    "vdotbelow": "\u1E7F",
    "vecyrillic": "\u0432",
    "veharabic": "\u06A4",
    "vehfinalarabic": "\uFB6B",
    "vehinitialarabic": "\uFB6C",
    "vehmedialarabic": "\uFB6D",
    "vekatakana": "\u30F9",
    "venus": "\u2640",
    "verticalbar": "\u007C",
    "verticallineabovecmb": "\u030D",
    "verticallinebelowcmb": "\u0329",
    "verticallinelowmod": "\u02CC",
    "verticallinemod": "\u02C8",
    "vewarmenian": "\u057E",
    "vhook": "\u028B",
    "vikatakana": "\u30F8",
    "viramabengali": "\u09CD",
    "viramadeva": "\u094D",
    "viramagujarati": "\u0ACD",
    "visargabengali": "\u0983",
    "visargadeva": "\u0903",
    "visargagujarati": "\u0A83",
    "vmonospace": "\uFF56",
    "voarmenian": "\u0578",
    "voicediterationhiragana": "\u309E",
    "voicediterationkatakana": "\u30FE",
    "voicedmarkkana": "\u309B",
    "voicedmarkkanahalfwidth": "\uFF9E",
    "vokatakana": "\u30FA",
    "vparen": "\u24B1",
    "vtilde": "\u1E7D",
    "vturned": "\u028C",
    "vuhiragana": "\u3094",
    "vukatakana": "\u30F4",
    "w": "\u0077",
    "wacute": "\u1E83",
    "waekorean": "\u3159",
    "wahiragana": "\u308F",
    "wakatakana": "\u30EF",
    "wakatakanahalfwidth": "\uFF9C",
    "wakorean": "\u3158",
    "wasmallhiragana": "\u308E",
    "wasmallkatakana": "\u30EE",
    "wattosquare": "\u3357",
    "wavedash": "\u301C",
    "wavyunderscorevertical": "\uFE34",
    "wawarabic": "\u0648",
    "wawfinalarabic": "\uFEEE",
    "wawhamzaabovearabic": "\u0624",
    "wawhamzaabovefinalarabic": "\uFE86",
    "wbsquare": "\u33DD",
    "wcircle": "\u24E6",
    "wcircumflex": "\u0175",
    "wdieresis": "\u1E85",
    "wdotaccent": "\u1E87",
    "wdotbelow": "\u1E89",
    "wehiragana": "\u3091",
    "weierstrass": "\u2118",
    "wekatakana": "\u30F1",
    "wekorean": "\u315E",
    "weokorean": "\u315D",
    "wgrave": "\u1E81",
    "whitebullet": "\u25E6",
    "whitecircle": "\u25CB",
    "whitecircleinverse": "\u25D9",
    "whitecornerbracketleft": "\u300E",
    "whitecornerbracketleftvertical": "\uFE43",
    "whitecornerbracketright": "\u300F",
    "whitecornerbracketrightvertical": "\uFE44",
    "whitediamond": "\u25C7",
    "whitediamondcontainingblacksmalldiamond": "\u25C8",
    "whitedownpointingsmalltriangle": "\u25BF",
    "whitedownpointingtriangle": "\u25BD",
    "whiteleftpointingsmalltriangle": "\u25C3",
    "whiteleftpointingtriangle": "\u25C1",
    "whitelenticularbracketleft": "\u3016",
    "whitelenticularbracketright": "\u3017",
    "whiterightpointingsmalltriangle": "\u25B9",
    "whiterightpointingtriangle": "\u25B7",
    "whitesmallsquare": "\u25AB",
    "whitesmilingface": "\u263A",
    "whitesquare": "\u25A1",
    "whitestar": "\u2606",
    "whitetelephone": "\u260F",
    "whitetortoiseshellbracketleft": "\u3018",
    "whitetortoiseshellbracketright": "\u3019",
    "whiteuppointingsmalltriangle": "\u25B5",
    "whiteuppointingtriangle": "\u25B3",
    "wihiragana": "\u3090",
    "wikatakana": "\u30F0",
    "wikorean": "\u315F",
    "wmonospace": "\uFF57",
    "wohiragana": "\u3092",
    "wokatakana": "\u30F2",
    "wokatakanahalfwidth": "\uFF66",
    "won": "\u20A9",
    "wonmonospace": "\uFFE6",
    "wowaenthai": "\u0E27",
    "wparen": "\u24B2",
    "wring": "\u1E98",
    "wsuperior": "\u02B7",
    "wturned": "\u028D",
    "wynn": "\u01BF",
    "x": "\u0078",
    "xabovecmb": "\u033D",
    "xbopomofo": "\u3112",
    "xcircle": "\u24E7",
    "xdieresis": "\u1E8D",
    "xdotaccent": "\u1E8B",
    "xeharmenian": "\u056D",
    "xi": "\u03BE",
    "xmonospace": "\uFF58",
    "xparen": "\u24B3",
    "xsuperior": "\u02E3",
    "y": "\u0079",
    "yaadosquare": "\u334E",
    "yabengali": "\u09AF",
    "yacute": "\u00FD",
    "yadeva": "\u092F",
    "yaekorean": "\u3152",
    "yagujarati": "\u0AAF",
    "yagurmukhi": "\u0A2F",
    "yahiragana": "\u3084",
    "yakatakana": "\u30E4",
    "yakatakanahalfwidth": "\uFF94",
    "yakorean": "\u3151",
    "yamakkanthai": "\u0E4E",
    "yasmallhiragana": "\u3083",
    "yasmallkatakana": "\u30E3",
    "yasmallkatakanahalfwidth": "\uFF6C",
    "yatcyrillic": "\u0463",
    "ycircle": "\u24E8",
    "ycircumflex": "\u0177",
    "ydieresis": "\u00FF",
    "ydotaccent": "\u1E8F",
    "ydotbelow": "\u1EF5",
    "yeharabic": "\u064A",
    "yehbarreearabic": "\u06D2",
    "yehbarreefinalarabic": "\uFBAF",
    "yehfinalarabic": "\uFEF2",
    "yehhamzaabovearabic": "\u0626",
    "yehhamzaabovefinalarabic": "\uFE8A",
    "yehhamzaaboveinitialarabic": "\uFE8B",
    "yehhamzaabovemedialarabic": "\uFE8C",
    "yehinitialarabic": "\uFEF3",
    "yehmedialarabic": "\uFEF4",
    "yehmeeminitialarabic": "\uFCDD",
    "yehmeemisolatedarabic": "\uFC58",
    "yehnoonfinalarabic": "\uFC94",
    "yehthreedotsbelowarabic": "\u06D1",
    "yekorean": "\u3156",
    "yen": "\u00A5",
    "yenmonospace": "\uFFE5",
    "yeokorean": "\u3155",
    "yeorinhieuhkorean": "\u3186",
    "yerahbenyomohebrew": "\u05AA",
    "yerahbenyomolefthebrew": "\u05AA",
    "yericyrillic": "\u044B",
    "yerudieresiscyrillic": "\u04F9",
    "yesieungkorean": "\u3181",
    "yesieungpansioskorean": "\u3183",
    "yesieungsioskorean": "\u3182",
    "yetivhebrew": "\u059A",
    "ygrave": "\u1EF3",
    "yhook": "\u01B4",
    "yhookabove": "\u1EF7",
    "yiarmenian": "\u0575",
    "yicyrillic": "\u0457",
    "yikorean": "\u3162",
    "yinyang": "\u262F",
    "yiwnarmenian": "\u0582",
    "ymonospace": "\uFF59",
    "yod": "\u05D9",
    "yoddagesh": "\uFB39",
    "yoddageshhebrew": "\uFB39",
    "yodhebrew": "\u05D9",
    "yodyodhebrew": "\u05F2",
    "yodyodpatahhebrew": "\uFB1F",
    "yohiragana": "\u3088",
    "yoikorean": "\u3189",
    "yokatakana": "\u30E8",
    "yokatakanahalfwidth": "\uFF96",
    "yokorean": "\u315B",
    "yosmallhiragana": "\u3087",
    "yosmallkatakana": "\u30E7",
    "yosmallkatakanahalfwidth": "\uFF6E",
    "yotgreek": "\u03F3",
    "yoyaekorean": "\u3188",
    "yoyakorean": "\u3187",
    "yoyakthai": "\u0E22",
    "yoyingthai": "\u0E0D",
    "yparen": "\u24B4",
    "ypogegrammeni": "\u037A",
    "ypogegrammenigreekcmb": "\u0345",
    "yr": "\u01A6",
    "yring": "\u1E99",
    "ysuperior": "\u02B8",
    "ytilde": "\u1EF9",
    "yturned": "\u028E",
    "yuhiragana": "\u3086",
    "yuikorean": "\u318C",
    "yukatakana": "\u30E6",
    "yukatakanahalfwidth": "\uFF95",
    "yukorean": "\u3160",
    "yusbigcyrillic": "\u046B",
    "yusbigiotifiedcyrillic": "\u046D",
    "yuslittlecyrillic": "\u0467",
    "yuslittleiotifiedcyrillic": "\u0469",
    "yusmallhiragana": "\u3085",
    "yusmallkatakana": "\u30E5",
    "yusmallkatakanahalfwidth": "\uFF6D",
    "yuyekorean": "\u318B",
    "yuyeokorean": "\u318A",
    "yyabengali": "\u09DF",
    "yyadeva": "\u095F",
    "z": "\u007A",
    "zaarmenian": "\u0566",
    "zacute": "\u017A",
    "zadeva": "\u095B",
    "zagurmukhi": "\u0A5B",
    "zaharabic": "\u0638",
    "zahfinalarabic": "\uFEC6",
    "zahinitialarabic": "\uFEC7",
    "zahiragana": "\u3056",
    "zahmedialarabic": "\uFEC8",
    "zainarabic": "\u0632",
    "zainfinalarabic": "\uFEB0",
    "zakatakana": "\u30B6",
    "zaqefgadolhebrew": "\u0595",
    "zaqefqatanhebrew": "\u0594",
    "zarqahebrew": "\u0598",
    "zayin": "\u05D6",
    "zayindagesh": "\uFB36",
    "zayindageshhebrew": "\uFB36",
    "zayinhebrew": "\u05D6",
    "zbopomofo": "\u3117",
    "zcaron": "\u017E",
    "zcircle": "\u24E9",
    "zcircumflex": "\u1E91",
    "zcurl": "\u0291",
    "zdot": "\u017C",
    "zdotaccent": "\u017C",
    "zdotbelow": "\u1E93",
    "zecyrillic": "\u0437",
    "zedescendercyrillic": "\u0499",
    "zedieresiscyrillic": "\u04DF",
    "zehiragana": "\u305C",
    "zekatakana": "\u30BC",
    "zero": "\u0030",
    "zeroarabic": "\u0660",
    "zerobengali": "\u09E6",
    "zerodeva": "\u0966",
    "zerogujarati": "\u0AE6",
    "zerogurmukhi": "\u0A66",
    "zerohackarabic": "\u0660",
    "zeroinferior": "\u2080",
    "zeromonospace": "\uFF10",
    "zerooldstyle": "\uF730",
    "zeropersian": "\u06F0",
    "zerosuperior": "\u2070",
    "zerothai": "\u0E50",
    "zerowidthjoiner": "\uFEFF",
    "zerowidthnonjoiner": "\u200C",
    "zerowidthspace": "\u200B",
    "zeta": "\u03B6",
    "zhbopomofo": "\u3113",
    "zhearmenian": "\u056A",
    "zhebrevecyrillic": "\u04C2",
    "zhecyrillic": "\u0436",
    "zhedescendercyrillic": "\u0497",
    "zhedieresiscyrillic": "\u04DD",
    "zihiragana": "\u3058",
    "zikatakana": "\u30B8",
    "zinorhebrew": "\u05AE",
    "zlinebelow": "\u1E95",
    "zmonospace": "\uFF5A",
    "zohiragana": "\u305E",
    "zokatakana": "\u30BE",
    "zparen": "\u24B5",
    "zretroflexhook": "\u0290",
    "zstroke": "\u01B6",
    "zuhiragana": "\u305A",
    "zukatakana": "\u30BA",
}
# --end
