#!/bin/bash
##########################################################################
#
# mplay.sh
#
# shell script for using with VDR and with mplayer-plugin
# inspired from 'mplayer.sh' but replaces it ;-)
#
# version: 0.5.3 (2007-04-28)
#
# Thomas Hildebrandt <toxym@web.de>
#
##########################################################################

# adapt this for your system configuration
CONFDIR=/etc/vdr/plugins/mplay


##########################################################################
# version
########################################
MPLAY_VERSION=0.5.3


##########################################################################
# functions
########################################

##########################################################################
# adds a file to the current playlist
# usage: add <file>
########################################
add(){
    local command

    # start with no error
    MPLAY_ERROR=

    log "Try to add '$1' to internal playlist."

    # set options for this file
    options "$1"

    # check for an error
    if [ "$MPLAY_ERROR" ]; then
        log "$MPLAY_ERROR"
        return
    fi

    # build command
    command="\"$1\" -vo $MPLAY_VOUT -ao $MPLAY_AOUT $MPLAY_OPTIONS"
    [ "$MPLAY_VF" ] && command="$command -vf $MPLAY_VF"
    [ "$MPLAY_AF" ] && command="$command -af $MPLAY_AF"

    # add command to play file
    log "Internal playlist entry is: '$command'"
    echo -n " $command" >> "$MPLAY_PLAY"

    ADDED=$(($ADDED+1))
}


##########################################################################
# log to syslog (if MPLAY_DEBUG is set and a logger is available)
# usage: log <message>
########################################
log(){
    if [ "$MPLAY_DEBUG" -a "$MPLAY_LOGGER" ]; then
        $MPLAY_LOGGER "$1"
    fi
}


##########################################################################
# set the options for a file
# usage: options <file>
###############.########################
options(){
    local extConf

    # begin with clean options
    MPLAY_OPTIONS=
    MPLAY_VF=
    MPLAY_AF=

    # trying to load special the special conf file
    # (this is maybe done already in global part of this script, but
    # if we process a playlist, the special options for the actual
    # file to play are not loaded)
    log "Searching for type config file for '${1##*.}'."
    extConf=`find $CONFDIR -iname ${1##*.}.conf`
    if [ -r "$extConf" ]; then
        . "$extConf"
        log "'$extConf' read"
    else
        log "'${1##*.}.conf' not found."
    fi

    # add type specific options to global options
    [ "$MPLAY_TYPE_OPTIONS" ] && MPLAY_OPTIONS="$MPLAY_OPTIONS $MPLAY_TYPE_OPTIONS"
    log "MPlayer options for this file are now: '$MPLAY_OPTIONS'"

    ### do the MPlayer identification (if requested)
    if [ "$MPLAY_IDENTIFY" = "yes" ]; then
        # let MPlayer identify the file
        FILE_ID=`"$MPLAY_MPLAYER" -identify -vo null -ao null -frames 1 -endpos 0.1 -nolirc "$1" 2>/dev/null`
        
	if [ -z "`echo \"$FILE_ID\" | grep ^ID_`" ]; then
	    MPLAY_ERROR="Error: MPlayer fails to identify '$1'."
	    return
	fi

        ### process video section
        log "Start processing video section..."
        MPLAY_TYPE_OPTIONS=
        optionsVideo "$FILE_ID"
        log "MPlayer options are now (after video processing): '$MPLAY_OPTIONS'"

        ### process audio section
        log "Start processing audio section..."
        MPLAY_TYPE_OPTIONS=
        optionsAudio "$FILE_ID"
        log "MPlayer options are now (after audio processing): '$MPLAY_OPTIONS'"

        ### process subtitles in extra file
        log "Start processing subtitle section..."
	if [ "`echo \"$FILE_ID\" | grep ^ID_FILE_SUB_ID`" -o "`echo \"$FILE_ID\" | grep ^ID_SUBTITLE_ID`" ]; then
            MPLAY_TYPE_OPTIONS=
            optionsSubtitle "$FILE_ID"
            log "MPlayer options are now (after subtitle processing): '$MPLAY_OPTIONS'"
	fi
    fi
}


##########################################################################
# takes care for the audio options of a file to play
# usage: optionsAudio <fileID>
########################################
optionsAudio(){
    local extConf format

    # get audio format of this file
    format=`echo "$1" | grep ID_AUDIO_CODEC | cut -d "=" -f 2`
    log "Audio format is: '$format'"

    # do more things only, if we have an audio format
    if [ "$format" ]; then
        # the options of playing audio may differ when playing the audio
        # standalone or from within a video file
        if [ "$MPLAY_VOUT" = "null" ]; then
            log "Searching for '$format.conf' and read it, if possible."
            extConf=`find $CONFDIR -iname $format.conf`
            if [ -r "$extConf" ]; then
                . "$extConf"
                log "'$extConf' read."
            else
                log "'$format.conf' not found."
            fi
        else
            # video found, try to load special video config
            log "Searching for 'video-$format.conf' and read it, if possible."
            extConf=`find $CONFDIR -iname video-$format.conf`
            if [ -r "$extConf" ]; then
                . "$extConf"
                log "'$extConf' read."
            else
                log "'video-$format.conf' not found."
            fi

            # care for DD 2.0
            if [ "$format" = "a52" -a "`echo "$1" | grep ^ID_AUDIO_BITRATE | tail -n 1 | cut -d "=" -f 2`" -lt 380000 ]; then
                log "Dolby Digital stream with low bitrate found, assuming, this is DD 2.0."

	        if [ "$MPLAY_DD20" ]; then
		    if [ "$MPLAY_AF" ]; then
		        MPLAY_AF="$MPLAY_AF,$MPLAY_DD20"
		    else
		        MPLAY_AF="$MPLAY_DD20"
		    fi
                    log "Special DD 2.0 settings applied."
		else
                    log "No special DD 2.0 settings found, using default settings for '$format'."
		fi
            fi	    
        fi

        # add type specific options to global options
        [ "$MPLAY_TYPE_OPTIONS" ] && MPLAY_OPTIONS="$MPLAY_OPTIONS $MPLAY_TYPE_OPTIONS"
    fi
}


##########################################################################
# takes care for the subtitle options of a file to play
# usage: optionsSubtitleFile <fileID>
########################################
optionsSubtitle(){
    local extConf format sid sub type

    # shall we show this subtitles?
    if [ "$MPLAY_SUB_SHOW" = "yes" ]; then
        # is a subtitle file already given?
	if [ "$MPLAY_SUB_FILE" ]; then
	    sub="$MPLAY_SUB_FILE"
	    type=file
	else
            # care for subtitle id
            sid=-1
            if [ "$MPLAY_SUB_ID" ]; then
	        sid=$MPLAY_SUB_ID
                log "Subtitle ID used: '$sid'"
	    else
	        log "Subtitle ID not specified, using language settings '$MPLAY_SUB_LANGUAGES'."
	    fi

            # try to get a filename of subtitle file
            type=file
            sub=`echo "$FILE_ID" | grep -A 1 ^ID_FILE_SUB_ID=$sid`
            sub=`echo "$sub" | grep ID_FILE_SUB_FILENAME | cut -d "=" -f 2`

            # when no file is found, get internal subtitle stream
            if [ -z "$sub" ]; then
                [ $sid -ge 0 ] $$ sub=`echo "$FILE_ID" | grep ^ID_SID_${sid}_LANG | cut -d "=" -f 2`
	        type=internal
            fi
	fi

        if [ "$type" = "file" ]; then
            log "Subtitles found in: '$sub'"

            # get file type
            format=${sub##*.}
            log "Subtitle format is: '$format'"

            # try to load the config file
            log "Searching for '$format.conf' and read it, if possible."
            extConf=`find $CONFDIR -iname $format.conf`
            if [ -r "$extConf" ]; then
                . "$extConf"
                log "'$extConf' read."
            else
                log "'$format.conf' not found."
            fi
        else
	    if [ $sid -ge 0 ]; then
	        log "Subtitles are an internal stream of '$MPLAY_FILE' in language '$sub'."
	    else
	        log "Subtitles are internal stream(s) of '$MPLAY_FILE'."
	    fi
        fi

        # sid
	[ $sid -ge 0 ] && MPLAY_TYPE_OPTIONS="$MPLAY_TYPE_OPTIONS -sid $sid"

        # languages
	[ "$MPLAY_SUB_LANGUAGES" ] && MPLAY_TYPE_OPTIONS="$MPLAY_TYPE_OPTIONS -slang $MPLAY_SUB_LANGUAGES"

        # vertical position
	[ "$MPLAY_SUB_POS" ] && MPLAY_TYPE_OPTIONS="$MPLAY_TYPE_OPTIONS -subpos $MPLAY_SUB_POS"

        # width on screen
	[ "$MPLAY_SUB_WIDTH" ] && MPLAY_TYPE_OPTIONS="$MPLAY_TYPE_OPTIONS -subwidth $MPLAY_SUB_WIDTH"

        # fps
	[ "$MPLAY_SUB_FPS" ] && MPLAY_TYPE_OPTIONS="$MPLAY_TYPE_OPTIONS -subfps $MPLAY_SUB_FPS"

        # delay
	[ "$MPLAY_SUB_DELAY" ] && MPLAY_TYPE_OPTIONS="$MPLAY_TYPE_OPTIONS -subdelay $MPLAY_SUB_DELAY"

        # background transparency
	[ "$MPLAY_SUB_BG_ALPHA" ] && MPLAY_TYPE_OPTIONS="$MPLAY_TYPE_OPTIONS -sub-bg-alpha $MPLAY_SUB_BG_ALPHA"

        # background 'color'
	[ "$MPLAY_SUB_BG_COLOR" ] && MPLAY_TYPE_OPTIONS="$MPLAY_TYPE_OPTIONS -sub-bg-color $MPLAY_SUB_BG_COLOR"
    else
        log "Subtitles are deactivated."
	MPLAY_TYPE_OPTIONS="-noautosub"
    fi

    # add type specific options to global options
    [ "$MPLAY_TYPE_OPTIONS" ] && MPLAY_OPTIONS="$MPLAY_OPTIONS $MPLAY_TYPE_OPTIONS"
}


##########################################################################
# takes care for the video options of a file to play
# usage: optionsVideo <fileID>
########################################
optionsVideo(){
    local aspect extConf fileX fileY format fps orgX orgY scale

    # get video format of this file
    format=`echo "$1" | grep ID_VIDEO_FORMAT | cut -d "=" -f 2`
    log "Video format is: '$format'"

    # do more things only, if we have a video format
    if [ "$format" ]; then
        # trying to load (another) special conf file
        # (this is made because for instance an AVI file can contain different
        #  video formats)
        log "Searching for '$format.conf' and read it, if possible."
        extConf=`find $CONFDIR -iname $format.conf`
        if [ -r "$extConf" ]; then
            . "$extConf"
            log "'$extConf' read."
        else
            log "'$format.conf' not found."
        fi

        # get resolution
        orgX=`echo "$1" | grep ID_VIDEO_WIDTH | cut -d "=" -f 2`
        orgY=`echo "$1" | grep ID_VIDEO_HEIGHT | cut -d "=" -f 2`
        fileX=$orgX
        fileY=$orgY
        log "Video resolution found: $orgX x $orgY"

        ### care for a possible set aspect ratio
	if [ "$MPLAY_ASPECT" ]; then
            aspect=`echo "$1" | grep ^ID_VIDEO_ASPECT | tail -n 1 | cut -d "=" -f 2`
            if [ "$aspect" -a "$aspect" != "0.0000" ]; then
	        log "Found aspect ratio set: '$aspect'"
                # okay, we have to process aspect settings
                # (MPlayer calculated this for us)
	        aspect=`echo "$1" | grep ^VO: | tail -n 1 | cut -d " " -f 5`
	        if [ "$aspect" ]; then
                    fileY=${aspect#*x}
                    fileX=${aspect%x*}
                    log "Cared for aspect ratio, resolution is now: $fileX x $fileY"
	        fi
            fi
	fi

        ### care for TV aspect correction
	if [ "$MPLAY_TV" ]; then
            fileY=$(($fileY*$MPLAY_TV))
	    log "TV aspect correction applied, resolution is now: $fileX x $fileY"
	fi	

        ### select correct output resolution
        # check for fixed set resolution
        # (this options can be set in the SET file)
        if [ "$MPLAY_RESOLUTION" ]; then
            MPLAY_X=${MPLAY_RESOLUTION%x*}
            MPLAY_Y=${MPLAY_RESOLUTION#*x}
        else
            if [ "$MPLAY_SCALE" = "up" ]; then
                # scaling video up
                log "Scaling video up (if needed)..."
                MPLAY_X=${MPLAY_RES1%x*}
                MPLAY_Y=${MPLAY_RES1#*x}
                if [ "$MPLAY_RES2" ]; then
		    if [ $fileX -lt ${MPLAY_RES2%x*} ]; then
                        MPLAY_X=${MPLAY_RES2%x*}
                        MPLAY_Y=${MPLAY_RES2#*x}
                        if [ "$MPLAY_RES3" ]; then
			    if [ $fileX -lt ${MPLAY_RES3%x*} ]; then
                                MPLAY_X=${MPLAY_RES3%x*}
                                MPLAY_Y=${MPLAY_RES3#*x}
			    fi
                        fi
		    fi
                fi
            else
                # scaling video down
                log "Scaling video down (if needed)..."
                MPLAY_X=${MPLAY_RES1%x*}
                MPLAY_Y=${MPLAY_RES1#*x}
                if [ "$MPLAY_RES2" ]; then
		    if [ $fileX -lt $MPLAY_X ]; then
                        MPLAY_X=${MPLAY_RES2%x*}
                        MPLAY_Y=${MPLAY_RES2#*x}
                        if [ "$MPLAY_RES3" ]; then
			    if [ $fileX -lt $MPLAY_X ]; then
                                MPLAY_X=${MPLAY_RES3%x*}
                                MPLAY_Y=${MPLAY_RES3#*x}
			    fi
                        fi
		    fi
                fi
            fi
        fi
        log "Calculated output resolution is: $MPLAY_X x $MPLAY_Y"

        # process X resolution
        if [ "$MPLAY_SCALE" = "up" -a $fileX -lt $MPLAY_X ]; then
            fileY=$(($MPLAY_X*$fileY/$fileX))
            fileX=$MPLAY_X
	fi
        if [ $fileX -gt $MPLAY_X ]; then
            fileY=$(($MPLAY_X*$fileY/$fileX))
            fileX=$MPLAY_X
        fi

        # process/correct Y if necessary
        if [ $fileY -gt $MPLAY_Y ]; then
            fileX=$(($MPLAY_Y*$fileX/$fileY))
            fileY=$MPLAY_Y
        fi

        log "Calculated video resolution is: $fileX x $fileY"

        # check and care for padding
        if [ "$MPLAY_PADDING" ]; then
            log "Padding option found: ${MPLAY_PADDING}px"
            paddedX=$(($fileX-2*$MPLAY_PADDING))
            fileY=$(($paddedX*$fileY/$fileX))
            fileX=$paddedX
            log "Padded video resolution is: $fileX x $fileY"
        fi

        # build video filter
	[ "$MPLAY_PP" ] && MPLAY_VF="pp=$MPLAY_PP,"

        if [ "$MPLAY_SCALE_ALWAYS" ]; then
            # scale always, no matter what the resolution says
            MPLAY_VF="${MPLAY_VF}scale=$fileX:$fileY,"
        else
            # scale only, if needed
            [ $fileX -ne $orgX -o $fileY -ne $orgY ] && MPLAY_VF="${MPLAY_VF}scale=$fileX:$fileY,"
        fi

        [ $fileX -ne $MPLAY_X -o $fileY -ne $MPLAY_Y ] && MPLAY_VF="${MPLAY_VF}expand=$MPLAY_X:$MPLAY_Y:::1,"

        [ "$MPLAY_VF" -a "$MPLAY_LAVC" ] && MPLAY_VF="${MPLAY_VF}lavc=$MPLAY_LAVC"


        # care for FPS
	# (this part is based on Juri's work, thx :-))
        fps=`echo "$1" | grep ID_VIDEO_FPS | cut -d "=" -f 2 | sed 's/\.//'`
	if [ $fps -ge 23000 -a $fps -le 24499 ]; then
	    log "Found $fps/1000 fps, applying speed correction"
	    MPLAY_OPTIONS="$MPLAY_OPTIONS -speed 25000/$fps"
	fi
    else
        # when no video resolution is found, it can't be a supported video
        # format and we don't need any video output
        MPLAY_VOUT="null"
        MPLAY_VF=
    fi

    log "MPlayer video output device is '$MPLAY_VOUT'."
    log "Video filter is now: '$MPLAY_VF'"
}


##########################################################################
# clean up and exit
########################################
quit(){
    # remove tmp file
    rm -f "$MPLAY_PLAY"

    # restore soundset
    if [ "$MPLAY_WITH_SNDCTL" -a "$MPLAY_SVDR" ]; then
        if [ "$SOUNDSET_BACKUP" ]; then
            log "sndctl: Restoring sound set to '$SOUNDSET_BACKUP'."
            "$MPLAY_SVDR" plug sndctl sset "$SOUNDSET_BACKUP"
        fi
    fi

    # restore VDR channel
    # FIXME: doesn't work up to now
    #[ "$CHANNEL_BACKUP" ] && "$MPLAY_SVDR" CHAN "$CHANNEL_BACKUP"

    log "Clean up complete"

    if [ "$1" ]; then
        log "$1"
        log ">>> mplay.sh version $MPLAY_VERSION ends with error(s)."
	#log ">>> MPlayer identification output was:"
	#log "$FILE_ID"
        exit 1
    else
        log ">>> mplay.sh version $MPLAY_VERSION normally ends"
        exit 0
    fi
}


#########################################################################
# processing starts here
########################################

#########################################################################
# read config file
########################################
if [ ! -r "$CONFDIR/mplayrc" ]; then
    echo "Error: Can't read $CONFDIR/mplayrc!"
    exit 1
fi

. "$CONFDIR/mplayrc"

log ">>> mplay.sh version $MPLAY_VERSION started (as user '$USER')"
log "Config file '$CONFDIR/mplayrc' read."
log "Started with: $0 $1 $2 $3 $4 $5 $6 $7 $8 $9"


#########################################################################
# checking MPlayer
########################################
FILE_ID=`$MPLAY_MPLAYER | grep ^MPlayer`
[ "$FILE_ID" ] || quit "Error: Can't find MPlayer as '$MPLAY_MPLAYER'!"

log "Using MPlayer: '$FILE_ID'"


##########################################################################
# initializing
########################################
# save current soundset
if [ "$MPLAY_WITH_SNDCTL" -a "$MPLAY_SVDR" ]; then
    SOUNDSET_BACKUP=`"$MPLAY_SVDR" plug sndctl sset | grep ^950 | cut -d " " -f 2-`
    log "sndctl: Active sound set saved ($SOUNDSET_BACKUP)."
fi


##########################################################################
# identifying file to play
########################################
# for file type detection we need only the file itself
argFile=${1##*/}

# files with an extension have a "." in its name
if [ "${argFile/*.*}" ]; then
    # no dot found, assuming, this is a set file
    # but let's be a little bit more sure
    if [ -r "$1" ]; then
        log "Thinking, I've found a set file: '$1'"

        argType=`file -b "$1"`
        [ "$argType" != "empty" -a "`echo $argType | grep -v ASCII`" ] && quit "Error: '$1' seems to be no SET file!"

        # trying to read this file
        . "$1"
        log "Set file '$1' read."
    fi

    # checking $MPLAY_FILE for a value
    if [ -z "$MPLAY_FILE" ]; then
       # no value was set in MPLAY_FILE, trying to find a file to play
       # within media directory
       for type in $MPLAY_TYPES; do
           log "Searching for '$argFile.$type' in '$MPLAY_MEDIA_DIR'..."
           MPLAY_FILE=`find "$MPLAY_MEDIA_DIR" -depth -iname "$argFile.$type"`
           [ "$MPLAY_FILE" ] && break
       done
    fi
        
else
    # no set file, nothing to do but set FILE to the given value
    MPLAY_FILE="$1"
fi

log "File to play seems: '$MPLAY_FILE'"

# MPLAY_FILE must now be set to continue
[ "$MPLAY_FILE" ] || quit "Error: No file to play was given!"

# MPLAY_FILE can contain only a file name or a complete path
# if MPLAY_FILE is not readable without a change, we try again to find
# this file in our media directory
if [ ! -r "$MPLAY_FILE" ]; then
    log "Searching for '$MPLAY_FILE' in '$MPLAY_MEDIA_DIR'"
    MPLAY_FILE=`find "$MPLAY_MEDIA_DIR" -depth -iname "$MPLAY_FILE"`
fi
log "File to play found: '$MPLAY_FILE'"

# checking content of MPLAY_FILE again
[ "$MPLAY_FILE" ] || quit "Error: No file to play was found!"

log "File to play is in the end: '$MPLAY_FILE'"


##########################################################################
# direct play?
########################################
if [ "$MPLAY_DIRECT" ]; then
    log "Direct playing requested: '$MPLAY_MPLAYER $MPLAY_DIRECT \"$MPLAY_FILE\"'"
    $MPLAY_MPLAYER $MPLAY_DIRECT "$FILE"
    quit
fi


##########################################################################
# checking and process file type
########################################
FILE_EXT=${MPLAY_FILE##*.}
log "Searching for a config file for type '$FILE_EXT'."

# if a configuration file for this file type exist: find and read it
# (it's done with 'find' because of the possibility of upper case letters
# within the file name)
extensionConf=`find $CONFDIR -iname $FILE_EXT.conf`
if [ -r "$extensionConf" ]; then
    . "$extensionConf"
    log "Type config file '$extensionConf' read."
else
    log "Type config file '$FILE_EXT.conf' not found."
fi


##########################################################################
# switch VDR channel?
########################################
# FIXME: this doesn't work
#if [ "$MPLAY_VDRCHANNEL" -a "$MPLAY_SVDR" ]; then
#    log "Switching VDR channel to $MPLAY_VDRCHANNEL"

    # save old channel
#    CHANNEL_BACKUP=`"$MPLAY_SVDR" CHAN | grep 250 | cut -d " " -f 2`

    # switch channel
#    ("$MPLAY_SVDR" CHAN "$MPLAY_VDRCHANNEL")
#fi

##########################################################################
# slave mode requested?
########################################
log "Check for slave mode..."
[ "$2" = "SLAVE" ] && MPLAY_GLOBAL="$MPLAY_GLOBAL -slave"


##########################################################################
# care for AID
########################################
# audio stream to play
# where is 'AID' in options (can be in $2 or $3)
log "Check for AID..."
[ "$2" = "AID" ] && MPLAY_GLOBAL="$MPLAY_GLOBAL -aid $3"
[ "$3" = "AID" ] && MPLAY_GLOBAL="$MPLAY_GLOBAL -aid $4"


##########################################################################
# start building play file
########################################
if [ "$MPLAY_MULTI" ]; then
    echo -n "\"$MPLAY_MULTI\" \"$MPLAY_MPLAYER\" $MPLAY_GLOBAL" > "$MPLAY_PLAY"
    log "Creation of player script '$MPLAY_PLAY' started with: '\"$MPLAY_MULTI\" \"$MPLAY_MPLAYER\" $MPLAY_GLOBAL'"
else
    echo -n "\"$MPLAY_MPLAYER\" $MPLAY_GLOBAL" > "$MPLAY_PLAY"
    log "Creation of player script '$MPLAY_PLAY' started with: '\"$MPLAY_MPLAYER\" $MPLAY_GLOBAL'"
fi


##########################################################################
# build (internal) playlist
########################################
# we have to process playlists by our own, because a playlist can include
# different file types, which may need different options for MPlayer

# up to now, 0 files are added
ADDED=0

if [ "$MPLAY_PLAYLIST" ]; then
    log "Start processing a playlist."

    # get path from $MPLAY_FILE
    path=${MPLAY_FILE%/*}

    # read this file step by step
    declare -a files
    i=0
    saveIFS="$IFS"
    IFS=$'\x0A'
    for playfile in `cat "$MPLAY_FILE"`; do
        files[$i]="$playfile"
        i=$(($i+1))
    done
    IFS="$saveIFS"

    j=0
    while [ $j -lt $i ]; do
        playfile="${files[$j]}"
        j=$(($j+1))

        # check for an absolute path
        if [ "${playfile%%/*}" ]; then
            add "$path/$playfile"
        else
            add "$playfile"
        fi
    done
else
    add "$MPLAY_FILE"
fi

# was at least one file added?
[ $ADDED -eq 0 ] && quit "Error: No file was added to internal playlist (nothing to play)."

log "Added $ADDED file(s) to internal playlist."


##########################################################################
# set soundset (if requested)
########################################
if [ "$MPLAY_WITH_SNDCTL" -a "$MPLAY_SVDR" ]; then
    if [ "$MPLAY_SOUNDSET" ]; then
        log "sndctl: Switching sound set to '$MPLAY_SOUNDSET'."
        "$MPLAY_SVDR" plug sndctl sset "$MPLAY_SOUNDSET"
    fi
fi


##########################################################################
# play
########################################
log "Starting '$MPLAY_PLAY' for playing."
chmod 744 "$MPLAY_PLAY"
# FIXME: This is a workaround for the problem VDR sound don't stop playing
[ "$MPLAY_FIXVDRSOUND" ] && "$MPLAY_MPLAYER" -vo null -ao mpegpes "$CONFDIR/silence.wav"
"$MPLAY_PLAY"


##########################################################################
# clean up and quit
########################################
quit
