# $Id: 73_km200.pm 26850 2022-12-13 09:15:29Z Sailor $
########################################################################################################################
#
#     73_km200.pm
#     Creates the possibility to access the Buderus central heating system via
#     Buderus KM200, KM100 or KM50 communication module. It uses HttpUtils_NonblockingGet
#     from Rudolf Koenig to avoid a full blockage of the fhem main system during the
#     polling procedure.
#
#     Author                     : Matthias Deeke 
#     Contributions              : Olaf Droegehorn, Andreas Hahn, Rudolf Koenig, Markus Bloch,
#     Contributions (continued)  : Stefan M., Furban, KaiKr, grossi33, Morkin, DLindner
#     e-mail                     : matthias.deeke(AT)deeke(PUNKT)eu
#     Fhem Forum                 : http://forum.fhem.de/index.php/topic,25540.0.html
#     Fhem Wiki                  : http://www.fhemwiki.de/wiki/Buderus_Web_Gateway
#
#     This file is part of fhem.
#
#     Fhem is free software: you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation, either version 2 of the License, or
#     (at your option) any later version.
#
#     Fhem is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with fhem.  If not, see <http://www.gnu.org/licenses/>.
#
#     fhem.cfg: define <devicename> km200 <IPv4-address> <GatewayPassword> <PrivatePassword>
#
#     Example 1 - Bare Passwords:
#     define myKm200 km200 192.168.178.200 GatewayGeheim        PrivateGeheim
#
#     Example 2 - base64 encoded passwords: Both passwords may be pre-encode with base64
#     define myKm200 km200 192.168.178.200 R2F0ZXdheUdlaGVpbQ== UHJpdmF0ZUdlaGVpbQ==
#
########################################################################################################################

########################################################################################################################
# List of open Problems:
#
#
########################################################################################################################

package main;

use strict;
use warnings;
use constant false => 0;
use constant true  => 1;

###START###### Initialize module ##############################################################################START####
sub km200_Initialize($) {
    my ($hash)  = @_;

	### Try to load perl libraries if installed or write log in case of unavailability. ###
	my @UseLibraries = split(/[\n,\t]/,"
		Blocking
		FHEM::Meta
		Time::HiRes qw(gettimeofday sleep usleep)
		Digest::MD5 qw(md5 md5_hex md5_base64)
		base qw( Exporter )
		List::MoreUtils qw(first_index)
		MIME::Base64
		LWP::UserAgent
		JSON
		Crypt::Rijndael
		HttpUtils
		Encode
		Text::Wrap
		Data::Dumper
	");

	foreach my $Library (grep(/\S/, @UseLibraries)) {	
		eval "use " . $Library;
		if (length($@) == 0) {
			Log3 undef, 5, "km200 - Successfully Installed Perl Module               : " . $Library
		}
		else {
			Log3 undef, 2, "km200 - Cannot find " . $Library . " in \@INC. Please install the Perl library first. Initialization of 73_km200.pm aborted!";
			return
		}
	}

    $hash->{STATE}           = "Init";
    $hash->{DefFn}           = "km200_Define";
    $hash->{UndefFn}         = "km200_Undefine";
    $hash->{SetFn}           = "km200_Set";
    $hash->{GetFn}           = "km200_Get";
    $hash->{AttrFn}          = "km200_Attr";
	$hash->{DbLog_splitFn}   = "km200_DbLog_splitFn";

    $hash->{AttrList}        = "do_not_notify:1,0 " .
							   "header " .
							   "disable:1,0 " .
						       "loglevel:0,1,2,3,4,5,6 " .
						       "IntervalDynVal " .
						       "PollingTimeout " .
							   "DoNotPoll " .
							   "ReadBackDelay " .
							   "ReadingTextLen " .
						       $readingFnAttributes;
	return FHEM::Meta::InitMod( __FILE__, $hash );
}
####END####### Initialize module ###############################################################################END#####


###START######  Activate module after module has been used via fhem command "define" ##########################START####
sub km200_Define($$) {
	my ($hash, $def)            = @_;
	my @a						= split("[ \t][ \t]*", $def);
	my $name					= $a[0];
								 #$a[1] just contains the "km200" module name and we already know that! :-)
	my $url						= $a[2];
	my $km200_gateway_password	= $a[3];
	my $km200_private_password	= $a[4];

	$hash->{NAME}				= $name;
	$hash->{STATE}              = "define";

	Log3 $name, 4, $name. " : km200 - Starting to define module";

	### Stop the current timer if one exists errornous 
	RemoveInternalTimer($hash);
	Log3 $name, 4, $name. " : km200 - InternalTimer has been removed.";

	
	###START###### Define known services of gateway ###########################################################START####
	my @KM200_AllServices = (
	"/",
	"/application",
	"/gservice-tariff",
	"/dhwCircuits",
	"/gateway",
	"/heatingCircuits",
	"/heatSources",
	"/notifications",
	"/recordings",
	"/solarCircuits",
	"/system",
	"/dhwCircuits",
	);
	####END####### Define known services of gateway ############################################################END#####
 
	
    ###START### Check whether all variables are available #####################################################START####
	if (int(@a) == 5) {
		###START### Check whether IPv4 address is valid
		if ($url =~ m/^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)(:[0-9]{1,5})?$/) {
			Log3 $name, 4, $name. " : km200 - IPv4-address is valid                  : " . $url;
		}
		else {
			return $name .": Error - IPv4 address is not valid \n Please use \"define <devicename> km200 <IPv4-address> <interval/[s]> <GatewayPassword> <PrivatePassword>\" instead";
		}
		####END#### Check whether IPv4 address is valid	

		
		###START### Check whether gateway password is base64 encoded or bare, has the right length and delete "-" if required
		my $PasswordEncrypted = false;
		my $EvalPassWord = $km200_gateway_password;
		$EvalPassWord =~ tr/-//d;

		if ( length($EvalPassWord) == 16) {
			$km200_gateway_password = $EvalPassWord;
			Log3 $name,4, $name. " : km200 - Provided GatewayPassword provided as bareword has the correct length at least.";		
		}
		# Check whether the password is eventually base64 encoded
		else {
		    # Remove additional encoding with base64
			my $decryptData = decode_base64($km200_gateway_password);
			$decryptData =~ tr/-//d;
			$decryptData =~ s/\r|\n//g;
			if ( length($decryptData) == 16) {
				$km200_gateway_password = $decryptData;
				$PasswordEncrypted = true;
				Log3 $name, 4, $name. " : km200 - Provided GatewayPassword encoded with base64 has the correct length at least.";
			}
			else {
				return $name .": Error - GatewayPassword does not have the correct length.\n". 
							  "          Please enter gateway password in the format of \"aaaabbbbccccdddd\" or \"aaaa-bbbb-cccc-dddd\"\n". 
							  "          You may encode your password with base64 first, in order to prevent bare passwords in fhem.cfg.\n".
							  "          If you choose to encrypt your gateway password with base64, you also must encrypt your private password the same way\n"; 

				Log3 $name, 3, $name. " : km200 - Provided Gateway Password does not follow the specifications";
			}
		}
		####END#### Check whether gateway password has the right length and delete "-" if required

		###START### Check whether private password is available and decode it with base64 if encoding is used
		if ($PasswordEncrypted == true) {
			my $decryptData = decode_base64($km200_private_password);
			$decryptData =~ s/\r|\n//g;
			if (length($decryptData) > 0) {
				$km200_private_password = $decryptData;
				Log3 $name, 4, $name. " : km200 - Provided PrivatePassword exists at least";
			}
			else {
				return $name .": Error - PrivatePassword does not have the minimum length.\n".
							  "          You may encode your password with base64 first, in order to prevent bare passwords in fhem.cfg.\n".
							  "          If you choose to encrypt your private password with base64, you also must encrypt your gateway password the same way\n"; 
			}
		}
		# If private password is provided as bare word 
		else {
			if (length($km200_private_password) > 0) {
					Log3 $name, 4, $name. " : km200 - Provided PrivatePassword exists at least";		
			}
			else {
				return $name .": Error - PrivatePassword has not been provided.\n".
							  "          You may encode your password with base64 first, in order to prevent bare passwords in fhem.cfg.\n".
							  "          If you choose to encrypt your private password with base64, you also must encrypt your gateway password the same way\n"; 
				Log3 $name, 3, $name. " : km200 - Provided Private Password does not follow the specifications";
		  }
		}
		####END#### Check whether private password is available and decode it with base64 if encoding is used
	}
	else {
	    return $name .": km200 - Error - Not enough parameter provided." . "\n" . "Gateway IPv4 address, Gateway and Private Passwords must be provided" ."\n". "Please use \"define <devicename> km200 <IPv4-address> <GatewayPassword> <PrivatePassword>\" instead";
	}
	####END#### Check whether all variables are available ######################################################END#####

	
	###START###### Create the secret SALT of the MD5-Hash for AES-encoding ####################################START####
	my $Buderus_MD5Salt = pack(
	'C*',
	0x86, 0x78, 0x45, 0xe9, 0x7c, 0x4e, 0x29, 0xdc,
	0xe5, 0x22, 0xb9, 0xa7, 0xd3, 0xa3, 0xe0, 0x7b,
	0x15, 0x2b, 0xff, 0xad, 0xdd, 0xbe, 0xd7, 0xf5,
	0xff, 0xd8, 0x42, 0xe9, 0x89, 0x5a, 0xd1, 0xe4
	);
	####END####### Create the secret SALT of the MD5-Hash for AES-encoding #####################################END#####
	

	###START###### Create keys with MD5 #######################################################################START####
	# Copy Salt
	my $km200_crypt_md5_salt = $Buderus_MD5Salt;

	# First half of the key: MD5 of (km200GatePassword . Salt)
	my $key_1 = md5($km200_gateway_password . $km200_crypt_md5_salt);  
	  
	# Second half of the key: - Initial: MD5 of ( Salt)
	my $key_2_initial = md5($km200_crypt_md5_salt);

	# Second half of the key: -  private: MD5 of ( Salt . km200PrivatePassword)
	my $key_2_private = md5($km200_crypt_md5_salt . $km200_private_password);

	# Create keys
	my $km200_crypt_key_initial = ($key_1 . $key_2_initial);
	my $km200_crypt_key_private = ($key_1 . $key_2_private);
	####END####### Create keys with MD5 #########################################################################END#####
  
  
	###START###### Writing values to global hash ###############################################################START####
	  $hash->{NAME}                             = $name;
	  $hash->{URL}                              = $url;
	  $hash->{INTERVALDYNVAL}                   = 300;
	  $hash->{DELAYDYNVAL}                      = 60;
	  $hash->{POLLINGTIMEOUT}                   = 5;
	  $hash->{READBACKDELAY}					= 100;
	  $hash->{READTEXTLEN}                      = 0;
	  $hash->{temp}{ServiceCounterInit}         = 0;
	  $hash->{temp}{ServiceCounterDyn}          = 0;
	  $hash->{temp}{ServiceDbLogSplitHash}      = ();
	  $hash->{status}{FlagInitRequest}          = false;
	  $hash->{status}{FlagGetRequest}           = false;
	  $hash->{status}{FlagSetRequest}           = false;
	  $hash->{status}{FlagDynRequest}           = false;
	  $hash->{status}{FlagStatRequest}          = false;
	  $hash->{Secret}{CRYPTKEYPRIVATE}          = $km200_crypt_key_private;
	  $hash->{Secret}{CRYPTKEYINITIAL}          = $km200_crypt_key_initial;
	@{$hash->{Secret}{KM200ALLSERVICES}}        = sort @KM200_AllServices;
	@{$hash->{Secret}{KM200ALLSERVICESBACKUP}}  = sort @KM200_AllServices;
	@{$hash->{Secret}{KM200RESPONDINGSERVICES}} = ();
	@{$hash->{Secret}{KM200WRITEABLESERVICES}}  = ();
	@{$hash->{Secret}{KM200DONOTPOLL}}  		= ();
	####END####### Writing values to global hash ################################################################END#####

	###START###### For Debugging purpose only ##################################################################START####  
	Log3 $name, 4, $name. " : km200 - Define D                                   : " . $def;
	Log3 $name, 4, $name. " : km200 - Define A                                   : " . @a;
	Log3 $name, 4, $name. " : km200 - Define Name                                : " . $name;
	Log3 $name, 4, $name. " : km200 - Define Adr                                 : " . $url;
	Log3 $name, 4, $name. " : km200 - Define H                                   : " . Dumper($hash);
	####END####### For Debugging purpose only ###################################################################END#####

	### Proceed with km200 Initialization as soon fhem is initialized
	# https://forum.fhem.de/index.php/topic,130351.msg1246281.html#msg1246281
	# Die InternalTimer Eintraege werden erst abgearbeitet, wenn $init_done = 1 ist.
	InternalTimer(15, \&km200_FirstInit, $hash );

	return 
}
####END####### Activate module after module has been used via fhem command "define" ############################END#####

###START###### Initialize km200 after fhem Initialization is done #############################################START####
sub km200_FirstInit($) {
	my ($hash) = @_;
	my $name = "73_km200";
	Log3 $name, 4, $name. " : km200_FirstInit -  hash                                  : " . Dumper($hash);

	### Stop the current timer if one exists errornous 
	RemoveInternalTimer($hash);
	Log3 $name, 4, $name. " : km200 - InternalTimer has been removed.";

	###START###### Reset fullResponse error message ############################################################START####
	readingsSingleUpdate( $hash, "fullResponse", "OK", 1);
	####END####### Reset fullResponse error message #############################################################END#####	
	
	###START###### Check whether communication to the physical unit is possible ################################START####
	my $Km200Info ="";
	$hash->{temp}{service} = "/gateway/DateTime";
	$Km200Info = km200_GetSingleService($hash);
	
	if ($Km200Info eq "ERROR") {
		$Km200Info = $hash->{temp}{TransferValue};
		$hash->{temp}{TransferValue} = "";

		## Communication with Gateway WRONG !! ##
		$hash->{STATE}="Error - No Communication";
		return ($name .": km200_FirstInit - ERROR - The communication between fhem and the Buderus KM200 failed! \n". 
		               "                  Please check physical connection, IP-address and passwords! \n");
	} 
	## Communication OK but service not available ##
	elsif ($Km200Info eq "SERVICE NOT AVAILABLE") {
		Log3 $name, 4, $name. " : km200_FirstInit -  /gateway/DateTime                     : NOT AVAILABLE";
	} 
	## Communication OK and service is available ##
	else {
		Log3 $name, 4, $name. " : km200_FirstInit - /gateway/DateTime                      : AVAILABLE";				
	}
	####END####### Check whether communication to the physical unit is possible ################################END#####

	###START###### Initiate the timer for first time polling of  values from KM200 but wait 10s ###############START####
	InternalTimer(gettimeofday()+10, "km200_GetInitService", $hash, 1);
	Log3 $name, 4, $name. " : km200_FirstInit - Internal timer for Initialisation of services started for the first time.";
	####END####### Initiate the timer for first time polling of  values from KM200 but wait 60s ################END#####
	
	return undef;
}
####END####### Initialize km200 after fhem Initialization is done ##############################################END#####


###START###### To bind unit of value to DbLog entries #########################################################START####
sub km200_DbLog_splitFn($$) {
	my ($event, $name)	= @_;
	my ($reading, $value, $unit);
    my $hash 			= $defs{$name};
	my @argument		= split("[ \t][ \t]*", $event);
	
	### Delete ":" and everything behind in readings name
	$argument[0] =~ s/:.*//;
 
	### Log entries for debugging
	Log3 $name, 5, $name. " : km200_DbLog_splitFn - Content of event             : " . $event;
	Log3 $name, 5, $name. " : km200_DbLog_splitFn - Content of argument[0]       : " . $argument[0];
#	Log3 $name, 5, $name. " : km200_DbLog_splitFn - Content of argument[1]       : " . $argument[1];

	### If the service to be changed is identical to the one where the unit received from
	if ($argument[0] = $hash->{temp}{ServiceDbLogSplitHash}{id}) {
		### Get values being changed from hash
		$reading = $argument[0];
		$value   = $argument[1];
		$unit    = $hash->{temp}{ServiceDbLogSplitHash}{unitOfMeasure};
	}

	### Delete temporary json-hash for DbLog-Split
	$hash->{temp}{ServiceDbLogSplitHash} = ();
	### Delete temporary json-hash for DbLog-Split

   return ($reading, $value, $unit);
}
####END####### To bind unit of value to DbLog entries ##########################################################END#####


###START###### Deactivate module module after "undefine" command by fhem ######################################START####
sub km200_Undefine($$) {
	my ($hash, $def)  = @_;
	my $name = $hash->{NAME};	
	my $url  = $hash->{URL};

  	### Stop the internal timer for this module
	RemoveInternalTimer($hash);

	Log3 $name, 3, $name. " - km200 has been undefined. The KM unit at $url will no longer polled.";

	return undef;
}
####END####### Deactivate module module after "undefine" command by fhem #######################################END#####


###START###### Handle attributes after changes via fhem GUI ###################################################START####
sub km200_Attr(@) {
	my @a               = @_;
	my $name            = $a[1];
	my $hash            = $defs{$name};
	my $IntervalDynVal  = $hash->{INTERVALDYNVAL};
	my $DelayDynVal     = $hash->{DELAYDYNVAL};
	my $ReadBackDelay   = $hash->{READBACKDELAY};
	my $ReadingTextLen  = $hash->{READTEXTLEN};
	
	
	### Check whether disable attribute has been provided
	if ($a[2] eq "disable") {
		###START### Check whether device shall be disabled
		if ($a[3] == 1) {
			### Set new status
			$hash->{STATE} = "Disabled";
			
			### Stop the current timer
			RemoveInternalTimer($hash);
			Log3 $name, 4, $name. " : km200 - InternalTimer has been removed.";

			### Delete all Readings
			fhem( "deletereading $name .*" );
			
			### Recover list of root services
			@{$hash->{Secret}{KM200ALLSERVICES}}= @{$hash->{Secret}{KM200ALLSERVICESBACKUP}};
			
			Log3 $name, 3, $name. " : km200 - Device disabled as per attribute.";
		}
		else {
			### Initiate the timer for first time polling of  values from KM200 but wait 10s
			InternalTimer(gettimeofday()+10, "km200_GetInitService", $hash, 1);
			Log3 $name, 4, $name. " : km200 - Internal timer for Initialisation of services re-started.";

			### Set new status
			$hash->{STATE} = "Initiating Sounding...";
			
			Log3 $name, 4, $name. " : km200 - Device enabled as per attribute.";
		}
		####END#### Check whether device shall be disabled
	}
	### Check whether dynamic interval attribute has been provided
	elsif ($a[2] eq "IntervalDynVal") {

		$IntervalDynVal = $a[3];
		###START### Check whether polling interval is not too short
		if ($IntervalDynVal > 19) {
			$hash->{INTERVALDYNVAL} = $IntervalDynVal;
			Log3 $name, 4, $name. " : km200 - IntervalDynVal set to attribute value:" . $IntervalDynVal ." s";
		}
		else {
			return $name .": Error - Gateway interval for IntervalDynVal too small - server response time longer than defined interval, please use something >=20, default is 90";
		}
		####END#### Check whether polling interval is not too short
	}
	### Check whether polling timeout attribute has been provided
	elsif($a[2] eq "PollingTimeout") {
		###START### Check whether timeout is not too short
		if ($a[3] >= 5) {
			$hash->{POLLINGTIMEOUT} = $a[3];
			Log3 $name, 4, $name. " : km200 - Polling timeout set to attribute value:" . $a[3] ." s";
		}
		else {
			Log3 $name, 4, $name. " : km200 - Error - Gateway polling timeout attribute too small: " . $a[3] ." s";
			return $name .": Error - Gateway polling timeout attribute is too small - server response time is 5s minimum, default is 5";
		}
		####END#### Check whether timeout is not too short
	}
    elsif($a[2] eq 'DoNotPoll') {
        #my @KM200_DONOTPOLL   = ();
        my @temp              = @a;

        ### Stop the current timer
        RemoveInternalTimer($hash);
        Log3 $name, 4, $name. " : km200 - InternalTimer has been removed.";
       
        ### Delete the first 3 items of the array
        splice @temp, 0, 3;

        ### Insert empty field as minimum entry
        push @temp, "";
       
        ### Transform string entries seperated by blank into array
        my @KM200_DONOTPOLL = split m/\s+/, $temp[0] // ();

        ### Remove trailing slash of each item if available
       
        ### For each item found in this empty parent directory
        for my $item (@KM200_DONOTPOLL) {
            ### Delete trailing slash
            $item =~ s/\/$//;
        }
       
        ### Save list of services not to be polled into hash
        @{$hash->{Secret}{KM200DONOTPOLL}} = @KM200_DONOTPOLL;

        ### Get original list of root services back
        @{$hash->{Secret}{KM200ALLSERVICES}} = @{$hash->{Secret}{KM200ALLSERVICESBACKUP}};
       
        ### For every blacklisted service
        for my $SearchWord(@KM200_DONOTPOLL) {
            last if @{$hash->{Secret}{KM200ALLSERVICES}}|| ref @{$hash->{Secret}{KM200ALLSERVICES}} ne 'ARRAY';
            ### Filter all blocked root services out of services to be polled
            my $FoundPosition = first_index{ $_ eq $SearchWord }@{$hash->{Secret}{KM200ALLSERVICES}};
            if ($FoundPosition >= 0) {
                splice(@{$hash->{Secret}{KM200ALLSERVICES}}, $FoundPosition, 1);
            }
        }
        ### Message for debugging purposes
        Log3 $name, 5, $name. "km200 module is only polling the following services! \n @{$hash->{Secret}{KM200ALLSERVICES}}";
        Log3 $name, 5, $name. "km200 module is NOT  polling the following services! \n @{$hash->{Secret}{KM200DONOTPOLL}}";
        Log3 $name, 4, $name. " : km200 - The following services will not be polled: ". $a[3];
       
        ### Interrupting all currently running Polling
        @{$hash->{Secret}{KM200DYNSERVICES}} = "";
        $hash->{temp}{ServiceCounterDyn} = 0;

        ### Delete all Readings
        fhem( "deletereading $name .*" );

        ### Re-start the sounding of  values from KM200 but wait the period of $hash->{POLLINGTIMEOUT} + 10s
        InternalTimer(gettimeofday()+$hash->{POLLINGTIMEOUT}+10, "km200_GetInitService", $hash, 1);
        Log3 $name, 4, $name. " : km200 - Sounding of services re-started after change of DoNotPoll attribute";
    }
	### Check whether time-out for Read-Back has been provided
	if($a[2] eq "ReadBackDelay") {
		$ReadBackDelay = $a[3];
		###START### Check whether ReadBackDelay is valid
		if ($ReadBackDelay >= 0) {
			$hash->{READBACKDELAY} = $ReadBackDelay;
			Log3 $name, 4, $name. " : km200 - ReadBackDelay set to attribute value:" . $ReadBackDelay ." s";
		}
		else {
			return $name .": Error - Read-Back delay time must be positive. Default is 0us";
		}
		####END#### Check whether ReadBackDelay is valid
	}
	### Check whether length for Readins has been provided
	if($a[2] eq "ReadingTextLen") {
		$ReadingTextLen = $a[3];
		###START### Check whether ReadBackDelay is valid
		if ($ReadingTextLen >= 0) {
			$hash->{READTEXTLEN} = $ReadingTextLen;
			Log3 $name, 4, $name. " : km200 - ReadingTextLen set to attribute value:" . $ReadingTextLen ." s";
		}
		else {
			return $name .": Error - Text Length must be positive. Default is 0";
		}
		####END#### Check whether ReadBackDelay is valid
	}
	### If no attributes of the above known ones have been selected
	else {
		# Do nothing
	}
	return undef;
}
####END####### Handle attributes after changes via fhem GUI ####################################################END#####


###START###### Obtain value after "get" command by fhem #######################################################START####
sub km200_Get($@) {
	my ( $hash, @a ) = @_;
	
	### If not enough arguments have been provided
	if ( @a < 2 ) {
		return "\"get km200\" needs at least one argument";
	}
		
	my $name     = shift @a;
	my $service  = shift @a;
	my $option   = shift @a;
	my %km200_gets;
	my $ReturnValue;
	my $ReturnMessage;
	
	### Get the list of possible services and create a hash out of it
	my @GetServices = @{$hash->{Secret}{KM200ALLSERVICES}};

	foreach my $item(@GetServices) {
		$km200_gets{$item} = ("1");
	}
	
	### Remove trailing slash if available
	$service = $1 if($service=~/(.*)\/$/);

	### For programming purpose only
	if ($service eq "TestErrorMsg") {
		my $classdescription;
		my $classworkaround;
		my $reason;
		my $checklist;
		my $correction;
		
		($classdescription, $classworkaround, $reason, $checklist, $correction) = km200_GetErrorMessage("A11", "3084", "4", $hash);
		$ReturnMessage = 
		"<style>table, th, td {border: 1px solid black; border-collapse: collapse;}</style>" .
		"<table n>" .
		"<tr><td>The error-class description is : </td><td>" . $classdescription . "</td></tr>" .
		"<tr><td>The error-class workaround is :  </td><td>" . $classworkaround .  "</td></tr>" .
		"<tr><td>The error-reason is :            </td><td>" . $reason .           "</td></tr>" .
		"<tr><td>The checklist is :               </td><td>" . $checklist .        "</td></tr>" .
		"<tr><td>The correction is :              </td><td>" . $correction .       "</td></tr>" .
		"</table>";
		return($ReturnMessage);
	}

	### If service chosen in GUI does not exist
	if(!$km200_gets{$service}) {
		my @cList = keys %km200_gets;
		push(@cList, "TestErrorMsg");
		return "Unknown argument $service, choose one of " . join(" ", @cList);
	}

	### Check whether the initialisation process has been finished
	if ($hash->{temp}{ServiceCounterInit} == false) {
		### Save chosen service into hash
		$hash->{temp}{service} = $service;

		### Read service-hash
		$ReturnValue = km200_GetSingleService($hash);

		### If the "get" - option has been set to "Json" for the return of the raw Json-string
		if ($option =~ m/json/i) {
			$ReturnMessage = $hash->{temp}{JsonRaw};
		}
		### If no option has been chosen, just return the result of the value.
		else {
			### If type is a floatvalue then format decimals
			if ($ReturnValue->{type} eq "floatValue") {
				$ReturnMessage = sprintf("%.1f", $ReturnValue->{value});
			}
			### If type is something else just pass throught
			else {
				$ReturnMessage = $ReturnValue->{value};
			}
		}
	}
	### If the initialisation process has NOT been finished
	else {
		$ReturnMessage = "The initialisation process is still ongoing. Please wait for the STATE changing to \"Standby\"";
	}
		
	
	### Delete temporary values
	$hash->{temp}{service} = "";
	$hash->{temp}{JsonRaw} = "";
	

	### Return value
	return($ReturnMessage);
}
####END####### Obtain value after "get" command by fhem ########################################################END#####


###START###### Manipulate service after "set" command by fhem #################################################START####
sub km200_Set($@) {
	my ( $hash, @a ) = @_;
	
	### If not enough arguments have been provided
	if ( @a < 2 ) {
		return "\"set km200\" needs at least one argument";
	}
		
	my $name = shift @a;
	my $service  = shift @a;
	my $value = join(" ", @a);
	my %km200_sets;
	my $ReturnMessage;

	### Get the list of possible services and create a hash out of it
	my @WriteableServices = @{$hash->{Secret}{KM200WRITEABLESERVICES}};

	

	foreach my $item(@WriteableServices) 	 {
		$km200_sets{$item} = ("1");
	}
	
	### If service chosen in GUI does not exist
	if(!$km200_sets{$service}) {
		my @cList = keys %km200_sets;
		return "Unknown argument $service, choose one of " . join(" ", @cList);
	}

	### Check whether the initialisation process has been finished
	if ($hash->{temp}{ServiceCounterInit} == false) {
		### Save chosen service into hash
		$hash->{temp}{service}  = $service;
		$hash->{temp}{postdata} = $value;	
		
		### Call set sub
		$ReturnMessage = km200_PostSingleService($hash);
	}
	### If the initialisation process has NOT been finished
	else {
		$ReturnMessage = "The initialisation process is still ongoing. Please wait for the STATE changing to \"Standby\"";
	}
	
	### Delete temporary hash values
	$hash->{temp}{postdata} = "";
	$hash->{temp}{service}  = "";
	
	return($ReturnMessage);
}
####END####### Manipulate service after "Set" command by fhem ##################################################END#####


###START####### Repeats "string" for "count" times ############################################################START####
sub str_repeat($$) {
    my $string = $_[0];
    my $count  = $_[1];
    return(${string}x${count});
}
####END######## Repeats "string" for "count" times #############################################################END#####


###START###### Subroutine Encrypt Data ########################################################################START####
sub km200_Encrypt($) {
	my ($hash, $def)            = @_;

	my $km200_crypt_key_private = $hash->{Secret}{CRYPTKEYPRIVATE};
	my $name                    = $hash->{NAME};
	my $encryptData             = $hash->{temp}{jsoncontent};

    # Create Rijndael encryption object
    my $cipher = Crypt::Rijndael->new($km200_crypt_key_private, Crypt::Rijndael::MODE_ECB() );

    # Get blocksize and add PKCS #7 padding
    my $blocksize =  $cipher->blocksize();
    my $encrypt_padchar = $blocksize - ( length( $encryptData ) % $blocksize );
    $encryptData .= str_repeat( chr( $encrypt_padchar ), $encrypt_padchar );

	# Do the encryption
    my $ciphertext = $cipher->encrypt( $encryptData );

	# Do additional encoding with base64
    $ciphertext = encode_base64($ciphertext);
   
    # Return the encoded text
    return($ciphertext);
}
####END####### Subroutine Encrypt Data #########################################################################END#####


###START###### Subroutine Decrypt Data ########################################################################START####
sub km200_Decrypt($) {
	my ($hash, $def)            = @_;

	my $km200_crypt_key_private = $hash->{Secret}{CRYPTKEYPRIVATE};
	my $name                    = $hash->{NAME};
	my $decryptData             = $hash->{temp}{decodedcontent};

	### Log entries for debugging purposes
    #Log3 $name, 5, $name. " : km200 - decryptData2 - decryptData                : " .$decryptData;
    
	# Remove additional encoding with base64
	$decryptData = decode_base64($decryptData);
	
	### Log entries for debugging purposes
	#Log3 $name, 5, $name. " : km200 - decryptData2 - base64decode               : " .$decryptData;

	# Check whether the length of the decryptData is NOT multiplies of 16
	if ((length($decryptData)&0xF) != 0) {
		# Return nothing which will end this subroutine
		return "";
	}
	
    # Create Rijndael decryption object and do the decryption
    my $cipher = Crypt::Rijndael->new($km200_crypt_key_private, Crypt::Rijndael::MODE_ECB() );
    my $deciphertext = $cipher->decrypt( $decryptData );

    # Remove zero padding
    $deciphertext =~ s/\x00+$//;

    # Remove PKCS #7 padding   
    my $decipher_len = length($deciphertext);
    my $decipher_padchar = ord(substr($deciphertext,($decipher_len - 1),1));
   
    my $i = 0;
    
    for ( $i = 0; $i < $decipher_padchar ; $i++ )
    {
        if ( $decipher_padchar != ord( substr($deciphertext,($decipher_len - $i - 1),1)))
        {
            last;
        }
    }
    
    # Return decrypted text
    if ( $i != $decipher_padchar )
    {
		### Log entries for debugging purposes
		Log3 $name, 5, $name. " : km200 - decryptData1 - decipher_len                : " .$decipher_len;
		$deciphertext =~ s/\x00+$//;
		Log3 $name, 5, $name. " : km200 - decryptData1 - deciphertext                : " .$deciphertext;
		### Log entries for debugging purposes
        return $deciphertext;
    }
    
	else
    {
		$deciphertext = substr($deciphertext,0,$decipher_len - $decipher_padchar);
		### Log entries for debugging purposes
		Log3 $name, 5, $name. " : km200 - decryptData2 - decipher_len                : " .$decipher_len;
		$deciphertext =~ s/\x00+$//;
		Log3 $name, 5, $name. " : km200 - decryptData2 - deciphertext                : " .$deciphertext;
		### Log entries for debugging purposes
        return $deciphertext;
    }
}
####END####### Subroutine Decrypt Data #########################################################################END#####

###START###### Subroutine set individual data value ###########################################################START####
sub km200_PostSingleService($) {
	my ($hash, $def)                = @_;
	my $Service                     = $hash->{temp}{service};
	my $km200_gateway_host          = $hash->{URL} ;
	my $name                        = $hash->{NAME} ;
	my $PollingTimeout              = $hash->{POLLINGTIMEOUT};
	my $err;
	my $data;
	my $jsonSend;
	my $jsonRead;
	my $JsonContent;
	
	### Log file entry for debugging
	Log3 $name, 5, $name. ("km200_Set - Writing value: " . $hash->{temp}{postdata} . " to the service                     : ". $Service . "\n");
	
	### Read the current json string 
	$jsonRead = km200_GetSingleService($hash);	
	
	#### If the get-command returns an error due to an unknown Service requested
	if ($jsonRead -> {type} eq "ERROR") {
		### Rescue original Service request
		my $WriteService = $Service;

		### Try to replace the Post-String with nothing
		$Service  =~ s/\/1-Mo//i;
		$Service  =~ s/\/2-Tu//i;
		$Service  =~ s/\/3-We//i;
		$Service  =~ s/\/4-Th//i;
		$Service  =~ s/\/5-Fr//i;
		$Service  =~ s/\/6-Sa//i;
		$Service  =~ s/\/7-Su//i;

		### Save corrected string in hash
		$hash->{temp}{service} = $Service;

		### Log file entry for debugging
		Log3 $name, 5, $name. "km200_Set - Trying to re-read Service - Assuming its a switchProgram list";

		### Try again to read the current json string again with the corrected service
		$jsonRead = km200_GetSingleService($hash);	

		### Check whether the type is an switchProgram. 
		### If true, the requested service was a particular week of the switchProgram
		if ($jsonRead -> {type} eq "switchProgram") {
			### Log file entry for debugging
			Log3 $name, 5, $name. "km200_Set - It is a switchProgram list!";

			### For each weekday, get current readings, delete all unnecessary blanks and transform to array
			my $TempReadingVal;
			
			   $TempReadingVal		= ReadingsVal($name,($Service . "/1-Mo"),"");
			   $TempReadingVal		=~ s/\s+/ /g;
			   $TempReadingVal		=~ s/\s+$//g;
			my @TempReadingMo 	    = split(/\s+/, $TempReadingVal,0);
			   
			   $TempReadingVal		= ReadingsVal($name,($Service . "/2-Tu"),"");
			   $TempReadingVal		=~ s/\s+/ /g;
			   $TempReadingVal		=~ s/\s+$//g;
			my @TempReadingTu 		= split(/\s+/, $TempReadingVal,0);
			   
			   $TempReadingVal		= ReadingsVal($name,($Service . "/3-We"),"");
			   $TempReadingVal		=~ s/\s+/ /g;
			   $TempReadingVal		=~ s/\s+$//g;
			my @TempReadingWe 		= split(/\s+/, $TempReadingVal,0);

			   $TempReadingVal		= ReadingsVal($name,($Service . "/4-Th"),"");
			   $TempReadingVal		=~ s/\s+/ /g;
			   $TempReadingVal		=~ s/\s+$//g;
			my @TempReadingTh 		= split(/\s+/, $TempReadingVal,0);
			
			   $TempReadingVal		= ReadingsVal($name,($Service . "/5-Fr"),"");
			   $TempReadingVal		=~ s/\s+/ /g;
			   $TempReadingVal		=~ s/\s+$//g;
			my @TempReadingFr 		= split(/\s+/, $TempReadingVal,0);
			
			   $TempReadingVal		= ReadingsVal($name,($Service . "/6-Sa"),"");
			   $TempReadingVal		=~ s/\s+/ /g;
			   $TempReadingVal		=~ s/\s+$//g;
			my @TempReadingSa 		= split(/\s+/, $TempReadingVal,0);
			
			   $TempReadingVal		= ReadingsVal($name,($Service . "/7-Su"),"");
			   $TempReadingVal		=~ s/\s+/ /g;
			   $TempReadingVal		=~ s/\s+$//g;
			my @TempReadingSu 		= split(/\s+/, $TempReadingVal,0);

			
			### For value to be written, delete all unnecessary blanks and transform to array and get length of array
			my $ReturnString		= $hash->{temp}{postdata};
			   $ReturnString 		=~ s/\s+/ /g;
			   $ReturnString 		=~ s/\s+$//g;
			my @TempReading			= split(/\s+/, $ReturnString);
			my $TempReadingLength	= @TempReading;

			
			### Obtain the allowed terminology for setpoints
			$hash->{temp}{service}	= $jsonRead -> {setpointProperty}{id};
			my $TempSetpointsJson	= km200_GetSingleService($hash);

			
			my @TempSetpointNames	=();
			### For each item found in this empty parent directory
			foreach my $item (@{ $TempSetpointsJson->{references} }) {
				my $TempSetPoint = substr($item->{id}, (rindex($item->{id}, "/") - length($item->{id}) +1));
				
				### Add service, which is one of the allowed terminologies at the same time, to the list of all known services
				push (@TempSetpointNames, $TempSetPoint);
			}
			
			### Restore the original service
			$hash->{temp}{service}	= $Service;
			
			### If number of switchpoints exceeds maximum allowed
			if (($TempReadingLength / 2) > $jsonRead -> {maxNbOfSwitchPointsPerDay}) {
				return ("ERROR - Too much Switchpoints for weeklist inserted. \n Do not add more than " . $jsonRead -> {maxNbOfSwitchPointsPerDay} . " SwitchPoints per day!\n");	
			}
			
			### If content of array is not even
			if (($TempReadingLength % 2)  != 0) {
				return "ERROR - At least one Switchtime or Switchpoint is missing. \n Make sure you always have couples of Switchtime and Switchpoint!\n";	
			}
			
			### Check whether description of setpoints is the same as referenced and the data is in the right order
			for (my $i=0;$i<$TempReadingLength;$i+=2) { 
				### If the even element behind the uneven index [1, 3, 5, ...] is not one of the pre-defined setpoints
				if (! grep /($TempReading[$i+1])/,@TempSetpointNames)
				{
					return "ERROR - At least for one Switchpoint the wrong terminology has been used. Only use one of the following items: " . join(' , ',@TempSetpointNames) ."\n";;
				}

				### If the uneven element behind the even index [0, 2, 4, ...]is not a number, hand back an error message
				if ($TempReading[$i] !~ /^[0-9.-]+$/)
				{
					return "ERROR - At least for one Switchtime a number is expected at that position. \n Ensure the correct syntax of time and switchpoint. (E.g. 0600 eco)\n";
				}
			
				### Convert timepoint into raster of defined switchPointTimeRaster
				my $TempHours    = substr($TempReading[$i], 0, length($TempReading[$i])-2);
				if ($TempHours > 23)
				{
					$TempHours = 23;
				}
				my $TempMinutes	 = substr($TempReading[$i], -2);
				if ($TempMinutes > 59)
				{
					$TempMinutes = 59;
				}
				$TempMinutes 	 = $TempMinutes / ($jsonRead -> {switchPointTimeRaster});
				$TempMinutes 	 =~ s/^(.*?)\..*$/$1/;
				$TempMinutes 	 = $TempMinutes * ($jsonRead -> {switchPointTimeRaster});
				$TempMinutes 	 = sprintf ('%02d', $TempMinutes);
				$TempReading[$i] = ($TempHours . $TempMinutes);

			}
			
			$hash->{temp}{postdata} = join(" ", @TempReading);
			
			### For the requested day to be changed, save new value
			if    ($WriteService =~ m/1-Mo/i) {
				@TempReadingMo   = @TempReading;
			}
			elsif ($WriteService =~ m/2-Tu/i) {
				@TempReadingTu   = @TempReading;
			}
			elsif ($WriteService =~ m/3-We/i) {
				@TempReadingWe   = @TempReading;
			}	
			elsif ($WriteService =~ m/4-Th/i) {
				@TempReadingTh   = @TempReading;
			}	
			elsif ($WriteService =~ m/5-Fr/i) {
				@TempReadingFr   = @TempReading;
			}	
			elsif ($WriteService =~ m/6-Sa/i) {
				@TempReadingSa   = @TempReading;
			}	
			elsif ($WriteService =~ m/7-Su/i) {
				@TempReadingSu   = @TempReading;
			}	

			
			### For every weekday create setpoint hash and push it to array of hashes of switchpoints to be send
			my @SwitchPointsSend =();
			
			for (my $i=0;$i<$#TempReadingMo;$i+=2) {
				my $TempHashSend;
				$TempHashSend->{"dayOfWeek"} = "Mo";
				my $TempHours                = substr($TempReadingMo[$i], 0, length($TempReadingMo[$i])-2);
				my $TempMinutes	             = substr($TempReadingMo[$i], -2);
				$TempHashSend->{"time"}      = ($TempHours * 60 ) + $TempMinutes;
				$TempHashSend->{"setpoint"}  = $TempReadingMo[$i+1];
				push @SwitchPointsSend, $TempHashSend;
			}

			for (my $i=0;$i<$#TempReadingTu;$i+=2) {
				my $TempHashSend;
				$TempHashSend->{"dayOfWeek"} = "Tu";
				my $TempHours                = substr($TempReadingTu[$i], 0, length($TempReadingTu[$i])-2);
				my $TempMinutes	             = substr($TempReadingTu[$i], -2);
				$TempHashSend->{"time"}      = ($TempHours * 60 ) + $TempMinutes;				
				$TempHashSend->{"setpoint"}  = $TempReadingTu[$i+1];
				push @SwitchPointsSend, $TempHashSend;
			}			
			
			for (my $i=0;$i<$#TempReadingWe;$i+=2) {
				my $TempHashSend;
				$TempHashSend->{"dayOfWeek"} = "We";
				my $TempHours                = substr($TempReadingWe[$i], 0, length($TempReadingWe[$i])-2);
				my $TempMinutes	             = substr($TempReadingWe[$i], -2);
				$TempHashSend->{"time"}      = ($TempHours * 60 ) + $TempMinutes;
				$TempHashSend->{"setpoint"}  = $TempReadingWe[$i+1];
				push @SwitchPointsSend, $TempHashSend;
			}
	
			for (my $i=0;$i<$#TempReadingTh;$i+=2) {
				my $TempHashSend;
				$TempHashSend->{"dayOfWeek"} = "Th";
				my $TempHours                = substr($TempReadingTh[$i], 0, length($TempReadingTh[$i])-2);
				my $TempMinutes	             = substr($TempReadingTh[$i], -2);
				$TempHashSend->{"time"}      = ($TempHours * 60 ) + $TempMinutes;
				$TempHashSend->{"setpoint"}  = $TempReadingTh[$i+1];
				push @SwitchPointsSend, $TempHashSend;
			}
			
			for (my $i=0;$i<$#TempReadingFr;$i+=2) {
				my $TempHashSend;
				$TempHashSend->{"dayOfWeek"} = "Fr";
				my $TempHours                = substr($TempReadingFr[$i], 0, length($TempReadingFr[$i])-2);
				my $TempMinutes	             = substr($TempReadingFr[$i], -2);
				$TempHashSend->{"time"}      = ($TempHours * 60 ) + $TempMinutes;
				$TempHashSend->{"setpoint"}  = $TempReadingFr[$i+1];
				push @SwitchPointsSend, $TempHashSend;
			}
			
			for (my $i=0;$i<$#TempReadingSa;$i+=2) {
				my $TempHashSend;
				$TempHashSend->{"dayOfWeek"} = "Sa";
				my $TempHours                = substr($TempReadingSa[$i], 0, length($TempReadingSa[$i])-2);
				my $TempMinutes	             = substr($TempReadingSa[$i], -2);
				$TempHashSend->{"time"}      = ($TempHours * 60 ) + $TempMinutes;
				$TempHashSend->{"setpoint"}  = $TempReadingSa[$i+1];
				push @SwitchPointsSend, $TempHashSend;
			}
			
			for (my $i=0;$i<$#TempReadingSu;$i+=2) {
				my $TempHashSend;
				$TempHashSend->{"dayOfWeek"} = "Su";
				my $TempHours                = substr($TempReadingSu[$i], 0, length($TempReadingSu[$i])-2);
				my $TempMinutes	 = substr($TempReadingSu[$i], -2);
				$TempHashSend->{"time"}      = ($TempHours * 60 ) + $TempMinutes;
				$TempHashSend->{"setpoint"}  = $TempReadingSu[$i+1];
				push @SwitchPointsSend, $TempHashSend;
			}

			### Save array of hashes of switchpoints into json hash to be send
			@{$jsonSend->{switchPoints}} = @SwitchPointsSend;
			
			### Create full URL of the current Service to be written
			my $url ="http://" . $km200_gateway_host . $Service;

			### Encode as json
			$JsonContent = encode_json($jsonSend);

			### Delete the name of hash, "{" and "}" out of json String. No idea why but result of Try-and-Error method
			$JsonContent =~ s/{"switchPoints"://;
			$JsonContent =~ s/]}/]/g;

			### Log file entry for debugging
			Log3 $name, 5, $name. "km200_Set - JsonContent                              : " . $JsonContent;

			### Encrypt 
			$hash->{temp}{jsoncontent} = $JsonContent;
			$data = km200_Encrypt($hash);

			### Log file entry for debugging
			Log3 $name, 5, $name. "km200_Set - Trying to push switchPoint list to KM-Unit";
			
			### Create parameter set for HttpUtils_BlockingGet
			my $param = {
							url        => $url,
							timeout    => $PollingTimeout * 5,
							data       => $data,
							method     => "POST",
							header     => "agent: TeleHeater/2.2.3\r\nUser-Agent: TeleHeater/2.2.3\r\nAccept: application/json",
						};
						
			### Block other scheduled and unscheduled routines
			$hash->{status}{FlagSetRequest} = true;

			### Write value with HttpUtils_BlockingGet
			($err, $data) = HttpUtils_BlockingGet($param);

			### Reset flag
			$hash->{status}{FlagSetRequest} = false;
			
			### If error message has been returned
			if($err ne "")  {
				Log3 $name, 2, $name . " - ERROR: $err";
				return $err;	
			}

			Log3 $name, 5, $name. ("Waiting for processing time (READBACKDELAY / [ms])     : " . $hash->{READBACKDELAY} . " \n");

			### Make a pause before ReadBack
			usleep ($hash->{READBACKDELAY}*1000);

		### Read service-hash and format it so it is compareable to the sent content
		my $ReReadContent = km200_GetSingleService($hash);
		   $ReReadContent = $ReReadContent->{switchPoints};
		   $ReReadContent = encode_json($ReReadContent);
		   $ReReadContent =~ s/{"switchPoints"://;
		   $ReReadContent =~ s/]}/]/g;
		   
		### Transform back into array of hashes
		eval  {
				$ReReadContent = decode_json(encode_utf8($ReReadContent));
				$JsonContent = decode_json(encode_utf8($JsonContent));
				1;
			}
		or do  {
			};
		
		### Set Counter for found items in SwitchPrograms
		my $FoundJsonItem = 0;
		
				
		### For every item of the array of SwitchPrograms to be send
		foreach my $ReReadItem (@{$ReReadContent}) {
			### Set Counter for found items of ReRead values
			my $FoundReReadItem = 0;
			
			### For every item of the array of SwitchPrograms after Re-Reading 
			foreach my $JsonItem (@{$JsonContent}) {
				### If the current Switchprogram - hash does not have the same amount of keys
				if (%$ReReadItem ne %$JsonItem) 
				{
					### Do nothing
					#print "they don't have the same number of keys\n";
				} 
				### If the current Switchprogram - hash do have the same amount of keys
				else 
				{
					### Compare key names and values
					my %cmp = map { $_ => 1 } keys %$ReReadItem;
					for my $key (keys %$JsonItem) 
					{
						last unless exists $cmp{$key};
						last unless $$ReReadItem{$key} eq $$JsonItem{$key};
						delete $cmp{$key};
					}
					if (%cmp) 
					{
						### Do nothing
						#print "they don't have the same keys or values\n";
					} 
					else 
					{
						### Inkrement Counter
						$FoundReReadItem = 1;
						#print "they have the same keys and values\n";
					}
				}
			}

			### If item has been found
			if ($FoundReReadItem == 1) { 
				### Inkrement Counter for found identical SwitchPoints
				$FoundJsonItem++;
			}
		}
		
		my $ReturnValue;
		
		if 	($FoundJsonItem == @{$ReReadContent}) {
			$ReturnValue = "The service " . $Service . " has been changed succesfully!";
			Log3 $name, 5, $name. "Writing $Service succesfully \n";
		}	
		else {
			$ReturnValue = "ERROR - The service " . $Service . " could not changed! \n";
		}

			### Return the status message
			return $ReturnValue;
		}
	}	
	### Check whether the type is an switchProgram. 
	### If true, the requested service is referring to the entire week but not a particular week.
	if ($jsonRead -> {type} eq "switchProgram") {
		### Create full URL of the current Service to be written
		my $url ="http://" . $km200_gateway_host . $Service;

		### Get the string to be send
		$JsonContent = $hash->{temp}{postdata};			

		### Encrypt 
		$hash->{temp}{jsoncontent} = $JsonContent;
		$data = km200_Encrypt($hash);
		
		### Create parameter set for HttpUtils_BlockingGet
		my $param = {
						url        => $url,
						timeout    => $PollingTimeout * 5,
						data       => $data,
						method     => "POST",
						header     => "agent: TeleHeater/2.2.3\r\nUser-Agent: TeleHeater/2.2.3\r\nAccept: application/json",
					};
					
		### Block other scheduled and unscheduled routines
		$hash->{status}{FlagSetRequest} = true;

		### Write value with HttpUtils_BlockingGet
		($err, $data) = HttpUtils_BlockingGet($param);

		### Reset flag
		$hash->{status}{FlagSetRequest} = false;
		
		### If error message has been returned
		if($err ne "")  {
			Log3 $name, 2, $name . " - ERROR: $err";
			return $err;	
		}

		Log3 $name, 5, $name. ("Waiting for processing time (READBACKDELAY / [ms])     : " . $hash->{READBACKDELAY} . " \n");
		### Make a pause before ReadBack
		usleep ($hash->{READBACKDELAY}*1000);

		### Read service-hash and format it so it is compareable to the sent content
		my $ReReadContent = km200_GetSingleService($hash);
		   $ReReadContent = $ReReadContent->{switchPoints};
		   $ReReadContent = encode_json($ReReadContent);
		   $ReReadContent =~ s/{"switchPoints"://;
		   $ReReadContent =~ s/]}/]/g;
		   
		### Transform back into array of hashes
		eval  {
				$ReReadContent = decode_json(encode_utf8($ReReadContent));
				$JsonContent = decode_json(encode_utf8($JsonContent));
				1;
			}
		or do  {
			};
		
		### Set Counter for found items in SwitchPrograms
		my $FoundJsonItem = 0;
		
				
		### For every item of the array of SwitchPrograms to be send
		foreach my $ReReadItem (@{$ReReadContent}) {
			### Set Counter for found items of ReRead values
			my $FoundReReadItem = 0;
			
			### For every item of the array of SwitchPrograms after Re-Reading 
			foreach my $JsonItem (@{$JsonContent}) {

				### If the current Switchprogram - hash does not have the same amount of keys
				if (%$ReReadItem != %$JsonItem) 
				{
					### Do nothing
					#print "they don't have the same number of keys\n";
				} 
				### If the current Switchprogram - hash do have the same amount of keys
				else 
				{
					### Compare key names and values
					my %cmp = map { $_ => 1 } keys %$ReReadItem;
					for my $key (keys %$JsonItem) 
					{
						last unless exists $cmp{$key};
						last unless $$ReReadItem{$key} eq $$JsonItem{$key};
						delete $cmp{$key};
					}
					if (%cmp) 
					{
						### Do nothing
						#print "they don't have the same keys or values\n";
					} 
					else 
					{
						### Inkrement Counter
						$FoundReReadItem = 1;
						#print "they have the same keys and values\n";
					}
				}
			}

			### If item has been found
			if ($FoundReReadItem == 1) { 
				### Inkrement Counter for found identical SwitchPoints
				$FoundJsonItem++;
			}
		}
		
		my $ReturnValue;
		
		if 	($FoundJsonItem == @{$ReReadContent}) {
			$ReturnValue = "The service " . $Service . " has been changed succesfully!";
			Log3 $name, 5, $name.  "The service $Service has been changed succesfully!";
			}	
		else {
			$ReturnValue = "ERROR - The service " . $Service . " could not changed! \n";
			Log3 $name, 5, $name. "Writing $Service was NOT succesfully";
		}

		### Return the status message
		return $ReturnValue;
	}	
	## Check whether the type is a single value containing a string
	elsif($jsonRead->{type} eq "stringValue") {	
		### Save chosen value into hash to be send
		$jsonSend->{value} = $hash->{temp}{postdata};

		### Log file entry for debugging
		Log3 $name, 5, $name. "km200_Set - String value";		

		### Create full URL of the current Service to be written
		my $url ="http://" . $km200_gateway_host . $Service;

		### Encode as json
		$JsonContent = encode_json($jsonSend);
		
		### Encrypt 
		$hash->{temp}{jsoncontent} = $JsonContent;
		$data = km200_Encrypt($hash);

		
		### Create parameter set for HttpUtils_BlockingGet
		my $param = {
						url        => $url,
						timeout    => $PollingTimeout,
						data       => $data,
						method     => "POST",
						header     => "agent: TeleHeater/2.2.3\r\nUser-Agent: TeleHeater/2.2.3\r\nAccept: application/json",
					};
					
		### Block other scheduled and unscheduled routines
		$hash->{status}{FlagSetRequest} = true;

		### Write value with HttpUtils_BlockingGet
		($err, $data) = HttpUtils_BlockingGet($param);

		### Reset flag
		$hash->{status}{FlagSetRequest} = false;
		
		### If error message has been returned
		if($err ne "")  {
			Log3 $name, 2, $name . " - ERROR: $err";
			return $err;	
		}

		### Make a pause before ReadBack
		usleep ($hash->{READBACKDELAY}*1000);
		
		### Read service-hash
		my $ReadValue = km200_GetSingleService($hash);
		
		### Return value
		my $ReturnValue = "";
		if ($ReadValue->{value} eq $hash->{temp}{postdata}) {	
			$ReturnValue = "The service " . $Service . " has been changed to: " . $ReadValue->{value};
			Log3 $name, 5, $name. "km200_Set - Writing " . $Service . " succesfully with value: " . $hash->{temp}{postdata};
		}
		else {
			$ReturnValue = "ERROR - The service " . $Service . " could not changed.";
			Log3 $name, 5, $name. "km200_Set - Writing " . $Service . " was NOT successful";
		}

		### Return the status message
		return $ReturnValue;
	}	
	## Check whether the type is a single value containing a float value
	elsif($jsonRead -> {type} eq "floatValue") {
		### Check whether value to be sent is numeric
		if ($hash->{temp}{postdata} =~ /^[0-9.-]+$/)  {
			### Save chosen value into hash to be send
			$jsonSend->{value} = ($hash->{temp}{postdata}) * 1;	

			### Log file entry for debugging
			Log3 $name, 5, $name. "km200_Set - Numeric value";
		
			### Create full URL of the current Service to be written
			my $url ="http://" . $km200_gateway_host . $Service;

			### Encode as json
			$JsonContent = encode_json($jsonSend);
			
			### Encrypt 
			$hash->{temp}{jsoncontent} = $JsonContent;
			$data = km200_Encrypt($hash);

			
			### Create parameter set for HttpUtils_BlockingGet
			my $param = {
							url        => $url,
							timeout    => $PollingTimeout,
							data       => $data,
							method     => "POST",
							header     => "agent: TeleHeater/2.2.3\r\nUser-Agent: TeleHeater/2.2.3\r\nAccept: application/json",
						};
						
			### Block other scheduled and unscheduled routines
			$hash->{status}{FlagSetRequest} = true;

			### Write value with HttpUtils_BlockingGet
			($err, $data) = HttpUtils_BlockingGet($param);

			### Reset flag
			$hash->{status}{FlagSetRequest} = false;
			
			### If error messsage has been returned
			if($err ne "")  {
				Log3 $name, 2, $name . " - ERROR: $err";
				return $err;	
			}

			### Make a pause before ReadBack
			usleep ($hash->{READBACKDELAY}*1000);	
			
			### Read service-hash
			my $ReadValue = km200_GetSingleService($hash);
			

			### Return value
			my $ReturnValue = "";
			if ($ReadValue->{value} eq $hash->{temp}{postdata}) {	
				$ReturnValue = "The service " . $Service . " has been changed to: " . $ReadValue->{value} . "\n";
				Log3 $name, 5, $name. "km200_Set - Writing " . $Service . " succesfully with value: " . $hash->{temp}{postdata};
			}
			elsif ($jsonRead -> {value} == $ReadValue->{value}) {
				$ReturnValue = "ERROR - The service " . $Service . " could not changed to: " . $hash->{temp}{postdata} . "\n The value is: " . $ReadValue->{value} . "\n";
				Log3 $name, 5, $name. "km200_Set - Writing " . $Service . " was NOT successful";
			}
			else {
				$ReturnValue = "The service " . $Service . " has been rounded to: " . $ReadValue->{value} . "\n";
				Log3 $name, 5, $name. "km200_Set - Writing " . $Service . " was rounded and changed successful";
			}

			### Return the status message
			return $ReturnValue;
		}
		### If the value to be sent is NOT numeric
		else {
			### Log file entry for debugging
			Log3 $name, 5, $name. "km200_Set - ERROR - Float value expected!";
			return ("km200_Set - ERROR - Float value expected!\n");
		}
	}
	## If the type is unknown
	else {
		### Log entries for debugging purposes
		Log3 $name, 4, $name. " : km200_SetSingleService - type unknown for : " .$Service;
	}
}
####END####### Subroutine set individual data value ############################################################END##### 


###START###### Subroutine get individual data value ###########################################################START####
sub km200_GetSingleService($) {
	my ($hash, $def)            	= @_;
	my $Service                 	= $hash->{temp}{service};
	my $km200_gateway_host      	= $hash->{URL};
	my $name                    	= $hash->{NAME};
	my $PollingTimeout              = $hash->{POLLINGTIMEOUT};
	my $json -> {type}				= "";
	   $json -> {value}				= "";
	my $err;
	my $data;

	### Log entries for debugging purposes
    Log3 $name, 5, $name. " : km200 - GetSingleService - service            : " .$Service;
	
	### Create full URL of the current Service to be read
	my $url ="http://" . $km200_gateway_host . $Service;

	### Log entries for debugging purposes
    Log3 $name, 5, $name. " : km200 - GetSingleService - url                : " .$url;
	
	### Create parameter set for HttpUtils_BlockingGet
	my $param = {
					url        => $url,
					timeout    => $PollingTimeout,
					method     => "GET",
					header     => "agent: TeleHeater/2.2.3\r\nUser-Agent: TeleHeater/2.2.3\r\nAccept: application/json",
				};

	### Block other scheduled and unscheduled routines
	$hash->{status}{FlagGetRequest} = true;

	### Retrieve data from km200
	($err, $data) = HttpUtils_BlockingGet($param);

	### Log entries for debugging purposes
    Log3 $name, 5, $name. " : km200 - GetSingleService - err                : " .$err;
	Log3 $name, 5, $name. " : km200 - GetSingleService - data               : " .$data;
	
	### Block other scheduled and unscheduled routines
	$hash->{status}{FlagGetRequest} = false;
	
	### If error message has been reported
	if($err ne "") {
		Log3 $name, 2, $name . " : ERROR: Service: ".$Service. ": No proper Communication with Gateway: " .$err;
		my $ReturnMessage ="ERROR";
		$json -> {type}   = $ReturnMessage;
		$json -> {value}  = $ReturnMessage;
		return $json;
	}
	### If NO error message has been reported
	else {
		$hash->{temp}{decodedcontent} = $data;
		my $decodedContent = km200_Decrypt($hash);

		if ($decodedContent ne "") {
			eval  {
				$json = decode_json(encode_utf8($decodedContent));
				1;
			}
			or do  {
			Log3 $name, 5, $name. " : km200_GetSingleService - Data cannot be parsed by JSON on km200 for http://" . $param->{url};
			};

			### Check whether the type is a single value containing a string or float value
			if(($json -> {type} eq "stringValue") || ($json -> {type} eq "floatValue")) {
				my $JsonId         = $json->{id};
				my $JsonType       = $json->{type};
				my $JsonValue      = $json->{value};

				### Save json-hash for DbLog-Split
				$hash->{temp}{ServiceDbLogSplitHash} = $json;
				$hash->{temp}{JsonRaw} = $decodedContent;
				
				
				### Write reading for fhem
				readingsSingleUpdate( $hash, $JsonId, $JsonValue, 1);

				return $json
			}	
			
			### Check whether the type is an switchProgram
			elsif ($json -> {type} eq "switchProgram") {
				my $JsonId         = $json->{id};
				my $JsonType       = $json->{type};

				### Log entries for debugging purposes
				Log3 $name, 4, $name. " : km200_GetSingleService  - value found for          : " .$Service;
				Log3 $name, 5, $name. " : km200_GetSingleService  - id                       : " .$JsonId;
				Log3 $name, 5, $name. " : km200_GetSingleService  - type                     : " .$JsonType;
				
				### Set up variables
				my $TempReturnVal = "";
				my $TempReadingMo = "";
				my $TempReadingTu = "";
				my $TempReadingWe = "";
				my $TempReadingTh = "";
				my $TempReadingFr = "";
				my $TempReadingSa = "";
				my $TempReadingSu = "";
				
				foreach my $item (@{ $json->{switchPoints} })
				{
					### Create string for time and switchpoint in fixed format and write part of Reading String
					
					my $time = $item->{time};
					my $temptime     = $time / 60;
					my $temptimeHH   = int($temptime);
					my $temptimeMM   = ($time - ($temptimeHH * 60));
					
					$temptimeHH = sprintf ('%02d', $temptimeHH);
					$temptimeMM = sprintf ('%02d', $temptimeMM);
					$temptime = $temptimeHH . $temptimeMM;
					
					my $tempsetpoint =  $item->{setpoint};
					$tempsetpoint    =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(8-length($1)))/e;
					my $TempReading  = $temptime . " " . $tempsetpoint;

					### Create ValueString for this day
					if ($item->{dayOfWeek} eq "Mo")
					{
						### If it is the first entry for this day
						if ($TempReadingMo eq "")
						{
							### Write the first entry
							$TempReadingMo = $TempReading;
						}
						### If it is NOT the first entry for this day
						else
						{
							### Add the next entry
							$TempReadingMo = $TempReadingMo . " " . $TempReading;
						}
					}
					elsif ($item->{dayOfWeek} eq "Tu")
					{
						### If it is the first entry for this day
						if ($TempReadingTu eq "")
						{
							### Write the first entry
							$TempReadingTu = $TempReading;
						}
						### If it is NOT the first entry for this day
						else
						{
							### Add the next entry
							$TempReadingTu = $TempReadingTu . " " . $TempReading;
						}
					}
					elsif ($item->{dayOfWeek} eq "We")
					{
						### If it is the first entry for this day
						if ($TempReadingWe eq "")
						{
							### Write the first entry
							$TempReadingWe = $TempReading;
						}
						### If it is NOT the first entry for this day
						else
						{
							### Add the next entry
							$TempReadingWe = $TempReadingWe . " " . $TempReading;
						}
					}
					elsif ($item->{dayOfWeek} eq "Th")
					{
						### If it is the first entry for this day
						if ($TempReadingTh eq "")
						{
							### Write the first entry
							$TempReadingTh = $TempReading;
						}
						### If it is NOT the first entry for this day
						else
						{
							### Add the next entry
							$TempReadingTh = $TempReadingTh . " " . $TempReading;
						}
					}
					elsif ($item->{dayOfWeek} eq "Fr")
					{
						### If it is the first entry for this day
						if ($TempReadingFr eq "")
						{
							### Write the first entry
							$TempReadingFr = $TempReading;
						}
						### If it is NOT the first entry for this day
						else
						{
							### Add the next entry
							$TempReadingFr = $TempReadingFr . " " . $TempReading;
						}
					}
					elsif ($item->{dayOfWeek} eq "Sa")
					{
						### If it is the first entry for this day
						if ($TempReadingSa eq "")
						{
							### Write the first entry
							$TempReadingSa = $TempReading;
						}
						### If it is NOT the first entry for this day
						else
						{
							### Add the next entry
							$TempReadingSa = $TempReadingSa . " " . $TempReading;
						}
					}
					elsif ($item->{dayOfWeek} eq "Su")
					{
						### If it is the first entry for this day
						if ($TempReadingSu eq "")
						{
							### Write the first entry
							$TempReadingSu = $TempReading;
						}
						### If it is NOT the first entry for this day
						else
						{
							### Add the next entry
							$TempReadingSu = $TempReadingSu . " " . $TempReading;
						}
					}
					else
					{
						Log3 $name, 5, $name. "dayOfWeek of unknow day: " . $item->{dayOfWeek};
					}
				}

				### Create new Service and write reading for fhem
				$TempReturnVal =                  "1-Mo: " . $TempReadingMo . "\n";
				$TempReturnVal = $TempReturnVal . "2-Tu: " . $TempReadingTu . "\n";
				$TempReturnVal = $TempReturnVal . "3-We: " . $TempReadingWe . "\n";
				$TempReturnVal = $TempReturnVal . "4-Th: " . $TempReadingTh . "\n";
				$TempReturnVal = $TempReturnVal . "5-Fr: " . $TempReadingFr . "\n";
				$TempReturnVal = $TempReturnVal . "6-Sa: " . $TempReadingSa . "\n";
				$TempReturnVal = $TempReturnVal . "7-Su: " . $TempReadingSu . "\n";

				### Save weeklist in "value"
				$json->{value} = $TempReturnVal;
				
				### Save raw Json string
				$hash->{temp}{JsonRaw} = $decodedContent;
	
				
				my $TempJsonId;
				
				### Create new Service and write reading for fhem
				$TempJsonId = $JsonId . "/" . "1-Mo";
				readingsSingleUpdate( $hash, $TempJsonId, $TempReadingMo, 1);

				### Create new Service and write reading for fhem
				$TempJsonId = $JsonId . "/" . "2-Tu";
				readingsSingleUpdate( $hash, $TempJsonId, $TempReadingTu, 1);

				### Create new Service and write reading for fhem
				$TempJsonId = $JsonId . "/" . "3-We";
				readingsSingleUpdate( $hash, $TempJsonId, $TempReadingWe, 1);

				### Create new Service and write reading for fhem
				$TempJsonId = $JsonId . "/" . "4-Th";
				readingsSingleUpdate( $hash, $TempJsonId, $TempReadingTh, 1);

				### Create new Service and write reading for fhem
				$TempJsonId = $JsonId . "/" . "5-Fr";
				readingsSingleUpdate( $hash, $TempJsonId, $TempReadingFr, 1);

				### Create new Service and write reading for fhem
				$TempJsonId = $JsonId . "/" . "6-Sa";
				readingsSingleUpdate( $hash, $TempJsonId, $TempReadingSa, 1);

				### Create new Service and write reading for fhem
				$TempJsonId = $JsonId . "/" . "7-Su";
				readingsSingleUpdate( $hash, $TempJsonId, $TempReadingSu, 1);

				return $json
			}
			### Check whether the type is an errorlist
			elsif ($json -> {type} eq "errorList") {
				my $TempErrorList    = "";
				
				### Sort list by timestamps descending
				my @TempSortedErrorList =  sort { $b->{t} cmp $a->{t} } @{ $json->{values} };
#				my @TempSortedErrorList =  @{ $json->{values} } ;
			
				### For every notification do
				foreach my $item (@TempSortedErrorList)
				{					
					### Create message string with fixed blocksize
					my $TempTime      = $item->{t};
					if ($TempTime) {$TempTime      =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(20-length($1)))/e;}
					my $TempErrorCode = $item->{dcd};
					   $TempErrorCode =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(3 -length($1)))/e;
					my $TempAddCode   = $item->{ccd};    
					   $TempAddCode   =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(4 -length($1)))/e;
					my $TempClassCode = $item->{cat};    
					   $TempClassCode =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(2- length($1)))/e;
					my $TempErrorMessage = "Time: " . $TempTime . "-ErrorCode: " . $TempErrorCode . " -AddCode: " . $TempAddCode . " -Category: " . $TempClassCode;

					### Create List
					$TempErrorList = $TempErrorList . $TempErrorMessage . "\n";
				}
				### Save raw Json string
				$hash->{temp}{JsonRaw} = $decodedContent;
				
				### Save errorList
				$json->{value} = $TempErrorList;
				
				return $json;
			}
			### Check whether the type is an refEnum which is indicating an empty parent directory
			elsif ($json -> {type} eq "refEnum") {
				### Initialise Return Message
				my $ReturnMessage = "";

				### For each item found in this empty parent directory
				foreach my $item (@{ $json->{references} })
				{
					### If it is the first item in the list
					if ($ReturnMessage eq "")
					{				
						$ReturnMessage = $item->{id};
					}
					### If it is not the first item in the list
					else
					{
						$ReturnMessage = $ReturnMessage . "\n" . $item->{id};
					}
				}
				### Return list of available directories
				$json->{value} = $ReturnMessage;
				
				### Save raw Json string
				$hash->{temp}{JsonRaw} = $decodedContent;
				
				return $json;
			}		
			### Check whether the type is a systeminfo
			elsif ($json -> {type} eq "systeminfo") {
				my $JsonId         = $json->{id};
				my $JsonType       = $json->{type};
				my @JsonValues     = $json->{values};

				### Log entries for debugging purposes
				Log3 $name, 4, $name. " : km200_GetSingleService  - value found for          : " .$Service;
				Log3 $name, 5, $name. " : km200_GetSingleService  - id                       : " .$JsonId;
				Log3 $name, 5, $name. " : km200_GetSingleService  - type                     : " .$JsonType;

				### Initialise Return Message
				my $ReturnMessage = "";
								
				### Initialise ArrayCounter 
				my $ArrayCounter = 0;
				
				foreach my $ArrayItem (@{ $json->{values} })
				{
					### Incrementation of ArrayCounter
					$ArrayCounter++;

					### Get array from scalar
					my %ArrayHash = %{$ArrayItem};
					
					while( my( $SystemInfoHashKey, $SystemInfoHashValue ) = each %ArrayHash )
					{
						### Create new Service and write reading for fhem
						my $TempJsonId = $JsonId . "/" . sprintf ('%02d', $ArrayCounter) . "/" . $SystemInfoHashKey;
						readingsSingleUpdate( $hash, $TempJsonId, $SystemInfoHashValue, 1);

						### If it is the first item in the list
						if ($ReturnMessage eq "")
						{				
							$ReturnMessage = $TempJsonId . " = " . $SystemInfoHashValue;
						}
						### If it is not the first item in the list
						else
						{
							$ReturnMessage = $ReturnMessage . "\n" . $TempJsonId . " = " . $SystemInfoHashValue;
						}
					}
				}

				### Return list of available directories
				$json->{value} = $ReturnMessage;
				
				### Save raw Json string
				$hash->{temp}{JsonRaw} = $decodedContent;
				
				return $json;
			}
			### If the type is unknown
			else {
				### Log entries for debugging purposes
				Log3 $name, 4, $name. " : km200_GetSingleService - type unknown for : " .$Service;
				### Log entries for debugging purposes
			}
		}
		else  {
			Log3 $name, 4, $name. " : km200_GetSingleService: ". $Service . " NOT available";
		
			my $ReturnMessage = "ERROR";
			$json -> {type}   = $ReturnMessage;
			$json -> {value}  = $ReturnMessage;
			return $json;
		}
	}
}
####END####### Subroutine get individual data value ############################################################END#####


###START###### Subroutine initial contact of services via HttpUtils ###########################################START####
sub km200_GetInitService($) {
	my ($hash, $def)                 = @_;
	my $km200_gateway_host           =   $hash->{URL} ;
	my $name                         =   $hash->{NAME} ;
	$hash->{status}{FlagInitRequest} = true;
	my @KM200_InitServices           = @{$hash->{Secret}{KM200ALLSERVICES}};
	my $ServiceCounterInit           = $hash->{temp}{ServiceCounterInit};
	my $PollingTimeout               = $hash->{POLLINGTIMEOUT};
	my $Service                      = $KM200_InitServices[$ServiceCounterInit];

	### Stop the current timer
	RemoveInternalTimer($hash);

	### If this this loop is accessed for the first time, stop the timer and set status
	if ($ServiceCounterInit == 0) {
		### Log file entry for debugging
		Log3 $name, 5, $name. "Sounding and importing of services started";
		
		### Set status of km200 fhem module
		$hash->{STATE} = "Sounding...";
	}


	### Get the values
	my $url ="http://" . $km200_gateway_host . $Service;

	my $param = {
					url        => $url,
					timeout    => $PollingTimeout,
					hash       => $hash,
					method     => "GET",
					header     => "agent: TeleHeater/2.2.3\r\nUser-Agent: TeleHeater/2.2.3\r\nAccept: application/json",
					callback   =>  \&km200_ParseHttpResponseInit
				};

	### Set flag for initialisation
	$hash->{status}{FlagInitRequest} = true;
	
	### Get the value
	HttpUtils_NonblockingGet($param);
}
####END####### Subroutine initial contact of services via HttpUtils ############################################END#####


###START###### Subroutine to download complete initial data set from gateway ##################################START####
# For all known, but not excluded services by attribute "DoNotPoll", try reading the respective values from gateway
sub km200_ParseHttpResponseInit($) {
    my ($param, $err, $data)     = @_;
    my $hash                     =   $param->{hash};
    my $name                     =   $hash ->{NAME};
	my $ServiceCounterInit       =   $hash ->{temp}{ServiceCounterInit};
	my @KM200_RespondingServices = @{$hash ->{Secret}{KM200RESPONDINGSERVICES}};
	my @KM200_WriteableServices  = @{$hash ->{Secret}{KM200WRITEABLESERVICES}};
	my @KM200_InitServices       = @{$hash ->{Secret}{KM200ALLSERVICES}};	
	my $NumberInitServices       = "";
	my $Service                  = $KM200_InitServices[$ServiceCounterInit];
	my $type;
    my $json ->{type} = "";
	
	if($err ne "") {
		### Create Log entry
		Log3 $name, 2, $name . " : km200_ParseHttpResponseInit - ERROR                : ".$Service. ": No proper Communication with Gateway: " .$err;
		
		### Set status of km200 fhem module
		$hash->{STATE} = "ERROR - Initial Connection failed... Try to re-connect in 10s";
		
		### Start the timer for polling again but wait 10s
		InternalTimer(gettimeofday()+10, "km200_GetInitService", $hash, 1);
		
		### Create Log entry
		Log3 $name, 2, $name . " : km200_ParseHttpResponseInit - ERROR                : Timer restarted to try again in 10s";
		Log3 $name, 5, $name . "______________________________________________________________________________________________________________________";
		return "ERROR";	
	}

	$hash->{temp}{decodedcontent} = $data;
	my $decodedContent = km200_Decrypt($hash);
	
	### Check whether the decoded content is not empty and therefore available
	if ($decodedContent ne "") {	
		eval  {
			$json = decode_json(encode_utf8($decodedContent));
			1;
		}
		or do  {
			Log3 $name, 4, $name. " : km200_ParseHttpResponseInit - CANNOT be parsed     : ". $Service;
		};

		### Check whether the type is a single value containing a string or float value
		if(($json -> {type} eq "stringValue") || ($json -> {type} eq "floatValue")) {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};
			my $JsonValue      = $json->{value};

			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - value found for      : " .$Service;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - id                   : " .$JsonId;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - type                 : " .$JsonType;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - value                : " .$JsonValue;

			### Add service to the list of responding services
			push (@KM200_RespondingServices, $Service);
			
			### Delete double entries in the list of responding services and sort in alphabetical order
			my %FilteredKM200RespondingServices;
			$FilteredKM200RespondingServices{$_}=0 for @KM200_RespondingServices;
			@KM200_RespondingServices = (keys %FilteredKM200RespondingServices);
			@KM200_RespondingServices = sort @KM200_RespondingServices;
			
			### Save json-hash for DbLog-Split
			$hash->{temp}{ServiceDbLogSplitHash} = $json;

			### Write reading for fhem
			readingsSingleUpdate( $hash, $JsonId, $JsonValue, 1);

			### Log file entry for debugging
			my $LogMessage = " : The following Service can be read";

			
			### Check whether service is writeable and write name of service in array
			if ($json->{writeable} == 1) {
				$LogMessage = $LogMessage . " and is writeable";
				push (@KM200_WriteableServices, $Service);
			}
			else {
				# Do nothing
				$LogMessage = $LogMessage . "                 ";
			}

			### Log file entry for debugging
			$LogMessage = $LogMessage . " : " . $JsonId;
			Log3 $name, 4, $name. $LogMessage;
		}	
		### Check whether the type is an switchProgram
		elsif ($json -> {type} eq "switchProgram") {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};
			my @JsonValues     = $json->{switchPoints};

			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - value found for      : " .$Service;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - id                   : " .$JsonId;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - type                 : " .$JsonType;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - value                : " .@JsonValues;

			### Add service to the list of responding services
			push (@KM200_RespondingServices, $Service);

			### Delete double entries in the list of responding services and sort in alphabetical order
			my %FilteredKM200RespondingServices;
			$FilteredKM200RespondingServices{$_}=0 for @KM200_RespondingServices;
			@KM200_RespondingServices = (keys %FilteredKM200RespondingServices);
			@KM200_RespondingServices = sort @KM200_RespondingServices;
			
			### Log file entry for debugging
			my $LogMessage = " : The following Service can be read";
			
			### Check whether service is writeable and write name of service in array
			if ($json->{writeable} == 1) {
				$LogMessage = $LogMessage . " and is writeable";
				push (@KM200_WriteableServices, $Service);
			}
			else {
				# Do nothing
				$LogMessage = $LogMessage . "                 ";
			}
			$LogMessage = $LogMessage . " : " .$JsonId;
			Log3 $name, 4, $name . $LogMessage;
			
			### Set up variables
			my $TempJsonId    = "";
			my $TempReadingMo = "";
			my $TempReadingTu = "";
			my $TempReadingWe = "";
			my $TempReadingTh = "";
			my $TempReadingFr = "";
			my $TempReadingSa = "";
			my $TempReadingSu = "";
			
			foreach my $item (@{ $json->{switchPoints} }) {
				### Create string for time and switchpoint in fixed format and write part of Reading String
				my $time         = $item->{time};
				my $temptime     = $time / 60;
				my $temptimeHH   = int($temptime);
				my $temptimeMM   = ($time - ($temptimeHH * 60));

				$temptimeHH = sprintf ('%02d', $temptimeHH);
				$temptimeMM = sprintf ('%02d', $temptimeMM);
				$temptime = $temptimeHH . $temptimeMM;
				
				my $tempsetpoint =  $item->{setpoint};
				$tempsetpoint    =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(8-length($1)))/e;
				my $TempReading  = $temptime . " " . $tempsetpoint;

				### Create ValueString for this day
				if ($item->{dayOfWeek} eq "Mo")
				{
					### If it is the first entry for this day
					if ($TempReadingMo eq "")
					{
						### Write the first entry
						$TempReadingMo = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingMo = $TempReadingMo . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Tu")
				{
					### If it is the first entry for this day
					if ($TempReadingTu eq "")
					{
						### Write the first entry
						$TempReadingTu = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingTu = $TempReadingTu . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "We")
				{
					### If it is the first entry for this day
					if ($TempReadingWe eq "")
					{
						### Write the first entry
						$TempReadingWe = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingWe = $TempReadingWe . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Th")
				{
					### If it is the first entry for this day
					if ($TempReadingTh eq "")
					{
						### Write the first entry
						$TempReadingTh = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingTh = $TempReadingTh . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Fr")
				{
					### If it is the first entry for this day
					if ($TempReadingFr eq "")
					{
						### Write the first entry
						$TempReadingFr = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingFr = $TempReadingFr . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Sa")
				{
					### If it is the first entry for this day
					if ($TempReadingSa eq "")
					{
						### Write the first entry
						$TempReadingSa = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingSa = $TempReadingSa . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Su")
				{
					### If it is the first entry for this day
					if ($TempReadingSu eq "")
					{
						### Write the first entry
						$TempReadingSu = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingSu = $TempReadingSu . " " . $TempReading;
					}
				}
				else
				{
					Log3 $name, 5, $name. "dayOfWeek of unknow day: " . $item->{dayOfWeek};
				}
			}

			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "1-Mo";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingMo, 1);
			### Log file entry for debugging
			Log3 $name, 5, $name. " : The following Service can be read and is writeable : " . $TempJsonId;
			### Add service to the list of writeable services
			push (@KM200_WriteableServices, $TempJsonId);

			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "2-Tu";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingTu, 1);
			### Log file entry for debugging
			Log3 $name, 5, $name. " : The following Service can be read and is writeable : " . $TempJsonId;
			### Add service to the list of writeable services
			push (@KM200_WriteableServices, $TempJsonId);
			
			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "3-We";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingWe, 1);
			### Log file entry for debugging
			Log3 $name, 5, $name. " : The following Service can be read and is writeable : " . $TempJsonId;
			### Add service to the list of writeable services
			push (@KM200_WriteableServices, $TempJsonId);

			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "4-Th";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingTh, 1);
			### Log file entry for debugging
			Log3 $name, 5, $name. " : The following Service can be read and is writeable : " . $TempJsonId;
			### Add service to the list of writeable services
			push (@KM200_WriteableServices, $TempJsonId);

			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "5-Fr";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingFr, 1);
			### Log file entry for debugging
			Log3 $name, 5, $name. " : The following Service can be read and is writeable : " . $TempJsonId;
			### Add service to the list of writeable services
			push (@KM200_WriteableServices, $TempJsonId);

			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "6-Sa";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingSa, 1);
			### Log file entry for debugging
			Log3 $name, 5, $name. " : The following Service can be read and is writeable : " . $TempJsonId;
			### Add service to the list of writeable services
			push (@KM200_WriteableServices, $TempJsonId);

			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "7-Su";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingSu, 1);
			### Log file entry for debugging
			Log3 $name, 5, $name. " : The following Service can be read and is writeable : " . $TempJsonId;
			### Add service to the list of writeable services
			push (@KM200_WriteableServices, $TempJsonId);

		}
		### Check whether the type is an errorlist
		elsif ($json -> {type} eq "errorList") {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};

			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - value found for      : " .$Service;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - id                   : " .$JsonId;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - type                 : " .$JsonType;

			### Add service to the list of responding services
			push (@KM200_RespondingServices, $Service);

			### Delete double entries in the list of responding services and sort in alphabetical order
			my %FilteredKM200RespondingServices;
			$FilteredKM200RespondingServices{$_}=0 for @KM200_RespondingServices;
			@KM200_RespondingServices = (keys %FilteredKM200RespondingServices);
			@KM200_RespondingServices = sort @KM200_RespondingServices;
			
			### Log file entry for debugging
			my $LogMessage = " : The following Service can be read";
			
			### Check whether service is writeable and write name of service in array
			if ($json->{writeable} == 1) {
				$LogMessage = $LogMessage . " and is writeable     ";
				push (@KM200_WriteableServices, $Service);
			}
			else {
				# Do nothing
				$LogMessage = $LogMessage . "                 ";
			}
			### Log file entry for debugging
			$LogMessage = $LogMessage . " : " . $JsonId;
			Log3 $name, 4, $name . $LogMessage;
			
			### Sort list by timestamps descending
			my $TempServiceIndex = 0;
			my @TempSortedErrorList =  sort { $b->{t} cmp $a->{t} } @{ $json->{values} };
           #my @TempSortedErrorList =  @{ $json->{values} };
			
			foreach my $item (@TempSortedErrorList) {
				### Increment Service-Index
				$TempServiceIndex++;
				
				### Create message string with fixed blocksize
				my $TempTime      = $item->{t};
				if ($TempTime) {$TempTime =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(20-length($1)))/e;} 
				my $TempErrorCode = $item->{dcd};
				   $TempErrorCode =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(3 -length($1)))/e;
				my $TempAddCode   = $item->{ccd};    
				   $TempAddCode   =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(4 -length($1)))/e;
				my $TempClassCode = $item->{cat};    
				   $TempClassCode =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(2- length($1)))/e;
				my $TempErrorMessage = "Time: " . $TempTime . "-ErrorCode: " . $TempErrorCode . " -AddCode: " . $TempAddCode . " -Category: " . $TempClassCode;
				
				### Create Service with Increment
				my $TempServiceString = $Service . "/Error-" . (sprintf("%02d", $TempServiceIndex));
				
				### Write Reading
				readingsSingleUpdate( $hash, $TempServiceString, $TempErrorMessage, 1);
				
				### Log file entry for debugging
				Log3 $name, 5, $name. " : The following Service can be read                       : " .$TempServiceString;
			}
		}
		### Check whether the type is an refEnum which is indicating an empty parent directory
		elsif ($json -> {type} eq "refEnum") {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};
			my @JsonReferences = $json->{references};
			
			### Log file entry for debugging
			Log3 $name, 5, $name. " : The following Service is an empty parent directory : " . $JsonId;
			
			### For each item found in this empty parent directory
			foreach my $item (@{ $json->{references} }) {
				my $SearchWord = $item->{id};
				
				### If the Service found is listed as blocked service
				if ((grep {$_ eq $SearchWord} @{$hash->{Secret}{KM200DONOTPOLL}}) == 1)
				{
					### Do nothing
					
					### Log file entry for debugging
					Log3 $name, 5, $name. "The following Service has been found but is blacklisted: " . $item->{id};
				}
				### If the Service found is NOT listed as blocked service
				else
				{
					### Add service to the list of all known services
					push (@{$hash ->{Secret}{KM200ALLSERVICES}}, $item->{id});
				}
			}
			### Sort the list of all services alphabetically
			@{$hash ->{Secret}{KM200ALLSERVICES}} = sort @{$hash ->{Secret}{KM200ALLSERVICES}};
		}
		### Check whether the type is a systeminfo
		elsif ($json -> {type} eq "systeminfo") {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};
			my @JsonValues     = $json->{values};

			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - value found for      : " .$Service;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - id                   : " .$JsonId;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - type                 : " .$JsonType;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - value                : " .@JsonValues;

			### Add service to the list of responding services
			push (@KM200_RespondingServices, $Service);

			### Delete double entries in the list of responding services and sort in alphabetical order
			my %FilteredKM200RespondingServices;
			$FilteredKM200RespondingServices{$_}=0 for @KM200_RespondingServices;
			@KM200_RespondingServices = (keys %FilteredKM200RespondingServices);
			@KM200_RespondingServices = sort @KM200_RespondingServices;
			
			### Log file entry for debugging
			Log3 $name, 4, $name . " : km200_ParseHttpResponseInit - Service can be read  : " .$JsonId;

			### Initialise ArrayCounter 
			my $ArrayCounter = 0;
			
			foreach my $ArrayItem (@{ $json->{values} }) {
				### Incrementation of ArrayCounter
				$ArrayCounter++;

				### Log file entry for debugging
				Log3 $name, 5, $name . " : The ArrayItem is                                   : " .       $ArrayItem  ;
				Log3 $name, 5, $name . " : The keys ArrayItem is                              : " . (keys %{$ArrayItem}) ;
				
				### Get array from scalar
				my %ArrayHash = %{$ArrayItem};
				
				while( my( $SystemInfoHashKey, $SystemInfoHashValue ) = each %ArrayHash )
				{
					### Log file entry for debugging
					Log3 $name, 5, $name . " : The ArrayHashKey is                                : " . $SystemInfoHashKey;
					Log3 $name, 5, $name . " : The ArrayHashValue is                              : " . $SystemInfoHashValue;

					### Create new Service and write reading for fhem
					my $TempJsonId = $JsonId . "/" . sprintf ('%02d', $ArrayCounter) . "/" . $SystemInfoHashKey;
					readingsSingleUpdate( $hash, $TempJsonId, $SystemInfoHashValue, 1);
					### Log file entry for debugging
					Log3 $name, 5, $name . " : km200_ParseHttpResponseInit - Service can be read  : " . $TempJsonId;

				}
			}
		}
		### Check whether the type is a recording and is a 'power' recording
		elsif ($json -> {type} eq "yRecording") {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};

			### Log entries for debugging purposes
			Log3 $name, 5, $name . " : km200_ParseHttpResponseInit___________________________________________________________________________________";
			Log3 $name, 5, $name . " : km200_ParseHttpResponseInit - value found for      : " .$Service;
			Log3 $name, 5, $name . " : km200_ParseHttpResponseInit - id                   : " .$JsonId;
			Log3 $name, 5, $name . " : km200_ParseHttpResponseInit - type                 : " .$JsonType;

			### Add service to the list of responding services
			push (@KM200_RespondingServices, $Service);

			### Delete double entries in the list of responding services and sort in alphabetical order
			my %FilteredKM200RespondingServices;
			$FilteredKM200RespondingServices{$_}=0 for @KM200_RespondingServices;
			@KM200_RespondingServices = (keys %FilteredKM200RespondingServices);
			@KM200_RespondingServices = sort @KM200_RespondingServices;

			### Log file entry for debugging
			Log3 $name, 5, $name . " : km200_ParseHttpResponseInit - Service can be read  : " .$JsonId;
			
			### Log file entry for debugging
			my $LogMessage = " : The following Service can be read";
			
			### Check whether service is writeable and write name of service in array
			if ($json->{writeable} == 1) {
				$LogMessage = $LogMessage . " and is writeable";
				push (@KM200_WriteableServices, $Service);
			}
			else {
				# Do nothing
				$LogMessage = $LogMessage . "                 ";
			}
			$LogMessage = $LogMessage . " : " .$JsonId;
			Log3 $name, 4, $name . $LogMessage;
		}
		## Check whether the type is unknown
		else {
			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - type unknown for     : " .$Service;
		}
	}
	### Check whether the decoded content is empty and therefore NOT available
	else {
		### Log entries for debugging purposes
		Log3 $name, 4, $name. " : km200_ParseHttpResponseInit -  NOT available       : ". $Service;
	}

	### Log entries for debugging purposes
	#Log3 $name, 5, $name. " : km200_ParseHttpResponseInit    : response          : " .$data;
	Log3 $name, 5, $name . "______________________________________________________________________________________________________________________";
	### Get the size of the array
	@KM200_InitServices       = @{$hash ->{Secret}{KM200ALLSERVICES}};
	$NumberInitServices       = @KM200_InitServices;
	
	### If the list of KM200ALLSERVICES has not been finished yet
	if ($ServiceCounterInit < ($NumberInitServices-1)) {
		++$ServiceCounterInit;
		$hash->{temp}{ServiceCounterInit}           = $ServiceCounterInit;
		@{$hash->{Secret}{KM200RESPONDINGSERVICES}} = @KM200_RespondingServices;
		@{$hash->{Secret}{KM200WRITEABLESERVICES}}  = @KM200_WriteableServices;
		km200_GetInitService($hash);
	}
	### If the list of KM200ALLSERVICES is finished
	else {
		my @KM200_DynServices = @KM200_RespondingServices;
		
		### Save arrays of services in hash
		@{$hash->{Secret}{KM200RESPONDINGSERVICES}} = @KM200_RespondingServices;
		@{$hash->{Secret}{KM200WRITEABLESERVICES}}  = @KM200_WriteableServices;
		@{$hash->{Secret}{KM200DYNSERVICES}}        = @KM200_DynServices;

		
		### Reset flag for initialisation
		$hash->{status}{FlagInitRequest}            = false;
		
		
		###START###### Initiate the timer for continuous polling of dynamical values from KM200 ###################START####
		InternalTimer(gettimeofday()+($hash->{INTERVALDYNVAL}), "km200_GetDynService", $hash, 1);
		Log3 $name, 4, $name. " : km200 - Define: InternalTimer for dynamic values started with interval of: ".($hash->{INTERVALDYNVAL});
		####END####### Initiate the timer for continuous polling of dynamical values from KM200 ####################END#####

		### Reset fullResponse error message
		readingsSingleUpdate( $hash, "fullResponse", "OK", 1);
		
		### Log file entry for debugging
		Log3 $name, 5, $name. "Sounding and importing of services is completed";

		### Set status of km200 fhem module
		$hash->{STATE} = "Standby";

		### Disable flag
		$hash->{temp}{ServiceCounterInit} = false;
	}
	### If the Initialisation process has been interuppted with an error message
	if (ReadingsVal($name,"fullResponse",0) eq "ERROR") {
		### Reset fullResponse error message
		readingsSingleUpdate( $hash, "fullResponse", "Restarted after ERROR", 1);
		
		### Reset timer for init procedure and start over again until it works
		InternalTimer(gettimeofday()+5, "km200_GetInitService", $hash, 1);
		Log3 $name, 5, $name. " : km200 - Internal timer for Initialisation of services restarted after fullResponse - error.";
	}
	
	### Clear up temporary variables
	$hash->{temp}{decodedcontent} = "";	
	
	return;
}
####END####### Subroutine to download complete initial data set from gateway ###################################END#####


###START###### Subroutine obtaining dynamic services via HttpUtils ############################################START####
sub km200_GetDynService($) {
	my ($hash, $def)                 = @_;
	my $km200_gateway_host           =   $hash->{URL};
	my $name                         =   $hash->{NAME};
	$hash->{STATE}                   = "Polling";
	my @KM200_DynServices            = @{$hash->{Secret}{KM200DYNSERVICES}};
	my $ServiceCounterDyn            =   $hash->{temp}{ServiceCounterDyn};
	my $PollingTimeout               =   $hash->{POLLINGTIMEOUT};
	

	### If at least one service to be polled is available
	if (@KM200_DynServices != 0) {
		my $Service = $KM200_DynServices[$ServiceCounterDyn];
		
		### Log file entry for debugging
		if ($ServiceCounterDyn == 0) {
			Log3 $name, 5, $name . "Starting download of dynamic services";
		}

		### Log file entry for debugging
		Log3 $name, 5, $name . " - km200_GetDynService - Polling                      : " . $Service;

		### Check whether Service is an parent yRecording entry
		if (($Service =~ m/recordings/ ) and ($Service !~ m/interval=/)) {
			my $SearchPattern = $Service . "?interval=";

			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_GetDynService____________________________________________________________________";
			Log3 $name, 5, $name. " : km200_GetDynService  - Service                     : " .$Service;
			Log3 $name, 5, $name. " : km200_GetDynService  - KM200_DynServices pre       :/n" .Dumper(@KM200_DynServices);
			Log3 $name, 5, $name. " : km200_GetDynService  - SearchPattern               : " .$SearchPattern;


			### Search in array for old interval entries and delete them
			for(my $i = 0; $i < @KM200_DynServices; $i++) {
				if (index($KM200_DynServices[$i], $SearchPattern) != -1) {
					### Log entries for debugging purposes
					Log3 $name, 5, $name. " : km200_GetDynService  - Service purged              : " . sprintf('%03d', $i) . " - " . $KM200_DynServices[$i];
					
					### Delete entry in Array
					splice(@KM200_DynServices,$i,1);
					
					### Correct index by due to deleted array-item
					$i--;
				} 
				else
				{
					### Log entries for debugging purposes
					Log3 $name, 5, $name. " : km200_GetDynService  - Service NOT purged          : " . sprintf('%03d', $i) . " - " . $KM200_DynServices[$i];
				}
			}

			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_GetDynService  - KM200_DynServices purge     :/n" .Dumper(@KM200_DynServices);

			### Define interval for current timestamp
			my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime();
			my @Intervals = (
								sprintf ('%04d-%02d-%02d', $year+1900,$mon+1,$mday), 
								sprintf ('%04d-%02d'     , $year+1900,$mon+1),
								sprintf ('%04d'          , $year+1900)
							);

			### For all possible intervals
			for(@Intervals) {
				my $tmpid = $Service.'?interval='.$_;
				### Log recording service for debugging
				Log3 $name, 5, $name. " : km200_GetDynService  - add service                 : ". $tmpid;
				
				### Add service to the list of all known services
				push (@KM200_DynServices, $tmpid);
			}

			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_GetDynService  - KM200_DynServices post      :/n" .Dumper(@KM200_DynServices);

			### Save Array of new DynServices into hash
			@{$hash->{Secret}{KM200DYNSERVICES}} = @KM200_DynServices;
			
			### Jump to next value
			++$ServiceCounterDyn;
			$hash->{temp}{ServiceCounterDyn} = $ServiceCounterDyn;
			InternalTimer(gettimeofday()+0, \&km200_GetDynService, $hash, 1);
#			km200_GetDynService($hash);
		}
		else {
			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_GetDynService  - Downloading Service         : " .$Service;

			my $url = "http://" . $km200_gateway_host . $Service;
			my $param = {
							url        => $url,
							timeout    => $PollingTimeout,
							hash       => $hash,
							method     => "GET",
							header     => "agent: TeleHeater/2.2.3\r\nUser-Agent: TeleHeater/2.2.3\r\nAccept: application/json",
							callback   =>  \&km200_ParseHttpResponseDyn
						};

			### Set Status Flag in order state running dynamic request
			$hash->{status}{FlagDynRequest} = true;

			### Get data
			HttpUtils_NonblockingGet($param);
		}
	}
	### If no service to be polled is available
	else {
		Log3 $name, 4, $name . " : km200_GetDynService  - No dynamic values available to be read. Skipping download.";
	}
}
####END####### Subroutine get dynamic data value ###############################################################END#####

###START###### Subroutine to download complete dynamic data set from gateway ##################################START####
# For all responding dynamic services read the respective values from gateway
sub km200_ParseHttpResponseDyn($) {
    my ($param, $err, $data)    = @_;
    my $hash                    =   $param->{hash};
    my $name                    =   $hash ->{NAME};
	my $ServiceCounterDyn       =   $hash ->{temp}{ServiceCounterDyn};
	my @KM200_DynServices       = @{$hash ->{Secret}{KM200DYNSERVICES}};	
	my $NumberDynServices       = @KM200_DynServices;
	my $Service                 = $KM200_DynServices[$ServiceCounterDyn];
	my $km200_gateway_host      =   $hash->{URL};
	my $PollingTimeout          =   $hash->{POLLINGTIMEOUT};
	my $json ->{type}           = "";
	my $type;

	Log3 $name, 5, $name. " : Parsing response of dynamic service received for   : " . $Service;

	### Reset Status Flag
	$hash->{status}{FlagDynRequest}           = false;
	
	if($err ne "") {
		Log3 $name, 2, $name . " : ERROR: Service: ".$Service. ": No proper Communication with Gateway: " .$err;
        readingsSingleUpdate($hash, "fullResponse", "ERROR", 1);
	}

	$hash->{temp}{decodedcontent} = $data;
	my $decodedContent = km200_Decrypt($hash);
	
	if ($decodedContent ne "") {
		eval  {
			$json = decode_json(encode_utf8($decodedContent));
			1;
		}
		or do  {
			Log3 $name, 5, $name. " - km200_parseHttpResponseDyn : Data cannot be parsed by JSON on km200 for http://" . $param->{url};
		};
		
		### Check whether the type is a single value containing a string or float value
		if(($json -> {type} eq "stringValue") || ($json -> {type} eq "floatValue")) {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};
			my $JsonValue      = $json->{value};
			
			### Log entries for debugging purposes
			Log3 $name, 4, $name. " : km200_parseHttpResponseDyn  - value found for      : " .$Service;
			Log3 $name, 5, $name. " : km200_parseHttpResponseDyn  - id                   : " .$JsonId;
			Log3 $name, 5, $name. " : km200_parseHttpResponseDyn  - type                 : " .$JsonType;
			Log3 $name, 5, $name. " : km200_parseHttpResponseDyn  - value                : " .$JsonValue;
			### Log entries for debugging purposes

			### Save json-hash for DbLog-Split
			$hash->{temp}{ServiceDbLogSplitHash} = $json;
			### Save json-hash for DbLog-Split

			### Write reading
			readingsSingleUpdate( $hash, $JsonId, $JsonValue, 1);
			### Write reading
		}			
		### Check whether the type is an switchProgram
		elsif ($json -> {type} eq "switchProgram") {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};

			### Log entries for debugging purposes
			Log3 $name, 4, $name. " : km200_parseHttpResponseDyn  - value found for      : " .$Service;
			Log3 $name, 5, $name. " : km200_parseHttpResponseDyn  - id                   : " .$JsonId;
			Log3 $name, 5, $name. " : km200_parseHttpResponseDyn  - type                 : " .$JsonType;
			
			### Set up variables
			my $TempJsonId    = "";
			my $TempReadingMo = "";
			my $TempReadingTu = "";
			my $TempReadingWe = "";
			my $TempReadingTh = "";
			my $TempReadingFr = "";
			my $TempReadingSa = "";
			my $TempReadingSu = "";
			
			foreach my $item (@{ $json->{switchPoints} }) {
				### Create string for time and switchpoint in fixed format and write part of Reading String
				my $time         = $item->{time};
				my $temptime     = $time / 60;
				my $temptimeHH   = int($temptime);
				my $temptimeMM   = ($time - ($temptimeHH * 60));

				$temptimeHH = sprintf ('%02d', $temptimeHH);
				$temptimeMM = sprintf ('%02d', $temptimeMM);
				$temptime = $temptimeHH . $temptimeMM;
				
				my $tempsetpoint =  $item->{setpoint};
				$tempsetpoint    =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(8-length($1)))/e;
				my $TempReading  = $temptime . " " . $tempsetpoint;

				### Create ValueString for this day
				if ($item->{dayOfWeek} eq "Mo")
				{
					### If it is the first entry for this day
					if ($TempReadingMo eq "")
					{
						### Write the first entry
						$TempReadingMo = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingMo = $TempReadingMo . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Tu")
				{
					### If it is the first entry for this day
					if ($TempReadingTu eq "")
					{
						### Write the first entry
						$TempReadingTu = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingTu = $TempReadingTu . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "We")
				{
					### If it is the first entry for this day
					if ($TempReadingWe eq "")
					{
						### Write the first entry
						$TempReadingWe = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingWe = $TempReadingWe . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Th")
				{
					### If it is the first entry for this day
					if ($TempReadingTh eq "")
					{
						### Write the first entry
						$TempReadingTh = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingTh = $TempReadingTh . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Fr")
				{
					### If it is the first entry for this day
					if ($TempReadingFr eq "")
					{
						### Write the first entry
						$TempReadingFr = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingFr = $TempReadingFr . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Sa")
				{
					### If it is the first entry for this day
					if ($TempReadingSa eq "")
					{
						### Write the first entry
						$TempReadingSa = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingSa = $TempReadingSa . " " . $TempReading;
					}
				}
				elsif ($item->{dayOfWeek} eq "Su")
				{
					### If it is the first entry for this day
					if ($TempReadingSu eq "")
					{
						### Write the first entry
						$TempReadingSu = $TempReading;
					}
					### If it is NOT the first entry for this day
					else
					{
						### Add the next entry
						$TempReadingSu = $TempReadingSu . " " . $TempReading;
					}
				}
				else
				{
					Log3 $name, 5, $name. "dayOfWeek of unknow day: " . $item->{dayOfWeek};
				}
			}

			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "1-Mo";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingMo, 1);
			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "2-Tu";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingTu, 1);
			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "3-We";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingWe, 1);
			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "4-Th";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingTh, 1);
			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "5-Fr";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingFr, 1);
			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "6-Sa";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingSa, 1);
			
			### Create new Service and write reading for fhem
			$TempJsonId = $JsonId . "/" . "7-Su";
			readingsSingleUpdate( $hash, $TempJsonId, $TempReadingSu, 1);
		}
		### Check whether the type is an errorlist
		elsif ($json -> {type} eq "errorList") {
			my $JsonId           = $json->{id};
			my $JsonType         = $json->{type};
			my $TempServiceIndex = 0;

			### Sort list by timestamps descending
			my @TempSortedErrorList =  sort { $b->{t} cmp $a->{t} } @{ $json->{values} };
#			my @TempSortedErrorList =  @{ $json->{values} } ;
			
			### For every notification do
			foreach my $item (@TempSortedErrorList) {
				### Increment Service-Index
				$TempServiceIndex++;
				
				### Create message string with fixed blocksize
				my $TempTime      = $item->{t};
				if ($TempTime) 
				{
					$TempTime     =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(20-length($1)))/e;
				}
				else
				{
					$TempTime     = "unknown";
				}
				
				my $TempErrorCode = $item->{dcd};
				   $TempErrorCode =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(3 -length($1)))/e;
				my $TempAddCode   = $item->{ccd};    
				   $TempAddCode   =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(4 -length($1)))/e;
				my $TempClassCode = $item->{cat};    
				   $TempClassCode =~ s/^(.+)$/sprintf("%s%s", $1, ' 'x(2- length($1)))/e;
				my $TempErrorMessage = "Time: " . $TempTime . "-ErrorCode: " . $TempErrorCode . " -AddCode: " . $TempAddCode . " -Category: " . $TempClassCode;

				### Create Service with Increment and leading 0
				my $TempServiceString = $Service . "/Error-" . (sprintf("%02d", $TempServiceIndex));

				### Open readings bulk update
				readingsBeginUpdate($hash);
				
				### Write Reading
				readingsBulkUpdate( $hash, $TempServiceString, $TempErrorMessage, 1);

				### Obtain detailed Information about error codes
				my $classdescription;
				my $classworkaround;
				my $reason;
				my $checklist;
				my $correction;
				($classdescription, $classworkaround, $reason, $checklist, $correction) = km200_GetErrorMessage($TempErrorCode, $TempAddCode, $TempClassCode, $hash);

				### Write Reading
				readingsBulkUpdate( $hash, $TempServiceString . "_Class-Description", $classdescription, 1);
				readingsBulkUpdate( $hash, $TempServiceString . "_Class-Workaround",  $classworkaround,  1);
				readingsBulkUpdate( $hash, $TempServiceString . "_Error-Reason",      $reason,           1);
				readingsBulkUpdate( $hash, $TempServiceString . "_Error-Checklist",   $checklist,        1);
				readingsBulkUpdate( $hash, $TempServiceString . "_Error-Correction",  $correction,       1);

				### Close readings bulk update
				readingsEndUpdate($hash, 1);
			}
		}
		### Check whether the type is a systeminfo
		elsif ($json -> {type} eq "systeminfo") {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};
			my @JsonValues     = $json->{values};

			### Log entries for debugging purposes
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - value found for      : " .$Service;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - id                   : " .$JsonId;
			Log3 $name, 5, $name. " : km200_ParseHttpResponseInit - type                 : " .$JsonType;

			### Initialise ArrayCounter 
			my $ArrayCounter = 0;
			
			foreach my $ArrayItem (@{ $json->{values} }) {
				### Incrementation of ArrayCounter
				$ArrayCounter++;


				### Get array from scalar
				my %ArrayHash = %{$ArrayItem};
				
				while( my( $SystemInfoHashKey, $SystemInfoHashValue ) = each %ArrayHash )
				{
					### Create new Service and write reading for fhem
					my $TempJsonId = $JsonId . "/" . sprintf ('%02d', $ArrayCounter) . "/" . $SystemInfoHashKey;
					readingsSingleUpdate( $hash, $TempJsonId, $SystemInfoHashValue, 1);
				}
			}
		}

		## Check whether the type is a recording
		elsif (($json -> {type} eq "yRecording" ) and ($json -> {interval} ne "")) {
			my $JsonId         = $json->{id};
			my $JsonType       = $json->{type};

			## If the response contains data based on the interval
			if ($json -> {interval} ne "") {
				## Log entries for debugging purposes
				Log3 $name, 5, $name. " : km200_ParseHttpResponseDyn________________________________________________________________________";
				Log3 $name, 5, $name. " : km200_ParseHttpResponseDyn - yRec value found for : " .$Service;
				Log3 $name, 5, $name. " : km200_ParseHttpResponseDyn - id                   : " .$JsonId;
				Log3 $name, 5, $name. " : km200_ParseHttpResponseDyn - type                 : " .$JsonType;

				my $interval = $json->{interval};
				my $sampleRate = $json->{sampleRate};
				
				## Log recording values for debugging
				Log3 $name, 5, $name. " : km200_ParseHttpResponseDyn                        : " . $JsonId . ' : ' . $interval . ' : ' . $sampleRate;
				Log3 $name, 5, $name. " : km200_ParseHttpResponseDyn  - recording values    : " . Dumper(@{$json->{recording}});
				
				my $energySum = 0;
				foreach my $item (@{ $json->{recording} })
				{
					if($item->{c} ne 0)
					{
						if($sampleRate eq "P1H")
						{
							$energySum = $energySum + $item->{y} / $item->{c}
						}
						elsif($sampleRate eq "P1D")
						{
							$energySum = $energySum + $item->{y} / ($item->{c} / 24)
						}
						elsif($sampleRate eq "P31D")
						{
							### don't know which values für 'c' should be used
							$energySum = $energySum + $item->{y} / ($item->{c} / 30 / 24)
						}
					}
				}
				
				### write reading for fhem
				my $intervalText = $sampleRate eq "P1H" ? "_Today_kWh" : $sampleRate eq "P1D" ? "_ThisMonth_kWh" : "_ThisYear_kWh";
				my $TempJsonId = '/recordings'.$json->{recordedResource}{id}. $intervalText;
				$energySum = sprintf('%.02f', $energySum);
				readingsSingleUpdate( $hash, $TempJsonId, $energySum, 1);

				### Log recording values for debugging
				Log3 $name, 5, $name. " : km200_ParseHttpRecordings energySum: ".$JsonId.':'.$energySum;
			}
		}		
		
		### Check whether the type is unknown
		else {
			### Log entries for debugging purposes
			Log3 $name, 4, $name. " : km200_parseHttpResponseDyn  - type unknown for : " .$Service;
		}
	}
	else {
		Log3 $name, 5, $name. " : km200_parseHttpResponseDyn  - Data not available on km200 for http://" . $param->{url};
	}

	Log3 $name, 5, $name . "______________________________________________________________________________________________________________________";
	### Clear up temporary variables
	$hash->{temp}{decodedcontent} = "";	
	$hash->{temp}{service}        = "";
	### Clear up temporary variables

	### If list is not complete yet
	if ($ServiceCounterDyn < ($NumberDynServices-1)) {
		++$ServiceCounterDyn;
		$hash->{temp}{ServiceCounterDyn} = $ServiceCounterDyn;
		km200_GetDynService($hash);
	}
	### If list is complete
	else {
		$hash->{STATE}                   = "Standby";
		$hash->{temp}{ServiceCounterDyn} = 0;
		
		###START###### Re-Start the timer #####################################START####
		InternalTimer(gettimeofday()+$hash->{INTERVALDYNVAL}, "km200_GetDynService", $hash, 1);
		####END####### Re-Start the timer ######################################END#####
		
		### Update fullResponse Reading
		readingsSingleUpdate( $hash, "fullResponse", "OK", 1);
		
		$hash->{status}{FlagDynRequest}  = false;
		
		Log3 $name, 5, $name . "______________________________________________________________________________________________________________________";
	}
	return undef;
}
####END####### Subroutine to download complete dynamic data set from gateway ###################################END#####


###START###### Load database for errorcodes in hash ###########################################################START####
sub km200_GetErrorMessage($$$$) {
	my ($ErrorCode, $SubCode, $Class, $hash) = @_;
	my $name                         =   $hash ->{NAME};
	my %ErrorCodeList;
	my %ClassCodeList;

	### Define database for error codes as hash
	$ErrorCodeList{"0001"} = {errorcode => "--",                                     subcode => "200",                                                class => "8",           reason => "W&auml;rmeerzeuger im Heizbetrieb",                                                                                                                                                                                                                                                                                                                checklist => "W&auml;rmeerzeuger ist im Heizbetrieb",                                                                                                                                                                                                                                                                                                                                                                                                                                                               correction => "Betriebsmeldung, keine St&ouml;rung"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0002"} = {errorcode => "--",                                     subcode => "201",                                                class => "8",           reason => "W&auml;rmeerzeuger im WW-Betrieb",                                                                                                                                                                                                                                                                                                                 checklist => "W&auml;rmeerzeuger ist im Warmwasserbetrieb",                                                                                                                                                                                                                                                                                                                                                                                                                                                         correction => "Betriebsmeldung, keine St&ouml;rung"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0003"} = {errorcode => "--",                                     subcode => "202",                                                class => "8",           reason => "Ger&auml;t im Schaltoptimierungsprogramm",                                                                                                                                                                                                                                                                                                         checklist => "Einstellungen am  am Basiscontroller/ Steuerger&auml;t &uuml;berpr&uuml;fen.",                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "Einstellungen im Basiscontroller/Steuerger&auml;t auf die Anlagenbedingungen anpassen"                                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0004"} = {errorcode => "--",                                     subcode => "203",                                                class => "8",           reason => "Ger&auml;t in Betriebsbereitschaft, kein W&auml;rmebedarf vorhanden",                                                                                                                                                                                                                                                                              checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Betriebsmeldung, keine St&ouml;rung"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0005"} = {errorcode => "--",                                     subcode => "204",                                                class => "8",           reason => "Aktuelle Heizwassertemperatur des W&auml;rmeerzeugers h&ouml;her als Sollwert",                                                                                                                                                                                                                                                                    checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Betriebsmeldung, keine St&ouml;rung"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0006"} = {errorcode => "--",                                     subcode => "208",                                                class => "8",           reason => "W&auml;rmeanforderung wegen Abgastest",                                                                                                                                                                                                                                                                                                            checklist => "Schornsteinfegerfunktion ist aktiviert",                                                                                                                                                                                                                                                                                                                                                                                                                                                              correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0007"} = {errorcode => "--",                                     subcode => "265",                                                class => "8",           reason => "W&auml;rmebedarf geringer als gelieferte Energie",                                                                                                                                                                                                                                                                                                 checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0008"} = {errorcode => "--",                                     subcode => "268",                                                class => "8",           reason => "Relaistest wurde aktiviert",                                                                                                                                                                                                                                                                                                                       checklist => "Im Servicemen&uuml; wurde der Relaistest aktiviert. Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                             correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0009"} = {errorcode => "--",                                     subcode => "283",                                                class => "8",           reason => "Brenner wird gestartet",                                                                                                                                                                                                                                                                                                                           checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0010"} = {errorcode => "--",                                     subcode => "284",                                                class => "8",           reason => "Gasarmatur/&Ouml;lMagnetventile wird ge&ouml;ffnet; Erste Sicherheitszeit",                                                                                                                                                                                                                                                                        checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Betriebsmeldung, keine St&ouml;rung"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0011"} = {errorcode => "--",                                     subcode => "305",                                                class => "8",           reason => "Kessel kann vor&uuml;bergehend nach Warmwasservorrang nicht starten",                                                                                                                                                                                                                                                                              checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Betriebsmeldung, keine St&ouml;rung"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0012"} = {errorcode => "--",                                     subcode => "309",                                                class => "8",           reason => "Heizund W&auml;rmebetrieb parallel",                                                                                                                                                                                                                                                                                                               checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0013"} = {errorcode => "--",                                     subcode => "357",                                                class => "8",           reason => "Entl&uuml;ftungsprogramm",                                                                                                                                                                                                                                                                                                                         checklist => "Entl&uuml;ftungsprogramm: Das Entl&uuml;ftungsprogramm des W&auml;rmetauchers l&auml;uft. Hierbei taktet die Pumpe und das 3Wegeventil",                                                                                                                                                                                                                                                                                                                                                              correction => "Betriebsmeldung, keine St&ouml;rung"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0014"} = {errorcode => "--",                                     subcode => "358",                                                class => "8",           reason => "Blockierschutz aktiv",                                                                                                                                                                                                                                                                                                                             checklist => "Blockierschutz f&uuml;r die Heizungspumpe und 3-Wegeventil aktiv",                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "Betriebsmeldung, keine St&ouml;rung"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0015"} = {errorcode => "--",                                     subcode => "370",                                                class => "4",           reason => "Zu geringer K&auml;ltemitteldruck",                                                                                                                                                                                                                                                                                                                checklist => "Verdampfer vereist&bull;K&auml;ltemittelkreis undicht",                                                                                                                                                                                                                                                                                                                                                                                                                                               correction => "Eisbildung am Verdampfer entfernen&bull;W&auml;rmepumpeneinheit austauschen"                                                                                                                                                                                                                                                                                                                     };
	$ErrorCodeList{"0016"} = {errorcode => "--",                                     subcode => "471",                                                class => "8",           reason => "Vorlaufphase Pumpe in der Inneneinheit",                                                                                                                                                                                                                                                                                                           checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0017"} = {errorcode => "--",                                     subcode => "472",                                                class => "8",           reason => "Vorheizphase der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                 checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0018"} = {errorcode => "--",                                     subcode => "473",                                                class => "8",           reason => "W&auml;rmepumpe in Betrieb",                                                                                                                                                                                                                                                                                                                       checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0019"} = {errorcode => "--",                                     subcode => "474",                                                class => "8",           reason => "Pumpe Hybrid-Manager: Nachlaufphase",                                                                                                                                                                                                                                                                                                              checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0020"} = {errorcode => "--",                                     subcode => "475",                                                class => "8",           reason => "W&auml;rmepumpe im Abtaubetrieb",                                                                                                                                                                                                                                                                                                                  checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0021"} = {errorcode => "--",                                     subcode => "476",                                                class => "8",           reason => "Interner Status",                                                                                                                                                                                                                                                                                                                                  checklist => "Betriebscode, keine zSt&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0022"} = {errorcode => "--",                                     subcode => "477",                                                class => "8",           reason => "Nur EMS-W&auml;rmeerzeuger in Betrieb",                                                                                                                                                                                                                                                                                                            checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0023"} = {errorcode => "--",                                     subcode => "478",                                                class => "8",           reason => "W&auml;rmepumpe blockiert: mehr als 4 Starts pro Stunde",                                                                                                                                                                                                                                                                                          checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0024"} = {errorcode => "--",                                     subcode => "480",                                                class => "8",           reason => "Differenztemperatur au&szlig;erhalb des zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                                                                 checklist => "Indikator f&uuml;r geringen Durchfluss im System. Filter auf Verstopfung pr&uuml;fen.",                                                                                                                                                                                                                                                                                                                                                                                                               correction => "Filter ggf. reinigen"                                                                                                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0025"} = {errorcode => "--",                                     subcode => "481",                                                class => "8",           reason => "W&auml;rmepumpe l&auml;uft bei blockiertem Heizkessel",                                                                                                                                                                                                                                                                                            checklist => "Betriebscode, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0026"} = {errorcode => "--",                                     subcode => "482",                                                class => "8",           reason => "W&auml;rmepumpe arbeitet im Servicemodus",                                                                                                                                                                                                                                                                                                         checklist => "Betriebscode, keine St&ouml;rung Die W&auml;rmepumpe arbeitet im Servicemodus (100 % Leistung)",                                                                                                                                                                                                                                                                                                                                                                                                      correction => "Nach erfolgten Servicearbeiten auf Normalbetrieb umschalten. Der Servicemodus endet automatisch nach 15 Minuten."                                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0027"} = {errorcode => "--",                                     subcode => "2908",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik / Basiscontroller",                                                                                                                                                                                                                                                                                       checklist => "Neustart Ger&auml;t, um zu sehen, ob das Problem erneut auftritt.",                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Bleibt die St&ouml;rung nach Reset erhalten, ist der Feuerungsautomat oder Fremdbrennermodul defekt und muss ausgetauscht werden"                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0028"} = {errorcode => "--",                                     subcode => "2909",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik / Basiscontroller",                                                                                                                                                                                                                                                                                       checklist => "Restart-Appliance, um zu sehen, wenn das Problem erneut auftritt.",                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Bleibt die St&ouml;rung nach Reset erhalten, ist der Feuerungsautomat oder Fremdbrennermodul defekt und muss ausgetauscht werden"                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0029"} = {errorcode => "--",                                     subcode => "2910",                                               class => "8",           reason => "Fehler im Abgassystem",                                                                                                                                                                                                                                                                                                                            checklist => "&Uuml;berpr&uuml;fen, ob ein Abgassystem angeschlossen ist&bull;Abgassystem auf freien Querschnitt &uuml;berpr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                             correction => "Abgassystem montieren&bull;Ablagerungen im Abgasssystem entfernen"                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0030"} = {errorcode => "--",                                     subcode => "2911",                                               class => "8",           reason => "Kalibrierung fehlgeschlagen",                                                                                                                                                                                                                                                                                                                      checklist => "Verkabelung pr&uuml;fen&bull;Ionistationselektrode pr&uuml;fen&bull;Gas-Ventil pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                        correction => "Fehlerhafte Komponente ersetzen&bull;-"                                                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0031"} = {errorcode => "--",                                     subcode => "2912",                                               class => "8",           reason => "Kein Flammensignal w&auml;hrend der Kalibrierung",                                                                                                                                                                                                                                                                                                 checklist => "Verkabelung pr&uuml;fen&bull;Ionistationselektrode pr&uuml;fen&bull;Gas-Ventil pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                        correction => "Fehlerhafte Komponente ersetzen&bull;-"                                                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0032"} = {errorcode => "--",                                     subcode => "2913",                                               class => "8",           reason => "Flammensignal zu niedrig in der Kalibrierung",                                                                                                                                                                                                                                                                                                     checklist => "Verkabelung pr&uuml;fen&bull;Ionistationselektrode pr&uuml;fen&bull;Gas-Ventil pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                        correction => "Ionistationselektrode ersetzen&bull;-"                                                                                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0033"} = {errorcode => "--",                                     subcode => "2914",                                               class => "4",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Reset am W&auml;rmeerzeuger durchf&uuml;hren.",                                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Bleibt die St&ouml;rung nach Reset erhalten, ist die Ger&auml;testeuerung oder das Brennermodul defekt und muss ausgetauscht werden"                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0034"} = {errorcode => "--",                                     subcode => "2915",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Verkabelung pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "1. Reset am Ger&auml;t ausf&uuml;hren; 2. W&auml;rmeanforderung abwarten/erzeugen; 3. Ende der W&auml;rmeanforderung abwarten/ W&auml;rmeanforderung beenden.; Wenn der Fehler erneut erscheint, muss der Feuerungsautomat getauscht werden&bull;-"                                                                                                                                              };
	$ErrorCodeList{"0035"} = {errorcode => "--",                                     subcode => "2916",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Verkabelung pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "1. Reset am Ger&auml;t ausf&uuml;hren; 2. W&auml;rmeanforderung abwarten/erzeugen; 3. Ende der W&auml;rmeanforderung abwarten/ W&auml;rmeanforderung beenden.; Wenn der Fehler erneut erscheint, muss der Feuerungsautomat getauscht werden&bull;-"                                                                                                                                              };
	$ErrorCodeList{"0036"} = {errorcode => "--",                                     subcode => "2917",                                               class => "8",           reason => "Kein Flammensignal w&auml;hrend der &Uuml;berpr&uuml;fung der Verbrennungsregelung",                                                                                                                                                                                                                                                               checklist => "Gas-Ventil pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                                                              correction => "1. Ger&auml;t auslaufen lassen; 2. W&auml;rmeanforderung erzeugen/abwarten; 3. 5 Minuten abwarten. Wenn der Fehler erneut auftritt, dann Ger&auml;t resetten. Bei erneutem Auftreten des Fehlers Feuerungsautomat tauschen"                                                                                                                                                                      };
	$ErrorCodeList{"0037"} = {errorcode => "--",                                     subcode => "2918",                                               class => "8",           reason => "St&ouml;rung in der Verbrennungsregelung",                                                                                                                                                                                                                                                                                                         checklist => "Siphon pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Siphon und Kondensatabflu&szlig; reinigen"                                                                                                                                                                                                                                                                                                                                                       };
	$ErrorCodeList{"0038"} = {errorcode => "--",                                     subcode => "2919",                                               class => "8",           reason => "St&ouml;rung in der Verbrennungsregelung",                                                                                                                                                                                                                                                                                                         checklist => "Verkabelung pr&uuml;fen&bull;Ionistationselektrode pr&uuml;fen&bull;Gas-Ventil pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                        correction => "Fehlerhafte Komponente tauschen&bull;-"                                                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0039"} = {errorcode => "--",                                     subcode => "2920",                                               class => "8",           reason => "St&ouml;rung Flammen&uuml;berwachung",                                                                                                                                                                                                                                                                                                             checklist => "Verkabelung pr&uuml;fen&bull;Brennerd&uuml;se pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                         correction => "Feuerungsautomat austauschen&bull;-"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0040"} = {errorcode => "--",                                     subcode => "2921",                                               class => "4",           reason => "Ger&auml;teelektronik im Testmode",                                                                                                                                                                                                                                                                                                                checklist => "Warten bis Test beendet",                                                                                                                                                                                                                                                                                                                                                                                                                                                                             correction => "Keine Ma&szlig;nahme erforderlich (W&auml;rmeerzeuger startet automatisch)"                                                                                                                                                                                                                                                                                                                      };
	$ErrorCodeList{"0041"} = {errorcode => "--",                                     subcode => "2922",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Verkabelung pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Feuerungsautomat austauschen&bull;-"                                                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0042"} = {errorcode => "--",                                     subcode => "2923",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Reset am W&auml;rmeerzeuger durchf&uuml;hren.",                                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Ger&auml;teelektronik austauschen"                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0043"} = {errorcode => "--",                                     subcode => "2924",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Reset am W&auml;rmeerzeuger durchf&uuml;hren.",                                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Ger&auml;teelektronik austauschen"                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0044"} = {errorcode => "--",                                     subcode => "2925",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Verkabelung pr&uuml;fen&bull;Ionistationselektrode pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                    correction => "Ger&auml;teelektronik austauschen&bull;-"                                                                                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0045"} = {errorcode => "--",                                     subcode => "2926",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Verkabelung pr&uuml;fen&bull;Ionistationselektrode pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                    correction => "Ger&auml;teelektronik austauschen&bull;-"                                                                                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0046"} = {errorcode => "--",                                     subcode => "2626",                                               class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik",                                                                                                                                                                                                                                                                                                         checklist => "Verkabelung pr&uuml;fen&bull;Ionistationselektrode pr&uuml;fen&bull;Feuerungsautomat pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                    correction => "Ger&auml;teelektronik austauschen&bull;-"                                                                                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0047"} = {errorcode => "--",                                     subcode => "2927",                                               class => "4",           reason => "Flamme w&auml;hrend Brennerbetrieb ausgefallen",                                                                                                                                                                                                                                                                                                   checklist => "Hauptabsperreinrichtung &uuml;berpr&uuml;fen&bull;Ger&auml;teabsperrhahn &uuml;berpr&uuml;fen&bull;Gasanschlu&szlig;flie&szlig;druck bei Nennw&auml;rmebelastung messen&bull;Ionisationsstrom messen&bull;Ionisationselektrode &uuml;berpr&uuml;fen&bull;Schutzleiteranschlu&szlig; im Steuerger&auml;t &uuml;berpr&uuml;fen&bull;Z&uuml;ndkabel auf Besch&auml;digungen &uuml;berpr&uuml;fen&bull;Anschlusskabel f&uuml;r die Ionisationselektrode auf Besch&auml;digungen &uuml;berpr&uuml;fen&bull;Ohmschen Widerstand der Sicherheitsventile an der Gasarmatur messen&bull;Brennereinstellwert bei Nennw&auml;rmebelastung bzw. eingebaute Brennerd&uuml;sen &uuml;berpr&uuml;fen&bull;Brennereinstellwert bei kleinster Leistung &uuml;berpr&uuml;fen&bull;Abgasanlage &uuml;berpr&uuml;fen&bull;Verbrennungsluftversorgung &uuml;berpr&uuml;fen&bull;W&auml;rmeblock auf abgasseitige Ablagerungen &uuml;berpr&uuml;fen&bull;-",correction => "Hauptabsperreinrichtung &ouml;ffnen&bull;Ger&auml;teabsperrhahn &ouml;ffnen&bull;Ger&auml;t stillegen und Gasleitung &uuml;berpr&uuml;fen&bull;Signalauswertung auf Leiterplatte defekt&bull;Ionisationselektrode austauschen&bull;Schutzleiteranschluss (PE) im Schaltkasten herstellen&bull;Z&uuml;ndkabel austauschen&bull;Anschlusskabel zur Ionisationselektrode austauschen&bull;Gasarmatur austauschen&bull;Brenner korrekt einstellen bzw. Brennerd&uuml;sen austauschen&bull;Brenner bei minimaler Nennbelastung einstellen&bull;Abgasanlage umbauen&bull;Verbrennungsluftverbund zu klein bzw. zu geringe Gr&ouml;&szlig;e der L&uuml;ftungs&ouml;ffnung&bull;W&auml;rmeblock abgasseitig reinigen&bull;Steuerger&auml;t/Feuerungsautomat austauschen"};
	$ErrorCodeList{"0048"} = {errorcode => "--",                                     subcode => "2928",                                               class => "8",           reason => "Interner Fehler",                                                                                                                                                                                                                                                                                                                                  checklist => "Ger&auml;t neustarten, pr&uuml;fen ob Fehler wieder erscheint",                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Steuerger&auml;t / Feuerungsautomat austauschen, falls das Problem erneut auftritt"                                                                                                                                                                                                                                                                                                              };
	$ErrorCodeList{"0049"} = {errorcode => "--",                                     subcode => "2929",                                               class => "8",           reason => "Siphon F&uuml;llprogramm",                                                                                                                                                                                                                                                                                                                         checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0050"} = {errorcode => "0",                                      subcode => "271; 471",                                           class => "BC",          reason => "Vorlaufphase Pumpe in der Inneneinheit",                                                                                                                                                                                                                                                                                                           checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0051"} = {errorcode => "0",                                      subcode => "370",                                                class => "8",           reason => "Zu geringer K&auml;ltemitteldruck",                                                                                                                                                                                                                                                                                                                checklist => "Verdampfer auf Vereisung &uuml;berpr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                correction => "Eisbildung am Verdampfer entfernen&bull;W&auml;rmepumpeneinheit austauschen"                                                                                                                                                                                                                                                                                                                     };
	$ErrorCodeList{"0052"} = {errorcode => "0",                                      subcode => "2516",                                               class => "BC",          reason => "Interner Status",                                                                                                                                                                                                                                                                                                                                  checklist => "Interner Status",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0053"} = {errorcode => "1",                                      subcode => "272; 472",                                           class => "BC",          reason => "Vorheizphase der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                 checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0054"} = {errorcode => "1",                                      subcode => "470",                                                class => "10",          reason => "Keine Kommunikation mit dem Systemregler",                                                                                                                                                                                                                                                                                                         checklist => "Kontakte, Kabel und Verbindungen zwischen Hybridmodul und Systemregler pr&uuml;fen&bull;Systemregler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                    correction => "Besch&auml;digte Leitungen und Kontakte austauschen&bull;Ggf. ersetzen"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0055"} = {errorcode => "2",                                      subcode => "273; 473",                                           class => "BC",          reason => "W&auml;rmepumpe in Betrieb",                                                                                                                                                                                                                                                                                                                       checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0056"} = {errorcode => "3",                                      subcode => "274; 474",                                           class => "BC",          reason => "Pumpe Hybrid-Manager: Nachlaufphase",                                                                                                                                                                                                                                                                                                              checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0057"} = {errorcode => "4",                                      subcode => "275; 475",                                           class => "BC",          reason => "W&auml;rmepumpe im Abtaubetrieb",                                                                                                                                                                                                                                                                                                                  checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0058"} = {errorcode => "5",                                      subcode => "276; 476",                                           class => "BC",          reason => "W&auml;rmepumpe im St&ouml;rungsmodus",                                                                                                                                                                                                                                                                                                            checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0059"} = {errorcode => "6",                                      subcode => "277; 477",                                           class => "BC",          reason => "Nur EMS-W&auml;rmeerzeuger in Betrieb",                                                                                                                                                                                                                                                                                                            checklist => "Pr&uuml;fen, ob ein Durchfluss gegeben ist. Ursache: Filter in der Logatherm  WHM (Inneneinheit Logatherm WPL); verschmutzt oder Absperrventile geschlossen&bull;Funktion und Anschluss des Str&ouml;mungsschalters &uuml;berpr&uuml;fen&bull;Au&szlig;eneinheit mit dem Servicetool pr&uuml;fen",                                                                                                                                                                                                    correction => "Filter reinigen und Absperrventile &ouml;ffnen&bull;Str&ouml;mungsschalter anschlie&szlig;en oder ggf. Str&ouml;mungschalter austauschen&bull;St&ouml;rung der Au&szlig;eneinheit beheben"                                                                                                                                                                                                       };
	$ErrorCodeList{"0060"} = {errorcode => "7",                                      subcode => "278; 478",                                           class => "BC",          reason => "W&auml;rmepumpe blockiert: mehr als 4 Starts pro Stunde",                                                                                                                                                                                                                                                                                          checklist => "W&auml;rmepumpe blockiert w&auml;hrend der Startversuche; dieser Zustand tritt auf, wenn das System mehr als 4-mal in einem 60-min&uuml;tigen Zeitraum startet; die Z&auml;hlung wird nach 60 Minuten zur&uuml;ckgesetzt; dieser Zustand tritt auch auf, um sicherzustellen, dass die Luft-Wasser-W&auml;rmepumpe f&uuml;r mindestens 20 Minuten ausgeschaltet ist",                                                                                                                                  correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0061"} = {errorcode => "8",                                      subcode => "479",                                                class => "BC",          reason => "St&ouml;rung Str&ouml;mungsschalter W&auml;rmepumpe w&auml;hrend Selbsttest",                                                                                                                                                                                                                                                                      checklist => "Str&ouml;mungsschalter pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Ggf. Str&ouml;mungsschalter austauschen"                                                                                                                                                                                                                                                                                                                                                         };
	$ErrorCodeList{"0062"} = {errorcode => "9",                                      subcode => "280; 480",                                           class => "BC",          reason => "Differenztemperatur au&szlig;erhalb des zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                                                                 checklist => "Indikator f&uuml;r geringen Durchfluss im System; Filter auf Verstopfung pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                correction => "Ggf. reinigen"                                                                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0063"} = {errorcode => "0A",                                     subcode => "202",                                                class => "BC",          reason => "Ger&auml;t im Schaltoptimierungsprogramm",                                                                                                                                                                                                                                                                                                         checklist => "Leistungseinstellung am Basiscontroller pr&uuml;fen&bull;Regelungseinstellungen im Regelger&auml;t Bedieneinheit pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                        correction => "Kesselleistung auf den erforderlichen W&auml;rmebedarf des Geb&auml;udes abstimmen&bull;Regelungseinstellungen an die Anlagenbedingungen anpassen"                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0064"} = {errorcode => "0A",                                     subcode => "305",                                                class => "BC",          reason => "Kessel kann vor&uuml;bergehend nach Warmwasser-Vorrang nicht starten",                                                                                                                                                                                                                                                                             checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0065"} = {errorcode => "0A",                                     subcode => "333",                                                class => "10",          reason => "Kessel vor&uuml;bergehend wegen zu geringen Wasserdrucks abgeschaltet",                                                                                                                                                                                                                                                                            checklist => "Kessel hat wegen kurzzeitig zu geringen Wasserdrucks abgeschaltet; Kessel l&auml;uft selbstst&auml;ndig nach kurzer Zeit wieder an",                                                                                                                                                                                                                                                                                                                                                                  correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0066"} = {errorcode => "0A",                                     subcode => "2505",                                               class => "BC",          reason => "W&auml;rmeanforderung blockiert wegen Antipendel",                                                                                                                                                                                                                                                                                                 checklist => "W&auml;rmeanforderung blockiert durch Antipendelzeit",                                                                                                                                                                                                                                                                                                                                                                                                                                                correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0067"} = {errorcode => "0C",                                     subcode => "283",                                                class => "BC",          reason => "Brenner wird gestartet",                                                                                                                                                                                                                                                                                                                           checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0068"} = {errorcode => "0C",                                     subcode => "2517",                                               class => "BC",          reason => "Vorbel&uuml;ftung",                                                                                                                                                                                                                                                                                                                                checklist => "Vorbel&uuml;ftung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0069"} = {errorcode => "0C",                                     subcode => "2518",                                               class => "BC",          reason => "Warten Mischraumtemperatur",                                                                                                                                                                                                                                                                                                                       checklist => "Warten, dass Mischraumtemperatur erreicht wird",                                                                                                                                                                                                                                                                                                                                                                                                                                                      correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0070"} = {errorcode => "0C",                                     subcode => "2519",                                               class => "BC",          reason => "Flamme bilden",                                                                                                                                                                                                                                                                                                                                    checklist => "Flamme bilden",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0071"} = {errorcode => "0d",                                     subcode => "2524",                                               class => "BC",          reason => "Nachfackelkontrolle aus Startphase",                                                                                                                                                                                                                                                                                                               checklist => "Nachfackelkontrolle aus Startphase",                                                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0072"} = {errorcode => "0d",                                     subcode => "2525",                                               class => "BC",          reason => "Nachfackelkontrolle aus Station&auml;rbetrieb",                                                                                                                                                                                                                                                                                                    checklist => "Nachfackelkontrolle aus Station&auml;rbetrieb",                                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0073"} = {errorcode => "0d",                                     subcode => "2526",                                               class => "BC",          reason => "Nachbel&uuml;ftung aus Startphase",                                                                                                                                                                                                                                                                                                                checklist => "Nachbel&uuml;ftung aus Startphase",                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0074"} = {errorcode => "0d",                                     subcode => "2527",                                               class => "BC",          reason => "Nachbel&uuml;ftung aus Station&auml;rbetrieb",                                                                                                                                                                                                                                                                                                     checklist => "Nachbel&uuml;ftung aus Station&auml;rbetrieb",                                                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0075"} = {errorcode => "0d",                                     subcode => "2528",                                               class => "BC",          reason => "Gebl&auml;se aus",                                                                                                                                                                                                                                                                                                                                 checklist => "Gebl&auml;se aus",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0076"} = {errorcode => "0d",                                     subcode => "2529",                                               class => "BC",          reason => "Sicherheitsrelais aus",                                                                                                                                                                                                                                                                                                                            checklist => "Sicherheitsrelais aus",                                                                                                                                                                                                                                                                                                                                                                                                                                                                               correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0077"} = {errorcode => "0E",                                     subcode => "265",                                                class => "BC",          reason => "W&auml;rmeerzeuger in Betriebsbereitschaft und W&auml;rmebedarf vorhanden, es wird jedoch zu viel Energie geliefert",                                                                                                                                                                                                                              checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0078"} = {errorcode => "0E",                                     subcode => "2512",                                               class => "BC",          reason => "W&auml;rmeanforderung blockiert aufgrund einer Leistungsbegrenzung",                                                                                                                                                                                                                                                                               checklist => "W&auml;rmeanforderung blockiert aufgrund einer Leistungsbegrenzung",                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0079"} = {errorcode => "0F",                                     subcode => "-",                                                  class => "",            reason => "Gasarmatur/&Ouml;lMagnetventil wird ge&ouml;ffnet",                                                                                                                                                                                                                                                                                                checklist => "Vorlauftemperatur mit Basiscontroller kontrollieren. R&uuml;cklauftemperaturf&uuml;hler mit Bedieneinheit oder Service Key kontrollieren&bull;Widerstandswert des Kesself&uuml;hlers (Sicherheitstemperaturbegrenzer) messen und mit der Kennlinie vergleichen&bull;Pr&uuml;fen, ob ein Gussglied durch Schmutz verstopft ist",                                                                                                                                                                       correction => "Einstellung der Kesselkreispumpe anpassen&bull;Oberf&auml;chentemperatur des mit dem Sicherheitstemperaturf&uuml;hler best&uuml;ckten Gussglieds mit Temperaturmessger&auml;t pr&uuml;fen&bull;Gussglied reinigen"                                                                                                                                                                               };
	$ErrorCodeList{"0080"} = {errorcode => "0F",                                     subcode => "2513",                                               class => "BC",          reason => "W&auml;rmeanforderung blockiert aufgrund von Temperaturdifferenzen",                                                                                                                                                                                                                                                                               checklist => "W&auml;rmeanforderung blockiert aufgrund; zu hoher Temperaturdifferenzen zwischen Vorlauf und R&uuml;cklauf",                                                                                                                                                                                                                                                                                                                                                                                         correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0081"} = {errorcode => "0H",                                     subcode => "203",                                                class => "BC",          reason => "Ger&auml;t in Betriebsbereitschaft, kein W&auml;rmebedarf vorhanden",                                                                                                                                                                                                                                                                              checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0082"} = {errorcode => "0H",                                     subcode => "2500",                                               class => "BC",          reason => "Keine W&auml;rmeanforderung",                                                                                                                                                                                                                                                                                                                      checklist => "Keine W&auml;rmeanforderung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                         correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0083"} = {errorcode => "0H",                                     subcode => "2530",                                               class => "BC",          reason => "Interner Status",                                                                                                                                                                                                                                                                                                                                  checklist => "Interner Status",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0084"} = {errorcode => "0L",                                     subcode => "284",                                                class => "BC",          reason => "Gasarmatur/&Ouml;l-Magnetventile werden ge&ouml;ffnet Erste Sicherheitszeit",                                                                                                                                                                                                                                                                      checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0085"} = {errorcode => "0L",                                     subcode => "2520",                                               class => "BC",          reason => "Flamme stabilisieren",                                                                                                                                                                                                                                                                                                                             checklist => "Flamme stabilisieren",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0086"} = {errorcode => "0L",                                     subcode => "2521",                                               class => "BC",          reason => "Stabilisieren W&auml;rmetauscher",                                                                                                                                                                                                                                                                                                                 checklist => "Stabilisieren W&auml;rmetauscher",                                                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0087"} = {errorcode => "0L",                                     subcode => "2522",                                               class => "BC",          reason => "Warten Aufheizung W&auml;rmetauscher",                                                                                                                                                                                                                                                                                                             checklist => "Warten Aufheizung W&auml;rmetauscher",                                                                                                                                                                                                                                                                                                                                                                                                                                                                correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0088"} = {errorcode => "0L",                                     subcode => "2523",                                               class => "BC",          reason => "Umschaltphase (von Start auf station&auml;r)",                                                                                                                                                                                                                                                                                                     checklist => "Umschaltphase (von Start auf station&auml;r)",                                                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0089"} = {errorcode => "0P",                                     subcode => "205",                                                class => "BC",          reason => "W&auml;rmeerzeuger wartet auf Luftstr&ouml;mung",                                                                                                                                                                                                                                                                                                  checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0090"} = {errorcode => "0U",                                     subcode => "270",                                                class => "BC",          reason => "W&auml;rmeerzeuger wird hochgefahren",                                                                                                                                                                                                                                                                                                             checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0091"} = {errorcode => "0Y",                                     subcode => "204",                                                class => "BC",          reason => "Aktuelle Heizwassertemperatur des W&auml;rmeerzeugers h&ouml;her als der Sollwert",                                                                                                                                                                                                                                                                checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0092"} = {errorcode => "0Y",                                     subcode => "276",                                                class => "10",          reason => "Temperatur am Vorlauftemperaturf&uuml;hler zu hoch",                                                                                                                                                                                                                                                                                               checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Vorlauftemperaturf&uuml;hler pr&uuml;fen&bull;Pumpenleistung oder Pumpenkennfeld pr&uuml;fen",                                                                                                                                                                                            correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck,; Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Vorlauftemperaturf&uuml;hler austauschen&bull;Pumpenleistung oder Pumpenkennfeld korrekt einstellen und auf maximale Leistung anpassen"                                                 };
	$ErrorCodeList{"0093"} = {errorcode => "0Y",                                     subcode => "277",                                                class => "10",          reason => "Temperatur am Sicherheitstemperaturf&uuml;hler zu hoch",                                                                                                                                                                                                                                                                                           checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Sicherheitstemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                            correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck, Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Sicherheitstemperaturf&uuml;hler austauschen"                                                                                                                                            };
	$ErrorCodeList{"0094"} = {errorcode => "0Y",                                     subcode => "285",                                                class => "10",          reason => "Temperatur am R&uuml;cklauftemperaturf&uuml;hler zu hoch",                                                                                                                                                                                                                                                                                         checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;R&uuml;cklauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                          correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. R&uuml;cklauftemperaturf&uuml;hler austauschen"                                                                                                                                           };
	$ErrorCodeList{"0095"} = {errorcode => "0Y",                                     subcode => "359",                                                class => "10",          reason => "Temperatur am Warmwasser-Temperaturf&uuml;hler zu hoch",                                                                                                                                                                                                                                                                                           checklist => "Korrekte Einbaulage des Temperaturf&uuml;hlers pr&uuml;fen&bull;Temperaturf&uuml;hler und Anschlusskabel auf Unterbrechung oder Kurzschluss pr&uuml;fen&bull;Kesselidentifikationsmodul richtig aufstecken",                                                                                                                                                                                                                                                                                          correction => "Temperaturf&uuml;hler korrekt einbauen&bull;Ggf. austauschen&bull;Ggf. austauschen"                                                                                                                                                                                                                                                                                                              };
	$ErrorCodeList{"0096"} = {errorcode => "0Y",                                     subcode => "2511",                                               class => "BC",          reason => "W&auml;rmeanforderung blockiert, weil GPA nicht kalibriert",                                                                                                                                                                                                                                                                                       checklist => "W&auml;rmeanforderung blockiert, weil Luftklappenstellmotor (GPA) nicht kalibriert",                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0097"} = {errorcode => "0Y",                                     subcode => "2515",                                               class => "BC",          reason => "W&auml;rmeanforderung blockiert, weil Kessel warm genug",                                                                                                                                                                                                                                                                                          checklist => "W&auml;rmeanforderung blockiert, weil Kessel warm genug",                                                                                                                                                                                                                                                                                                                                                                                                                                             correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0098"} = {errorcode => "0Y",                                     subcode => "2531",                                               class => "BC",          reason => "W&auml;rmeanforderung blockiert weil Mischraum zu kalt ist",                                                                                                                                                                                                                                                                                       checklist => "W&auml;rmeanforderung blockiert weil Mischraum zu kalt ist.",                                                                                                                                                                                                                                                                                                                                                                                                                                         correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0099"} = {errorcode => "10",                                     subcode => "481",                                                class => "BC",          reason => "W&auml;rmepumpe l&auml;uft; bei blockiertem Heizkessel",                                                                                                                                                                                                                                                                                           checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0100"} = {errorcode => "11",                                     subcode => "482",                                                class => "BC",          reason => "W&auml;rmepumpe arbeitet im Servicemodus",                                                                                                                                                                                                                                                                                                         checklist => "Die W&auml;rmepumpe arbeitet im Servicemodus (100 % Leistung)",                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Nach erfolgten Servicearbeiten auf Normalbetrieb umschalten; der Servicemodus endet automatisch nach 15 min"                                                                                                                                                                                                                                                                                     };
	$ErrorCodeList{"0101"} = {errorcode => "1C",                                     subcode => "210",                                                class => "4",           reason => "Abgasthermostat hat angesprochen",                                                                                                                                                                                                                                                                                                                 checklist => "Abgastemperaturf&uuml;hler unter 105°C abk&uuml;hlen lassen und anschlie&szlig;end auf Durchgang pr&uuml;fen.&bull;W&auml;rmetauscher und Brenner auf Verschmutzung pr&uuml;fen",                                                                                                                                                                                                                                                                                                                    correction => "Ist kein Durchgang vorhanden, Abgastemperaturf&uuml;hler austauschen&bull;W&auml;rmetauscher und Brenner entsprechend Montageund Wartungsanleitung reinigen"                                                                                                                                                                                                                                     };
	$ErrorCodeList{"0102"} = {errorcode => "1C",                                     subcode => "526",                                                class => "4",           reason => "F&uuml;hlerdifferenz Abgastemperaturf&uuml;hler zu gro&szlig;",                                                                                                                                                                                                                                                                                    checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;-&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder ggf. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen&bull;Bei Abweichungen Ger&auml;teelektronik SAFe austauschen"                                                                                                   };
	$ErrorCodeList{"0103"} = {errorcode => "1F",                                     subcode => "525",                                                class => "4",           reason => "Abgastemperatur zu hoch",                                                                                                                                                                                                                                                                                                                          checklist => "Im Men&uuml; „Service“ der Bedieneinheit den Wert der Abgastemperatur pr&uuml;fen und mit der wirklichen Abgastemperatur vergleichen&bull;Wenn Abweichungen bestehen, dann muss der Abgastemperaturf&uuml;hler mit einem Ohmmeter gemessen und nach Tabelle gepr&uuml;ft werden&bull;Kessel auf Verschmutzung pr&uuml;fen&bull;Die Position des Abgastemperaturf&uuml;hlers pr&uuml;fen&bull;Pr&uuml;fen, ob die Heizgaslenkplatten vorhanden oder eventuell besch&auml;digt sind",               correction => "-&bull;Bei Abweichungen der Messwerte den F&uuml;hler austauschen&bull;Bei verschmutztem Kessel diesen reinigen&bull;Wenn der Abgastemperaturf&uuml;hler falsch positioniert ist, diesen an der richtigen Stelle montieren&bull;Heizgaslenkplatten ersetzen bzw. vervollst&auml;ndigen"                                                                                                          };
	$ErrorCodeList{"0104"} = {errorcode => "1H",                                     subcode => "530",                                                class => "10",          reason => "Abgastemperatur zu hoch",                                                                                                                                                                                                                                                                                                                          checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;-&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder ggf. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen&bull;Bei Abweichungen Ger&auml;teelektronik SAFe austauschen"                                                                                                   };
	$ErrorCodeList{"0105"} = {errorcode => "1H",                                     subcode => "562",                                                class => "10",          reason => "Abgasaustrittssicherung zu hohe Temperatur",                                                                                                                                                                                                                                                                                                       checklist => "Zugbedarf im Abgasrohr pr&uuml;fen&bull;Schornsteindimensionierung pr&uuml;fen&bull;Abgas&uuml;berwachungseinrichtung defekt",                                                                                                                                                                                                                                                                                                                                                                        correction => "Abgasweg frei machen&bull;-&bull;Abgas&uuml;berwachungseinrichtung pr&uuml;fen bzw. austauschen"                                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0106"} = {errorcode => "1H",                                     subcode => "563",                                                class => "4",           reason => "Zu h&auml;ufiger Abgasaustritt",                                                                                                                                                                                                                                                                                                                   checklist => "Zugbedarf im Abgasrohr pr&uuml;fen&bull;Schornsteindimensionierung pr&uuml;fen&bull;Abgas&uuml;berwachungseinrichtung defekt",                                                                                                                                                                                                                                                                                                                                                                        correction => "Abgasweg frei machen&bull;-&bull;Abgas&uuml;berwachungseinrichtung pr&uuml;fen bzw. austauschen"                                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0107"} = {errorcode => "1L",                                     subcode => "211",                                                class => "10",          reason => "Installation Ger&auml;teelektronik UBA fehlerhaft",                                                                                                                                                                                                                                                                                                checklist => "Kontakt zwischen Ger&auml;teelektronik UBA und Montagefu&szlig; pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                         correction => "Ger&auml;teelektronik UBA ordnungsgem&auml;&szlig; aufsetzen und nachziehen, Abdeckung hinter Ger&auml;teelektronik UBA abnehmen und Stecker richtig auf Ger&auml;teelektronik-R&uuml;ckseite aufstecken"                                                                                                                                                                                        };
	$ErrorCodeList{"0108"} = {errorcode => "1L",                                     subcode => "527",                                                class => "4",           reason => "Kurzschluss Abgastemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                           checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;-&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder ggf. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen&bull;Bei Abweichungen Ger&auml;teelektronik UBA austauschen"                                                                                                    };
	$ErrorCodeList{"0109"} = {errorcode => "1L",                                     subcode => "529",                                                class => "4",           reason => "Kurzschluss Abgastemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                           checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;-&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder ggf. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen&bull;Bei Abweichungen Ger&auml;teelektronik SAFe austauschen"                                                                                                   };
	$ErrorCodeList{"0110"} = {errorcode => "1P",                                     subcode => "528",                                                class => "4",           reason => "Unterbrechung Abgastemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                         checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;-&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder ggf. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen&bull;Bei Abweichungen Ger&auml;teelektronik SAFe austauschen"                                                                                                   };
	$ErrorCodeList{"0111"} = {errorcode => "2A",                                     subcode => "531",                                                class => "4",           reason => "Wassermangel im W&auml;rmeerzeuger",                                                                                                                                                                                                                                                                                                               checklist => "Pr&uuml;fen, ob die Absperrh&auml;hne ge&ouml;ffnet sind&bull;-&bull;Ist Luft im System?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?",                                                                                                                                                                                                                                                                                                                                          correction => "Anlage entl&uuml;ften&bull;Absperrh&auml;hne &ouml;ffnen&bull;Anlage entl&uuml;ften&bull;Bei zu geringem Wasserdruck, Wasser nachf&uuml;llen und Anlage entl&uuml;ften"                                                                                                                                                                                                                          };
	$ErrorCodeList{"0112"} = {errorcode => "2E",                                     subcode => "207",                                                class => "4",           reason => "Betriebsdruck zu niedrig",                                                                                                                                                                                                                                                                                                                         checklist => "Ist Luft im System?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Drucksensor &uuml;berpr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                         correction => "Anlage entl&uuml;ften&bull;Bei zu geringem Wasserdruck, Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Drucksensor austauschen"                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0113"} = {errorcode => "2E",                                     subcode => "357",                                                class => "BC",          reason => "Entl&uuml;ftungsprogramm",                                                                                                                                                                                                                                                                                                                         checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0114"} = {errorcode => "2E",                                     subcode => "358",                                                class => "BC",          reason => "Entl&uuml;ftungsprogramm",                                                                                                                                                                                                                                                                                                                         checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0115"} = {errorcode => "2F",                                     subcode => "260",                                                class => "12",          reason => "Kein Temperaturanstieg nach Brennerstart",                                                                                                                                                                                                                                                                                                         checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Sicherheitsund Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                 correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Sicherheitsoder Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                 };
	$ErrorCodeList{"0116"} = {errorcode => "2F",                                     subcode => "271",                                                class => "4",           reason => "Temperaturdifferenz Kesselvorlauf-/Sicherheitstemperaturf&uuml;hler; zu gro&szlig;",                                                                                                                                                                                                                                                               checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Ist Luft im System?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Sicherheitsund Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                          correction => "Absperrh&auml;hne &ouml;ffnen&bull;Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Sicherheitsoder Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                                                                        };
	$ErrorCodeList{"0117"} = {errorcode => "2F",                                     subcode => "338",                                                class => "4",           reason => "Zu viele erfolglose Brennerstartversuche",                                                                                                                                                                                                                                                                                                         checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Sicherheitsund Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                 correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Sicherheitsoder Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                 };
	$ErrorCodeList{"0118"} = {errorcode => "2H",                                     subcode => "357",                                                class => "BC",          reason => "Blockierschutz aktiv",                                                                                                                                                                                                                                                                                                                             checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0119"} = {errorcode => "2H",                                     subcode => "358",                                                class => "BC",          reason => "Blockierschutz aktiv",                                                                                                                                                                                                                                                                                                                             checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0120"} = {errorcode => "2L",                                     subcode => "266",                                                class => "4",           reason => "Pumpendruckerh&ouml;hung zu niedrig",                                                                                                                                                                                                                                                                                                              checklist => "Ist Luft im System?&bull;Absperrh&auml;hne nicht vollst&auml;ndig ge&ouml;ffnet&bull;Durchfluss zu gest&ouml;rt, nicht ausreichend&bull;Umw&auml;lzpumpe auf mechanisches Klemmen pr&uuml;fen&bull;Ansteuerung der Umw&auml;lzpumpe pr&uuml;fen&bull;Laufrad der Umw&auml;lzpumpe auf Verschmutzungen pr&uuml;fen&bull;Druckf&uuml;hler auf Verschmutzung pr&uuml;fen&bull;Pr&uuml;fen, ob bei Einsatz einer hydraulischen Weiche Vorund R&uuml;cklauf an den Heizkreisgruppen richtig angeschlossen sind&bull;Pr&uuml;fen, ob das Ausdehnungsgef&auml;&szlig; am Heizkesselvorlauf angeschlossen ist",correction => "Anlage entl&uuml;ften&bull;Absperrh&auml;hne ganz &ouml;ffnen&bull;Ggf. Filter reinigen bzw. ersetzen&bull;Versuchen, die Umw&auml;lzpumpe zu l&ouml;sen&bull;Erfolgt keine Spannungsversorgung der Umw&auml;lzpumpe, Versorgungskabel durchmessen und bei Besch&auml;digung austauschen&bull;Laufrad der Umw&auml;lzpumpe reinigen&bull;Druckf&uuml;hler reinigen ggf. ersetzen&bull;Ggf. Anlagenhydraulik ab&auml;ndern&bull;Hydraulik ab&auml;ndern, Ausdehnungsgef&auml;&szlig; muss in den Anlagenr&uuml;cklauf"};
	$ErrorCodeList{"0121"} = {errorcode => "2L",                                     subcode => "329",                                                class => "10",          reason => "Pumpendruckerh&ouml;hung zu niedrig",                                                                                                                                                                                                                                                                                                              checklist => "Umw&auml;lzpumpe auf mechanisches Klemmen pr&uuml;fen&bull;Ansteuerung der Umw&auml;lzpumpe pr&uuml;fen&bull;Laufrad der Umw&auml;lzpumpe auf Verschmutzung pr&uuml;fen&bull;Pr&uuml;fen, ob das Ausdehnungsgef&auml;&szlig; am Heizkesselvorlauf angeschlossen ist&bull;Pr&uuml;fen, ob bei Einsatz einer hydraulischen Weiche Vorund R&uuml;cklauf an den Heizkreisgruppen richtig angeschlossen sind&bull;Druckf&uuml;hler auf Verschmutzung pr&uuml;fen",                                         correction => "Versuchen, die Umw&auml;lzpumpe zu l&ouml;sen oder die Umw&auml;lzpumpe auszutauschen&bull;Erfolgt keine Spannungsversorgung der Umw&auml;lzpumpe, Versorgungskabel durchmessen und bei Besch&auml;digung austauschen&bull;Laufrad der Umw&auml;lzpumpe reinigen&bull;Hydraulik &auml;ndern, Ausdehnungsgef&auml;&szlig; muss in den Anlagenr&uuml;cklauf&bull;Ggf. Anlagenhydraulik &auml;ndern&bull;Druckf&uuml;hler reinigen, ggf. ersetzen"};
	$ErrorCodeList{"0122"} = {errorcode => "2P",                                     subcode => "212",                                                class => "12",          reason => "Temperaturanstieg Sicherheitsoder Kesselvorlauftemperaturf&uuml;hler zu schnell",                                                                                                                                                                                                                                                                  checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Sicherheitsund Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                 correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Sicherheitsoder Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                 };
	$ErrorCodeList{"0123"} = {errorcode => "2P",                                     subcode => "341",                                                class => "10",          reason => "Temperaturanstieg W&auml;rmeerzeugertemperatur zu schnell",                                                                                                                                                                                                                                                                                        checklist => "Pr&uuml;fen, ob Kessel durchstr&ouml;mt wird&bull;Stecker pr&uuml;fen&bull;Systemwasserdruck sowie Pumpe pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                correction => "W&auml;rmeabnahme sicherstellen&bull;Ggf. richtig aufstecken&bull;Pumpenleistung oder Pumpenkennfeld korrekt einstellen und an maximale Leistung anpassen; W&auml;rmeabnahme sicherstellen"                                                                                                                                                                                                      };
	$ErrorCodeList{"0124"} = {errorcode => "2P",                                     subcode => "342",                                                class => "12",          reason => "Temperaturanstieg Warmwasserbetrieb zu schnell",                                                                                                                                                                                                                                                                                                   checklist => "Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Pumpe pr&uuml;fen&bull;Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                   correction => "Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ggf. Pumpe austauschen&bull;Ggf. Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                                                                                                                     };
	$ErrorCodeList{"0125"} = {errorcode => "2P",                                     subcode => "564",                                                class => "10",          reason => "Temperaturanstieg W&auml;rmeerzeugertemperatur zu schnell",                                                                                                                                                                                                                                                                                        checklist => "Pr&uuml;fen, ob Kessel durchstr&ouml;mt wird&bull;Pumpe pr&uuml;fen&bull;Ablagerungen im Kesselw&auml;rmetauscher (Kalk, Schmutz aus der Heizungsanlage)",                                                                                                                                                                                                                                                                                                                                            correction => "Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;W&auml;rmeabnahme sicherstellen&bull;W&auml;rmetauscher mit einem f&uuml;r Aluminium geeigneten und freigegebenen Mittel reinigen"                                                                                                                                                                             };
	$ErrorCodeList{"0126"} = {errorcode => "2U",                                     subcode => "213",                                                class => "12",          reason => "Differenz zwischen Vorlaufund R&uuml;cklauftemperatur zu gro&szlig;; (> 50 K)",                                                                                                                                                                                                                                                                    checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?; Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?; Ist mindestens ein Thermostatventil ge&ouml;ffnet? Umw&auml;lzpumpe pr&uuml;fen; Vorlaufund R&uuml;cklauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                correction => "Absperrh&auml;hne &ouml;ffnen; Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften; Ein Thermostatventil &ouml;ffnen; Ggf. Umw&auml;lzpumpe austauschen; Ggf. Vorlaufod. R&uuml;cklauftemperaturf&uuml;hler austauschen"                                                                                                                                                };
	$ErrorCodeList{"0127"} = {errorcode => "2U",                                     subcode => "533",                                                class => "4",           reason => "Regelung des W&auml;rmeerzeugers hat wasserseitig falsche Durchstr&ouml;mung erkannt",                                                                                                                                                                                                                                                             checklist => "Pr&uuml;fen, ob Kesselvorund -r&uuml;cklauf vertauscht sind&bull;Umw&auml;lzpumpen auf korrekte Flie&szlig;richtung pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                     correction => "Kesselvorund r&uuml;cklauf korrekt anschlie&szlig;en&bull;Korrekte Flie&szlig;richtung der Pumpen sicherstellen"                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0128"} = {errorcode => "2U",                                     subcode => "565",                                                class => "10",          reason => "Differenz zwischen Vorlaufund R&uuml;cklauftemperatur zu gro&szlig;",                                                                                                                                                                                                                                                                              checklist => "Kann bei normaler Anlagenkonfiguration nicht auftreten",                                                                                                                                                                                                                                                                                                                                                                                                                                              correction => "Anlagenhydraulik pr&uuml;fen"                                                                                                                                                                                                                                                                                                                                                                    };
	$ErrorCodeList{"0129"} = {errorcode => "2U",                                     subcode => "575",                                                class => "4",           reason => "Kesselvorlauftemperatur hat maximal zul&auml;ssigen Wert &uuml;berschritten",                                                                                                                                                                                                                                                                      checklist => "Wasserseitige Durchstr&ouml;mung pr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "Ausreichende Durchstr&ouml;mung sicherstellen&bull;Kesself&uuml;hler/Sicherheitstemperaturbegrenzer-F&uuml;hler austauschen&bull;Z&uuml;nd-/&Uuml;berwachungselektrode austauschen"                                                                                                                                                                                                              };
	$ErrorCodeList{"0130"} = {errorcode => "2U",                                     subcode => "2050",                                               class => "10",          reason => "Falschdurchstr&ouml;mung des Kessels",                                                                                                                                                                                                                                                                                                             checklist => "Fehlerhafter hydraulischer Anschluss des Kessels&bull;Fehlerhafter Einbau der Heizkreispumpe(n) (Flie&szlig;richtung)&bull;Fehlerhafte Montage des HeizkreisSchnellmontage-Sets&bull;R&uuml;cklauftemperaturf&uuml;hler besch&auml;digt",                                                                                                                                                                                                                                                             correction => "Hydraulischer Anschluss (Vorlauf/R&uuml;cklauf) am Kessel pr&uuml;fen, ggf. korrigieren&bull;Einbaulage der Heizkreispumpe(n) pr&uuml;fen, ggf. korrigieren&bull;Montage des Heizkreis-Schnellmontage-Sets pr&uuml;fen, ggf. korrigieren&bull;R&uuml;cklauftemperaturf&uuml;hler pr&uuml;fen, ggf. austauschen"                                                                                  };
	$ErrorCodeList{"0131"} = {errorcode => "2Y",                                     subcode => "281",                                                class => "12",          reason => "Pumpendruckerh&ouml;hung zu niedrig",                                                                                                                                                                                                                                                                                                              checklist => "Luft in der Umw&auml;lzpumpe&bull;Tachokabel auf Kabelbruch pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                             correction => "Pumpe entfl&uuml;ften&bull;Ggf. Tachokabel der Umw&auml;lzpumpe austauschen"                                                                                                                                                                                                                                                                                                                     };
	$ErrorCodeList{"0132"} = {errorcode => "2Y",                                     subcode => "282",                                                class => "10",          reason => "Keine Drehzahlr&uuml;ckmeldung Kesselpumpe",                                                                                                                                                                                                                                                                                                       checklist => "Pr&uuml;fen, ob die Steckverbindung des  Tachokabels und Versorgungskabel der Umw&auml;lzpumpe richtig montiert sind&bull;Tachokabel auf Kabelbruch pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                     correction => "Steckverbindung des Tachokabels der Umw&auml;lzpumpe wiederherstellen&bull;Ggf. Tachokabel der Umw&auml;lzpumpe austauschen"                                                                                                                                                                                                                                                                     };
	$ErrorCodeList{"0133"} = {errorcode => "2Y",                                     subcode => "307",                                                class => "12",          reason => "Kesselpumpe blockiert",                                                                                                                                                                                                                                                                                                                            checklist => "Kesselinterne Pumpe defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Kesselinterne Pumpe austauschen"                                                                                                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0134"} = {errorcode => "2Y",                                     subcode => "308",                                                class => "12",          reason => "Kesselpumpe dreht ohne Widerstand",                                                                                                                                                                                                                                                                                                                checklist => "Kesselinterne Pumpe defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Kesselinterne Pumpe austauschen"                                                                                                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0135"} = {errorcode => "3A",                                     subcode => "264",                                                class => "10",          reason => "Lufttransport w&auml;hrend Betriebsphase ausgefallen",                                                                                                                                                                                                                                                                                             checklist => "Beide Steckverbindungen am Gebl&auml;se pr&uuml;fen&bull;Ansteuerung 230 V AC des Gebl&auml;ses pr&uuml;fen, Versorgungskabel des Gebl&auml;ses pr&uuml;fen&bull;Tachokabel des Gebl&auml;ses pr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                    correction => "Steckverbindungen am Gebl&auml;se ordnungsgem&auml;&szlig; wiederherstellen&bull;Ggf. Versorgungskabel des Gebl&auml;ses austauschen&bull;Ggf. Tachokabel des Gebl&auml;ses austauschen&bull;Ggf. Gebl&auml;se austauschen"                                                                                                                                                                      };
	$ErrorCodeList{"0136"} = {errorcode => "3C",                                     subcode => "217",                                                class => "4",           reason => "Kein Lufttransport nach mehreren Minuten",                                                                                                                                                                                                                                                                                                         checklist => "Feuerungsautomat, Ger&auml;teelektronik oder Gebl&auml;se defekt&bull;Gebl&auml;sekabel mit Stecker und Gebl&auml;se pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                    correction => "Feuerungsautomat oder Gebl&auml;se austauschen&bull;Ggf. austauschen"                                                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0137"} = {errorcode => "3C",                                     subcode => "537",                                                class => "4",           reason => "Keine Drehzahlr&uuml;ckmeldung vom Gebl&auml;se",                                                                                                                                                                                                                                                                                                  checklist => "Verbindungsleitung zwischen Gebl&auml;se und Feuerungsautomat SAFe pr&uuml;fen&bull;Steckverbindung an Feuerungsautomat SAFe und Gebl&auml;se pr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                                    correction => "Wenn besch&auml;digt, austauschen&bull;Wenn besch&auml;digt, austauschen oder Stecker wieder aufstecken&bull;Wenn Spannung anliegt und das Gebl&auml;se nicht l&auml;uft, ist das Gebl&auml;se defekt und muss ausgetauscht werden&bull;Wenn keine Spannung anliegt, ist die Ger&auml;teelektronik SAFe defekt und muss ausgetauscht werden"                                                     };
	$ErrorCodeList{"0138"} = {errorcode => "3C",                                     subcode => "538",                                                class => "4",           reason => "Gebl&auml;se zu langsam",                                                                                                                                                                                                                                                                                                                          checklist => "Gebl&auml;se auf Verschmutzung oder Defekt pr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Gebl&auml;se reinigen&bull;Ggf. Gebl&auml;se austauschen"                                                                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0139"} = {errorcode => "3C",                                     subcode => "539",                                                class => "4",           reason => "Gebl&auml;sedrehzahl au&szlig;erhalb des zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                                                                checklist => "&Uuml;berp&uuml;fen des PWM-Verbindungskabels&bull;Brennergebl&auml;se auf Funktion pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                     correction => "Ggf. PWM-Verbindungskabel austauschen&bull;Ggf. Brennergebl&auml;se austauschen"                                                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0140"} = {errorcode => "3C",                                     subcode => "540",                                                class => "4",           reason => "Gebl&auml;se zu schnell",                                                                                                                                                                                                                                                                                                                          checklist => "Pr&uuml;fen, ob PWM-Signal &uuml;bertragen wird",                                                                                                                                                                                                                                                                                                                                                                                                                                                     correction => "PWM Verbindungskabel pr&uuml;fen, ggf. austauschen"                                                                                                                                                                                                                                                                                                                                              };
	$ErrorCodeList{"0141"} = {errorcode => "3C",                                     subcode => "2036",                                               class => "4",           reason => "Gebl&auml;sedrehzahl entspricht nicht Sollwert",                                                                                                                                                                                                                                                                                                   checklist => "Verbindungsleitung (PWM-Signal) zwischen Feuerungsautomat und Brennergebl&auml;se besch&auml;digt oder nicht aufgesteckt&bull;Spannungsversorgung des Brennergebl&auml;ses fehlerhaft&bull;Brennergebl&auml;se besch&auml;digt&bull;Feuerungsautomat besch&auml;digt",                                                                                                                                                                                                                                correction => "Verbindungsleitung (PWM-Signal) zwischen Feuerungsautomat und Brennergebl&auml;se pr&uuml;fen, aufstecken, ggf. austauschen&bull;Spannungsversorgung des Brennergebl&auml;ses pr&uuml;fen, ggf. austauschen&bull;L&uuml;fterrad des Brennergebl&auml;ses pr&uuml;fen, reinigen oder Brennergebl&auml;se ggf. austauschen&bull;Feuerungsautomat pr&uuml;fen, ggf. austauschen"                    };
	$ErrorCodeList{"0142"} = {errorcode => "3C",                                     subcode => "2037",                                               class => "4",           reason => "Startdrehzahl am Gebl&auml;se nicht erreicht",                                                                                                                                                                                                                                                                                                     checklist => "Verbindungsleitung (PWM-Signal) zwischen Feuerungsautomat und Brennergebl&auml;se besch&auml;digt&bull;Brennergebl&auml;se verschmutzt&bull;Brennergebl&auml;se besch&auml;digt&bull;Feuerungsautomat besch&auml;digt",                                                                                                                                                                                                                                                                               correction => "Verbindungsleitung (PWM-Signal) zwischen Feuerungsautomat und Brennergebl&auml;se pr&uuml;fen, ggf. austauschen&bull;Brennergebl&auml;se auf Verschmutzung pr&uuml;fen, ggf. reinigen&bull;Brennergebl&auml;se pr&uuml;fen, ggf. austauschen&bull;Feuerungsautomat pr&uuml;fen, ggf. austauschen"                                                                                                };
	$ErrorCodeList{"0143"} = {errorcode => "3C",                                     subcode => "2046",                                               class => "4",           reason => "Mindestdrehzahl Gebl&auml;se unterschritten",                                                                                                                                                                                                                                                                                                      checklist => "Verbindungsleitung (PWM-Signal) zwischen Feuerungsautomat und Brennergebl&auml;se besch&auml;digt oder nicht aufgesteckt&bull;Spannungsversorgung des Brennergebl&auml;ses fehlerhaft&bull;Brennergebl&auml;se besch&auml;digt&bull;Feuerungsautomat besch&auml;digt",                                                                                                                                                                                                                                correction => "Verbindungsleitung (PWM-Signal) zwischen Feuerungsautomat und Brennergebl&auml;se pr&uuml;fen, aufstecken, ggf. austauschen&bull;Spannungsversorgung des Brennergebl&auml;ses pr&uuml;fen, ggf. herstellen&bull;Brennergebl&auml;se pr&uuml;fen, ggf. austauschen&bull;Feuerungsautomat pr&uuml;fen, ggf. austauschen"                                                                           };
	$ErrorCodeList{"0144"} = {errorcode => "3C",                                     subcode => "2114",                                               class => "4",           reason => "Schwerg&auml;ngiges Gebl&auml;se Das Ansteuer-Signal (PWM) des Gebl&auml;ses passt nicht zur Drehzahl",                                                                                                                                                                                                                                            checklist => "Schwerg&auml;ngiges Gebl&auml;se&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                             correction => "Gebl&auml;se pr&uuml;fen, ggf. reinigen&bull;Gebl&auml;se austauschen"                                                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0145"} = {errorcode => "3F",                                     subcode => "273",                                                class => "10",          reason => "Betriebsunterbrechung: Brenner und Gebl&auml;se",                                                                                                                                                                                                                                                                                                  checklist => "W&auml;rmeanforderung vollst&auml;ndig wegschalten und nach einer Minute pr&uuml;fen, ob das Gebl&auml;se in Betrieb bleibt",                                                                                                                                                                                                                                                                                                                                                                         correction => "Ggf. Ger&auml;teelektronik UBA austauschen"                                                                                                                                                                                                                                                                                                                                                      };
	$ErrorCodeList{"0146"} = {errorcode => "3H",                                     subcode => "535",                                                class => "10",          reason => "Lufttemperatur zu hoch",                                                                                                                                                                                                                                                                                                                           checklist => "Pr&uuml;fen, ob der Luftf&uuml;hler richtig positioniert ist&bull;Widerstandswerte des F&uuml;hlers nach Tabelle pr&uuml;fen&bull;F&uuml;hlerleitung auf Besch&auml;digung pr&uuml;fen&bull;Kessel auf Verschmutzung pr&uuml;fen",                                                                                                                                                                                                                                                                    correction => "F&uuml;hler richtig positionieren&bull;Bei Abweichungen F&uuml;hler austauschen&bull;Bei Besch&auml;digungen F&uuml;hler austauschen&bull;Wenn erforderlich, Kessel reinigen"                                                                                                                                                                                                                    };
	$ErrorCodeList{"0147"} = {errorcode => "3L",                                     subcode => "214",                                                class => "4",           reason => "Gebl&auml;se wird w&auml;hrend Sicherheitszeit abgeschaltet",                                                                                                                                                                                                                                                                                      checklist => "Beide Steckverbindungen am Gebl&auml;se pr&uuml;fen&bull;Ansteuerung 230 V AC des Gebl&auml;ses pr&uuml;fen, Versorgungskabel des Gebl&auml;ses pr&uuml;fen&bull;Tachokabel des Gebl&auml;ses auf Kabelbruch pr&uuml;fen&bull;Gebl&auml;sekabel mit Stecker und Gebl&auml;se pr&uuml;fen",                                                                                                                                                                                                            correction => "Steckverbindungen am Gebl&auml;se ordnungsgem&auml;&szlig; wiederherstellen&bull;Ggf. Versorgungskabel des Gebl&auml;ses austauschen&bull;Ggf. Tachokabel des Gebl&auml;ses austauschen&bull;Ggf. Gebl&auml;se austauschen"                                                                                                                                                                      };
	$ErrorCodeList{"0148"} = {errorcode => "3P",                                     subcode => "216",                                                class => "4",           reason => "Gebl&auml;se zu langsam",                                                                                                                                                                                                                                                                                                                          checklist => "Pr&uuml;fen, ob das Gebl&auml;se verschmutzt oder feucht ist&bull;Pr&uuml;fen, ob die Netzspannung des Ger&auml;ts zwischen 195 V AC und 253 V AC liegt",                                                                                                                                                                                                                                                                                                                                             correction => "Gebl&auml;se reinigen oder austauschen&bull;Elektroanlage pr&uuml;fen"                                                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0149"} = {errorcode => "3P",                                     subcode => "2035",                                               class => "4",           reason => "Luftklappenstellung entspricht nicht dem Sollwert",                                                                                                                                                                                                                                                                                                checklist => "Luftklappendichtung besch&auml;digt&bull;Luftkanal verschmutzt&bull;Luftklappe besch&auml;digt&bull;Stellmotor besch&auml;digt",                                                                                                                                                                                                                                                                                                                                                                      correction => "Luftklappendichtung pr&uuml;fen, ggf. austauschen&bull;Luftkanal auf Verschmutzung pr&uuml;fen, ggf. reinigen&bull;Luftklappe pr&uuml;fen, ggf. austauschen&bull;Stellmotor pr&uuml;fen, ggf. austauschen"                                                                                                                                                                                       };
	$ErrorCodeList{"0150"} = {errorcode => "3P",                                     subcode => "2042",                                               class => "4",           reason => "Heizpatronentemperatur entspricht nicht Vorgabe Heizpatronentemperatur zu hoch",                                                                                                                                                                                                                                                                   checklist => "Heizpatrone wird nicht korrekt durchstr&ouml;mt&bull;Luftklappe undicht&bull;Luftklappe ist in fehlerhafter Position&bull;Startluftkanal ist verschmutzt&bull;Stellmotor ist fehlerhaft montiert&bull;Stellmotor ist besch&auml;digt&bull;Gebl&auml;se ist verschmutzt&bull;Gebl&auml;se ist besch&auml;digt",                                                                                                                                                                                        correction => "Luftklappendichtung pr&uuml;fen, ggf. austauschen&bull;-&bull;Startluftkanal auf Verschmutzung pr&uuml;fen, ggf. reinigen&bull;Stellmotor pr&uuml;fen, ggf. austauschen&bull;-&bull;Gebl&auml;se pr&uuml;fen, ggf. L&uuml;fterrad reinigen oder Gebl&auml;se austauschen&bull;-"                                                                                                                 };
	$ErrorCodeList{"0151"} = {errorcode => "3P",                                     subcode => "2083",                                               class => "4",           reason => "Positionskalibrierung Luftklappe fehlgeschlagen",                                                                                                                                                                                                                                                                                                  checklist => "Luftklappendichtung fehlerhaft&bull;Luftklappe besch&auml;digt&bull;Startluftkanal verschmutzt&bull;Stellmotor fehlerhaft montiert&bull;Stellmotor besch&auml;digt",                                                                                                                                                                                                                                                                                                                                  correction => "Luftklappendichtung pr&uuml;fen, ggf. austauschen&bull;Luftklappe auf Besch&auml;digung pr&uuml;fen, ggf. austauschen&bull;Startluftkanal auf Verschmutzung pr&uuml;fen, ggf. reinigen&bull;Stellmotor und Luftklappe auf fehlerhafte Montage pr&uuml;fen, ggf. korrigieren&bull;Stellmotor pr&uuml;fen, ggf. austauschen"                                                                       };
	$ErrorCodeList{"0152"} = {errorcode => "3P",                                     subcode => "2091",                                               class => "4",           reason => "Stellklappe schlie&szlig;t schwerg&auml;ngig; Der Strom des Luftklappenstellmotors (GPA) ist im oberen Anschlag zu hoch",                                                                                                                                                                                                                          checklist => "Luftklappendichtung fehlerhaft&bull;Luftklappe besch&auml;digt&bull;Startluftkanal verschmutzt&bull;Stellmotor fehlerhaft montiert&bull;Stellmotor besch&auml;digt",                                                                                                                                                                                                                                                                                                                                  correction => "Luftklappendichtung pr&uuml;fen, ggf. austauschen&bull;Luftklappe auf Besch&auml;digung pr&uuml;fen, ggf. austauschen&bull;Startluftkanal auf Verschmutzung pr&uuml;fen, ggf. reinigen&bull;Stellmotor und Luftklappe auf fehlerhafte Montage pr&uuml;fen, ggf. korrigieren&bull;Stellmotor pr&uuml;fen, ggf. austauschen"                                                                       };
	$ErrorCodeList{"0153"} = {errorcode => "3P",                                     subcode => "2112",                                               class => "4",           reason => "Heizpatrone k&uuml;hlt nach Abschaltung nicht ab",                                                                                                                                                                                                                                                                                                 checklist => "Interne St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                correction => "Entriegeln"                                                                                                                                                                                                                                                                                                                                                                                      };
	$ErrorCodeList{"0154"} = {errorcode => "3U",                                     subcode => "536",                                                class => "4",           reason => "Lufttemperaturf&uuml;hler/Abgastemperaturf&uuml;hler falsch positioniert",                                                                                                                                                                                                                                                                         checklist => "Positionierung des Abgastemperaturf&uuml;hlers pr&uuml;fen&bull;Positionierung des Lufttemperaturf&uuml;hlers pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                           correction => "Bei falscher Positionierung die F&uuml;hlerposition korrigieren&bull;Bei falscher Positionierung die F&uuml;hlerposition korrigieren"                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0155"} = {errorcode => "3Y",                                     subcode => "215",                                                class => "4",           reason => "Gebl&auml;se zu schnell",                                                                                                                                                                                                                                                                                                                          checklist => "Steckverbindung des Tachokabels am Gebl&auml;se pr&uuml;fen&bull;Tachokabel des Gebl&auml;ses auf Kabelbruch pr&uuml;fen&bull;Pr&uuml;fen, ob die Netzspannung des Ger&auml;ts zwischen 195 V AC und 253 V AC liegt&bull;Pr&uuml;fen, ob eine Verstopfung im Brenner, W&auml;rmetauscher oder Abgassystem vorliegt&bull;Pr&uuml;fen, ob das Gebl&auml;serad lose auf der Motorwelle sitzt",                                                                                                           correction => "Steckverbindung ordnungsgem&auml;&szlig; wiederherstellen&bull;Ggf. Tachokabel des Gebl&auml;ses austauschen&bull;Elektroanlage pr&uuml;fen&bull;Verstopfung/Verunreinigung beheben&bull;Ggf. Gebl&auml;se austauschen"                                                                                                                                                                          };
	$ErrorCodeList{"0156"} = {errorcode => "4A",                                     subcode => "218; 332",                                           class => "4",           reason => "Temperatur am Kesselvorlauftemperaturf&uuml;hler zu hoch",                                                                                                                                                                                                                                                                                         checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                                 };
	$ErrorCodeList{"0157"} = {errorcode => "4A",                                     subcode => "505",                                                class => "4",           reason => "Kein Temperaturanstieg am Sicherheitstemperaturbegrenzer",                                                                                                                                                                                                                                                                                         checklist => "Pr&uuml;fen, ob der Sicherheitstemperaturbegrenzer richtig in der Tauchh&uuml;lse steckt",                                                                                                                                                                                                                                                                                                                                                                                                            correction => "Sicherheitstemperaturbegrenzer richtig positionieren"                                                                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0158"} = {errorcode => "4A",                                     subcode => "506",                                                class => "4",           reason => "Temperaturanstieg am Sicherheitstemperaturbegrenzer zu hoch",                                                                                                                                                                                                                                                                                      checklist => "Hydraulik der Anlage bzw. Position des F&uuml;hlers pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                     correction => "F&uuml;r eine ausreichende Durchstr&ouml;mung; des Kessels sorgen bzw. F&uuml;hler richtig positionieren"                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0159"} = {errorcode => "4A",                                     subcode => "507",                                                class => "10",          reason => "Sicherheitstemperaturbegrenzer hat w&auml;hrend Sicherheitstemperaturbegrenzer-Test ausgel&ouml;st",                                                                                                                                                                                                                                               checklist => "Servicetaste an der Ger&auml;teelektronik BC10 h&auml;ngt oder wurde zu lange gedr&uuml;ckt",                                                                                                                                                                                                                                                                                                                                                                                                         correction => "Bei h&auml;ngender Taste die Taste l&ouml;sen oder Ger&auml;teelektronik BC10 austauschen"                                                                                                                                                                                                                                                                                                       };
	$ErrorCodeList{"0160"} = {errorcode => "4A",                                     subcode => "520",                                                class => "4",           reason => "Vorlauftemperatur hat maximal zul&auml;ssigen Wert &uuml;berschritten (100°C)",                                                                                                                                                                                                                                                                   checklist => "Da &uuml;ber den Kesself&uuml;hler der Temperatur-anstieg im Kessel &uuml;berwacht wird und dadurch der Brenner rechtzeitig ausge-; schaltet wird, kann diese St&ouml;rungsmeldung unter normalen Umst&auml;nden nicht erscheinen Anlagenhydraulik pr&uuml;fen, ggf. &auml;ndern",                                                                                                                                                                                                                    correction => "Die St&ouml;rung kann nur bei ung&uuml;nstiger Hydraulik in Zweikessel-Anlagen auftreten, wenn Kessel sich z. B. &uuml;ber den R&uuml;cklauf oder Vorlauf gegenseitig beeinflussen"                                                                                                                                                                                                              };
	$ErrorCodeList{"0161"} = {errorcode => "4A",                                     subcode => "575",                                                class => "4",           reason => "Kesselvorlauftemperatur hat maximal zul&auml;ssigen Wert &uuml;berschritten",                                                                                                                                                                                                                                                                      checklist => "Wasserseitige Durchstr&ouml;mung pr&uuml;fen Pr&uuml;fung ob Fremdw&auml;rmequelle vorhanden (FestbrennstoffKessel)&bull;Kesself&uuml;hler/SicherheitstemperaturbegrenzerF&uuml;hler pr&uuml;fen&bull;Z&uuml;nd-/&Uuml;berwachungselektrode pr&uuml;fen",                                                                                                                                                                                                                                             correction => "Ausreichende Durchstr&ouml;mung sicherstellen&bull;Ggf. Kesself&uuml;hler/Sicherheitstemperaturbegrenzer-F&uuml;hler austauschen&bull;Ggf. Z&uuml;nd-/&Uuml;berwachungselektrode austauschen"                                                                                                                                                                                                    };
	$ErrorCodeList{"0162"} = {errorcode => "4A",                                     subcode => "700",                                                class => "4",           reason => "Werksauslieferungszustand",                                                                                                                                                                                                                                                                                                                        checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Kessel durch Reset entriegeln"                                                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0163"} = {errorcode => "4A",                                     subcode => "2038",                                               class => "4",           reason => "Solltemperatur im Mischraum nicht erreicht",                                                                                                                                                                                                                                                                                                       checklist => "Dichtung Luftklappe verschlie&szlig;en&bull;Heizpatrone defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                      correction => "Dichtung Luftklappe optisch pr&uuml;fen, ggf. austauschen&bull;Heizpatrone pr&uuml;fen, ggf. austauschen"                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0164"} = {errorcode => "4A",                                     subcode => "2043",                                               class => "4",           reason => "Mischraumtemperatur entspricht nicht Vorgabe, Mischraumtemperatur zu niedrig oder zu hoch",                                                                                                                                                                                                                                                        checklist => "-&bull;Mischraum-Temperaturf&uuml;hler besch&auml;digt&bull;Dichtung Luftklappe undicht&bull;Verbrennungsluft-W&auml;rmetauscher verstopft&bull;Zu hoher Restsauerstoffanteil (Lambda) im Abgas&bull;-",                                                                                                                                                                                                                                                                                              correction => "Entriegeln&bull;Mischraum-Temperaturf&uuml;hler pr&uuml;fen, ggf. austauschen&bull;Luftklappendichtung pr&uuml;fen, ggf. austauschen&bull;Verbrennungsluft-W&auml;rmetauscher auf Verschmutzung pr&uuml;fen, ggf. reinigen&bull;Abgasmessung durchf&uuml;hren: CO2-Wert pr&uuml;fen&bull;&Ouml;lpumpendruck messen, ggf. &Ouml;lpumpe austauschen&bull;Einspritzventil pr&uuml;fen, ggf. austauschen&bull;Brennergebl&auml;se pr&uuml;fen, ggf. austauschen"};
	$ErrorCodeList{"0165"} = {errorcode => "4A",                                     subcode => "2090",                                               class => "4",           reason => "Temperaturanstieg der Heizpatrone zu gering",                                                                                                                                                                                                                                                                                                      checklist => "Heizpatronen-Temperaturf&uuml;hler nicht korrekt angeschlossen&bull;Heizpatrone besch&auml;digt&bull;Verbrennungsluftdurchstr&ouml;mung der Heizpatrone zu gro&szlig;&bull;Feuerungsautomat defekt",                                                                                                                                                                                                                                                                                                  correction => "Anschluss Heizpatronen-Temperaturf&uuml;hler pr&uuml;fen, ggf. korrigieren&bull;Spannungsversorgung Heizpatrone pr&uuml;fen, ggf. korrigieren&bull;Widerstand Heizpatronenwendel (< 100 Ohm) pr&uuml;fen, ggf. Heizpatrone austauschen&bull;Feuerungsautomat pr&uuml;fen, ggf. austauschen"                                                                                                      };
	$ErrorCodeList{"0166"} = {errorcode => "4A",                                     subcode => "2129",                                               class => "8",           reason => "W&auml;rmeanforderung blockiert weil Mischraum zu hei&szlig;",                                                                                                                                                                                                                                                                                     checklist => "Betriebsmeldung, keine St&ouml;rung; aktueller Betriebszustand des W&auml;rmeerzeugers",                                                                                                                                                                                                                                                                                                                                                                                                              correction => "Keine Ma&szlig;nahme notwendig"                                                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0167"} = {errorcode => "4C",                                     subcode => "224",                                                class => "4",           reason => "Sicherheitsoder Abgastemperaturbegrenzer hat ausgel&ouml;st",                                                                                                                                                                                                                                                                                      checklist => "Sicherheitstemperaturbegrenzer auf Unterbrechung oder Kurzschluss pr&uuml;fen&bull;Abgastemperatur auf Unterbrechung oder Kurzschluss pr&uuml;fen&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Pumpenanlauf pr&uuml;fen",                                                                                                                                                                                                                                                   correction => "Ggf. Sichheitstemperaturbegrenzer austauschen&bull;Ggf. Abgastemperaturbegrenzer austauschen&bull;Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ggf. Pumpe austauschen"                                                                                                                                                                                      };
	$ErrorCodeList{"0168"} = {errorcode => "4E",                                     subcode => "225",                                                class => "4",           reason => "Temperatur",                                                                                                                                                                                                                                                                                                                                       checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Sicherheitsund Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                 correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Doppeltemperaturf&uuml;hler austauschen"                                                                                                                                                  };
	$ErrorCodeList{"0169"} = {errorcode => "4E",                                     subcode => "278",                                                class => "10",          reason => "F&uuml;hlertest fehlgeschlagen",                                                                                                                                                                                                                                                                                                                   checklist => "Mittels Spannungsmessung pr&uuml;fen,; ob zwischen der Verkabelung des Vorlaufund des Sicherheitstemperaturf&uuml;hlers ein Kurzschluss vorliegt",                                                                                                                                                                                                                                                                                                                                                    correction => "Kabelbaum oder entsprechenden Teil davon ersetzen"                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0170"} = {errorcode => "4F",                                     subcode => "219",                                                class => "4",           reason => "Temperatur am Sicherheitstemperaturf&uuml;hler zu hoch",                                                                                                                                                                                                                                                                                           checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Sicherheitstemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                            correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck,; Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Ggf. Sicherheitstemperaturf&uuml;hler austauschen"                                                                                                                                           };
	$ErrorCodeList{"0171"} = {errorcode => "4L",                                     subcode => "220",                                                class => "4",           reason => "Temperatur am Sicherheitstemperaturf&uuml;hler zu hoch",                                                                                                                                                                                                                                                                                           checklist => "Sind alle Absperrh&auml;hne ge&ouml;ffnet?&bull;Betr&auml;gt der Wasserdruck in der Anlage mindestens 1 bar?&bull;Ist mindestens ein Thermostatventil ge&ouml;ffnet?&bull;Umw&auml;lzpumpe pr&uuml;fen&bull;Kabel im Kabelbaum zum Sicherheitsf&uuml;hler auf Kurzschluss pr&uuml;fen",                                                                                                                                                                                                               correction => "Absperrh&auml;hne &ouml;ffnen&bull;Bei zu geringem Wasserdruck,; Wasser nachf&uuml;llen und Anlage entl&uuml;ften&bull;Ein Thermostatventil &ouml;ffnen&bull;Ggf. Umw&auml;lzpumpe austauschen&bull;Bei Besch&auml;digung Kabelbaum austauschen"                                                                                                                                                 };
	$ErrorCodeList{"0172"} = {errorcode => "4P",                                     subcode => "221",                                                class => "4",           reason => "Keine Verbindung zum Sicherheitstemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                            checklist => "Steckverbindung zum Sicherheitsf&uuml;hler auf losen Kontakt pr&uuml;fen&bull;Kabel im Kabelbaum zum Sicherheitsf&uuml;hler auf Besch&auml;digungen pr&uuml;fen&bull;Widerstandswert des Sicherheitsf&uuml;hlers pr&uuml;fen",                                                                                                                                                                                                                                                                        correction => "Steckverbindung ordnungsgem&auml;&szlig; anbringen&bull;Bei Besch&auml;digung Kabelbaum austauschen&bull;Ggf. Sicherheitsf&uuml;hler austauschen"                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0173"} = {errorcode => "4U",                                     subcode => "222",                                                class => "4",           reason => "Kurzschluss Vorlauftemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                         checklist => "Kabel im Kabelbaum zum Vorlauftemperaturf&uuml;hler auf Kurzschluss pr&uuml;fen&bull;Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                       correction => "Bei Besch&auml;digung Kabelbaum austauschen&bull;Ggf. Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0174"} = {errorcode => "4U",                                     subcode => "350",                                                class => "10",          reason => "Kurzschluss Vorlauftemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                         checklist => "Kabel im Kabelbaum zum Vorlauftemperaturf&uuml;hler auf Kurzschluss pr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                              correction => "Bei Besch&auml;digung Kabelbaum austauschen; Ggf. Vorlauftemperaturf&uuml;hler austauschen&bull;-"                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0175"} = {errorcode => "4U",                                     subcode => "521",                                                class => "4",           reason => "F&uuml;hlerdifferenz zwischen Kesselvorlauftemperaturf&uuml;hler 1 und 2 zu gro&szlig;",                                                                                                                                                                                                                                                           checklist => "Pr&uuml;fen, ob die „Reset“-Taste der Ger&auml;teelektronik SAFe leuchtet (am Brenner)&bull;Pr&uuml;fen, ob die R&uuml;ckschlagklappe an der Speicherladepumpe geschlossen ist&bull;Pr&uuml;fen, ob Heizkesselvorlauf und -r&uuml;cklauf richtig angeschlossen sind&bull;Steckverbindung am Kesself&uuml;hler und an Ger&auml;teelektronik SAFe oder Fremdbrennermodul auf Verschmutzung bzw. Besch&auml;digung pr&uuml;fen&bull;Widerstandswerte der Kesself&uuml;hler nach Tabelle pr&uuml;fen bzw. Stecker am F&uuml;hler optisch pr&uuml;fen&bull;Spannungswerte am Kesself&uuml;hler nach Tabelle pr&uuml;fen",correction => "„Reset“ an Ger&auml;teelektronik SAFe bet&auml;tigen (am Brenner)&bull;Sollte die Klappe nicht geschlossen sein, diese schlie&szlig;en&bull;Sollte eine Vertauschung vorliegen, Kessel richtig anschlie&szlig;en&bull;Ggf. Kontakte reinigen bzw. Verbindungsleitung austauschen&bull;Wenn Abweichungen der F&uuml;hlerwerte bestehen oder der Stecker defekt ist,; Kesself&uuml;hler austauschen&bull;Wenn Abweichungen bestehen, Ger&auml;teelektronik SAFe oder Fremdbrennermodul austauschen"};
	$ErrorCodeList{"0176"} = {errorcode => "4U",                                     subcode => "522",                                                class => "4",           reason => "Kurzschluss zwischen Kesselvorlauftemperaturf&uuml;hler 1 und 2",                                                                                                                                                                                                                                                                                  checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;-&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder ggf. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen&bull;Bei Abweichungen Ger&auml;teelektronik SAFe oder Fremdbrennermodul austauschen"                                                                            };
	$ErrorCodeList{"0177"} = {errorcode => "4U",                                     subcode => "524",                                                class => "4",           reason => "Kurzschluss Kesselvorlauftemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                   checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;-&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder ggf. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen&bull;Bei Abweichungen Ger&auml;teelektronik SAFe oder Fremdbrennermodul austauschen"                                                                            };
	$ErrorCodeList{"0178"} = {errorcode => "4U",                                     subcode => "532",                                                class => "10",          reason => "Netzspannung zu niedrig",                                                                                                                                                                                                                                                                                                                          checklist => "Verdrahtung oder Netzspannung pr&uuml;fen&bull;Fremdbrennermodul defekt&bull;EMV-Probleme",                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Verdrahtung korrigieren oder f&uuml;r ausreichend Netzspannung sorgen&bull;Fremdbrennermodul austauschen&bull;EMV-Probleme beseitigen"                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0179"} = {errorcode => "4U",                                     subcode => "2006",                                               class => "4",           reason => "Kurzschluss MischraumTemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                       checklist => "Zu geringe Temperatur vom MischraumTemperaturf&uuml;hler gemessen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                            correction => "Mischraum-Temperaturf&uuml;hler pr&uuml;fen, ggf. austauschen; F&uuml;hlerwerte nach Tabelle pr&uuml;fen, Spannungswerte am Kesself&uuml;hler nach Tabelle pr&uuml;fen&bull;Steckverbindung pr&uuml;fen, ggf. Kabelbaum oder Temperaturf&uuml;hler austauschen&bull;Feuerungsautomat austauschen"                                                                                                };
	$ErrorCodeList{"0180"} = {errorcode => "4U",                                     subcode => "2009",                                               class => "4",           reason => "Differenz zwischen Mischraum-Temperaturf&uuml;hler 1 und 2 zu gro&szlig;",                                                                                                                                                                                                                                                                         checklist => "Mischraum-Temperaturf&uuml;hler misst fehlerhafte Werte&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                      correction => "F&uuml;hlerkabel pr&uuml;fen, ggf. austauschen&bull;Temperaturf&uuml;hler pr&uuml;fen, ggf. austauschen&bull;Feuerungsautomat austauschen&bull;Kundendienst verst&auml;ndigen"                                                                                                                                                                                                                   };
	$ErrorCodeList{"0181"} = {errorcode => "4U",                                     subcode => "2023",                                               class => "4",           reason => "Kurzschluss F&uuml;hler Heizpatrone",                                                                                                                                                                                                                                                                                                              checklist => "Heizpatronen-Temperaturf&uuml;hler liefert fehlerhafte Werte&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "F&uuml;hlerleitung pr&uuml;fen, ggf. austauschen&bull;Spannungsversorgung Kesselregelger&auml;t pr&uuml;fen, ggf. Anschluss korrigieren&bull;Polung der Anschlussleitung pr&uuml;fen, ggf. Anschluss korrigieren"                                                                                                                                                                                };
	$ErrorCodeList{"0182"} = {errorcode => "4U",                                     subcode => "2100",                                               class => "4",           reason => "Kurzschluss MischraumTemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                       checklist => "Mischraum-Temperaturf&uuml;hler defekt&bull;Steckverbindung Mischraum-Temperaturf&uuml;hler besch&auml;digt oder nicht aufgesteckt&bull;Feuerungsautomat defekt",                                                                                                                                                                                                                                                                                                                                     correction => "Temperaturf&uuml;hler pr&uuml;fen, ggf. austauschen&bull;Steckverbindung Mischraum-Temperaturf&uuml;hler pr&uuml;fen, ggf. aufstecken&bull;Feuerungsautomat austauschen"                                                                                                                                                                                                                         };
	$ErrorCodeList{"0183"} = {errorcode => "4Y",                                     subcode => "223",                                                class => "4",           reason => "Kontakt Vorlauftemperaturf&uuml;hler lose oder defekt",                                                                                                                                                                                                                                                                                            checklist => "Steckverbindung zum Vorlauftemperaturf&uuml;hler auf losen Kontakt pr&uuml;fen&bull;Kabel im Kabelbaum zum Vorlauftemperaturf&uuml;hler auf Besch&auml;digungen pr&uuml;fen&bull;Widerstandswert des Vorlauftemperaturf&uuml;hlers pr&uuml;fen",                                                                                                                                                                                                                                                      correction => "Steckverbindung ordnungsgem&auml;&szlig; anbringen&bull;Bei Besch&auml;digung Kabelbaum austauschen&bull;Ggf. Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0184"} = {errorcode => "4Y",                                     subcode => "351",                                                class => "10",          reason => "Unterbrechung Vorlauftemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                       checklist => "Kabel im Kabelbaum zum Vorlauftemperaturf&uuml;hler pr&uuml;fen&bull;Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                       correction => "Bei Besch&auml;digung Kabelbaum austauschen&bull;Ggf. Vorlauftemperaturf&uuml;hler austauschen"                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0185"} = {errorcode => "4Y",                                     subcode => "523",                                                class => "4",           reason => "Unterbrechung Kesselvorlauftemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                 checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;-&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder ggf. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen&bull;Bei Abweichungen Ger&auml;teelektronik SAFe oder Fremdbrennermodul austauschen"                                                                            };
	$ErrorCodeList{"0186"} = {errorcode => "4Y",                                     subcode => "2005",                                               class => "4",           reason => "Unterbrechung MischraumTemperaturf&uuml;hler",                                                                                                                                                                                                                                                                                                     checklist => "Zu hohe Temperatur vom MischraumTemperaturf&uuml;hler gemessen&bull;Steckverbindung Mischraum-Temperaturf&uuml;hler fehlerhaft oder besch&auml;digt&bull;Feuerungsautomat defekt",                                                                                                                                                                                                                                                                                                                    correction => "Mischraum-Temperaturf&uuml;hler pr&uuml;fen, ggf. austauschen; F&uuml;hlerwerte nach Tabelle pr&uuml;fen, Spannungswerte am Kesself&uuml;hler nach Tabelle pr&uuml;fen&bull;Steckverbindung pr&uuml;fen, ggf. Kabelbaum oder Temperaturf&uuml;hler austauschen&bull;Feuerungsautomat austauschen"                                                                                                };
	$ErrorCodeList{"0187"} = {errorcode => "5A",                                     subcode => "275",                                                class => "20",          reason => "Ger&auml;teelektronik UBA im Testmode",                                                                                                                                                                                                                                                                                                            checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0188"} = {errorcode => "5A",                                     subcode => "507",                                                class => "BC",          reason => "Sicherheitstemperaturbegrenzer-Test erfolgreich durchgef&uuml;hrt",                                                                                                                                                                                                                                                                                checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0189"} = {errorcode => "5C",                                     subcode => "226",                                                class => "BC",          reason => "Kennzeichnung f&uuml;r Handterminal",                                                                                                                                                                                                                                                                                                              checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0190"} = {errorcode => "5E",                                     subcode => "586",                                                class => "4",           reason => "Ger&auml;teelektronik SAFe alter Softwarestand",                                                                                                                                                                                                                                                                                                   checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Ger&auml;teelektronik SAFe mit neuem Softwarestand (ab Version 4.X) einsetzen"                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0191"} = {errorcode => "5H",                                     subcode => "268",                                                class => "BC",          reason => "Relaistest wurde aktiviert",                                                                                                                                                                                                                                                                                                                       checklist => "Betriebsmeldung, keine St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0192"} = {errorcode => "5H",                                     subcode => "310",                                                class => "10",          reason => "Keine Kommunikation mit dem EMS W&auml;rmeerzeuger",                                                                                                                                                                                                                                                                                               checklist => "Es kann keine Kommunikation zwischen dem Hybridmodul (Inneneinheit) und dem EMSW&auml;rmeerzeuger hergestellt werden&bull;Kontakte, Kabel und Verbindungen zwischen Hybridmodul und EMS-W&auml;rmeerzeuger pr&uuml;fen",                                                                                                                                                                                                                                                                              correction => "EMS-Verbindung zwischen dem Hybridmodul (Inneneinheit) und dem EMS-W&auml;rmeerzeuger &uuml;berpr&uuml;fen&bull;Besch&auml;digte Leitungen und Kontakte austauschen, ggf. Ger&auml;teelektronik Hybridmodul (Inneneinheit) und EMS W&auml;rmeerzeuger MC10/ MC40/MC100/BC10/BC25 austauschen"                                                                                                    };
	$ErrorCodeList{"0193"} = {errorcode => "5H",                                     subcode => "323",                                                class => "n.a.",        reason => "EMS Bussignal kann im Ger&auml;t nicht verarbeitet werden",                                                                                                                                                                                                                                                                                        checklist => "&Uuml;berpr&uuml;fen, ob alle Anschlussstecker f&uuml;r den EMS BUS aufgesteckt sind&bull;Anschlussstecker auf korrekte Verdrahtung pr&uuml;fen&bull;Anschlusskabel pr&uuml;fen",                                                                                                                                                                                                                                                                                                                     correction => "Anschlussstecker mit der Beschriftung EMS im Steuerger&auml;t aufstecken&bull;Anschlusstecker mit der Beschrifung EMS austauschen&bull;Anschlusskabel f&uuml;r den 2 Draht Bus austauschen"                                                                                                                                                                                                      };
	$ErrorCodeList{"0194"} = {errorcode => "5H",                                     subcode => "470",                                                class => "10",          reason => "Keine Kommunikation mit dem Systemregler",                                                                                                                                                                                                                                                                                                         checklist => "Kontakte, Kabel und Verbindungen zwischen Hybridmodul und Systemregler pr&uuml;fen&bull;Systemregler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                    correction => "Besch&auml;digte Leitungen und Kontakte austauschen&bull;Ggf. ersetzen"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0195"} = {errorcode => "5H",                                     subcode => "2113",                                               class => "4",           reason => "Interne St&ouml;rung",                                                                                                                                                                                                                                                                                                                             checklist => "Interne St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                correction => "Entriegeln"                                                                                                                                                                                                                                                                                                                                                                                      };
	$ErrorCodeList{"0196"} = {errorcode => "5H",                                     subcode => "2504",                                               class => "BC",          reason => "W&auml;rmeanforderung wegen Relaistest",                                                                                                                                                                                                                                                                                                           checklist => "W&auml;rmeanforderung wegen Relaistest",                                                                                                                                                                                                                                                                                                                                                                                                                                                              correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0197"} = {errorcode => "5L",                                     subcode => "542",                                                class => "10",          reason => "Kommunikation mit Ger&auml;teelektronik SAFe oder BRM10/Regelger&auml;t/Feuerungsautomat oder Fremdbrennermodul unvollst&auml;ndig",                                                                                                                                                                                                               checklist => "Leitungsverbindungen zwischen Feuerungsautomat oder Fremdbrennermodul und Regelger&auml;t pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                               correction => "Ger&auml;teelektronik SAFe oder Fremdbrennermodul austauschen"                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0198"} = {errorcode => "5L",                                     subcode => "543",                                                class => "10",          reason => "Keine Kommunikation mit Ger&auml;teelektronik/ Fremdbrennermodul; (Auswirkung: schnelles Blinken; der LED am SAFe bzw. UM10 bzw. BRM10 (= Not-; betrieb))",                                                                                                                                                                                        checklist => "Pr&uuml;fen, ob die Stecker der Kabel (Busleitung und Netzleitung) zwischen Ger&auml;teelektronik oder Fremdbrennermodul und Regelger&auml;t richtig aufgesteckt sind&bull;Im Regelger&auml;t an den Anschlussklemmen „Netz Ger&auml;teelektronik“ pr&uuml;fen ob 230 V AC anliegen&bull;Pr&uuml;fen, ob die Verbindungsleitungen (Busleitung und Netzleitung) zwischen Ger&auml;teelektronik und Regelger&auml;t evtl. besch&auml;digt sind&bull;Die Busverbindungsleitung zwischen Ger&auml;teelektronik trennen und pr&uuml;fen, ob der Kessel in Notbetrieb geht (l&auml;uft auf 60°C Kesseltemperatur)&bull;Pr&uuml;fen, ob am Ger&auml;teelektronik die gr&uuml;ne Signallampe leuchtet&bull;Wenn die Ger&auml;teelektronik dunkel bleibt, eine gewisse Zeit abwarten, da bei kalter Ger&auml;teelektronik das Ger&auml;t evtl. nicht anl&auml;uft&bull;Durch Austausch pr&uuml;fen, ob Regelger&auml;t ger&auml;teelektronik oder fremdbrennermodul defekt ist&bull;Durch Austausch pr&uuml;fen, ob der Ger&auml;teelektronik oder Fremdbrennermodul defekt ist",correction => "Stecker richtig aufstecken&bull;Wenn keine 230 V AC anliegen, dann ist das Regelger&auml;t defekt und muss ausgetauscht werden&bull;Verbindungsleitung austauschen&bull;Wenn die Lampen nicht leuchten, dann ist vermutlich die Ger&auml;teelektronik defekt und muss ausgetauscht werden&bull;Wenn der Kessel nicht anl&auml;uft, dann ist die Ger&auml;teelektronik defekt und muss ausgetauscht werden&bull;Max. 30 Minuten warten und pr&uuml;fen, ob dann die gr&uuml;ne Lampe an der Ger&auml;teelektronik; wieder leuchtet; Wenn dies nicht der Fall ist, dann muss die Ger&auml;teelektronik ausgetauscht werden&bull;Regelger&auml;t austauschen&bull;Max. 30 Minuten warten und pr&uuml;fen, ob; dann die gr&uuml;ne Lampe an der Ger&auml;teelektronik wieder leuchtet; Wenn dies nicht der Fall ist, dann muss die Ger&auml;teelektronik ausgetauscht werden"};
	$ErrorCodeList{"0199"} = {errorcode => "5L",                                     subcode => "2051",                                               class => "10",          reason => "Interne St&ouml;rung",                                                                                                                                                                                                                                                                                                                             checklist => "Sicherheits-Controller blockiert&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                             correction => "Anlage f&uuml;r 30 Sekunden spannungsfrei schalten&bull;Feuerungsautomat austauschen&bull;Kundendienst verst&auml;ndigen"                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0200"} = {errorcode => "5P",                                     subcode => "552",                                                class => "4",           reason => "„Reset“-Taste zu h&auml;ufig bet&auml;tigt; (Wird innerhalb kurzer Zeit die „Reset“-Taste am MC10 zu haufig betatigt, wird diese Fehler meldung erzeugt.; ACHTUNG: Dieser Fehler lasst sich nur uber die Taste am SAFe entriegeln!)",                                                                                                      checklist => "Pr&uuml;fen, ob die „Reset“-Taste am Regelger&auml;t evtl. klemmt&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "Taste wieder l&ouml;sen&bull;Wenn die „Reset“-Taste in Ordnung ist, dann muss der Basiscontroller ausgetauscht werden"                                                                                                                                                                                                                                                                       };
	$ErrorCodeList{"0201"} = {errorcode => "5U",                                     subcode => "582",                                                class => "10",          reason => "Keine Kommunikation mit Umschaltmodul",                                                                                                                                                                                                                                                                                                            checklist => "Verbindungsleitung zwischen Feuerungsautomat oder Fremdbrennermodul und Umschaltmodul pr&uuml;fen&bull;Sicherung am Umschaltmodul pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                       correction => "Bei defekten Kabeln diese austauschen, bei losem Stecker diesen richtig aufstecken&bull;Sicherung austauschen"                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0202"} = {errorcode => "5U",                                     subcode => "588",                                                class => "4",           reason => "Mehr als ein Umschaltmodul im System",                                                                                                                                                                                                                                                                                                             checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Alle Module bis auf ein Umschaltmodul entfernen"                                                                                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0203"} = {errorcode => "5Y",                                     subcode => "585",                                                class => "4",           reason => "Umschaltmodul nicht vorhanden",                                                                                                                                                                                                                                                                                                                    checklist => "Wenn das Umschaltmodul ausgebaut wird, muss es auch softwareseitig deinstalliert werden (siehe Montageund Bedienungsanleitung)",                                                                                                                                                                                                                                                                                                                                                                      correction => "Entsprechend der Montageund Bedienungsanleitung einen Reset am Raumcontroller durchf&uuml;hren, um das Umschaltmodul zu deinstallieren bzw. das Modul auszutauschen"                                                                                                                                                                                                                             };
	$ErrorCodeList{"0204"} = {errorcode => "6A",                                     subcode => "227",                                                class => "4",           reason => "Kein Flammensignal nach Z&uuml;ndung; (Keine Ionisationsmeldung nach; der Zundung. Nach dem 4. Versuch (GB172: 5. Versuch) wird die; St&ouml;rung 6A erzeugt. Zwischen erfolglosen Zundungsversuchen Anzeige blockierend (6A nicht blinkend). Nach 5 erfolglosen; Zundungsversuchen Anzeige; verriegelnd (6A blinkend),; „Reset“ zum Entriegeln)", checklist => "Gasanschlussdruck messen, Gasd&uuml;se pr&uuml;fen&bull;Ist die Gasleitung entl&uuml;ftet?&bull;Gl&uuml;hz&uuml;nder in Ordnung?&bull;Die Spannung zum Gl&uuml;hz&uuml;nder muss 120 V AC betragen&bull;&Ouml;ffnet die Gasarmatur bei 0L? Elektrischen Anschluss Gasarmatur pr&uuml;fen&bull;Zuluftund Abgasweg sowie Brennkammer auf Verschmutzung pr&uuml;fen&bull;Ionisationsstrom messen&bull;Ist das Kesselidentifikationsmodul richtig aufgesteckt?&bull;Bei raumluftabh&auml;ngiger Betriebsweise den Raumluftverbund oder die L&uuml;ftungs&ouml;ffnung pr&uuml;fen&bull;Abgasweg kontollieren und den Kondensatsiphon pr&uuml;fen&bull;W&auml;rmetauscher auf Verschmutzung pr&uuml;fen&bull;Ist ein Stromnetz (IT) vorhanden, in Deutschland praktisch nicht vorkommend (au&szlig;er Insell&ouml;sungen)?",correction => "Ist der ben&ouml;tigte Gasanschlussdruck nicht vorhanden, mit dem zust&auml;ndigen Gasversorger R&uuml;cksprache halten; Gas-Luft-Verh&auml;ltnis auf: 5 Pa einstellen F&uuml;r die Gasart korrekte Gasd&uuml;se einsetzen&bull;Gasleitung entl&uuml;ften&bull;Widerstand 50: 300 Ω, dann Gl&uuml;hz&uuml;nder in Ordnung, sonst auswechseln&bull;Wenn keine Spannung anliegt, Steckverbindung pr&uuml;fen oder defekten Trafo austauschen&bull;Spannung muss w&auml;hrend 0L zwischen 20 und 24 V AC liegen&bull;Verschmutzung beseitigen&bull;Ionisationsstrom muss in der Teillast > 1,4 μA betragen, sonst &Uuml;berwachungselektrode austauschen&bull;Kesselidentifikationsmodul richtig aufstecken oder ggf. austauschen&bull;-&bull;Ggf. Abgasweg reinigen bzw. instand setzen und den Kondensatsiphon reinigen&bull;Ggf. W&auml;rmetauscher reinigen&bull;Am Netzanschluss an der Leiterplatte muss dann ein Widerstand von 2 M Ω eingebaut werden"};
	$ErrorCodeList{"0205"} = {errorcode => "6A",                                     subcode => "504",                                                class => "10",          reason => "Brennerst&ouml;rung",                                                                                                                                                                                                                                                                                                                              checklist => "Der vorhandene Nicht-EMS-Brenner ist auf St&ouml;rung",                                                                                                                                                                                                                                                                                                                                                                                                                                               correction => "Brenner am Feuerungsautomaten entriegeln"                                                                                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0206"} = {errorcode => "6A",                                     subcode => "577",                                                class => "10",          reason => "Keine Flamme innerhalb Sicherheitszeit; (Innerhalb der Sicherheitszeit ist Ionisationsstrom <; 1.1 μA)",                                                                                                                                                                                                                                          checklist => "Pr&uuml;fen, ob der Gashahn ge&ouml;ffnet ist&bull;Gasanschlussdruck pr&uuml;fen&bull;Querschnitte der Gasleitung nicht ausreichend dimensioniert&bull;Gegendruck der Abgasanlage zu hoch durch ung&uuml;nstige Ausf&uuml;hrung (zu viele Umlenkungen, Querschnitte zu gering, zu lange waagerechte Strecken)&bull;Verbindungsleitung zwischen Ger&auml;teelektronik SAFe und &Uuml;berwachungselektrode auf fehlerhafte Kontaktierung, Unterbrechung und Besch&auml;digung pr&uuml;fen&bull;Elektrodenabst&auml;nde und Z&uuml;nd-/&Uuml;berwachungselektrode auf Besch&auml;digungen pr&uuml;fen&bull;Luft in der Gasleitung&bull;Startgasd&uuml;se verschmutzt&bull;Gasarmatur defekt (&uuml;ber Klickger&auml;usch der Armatur h&ouml;rbar)&bull;Masseanschluss der &Uuml;berwachungselektrode pr&uuml;fen&bull;&Uuml;berwachungselektrode verschmutzt&bull;Z&uuml;ndung und Flammen&uuml;berwachung pr&uuml;fen&bull;Z&uuml;ndtrafo defekt (kein oder verz&ouml;gerter Z&uuml;ndfunke, „harter Start“)&bull;Feuerungsautomat/Ger&auml;teelektronik SAFe defekt",correction => "Gashahn &ouml;ffnen&bull;Bei zu niedrigem Gasanschlussdruck den Gasversorger informieren&bull;Ausreichend dimensionierte Gasleitung installieren&bull;Abgasanlage korrekt dimensionieren&bull;Korrekte Kontaktierung herstellen, ggf. Kabel austauschen&bull;Brennstab und Elektrode ausrichten, defekte Elektroden austauschen&bull;Gasleitung entl&uuml;ften&bull;Startgasd&uuml;se reinigen&bull;Gasarmatur austauschen&bull;Masseverbindung der &Uuml;berwachungselektrode herstellen&bull;&Uuml;berwachungselektrode reinigen bzw. austauschen&bull;Austausch Z&uuml;ndung und Flammen&uuml;berwachung&bull;Z&uuml;ndtrafo austauschen&bull;Feuerungsautomat/Ger&auml;teelektronik SAFe austauschen"};
	$ErrorCodeList{"0207"} = {errorcode => "6A",                                     subcode => "578",                                                class => "4",           reason => "Wiederholt keine Flamme innerhalb Sicherheitszeit",                                                                                                                                                                                                                                                                                                checklist => "Hauptabsperreinrichtung &uuml;berpr&uuml;fen&bull;Ger&auml;teabsperrhan &uuml;berpr&uuml;fen&bull;Gasanschlu&szlig;flie&szlig;druck bei Nennw&auml;rmebelastung messen&bull;Anschlusstecker an der Ionisationselektrode &uuml;berpr&uuml;fen&bull;Ionisationselektrode &uuml;berpr&uuml;fen&bull;Z&uuml;ndkabel auf Besch&auml;digungen &uuml;berpr&uuml;fen&bull;Anschlusskabel f&uuml;r die Ionisationselektrode auf Besch&auml;digungen &uuml;berpr&uuml;fen&bull;Z&uuml;ndtransformator &uuml;berpr&uuml;fen&bull;Brennereinstellwert bei kleinster Leistung &uuml;berpr&uuml;fen&bull;Verbrennungsluftversorgung &uuml;berpr&uuml;fen",correction => "Hauptabsperreinrichtung &ouml;ffnen&bull;Ger&auml;teabsperrhahn &ouml;ffnen&bull;Ger&auml;t stilllegen und Gasleitung &uuml;berpr&uuml;fen&bull;Anschlussstecker an der Ionisationelektrode aufstecken&bull;Ionisationselektrode austauschen&bull;Z&uuml;ndkabel austauschen&bull;Anschlusskabel zur Ionisationselektrode austauschen&bull;Z&uuml;ndtransformator austauschen&bull;Brenner bei minimaler Nennbelastung einstellen&bull;Verbrennungsluftverbund zu klein bzw. zu geringe Gr&ouml;&szlig;e der L&uuml;ftungs&ouml;ffnung"};
	$ErrorCodeList{"0208"} = {errorcode => "6C",                                     subcode => "228",                                                class => "4",           reason => "Flammensignal trotz nicht vorhandener Flamme",                                                                                                                                                                                                                                                                                                     checklist => "&Uuml;berwachungselektrode auf Besch&auml;digung bzw. Verschmutzung pr&uuml;fen&bull;Leiterplatte auf Feuchtigkeit pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                      correction => "&Uuml;berwachungselektrode austauschen bzw. reinigen&bull;Leiterplatte trocknen"                                                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0209"} = {errorcode => "6C",                                     subcode => "306",                                                class => "4",           reason => "Flammensignal nach Schlie&szlig;en der Brennstoffversorgung",                                                                                                                                                                                                                                                                                      checklist => "Str&ouml;mt nach Abschaltung Gas-Luft-Gemisch in Brennkammer, obwohl Gasarmatur spannungslos ist?&bull;Elektroden und Anschlusskabel pr&uuml;fen&bull;Kondensatablauf pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Gasarmatur austauschen&bull;Ggf. austauschen&bull;Kondensatablauf reinigen"                                                                                                                                                                                                                                                                                                                      };
	$ErrorCodeList{"0210"} = {errorcode => "6C",                                     subcode => "508",                                                class => "4",           reason => "Flammensignal zu hoch",                                                                                                                                                                                                                                                                                                                            checklist => "„Reset“-Taste bet&auml;tigen und warten, ob die St&ouml;rung behoben ist",                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "Bleibt die St&ouml;rung nach Reset erhalten, ist die Ger&auml;teelektronik SAFe defekt und muss ausgetauscht werden"                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0211"} = {errorcode => "6C",                                     subcode => "509",                                                class => "4",           reason => "Eingang Flammenf&uuml;hler defekt; (Bei der &Uuml;berpr&uuml;fung der Eingangsbeschaltung des Flammenf&uuml;hlers wurde ein Fehler erkannt.)",                                                                                                                                                                                                     checklist => "&Uuml;ber die Bedieneinheit die Anlage in einen Ruhezustand, z. B. Absenkbetrieb, setzen, sodass der Brenner ausgeschaltet ist&bull;Im Ruhezustand in der Bedieneinheit im Men&uuml;; „Monitor“ den F&uuml;hlerstrom pr&uuml;fen&bull;-",                                                                                                                                                                                                                                                         correction => "-&bull;Falls der F&uuml;hlerstrom zu gro&szlig; ist, die Position des Flammenf&uuml;hlers pr&uuml;fen; Evtl. gelangt Fremdlicht an den Flammenf&uuml;hler&bull;Falls ein F&uuml;hlerstrom von ca. 0 μA angezeigt wird, ist die Ger&auml;teelektronik SAFe defekt und muss ausgetauscht werden"                                                                                                  };
	$ErrorCodeList{"0212"} = {errorcode => "6C",                                     subcode => "519",                                                class => "4",           reason => "Kein Flammenabriss/Nachbel&uuml;ftung; (W&auml;hrend der Nachbel&uuml;ftungsphase; ging das Flammensignal nicht aus.)",                                                                                                                                                                                                                            checklist => "Stecker Magnetventil 1. Stufe an Ger&auml;teelektronik SAFe entfernen und im Men&uuml; „Monitor“ der Bedieneinheit den Flammenstrom pr&uuml;fen&bull;Erneut den Flammenstrom im Men&uuml; „Monitor“ der Bedieneinheit pr&uuml;fen",                                                                                                                                                                                                                                                           correction => "Sollte ein Flammenstrom von mehr als ca. 0 μA angezeigt werden, das Magnetventil; der 1. Stufe austauschen&bull;Sollte weiterhin ein Flammenstrom von mehr als ca. 0 μA angezeigt werden, den Flammenf&uuml;hler austauschen"                                                                                                                                                                  };
	$ErrorCodeList{"0213"} = {errorcode => "6C",                                     subcode => "576",                                                class => "4",           reason => "Flammensignal w&auml;hrend Vorbel&uuml;ftung",                                                                                                                                                                                                                                                                                                     checklist => "Flammenf&uuml;hler auf Defekte pr&uuml;fen&bull;Gasarmatur auf Funktion pr&uuml;fen&bull;Wenn Gasarmatur in Ordnung ist",                                                                                                                                                                                                                                                                                                                                                                             correction => "Flammenf&uuml;hler austauschen&bull;Gasarmatur austauschen&bull;Ger&auml;teelektronik SAFe austauschen"                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0214"} = {errorcode => "6C",                                     subcode => "2041",                                               class => "4",           reason => "Fremdlicht im Feuerraum w&auml;hrend Nachbel&uuml;ftung erkannt; Flamme erlischt nicht, nachdem Magnetventil &Ouml;lpumpe geschlossen",                                                                                                                                                                                                            checklist => "Magnetventil &Ouml;lpumpe defekt&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                             correction => "Entriegeln&bull;Magnetventileinsatz &Ouml;lpumpe austauschen"                                                                                                                                                                                                                                                                                                                                    };
	$ErrorCodeList{"0215"} = {errorcode => "6E",                                     subcode => "556",                                                class => "10",          reason => "Hauptflamme zu fr&uuml;h",                                                                                                                                                                                                                                                                                                                         checklist => "Diese Funktion ist zurzeit nicht aktiv.",                                                                                                                                                                                                                                                                                                                                                                                                                                                             correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0216"} = {errorcode => "6F",                                     subcode => "2510",                                               class => "BC",          reason => "W&auml;rmeanforderung blockiert, weil 24 Stunden vorbei",                                                                                                                                                                                                                                                                                          checklist => "Sicherheitsabschaltung nach 24 Stunden",                                                                                                                                                                                                                                                                                                                                                                                                                                                              correction => "Keine Ma&szlig;nahme, W&auml;rmeerzeuger versucht Wiederanlauf"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0217"} = {errorcode => "6L",                                     subcode => "229",                                                class => "10",          reason => "Flamme w&auml;hrend Brennerbetrieb ausgefallen; (Flamme wahrend des Brenner-betriebs ausgefallen.; Der Brenner startet von Neuem, misslingt dieser Start auch, wird der Fehler zu einer blockierenden 6A Storung. Nach weiteren vier Startversuchen wird diese zu einer verriegelnden 6A St&ouml;rung.)",                                          checklist => "Gasanschlussdruck messen, Gas-Luft-Verh&auml;ltnis messen, Gasd&uuml;se pr&uuml;fen; Bei Fl&uuml;ssiggas: Gen&uuml;gend Gasvorrat im Tank oder Tank neu bef&uuml;llt? Kondensatsiphon pr&uuml;fen&bull;&Uuml;berwachungselektrode auf Besch&auml;digung bzw. Verschmutzung pr&uuml;fen&bull;Kondensatablauf pr&uuml;fen",                                                                                                                                                                             correction => "Ist der ben&ouml;tigte Gasanschlussdruck nicht vorhanden, mit dem zust&auml;ndigen Gasversorger R&uuml;cksprache halten; Gas-Luft-Verh&auml;ltnis auf -5 Pa einstellen  F&uuml;r die Gasart korrekte Gasd&uuml;se einsetzen&bull;&Uuml;berwachungselektrode austauschen bzw. reinigen&bull;Kondensatablauf reinigen"                                                                             };
	$ErrorCodeList{"0218"} = {errorcode => "6L",                                     subcode => "512",                                                class => "10",          reason => "Flammenabriss innerhalb Sicherheitszeit",                                                                                                                                                                                                                                                                                                          checklist => "Flammensignal ging innerhalb der Sicherheitszeit aus.",                                                                                                                                                                                                                                                                                                                                                                                                                                               correction => "Keine Ma&szlig;nahme, Ger&auml;teelektronik SAFe versucht Wiederanlauf"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0219"} = {errorcode => "6L",                                     subcode => "513",                                                class => "10",          reason => "Flammenabriss innerhalb Nachz&uuml;ndzeit",                                                                                                                                                                                                                                                                                                        checklist => "Das Flammensignal ging innerhalb der Nachzundzeit aus.",                                                                                                                                                                                                                                                                                                                                                                                                                                              correction => "Keine Ma&szlig;nahme, Ger&auml;teelektronik SAFe versucht Wiederanlauf"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0220"} = {errorcode => "6L",                                     subcode => "514",                                                class => "10",          reason => "Flammenabriss innerhalb Stabilisierungszeit",                                                                                                                                                                                                                                                                                                      checklist => "Das Flammensignal ging innerhalb der Stabilisierungszeit aus.",                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Keine Ma&szlig;nahme, Ger&auml;teelektronik SAFe versucht Wiederanlauf"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0221"} = {errorcode => "6L",                                     subcode => "515",                                                class => "10",          reason => "Flammenabriss in Betrieb; 1. + 2. Stufe",                                                                                                                                                                                                                                                                                                          checklist => "Das Flammensignal ging wahrend des Betriebs von der 1. + 2. Stufe aus.",                                                                                                                                                                                                                                                                                                                                                                                                                              correction => "Keine Ma&szlig;nahme, Ger&auml;teelektronik SAFe versucht Wiederanlauf"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0222"} = {errorcode => "6L",                                     subcode => "516",                                                class => "10",          reason => "Flammenabriss Umschaltung 1. Stufe",                                                                                                                                                                                                                                                                                                               checklist => "Das Flammensignal ging w&auml;hrend der Umschaltung auf die 1. Stufe aus.",                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Keine Ma&szlig;nahme, Ger&auml;teelektronik SAFe versucht Wiederanlauf"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0223"} = {errorcode => "6L",                                     subcode => "517",                                                class => "10",          reason => "Flammenabriss in Betrieb 1. Stufe",                                                                                                                                                                                                                                                                                                                checklist => "Das Flammensignal ging w&auml;hrend des Betriebs von der 1. Stufe aus.",                                                                                                                                                                                                                                                                                                                                                                                                                              correction => "Keine Ma&szlig;nahme, Ger&auml;teelektronik SAFe versucht Wiederanlauf"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0224"} = {errorcode => "6L",                                     subcode => "518",                                                class => "10",          reason => "Flammenabriss Umschaltung 1. + 2. Stufe",                                                                                                                                                                                                                                                                                                          checklist => "Das Flammensignal ging w&auml;hrend der Umschaltung von 1. Stufe auf 2. Stufe bzw. von 2. Stufe auf 1. Stufe aus.",                                                                                                                                                                                                                                                                                                                                                                                   correction => "Keine Ma&szlig;nahme, Ger&auml;teelektronik SAFe versucht Wiederanlauf"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0225"} = {errorcode => "6L",                                     subcode => "548",                                                class => "4",           reason => "Zu viele Brennerstartversuche; (W&auml;hrend einer W&auml;rmeanforderung sind 6 Flammenabrisse aufgetreten.)",                                                                                                                                                                                                                                     checklist => "In der Serviceebene der Bedieneinheit den Men&uuml;punkt „St&ouml;rungsspeicher“ und danach das Untermen&uuml; „blockierende St&ouml;rung“ aufrufen; Sollten ausschlie&szlig;lich St&ouml;rungsmeldungen 6U/511 vorliegen, folgendes Vorgehen w&auml;hlen:&bull;Pr&uuml;fen, ob eine korrekte Brennstoffversorgung sichergestellt ist; Kapitel „Brennstoffversorgung“ in der Serviceanleitung beachten!&bull;In der Serviceebene der Bedieneinheit den Men&uuml;punkt „Relaistest“ aufrufen, hier die Z&uuml;ndung einschalten und pr&uuml;fen, ob die Z&uuml;ndung korrekt arbeitet&bull;Sollte keine Z&uuml;ndung vorhanden sein, folgende Punkte pr&uuml;fen:&bull;Abstand Z&uuml;ndelektroden&bull;Zustand Z&uuml;ndelektroden&bull;Zustand der Z&uuml;ndleitungen zwischen Z&uuml;ndtrafo und Elektroden&bull;Sind die Stecker richtig auf die Z&uuml;ndelektroden aufgesteckt?&bull;Zustand der Verbindungsleitung zwischen Feuerungsautomat und Z&uuml;ndtrafo pr&uuml;fen&bull;Ist der Stecker des Z&uuml;ndtrafos richtig am Feuerungsautomaten aufgesteckt?&bull;In der Serviceebene der Bedieneinheit den Men&uuml;punkt „Monitor“ aufrufen und hier den Flammenstrom pr&uuml;fen; Sollte dieser schwanken oder st&auml;ndig kleiner als die Sollvorgaben aus der Serviceanleitung sein, folgendes Vorgehen w&auml;hlen:&bull;Flammen&uuml;berwachung auf Verschmutzung pr&uuml;fen&bull;Position der Flammen&uuml;berwachung pr&uuml;fen (bei &Ouml;lkesseln Winkelhalter pr&uuml;fen)&bull;Leitungsverbindung zwischen Feuerungsautomat und Flammenf&uuml;hler pr&uuml;fen&bull;Steckverbindung der Flammen&uuml;berwachung am Feuerungsautomaten pr&uuml;fen&bull;D&uuml;sen pr&uuml;fen&bull;&Ouml;labschlussventil bei &Ouml;lkesseln optisch pr&uuml;fen&bull;Mischsystem bei &Ouml;lkesseln pr&uuml;fen&bull;In der Serviceebene der Bedieneinheit den Men&uuml;punkt „Diagnose/Fehler/blockierende Fehler“ (RC35) bzw. „Diagnose/St&ouml;rungsanzeigen/ aktuelle St&ouml;rungen“ (RC300) pr&uuml;fen ob St&ouml;rmeldung 6L/516 vorliegt; sollte diese vorliegen, dann pr&uuml;fen, ob Magnetventile 1 und/oder 2 von der Zuordnung richtig am Feuerungsautomaten aufgesteckt sind&bull;Sollten andere oder auch keine blockierenden St&ouml;rungen vorliegen, folgendes Vorgehen w&auml;hlen:&bull;Bei raumluftunabh&auml;ngiger Betriebsweise den CO-Gehalt in der Zuluft messen, wenn CO gemessen wird, ist das Abgassystem undicht&bull;Pr&uuml;fen, ob eine korrekte Brennstoffversorgung sichergestellt ist; Kapitel „Brennstoffversorgung“ in der Serviceanleitung beachten!&bull;Magnetventil auf Funktion pr&uuml;fen, ob evtl. defekt",correction => "-&bull;Bei St&ouml;rung der Brennstoffversorgung diese beheben&bull;-&bull;Bei Abweichungen folgende Ma&szlig;nahmen durchf&uuml;hren:&bull;Korrekten Abstand herstellen&bull;Wenn Elektroden verbraucht oder besch&auml;digt sind, diese austauschen&bull;Wenn besch&auml;digt, austauschen&bull;Wenn die Kontakte nicht richtig befestigt sind, St&ouml;rung beheben&bull;Wenn besch&auml;digt, austauschen&bull;Ggf. Kontaktprobleme beheben (z. B. Stecker richtig aufstecken)&bull;-&bull;Bei Verschmutzung des Flammenf&uuml;hlers diesen reinigen&bull;Flammenf&uuml;hler richtig positionieren bzw. den Winkelhalter austauschen&bull;Bei defekten Kabeln diese austauschen&bull;Wenn der Stecker nicht richtig aufgesteckt ist, dann den Stecker wieder richtig aufstecken&bull;D&uuml;sen ggf. austauschen&bull;&Ouml;labschlussventil des &Ouml;lvorw&auml;rmers ggf. austauschen&bull;Mischsystem ggf. reinigen&bull;Bei Abweichungen korrigieren,; bei Vertauschungen diese beheben&bull;-&bull;Abgassystem abdichten, ggf. erneut montieren, Dichtheitspr&uuml;fung durchf&uuml;hren&bull;Bei St&ouml;rungen in der Brennstoffversorgung diese beheben&bull;Bei Defekt das Magnetventil austauschen"};
	$ErrorCodeList{"0226"} = {errorcode => "6L",                                     subcode => "553",                                                class => "4",           reason => "Zu viele Flammenabrisse (Wenn direkt hinter einander 15 Flammen abrisse registriert; werden, wird diese Storungs-; meldung erzeugt. ACHTUNG: Ein „Reset“ ist nur am SAFe moglich!)",                                                                                                                                                           checklist => "In der Serviceebene der Bedieneinheit den Men&uuml;punkt „Diagnose/Fehler“ (RC35) bzw. „Diagnose/ St&ouml;rungsanzeigen/aktuelle St&ouml;rungen“ (RC300) aufrufen „St&ouml;rungsspeicher“; Sollten ausschlie&szlig;lich St&ouml;rungsmeldungen 6U/511 vorliegen, folgendes Vorgehen w&auml;hlen:&bull;Pr&uuml;fen, ob eine korrekte Brennstoffversorgung sichergestellt ist; Kapitel „Brennstoffversorgung“ in der Serviceanleitung beachten!&bull;In der Serviceebene der Bedieneinheit den Men&uuml;punkt „Relaistest“ aufrufen, hier die Z&uuml;ndung einschalten und pr&uuml;fen, ob die Z&uuml;ndung korrekt arbeitet&bull;Sollte keine Z&uuml;ndung vorhanden sein, folgende Punkte pr&uuml;fen:&bull;Abstand Z&uuml;ndelektroden pr&uuml;fen&bull;Zustand Z&uuml;ndelektroden pr&uuml;fen&bull;Zustand der Z&uuml;ndleitungen zwischen Z&uuml;ndtrafo und Elektroden pr&uuml;fen&bull;Sind die Stecker richtig auf die Z&uuml;ndelektroden aufgesteckt?&bull;Zustand der Verbindungsleitung zwischen Feuerungsautomaten und Z&uuml;ndtrafo pr&uuml;fen&bull;Ist der Stecker des Z&uuml;ndtrafos richtig am Feuerungsautomat aufgesteckt?&bull;In der Serviceebene der Bedieneinheit den Men&uuml;punkt „Monitor“ aufrufen und hier den Flammenstrom pr&uuml;fen; sollte dieser schwanken oder st&auml;ndig kleiner als die Sollvorgaben aus der Serviceanleitung sein, folgendes Vorgehen w&auml;hlen:&bull;Flammen&uuml;berwachung auf Verschmutzung pr&uuml;fen&bull;Position der Flammen&uuml;berwachung pr&uuml;fen (Winkelhalter pr&uuml;fen)&bull;Leitungsverbindung zwischen Feuerungs-automat und Flammenf&uuml;hler pr&uuml;fen&bull;Steckverbindung der Flammen&uuml;berwachung am Feuerungsautomaten pr&uuml;fen&bull;D&uuml;sen pr&uuml;fen&bull;&Ouml;labschlussventil bei &Ouml;lkesseln optisch pr&uuml;fen&bull;Bei raumluftunabh&auml;ngiger Betriebsweise den CO-Gehalt in der Zuluft messen, wenn CO gemessen wird, ist das Abgassystem undicht&bull;Mischsystem bei &Ouml;lkesseln pr&uuml;fen&bull;Brennereinstellung pr&uuml;fen&bull;In der Serviceebene der Bedieneinheit den Men&uuml;punkt „Diagnose/Fehler/blockierende Fehler“ (RC35) bzw. „Diagnose/St&ouml;rungsanzeigen/ aktuelle St&ouml;rungen“ (RC300) auf-rufen; sollten St&ouml;rungsmeldungen 6L/516 vorliegen, pr&uuml;fen, ob Magnetventile 1 und/oder 2 von der Zuordnung richtig am Feuerungsautomaten aufgesteckt sind&bull;Sollten andere oder auch keine anderen blockierenden Fehler vorliegen, folgendes Vorgehen w&auml;hlen:&bull;Pr&uuml;fen, ob eine korrekte Brennstoffversorgung sichergestellt ist; entsprechendes Kapitel; „Brennstoffversorgung“ in der Serviceanleitung beachten!&bull;Magnetventil auf Funktion pr&uuml;fen, ob eventuell defekt",correction => "-&bull;Bei St&ouml;rung der Brennstoffversorgung diese beheben&bull;-&bull;Bei Abweichungen folgende Ma&szlig;nahmen durchf&uuml;hren:&bull;Korrekten Abstand herstellen&bull;Wenn Elektroden verbraucht oder besch&auml;digt sind, diese austauschen&bull;Wenn besch&auml;digt, austauschen&bull;Wenn die Kontakte nicht richtig befestigt sind, St&ouml;rung beheben&bull;Wenn besch&auml;digt, austauschen&bull;Ggf. Kontaktprobleme beheben (z. B. Stecker richtig aufstecken )&bull;-&bull;Bei Verschmutzung des Flammenf&uuml;hlers diesen reinigen&bull;Flammenf&uuml;hler richtig positionieren bzw. den Winkelhalter austauschen&bull;Bei defekten Kabeln diese austauschen&bull;Sollte der Stecker nicht richtig aufgesteckt sein, den Stecker wieder richtig aufstecken&bull;D&uuml;sen ggf. austauschen&bull;&Ouml;labschlussventil des &Ouml;lvorw&auml;rmers ggf. austauschen&bull;Abgassystem abdichten, ggf. erneut montieren, Dichtheitspr&uuml;fung durchf&uuml;hren&bull;Mischsystem ggf. reinigen&bull;Bei Abweichungen korrigieren&bull;Bei Vertauschungen diese beheben&bull;-&bull;Bei St&ouml;rung der &Ouml;lversorgung, diese beheben&bull;Bei Defekt austauschen"};
	$ErrorCodeList{"0227"} = {errorcode => "6L",                                     subcode => "555",                                                class => "10",          reason => "Flammenabriss innerhalb Stabilisierung Z&uuml;ndgas",                                                                                                                                                                                                                                                                                              checklist => "Kabelverbindung zur &Uuml;berwachungselektrode pr&uuml;fen&bull;&Uuml;berwachungselektrode auf Verschmutzung pr&uuml;fen&bull;Position der &Uuml;berwachungselektrode pr&uuml;fen",                                                                                                                                                                                                                                                                                                                   correction => "Defekte Kabelverbindung wiederherstellen&bull;Verschmutzung beheben bzw. Elektrode austauschen&bull;&Uuml;berwachungselektrode korrekt positionieren"                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0228"} = {errorcode => "6L",                                     subcode => "557",                                                class => "10",          reason => "Flammenabriss bei Hauptgas ein",                                                                                                                                                                                                                                                                                                                   checklist => "Feuerungsautomat entriegeln Gasanschlussdruck pr&uuml;fen; Feuerungsautomat/Ger&auml;teelektronik SAFe pr&uuml;fen; D&uuml;sendruck laut technischer Dokumente des Kessels pr&uuml;fen",                                                                                                                                                                                                                                                                                                              correction => "Feuerungsautomat entriegeln; Bei Abweichungen ggf. mit Gasversorger in Verbindung setzen; Feuerungsautomat/Ger&auml;teelektronik SAFe auswechseln; D&uuml;sendruck ggf. richtig einstellen"                                                                                                                                                                                                      };
	$ErrorCodeList{"0229"} = {errorcode => "6L",                                     subcode => "561",                                                class => "4",           reason => "Spannungsunterbrechung w&auml;hrend Brennerstart (Wenn der Brennerautomat 5-mal hintereinander wahrend des ersten Brenner anlaufs direkt nach dem; „Power Up“ ausgeschaltet wurde, wird diese Meldung erzeugt. Tritt; die Meldung mehr als 5-mal nach -; einander auf, wird der Fehler ein; verriegelnder Fehler.)",                           checklist => "-&bull;Die 230-V-Spannungsversorgung zum Regelger&auml;t bzw. Feuerungsautomat pr&uuml;fen; Evtl. liegt ein Wackelkontakt vor und der Feuerungsautomat bekommt st&auml;ndig eine Spannungsunterbrechung&bull;Eventuell liegen St&ouml;rungen im Rahmen von EMV vor; daher das Umfeld der Anlage auf m&ouml;gliche St&ouml;rquellen pr&uuml;fen&bull;Sollten keine der genannten Probleme vorliegen, ist vermutlich der Feuerungsautomat defekt",                                                      correction => "Feuerungsautomat entriegeln&bull;Problem in der Spannungsversorgung beheben&bull;Ursachen f&uuml;r EMV-St&ouml;rungen beheben&bull;Feuerungsautomat (SAFe) austauschen"                                                                                                                                                                                                                          };
	$ErrorCodeList{"0230"} = {errorcode => "6L",                                     subcode => "587",                                                class => "10",          reason => "Flammenabriss Stabilisierung Teillast",                                                                                                                                                                                                                                                                                                            checklist => "Gasanschlussdruck messen&bull;Gasarmatur auf Funktion pr&uuml;fen&bull;Ionisationsstrom in der Serviceebene der Bedieneinheit pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                           correction => "F&uuml;r korrekten Gasanschlussdruck sorgen&bull;Bei Defekt Gasarmatur austauschen&bull;Bei Abweichungen Z&uuml;nd-/&Uuml;berwachungselektrode austauschen"                                                                                                                                                                                                                                      };
	$ErrorCodeList{"0231"} = {errorcode => "6P",                                     subcode => "269",                                                class => "4",           reason => "Flammen&uuml;berwachung",                                                                                                                                                                                                                                                                                                                          checklist => "BCM/Kesselidentifikationsmodul defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                                               correction => "BCM bzw. Kesselidentifikationsmodul durch Buderus Service austauschen lassen"                                                                                                                                                                                                                                                                                                                    };
	$ErrorCodeList{"0232"} = {errorcode => "6P",                                     subcode => "363",                                                class => "8",           reason => "Systemst&ouml;rung Ger&auml;teelektronik / Basiscontroller",                                                                                                                                                                                                                                                                                       checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Steuerger&auml;t/ Feuerungsautomat austauschen"                                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0233"} = {errorcode => "6U",                                     subcode => "511",                                                class => "10",          reason => "Keine Flamme innerhalb Sicherheitszeit",                                                                                                                                                                                                                                                                                                           checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Keine Ma&szlig;nahme, Ger&auml;teelektronik SAFe versucht Wiederanlauf&bull;Ggf. entriegeln/reseten&bull;Hinweis: siehe auch St&ouml;rung 6L/548 und 6L/553"                                                                                                                                                                                                                                     };
	$ErrorCodeList{"0234"} = {errorcode => "6Y",                                     subcode => "510",                                                class => "4",           reason => "Fremdlicht Vorbel&uuml;ftung",                                                                                                                                                                                                                                                                                                                     checklist => "Position des Flammenf&uuml;hlers pr&uuml;fen, ob evtl. Fremdlicht einf&auml;llt&bull;Mit abgezogenem und mit der Hand abgedunkeltem Flammenf&uuml;hler Startversuch durchf&uuml;hren und pr&uuml;fen, ob St&ouml;rungsmeldung 6Y/510 angezeigt wird&bull;Das Magnetventil ist undicht, wodurch die Flamme nicht verlischt (im Sichtfenster am Brenner optisch pr&uuml;fen)&bull;Mit abgezogenem und mit der Hand abgedunkeltem Flammenf&uuml;hler Startversuch durchf&uuml;hren und pr&uuml;fen, ob St&ouml;rungsmeldung 6U/511 angezeigt wird; (Die Ger&auml;teelektronik SAFe versucht einen Wiederanlauf) Sollte dies der Fall sein, erkennt der Flammenf&uuml;hler Fremdlicht&bull;Wackelkontakt Flammenf&uuml;hler oder Stecker Ger&auml;teelektronik SAFe",correction => "Wenn erforderlich, F&uuml;hler richtig positionieren&bull;Der Flammenf&uuml;hler ist defekt, wenn immer noch die St&ouml;rungsmeldung 6Y/510 erscheint Der Flammenf&uuml;hler muss ausgetauscht werden&bull;Magnetventil austauschen&bull;Evtl. ist das Z&uuml;ndelement nicht richtig positioniert und sollte richtig eingebaut werden&bull;Flammenf&uuml;hler oder Ger&auml;teelektronik SAFe austauschen"};
	$ErrorCodeList{"0235"} = {errorcode => "6Y",                                     subcode => "2039",                                               class => "4",           reason => "Fremdlicht im Feuerraum w&auml;hrend Vorbel&uuml;ftung erkannt; Flamme wurde zu einem unzul&auml;ssigen Zeitpunkt erkannt",                                                                                                                                                                                                                        checklist => "Magnetventil undicht&bull;Einspritzventil undicht&bull;Ionisationsstromsensor besch&auml;digt&bull;Feuerungsautomat besch&auml;digt",                                                                                                                                                                                                                                                                                                                                                                 correction => "Magnetventil auf Dichtheit pr&uuml;fen, ggf. austauschen&bull;Einspritzventil auf Dichtheit pr&uuml;fen, ggf. austauschen&bull;&Uuml;berwachungselektrode pr&uuml;fen, ggf. austauschen&bull;Feuerungsautomat pr&uuml;fen, ggf. austauschen"                                                                                                                                                     };
	$ErrorCodeList{"0236"} = {errorcode => "7A",                                     subcode => "550",                                                class => "10",          reason => "Netzspannung zu niedrig",                                                                                                                                                                                                                                                                                                                          checklist => "Netzspannung pr&uuml;fen. Die Spannung darf; 187 V AC nicht unterschreiten (GB402 195 V AC)",                                                                                                                                                                                                                                                                                                                                                                                                         correction => "F&uuml;r eine korrekte Spannungsversorgung sorgen"                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0237"} = {errorcode => "7A",                                     subcode => "551",                                                class => "10",          reason => "Netzspannung kurzzeitig unterbrochen",                                                                                                                                                                                                                                                                                                             checklist => "Netzzuleitung auf evtl. Wackelkontakte pr&uuml;fen&bull;Verdrahtung und richtige Kontaktierung des Netzsteckers an Regelger&auml;t bzw. Ger&auml;teelektronik SAFe oder Fremdbrennermodul pr&uuml;fen",                                                                                                                                                                                                                                                                                               correction => "Ggf. Kontaktprobleme beheben&bull;Ggf. Kontaktprobleme beheben"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0238"} = {errorcode => "7C",                                     subcode => "231",                                                class => "4",           reason => "Netzspannungsunterbrechung nach St&ouml;rungsanzeige",                                                                                                                                                                                                                                                                                             checklist => "Die Netzspannung wurde nach einer St&ouml;rungsmeldung unterbrochen und wieder eingeschaltet.",                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Taste „Reset“ bet&auml;tigen"                                                                                                                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0239"} = {errorcode => "7H",                                     subcode => "328",                                                class => "4",           reason => "Netzspannung kurzzeitig unterbrochen; (Der UBA &uuml;berpr&uuml;ft die eingehende Netzspannung.; Wird ein Spannungsabfall registriert, geht der Kessel auf St&ouml;rung.)",                                                                                                                                                                        checklist => "Netzspannung pr&uuml;fen Lag ein Stromausfall vor?",                                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Eventuell Trafo pr&uuml;fen, ggf. austauschen"                                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0240"} = {errorcode => "7H",                                     subcode => "356",                                                class => "4",           reason => "Versorgungsspannung f&uuml;r den W&auml;rmeerzeuger ist zu niedrig",                                                                                                                                                                                                                                                                               checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Versorgungsspannung von mindestens 196 VAC herstellen"                                                                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0241"} = {errorcode => "7L",                                     subcode => "261",                                                class => "4",           reason => "Zeitfehler bei erster Sicherheitszeit",                                                                                                                                                                                                                                                                                                            checklist => "Kontakt zwischen Ger&auml;teelektronik UBA u. Montagefu&szlig; sowie Basiscontroller BC25 und der Grundplatte und allen sonstigen Steckverbindungen pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                     correction => "Kontaktproblem beseitigen, ggf. Ger&auml;teelektronik UBA bzw. Ger&auml;teelektronik BC25 austauschen"                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0242"} = {errorcode => "7L",                                     subcode => "280",                                                class => "4",           reason => "Zeitfehler bei Wiederanlaufversuch",                                                                                                                                                                                                                                                                                                               checklist => "Kontakt zwischen Ger&auml;teelektronik UBA u. Montagefu&szlig; sowie Basiscontroller BC25 und der Grundplatte und allen sonstigen Steckverbindungen pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                     correction => "Kontaktproblem beseitigen, ggf. Ger&auml;teelektronik UBA bzw. Ger&auml;teelektronik BC25 austauschen"                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0243"} = {errorcode => "7P",                                     subcode => "549",                                                class => "4",           reason => "Sicherheitskette offen (SI 17-18)",                                                                                                                                                                                                                                                                                                                checklist => "Den Wasserdruck der Anlage pr&uuml;fen. Der Wasserdruck darf 0,8 bar nicht unterschreiten&bull;Pr&uuml;fen, ob der Steckverbinder des Druckw&auml;chters richtig aufgesteckt ist&bull;Eingangsspannung Druckw&auml;chter pr&uuml;fen&bull;Ausgangsspannung Druckw&auml;chter pr&uuml;fen&bull;Pr&uuml;fen, ob der Gasdruckw&auml;chter geschaltet hat oder einen Verdrahtungsfehler vorliegt&bull;Angeschlossene Sicherheitsger&auml;te auf Ausl&ouml;sung pr&uuml;fen (Sicherheitskette SI17-18)&bull;Regelger&auml;t defekt",correction => "Wasser bis mindestens 1 bar Wasserdruck nachf&uuml;llen&bull;-&bull;Gasdruckw&auml;chter und Gasdruck pr&uuml;fen&bull;Angeschlossene Sicherheitsger&auml;te (z.B. AbgasSTB)  entriegeln, St&ouml;rungsursache beseitigen&bull;Regelger&auml;t austauschen"                                                                                                                                      };
	$ErrorCodeList{"0244"} = {errorcode => "7U",                                     subcode => "2052",                                               class => "10",          reason => "Maximale Einschaltdauer Z&uuml;ndtrafo &uuml;berschritten",                                                                                                                                                                                                                                                                                        checklist => "Zu viele Wiederanl&auml;ufe des Brenners haben die Einschaltdauer des Z&uuml;ndtrafos &uuml;berschritten&bull;Fehlerhafte &Ouml;lversorgung&bull;Fehlerhafte Brennerkomponenten&bull;Feuerungsautomat besch&auml;digt&bull;Hinweis: siehe auch Beschreibung St&ouml;rungscode 6L/548",                                                                                                                                                                                                                correction => "Fehler in der &Ouml;lversorgung pr&uuml;fen, ggf. beseitigen&bull;-&bull;Brennerkomponenten pr&uuml;fen, ggf. austauschen&bull;Feuerungsautomat pr&uuml;fen, ggf. austauschen&bull;-"                                                                                                                                                                                                            };
	$ErrorCodeList{"0245"} = {errorcode => "8L",                                     subcode => "534",                                                class => "10",          reason => "Kein Gasdruck oder zus&auml;tzlicher Abgasdruckbegrenzer abgeschaltet; (Obwohl das Magnetventil 1 ge&ouml;ffnet haben m&uuml;sste, steht kein Gasdruck an.; Brenner macht nach-einander 3 Startversuche, dann wartet er eine Stunde, um dann erneut 3 Startversuche durchzu-; f&uuml;hren.)",                                                      checklist => "Pr&uuml;fen, ob der Gashahn ge&ouml;ffnet ist&bull;Pr&uuml;fen, ob Gasdruck vorhanden ist&bull;Pr&uuml;fen, ob Abgasdruckbegrenzer geschaltet hat&bull;Elektrode pr&uuml;fen&bull;Z&uuml;ndtrafo pr&uuml;fen&bull;Gasfilter auf Verschmutzung pr&uuml;fen",                                                                                                                                                                                                                                           correction => "Ggf. Gasarmatur austauschen&bull;Gasdruck messen&bull;Abgasdruckbegrenzer entriegeln Abgasweg auf Verstopfung pr&uuml;fen&bull;Ggf. Elektrode austauschen&bull;Ggf. Z&uuml;ndtrafo austauschen&bull;Ggf. Gasfilter austauschen"                                                                                                                                                                  };
	$ErrorCodeList{"0246"} = {errorcode => "8L",                                     subcode => "579",                                                class => "4",           reason => "Kein Gasdruck; (Obwohl das Magnetventil 1 ge&ouml;ffnet haben m&uuml;sste, steht kein Gasdruck an. Der Brenner macht hinter einander 3 Startversuche, dann; wartet er eine Stunde, um erneut; 3 Startversuche durchzuf&uuml;hren.)",                                                                                                               checklist => "Pr&uuml;fen, ob der Gashahn ge&ouml;ffnet ist",                                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Gasmagnetventil 1 austauschen"                                                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0247"} = {errorcode => "8U",                                     subcode => "364",                                                class => "L",           reason => "Magnetventil EV2 undicht",                                                                                                                                                                                                                                                                                                                         checklist => "Gasarmatur auf Verschmutzung pr&uuml;fen&bull;Gasfilter vorhanden?",                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Gasarmatur austauschen&bull;-"                                                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0248"} = {errorcode => "8U",                                     subcode => "365",                                                class => "L",           reason => "Magnetventil EV1 undicht",                                                                                                                                                                                                                                                                                                                         checklist => "Gasarmatur auf Verschmutzung pr&uuml;fen&bull;Gasfilter vorhanden?",                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Gasarmatur austauschen&bull;-"                                                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0249"} = {errorcode => "8U",                                     subcode => "581",                                                class => "4",           reason => "Magnetventil 2 undicht",                                                                                                                                                                                                                                                                                                                           checklist => "Gasarmatur auf Verschmutzung pr&uuml;fen Gasfilter vorhanden?",                                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Gasarmatur austauschen"                                                                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0250"} = {errorcode => "8U",                                     subcode => "584",                                                class => "10",          reason => "Umschaltmodul keine R&uuml;ckmeldung",                                                                                                                                                                                                                                                                                                             checklist => "Externe Komponenten m&uuml;ssen &uuml;ber die Anschlussklemme 7 eine R&uuml;ckmeldung (230 V AC) geben; diese R&uuml;ckmeldung fehlt;; eventuell ist ein Draht lose, ein Kabel besch&auml;digt oder die externe Komponente defekt",                                                                                                                                                                                                                                                                   correction => "Defektes Kabel austauschen bzw. Steckverbindung richtig herstellen oder auch defekte externe Komponente austauschen"                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0251"} = {errorcode => "8U",                                     subcode => "591",                                                class => "4",           reason => "Abgassperrklappe &ouml;ffnet nicht",                                                                                                                                                                                                                                                                                                               checklist => "Die Verdrahtung zwischen Universalschaltmodul und Abgassperrklappe pr&uuml;fen&bull;-&bull;Abgassperrklappenmotor defekt&bull;Universalschaltmodul defekt",                                                                                                                                                                                                                                                                                                                                           correction => "Bei Fehlverdrahtung den Verdrahtungsfehler korrigieren&bull;Bei Defekt Kabel ersetzen&bull;Abgassperrklappenmotor austauschen&bull;Modul austauschen"                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0252"} = {errorcode => "8U",                                     subcode => "592",                                                class => "4",           reason => "Abgassperrklappe dauerhaft ge&ouml;ffnet",                                                                                                                                                                                                                                                                                                         checklist => "Die Verdrahtung zwischen Universalschaltmodul und Abgassperrklappe pr&uuml;fen&bull;Verbindungsleitung zwischen Universalschaltmodul und Abgassperrklappe pr&uuml;fen&bull;Abgassperrklappenmotor defekt&bull;Universalschaltmodul defekt",                                                                                                                                                                                                                                                           correction => "Bei Fehlverdrahtung den Verdrahtungsfehler korrigieren&bull;Bei Defekt Kabel ersetzen&bull;Abgassperrklappenmotor austauschen&bull;Modul austauschen"                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0253"} = {errorcode => "8U",                                     subcode => "593",                                                class => "4",           reason => "Br&uuml;cke am Eingang K&uuml;chenl&uuml;fter (Dunstabzugshaube) fehlt",                                                                                                                                                                                                                                                                           checklist => "K&uuml;chenl&uuml;fter &uuml;berpr&uuml;fen; Verbindung K&uuml;chenl&uuml;fter zu Modul LM10 &uuml;berpr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                   correction => "K&uuml;chenl&uuml;fter abschalten, Installation korrigieren, Br&uuml;cke einsetzen"                                                                                                                                                                                                                                                                                                              };
	$ErrorCodeList{"0254"} = {errorcode => "8Y",                                     subcode => "232",                                                class => "20",          reason => "W&auml;rmeerzeuger durch externen Schaltkontakt verriegelt (EV bzw. I3)",                                                                                                                                                                                                                                                                          checklist => "Pr&uuml;fen, ob ein externer Schaltkontakt, z. B. Fu&szlig;bodenthermostat oder Festbrennstoff-Kessel, ge&ouml;ffnet hat&bull;Kabelbaum zwischen Ger&auml;teelektronik UBA und der Klemmleiste pr&uuml;fen",                                                                                                                                                                                                                                                                                          correction => "Kein Handlungsbedarf, wenn der Zustand korrekt und gew&uuml;nscht ist&bull;Ggf. Kabelbaum oder entsprechenden Teil davon ersetzen"                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0255"} = {errorcode => "8Y",                                     subcode => "572",                                                class => "10",          reason => "Externe Sperrung durch Klemme EV",                                                                                                                                                                                                                                                                                                                 checklist => "Pr&uuml;fen, ob evtl. ein Kabel von den Anschlussklemmen EV defekt ist; Ansonsten liegt keine St&ouml;rung vor, da die Verriegelung eine gewollte Funktion ist",                                                                                                                                                                                                                                                                                                                                      correction => "Bei defektem Kabel oder losen Dr&auml;hten den Defekt beheben"                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0256"} = {errorcode => "8Y",                                     subcode => "583",                                                class => "10",          reason => "Umschaltmodul externe Verriegelung",                                                                                                                                                                                                                                                                                                               checklist => "Pr&uuml;fen, ob evtl. ein Kabel von den Anschlussklemmen EV defekt ist; Ansonsten liegt keine St&ouml;rung vor, da die Verriegelung eine gewollte Funktion ist",                                                                                                                                                                                                                                                                                                                                      correction => "Bei defektem Kabel oder losen Dr&auml;hten den Defekt beheben"                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0257"} = {errorcode => "8Y",                                     subcode => "589",                                                class => "10",          reason => "W&auml;rmeerzeuger durch externen Schaltkontakt verriegelt (BRM10)",                                                                                                                                                                                                                                                                               checklist => "Pr&uuml;fen, ob evtl. ein Kabel des BRM10 von den Anschlussklemmen 15/16 defekt ist; Ansonsten liegt keine St&ouml;rung vor, da die Verriegelung eine gewollte Funktion ist",                                                                                                                                                                                                                                                                                                                         correction => "Bei defektem Kabel oder losen Dr&auml;hten den Defekt beheben"                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0258"} = {errorcode => "8Y",                                     subcode => "590",                                                class => "4",           reason => "Druckschalter w&auml;hrend Betrieb ge&ouml;ffnet",                                                                                                                                                                                                                                                                                                 checklist => "Unterbrechung Zuluftdruckschalter im Betrieb&bull;Unterbrechung Abgasdruckschalter im Betrieb&bull;Unterbrechung am Gaseingangsdruckschalter&bull;Unterbrechung an Ger&auml;teelektronik SAFe Druckschalter&bull;Unterbrechung am Gasausgangsdruckschalter",                                                                                                                                                                                                                                          correction => "Zuluftweg auf freien Durchgang pr&uuml;fen&bull;Abgasweg auf freien Durchgang pr&uuml;fen&bull;Einstellung des Druckschalters pr&uuml;fen (Erdgas Soll= 3 inch wc, FG Soll = 7 inch wc), Gas-Versorgungsdruck > 3,5 inch wc auch bei 100 %-Modulation sicherstellen&bull;Gasarmatur austauschen (Nulldruckregler defekt)&bull;Gasarmatur austauschen (Nulldruckregler defekt)"                   };
	$ErrorCodeList{"0259"} = {errorcode => "8Y",                                     subcode => "2514",                                               class => "BC",          reason => "W&auml;rmeanforderung blockiert wegen UM10",                                                                                                                                                                                                                                                                                                       checklist => "W&auml;rmeanforderung blockiert durch Umschaltmodul",                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0260"} = {errorcode => "92",                                     subcode => "84",                                                 class => "B",           reason => "R&uuml;cklauftemperaturf&uuml;hler defekt",                                                                                                                                                                                                                                                                                                        checklist => "R&uuml;cklauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                                      correction => "F&uuml;hler wieder richtig anschlie&szlig;en, ggf. austauschen"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0261"} = {errorcode => "92",                                     subcode => "85",                                                 class => "B",           reason => "R&uuml;cklauftemperaturf&uuml;hler defekt",                                                                                                                                                                                                                                                                                                        checklist => "R&uuml;cklauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                                      correction => "F&uuml;hler wieder richtig anschlie&szlig;en, ggf. austauschen"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0262"} = {errorcode => "93",                                     subcode => "86",                                                 class => "B",           reason => "Vorlauftemperaturf&uuml;hler defekt",                                                                                                                                                                                                                                                                                                              checklist => "Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                                            correction => "F&uuml;hler wieder richtig anschlie&szlig;en, ggf. austauschen"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0263"} = {errorcode => "93",                                     subcode => "87",                                                 class => "B",           reason => "Vorlauftemperaturf&uuml;hler defekt",                                                                                                                                                                                                                                                                                                              checklist => "Vorlauftemperaturf&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                                            correction => "F&uuml;hler wieder richtig anschlie&szlig;en, ggf. austauschen"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0264"} = {errorcode => "94",                                     subcode => "88",                                                 class => "B",           reason => "H&ouml;here R&uuml;cklauftemperatur als Vorlauftemperatur",                                                                                                                                                                                                                                                                                        checklist => "Pr&uuml;fen, ob Vorund R&uuml;cklauf vertauscht sind&bull;DIP-Schalter-Stellung auf der Leiterplatte auf dem W&auml;rmepumpenmodul pr&uuml;fen (Normalbetrieb)&bull;Die F&uuml;hler und die Position der F&uuml;hler pr&uuml;fen",                                                                                                                                                                                                                                                                    correction => "Vorund R&uuml;cklauf richtig anschlie&szlig;en&bull;DIP-Schalter auf Normalbetrieb stellen&bull;Die F&uuml;hler richtig positionieren"                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0265"} = {errorcode => "95",                                     subcode => "90",                                                 class => "B",           reason => "Au&szlig;entemperaturf&uuml;hler defekt",                                                                                                                                                                                                                                                                                                          checklist => "F&uuml;hler anhand der Spannungsund Widerstandswerte pr&uuml;fen&bull;F&uuml;hleranschluss pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                              correction => "Bei Defekt F&uuml;hler austauschen&bull;F&uuml;hler wieder richtig anschlie&szlig;en"                                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0266"} = {errorcode => "96",                                     subcode => "89",                                                 class => "10",          reason => "Vorlaufoder R&uuml;cklauftemperatur au&szlig;erhalb zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                                                     checklist => "Vorlaufoder R&uuml;cklauftemperatur am HybridManager au&szlig;erhalb des zul&auml;ssigen Bereichs Der W&auml;rmeerzeuger erw&auml;rmt zun&auml;chst das Wasser, bevor die Luft-Wasser-W&auml;rmepumpe gestartet wird",                                                                                                                                                                                                                                                                                correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0267"} = {errorcode => "97",                                     subcode => "91",                                                 class => "4",           reason => "St&ouml;rung Au&szlig;eneinheit/ Wasserdurchfluss",                                                                                                                                                                                                                                                                                                checklist => "St&ouml;rung an der W&auml;rmepumpe/Wasserdurchfluss gest&ouml;rt&bull;LED-Signal am Hybridmodul pr&uuml;fen&bull;Durchfluss der Luft-Wasser-W&auml;rmepumpe pr&uuml;fen, Filter pr&uuml;fen&bull;Str&ouml;mungsschalter pr&uuml;fen&bull;System pr&uuml;fen",                                                                                                                                                                                                                                        correction => "-&bull;Ggf. Filter reinigen bzw. ersetzen&bull;Ggf. ersetzen&bull;-"                                                                                                                                                                                                                                                                                                                             };
	$ErrorCodeList{"0268"} = {errorcode => "9A",                                     subcode => "235",                                                class => "8",           reason => "Versionskonflikt Ger&auml;teelektronik/Kesselidentifikationsmodul",                                                                                                                                                                                                                                                                                checklist => "Softwareversion im Kessel: IdentifikationsModul/ Kodierstecker ist nicht kompatibel zum Steuerger&auml;t/Feuerungsautomat",                                                                                                                                                                                                                                                                                                                                                                           correction => "Kesselidentifikationsmodul/Feuerungsautomat austauschen"                                                                                                                                                                                                                                                                                                                                         };
	$ErrorCodeList{"0269"} = {errorcode => "9A",                                     subcode => "361",                                                class => "8",           reason => "Kesselidentifikationsmodul oder Ger&auml;teelektronik St&ouml;rung",                                                                                                                                                                                                                                                                               checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Kessel Idendifikations Modul/ Kodierstecker korrekt aufstecken bzw. austauschen"                                                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0270"} = {errorcode => "9H",                                     subcode => "237",                                                class => "4",           reason => "Systemst&ouml;rung",                                                                                                                                                                                                                                                                                                                               checklist => "Kesselidentifikationsmodul KIM oder Ger&auml;teelektronik UBA defekt oder Kurzschluss im Anschlusskabel der Gasarmatur",                                                                                                                                                                                                                                                                                                                                                                              correction => "Anschlusskabel der Gasarmatur austauschen, Ger&auml;teelektronik UBA austauschen oder Kesselidentifikationsmodul durch Buderus Service austauschen lassen"                                                                                                                                                                                                                                       };
	$ErrorCodeList{"0271"} = {errorcode => "9H",                                     subcode => "267",                                                class => "4",           reason => "Systemst&ouml;rung",                                                                                                                                                                                                                                                                                                                               checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Ger&auml;teelektronik UBA austauschen"                                                                                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0272"} = {errorcode => "9H",                                     subcode => "272",                                                class => "4",           reason => "Systemst&ouml;rung",                                                                                                                                                                                                                                                                                                                               checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Ger&auml;teelektronik UBA austauschen (Ger&auml;teelektronik mit aktuellerem Softwarestand einsetzen)"                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0273"} = {errorcode => "9L",                                     subcode => "230",                                                class => "10",          reason => "St&ouml;rung Regelventil",                                                                                                                                                                                                                                                                                                                         checklist => "Spule des Regelventils unterbrochen, Kabel zum Regelventil unterbrochen, Regelventilansteuerung defekt",                                                                                                                                                                                                                                                                                                                                                                                              correction => "Bei defektem Kabel oder losen Dr&auml;hten den Defekt beheben"                                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0274"} = {errorcode => "9L",                                     subcode => "234",                                                class => "4",           reason => "Elektrische St&ouml;rung Gasarmatur",                                                                                                                                                                                                                                                                                                              checklist => "Anschlusskabel der Gasarmatur pr&uuml;fen&bull;Gasarmatur defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "Anschlusskabel austauschen und Reset nach dem Austausch&bull;Gasarmatur austauschen und Reset nach dem Austausch"                                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0275"} = {errorcode => "9L",                                     subcode => "238",                                                class => "8",           reason => "Ger&auml;teelektronik ist defekt",                                                                                                                                                                                                                                                                                                                 checklist => "Relais f&uuml;r die Ansteuerung der Gasarmatur defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                               correction => "Feuerungsautomat austauschen"                                                                                                                                                                                                                                                                                                                                                                    };
	$ErrorCodeList{"0276"} = {errorcode => "9P",                                     subcode => "239",                                                class => "4",           reason => "Systemst&ouml;rung",                                                                                                                                                                                                                                                                                                                               checklist => "Ger&auml;teelektronik UBA defekt oder Kurzschluss im Anschlusskabel der Gasarmatur&bull;Kesselidentifikationsmodul nicht richtig aufgesteckt oder defekt",                                                                                                                                                                                                                                                                                                                                            correction => "Anschlusskabel der Gasarmatur austauschen, ggf. Ger&auml;teelektronik UBA austauschen&bull;Kesselidentifikationsmodul richtig aufstecken oder durch Buderus Service austauschen lassen"                                                                                                                                                                                                          };
	$ErrorCodeList{"0277"} = {errorcode => "9U",                                     subcode => "230",                                                class => "8",           reason => "Interne St&ouml;rung Ger&auml;teelektronik, Modulationsspule St&ouml;rung",                                                                                                                                                                                                                                                                        checklist => "Modulationsspule defekt, oder Dr&auml;hte an der Spule lose",                                                                                                                                                                                                                                                                                                                                                                                                                                         correction => "Ionisation pr&uuml;fen (au&szlig;erhalb Grenzwert), Steuerger&auml;t austauschen"                                                                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0278"} = {errorcode => "9U",                                     subcode => "233",                                                class => "4",           reason => "St&ouml;rung Kesselidentifikationsmodul oder Ger&auml;teelektronik (KIM oder UBA defekt)",                                                                                                                                                                                                                                                         checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Ger&auml;teelektronik UBA austauschen oder Kesselidentifikationsmodul durch Buderus Service austauschen lassen; KesselidentifikationsmodulSteckverbindung wiederherstellen; Ggf. Kesselidentifikationsmodul, als letzte Ma&szlig;nahme Kabelbaum oder Ger&auml;teelektronik UBA austauschen"                                                                                                     };
	$ErrorCodeList{"0279"} = {errorcode => "9Y",                                     subcode => "500",                                                class => "8",           reason => "Interne St&ouml;rung Fremdbrennermodul, keine Spannung Sicherheitsrelais",                                                                                                                                                                                                                                                                         checklist => "„Reset“-Taste bet&auml;tigen und warten, ob die St&ouml;rung behoben ist",                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "Bleibt die St&ouml;rung nach Reset erhalten, ist der Feuerungsautomat oder Fremdbrennermodul defekt und muss ausgetauscht werden"                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0280"} = {errorcode => "9Y",                                     subcode => "501",                                                class => "8",           reason => "Interne St&ouml;rung Fremdbrennermodul, Sicherheitsrelais h&auml;ngt",                                                                                                                                                                                                                                                                             checklist => "„Reset“-Taste bet&auml;tigen und warten, ob die St&ouml;rung behoben ist",                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "Bleibt die St&ouml;rung nach Reset erhalten, ist der Feuerungsautomat oder Fremdbrennermodul defekt und muss ausgetauscht werden"                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0281"} = {errorcode => "9Y",                                     subcode => "502",                                                class => "4",           reason => "Interne St&ouml;rung Ger&auml;teelektronik SAFe/Fremdbrennermodul, keine Spannung Brennstoffrelais 1",                                                                                                                                                                                                                                             checklist => "„Reset“-Taste bet&auml;tigen und warten, ob die St&ouml;rung behoben ist",                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "Bleibt die St&ouml;rung nach Reset erhalten, ist der Feuerungsautomat oder das Fremdbrennermodul defekt und muss ausgetauscht werden"                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0282"} = {errorcode => "9Y",                                     subcode => "503",                                                class => "4",           reason => "Interne St&ouml;rung Ger&auml;teelektronik SAFe/Fremdbrennermodul, Brennstoffrelais 1 h&auml;ngt",                                                                                                                                                                                                                                                 checklist => "„Reset“-Taste bet&auml;tigen und warten, ob die St&ouml;rung behoben ist",                                                                                                                                                                                                                                                                                                                                                                                                                        correction => "Bleibt die St&ouml;rung nach Reset erhalten, ist der Feuerungsautomat oder das Fremdbrennermodul defekt und muss ausgetauscht werden"                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0283"} = {errorcode => "9Y",                                     subcode => "2000",                                               class => "4",           reason => "Interne St&ouml;rung Feuerungsautomat, keine Spannung Z&uuml;ndtraforelais; hinter dem Z&uuml;ndtraforelais wird keine Spannung gemessen, obwohl das Relais eingeschaltet ist",                                                                                                                                                                    checklist => "St&ouml;rung im Feuerungsautomaten&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Entriegeln&bull;Anlage f&uuml;r 30 Sekunden spannungsfrei schalten&bull;Feuerungsautomat austauschen"                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0284"} = {errorcode => "9Y",                                     subcode => "2001",                                               class => "4",           reason => "Interne St&ouml;rung Feuerungsautomat, Z&uuml;ndtraforelais h&auml;ngt; hinter dem Z&uuml;ndtraforelais wird eine Spannung gemessen, obwohl das Relais abgeschaltet ist",                                                                                                                                                                          checklist => "St&ouml;rung im Feuerungsautomaten&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Entriegeln&bull;Anlage f&uuml;r 30 Sekunden spannungsfrei schalten&bull;Feuerungsautomat austauschen"                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0285"} = {errorcode => "9Y",                                     subcode => "2002",                                               class => "4",           reason => "Interne St&ouml;rung Feuerungsautomat, keine Spannung Heizpatronenrelais; hinter dem Heizpatronenrelais wird keine Spannung gemessen, obwohl das Relais eingeschaltet ist und hinter dem Sicherheitsrelais eine Spannung gemessen wird",                                                                                                           checklist => "St&ouml;rung im Feuerungsautomaten&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Entriegeln&bull;Anlage f&uuml;r 30 Sekunden spannungsfrei schalten&bull;Feuerungsautomat austauschen"                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0286"} = {errorcode => "9Y",                                     subcode => "2003",                                               class => "4",           reason => "Interne St&ouml;rung Feuerungsautomat, Heizpatronenrelais h&auml;ngt; hinter dem Heizpatronenrelais wird eine Spannung gemessen, obwohl das Relais abgeschaltet ist",                                                                                                                                                                              checklist => "St&ouml;rung im Feuerungsautomaten&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Entriegeln&bull;Anlage f&uuml;r 30 Sekunden spannungsfrei schalten&bull;Feuerungsautomat austauschen-A"                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0287"} = {errorcode => "A01",                                    subcode => "594",                                                class => "20",          reason => "Temperaturf&uuml;hler anstelle Kodierbr&uuml;cken angeschlossen",                                                                                                                                                                                                                                                                                  checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Verdrahtungsfehler im Bereich Kodierbr&uuml;cke beheben"                                                                                                                                                                                                                                                                                                                                         };
	$ErrorCodeList{"0288"} = {errorcode => "A01",                                    subcode => "800",                                                class => "14",          reason => "Au&szlig;entemperaturf&uuml;hler defekt; (Auswirkung: es wird die minimale Au&szlig;entemperatur angenommen)",                                                                                                                                                                                                                                     checklist => "Konfiguration pr&uuml;fen; Mit der gew&auml;hlten Einstellung ist ein Au&szlig;entemperaturf&uuml;hler erforderlich&bull;Verbindungsleitung zwischen Regelger&auml;t; und Au&szlig;entemperaturf&uuml;hler auf Durchgang pr&uuml;fen&bull;Elektrischen Anschluss der Verbindungsleitung im Au&szlig;entemperaturf&uuml;hler bzw. am Stecker im Regelger&auml;t pr&uuml;fen&bull;Au&szlig;entemperaturf&uuml;hler laut Tabelle pr&uuml;fen&bull;Spannung an den Anschlussklemmen des Au&szlig;entemperaturf&uuml;hlers im Regelger&auml;t laut Tabelle pr&uuml;fen",correction => "Konfiguration &auml;ndern&bull;Wenn kein Durchgang vorhanden ist, die St&ouml;rung beheben&bull;Falls Schrauben oder ein Stecker lose sind, das Kontaktproblem beheben&bull;Wenn Werte nicht &uuml;bereinstimmen, dann den F&uuml;hler austauschen&bull;Wenn die F&uuml;hlerwerte gestimmt haben, aber die Spannungswerte nicht &uuml;bereinstimmen, dann das Regelger&auml;t austauschen"       };
	$ErrorCodeList{"0289"} = {errorcode => "A01",                                    subcode => "808",                                                class => "14",          reason => "Warmwasserbereitung: Warmwasser-Temperaturf&uuml;hler 1 defekt (Auswirkung: es wird kein Warmwasser geladen) Falls keine Warmwasserfunktion gew&uuml;nscht; ist, diese in der Bedieneinheit; deaktivieren",                                                                                                                                        checklist => "Verbindungsleitung zwischen Regelger&auml;t und Warmwasser-Temperaturf&uuml;hler pr&uuml;fen&bull;Elektrischen Anschluss der Verbindungsleitung im Regelger&auml;t pr&uuml;fen&bull;Warmwasser-Temperaturf&uuml;hler laut Tabelle pr&uuml;fen&bull;Spannung an den Anschlussklemmen des Warmwasser-Temperaturf&uuml;hlers im Regelger&auml;t laut Tabelle pr&uuml;fen&bull;Beim GB142 sollte ein Reset durchgef&uuml;hrt werden",                                                                     correction => "Wenn ein Defekt vorliegt, F&uuml;hler austauschen&bull;Falls Schrauben oder ein Stecker lose sind, das Kontaktproblem beheben&bull;Wenn Werte nicht &uuml;bereinstimmen, dann den F&uuml;hler austauschen&bull;Wenn die F&uuml;hlerwerte gestimmt haben, aber die Spannungswerte nicht &uuml;bereinstimmen, dann das Regelger&auml;t austauschen&bull;Wenn nach dem Reset die St&ouml;rung noch besteht, dann ist die Ger&auml;teelektronik UBA; im W&auml;rmeerzeuger auszutauschen"};
	$ErrorCodeList{"0290"} = {errorcode => "A01",                                    subcode => "809",                                                class => "14",          reason => "Warmwasserbereitung: Warmwasser-Temperaturf&uuml;hler 2 defekt. Falls keine Warmwasserfunktion gew&uuml;nscht ist, diese in der Bedieneinheit; deaktivieren",                                                                                                                                                                                      checklist => "Verbindungsleitung zwischen Regelger&auml;t und Warmwasser-Temperaturf&uuml;hler pr&uuml;fen&bull;Elektrischen Anschluss der Verbindungsleitung im Regelger&auml;t pr&uuml;fen&bull;Warmwasser-Temperaturf&uuml;hler laut Tabelle pr&uuml;fen&bull;Spannung an den Anschlussklemmen des Warmwasser-Temperaturf&uuml;hlers im Regelger&auml;t laut Tabelle pr&uuml;fen&bull;Beim GB142 sollte ein Reset durchgef&uuml;hrt werden",                                                                     correction => "Wenn ein Defekt vorliegt, F&uuml;hler austauschen&bull;Falls Schrauben oder ein Stecker lose sind, das Kontaktproblem beheben&bull;Wenn die Werte nicht &uuml;bereinstimmen, dann den F&uuml;hler austauschen&bull;Wenn die F&uuml;hlerwerte gestimmt haben, aber die Spannungswerte nicht &uuml;bereinstimmen, dann das Regelger&auml;t austauschen&bull;Wenn nach dem Reset die St&ouml;rung noch besteht, dann ist die Ger&auml;teelektronik UBA; im W&auml;rmeerzeuger auszutauschen"};
	$ErrorCodeList{"0291"} = {errorcode => "A01",                                    subcode => "810",                                                class => "14",          reason => "Warmwasser bleibt kalt. Warmwassertemperatur ist w&auml;hrend der Ladung; 2 Std. lang nicht angestiegen. Auswirkung: Warmwasserladung l&auml;uft weiter, Warmwasser-Vorrang wird deaktiviert.",                                                                                                                                                    checklist => "Pumpe oder 3-Wege-Umschaltventil defekt&bull;Pr&uuml;fen, ob evtl. st&auml;ndig Wasser aufgrund von Zapfungen oder einer Leckage aus dem Warmwasserspeicher entnommen wird&bull;Die Position des Warmwasser-Temperaturf&uuml;hlers pr&uuml;fen, evtl. ist dieser falsch angebracht&bull;Die Verbindungsrohre zwischen Kessel und Speicher kontrollieren und nach Montageanleitung pr&uuml;fen, ob diese richtig angeschlossen sind&bull;Pr&uuml;fen, ob die Heizschlange im Speicher vollst&auml;ndig entl&uuml;ftet ist&bull;Entsprechend den technischen Unterlagen pr&uuml;fen, ob die eingebaute Speicherladepumpe &uuml;ber die erforderliche Leistung verf&uuml;gt&bull;Wenn der Warmwasser-Vorrang abgew&auml;hlt wurde und Heizung und Warmwasser im Parallelbetrieb laufen, kann evtl. die Leistung des Kessels nicht ausreichen&bull;Zu gro&szlig;e Verluste Zirkulationsleitung&bull;Den Warmwasser-Temperaturf&uuml;hler laut Tabelle pr&uuml;fen",correction => "Funktionstest durchf&uuml;hren&bull;Ggf. st&auml;ndige Warmwasserentnahme unterbinden&bull;Warmwasser-Temperaturf&uuml;hler richtig positionieren&bull;Bei Fehlern in der Verrohrung diese beheben&bull;Ggf. entl&uuml;ften&bull;Wenn Abweichungen bestehen, die Pumpe austauschen&bull;Warmwasserbereitung auf „Vorrang“ einstellen&bull;Zirkulationsleitung pr&uuml;fen&bull;Bei Abweichungen zu den Tabellenwerten den F&uuml;hler austauschen"};
	$ErrorCodeList{"0292"} = {errorcode => "A01",                                    subcode => "811",                                                class => "16",          reason => "Warmwasserbereitung: thermische Desinfektion misslungen; (Nach 3 Std. wurde die eingestellte Temperatur nicht erreicht.; Auswirkung: nach St&ouml;rmeldung; wird die thermische Desinfektion abgebrochen.)",                                                                                                                                       checklist => "Pr&uuml;fen, ob evtl. st&auml;ndig Wasser aufgrund von Zapfungen oder einer Leckage aus dem Warmwasserspeicher entnommen wird&bull;Die Position des Warmwasser-Temperaturf&uuml;hlers pr&uuml;fen, evtl. ist dieser falsch angebracht&bull;Die Verbindungsrohre zwischen Kessel und Speicher kontrollieren und nach Montageanleitung pr&uuml;fen, ob diese richtig angeschlossen sind&bull;Pr&uuml;fen, ob die Heizschlange im Speicher vollst&auml;ndig entl&uuml;ftet ist&bull;Entsprechend den technischen Unterlagen pr&uuml;fen, ob die eingebaute Speicherladepumpe &uuml;ber die erforderliche Leistung verf&uuml;gt&bull;Wenn der Warmwasser-Vorrang abgew&auml;hlt wurde und Heizung und Warmwasser im Parallelbetrieb laufen, kann evtl. die Leistung des Kessels nicht ausreichen&bull;Zu gro&szlig;e Verluste Zirkulationsleitung&bull;Den Warmwasser-Temperaturf&uuml;hler laut Tabelle pr&uuml;fen",correction => "Ggf. st&auml;ndige Warmwasserentnahme unterbinden&bull;Warmwasser-Temperaturf&uuml;hler richtig positionieren&bull;Bei Fehlern in der Verrohrung diese beheben&bull;Ggf. entl&uuml;ften&bull;Wenn Abweichungen bestehen, die Pumpe austauschen&bull;Warmwasserbereitung auf „Vorrang“ einstellen&bull;Zirkulationsleitung pr&uuml;fen&bull;Bei Abweichungen zu den Tabellenwerten den F&uuml;hler austauschen"};
	$ErrorCodeList{"0293"} = {errorcode => "A01",                                    subcode => "815",                                                class => "14",          reason => "Temperaturf&uuml;hler hydraulische Weiche defekt (Pumpeneffizienzmodul)",                                                                                                                                                                                                                                                                          checklist => "F&uuml;hler FK des Pumpeneffizienzmoduls defekt oder nicht korrekt angeschlossen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                             correction => "F&uuml;hleranschluss pr&uuml;fen&bull;Weichentemperaturf&uuml;hler auf falsche Einbauposition oder auf Bruchstelle pr&uuml;fen"                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0294"} = {errorcode => "A01",                                    subcode => "816",                                                class => "A",           reason => "Keine Kommunikation mit Busmaster Ger&auml;teelektronik (UBA oder MC10 oder MC40 oder MC100)/Kaskadenmodul/Feuerungsautomat/HM10 (Auswirkung: es findet kein Heiz-betrieb statt)",                                                                                                                                                                 checklist => "Pr&uuml;fen, ob Busleitung falsch angeschlossen wurde&bull;Pr&uuml;fen, ob Busleitung defekt ist; Erweiterungsmodule vom EMS-BUS entfernen und Regelger&auml;t ausund wieder einschalten; pr&uuml;fen, ob St&ouml;rungsursache Modul oder Modulverdrahtung&bull;-",                                                                                                                                                                                                                                   correction => "Verdrahtungsfehler beseitigen und Regelger&auml;t ausund wieder einschalten&bull;Busleitung reparieren bzw. austauschen&bull;Defekten EMS-BUS-Teilnehmer austauschen (EMS-Modul bzw. Feuerungsautomat bzw. Ger&auml;teelektronik bzw. Regelger&auml;t bzw. HM10)"                                                                                                                                };
	$ErrorCodeList{"0295"} = {errorcode => "A01",                                    subcode => "817",                                                class => "14",          reason => "Lufttemperaturf&uuml;hler defekt",                                                                                                                                                                                                                                                                                                                 checklist => "Lufttemperaturf&uuml;hler inkl. Steckverbindung der Ger&auml;teelektronik SAFe pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Steckverbindung richtig aufstecken oder ggf. Temperaturf&uuml;hler austauschen"                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0296"} = {errorcode => "A01",                                    subcode => "818",                                                class => "14",          reason => "W&auml;rmeerzeuger bleibt kalt",                                                                                                                                                                                                                                                                                                                   checklist => "Die Version der Ger&auml;teelektronik SAFe pr&uuml;fen&bull;Evtl. liegen Probleme im Bereich des Kesself&uuml;hlers bzw. Luftprobleme im Kessel vor&bull;Es wurde ein Weichenmodul installiert, obwohl dieses nicht installiert werden darf, und damit ist die Pumpenlogik au&szlig;er Betrieb&bull;Die Auslegung der Kesselleistung bezogen auf die Anlage pr&uuml;fen&bull;Unter Umst&auml;nden ist die Kesselleistung nicht f&uuml;r einen Parallelbetrieb von Warmwasser und Heizung ausgelegt und es ist trotzdem Parallelbetrieb von Heizung und Warmwasser parametriert",correction => "Ist die Feuerungsautomat-Version nicht mindestens 2.14, sollte die Ger&auml;teelektronik ausgetauscht werden&bull;Anlage entl&uuml;ften bzw. Anschl&uuml;sse und Anschlusskabel des Kesself&uuml;hlers pr&uuml;fen oder den F&uuml;hler austauschen&bull;Weichenmodul deaktivieren&bull;Ggf. gr&ouml;&szlig;eren Kessel ausw&auml;hlen&bull;Heizungsanlage auf „Warmwasser-Vorrang“ einstellen"};
	$ErrorCodeList{"0297"} = {errorcode => "A01",                                    subcode => "819",                                                class => "14",          reason => "&Ouml;lvorw&auml;rmer meldet Dauersignal",                                                                                                                                                                                                                                                                                                         checklist => "Verbindungsleitung zwischen Ger&auml;teelektronik SAFe und &Ouml;lvorw&auml;rmer optisch auf Besch&auml;digungen pr&uuml;fen&bull;Verbindungsleitung zwischen Ger&auml;teelektronik SAFe und &Ouml;lvorw&auml;rmer abziehen und auf Kurzschl&uuml;sse pr&uuml;fen&bull;&Ouml;lvorw&auml;rmer im kalten Zustand pr&uuml;fen",                                                                                                                                                                          correction => "Sollte das Kabel besch&auml;digt sein, dieses austauschen&bull;Sollten Kurzschl&uuml;sse vorhanden sein, Kabel austauschen&bull;Sollte eine Verbindung vorhanden sein, &Ouml;lvorw&auml;rmer austauschen"                                                                                                                                                                                        };
	$ErrorCodeList{"0298"} = {errorcode => "A01",                                    subcode => "820",                                                class => "14",          reason => "&Ouml;l-Betriebstemperatur wird nicht erreicht",                                                                                                                                                                                                                                                                                                   checklist => "Verbindungsleitung zwischen Ger&auml;teelektronik SAFe und &Ouml;lvorw&auml;rmer optisch auf Besch&auml;digungen pr&uuml;fen&bull;Pr&uuml;fen, ob an Ger&auml;teelektronik SAFe und am &Ouml;lvorw&auml;rmer die Stecker richtig aufgesteckt sind&bull;Verbindungsleitung zwischen Ger&auml;teelektronik SAFe und &Ouml;lvorw&auml;rmer auf; Durchgang pr&uuml;fen&bull;Im Men&uuml; „Service“ der Bedieneinheit den Men&uuml;punkt „Relaistest“ aufrufen und hier den &Ouml;lvorw&auml;rmer einschalten; Mit der Hand pr&uuml;fen, ob der &Ouml;lvorw&auml;rmer sich erw&auml;rmt&bull;-",correction => "Falls das Kabel besch&auml;digt ist, dieses austauschen&bull;Stecker richtig aufstecken&bull;Erw&auml;rmt sich der &Ouml;lvorw&auml;rmer und die St&ouml;rungsmeldung bleibt erhalten, dann ist der Schaltkontakt im &Ouml;lvorw&auml;rmer defekt; Der &Ouml;lvorw&auml;rmer muss ausgetauscht werden&bull;Erw&auml;rmt sich der &Ouml;lvorw&auml;rmer und die St&ouml;rungsmeldung bleibt erhalten, dann ist der Schaltkontakt im &Ouml;lvorw&auml;rmer defekt; Der &Ouml;lvorw&auml;rmer muss ausgetauscht werden&bull;Erw&auml;rmt sich der &Ouml;lvorw&auml;rmer nicht, ist das Heizelement im &Ouml;lvorw&auml;rmer defekt; Der &Ouml;lvorw&auml;rmer muss ausgetauscht werden"};
	$ErrorCodeList{"0299"} = {errorcode => "A01",                                    subcode => "828",                                                class => "14",          reason => "Wasserdruckf&uuml;hler defekt",                                                                                                                                                                                                                                                                                                                    checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Wasserdruckf&uuml;hler austauschen"                                                                                                                                                                                                                                                                                                                                                              };
	$ErrorCodeList{"0300"} = {errorcode => "A01",                                    subcode => "845",                                                class => "14",          reason => "Hydraulische Konfiguration wird nicht unterst&uuml;tzt",                                                                                                                                                                                                                                                                                           checklist => "W&auml;rmeerzeuger unterst&uuml;tzt die vorgegebene hydraulische Konfiguration nicht (z. B. weil mehr Pumpenausg&auml;nge ben&ouml;tigt werden; als vorhanden)&bull;-",                                                                                                                                                                                                                                                                                                                               correction => "Warmwasser auf Modul konfigurieren oder deinstallieren&bull;Heizkreis 1 auf Modul konfigurieren oder deinstallieren&bull;Systempumpe auf „Keine“ stellen"                                                                                                                                                                                                                                    };
	$ErrorCodeList{"0301"} = {errorcode => "A01",                                    subcode => "1010",                                               class => "8",           reason => "Keine Kommunikation &uuml;ber BUS-Verbindung EMS",                                                                                                                                                                                                                                                                                                 checklist => "Pr&uuml;fen, ob Busleitung falsch angeschlossen wurde&bull;Pr&uuml;fen, ob Busleitung defekt ist. Erweiterungsmodule vom EMS-BUS entfernen und Regelger&auml;t ausund wieder einschalten. Pr&uuml;fen, ob St&ouml;rungsursache Modul oder Modulverdrahtung.&bull;-",                                                                                                                                                                                                                                  correction => "Verdrahtungsfehler beseitigen und Regelger&auml;t Ausund wieder Einschalten&bull;Busleitung reparieren bzw. austauschen.&bull;Defekten EMS-BUS-Teilnehmer austauschen"                                                                                                                                                                                                                           };
	$ErrorCodeList{"0302"} = {errorcode => "A01",                                    subcode => "1030",                                               class => "8",           reason => "Etagenstation: Interner Datenfehler, Regelungsmodul austauschen",                                                                                                                                                                                                                                                                                  checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Modul austauschen"                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0303"} = {errorcode => "A01",                                    subcode => "1033; 1034; 1035; 1036",                             class => "8",           reason => "Etagenstation: interner Fehler, Ger&auml;t tauschen",                                                                                                                                                                                                                                                                                              checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Modul austauschen"                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0304"} = {errorcode => "A01",                                    subcode => "1038",                                               class => "8",           reason => "Etagenstation: Zeit/Datum ung&uuml;ltiger Wert",                                                                                                                                                                                                                                                                                                   checklist => "Datum/Zeit noch nicht eingestellt&bull;Spannungsversorgung &uuml;ber l&auml;ngere Zeit ausgefallen",                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Datum/Zeit einstellen&bull;l&auml;ngere Spannungsausf&auml;lle vermeiden"                                                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0305"} = {errorcode => "A01",                                    subcode => "1042",                                               class => "8",           reason => "Etagenstation: Interner Fehler: Zugriff auf Uhrenbaustein blockiert",                                                                                                                                                                                                                                                                              checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Modul austauschen"                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0306"} = {errorcode => "A01",                                    subcode => "1705",                                               class => "8",           reason => "Etagenstation: Modul erh&auml;lt unzul&auml;ssige Werte vom R&uuml;cklauftemperaturf&uuml;hler (Prim&auml;rkreis)",                                                                                                                                                                                                                                checklist => "Anschlussstecker am R&uuml;cklauftemperaturf&uuml;hler nicht aufgesteckt&bull;Anschlusskabel zum R&uuml;cklauftemperaturf&uuml;hler besch&auml;digt&bull;R&uuml;cklauftemperaturf&uuml;hler (Prim&auml;rkreis) defekt",                                                                                                                                                                                                                                                                               correction => "Anschlussstecker am R&uuml;cklauftemperaturf&uuml;hler (Prim&auml;rkreis) aufstecken&bull;Anschlusskabel zum R&uuml;cklauftemperaturf&uuml;hler (Prim&auml;rkreis) austauschen&bull;R&uuml;cklauftemperaturf&uuml;hler (Prim&auml;rkreis) austauschen"                                                                                                                                           };
	$ErrorCodeList{"0307"} = {errorcode => "A01",                                    subcode => "1706",                                               class => "8",           reason => "Etagenstation: Modul erh&auml;lt unzul&auml;ssige Werte vom Vorlauftemperaturf&uuml;hler (Prim&auml;rkreis)",                                                                                                                                                                                                                                      checklist => "Anschlussstecker am Vorlauftemperaturf&uuml;hler (Prim&auml;rkreis) nicht aufgesteckt&bull;Anschlusskabel zum Vorlauftemperaturf&uuml;hler (Prim&auml;rkreis) besch&auml;digt&bull;Vorlauftemperaturf&uuml;hler (Prim&auml;rkreis) defekt",                                                                                                                                                                                                                                                           correction => "Anschlussstecker am Vorlauftemperaturf&uuml;hler (Prim&auml;rkreis) aufstecken&bull;Anschlusskabel zum Vorlauftemperaturf&uuml;hler (Prim&auml;rkreis) austauschen&bull;Vorlauftemperaturf&uuml;hler (Prim&auml;rkreis) austauschen"                                                                                                                                                             };
	$ErrorCodeList{"0308"} = {errorcode => "A01",                                    subcode => "1707",                                               class => "8",           reason => "Etagenstation: Ung&uuml;ltige Position des Koodierschalters f&uuml;r Warmwasser",                                                                                                                                                                                                                                                                  checklist => "Falsche Position f&uuml;r den Kodierschalter (Warmwasser) eingestellt&bull;Kodierschalter f&uuml;r Warmwasser defekt",                                                                                                                                                                                                                                                                                                                                                                                correction => "Position des Kodierschalters f&uuml;r Warmwasser auf die korrekte Kennzahl drehen.&bull;Modul austauschen"                                                                                                                                                                                                                                                                                       };
	$ErrorCodeList{"0309"} = {errorcode => "A01",                                    subcode => "1708",                                               class => "8",           reason => "Etagenstation: Ung&uuml;ltige Position des Kodierschalters f&uuml;r Heizung",                                                                                                                                                                                                                                                                      checklist => "Falsche Position f&uuml;r den Kodierschalter (Heizung) eingestellt&bull;Kodierschalter f&uuml;r Heizung defekt",                                                                                                                                                                                                                                                                                                                                                                                      correction => "Position des Kodierschalters f&uuml;r Heizung auf die korrekte Kennzahl drehen.&bull;Modul austauschen"                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0310"} = {errorcode => "A01",                                    subcode => "1710",                                               class => "4",           reason => "Etagenstation: Vorlauftemperatur (Prim&auml;rkreis) zu niedrig",                                                                                                                                                                                                                                                                                   checklist => "Position der Kodierschalter am Modul &uuml;berpr&uuml;fen&bull;Vorlauftemperatur in der Systemauslegung zu niedrig&bull;Vorlauftemperaturf&uuml;hler (Prim&auml;r) ist nicht korrekt montiert",                                                                                                                                                                                                                                                                                                       correction => "Kodierschalter auf die notwendige Versorgungstemperatur anheben&bull;Einstellungen am W&auml;rmeerzeuger auf das System anpassen&bull;Vorlauftemperaturf&uuml;hler (Prim&auml;rkreis) an der korrekten Position montieren"                                                                                                                                                                       };
	$ErrorCodeList{"0311"} = {errorcode => "A01",                                    subcode => "1711",                                               class => "8",           reason => "Etagenstation: Vorlauftemperatur (Prim&auml;rkreis) zu hoch",                                                                                                                                                                                                                                                                                      checklist => "Systemauslegung bzw. Einstellungen; am W&auml;rmeerzeuger ist nicht auf die HDU abgestimmt&bull;Systemauslegung bei Fernw&auml;rmeanschluss nicht f&uuml;r HDU geeignet",                                                                                                                                                                                                                                                                                                                             correction => "System bzw. Einstellungen am W&auml;rmeerzeuger anpassen&bull;Fernw&auml;rmeanschlu&szlig; an HDU-Parameter anpassen."                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0312"} = {errorcode => "A01",                                    subcode => "1712",                                               class => "8",           reason => "Etagenstation: Modul erh&auml;lt unzul&auml;ssige Werte f&uuml;r den Warmwassertemperaturf&uuml;hler",                                                                                                                                                                                                                                             checklist => "Anschlusskabel vom Warmwassertemperaturf&uuml;hler im Modul &uuml;berpr&uuml;fen&bull;Anschlusskabel zum Warmwassertemperaturf&uuml;hler &uuml;berpr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                                correction => "Anschlusskabel vom Warmwassertemperaturf&uuml;hler im Modul anschlie&szlig;en&bull;Warmwassertemperaturf&uuml;hler komplett austauschen&bull;Warmwassertemperaturf&uuml;hler komplett austauschen"                                                                                                                                                                                               };
	$ErrorCodeList{"0313"} = {errorcode => "A01",                                    subcode => "1713",                                               class => "8",           reason => "Etagenstation: Modul erh&auml;lt unzul&auml;ssige Werte f&uuml;r den Warmnwasservolumenstrom",                                                                                                                                                                                                                                                     checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Volumenstrommesser im Warmwasserkreis austauschen"                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0314"} = {errorcode => "A01",                                    subcode => "1716",                                               class => "8",           reason => "Etagenstation: Modul erh&auml;lt unzul&auml;ssige Werte f&uuml;r den Vorlauftemperaturf&uuml;hler (Sekund&auml;rkreis)",                                                                                                                                                                                                                           checklist => "Anschlussstecker am Vorlauftemperaturf&uuml;hler (Sekund&auml;rkreis) &uuml;berpr&uuml;fen&bull;Anschlusskabel zum Vorlauftemperaturf&uuml;hler (Sekund&auml;rkreis) &uuml;berpr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                    correction => "Anschlussstecker am Vorlauftemperaturf&uuml;hler (Sekund&auml;rkreis) aufstecken&bull;Anschlusskabel zum Vorlauftemperaturf&uuml;hler (Sekund&auml;rkreis) austauschen&bull;Vorlauftemperaturf&uuml;hler (Sekund&auml;rkreis) austauschen"                                                                                                                                                       };
	$ErrorCodeList{"0315"} = {errorcode => "A01",                                    subcode => "1717",                                               class => "4",           reason => "Etagenstation: Modul erh&auml;lt unzul&auml;ssige Werte vom Au&szlig;entemperaturf&uuml;hler",                                                                                                                                                                                                                                                     checklist => "Konfiguration &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob ein Au&szlig;entemperaturf&uuml;hler montiert ist&bull;Anschluss des Au&szlig;entemperaturf&uuml;hlers am Modul &uuml;berpr&uuml;fen&bull;Anschlusskabel am Au&szlig;entemperaturf&uuml;hler &uuml;berpr&uuml;fen&bull;Au&szlig;entemperaturf&uuml;hler &uuml;berpr&uuml;fen&bull;Anschlusskabel zum Au&szlig;entemperatuf&uuml;hler &uuml;berpr&uuml;fen",                                                                          correction => "Konfiguration auf raumtemperaturgef&uuml;hrt&bull;Au&szlig;entemperaturf&uuml;hler montieren&bull;Anschlusskabel vom Au&szlig;entemperaturf&uuml;hler am Modul anschlie&szlig;en&bull;Anschlusskabel am Au&szlig;entemperaturf&uuml;hler anschlie&szlig;en&bull;Au&szlig;entemperaturf&uuml;hler austauschen&bull;Anschlusskabel zum Au&szlig;entemperaturf&uuml;hler instandsetzen oder austauschen"};
	$ErrorCodeList{"0316"} = {errorcode => "A01",                                    subcode => "1718",                                               class => "8",           reason => "Etagenstation: Modul erh&auml;lt unzul&auml;ssige Werte f&uuml;r den R&uuml;cklauftemperaturf&uuml;hler (Sekund&auml;rkreis)",                                                                                                                                                                                                                     checklist => "Anschlussstecker am R&uuml;cklauftemperaturf&uuml;hler (Sekund&auml;rkreis) &uuml;berpr&uuml;fen&bull;Anschlusskabel zum R&uuml;cklauftemperaturf&uuml;hler (Sekund&auml;rkreis) &uuml;berpr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                        correction => "Anschlussstecker am R&uuml;cklauftemperaturf&uuml;hler (Sekund&auml;rkreis) aufstecken&bull;Anschlusskabel zm R&uuml;cklauftemperaturf&uuml;hler (Sekund&auml;rkreis) austauschen&bull;R&uuml;cklauftemperaturf&uuml;hler (Sekund&auml;rkreis) austauschen"                                                                                                                                      };
	$ErrorCodeList{"0317"} = {errorcode => "A01",                                    subcode => "1719",                                               class => "8",           reason => "Etagenstation: Regelventil f&uuml;r Heizung in HDU defekt",                                                                                                                                                                                                                                                                                        checklist => "PWM-R&uuml;cksignal vom Regelventil (Heizung) liegt au&szlig;erhalb des zul&auml;ssigen Bereiches",                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Regelventil f&uuml;r den Heizbetrieb austauschen"                                                                                                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0318"} = {errorcode => "A01",                                    subcode => "1720",                                               class => "8",           reason => "Etagenstation: Regelventil f&uuml;r Warmwasser in HDU defekt",                                                                                                                                                                                                                                                                                     checklist => "PWM-R&uuml;cksignal vom Regelventil (Warmwasser) liegt au&szlig;erhalb des zul&auml;ssigen Bereiches",                                                                                                                                                                                                                                                                                                                                                                                                correction => "Regelventil f&uuml;r den Warmwasserbetrieb austauschen"                                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0319"} = {errorcode => "A01",                                    subcode => "1900",                                               class => "8",           reason => "AM200: Sensor defekt (Sys  Kabel zum Sensor pr&uuml;fen              ",                                                                                                                                                                                                                                                                            checklist => "Sensor ersetzen temvorlauftemperatur)             Elektrische Verbindung zum Sensor pr&uuml;fen            Sensorkabel ersetzen; Sensorkennlinie pr&uuml;fen       ",                                                                                                                                                                                                                                                                                                                                 correction => "Kaskadenmodul ersetzen Spannung am Sensoranschlu&szlig; pr&uuml;fen; Spannung am Kaskadenmodul abschalten/ anschalten"                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0320"} = {errorcode => "A01",                                    subcode => "1901",                                               class => "8",           reason => "AM200: Sensor defekt (Systemr&uuml;cklauftemperatur)",                                                                                                                                                                                                                                                                                             checklist => "Kabel zum Sensor pr&uuml;fen&bull;Elektrische Verbindung zum Sensor pr&uuml;fen&bull;Sensorkennlinie pr&uuml;fen&bull;Spannung am Sensoranschlu&szlig; pr&uuml;fen&bull;Spannung am Kaskadenmodul abschalten/ anschalten",                                                                                                                                                                                                                                                                            correction => "Sensor ersetzen&bull;Sensorkabel ersetzen&bull;Kaskadenmodul ersetzen&bull;-"                                                                                                                                                                                                                                                                                                                    };
	$ErrorCodeList{"0321"} = {errorcode => "A01",                                    subcode => "1902",                                               class => "8",           reason => "AM200: Sensor defekt (Au&szlig;entemperatur)",                                                                                                                                                                                                                                                                                                     checklist => "Kabel zum Sensor pr&uuml;fen&bull;Elektrische Verbindung zum Sensor pr&uuml;fen&bull;Sensorkennlinie pr&uuml;fen&bull;Spannung am Sensoranschlu&szlig; pr&uuml;fen&bull;Spannung am Kaskadenmodul abschalten/ anschalten",                                                                                                                                                                                                                                                                            correction => "Sensor ersetzen&bull;Sensorkabel ersetzen&bull;Kaskadenmodul ersetzen&bull;-"                                                                                                                                                                                                                                                                                                                    };
	$ErrorCodeList{"0322"} = {errorcode => "A01",                                    subcode => "1903",                                               class => "8",           reason => "AM200: Schalterposition Drehcodierschalter ung&uuml;ltig",                                                                                                                                                                                                                                                                                         checklist => "Schalterstellung pr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "Schalterstellung korrigieren&bull;Netzspannung aus und wieder einschalten&bull;Kaskadenmodul ersetzen"                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0323"} = {errorcode => "A01",                                    subcode => "1904",                                               class => "8",           reason => "Etagenstation: Sensor defekt (Systemr&uuml;cklauftemperatur)",                                                                                                                                                                                                                                                                                     checklist => "Konfiguration pr&uuml;fen. Mit der gew&auml;hlten Einstellung ist ein F&uuml;hler Versorgungsr&uuml;cklauf angebracht&bull;Verbindungsleitung zwischen Modul HDU und Versorgungsr&uuml;cklauf pr&uuml;fen&bull;F&uuml;hler Versorgungsr&uuml;cklauf laut Tabelle pr&uuml;fen&bull;Spannung an den Anschlussklemmen des F&uuml;hler Versorgungsr&uuml;cklauf laut Tabelle; pr&uuml;fen",                                                                                                               correction => "Konfiguration &auml;ndern&bull;Verbindung ordnungsgem&auml;&szlig; herstellen&bull;Wenn die Werte nicht &uuml;bereinstimmen, dann den F&uuml;hler austauschen&bull;Wenn die F&uuml;hlerwerte gestimmt haben, aber die Spannungswerte nicht &uuml;bereinstimmen, dann das Modul austauschen"                                                                                                      };
	$ErrorCodeList{"0324"} = {errorcode => "A01",                                    subcode => "1905",                                               class => "8",           reason => "Etagenstation: Modul erh&auml;lt unzul&auml;ssige Werte f&uuml;r den Systemvorlauf-temperaturf&uuml;hler",                                                                                                                                                                                                                                         checklist => "Konfiguration pr&uuml;fen. Mit der gew&auml;hlten Einstellung ist ein F&uuml;hler Versorgungsvorlauf angebracht&bull;Verbindungsleitung zwischen Modul HDU und Versorgungsvorlauf pr&uuml;fen&bull;F&uuml;hler Versorgungsvorlauf laut Tabelle pr&uuml;fen&bull;Spannung an den Anschlussklemmen des F&uuml;hler Versorgungsvorlauf laut Tabelle pr&uuml;fen",                                                                                                                                        correction => "Konfiguration &auml;ndern&bull;Verbindung ordnungsgem&auml;&szlig; herstellen&bull;Wenn die Werte nicht &uuml;bereinstimmen, dann den F&uuml;hler austauschen&bull;Wenn die F&uuml;hlerwerte gestimmt haben, aber die Spannungswerte nicht &uuml;bereinstimmen, dann das Modul austauschen"                                                                                                      };
	$ErrorCodeList{"0325"} = {errorcode => "A01",                                    subcode => "1906",                                               class => "8",           reason => "Etagenstation: Schalterposition ung&uuml;ltig",                                                                                                                                                                                                                                                                                                    checklist => "Modul verh&auml;lt sich wie in Stellung 'off', Funktionen sind deaktiviert&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Beide Poti-Drehknopfpositonen &uuml;berpr&uuml;fen&bull;Poti-Drehknopfpositon auf 0 und wieder auf neuen Wert setzen"                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0326"} = {errorcode => "A01",                                    subcode => "1908",                                               class => "4",           reason => "Etagenstation: Systemvorlauftemperatur zu niedrig",                                                                                                                                                                                                                                                                                                checklist => "Konfiguration pr&uuml;fen. Mit der gew&auml;hlten Einstellung ben&ouml;tigt die HDU die an den; Poti-Drehkn&ouml;pfen angegebene Versorgungstemperatur&bull;Zuleitungstemperatur am Kessel bzw. Speicher pr&uuml;fen&bull;Auf Lufteinschl&uuml;sse pr&uuml;fen",                                                                                                                                                                                                                                      correction => "Versorgungstemperatur anheben&bull;Versorgungsleistung zur Verf&uuml;gung stellen, Volumenstrom anheben bzw. Anlage entl&uuml;ften&bull;-"                                                                                                                                                                                                                                                       };
	$ErrorCodeList{"0327"} = {errorcode => "A01",                                    subcode => "1909",                                               class => "8",           reason => "Etagenstation: System-vorlauftemperatur zu hoch",                                                                                                                                                                                                                                                                                                  checklist => "Wasserseitige Durchstr&ouml;mung pr&uuml;fen -; Pr&uuml;fung, ob Fremdw&auml;rmequelle vorhanden (Feststoffkessel)&bull;Kesself&uuml;hler/SicherheitstemperaturbegrenzerF&uuml;hler pr&uuml;fen.&bull;Z&uuml;nd-/&Uuml;berwachungselektrode pr&uuml;fen.",                                                                                                                                                                                                                                            correction => "Ausreichende Durchstr&ouml;mung sicherstellen&bull;Ggf. Kesself&uuml;hler/Sicherheitstemperaturbegrenzer-F&uuml;hler austauschen&bull;Ggf. Z&uuml;nd-/&Uuml;berwachungselektrode am Kessel austauschen"                                                                                                                                                                                          };
	$ErrorCodeList{"0328"} = {errorcode => "A01",                                    subcode => "1920",                                               class => "8",           reason => "AM200: Signal vom Vorlauftemperaturf&uuml;hler (System) liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                                                      checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                   correction => "Bei Besch&auml;digung austauschen&bull;Bei Verschmutzung reinigen oder evtl. austauschen&bull;Bei Besch&auml;digung austauschen&bull;Bei losem Stecker diesen wieder aufstecken&bull;Bei Abweichungen den F&uuml;hler austauschen"                                                                                                                                                               };
	$ErrorCodeList{"0329"} = {errorcode => "A01",                                    subcode => "1921",                                               class => "8",           reason => "AM200: Signal vom R&uuml;cklauftemperaturf&uuml;hler (System) liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                                                checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0330"} = {errorcode => "A01",                                    subcode => "1922",                                               class => "8",           reason => "AM200: Signal vom Vorlauftemperaturf&uuml;hler am alternativen W&auml;rmeerzeuger liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                            checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0331"} = {errorcode => "A01",                                    subcode => "1923",                                               class => "8",           reason => "AM200: Signal vom R&uuml;cklauftemperaturf&uuml;hler am alternativen W&auml;rmeerzeuger liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                      checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0332"} = {errorcode => "A01",                                    subcode => "1924",                                               class => "8",           reason => "AM200: Signal vom Temperaturf&uuml;hler (Pufferspeicher oben) liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                                                checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0333"} = {errorcode => "A01",                                    subcode => "1925",                                               class => "8",           reason => "AM200: Signal vom Temperaturf&uuml;hler (Pufferspeicher Mitte) liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                                               checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0334"} = {errorcode => "A01",                                    subcode => "1926",                                               class => "8",           reason => "AM200: Signal vom Temperaturf&uuml;hler (Pufferspeicher unten) liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                                               checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0335"} = {errorcode => "A01",                                    subcode => "1927",                                               class => "8",           reason => "AM200: Signal vom Warmwasser Temperaturf&uuml;hler liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                                                           checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0336"} = {errorcode => "A01",                                    subcode => "1928",                                               class => "8",           reason => "AM200: Signal vom Au&szlig;entemperaturf&uuml;hler liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                                                           checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0337"} = {errorcode => "A01",                                    subcode => "1929",                                               class => "8",           reason => "AM200: Signal vom Abgastemperaturf&uuml;hler liegt au&szlig;erhalb der Kennlinie",                                                                                                                                                                                                                                                                 checklist => "F&uuml;hlerleitung pr&uuml;fen&bull;Steckverbindung pr&uuml;fen&bull;F&uuml;hlerwerte nach Tabelle pr&uuml;fen&bull;Spannungswerte am F&uuml;hler laut Tabelle pr&uuml;fen",                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0338"} = {errorcode => "A01",                                    subcode => "1930",                                               class => "8",           reason => "AM200: Systemkonfiguration nicht best&auml;tigt",                                                                                                                                                                                                                                                                                                  checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Best&auml;tigung Systemkonfiguration in der Bedieneinheit erforderlich"                                                                                                                                                                                                                                                                                                                          };
	$ErrorCodeList{"0339"} = {errorcode => "A01",                                    subcode => "1931",                                               class => "8",           reason => "AM200: Schalterposition ung&uuml;ltig",                                                                                                                                                                                                                                                                                                            checklist => "Drehkodierschalter AM200 zwischen 2 g&uuml;ltigen Positionen&bull;Schalterstellung pr&uuml;fen&bull;Schalter defekt",                                                                                                                                                                                                                                                                                                                                                                                 correction => "Schalterstellung korrigieren&bull;Netzspannung aus und wieder einschalten&bull;Modul ersetzen"                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0340"} = {errorcode => "A01",                                    subcode => "3818",                                               class => "4",           reason => "Keine Buskommunikation zwischen LM10/IUM10 und Ger&auml;teelektronik UBA-H3",                                                                                                                                                                                                                                                                      checklist => "Die Verdrahtung zwischen LM10/IUM10 und Ger&auml;teelektronik UBA-H3 pr&uuml;fen&bull;Verbindungsleitung zwischen LM10/IUM10 und Ger&auml;teelektronik UBA-H3 pr&uuml;fen&bull;LM10/IUM10 defekt",                                                                                                                                                                                                                                                                                                    correction => "Bei Fehlverdrahtung den Verdrahtungsfehler korrigieren&bull;Bei Defekt Datenleitung ersetzen&bull;LM10/IUM10 austauschen"                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0341"} = {errorcode => "A01",                                    subcode => "4111",                                               class => "8",           reason => "Etagenstation: Trinkwasservolumensensor defekt",                                                                                                                                                                                                                                                                                                   checklist => "Konfiguration pr&uuml;fen. Mit der gew&auml;hlten Einstellung ist ein Trinkwasservolumenstromsensor erforderlich.&bull;Verbindungsleitung zwischen Regelger&auml;t und Trinkwasservolumenstromsensor auf Durchgang pr&uuml;fen&bull;Elektrischer Anschluss der Verbindungsleitung in Trinkwasservolumenstromsensor bzw. am Stecker im Regelger&auml;t pr&uuml;fen&bull;-",                                                                                                                            correction => "Konfiguration &auml;ndern&bull;Verbindung ordnungsgem&auml;&szlig; herstellen&bull;Wenn die Werte nicht &uuml;bereinstimmen, dann den F&uuml;hler austauschen&bull;Wenn die F&uuml;hlerwerte gestimmt haben, aber die Spannungswerte nicht &uuml;bereinstimmen, dann das Modul austauschen"                                                                                                      };
	$ErrorCodeList{"0342"} = {errorcode => "A01",                                    subcode => "5100; 5101; 5104",                                   class => "8",           reason => "Interner Fehler im Steuerger&auml;t (MCU)",                                                                                                                                                                                                                                                                                                        checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Versorgungsspannung kurz unterbrechen&bull;Steuerger&auml;t (MCU) austauschen"                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0343"} = {errorcode => "A01",                                    subcode => "5105",                                               class => "8",           reason => "Gas-W&auml;rmepumpe: Interner Fehler im Steuerger&auml;t; (CU I/O)",                                                                                                                                                                                                                                                                               checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Versorgungsspannung kurz unterbrechen&bull;Steuerger&auml;t (CU I/O) austauschen"                                                                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0344"} = {errorcode => "A01",                                    subcode => "5106",                                               class => "8",           reason => "Gas-W&auml;rmepumpe: Interner Fehler im Steuerger&auml;t  der Au&szlig;eneinheit",                                                                                                                                                                                                                                                                 checklist => "-",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Versorgungsspannung kurz unterbrechen&bull;Steuerger&auml;t in der Au&szlig;eneinheit austauschen"                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0345"} = {errorcode => "A01",                                    subcode => "5110",                                               class => "8",           reason => "Gas-W&auml;rmepumpe: Kein CAN Bussignal am Steuerger&auml;t (CU I/O) vorhanden",                                                                                                                                                                                                                                                                   checklist => "&Uuml;berpr&uuml;fen, ob das CAN  Buskabel  am Steuerger&auml;t (CU I/O) korrekt angeschlossen ist.&bull;&Uuml;berpr&uuml;fen, ob das CAN Buskabel zwischen den Steuerger&auml;ten MC und CU I/O besch&auml;digt ist.&bull;-",                                                                                                                                                                                                                                                                        correction => "CAN Buskabel im Steuerger&auml;t (CU I/O) korrekt anschlie&szlig;en&bull;CAN Buskabel zwischen den Steuerger&auml;ten MCU und CU I/O austauschen&bull;Steuerger&auml;t (CU I/O) austauschen"                                                                                                                                                                                                     };
	$ErrorCodeList{"0346"} = {errorcode => "A01",                                    subcode => "5111",                                               class => "8",           reason => "Gas-W&auml;rmepumpe: St&ouml;rung in der CAN Buskommunikation zwischen Innenund Au&szlig;eneinheit",                                                                                                                                                                                                                                               checklist => "&Uuml;berpr&uuml;fen, ob das CAN -Buskabel am Steuerger&auml;t der Inneneinheit korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das CAN Buskabel am Steuerger&auml;t der Au&szlig;eneinheit korrekt angeschlossen ist.&bull;Abstand zwischen dem CAN Buskabel und dem Anschlusskabel f&uuml;r die Spannungsversorgung der Au&szlig;eneinheit &uuml;berpr&uuml;fen.&bull;-",                                                                                                                   correction => "CAN Buskabel im Steuerger&auml;t der Inneneinheit korrekt anschlie&szlig;en&bull;CAN Buskabel im Steuerger&auml;t der Au&szlig;eneinheit korrekt anschlie&szlig;en&bull;Abstand von mindestens 100 mm zwischen CAN Buskabel und Anschlusskabel f&uuml;r die Spannungsversorgung sicherstellen&bull;CAN Buskabel zwischen Au&szlig;enund Inneneinheit instandsetzen oder austauschen"             };
	$ErrorCodeList{"0347"} = {errorcode => "A01",                                    subcode => "5112",                                               class => "8",           reason => "Gas-W&auml;rmepumpe: Kein CAN Bussignal am Steuerger&auml;t (GHP-SC) vorhanden",                                                                                                                                                                                                                                                                   checklist => "&Uuml;berpr&uuml;fen, ob das CAN  Buskabel  am Steuerger&auml;t (GHP-SC) korrekt angeschlossen ist.&bull;&Uuml;berpr&uuml;fen, ob das CAN  Buskabel  am Steuerger&auml;t (CU I/O) korrekt angeschlossen ist.&bull;&Uuml;berpr&uuml;fen, ob das CAN Buskabel zwischen den Steuerger&auml;ten GHP-SC und CU I/O besch&auml;digt ist.&bull;Spannungsversorgung am Steuerger&auml;t CU I/O pr&uuml;fen&bull;-",                                                                                           correction => "CAN Buskabel im Steuerger&auml;t (GHP-SC) korrekt anschlie&szlig;en&bull;CAN Buskabel im Steuerger&auml;t (CU I/O) korrekt anschlie&szlig;en&bull;CAN Buskabel zwischen den Steuerger&auml;ten GHP-SC und CU I/O austauschen&bull;Spannungsversorgung  am Steuerger&auml;t (CU I/O) korrigieren&bull;Steuerger&auml;t CU I/O tauschen&bull;Steuerger&auml;t GHP-SC tauschen"                     };
	$ErrorCodeList{"0348"} = {errorcode => "A01",                                    subcode => "5130",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Vorlauftemperaturf&uuml;hler",                                                                                                                                                                                                                                        checklist => "&Uuml;berpr&uuml;fen ob der Anschlussstecker am Vorlauftemperaturf&uuml;hler TTI-6 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Vorlauftemperaturf&uuml;hler TTI-6 am Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-6 messen&bull;&Uuml;berpr&uuml;fen, ob der Vorlauftemperaturf&uuml;hler TTI-6 korrekten Kontakt an der Vorlaufleitung hat.&bull;Position des Vorlauftemperaturf&uuml;hlers TTI-6 an der Vorlaufleitung &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Vorlauftemperaturf&uuml;hler TTI-6 vorhanden bzw. besch&auml;digt ist&bull;-",correction => "Anschlussstecker am Vorlauftemperaturf&uuml;hler TTI-6 komplett aufstecken.&bull;Anschlusskabel vom Vorlauftemperaturf&uuml;hler TTI-6 am Steuerger&auml;t (CU i/O) korrekt anschlie&szlig;en&bull;Vorlauftemperaturf&uuml;hler TTI-6 austauschen&bull;Kontakt zwischen Vorlauftemperaturf&uuml;hler TTI-6 und Vorlaufleitung herstellen&bull;Vorlauftemperaturf&uuml;hler TTI-6 an der korrekten Position der Vorlaufleitung montieren&bull;Isolierung am Vorlauftemperaturf&uuml;hler TTI-6 austauschen bzw. montieren&bull;Kabelbaum austauschen"};
	$ErrorCodeList{"0349"} = {errorcode => "A01",                                    subcode => "5131",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom R&uuml;cklauftemperaturf&uuml;hle",                                                                                                                                                                                                                                   checklist => "&Uuml;berpr&uuml;fen ob der Anschlussstecker am  R&uuml;cklauftemperaturf&uuml;hler TTI-5 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom R&uuml;cklauftemperaturf&uuml;hler TTI-5 am Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;Ohmschen Widerstand des R&uuml;cklauftemperaturf&uuml;hlers TTI-5 messen&bull;&Uuml;berpr&uuml;fen, ob der R&uuml;cklauftemperaturf&uuml;hler; TTI-5 korrekten Kontakt an der Vorlaufleitung hat.&bull;Position des R&uuml;cklauftemperaturf&uuml;hlers TTI-5 an der Vorlaufleitung &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am R&uuml;cklauftemperaturf&uuml;hler TTI-5 vorhanden bzw. besch&auml;digt ist&bull;-",correction => "Anschlussstecker am R&uuml;cklauftemperaturf&uuml;hler TTI-5 komplett aufstecken.&bull;Anschlusskabel vom R&uuml;cklauftemperaturf&uuml;hler TTI-5 am Steuerger&auml;t (CU i/O) korrekt anschlie&szlig;en&bull;R&uuml;cklauftemperaturf&uuml;hler TTI-5 austauschen&bull;Kontakt zwischen R&uuml;cklauftemperaturf&uuml;hler TTI-5 und R&uuml;cklaufleitung herstellen&bull;R&uuml;cklauftemperaturf&uuml;hler TTI-5 an der korrekten Position der R&uuml;cklaufleitung montieren&bull;Isolierung am R&uuml;cklauftemperaturf&uuml;hler TTI-6 austauschen bzw. montieren&bull;Kabelbaum austauschen"};
	$ErrorCodeList{"0350"} = {errorcode => "A01",                                    subcode => "5132",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Temperaturf&uuml;hler im Solevorlauf",                                                                                                                                                                                                                                checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Temperaturf&uuml;hler TTI-7 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TTI-7 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-7 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-7 &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-7 korrekten Kontakt zur Solevorlaufleitung hat&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Temperaturf&uuml;hler TTI-7 vorhanden bzw besch&auml;digt ist&bull;Solekreis auf Entnahmeleistung &uuml;berpr&uuml;fen&bull;-",correction => "Anschlussstecker am Temperaturf&uuml;hler TTI-7 komplett aufstecken&bull;Anschlusskabel vom Temperaturf&uuml;hler TTI-7 korrekt im Steuerger&auml;t (CU I/O) anschlie&szlig;en&bull;Kabelbaum autauschen&bull;Temperaturf&uuml;hler TTI-7 austauschen&bull;Temperaturf&uuml;hler TTI-7 an der korrekten Position der Solevorlaufleitung montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-7 austauschen bzw. montieren&bull;Korrekten Volumenstrom im Solekreis einstellen&bull;Umw&auml;lzpumpe im Solekreis austauschen"};
	$ErrorCodeList{"0351"} = {errorcode => "A01",                                    subcode => "5133",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Soler&uuml;cklauftemperaturf&uuml;hler",                                                                                                                                                                                                                              checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Temperaturf&uuml;hler TTI-8 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TTI-8 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-8 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-8 &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-8 korrekten Kontakt zur Soler&uuml;cklaufleitung hat&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Temperaturf&uuml;hler TTI-8 vorhanden bzw besch&auml;digt ist&bull;Solekreis auf Entnahmeleistung &uuml;berpr&uuml;fen&bull;-",correction => "Anschlussstecker am Temperaturf&uuml;hler TTI-8 komplett aufstecken&bull;Anschlusskabel vom Temperaturf&uuml;hler TTI-8 korrekt im Steuerger&auml;t (CU I/O) anschlie&szlig;en&bull;Kabelbaum autauschen&bull;Temperaturf&uuml;hler TTI-8 austauschen&bull;Temperaturf&uuml;hler TTI-8 an der korrekten Position der Soler&uuml;cklaufleitung montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-8 austauschen bzw. montieren&bull;Korrekten Volumenstrom im Solekreis einstellen&bull;Umw&auml;lzpumpe im Solekreis austauschen"};
	$ErrorCodeList{"0352"} = {errorcode => "A01",                                    subcode => "5134",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Temperaturf&uuml;hler am Verfl&uuml;ssigerausgang",                                                                                                                                                                                                                   checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Temperaturf&uuml;hler TTI-3 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TTI-3 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-3 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-3 &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-3 korrekten Kontakt zur K&auml;ltemittelleitung am Verfl&uuml;ssigerausgang hat&bull;-",correction => "Anschlussstecker am Temperaturf&uuml;hler TTI-3 komplett aufstecken&bull;Anschlusskabel vom Temperaturf&uuml;hler TTI-3 korrekt im Steuerger&auml;t (CU I/O) anschlie&szlig;en&bull;Kabelbaum autauschen&bull;Temperaturf&uuml;hler TTI-3 austauschen&bull;Temperaturf&uuml;hler TTI-3 an der korrekten Position der K&auml;ltemittelleitung am Verfl&uuml;ssigerausgang montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-3 austauschen bzw. montieren"};
	$ErrorCodeList{"0353"} = {errorcode => "A01",                                    subcode => "5135",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Temperaturf&uuml;hler am Verfl&uuml;ssigerausgang (Heizwasser)",                                                                                                                                                                                                      checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Temperaturf&uuml;hler TTI-12 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TTI-12 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-12 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-12 &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-12 korrekten Kontakt zur Heizwasserleitung am Verfl&uuml;ssigerausgang hat&bull;-",correction => "Anschlussstecker am Temperaturf&uuml;hler TTI-12 komplett aufstecken&bull;Anschlusskabel vom Temperaturf&uuml;hler TTI-12 korrekt im Steuerger&auml;t (CU I/O) anschlie&szlig;en&bull;Kabelbaum autauschen&bull;Temperaturf&uuml;hler TTI-12 austauschen&bull;Temperaturf&uuml;hler TTI-12 an der korrekten Position der Heizwasserleitung am Verfl&uuml;ssigerausgang montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-12 austauschen bzw. montieren"};
	$ErrorCodeList{"0354"} = {errorcode => "A01",                                    subcode => "5136",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Temperaturf&uuml;hler am Verdampferausgang",                                                                                                                                                                                                                          checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Temperaturf&uuml;hler TTI-4 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TTI-4 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-4 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-4 &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-12 korrekten Kontakt zur K&auml;ltemittelleitung am Verdampferausgang hat&bull;-",correction => "Anschlussstecker am Temperaturf&uuml;hler TTI-4 komplett aufstecken&bull;Anschlusskabel vom Temperaturf&uuml;hler TTI-4 korrekt im Steuerger&auml;t (CU I/O) anschlie&szlig;en&bull;Kabelbaum autauschen&bull;Temperaturf&uuml;hler TTI-4 austauschen&bull;Temperaturf&uuml;hler TTI-4 an der korrekten Position der K&auml;ltemittelleitung am Verdampferausgang montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-4 austauschen bzw. montieren"};
	$ErrorCodeList{"0355"} = {errorcode => "A01",                                    subcode => "5137",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Temperaturf&uuml;hler am Austreiberfu&szlig;",                                                                                                                                                                                                                        checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Temperaturf&uuml;hler TTI-1 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TTI-1 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-1 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-1 &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-1 korrekten Kontakt zum Generatorfu&szlig; hat&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-1 korrrekt isoliert ist&bull;Nennw&auml;rmebelastung am Brenner &uuml;berpr&uuml;fen&bull;Funktion der Umw&auml;lzpumpe im K&auml;ltemittelkreis (NH3) &uuml;berpr&uuml;fen",correction => "Anschlussstecker am Temperaturf&uuml;hler TTI-1 komplett aufstecken&bull;Anschlusskabel vom Temperaturf&uuml;hler TTI-1 korrekt im Steuerger&auml;t (CU I/O) anschlie&szlig;en&bull;Kabelbaum autauschen&bull;Temperaturf&uuml;hler TTI-1 austauschen&bull;Temperaturf&uuml;hler TTI-1 an der korrekten Position am Austreiberfu&szlig; montieren&bull;Isolierung am Temperaturf&uuml;hler T1I-4 austauschen bzw. montieren&bull;Brennereinstellwert f&uuml;r die Nennw&auml;rmebelastung korrigieren&bull;Werkskundendienst benachrichtigen"};
	$ErrorCodeList{"0356"} = {errorcode => "A01",                                    subcode => "5138",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Temperaturf&uuml;hler f&uuml;r Hei&szlig;gas",                                                                                                                                                                                                                        checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Temperaturf&uuml;hler TTI-2 komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TTI-2 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-2 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-2 &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-1 korrekten Kontakt zur K&auml;ltemittelleitung am Verfl&uuml;ssigerausgang hat&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-2 korrrekt isoliert ist&bull;Nennw&auml;rmebelastung am Brenner &uuml;berpr&uuml;fen&bull;Funktion der Umw&auml;lzpumpe im K&auml;ltemittelkreis (NH3) &uuml;berpr&uuml;fen",correction => "Anschlussstecker am Temperaturf&uuml;hler TTI-2 komplett aufstecken&bull;Anschlusskabel vom Temperaturf&uuml;hler TTI-2 korrekt im Steuerger&auml;t (CU I/O) anschlie&szlig;en&bull;Kabelbaum autauschen&bull;Temperaturf&uuml;hler TTI-2 austauschen&bull;Temperaturf&uuml;hler TTI-2 an der korrekten Position Der K&auml;ltemittelleitung am Verfl&uuml;ssigerausgang montieren&bull;Isolierung am Temperaturf&uuml;hler TT1-2 austauschen bzw. montieren&bull;Brennereinstellwert f&uuml;r die Nennw&auml;rmebelastung korrigieren&bull;Werkskundendienst benachrichtigen"};
	$ErrorCodeList{"0357"} = {errorcode => "A01",                                    subcode => "5139",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Niederdruckf&uuml;hler (K&auml;ltemittel)",                                                                                                                                                                                                                           checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Niederdruckf&uuml;hler PTI12 im K&auml;ltemittelkreis komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das  Anschlusskabel vom Niederdruckf&uuml;hler PTI12 am Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Niederdruckf&uuml;hler PTI12 im K&auml;ltemittelkreis besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob der Motor am 2 Wege Ventil korrekt montiert ist&bull;&Uuml;berpr&uuml;fen, ob der Motor am 2 Wege Ventil defekt ist&bull;Funktion des 2 Wege Ventiles vor dem Verfl&uuml;ssiger &uuml;berpr&uuml;fen&bull;-",correction => "Anschlussstecker am Niederdruckf&uuml;hler PTI12 im K&auml;ltemittelkreis komplett aufstecken&bull;Anschlusskabel vom Niederdruckf&uuml;hler PTI12 am Steuerger&auml;t (CU I/O) korrekt anschlie&szlig;en&bull;Kabelbaum austauschen&bull;Motor am 2 Wege Ventil korrekt auf das Ventilunterteil montieren&bull;Motor am 2 Wege Ventil austauschen&bull;2 Wege Ventil vor dem Verfl&uuml;ssiger austauschen&bull;Werkskundendienst benachrichtigen"};
	$ErrorCodeList{"0358"} = {errorcode => "A01",                                    subcode => "5140",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Steuerger&auml;t erh&auml;lt unzul&auml;ssige Werte vom Hochdruckf&uuml;hler (K&auml;ltemittel)",                                                                                                                                                                                                                             checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Hochdruckf&uuml;hler PTI11 im K&auml;ltemittelkreis komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das  Anschlusskabel vom Hochdruckf&uuml;hler PTI11 am Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Motor am 2 -Wege Ventil  besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob der Motor am 2 Wege Ventil korrekt montiert ist&bull;&Uuml;berpr&uuml;fen, ob der Motor am 2 Wege Ventil defekt ist&bull;Funktion des 2 Wege Ventiles vor dem Verfl&uuml;ssiger &uuml;berpr&uuml;fen&bull;-",correction => "Anschlussstecker am Hochdruckf&uuml;hler PTI11 im K&auml;ltemittelkreis komplett aufstecken&bull;Anschlusskabel vom Hochdruckf&uuml;hler PTI11 am Steuerger&auml;t (CU I/O) korrekt anschlie&szlig;en&bull;Anschlusskabel zum Motor am 2 Wege Ventil austauschen&bull;Motor am 2 Wege Ventil korrekt auf das Ventilunterteil montieren&bull;Motor am 2 Wege Ventil austauschen&bull;2 Wege Ventil vor dem Verfl&uuml;ssiger austauschen&bull;Werkskundendienst benachrichtigen"};
	$ErrorCodeList{"0359"} = {errorcode => "A01",                                    subcode => "5141",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Volumenstrom im Prim&auml;rkreis liegt au&szlig;erhalb des zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                         checklist => "Stellung der Wartungsh&auml;hne zwischen W&auml;rmepumpe und Pufferspeicher &uuml;berpr&uuml;fen.&bull;&Uuml;berpr&uuml;fen, ob sich Luft in den Verbindungsleitungen zwischen W&auml;rmepumpe und Pufferspeicher befindet.&bull;Funktion der Umw&auml;lzpumpe &uuml;berpr&uuml;fen&bull;Volumenstrom &uuml;berpr&uuml;fen&bull;Volumenstrom &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Volumenstromsensor FTI18 aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Volumenstromsensor FTI18 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;Volumenstrommesser FTI18 auf wasserseitige Verschmutzungen &uuml;berpr&uuml;fen&bull;-",correction => "Wartungsh&auml;hne zwischen W&auml;rmepumpe und Pufferspeicher &ouml;ffnen&bull;Verbindungsleitungen zwischen W&auml;rmepumpe und Pufferspeicher entl&uuml;ften&bull;Umw&auml;lzpumpe anwerfen ggf. austauschen&bull;Volumenstrom auf den zul&auml;ssigen Wert (7 100 l/ min) reduzieren&bull;Volumenstrom auf den zul&auml;ssigen Wert (7 100 l/ min) erh&ouml;hen&bull;Anschlussstecker am Volumenstromsensor FTI18 komplett aufstecken&bull;Anschlusskabel vom Volumenstromsensor FTI18 am Steuerger&auml;t (CU I/O) korrekt anschlie&szlig;en&bull;Ablagerungen/ Schmutz aus Volumenstromsensor FTI18 entfernen&bull;Volumenstromsensor FTI18 austauschen"};
	$ErrorCodeList{"0360"} = {errorcode => "A01",                                    subcode => "5142",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Volumenstrom im Solekreis liegt au&szlig;erhalb des zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                                checklist => "Stellung der Wartungsh&auml;hne im Solekreis &uuml;berpr&uuml;fen.&bull;Filter im Solekreis auf Verschmutzung &uuml;berpr&uuml;fen.&bull;Funktion der Umw&auml;lzpumpe  im Solekreis &uuml;berpr&uuml;fen&bull;Volumenstrom im Solekreis &uuml;berpr&uuml;fen&bull;Volumenstrom im Solekreis &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Volumenstromsensor FTI19 im Solekreis aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Volumenstromsensor FTI19 im Steuerger&auml;t (CU I/O) korrekt angeschlossen ist&bull;Volumenstrommesser FTI19 im Solekreis auf wasserseitige Verschmutzungen &uuml;berpr&uuml;fen&bull;-",correction => "Wartungsh&auml;hne im Solekreis &ouml;ffnen&bull;Filter im Solekreis reinigen&bull;Umw&auml;lzpumpe anwerfen ggf. austauschen&bull;Volumenstrom auf den zul&auml;ssigen Wert im Solekreis  (12 100 l/min) reduzieren&bull;Volumenstrom auf den zul&auml;ssigen Wert im Solrkeis  (12 100 l/min) erh&ouml;hen&bull;Anschlussstecker am Volumenstromsensor FTI19 im Solekreis komplett aufstecken&bull;Anschlusskabel vom Volumenstromsensor FTI19 am Steuerger&auml;t (CU I/O) korrekt anschlie&szlig;en&bull;Ablagerungen/ Schmutz aus Volumenstromsensor FTI19 im Solekreis entfernen&bull;Volumenstromsensor FTI19 im Solekreisaustauschen"};
	$ErrorCodeList{"0361"} = {errorcode => "A01",                                    subcode => "5143",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Unzul&auml;ssige Lufteintrittstemperatur an der Au&szlig;eneinheit",                                                                                                                                                                                                                                                          checklist => "&Uuml;berpr&uuml;fen, ob der Lufteintrittstemperaturf&uuml;hler an der Au&szlig;eneinheit verschmutzt ist&bull;&Uuml;berpr&uuml;fen, ob der Anschlusstecker am Lufteintrittstemperaturf&uuml;hler in der Au&szlig;eneinheit aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Lufteintrittstemperaturf&uuml;hler in der Au&szlig;eneinheit besch&auml;digt ist&bull;-",                                                                                                            correction => "Ablagerungen amLufteintrittstemperaturf&uuml;hler in der Au&szlig;eneinheit entfernen&bull;Anschlussstecker am Lufteintrittstemperaturf&uuml;hler in der Au&szlig;eneinheit aufstecken&bull;Anschlusskabel zum Lufteintrittstemperaturf&uuml;hler in der Au&szlig;eneinheit austauschen&bull;Lufteintrittstemperaturf&uuml;hler in der Au&szlig;eneinheit austauschen"                           };
	$ErrorCodeList{"0362"} = {errorcode => "A01",                                    subcode => "5144",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Unzul&auml;ssige Luftaustrittstemperatur an der Au&szlig;eneinheit",                                                                                                                                                                                                                                                          checklist => "&Uuml;berpr&uuml;fen, ob der Luftaustrittstemperaturf&uuml;hler an der Au&szlig;eneinheit verschmutzt ist&bull;&Uuml;berpr&uuml;fen, ob der Anschlusstecker am Luftaustrittstemperaturf&uuml;hler in der Au&szlig;eneinheit aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Luftaustrittstemperaturf&uuml;hler in der Au&szlig;eneinheit besch&auml;digt ist&bull;-",                                                                                                            correction => "Ablagerungen amLuftaustrittstemperaturf&uuml;hler in der Au&szlig;eneinheit entfernen&bull;Anschlussstecker am Luftaustrittstemperaturf&uuml;hler in der Au&szlig;eneinheit aufstecken&bull;Anschlusskabel zum Luftaustrittstemperaturf&uuml;hler in der Au&szlig;eneinheit austauschen&bull;Luftaustrittstemperaturf&uuml;hler in der Au&szlig;eneinheit austauschen"                           };
	$ErrorCodeList{"0363"} = {errorcode => "A01",                                    subcode => "5150",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Temperaturdifferenz im Solekreis liegt au&szlig;erhalb des zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                         checklist => "Filter im Solekreis reinigen&bull;Volumenstrom im Solekreis &uuml;berpr&uuml;fen&bull;Volumenstrom im Solekreis &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-7 korrekten Kontakt zur Solevorlaufleitung hat&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Temperaturf&uuml;hler TTI-7 am Solevorlauf vorhanden bzw besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-8 korrekten Kontakt zur Soler&uuml;cklaufleitung hat.&bull;&Uuml;berpr&uuml;fen, ob ob die Isolierung am Temperaturf&uuml;hler TTI-8 am Soler&uuml;cklauf vorhanden  bzw. besch&auml;digt ist&bull;Entnahmeleistung der Erdsonde &uuml;berpr&uuml;fen&bull;-",correction => "Filter im Solekreis reinigen&bull;Volumenstrom im Solekreis auf den zul&auml;ssigen Wert reduzieren (12 100 l/min)&bull;Volumenstrom im Solekreis auf den zul&auml;ssigen Wert erh&ouml;hen (12 100 l/min)&bull;Temperaturf&uuml;hler TTI-7 an der korrekten Position der Solevorlaufleitung montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-7 austauschen bzw. montieren&bull;Temperaturf&uuml;hler TTI-8 an der korrekten Position der Soler&uuml;cklaufleitung montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-8 austauschen bzw. montieren&bull;Erdsonde &uuml;berpr&uuml;fen&bull;Werkskundendienst beauftragen"};
	$ErrorCodeList{"0364"} = {errorcode => "A01",                                    subcode => "5151",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Temperaturdifferenz zwischen Sole und K&auml;ltemittelaustritt nach Verdampfer  ist au&szlig;erhalb des zul&auml;ssigen Bereiches",                                                                                                                                                                                           checklist => "&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-7 korrekten Kontakt zur Solevorlaufleitung hat&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Temperaturf&uuml;hler TTI-7 am Solevorlauf vorhanden bzw besch&auml;digt ist.&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-4 korrekten Kontakt zur K&auml;ltemittelleitung am Verdampferausgang hat.&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Temperaturf&uuml;hler TTI-4 vorhanden bzw. besch&auml;digt ist.&bull;Volumenstrom im Solekreis &uuml;berpr&uuml;fen",correction => "Temperaturf&uuml;hler TTI-7 an der korrekten Position der Solevorlaufleitung montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-7 austauschen bzw. montieren&bull;Temperaturf&uuml;hler TTI-4 an der korrekten Position der K&auml;ltemittelleitung am Verdampferausgang montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-4 austauschen bzw. montieren&bull;Volumenstrom im Solekreis auf die zul&auml;ssigen Werte (12 -100 l/min) erh&ouml;hen"};
	$ErrorCodeList{"0365"} = {errorcode => "A01",                                    subcode => "5152",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Temperaturdifferenz zwischen Verl&uuml;ssigeraustritt K&auml;ltekreis und Heizungsr&uuml;cklauf ist au&szlig;erhalb des zul&auml;ssigen Bereiches",                                                                                                                                                                           checklist => "&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-3 korrekten Kontakt zur K&auml;ltemittelleitung am Verfl&uuml;ssigerausgang hat&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Temperaturf&uuml;hler TTI-3 vorhanden bzw. besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-5 korrekten Kontakt an der R&uuml;ckllaufleitung hat.&bull;Position des R&uuml;cklauftemperaturf&uuml;hlers TTI-5 an der Vorlaufleitung &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am R&uuml;cklauftemperaturf&uuml;hler TTI-5 vorhanden bzw. besch&auml;digt ist",correction => "Temperaturf&uuml;hler TTI-3 an der korrekten Position der K&auml;ltemittelleitung am Verfl&uuml;ssigerausgang montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-3 austauschen bzw. montieren&bull;Kontakt zwischen R&uuml;cklauftemperaturf&uuml;hler TTI-5 und R&uuml;cklaufleitung herstellen&bull;R&uuml;cklauftemperaturf&uuml;hler TTI-5 an der korrekten Position der R&uuml;cklaufleitung montieren&bull;Isolierung am R&uuml;cklauftemperaturf&uuml;hler TTI-5 austauschen bzw. montieren"};
	$ErrorCodeList{"0366"} = {errorcode => "A01",                                    subcode => "5153",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Druckdifferenz im K&auml;ltemittelkreis ist au&szlig;erhalb des zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                    checklist => "Funktion der L&ouml;sungspumpe im K&auml;ltemittelkreis (NH3) &uuml;berpr&uuml;fen&bull;Funktion der Regelventile im K&auml;ltemittelkreis (NH3) &uuml;berpr&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                        correction => "Werkskundendienst benachrichtigen&bull;Werkskundendienst benachrichtigen&bull;Werkskundendienst benachrichtigen"                                                                                                                                                                                                                                                                                 };
	$ErrorCodeList{"0367"} = {errorcode => "A01",                                    subcode => "5161",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Fehler im Frequenzumrichter f&uuml;r Ansteuerung L&ouml;sungspumpe",                                                                                                                                                                                                                                                          checklist => "Motor an der L&ouml;sungspumpe im K&auml;ltemittelkreis &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob alle Anschlusskabel im Frequenzumrichter aufgesteckt sind&bull;-",                                                                                                                                                                                                                                                                                                                         correction => "Motor an der L&ouml;sungspumpe im K&auml;ltemittelkreis austauschen&bull;Anschlusskabel im Frequenzumrichter aufstecken&bull;Frequenzumrichter austauschen"                                                                                                                                                                                                                                      };
	$ErrorCodeList{"0368"} = {errorcode => "A01",                                    subcode => "5162",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Drehzahl vom Abluftventilator liegt au&szlig;erhalb des zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                            checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker f&uuml;r die Drehzahlregelung am Abluftventilator komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel f&uuml;r die Drehzahlregelung am Abluftventilator besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel f&uuml;r die Spannungsversorgung des Abluftventilators besch&auml;digt ist&bull;Funktion des Abluftventilators &uuml;berpr&uuml;fen&bull;-",                                                         correction => "Anschlussstecker f&uuml;r die Drehzahlregelung am Abluftventilator komplett aufstecken&bull;Anschlusskabel f&uuml;r die Drehzahlregelung des Abluftventilators austauschen&bull;Anschlusskabel f&uuml;r die Spannungsversorgung des Abluftventilators austauschen&bull;Abluftventilator austauschen&bull;Vorwiderstand f&uuml;r die Drehzahlbegrenzung des Abluftventilators austauschen"        };
	$ErrorCodeList{"0369"} = {errorcode => "A01",                                    subcode => "5163",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Gebl&auml;sedrehzahl au&szlig;erhalb des zul&auml;ssigen Bereiches",                                                                                                                                                                                                                                                          checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker f&uuml;r die Drehzahlregelung am Gebl&auml;se  komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel f&uuml;r die Drehzahlregelung am Gebl&auml;se besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel f&uuml;r die Spannungsversorgung des Gebl&auml;ses besch&auml;digt ist&bull;Funktion des Gebl&auml;sese &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Anschlussstecker f&uuml;r die Spannungsversorgung (230 VAC) am Gebl&auml;se komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob der Anschlussstecker f&uuml;r die Spannungsversorgung (230 VAC) des Gebl&auml;sese im Feuerungsautomat komplett aufgesteckt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel f&uuml;r die Spannungsversorgung (230 VAC) zum Gebl&auml;se besch&auml;digt ist&bull;-",correction => "Anschlussstecker f&uuml;r die Drehzahlregelung am Gebl&auml;se komplett aufstecken&bull;Anschlusskabel f&uuml;r die Drehzahlregelung des Gebl&auml;ses austauschen&bull;Anschlusskabel f&uuml;r die Spannungsversorgung des Gebl&auml;ses austauschen&bull;Gebl&auml;se austauschen&bull;Anschlussstecker f&uuml;r die Spannungsversorgung (230 VAC) am Gebl&auml;se komplett aufstecken&bull;Anschlussstecker f&uuml;r die Spannungsversorgung (230 VAC) zum Gebl&auml;se im Feuerungsautomat komplett aufstecken&bull;Anschlusskabel f&uuml;r die Spannungsversorgung (230 VAC) zum Gebl&auml;se austauschen&bull;Feuerungsautomat austauschen"};
	$ErrorCodeList{"0370"} = {errorcode => "A01",                                    subcode => "5164",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Fehler 2-Wege-Ventil am Verfl&uuml;ssiger (Heizungswasser)",                                                                                                                                                                                                                                                                  checklist => "&Uuml;berpr&uuml;fen Sie die Position des Motors am 2-Wege-Ventil&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel f&uuml;r das R&uuml;ckmeldesignal am Motor f&uuml;r das 2-Wege-Ventil besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel f&uuml;r die Spannungsversorgung des Motors am 2-Wege-Ventil besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob der Motor vom 2-Wege-Ventil blockiert ist&bull;-",                                                                       correction => "Position des Motors am 2-Wege-Ventil korrigieren&bull;Anschlusskabel f&uuml;r das R&uuml;ckmeldesignal am Motor des 2-Wege-Ventiles austauschen&bull;Anschlusskabel f&uuml;r die Spannungsversorgung des Abluftventilators austauschen&bull;Motor am 2-Wege-Ventil austauschen&bull;2-Wege-Ventil austauschen"                                                                                   };
	$ErrorCodeList{"0371"} = {errorcode => "A01",                                    subcode => "5170",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: F&uuml;llstand des K&auml;ltemittels (NH3) im Austreiber zu niedrig",                                                                                                                                                                                                                                                         checklist => "Anschlusskabel zum F&uuml;llstandssensor &uuml;berpr&uuml;fen&bull;F&uuml;llstandssensor f&uuml;r das K&auml;ltemittel im Austreiber &uuml;berpr&uuml;fen&bull;Anlagenhydraulik &uuml;berpr&uuml;fen&bull;Fremdgas aus K&auml;ltemittelkreis (NH3) entfernen&bull;Funktion der Regelventile im K&auml;ltemittelkreis (NH3) &uuml;berpr&uuml;fen&bull;-",                                                                                                                                              correction => "Anschlusskabel zum F&uuml;llstandssensor austauschen&bull;Werkskundendienst benachrichtigen&bull;Anlagenhydraulik korrigieren&bull;Werkskundendienst benachrichtigen&bull;Werkskundendienst benachrichtigen&bull;Werkskundendienst benachrichtigen"                                                                                                                                              };
	$ErrorCodeList{"0372"} = {errorcode => "A01",                                    subcode => "5171",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: R&uuml;ckstau Abgaskondensat im Abgasw&auml;rmetauscher",                                                                                                                                                                                                                                                                     checklist => "&Uuml;berpr&uuml;fen, ob der Kondensatanschlu&szlig; am Abgasw&auml;rmetauscher durch Ablagerungen blockiert ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Sensor FLS besch&auml;digt ist&bull;Funktion des Sensors FLS &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob die Kondensatleitung mit Gef&auml;lle verlegt worden ist&bull;-",                                                                                                                                               correction => "R&uuml;ckst&auml;nde am Kondensatanschlu&szlig; im Abgasw&auml;rmetauscher entfernn&bull;Anschlusskabel zum Sensor FLS austauschen&bull;Sensor FLS am Abgasw&auml;rmetauscher austauschen&bull;Kondensatleitung mit Gef&auml;lle verlegen&bull;Externen Kondensatsiphon reinigen"                                                                                                                };
	$ErrorCodeList{"0373"} = {errorcode => "A01",                                    subcode => "5172",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Aufheizzeit imK&auml;ltekreis nach Start zu lang",                                                                                                                                                                                                                                                                            checklist => "Ohmschen Widerstand des R&uuml;cklauftemperaturf&uuml;hlers TTI-5 messen&bull;&Uuml;berpr&uuml;fen, ob der R&uuml;cklauftemperaturf&uuml;hler; TTI-5 korrekten Kontakt an der Vorlaufleitung hat.&bull;Position des R&uuml;cklauftemperaturf&uuml;hlers TTI-5 an der Vorlaufleitung &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am R&uuml;cklauftemperaturf&uuml;hler TTI-5 vorhanden bzw. besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum R&uuml;cklauftemperaturf&uuml;hler TTI-5 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-6 messen&bull;&Uuml;berpr&uuml;fen, ob der Vorlauftemperaturf&uuml;hler TTI-6 korrekten Kontakt an der Vorlaufleitung hat&bull;Position des Vorlauftemperaturf&uuml;hlers TTI-6 an der Vorlaufleitung &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Vorlauftemperaturf&uuml;hler TTI-6 vorhanden bzw. besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-6 besch&auml;digt ist&bull;&Uuml;berpr&uuml;fen, ob das Anschlusskabel zum Temperaturf&uuml;hler TTI-12 besch&auml;digt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hlers TTI-12 &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob der Temperaturf&uuml;hler TTI-12 korrekten Kontakt zur Heizwasserleitung am Verfl&uuml;ssigerausgang hat&bull;&Uuml;berpr&uuml;fen, ob die Isolierung am Temperaturf&uuml;hler TTI-12 vorhanden bzw. besch&auml;digt ist&bull;Brennereinstellung bei minimaler Nennw&auml;rmebelastung &uuml;berpr&uuml;fen",correction => "R&uuml;cklauftemperaturf&uuml;hler TTI-5 austauschen&bull;Kontakt zwischen R&uuml;cklauftemperaturf&uuml;hler TTI-5 und R&uuml;cklaufleitung herstellen&bull;R&uuml;cklauftemperaturf&uuml;hler TTI-5 an der korrekten Position der R&uuml;cklaufleitung montieren&bull;Isolierung am R&uuml;cklauftemperaturf&uuml;hler TTI-5 austauschen bzw. montieren&bull;Kabelbaum austauschen&bull;Vorlauftemperaturf&uuml;hler TTI-6 austauschen&bull;Kontakt zwischen Vorlauftemperaturf&uuml;hler TTI-6 und Vorlaufleitung herstellen&bull;Vorlauftemperaturf&uuml;hler TTI-6 an der korrekten Position der Vorlaufleitung montieren&bull;Isolierung am Vorlauftemperaturf&uuml;hler TTI-6 austauschen bzw. montieren&bull;Kabelbaum austauschen&bull;Kabelbaum autauschen&bull;Temperaturf&uuml;hler TTI-12 austauschen&bull;Temperaturf&uuml;hler TTI-12 an der korrekten Position der Heizwasserleitung am Verfl&uuml;ssigerausgang montieren&bull;Isolierung am Temperaturf&uuml;hler TTI-12 austauschen bzw. montieren&bull;Brennereinstellung bei minimaler Nennw&auml;rmebelastung korrigieren"};
	$ErrorCodeList{"0374"} = {errorcode => "A01",                                    subcode => "5172",                                               class => "4",           reason => "",                                                                                                                                                                                                                                                                                                                                                 checklist => "Brennereinstellung  bei maximaler Nennw&auml;rmebelastung &uuml;berpr&uuml;fen&bull;Funktion der Regelventile im K&auml;ltemittelkreis &uuml;berp&uuml;fen&bull;-",                                                                                                                                                                                                                                                                                                                                   correction => "Brennereinstellung bei maximaler Nennw&auml;rmebelastung korrigieren&bull;Werkskundendienst benachrichtigen&bull;Werkskundendienst benachrichtigen&bull;Brennereinstellung bei maximaler Nennw&auml;rmebelastung korrigieren&bull;Werkskundendienst benachrichtigen"                                                                                                                             };
	$ErrorCodeList{"0375"} = {errorcode => "A01",                                    subcode => "5201",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Warnung Aussentemperaturf&uuml;hler T1 Kabelbruch",                                                                                                                                                                                                                                                                           checklist => "Anschluss Au&szlig;entemperaturf&uuml;hler pr&uuml;fen Au&szlig;entemperatuf&uuml;hler liegt au&szlig;erhalb des Messbereichs (Ohm-Werte pr&uuml;fen, s.u. Werte in Installationsanleitung)",                                                                                                                                                                                                                                                                                                         correction => "Au&szlig;entemperaturf&uuml;hler montieren Au&szlig;entemperaturf&uuml;hler tauschen Installationsmodul tauschen"                                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0376"} = {errorcode => "A01",                                    subcode => "5202",                                               class => "4",           reason => "Gas-W&auml;rmepumpe: Warnung Aussentemperaturf&uuml;hler T1 Kurzschluss",                                                                                                                                                                                                                                                                          checklist => "Anschlusskabel und Steckanschluss pr&uuml;fen (F&uuml;hlerwerte-Tabelle in Installationsanleitung)",                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Au&szlig;entemperaturf&uuml;hler tauschen Kabel erneuern Installationsmodul tauschen"                                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0377"} = {errorcode => "A01",                                    subcode => "5203",                                               class => "WPL AR",      reason => "Alarm Au&szlig;entemperaturf&uuml;hler T1 defekt",                                                                                                                                                                                                                                                                                                 checklist => "Siehe Pr&uuml;fvorgang/Ursache f&uuml;r Ursachencode 5201, 5202",                                                                                                                                                                                                                                                                                                                                                                                                                                     correction => "Siehe m&ouml;gliche Ma&szlig;nahmen f&uuml;r Ursachencode 5201, 5202"                                                                                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0378"} = {errorcode => "A01",                                    subcode => "5204",                                               class => "WPL AR",      reason => "Warnung Vorlauff&uuml;hler T0 Unterbrechung",                                                                                                                                                                                                                                                                                                      checklist => "Vorlauff&uuml;hler T0/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Vorlauff&uuml;hler T0 ej angezogen&bull;Vorlauff&uuml;hler T0/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                      correction => "Ohmwert von F&uuml;hler T0/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von I/O-Board-Klemme getrennt&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Vorlauff&uuml;hler T0 wechseln&bull;I/O-Board wechseln"                                                                                                                            };
	$ErrorCodeList{"0379"} = {errorcode => "A01",                                    subcode => "5205",                                               class => "WPL AR",      reason => "Warnung Vorlauff&uuml;hler T0 kurzgeschlossen",                                                                                                                                                                                                                                                                                                    checklist => "Vorlauff&uuml;hler T0/Signalkabel kurzgeschlossen&bull;Vorlauff&uuml;hler T0 kurzgeschlossen&bull;Signalkabel zu Vorlauff&uuml;hler T0 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                          correction => "Ohmwert von F&uuml;hler T0/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von I/O-Board-Klemme getrennt&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/Au&szlig;entemperatursensor T0 wechseln&bull;Signalkabel zu Vorlauff&uuml;hler T0 reparieren/wechseln&bull;I/O-Board wechseln"                  };
	$ErrorCodeList{"0380"} = {errorcode => "A01",                                    subcode => "5207",                                               class => "4",           reason => "Warnung Temperaturf&uuml;hler TC1 Kabelbruch",                                                                                                                                                                                                                                                                                                     checklist => "&Uuml;berpr&uuml;fen, ob der Anschlussstecker am Temperaturf&uuml;hler TC1 komplett aufgesteckt ist&bull;Ohmschen Widerstand des Temperaturf&uuml;hler TC1 messen&bull;-",                                                                                                                                                                                                                                                                                                                            correction => "Anschlussstecker am Temperaturf&uuml;hler TC1 komplett aufstecken.&bull;Temperaturf&uuml;hler TC 1 austauschen&bull;Anschlusskabel zum Temperatuf&uuml;hler TC1 austauschen"                                                                                                                                                                                                                     };
	$ErrorCodeList{"0381"} = {errorcode => "A01",                                    subcode => "5208",                                               class => "8",           reason => "Warnung Temperaturf&uuml;hler TC1 Kurzschluss",                                                                                                                                                                                                                                                                                                    checklist => "Ohmschen Widerstand des Temperaturf&uuml;hlers TC1 messen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                                    correction => "Temperaturf&uuml;hler TC1 austauschen&bull;Anschlusskabel zum Temperatuf&uuml;hler TC1 austauschen"                                                                                                                                                                                                                                                                                              };
	$ErrorCodeList{"0382"} = {errorcode => "A01",                                    subcode => "5210",                                               class => "WPL AR",      reason => "Warnung Z2; W&auml;rmetr&auml;ger in F&uuml;hler TC1 Unterbrechung",                                                                                                                                                                                                                                                                               checklist => "W&auml;rmetr&auml;ger in F&uuml;hler TC1 W&auml;rmepumpe 2/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Vorlauff&uuml;hler TC1 W&auml;rmepumpe 2 nicht angezogen&bull;W&auml;rmetr&auml;ger in F&uuml;hler TC1 W&auml;rmepumpe 2/ Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                     correction => "Ohmwert von F&uuml;hler TC1 W&auml;rmepumpe 2/ Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen. Das Innenteil ist dabei von I/O-Board-Klemme getrennt&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;W&auml;rmetr&auml;ger in F&uuml;hler TC1 W&auml;rmepumpe 2 wechseln&bull;I/O-Board wechseln"                                                                   };
	$ErrorCodeList{"0383"} = {errorcode => "A01",                                    subcode => "5211",                                               class => "WPL AR",      reason => "Warnung Z2 W&auml;rmetr&auml;ger in F&uuml;hler TC1 kurzgschlossen",                                                                                                                                                                                                                                                                               checklist => "W&auml;rmetr&auml;ger in F&uuml;hler TC1 W&auml;rmepumpe 2/ Signalkabel kurzgeschlossen&bull;W&auml;rmetr&auml;ger in F&uuml;hler TC1 W&auml;rmepumpe 2 kurzgeschlossen&bull;Signalkabel zu Vorlauff&uuml;hler T0 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                               correction => "Ohmwert von F&uuml;hler TC1 W&auml;rmepumpe 2/ Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen. Das Innenteil ist dabei von I/O-Board-Klemme getrennt.&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/W&auml;rmetr&auml;ger in F&uuml;hler TC1 W&auml;rmepumpe 2 wechseln&bull;Signalkabel zu W&auml;rmetr&auml;ger in F&uuml;hler TC1 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0384"} = {errorcode => "A01",                                    subcode => "5213",                                               class => "WPL AR",      reason => "Warnung Z1; W&auml;rmetr&auml;ger in F&uuml;hler TC0 Unterbrechung",                                                                                                                                                                                                                                                                               checklist => "W&auml;rmetr&auml;ger in F&uuml;hler TC0/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Vorlauff&uuml;hler TC0 ej angezogen&bull;Vorlauff&uuml;hler TC0/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                 correction => "Ohmwert von F&uuml;hler TC0/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von I/O-Board-Klemme getrennt&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Vorlauff&uuml;hler TC0 wechseln&bull;I/O-Board wechseln"                                                                                                                          };
	$ErrorCodeList{"0385"} = {errorcode => "A01",                                    subcode => "5214",                                               class => "WPL AR",      reason => "Warnung Z1; W&auml;rmetr&auml;ger in F&uuml;hler TC0 kurzgschlossen",                                                                                                                                                                                                                                                                              checklist => "W&auml;rmetr&auml;ger in F&uuml;hler TC0/Signalkabel kurzgeschlossen&bull;W&auml;rmetr&auml;ger in F&uuml;hler TC0 kurzgeschlossen&bull;Signalkabel zu W&auml;rmetr&auml;ger aus F&uuml;hler TC0 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                correction => "Ohmwert von F&uuml;hler TC0/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von I/O-Board-Klemme getrennt&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/W&auml;rmetr&auml;ger in F&uuml;hler TC0 wechseln&bull;Signalkabel zu W&auml;rmetr&auml;ger aus F&uuml;hler TC0 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0386"} = {errorcode => "A01",                                    subcode => "5216",                                               class => "WPL AR",      reason => "Warnung Z2 W&auml;rmetr&auml;ger aus F&uuml;hler TC0 Unterbrechung",                                                                                                                                                                                                                                                                               checklist => "W&auml;rmetr&auml;ger aus F&uuml;hler TC0 W&auml;rmepumpe 2/ Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Vorlauff&uuml;hler TC0 W&auml;rmepumpe 2 nicht angezogen&bull;Vorlauff&uuml;hler TC0/W&auml;rmepumpe 2 Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                      correction => "Ohmwert von F&uuml;hler TC0 W&auml;rmepumpe 2/ Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen. Das Innenteil ist dabei von I/O-Board-Klemme getrennt.&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Vorlauff&uuml;hler TC1 W&auml;rmepumpe 2 wechseln&bull;I/O-Board wechseln"                                                                                    };
	$ErrorCodeList{"0387"} = {errorcode => "A01",                                    subcode => "5217",                                               class => "WPL AR",      reason => "Warnung Z2 W&auml;rmetr&auml;ger aus F&uuml;hler TC0 kurzgschlossen",                                                                                                                                                                                                                                                                              checklist => "W&auml;rmetr&auml;ger inaus F&uuml;hler TC0 W&auml;rmepumpe 2/Signalkabel kurzgeschlossen&bull;W&auml;rmetr&auml;ger in F&uuml;hler TC0 W&auml;rmepumpe 2 kurzgeschlossen&bull;Signalkabel zu W&auml;rmetr&auml;ger aus F&uuml;hler TC0 W&auml;rmepumpe 2 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                       correction => "Ohmwert von F&uuml;hler TC0 W&auml;rmepumpe 2/ Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen. Das Innenteil ist dabei  von I/O-Board-Klemme getrennt.&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/W&auml;rmetr&auml;ger aus F&uuml;hler TC0 W&auml;rmepumpe 2 wechseln&bull;Signalkabel zu W&auml;rmetr&auml;ger aus F&uuml;hler TC0 W&auml;rmepumpe 2 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0388"} = {errorcode => "A01",                                    subcode => "5220",                                               class => "8",           reason => "Warnung Z1 Sole Eintrittstemperaturf&uuml;hler TB0 Kurzschluss",                                                                                                                                                                                                                                                                                   checklist => "&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TC3 im Steuerger&auml;t korrekt angeschlossen ist&bull;-",                                                                                                                                                                                                                                                                                                                                                                      correction => "Anschlusskabel vom Temperaturf&uuml;hler TC3 im Steuerger&auml;t korrekt anschlie&szlig;en&bull;Temperaturf&uuml;hler TC3 austauschen"                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0389"} = {errorcode => "A01",                                    subcode => "5222",                                               class => "8",           reason => "Warnung Z2 Sole Eintrittstemperaturf&uuml;hler TB0 Kabelbruch",                                                                                                                                                                                                                                                                                    checklist => "F&uuml;hler TB0 (Sole-Eintrittstemperaturf&uuml;hler) pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Sole-Eintrittstemperaturf&uuml;hler tauschen Installationsmodul tauschen"                                                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0390"} = {errorcode => "A01",                                    subcode => "5234",                                               class => "WPL AR",      reason => "Warnung Schwimmbadtemperaturf&uuml;hler TP1 Unterbrechung",                                                                                                                                                                                                                                                                                        checklist => "Schwimmbadtemperaturf&uuml;hler TP1/ Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Schwimmbadtemperaturf&uuml;hler TP1 nicht angezogen&bull;Schwimmbadtemperaturf&uuml;hler TP1/ Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                       correction => "Ohmwert von F&uuml;hler TP1/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von I/O-Board-Klemme getrennt&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Schwimmbadtemperaturf&uuml;hler TP1 wechseln&bull;I/O-Board wechseln"                                                                                                             };
	$ErrorCodeList{"0391"} = {errorcode => "A01",                                    subcode => "5235",                                               class => "WPL AR",      reason => "Warnung Schwimmbadtemperaturf&uuml;hler TP1 kurzgeschlossen",                                                                                                                                                                                                                                                                                      checklist => "Schwimmbadtemperaturf&uuml;hler TP1/ Signalkabel Unterbrechung&bull;Schwimmbadtemperatur TP1 kurzgeschlossen&bull;Signalkabel zu W&auml;rmetr&auml;ger aus Schwimmbadtemperaturf&uuml;hler TP1 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                  correction => "Ohmwert von Schwimmbadtemperaturf&uuml;hler TC1/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; Das Innenteil ist dabei von I/O-BoardKlemme getrennt&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/Schwimmbadtemperaturf&uuml;hler TP1 wechseln&bull;Signalkabel zu Schwimmbadtemperaturf&uuml;hler TP1 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0392"} = {errorcode => "A01",                                    subcode => "5237",                                               class => "4",           reason => "Warnung Warmwasserf&uuml;hler TW1 Kabelbruch",                                                                                                                                                                                                                                                                                                     checklist => "&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Warmwassertemperaturf&uuml;hler TW1 im Steuerger&auml;t korrekt angeschlossen ist&bull;Ohmschen Widerstand des Warmwassertemperaturf&uuml;hlers TW1 messen&bull;-",                                                                                                                                                                                                                                                                                   correction => "Anschlusskabel vom Warmwassertemperaturf&uuml;hler TW1 im Steuerger&auml;t korrekt anschlie&szlig;en&bull;Warmwassertemperaturf&uuml;hler TW1 austauschen&bull;Anschlusskabel zum Warmwassertemperaturf&uuml;hler TW1 austauschen"                                                                                                                                                               };
	$ErrorCodeList{"0393"} = {errorcode => "A01",                                    subcode => "5238",                                               class => "4",           reason => "Warnung Warmwasserf&uuml;hler TW1 Kurzschluss",                                                                                                                                                                                                                                                                                                    checklist => "Ohmschen Widerstand des Warmwassertemperaturf&uuml;hlers TW1 messen&bull;-",                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Warmwassertemperaturf&uuml;hler TW1 austauschen&bull;Anschlusskabel zum Warmwassertemperaturf&uuml;hler TW1 austauschen"                                                                                                                                                                                                                                                                         };
	$ErrorCodeList{"0394"} = {errorcode => "A01",                                    subcode => "5246",                                               class => "WPL AR",      reason => "Alarm Z1 ElektroHeizkassette E2 ausgel&ouml;st &Uuml;berhitzungsschutz/ Druckw&auml;chter",                                                                                                                                                                                                                                                        checklist => "Niedriger Systemdruck, Luft im Heizsystem&bull;Schmutz im Systemfilter/Filterball SC1&bull;Schlechte Zirkulation im W&auml;rmetr&auml;gersystem/Heizsystem&bull;Sicherung F1 defekt&bull;Druckw&auml;chter defekt&bull;&Uuml;berhitzungsschutz defekt&bull;I/O-Board defekt, kein PWM-Signal zur Umw&auml;lzpumpe&bull;Sicherung F50 I/O-Board defekt&bull;I/O-Board defekt, keine 230 V an Klemme 01 (51, N)&bull;I/O-Board defekt&bull;Umw&auml;lzpumpe defekt&bull;Sch&uuml;tz zu elektr. Zuheizer bleibt in der geschlossenen Position stecken",correction => "Das Heizsystem laut Anweisungen in der Gebrauchsanweisung entl&uuml;ften, Wasser im Heizzystem&bull;Systemfilter/Filterball SC1 reinigen&bull;Einstellventile/Heizk&ouml;rperthermostate &uuml;berpr&uuml;fen&bull;Sicherung F1 wechseln&bull;Druckw&auml;chter wechseln&bull;Schalttemperatur (96°C) &uuml;berpr&uuml;fen &Uuml;berhitzungsschutz wechseln&bull;PWM-Signal an Klemme 36,37 am I/O-Board abschalten, Umw&auml;lzpumpe auf 100 % erh&ouml;hen&bull;Sicherung F50 wechseln&bull;&Uuml;berpr&uuml;fen, ob 230 V an Klemme 01 (51, N) am I/O-Board&bull;I/O-Board wechseln&bull;Umw&auml;lzpumpe wechseln&bull;&Uuml;berpr&uuml;fen/Sch&uuml;tz wechseln"};
	$ErrorCodeList{"0395"} = {errorcode => "A01",                                    subcode => "5247",                                               class => "WPL AR",      reason => "Alarm Z2 Elektro-Heiz-kassette E2 ausgel&ouml;st &Uuml;berhitzungsschutz/Druckw&auml;chter",                                                                                                                                                                                                                                                       checklist => "Siehe Alarm 5246 oben",                                                                                                                                                                                                                                                                                                                                                                                                                                                                               correction => "Siehe Ma&szlig;nahmen f&uuml;r ALARM 5246 oben"                                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0396"} = {errorcode => "A01",                                    subcode => "5265",                                               class => "WPL AR",      reason => "Warnung Heizpumpenkarte Z1 getrennt",                                                                                                                                                                                                                                                                                                              checklist => "Loser Kontakt in CANbus-Anschl&uuml;ssen an Installer-Board/ I/O-Board W&auml;rmepumpe&bull;Unterbrechung am CANbus-Kabel zwischen Tower und W&auml;rmepumpe&bull;Falscher Typ CANbus-Kabel&bull;CANbus-Kabel zusammen mit Stromversorgung der W&auml;rmepumpe verlegt&bull;Fehlerhafte Erdung des CANbus-Kabels",                                                                                                                                                                                    correction => "CANbus-Anschl&uuml;sse an Installer-Board/ I/O-Board W&auml;rmepumpe &uuml;berpr&uuml;fen&bull;CANbus-Kabel zwischen Tower und W&auml;rmepumpe &uuml;berpr&uuml;fen&bull;Zum richtigen Typ von Kabel wechseln&bull;CANbusund Stromkabel mit einem Abstand von mindestens 100 mm verlegen&bull;Kabelschutz zur/von der Erdung entfernen/abschlie&szlig;en"                                        };
	$ErrorCodeList{"0397"} = {errorcode => "A01",                                    subcode => "5267",                                               class => "WPL AR",      reason => "Warnung Heizpumpenkarte Z2 getrennt",                                                                                                                                                                                                                                                                                                              checklist => "Loser Kontakt in CANbus-Anschl&uuml;ssen an Installer-Board/ I/O-Board W&auml;rmepumpe 2&bull;Unterbrechung am CANbus-Kabel zwischen Tower und W&auml;rmepumpe 2&bull;Falscher Typ CANbus-Kabel&bull;CANbus-Kabel zusammen mit Stromversorgung der W&auml;rmepumpe 2 verlegt&bull;Fehlerhafte Erdung des CANbus-Kabels",                                                                                                                                                                              correction => "CANbus-Anschl&uuml;sse an Installer-Board/ I/O-Board W&auml;rmepumpe 2 &uuml;berpr&uuml;fen&bull;CANbus-Kabel zwischen Tower und W&auml;rmepumpe 2 wechseln&bull;Zum richtigen Typ von Kabel wechseln&bull;CANbusund Stromkabel mit einem Abstand von mindestens 100 mm verlegen&bull;Kabelschutz zur/von der Erdung entfernen/abschlie&szlig;en"                                                };
	$ErrorCodeList{"0398"} = {errorcode => "A01",                                    subcode => "5269",                                               class => "WPL AR",      reason => "Alarm Z1 Elektro-Heizkassette EE zu hohe Temperatur",                                                                                                                                                                                                                                                                                              checklist => "Schmutz im Systemfilter/Filterball SC1&bull;Schlechte Zirkulation im W&auml;rmetr&auml;gersystem/ Heizsystem&bull;F&uuml;hler TC1 defekt",                                                                                                                                                                                                                                                                                                                                                            correction => "Systemfilter/Filterball SC1 reinigen&bull;Einstellventile/Heizk&ouml;rperthermostate &uuml;berpr&uuml;fen&bull;F&uuml;hler TC1 im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/F&uuml;hler TC1 wechseln"                                                                                                                                                                      };
	$ErrorCodeList{"0399"} = {errorcode => "A01",                                    subcode => "5270",                                               class => "WPL AR",      reason => "Alarm Z2 Elektro-Heizkassette EE zu hohe Temperatur",                                                                                                                                                                                                                                                                                              checklist => "Siehe Ursachen f&uuml;r Warnung 5269 oben",                                                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Siehe Ma&szlig;nahmen f&uuml;r ALARM 5269 oben"                                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0400"} = {errorcode => "A01",                                    subcode => "5271",                                               class => "WPL AR",      reason => "Alarm Heizsystem 1 hohe Vorlauftemperatur",                                                                                                                                                                                                                                                                                                        checklist => "F&uuml;hler T0 defekt&bull;3-Wege-Ventil VW1 wechselt nicht aus WW-Betriebsart&bull;&Uuml;berpr&uuml;fen, ob Klemme 53 (O3) an I/O-Board im WW-Betrieb 230 V abgibt&bull;&Uuml;berpr&uuml;fen, ob Klemme 53 (O3) an I/O-Board im WW-Betrieb 230 V abgibt&bull;&Uuml;berhitzungsschutz ausgel&ouml;st",                                                                                                                                                                                                correction => "F&uuml;hler TC0 im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/ F&uuml;hler TC0 wechseln&bull;Funktion 3-Wege-Ventil VW1 &uuml;berpr&uuml;fen&bull;Wenn 230 V im WW-Betrieb, VW1 Motorteil/Kabel wechseln&bull;Wenn keine 230 V im WW-Betrieb, I/O-Board wechseln&bull;&Uuml;berhitzungsschutz r&uuml;cksetzen"                                                              };
	$ErrorCodeList{"0401"} = {errorcode => "A01",                                    subcode => "5272",                                               class => "WPL AR",      reason => "Alarm externe Zuheizung EM funktioniert nicht",                                                                                                                                                                                                                                                                                                    checklist => "Siehe Ursachen f&uuml;r Alarm 5246&bull;Sicherung in Stromzentrale ausgel&ouml;st",                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Siehe Ma&szlig;nahmen f&uuml;r Alarm 5246&bull;Sicherung in Stromzentrale wechseln/r&uuml;cksetzen"                                                                                                                                                                                                                                                                                              };
	$ErrorCodeList{"0402"} = {errorcode => "A01",                                    subcode => "5273",                                               class => "WPL AR",      reason => "Alarm W&auml;rmepumpe Z1 Phasenw&auml;chter",                                                                                                                                                                                                                                                                                                      checklist => "Sicherung im Tower ausgel&ouml;st&bull;Phase/Phasen fehlen an ankommender Klemme W&auml;rmepumpe&bull;Phase/Phasen fehlen an ankommender Klemme am EMI-Filter im Inverter&bull;Wenn alle Phasen am EMI-Filter unter Spannung stehen, ist der Inverter defekt",                                                                                                                                                                                                                                        correction => "Sicherung im Tower r&uuml;cksetzen&bull;&Uuml;berpr&uuml;fen, ob alle Phasen an ankommender Klemme der W&auml;rmepumpe vorhanden sind&bull;&Uuml;berpr&uuml;fen, ob alle Phasen an ankommender Klemme am EMI-Filter vorhanden sind&bull;Inverter wechseln"                                                                                                                                       };
	$ErrorCodeList{"0403"} = {errorcode => "A01",                                    subcode => "5274",                                               class => "WPL AR",      reason => "Alarm W&auml;rmepumpe Z2 Phasenw&auml;chter",                                                                                                                                                                                                                                                                                                      checklist => "Siehe Ursachen f&uuml;r Alarm 5273",                                                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Siehe Ma&szlig;nahmen f&uuml;r Alarm 5273"                                                                                                                                                                                                                                                                                                                                                       };
	$ErrorCodeList{"0404"} = {errorcode => "A01",                                    subcode => "5298",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 zu hoher Druck an JR1",                                                                                                                                                                                                                                                                                                 checklist => "Schmutz im Systemfilter/Filterball SC1&bull;Schlechte Zirkulation im W&auml;rmetr&auml;gersystem/ Heizsystem&bull;Luft im W&auml;rmetr&auml;gersystem/Heizsystem&bull;Fehler an F&uuml;hler TC3, TC0, T0&bull;VW1 wechselt niciht von Warmwasser zu Heizung&bull;Installer-Board defekt&bull;Installer-Board defekt, kein PWM-Signal&bull;Installer-Board defekt, keine 230 V an Klemme 01 (51, N)&bull;Installer-Board defekt",                                                                      correction => "Systemfilter/Filterball SC1 reinigen&bull;Ausreichende Stromrate sicherstellen&bull;Das Heizsystem laut Anweisungen in der Gebrauchsanweisung entl&uuml;ften, Wasser im Heizzystem auff&uuml;llen&bull;F&uuml;hlerwert im Vergleich zur aktuellen F&uuml;hlertabelle und&bull;VW1 Position A = Warmwasser, B = Heizk&ouml;rper &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, dass Klemme 53 nur bei Warmwasserbedarf 230 V gibt&bull;PWM-Signal an Klemme 36,37 am InstallerBoard abschalten, Umw&auml;lzpumpe auf 100 % erh&ouml;hen, wenn nicht Umw&auml;lzpumpe wechseln&bull;&Uuml;berpr&uuml;fen, ob 230 V an Klemme 01 (51, N) am Installer-Board vorhanden sind&bull;Installer-Board wechseln"};
	$ErrorCodeList{"0405"} = {errorcode => "A01",                                    subcode => "5300",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 ausgel&ouml;ster Hochdruckpressostat",                                                                                                                                                                                                                                                                                  checklist => "Pressostat au&szlig;er Funktion&bull;Unterbrechung/schlechte Verbindung im Kabelbaum zwischen MR1 und Inverter&bull;Unterbrechung/schlechte Verbindung im Kabelbaum zwischen MR1 und Inverter",                                                                                                                                                                                                                                                                                                       correction => "Aufgebrochener Kreis <43bar = Pressostat wechseln&bull;Kabelbaum/Anschl&uuml;sse zwischen MR1 und Inverter &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob Hochdruckw&auml;chter angeschlossen ist"                                                                                                                                                                                           };
	$ErrorCodeList{"0406"} = {errorcode => "A01",                                    subcode => "5302",                                               class => "WPL AR",      reason => "Warnung Z1 hohe Temperatur am Kompressortreiber",                                                                                                                                                                                                                                                                                                  checklist => "Schlechte W&auml;rme&uuml;bertragung zur K&uuml;hlschlange",                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Verschraubung zur K&uuml;hlschlange &uuml;berpr&uuml;fen"                                                                                                                                                                                                                                                                                                                                        };
	$ErrorCodeList{"0407"} = {errorcode => "A01",                                    subcode => "5304",                                               class => "WPL AR",      reason => "Warnung Z2 hohe Temperatur am Kompressortreiber",                                                                                                                                                                                                                                                                                                  checklist => "Siehe Warnung 5302 oben",                                                                                                                                                                                                                                                                                                                                                                                                                                                                             correction => "Siehe Ma&szlig;nahmen f&uuml;r Warnung 5302 oben"                                                                                                                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0408"} = {errorcode => "A01",                                    subcode => "5310",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Hei&szlig;gastemperatur zu hoch",                                                                                                                                                                                                                                                                                       checklist => "Sauggas-&Uuml;berhitzung zu hoch&bull;Sauggas-&Uuml;berhitzung &uuml;berpr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "&Uuml;berpr&uuml;fen, dass Sauggas-&Uuml;berhitzung 10C nicht kontinuierlich &uuml;bersteigt&bull;Sauggas-&Uuml;berhitzung = TR5-JR0"                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0409"} = {errorcode => "A01",                                    subcode => "5312",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Hei&szlig;gastemperatur; zu hoch",                                                                                                                                                                                                                                                                                      checklist => "Sauggas-&Uuml;berhitzung zu hoch&bull;Sauggas-&Uuml;berhitzung &uuml;berpr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "&Uuml;berpr&uuml;fen, dass Sauggas-&Uuml;berhitzung 10C nicht kontinuierlich &uuml;bersteigt&bull;Sauggas-&Uuml;berhitzung = TR5-JR0"                                                                                                                                                                                                                                                            };
	$ErrorCodeList{"0410"} = {errorcode => "A01",                                    subcode => "5314",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Heizgasf&uuml;hler TR6 Unterbrechung",                                                                                                                                                                                                                                                                                  checklist => "Sauggas-&Uuml;berhitzung &uuml;berpr&uuml;fen&bull;Heizgasf&uuml;hler TR6/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Heizgasf&uuml;hler TR6 nicht angezogen&bull;Heizgasf&uuml;hler TR6/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                             correction => "Sauggas-&Uuml;berhitzung = TR5-JR0&bull;Ohmwert von F&uuml;hler TR6/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/O-Board-Klemme, &uuml;berpr&uuml;fen&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Heizgasf&uuml;hler TR6 wechseln&bull;I/O-Board wechseln"                                                               };
	$ErrorCodeList{"0411"} = {errorcode => "A01",                                    subcode => "5315",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Heizgasf&uuml;hler TR6 Kurzschluss",                                                                                                                                                                                                                                                                                    checklist => "Heizgasf&uuml;hler TR6/Signalkabel kurzgeschlossen&bull;Signalkabel zum Heizgasf&uuml;hler kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                                                      correction => "Ohmwert von F&uuml;hler TR6/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/O-Board-Klemme, &uuml;berpr&uuml;fen&bull;Signalkabel zum Heizgasf&uuml;hler TR6 reparieren/wechseln&bull;I/O-Board wechseln"                                                                                                                    };
	$ErrorCodeList{"0412"} = {errorcode => "A01",                                    subcode => "5317",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z2 Heizgasf&uuml;hler TR6 Unterbrechung",                                                                                                                                                                                                                                                                                 checklist => "Heizgasf&uuml;hler TR6/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Heizgasf&uuml;hler TR6 nicht angezogen&bull;Heizgasf&uuml;hler TR6/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                correction => "Ohmwert von F&uuml;hler TR6/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/O-Board-Klemme, &uuml;berpr&uuml;fen&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Heizgasf&uuml;hler TR6 wechseln&bull;I/O-Board wechseln"                                                                                                       };
	$ErrorCodeList{"0413"} = {errorcode => "A01",                                    subcode => "5318",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z2 Heizgasf&uuml;hler TR6 Kurzschluss",                                                                                                                                                                                                                                                                                   checklist => "Heizgasf&uuml;hler TR6/Signalkabel kurzgeschlossen&bull;Heizgasf&uuml;hler TR6 kurzgeschlossen&bull;Signalkabel zum Heizgasf&uuml;hler kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                          correction => "Ohmwert von F&uuml;hler TR6/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/Heizgasf&uuml;hler TR6 wechseln&bull;Signalkabel zu W&auml;rmetr&auml;ger in F&uuml;hler TC1 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0414"} = {errorcode => "A01",                                    subcode => "5320",                                               class => "8",           reason => "Warnung Temperaturf&uuml;hler TC3 Kabelbruch",                                                                                                                                                                                                                                                                                                     checklist => "&Uuml;berpr&uuml;fen, ob das Anschlusskabel vom Temperaturf&uuml;hler TC3 im Steuerger&auml;t korrekt angeschlossen ist&bull;-",                                                                                                                                                                                                                                                                                                                                                                      correction => "Anschlusskabel vom Temperaturf&uuml;hler TC3 im Steuerger&auml;t korrekt anschlie&szlig;en&bull;Temperaturf&uuml;hler TC3 austauschen"                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0415"} = {errorcode => "A01",                                    subcode => "5321",                                               class => "4",           reason => "Warnung Temperaturf&uuml;hler TC3 Kurzschluss",                                                                                                                                                                                                                                                                                                    checklist => "Temperaturf&uuml;hler/Kabel TC3 pr&uuml;fen",                                                                                                                                                                                                                                                                                                                                                                                                                                                         correction => "Temperaturf&uuml;hler TC3 austauschen"                                                                                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0416"} = {errorcode => "A01",                                    subcode => "5323",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Kondensatorf&uuml;hler TC3 Unterbrechung",                                                                                                                                                                                                                                                                              checklist => "Kondensatorf&uuml;hler TC3/Signalkabel kurzgeschlossen&bull;Kondensatorf&uuml;hler TC3 kurzgeschlossen&bull;Signalkabel zu Vorlauff&uuml;hler T0 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                correction => "Ohmwert von F&uuml;hler TC3/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle und tats&auml;chlichen Temperatur &uuml;ber-pr&uuml;fen/Kondensatorf&uuml;hler TC3 wechseln&bull;Signalkabel zu Kondensatorf&uuml;hler TC3 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0417"} = {errorcode => "A01",                                    subcode => "5324",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Kondensatorf&uuml;hler TC3 kurzgeschlossen",                                                                                                                                                                                                                                                                            checklist => "Kondensatorf&uuml;hler TC3/Signalkabel kurzgeschlossen&bull;Kondensatorf&uuml;hler TC3 kurzgeschlossen&bull;Signalkabel zu Kondensatorf&uuml;hler/ F&uuml;hler TC3 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                              correction => "Ohmwert von F&uuml;hler TC3/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hler-tabelle und tats&auml;chlichen Temperatur &uuml;ber-pr&uuml;fen/Kondensatorf&uuml;hler TC3 wechseln&bull;Signalkabel/F&uuml;hler zu Kondensatorf&uuml;hler TC3 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0418"} = {errorcode => "A01",                                    subcode => "5330",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Kommunikations-; fehler zum Kompressortreiber",                                                                                                                                                                                                                                                                         checklist => "Keine Spannung zum Au&szlig;enteil&bull;St&ouml;rung am MODbus&bull;CANbus-Kabel zwischen IDU, ODU fehlerhaft verlegt",                                                                                                                                                                                                                                                                                                                                                                               correction => "230/400 V-Anschl&uuml;sse in Tower und ODU &uuml;berpr&uuml;fen&bull;MODbus-Kabel/Anschlussklemmen zwischen I/O-Board und Inverter &uuml;berpr&uuml;fen&bull;Kabelbaum und Verlegung (mindestens 100 mm zwischen Stromkabel und Canbus-Kabel) &uuml;berpr&uuml;fen"                                                                                                                              };
	$ErrorCodeList{"0419"} = {errorcode => "A01",                                    subcode => "5332",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Kommunikations-; fehler zum Kompressortreiber",                                                                                                                                                                                                                                                                         checklist => "Keine Spannung zum Au&szlig;enteil&bull;St&ouml;rung am MODbus&bull;CANbus-Kabel zwischen IDU, ODU fehlerhaft verlegt",                                                                                                                                                                                                                                                                                                                                                                               correction => "230/400 V-Anschl&uuml;sse in Tower und ODU &uuml;berpr&uuml;fen&bull;MODbus-Kabel/Anschlussklemmen zwischen I/O-Board und Inverter &uuml;berpr&uuml;fen&bull;Kabelbaum und Verlegung (mindestens 100 mm zwischen Stromkabel und Canbus-Kabel) &uuml;berpr&uuml;fen"                                                                                                                              };
	$ErrorCodeList{"0420"} = {errorcode => "A01",                                    subcode => "5350",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 asynchroner Betrieb am Kompressor",                                                                                                                                                                                                                                                                                     checklist => "Spiel im Kabelbaum zwischen Kompressor und Inverter&bull;&Ouml;l/Fl&uuml;ssigkeit im Kompressor bei Startversuch&bull;Kompressorf&uuml;hler TR1 zeigt falschen Wert&bull;Wenn I/O-Board ausgetauscht wird, fehlerhafte Einstellung des Drehgebers am I/O-Board",                                                                                                                                                                                                                                      correction => "Kabelbaum/Anschl&uuml;sse zwischen Kompressor und Inverter &uuml;berpr&uuml;fen&bull;Kompressorf&uuml;hler TR1 zeigt falschen Wert&bull;Kompressorf&uuml;hler TR1 im Vergleich zur aktuellen F&uuml;hlertabelle und tats&auml;chlichen Temperatur &uuml;berpr&uuml;fen&bull;Einstellung im Vergleich zum aktuellen elektrischen Schaltplan &uuml;berpr&uuml;fen"                                 };
	$ErrorCodeList{"0421"} = {errorcode => "A01",                                    subcode => "5351",                                               class => "WPL AR",      reason => "Alarm W&auml;rmepumpe Z1 asynchroner Betrieb am Kompressor",                                                                                                                                                                                                                                                                                       checklist => "Siehe Pr&uuml;fvorgang/Ursachen f&uuml;r Ursachencode 5350",                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Siehe m&ouml;gliche Ma&szlig;nahmen f&uuml;r Ursachencode 5350"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0422"} = {errorcode => "A01",                                    subcode => "5353",                                               class => "WPL AR",      reason => "Alarm W&auml;rmepumpe Z2 asynchroner Betrieb am Kompressor",                                                                                                                                                                                                                                                                                       checklist => "Siehe Pr&uuml;fvorgang/Ursachen f&uuml;r Ursachencode 5352",                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Siehe m&ouml;gliche Ma&szlig;nahmen f&uuml;r Ursachencode 5352"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0423"} = {errorcode => "A01",                                    subcode => "5354",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 &Uuml;berstrom Kompressor",                                                                                                                                                                                                                                                                                             checklist => "Unterbrechung/Kurzschluss im Kabelbaum zwischen Inverter und Kompressor&bull;Stromfehler im Kompressor&bull;Inverter au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                                                           correction => "Kabelbaum zwischen Inverter und Kompressor &uuml;berpr&uuml;fen&bull;Widerstand zwischen Wicklungen und Erdung messen; bei Widerstand; < 10 kOhm, Kompressor wechseln&bull;Inverter wechseln"                                                                                                                                                                                                    };
	$ErrorCodeList{"0424"} = {errorcode => "A01",                                    subcode => "5356",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 &Uuml;berstrom Kompressor",                                                                                                                                                                                                                                                                                             checklist => "Unterbrechung/Kurzschluss im Kabelbaum zwischen Inverter und Kompressor&bull;Stromfehler im Kompressor&bull;Inverter au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                                                           correction => "Kabelbaum zwischen Inverter und Kompressor &uuml;berpr&uuml;fen&bull;Widerstand zwischen Wicklungen und Erdung messen; bei Widerstand < 10 kOhm, Kompressor wechseln&bull;Inverter wechseln"                                                                                                                                                                                                     };
	$ErrorCodeList{"0425"} = {errorcode => "A01",                                    subcode => "5360",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 PFC &Uuml;berstrom des Kompressortreibers",                                                                                                                                                                                                                                                                             checklist => "Kurze Tiefe bei der Strommessung/ Spannungsmessung&bull;Interner Fehler im Inverter",                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "Bei wiederholten Warnungen Kontakt zum Stromanbieter aufnehmen&bull;Inverter wechseln"                                                                                                                                                                                                                                                                                                           };
	$ErrorCodeList{"0426"} = {errorcode => "A01",                                    subcode => "5362",                                               class => "WPL AR",      reason => "Info W&auml;rmepumpe Z1 &Uuml;berspannung",                                                                                                                                                                                                                                                                                                        checklist => "Zu hohe Spannung zur W&auml;rmepumpe&bull;Zu hohe eingehende Netzspannung",                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Hauptsicherungen in Bezug auf schlechten Kontakt &uuml;berpr&uuml;fen&bull;Bei wiederholten Warnungen Kontakt zum Stromanbieter aufnehmen"                                                                                                                                                                                                                                                       };
	$ErrorCodeList{"0427"} = {errorcode => "A01",                                    subcode => "5366",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 niedrige &Uuml;berhitzung",                                                                                                                                                                                                                                                                                             checklist => "Motor nicht ordnungsgem&auml;&szlig; montiert&bull;&Uuml;berpr&uuml;fen, dass Ventile sich in der richtigen Reihenfolge &ouml;ffnen/schlie&szlig;en&bull;Expansionsventilmotor defekt&bull;Expansionsventil VR1 &ouml;ffnet sich zu weit/bleibt in der offenen Stellung stecken",                                                                                                                                                                                                                     correction => "Ventilmotor &uuml;berpr&uuml;fen&bull;Test Au&szlig;eneinheit aktivieren&bull;Widerstand zwischen Kabel Orange, Rot, Gelb und Schwarz zu Grau messen; 46 kOhm = OK, bei Unterbrechung oder Kurzschluss, Ventilmotor wechseln&bull;Bei wiederholten Warnungen, Expansionsventil wechseln"                                                                                                         };
	$ErrorCodeList{"0428"} = {errorcode => "A01",                                    subcode => "5368",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 niedrige &Uuml;berhitzung",                                                                                                                                                                                                                                                                                             checklist => "Motor nicht ordnungsgem&auml;&szlig; montiert&bull;&Uuml;berpr&uuml;fen, dass Ventile sich in der richtigen Reihenfolge &ouml;ffnen/schlie&szlig;en&bull;Expansionsventilmotor defekt&bull;Expansionsventil VR1 &ouml;ffnet sich zu weit/ bleibt in der offenen Stellung stecken",                                                                                                                                                                                                                    correction => "Ventilmotor &uuml;berpr&uuml;fen&bull;Test Au&szlig;eneinheit aktivieren&bull;Widerstand zwischen Kabel Orange, Rot, Gelb und Schwarz zu Grau messen; 46 kOhm = OK, bei Unterbrechung oder Kurzschluss, Ventilmotor wechseln&bull;Bei wiederholten Warnungen, Expansionsventil wechseln"                                                                                                         };
	$ErrorCodeList{"0429"} = {errorcode => "A01",                                    subcode => "5374",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Frostschutz des Kondensators",                                                                                                                                                                                                                                                                                          checklist => "Schlechte/ausgebliebene Zirkulation im W&auml;rmetr&auml;gersystem/Heizsystem&bull;Schmutz im Systemfilter/Filterball SC1&bull;Luft im W&auml;rmetr&auml;gersystem/ Heizsystem&bull;F&uuml;hler TC3 defekt&bull;I/O-Board defekt, 230 V fehlt&bull;I/O-Board defekt, kein PWM-Signal&bull;I/O-Board defekt, keine 230 V an Klemme 01 (51, N)&bull;I/O-Board defekt&bull;Umw&auml;lzpumpe defekt",                                                                                                     correction => "Einstellventile/Heizk&ouml;rperthermostate &uuml;berpr&uuml;fen&bull;Systemfilter/Filterball SC1 reinigen&bull;Das Heizsystem laut Anweisungen in der Gebrauchsanweisung entl&uuml;ften, Wasser im Heizsystem auff&uuml;llen&bull;F&uuml;hler TC3 im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/F&uuml;hler TC3 wechseln&bull;&Uuml;berpr&uuml;fen, ob 230 V an Klemme 01 (51, N) am I/O-Board&bull;PWM-Signal an Klemme 36,37 am I/O-Board abschalten, Umw&auml;lzpumpe auf 100 % erh&ouml;hen&bull;&Uuml;berpr&uuml;fen, ob 230 V an Klemme 01 (51, N) am I/O-Board&bull;I/O-Board wechseln&bull;Umw&auml;lzpumpe wechseln"};
	$ErrorCodeList{"0430"} = {errorcode => "A01",                                    subcode => "5376",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Frostschutz des Kondensators",                                                                                                                                                                                                                                                                                          checklist => "Schlechte/ausgebliebene Zirkulation im W&auml;rmetr&auml;gersystem/Heizsystem&bull;Schmutz im Systemfilter/Filterball SC1&bull;Luft im W&auml;rmetr&auml;gersystem/ Heizsystem&bull;F&uuml;hler TC3 defekt&bull;I/O-Board defekt, 230 V fehlt&bull;I/O-Board defekt, kein PWM-Signal&bull;I/O-Board defekt, keine 230 V an Klemme 01 (51, N)&bull;I/O-Board defekt&bull;Umw&auml;lzpumpe defekt",                                                                                                     correction => "Einstellventile/Heizk&ouml;rperthermostate &uuml;berpr&uuml;fen&bull;Systemfilter/Filterball SC1 reinigen&bull;Das Heizsystem laut Anweisungen in der Gebrauchsanweisung entl&uuml;ften, Wasser im Heizzystem auff&uuml;llen&bull;F&uuml;hler TC3 im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/ F&uuml;hler TC3 wechseln&bull;&Uuml;berpr&uuml;fen, ob 230 V an Klemme 01 (51, N) am I/O-Board&bull;PWM-Signal an Klemme 36,37 am I/O-Board abschalten, Umw&auml;lzpumpe auf 100 % erh&ouml;hen&bull;&Uuml;berpr&uuml;fen, ob 230 V an Klemme 01 (51, N) am I/O-Board&bull;I/O-Board wechseln&bull;Umw&auml;lzpumpe wechseln"};
	$ErrorCodeList{"0431"} = {errorcode => "A01",                                    subcode => "5378",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 fehlgeschlagene; Abtauung",                                                                                                                                                                                                                                                                                             checklist => "Temperatur des Heizsystems zu niedrig&bull;F&uuml;hler TL2 defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Mehrere Thermostate des Heizzystems &ouml;ffnen&bull;F&uuml;hler TL2 im Vergleich zur aktuellen F&uuml;hlertabelle und tats&auml;chlichen Temperatur &uuml;berpr&uuml;fen. Bei Abweichungen F&uuml;hler TL2 wechseln."                                                                                                                                                                           };
	$ErrorCodeList{"0432"} = {errorcode => "A01",                                    subcode => "5380",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 fehlgeschlagene Abtauung",                                                                                                                                                                                                                                                                                              checklist => "Temperatur des Heizsystems zu niedrig&bull;F&uuml;hler TL2 defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Mehrere Thermostate des Heizzystems &ouml;ffnen&bull;F&uuml;hler TL2 im Vergleich zur aktuellen F&uuml;hlertabelle und tats&auml;chlichen Temperatur &uuml;berpr&uuml;fen. Bei Abweichungen F&uuml;hler TL2 wechseln."                                                                                                                                                                           };
	$ErrorCodeList{"0433"} = {errorcode => "A01",                                    subcode => "5394",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Interner Kompressortreiberfehler 1",                                                                                                                                                                                                                                                                                    checklist => "Unterbrechung/Kurzschluss im Kabelbaum zwischen Inverter und Kompressor&bull;Stromfehler im Kompressor&bull;Inverter au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                                                           correction => "Kabelbaum zwischen Inverter und Kompressor &uuml;berpr&uuml;fen&bull;Widerstand zwischen Wicklungen und Erdung messen; bei Widerstand; < 10 kOhm, Kompressor wechseln&bull;Inverter wechseln"                                                                                                                                                                                                    };
	$ErrorCodeList{"0434"} = {errorcode => "A01",                                    subcode => "5396",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Interner Kompressortreiberfehler 1",                                                                                                                                                                                                                                                                                    checklist => "Unterbrechung/Kurzschluss im Kabelbaum zwischen Inverter und Kompressor&bull;Stromfehler im Kompressor&bull;Inverter au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                                                           correction => "Kabelbaum zwischen Inverter und Kompressor &uuml;berpr&uuml;fen&bull;Widerstand zwischen Wicklungen und Erdung messen; bei Widerstand; < 10 kOhm, Kompressor wechseln&bull;Inverter wechseln"                                                                                                                                                                                                    };
	$ErrorCodeList{"0435"} = {errorcode => "A01",                                    subcode => "5408",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Lufteintrittstemperaturf&uuml;hler TL2 Unterbrechung",                                                                                                                                                                                                                                                                  checklist => "Lufteintrittsf&uuml;hler TL2 T0/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Lufteintrittsf&uuml;hler TL2 nicht angezogen&bull;Lufteintrittsf&uuml;hler TL2 T0/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                        correction => "Ohmwert von F&uuml;hler TL2/Signalkabel; im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Lufteintrittsf&uuml;hler/Signalkabel F&uuml;hler TL2 wechseln&bull;I/O-Board wechseln"                                                                         };
	$ErrorCodeList{"0436"} = {errorcode => "A01",                                    subcode => "5409",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Lufteintrittstemperaturf&uuml;hler TL2 kurzgeschlossen",                                                                                                                                                                                                                                                                checklist => "Au&szlig;entemperaturf&uuml;hler TL2/Signalkabel kurzgeschlossen&bull;Lufteintrittsf&uuml;hler TC3 kurzgeschlossen&bull;Signalkabel zu Lufteintrittsf&uuml;hler/F&uuml;hler TL2 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                 correction => "Ohmwert von F&uuml;hler TL2/Signalkabel; im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle und tats&auml;chlichen Temperatur &uuml;berpr&uuml;fen/Lufteintrittsf&uuml;hler TL2 wechseln&bull;Signalkabel/F&uuml;hler zu Kondensatorf&uuml;hler TL2 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0437"} = {errorcode => "A01",                                    subcode => "5411",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Lufteintrittstemperaturf&uuml;hler TL2 Unterbrechung",                                                                                                                                                                                                                                                                  checklist => "Lufteintrittsf&uuml;hler TL2 T0/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Lufteintrittsf&uuml;hler TL2 nicht angezogen&bull;Lufteintrittsf&uuml;hler TL2 T0/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                        correction => "Ohmwert von F&uuml;hler TL2/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Lufteintrittsf&uuml;hler/Signalkabel F&uuml;hler TL2 wechseln&bull;I/O-Board wechseln"                                                                          };
	$ErrorCodeList{"0438"} = {errorcode => "A01",                                    subcode => "5412",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Lufteintrittstemperaturf&uuml;hler TL2 kurzgeschlossen",                                                                                                                                                                                                                                                                checklist => "Au&szlig;entemperaturf&uuml;hler TL2/Signalkabel kurzgeschlossen&bull;Lufteintrittsf&uuml;hler TL2 kurzgeschlossen&bull;Signalkabel zu Lufteintrittsf&uuml;hler/F&uuml;hler TL2 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                 correction => "Ohmwert von F&uuml;hler TL2/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hler-tabelle und tats&auml;chlichen Temperatur &uuml;berpr&uuml;fen/Lufteintrittsf&uuml;hler TL2 wechseln&bull;Signalkabel/F&uuml;hler zu Kondensatorf&uuml;hler TL2 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0439"} = {errorcode => "A01",                                    subcode => "5414",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z1 Fl&uuml;ssigkeitsleitung in Heizbetrieb F&uuml;hler TR3 Unterbrechung",                                                                                                                                                                                                                                                checklist => "F&uuml;hler TR3/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu F&uuml;hler TR3 nicht angezogen&bull;F&uuml;hler TR3/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                     correction => "Ohmwert von F&uuml;hler TR3/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von; I/O-Board-Klemme getrennt&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;F&uuml;hler TR3 wechseln&bull;I/O-Board wechseln"                                                                                                                                };
	$ErrorCodeList{"0440"} = {errorcode => "A01",                                    subcode => "5415",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z1 Fl&uuml;ssigkeitsleitung in Heizbetrieb F&uuml;hler TR3 kurzgeschlossen",                                                                                                                                                                                                                                              checklist => "W&auml;rmetr&auml;ger in F&uuml;hler TR3/Signalkabel kurzgeschlossen&bull;F&uuml;hler TR3 kurzgeschlossen&bull;Signalkabel/F&uuml;hler TR3 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                      correction => "Ohmwert von F&uuml;hler TR3/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von; I/O-Board-Klemme getrennt&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hler-tabelle &uuml;berpr&uuml;fen/F&uuml;hler TR3 wechseln&bull;F&uuml;hler TR3 wechseln&bull;I/O-Board wechseln"                                                              };
	$ErrorCodeList{"0441"} = {errorcode => "A01",                                    subcode => "5417",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z2 Fl&uuml;ssigkeitsleitung in Heizbetrieb F&uuml;hler TR3 Unterbrechung",                                                                                                                                                                                                                                                checklist => "F&uuml;hler TR3/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu F&uuml;hler TR3 nicht angezogen&bull;F&uuml;hler TR3/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion&bull;-",                                                                                                                                                                                                                                                                                              correction => "Ohmwert von F&uuml;hler TR3/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von; I/O-Board-Klemme getrennt&bull;Schraubklemme an I/O-Board &uuml;berpr&uuml;fen&bull;F&uuml;hler TR3 wechseln&bull;I/O-Board wechseln&bull;-"                                                                                                            };
	$ErrorCodeList{"0442"} = {errorcode => "A01",                                    subcode => "5418",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z2 Fl&uuml;ssigkeitsleitung in Heizbetrieb F&uuml;hler TR3 kurzgeschlossen",                                                                                                                                                                                                                                              checklist => "W&auml;rmetr&auml;ger in F&uuml;hler TR3/Signalkabel kurzgeschlossen&bull;F&uuml;hler TR3 kurzgeschlossen&bull;Signalkabel/F&uuml;hler TR3 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                      correction => "Ohmwert von F&uuml;hler TR3/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von; I/O-Board-Klemme getrennt&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/F&uuml;hler TR3 wechseln&bull;F&uuml;hler TR3 wechseln&bull;I/O-Board wechseln"                                                               };
	$ErrorCodeList{"0443"} = {errorcode => "A01",                                    subcode => "5420",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z1 Fl&uuml;ssigkeitsleitung im K&uuml;hlbetrieb F&uuml;hler TR4 Unterbrechung",                                                                                                                                                                                                                                           checklist => "F&uuml;hler TR4/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu F&uuml;hler TR4 nicht angezogen&bull;Lufteintrittsf&uuml;hler TR4/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                        correction => "Ohmwert von F&uuml;hler TR4/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Schraubklemme an I/O-Board &uuml;berpr&uuml;fen&bull;Byt F&uuml;hler/Signalkabel F&uuml;hler TR4&bull;I/O-Board wechseln"                                                                               };
	$ErrorCodeList{"0444"} = {errorcode => "A01",                                    subcode => "5421",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z1 Fl&uuml;ssigkeitsleitung im K&uuml;hlbetrieb F&uuml;hler TR4 kurzgeschlossen",                                                                                                                                                                                                                                         checklist => "F&uuml;hler TR4/Signalkabel kurzgeschlossen&bull;F&uuml;hler TR4 kurzgeschlossen&bull;Signalkabel/F&uuml;hler TR4 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                               correction => "Ohmwert von F&uuml;hler TR4/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle und tats&auml;chlichen Temperatur &uuml;ber-pr&uuml;fen/F&uuml;hler TR4 wechseln&bull;Signalkabel/F&uuml;hler zu F&uuml;hler TR4 wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0445"} = {errorcode => "A01",                                    subcode => "5423",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z2 Fl&uuml;ssigkeitsleitung im K&uuml;hlbetrieb F&uuml;hler TR4 Unterbrechung",                                                                                                                                                                                                                                           checklist => "F&uuml;hler TR4/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu F&uuml;hler TR4 nicht angezogen&bull;Lufteintrittsf&uuml;hler TR4/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                        correction => "Ohmwert von F&uuml;hler TR4/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Schraubklemme an I/O-Board &uuml;berpr&uuml;fen&bull;Byt F&uuml;hler/Signalkabel F&uuml;hler TR4&bull;I/O-Board wechseln"                                                                               };
	$ErrorCodeList{"0446"} = {errorcode => "A01",                                    subcode => "5424",                                               class => "WPL AR",      reason => "Warnung; W&auml;rmepumpe Z2 Fl&uuml;ssigkeitsleitung im K&uuml;hlbetrieb F&uuml;hler TR4 kurzgeschlossen",                                                                                                                                                                                                                                         checklist => "F&uuml;hler TR4/Signalkabel kurzgeschlossen&bull;F&uuml;hler TR4 kurzgeschlossen&bull;Signalkabel/F&uuml;hler TR4 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                               correction => "Ohmwert von F&uuml;hler TR4/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hler-tabelle und tats&auml;chlichen Temperatur &uuml;ber-pr&uuml;fen/F&uuml;hler TR4 wechseln&bull;Signalkabel/F&uuml;hler zu F&uuml;hler TR4 wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0447"} = {errorcode => "A01",                                    subcode => "5426",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Heizgasf&uuml;hler TR5 Unterbrechung",                                                                                                                                                                                                                                                                                  checklist => "F&uuml;hler TR5/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu F&uuml;hler TR5 nicht angezogen&bull;F&uuml;hler TR5/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                     correction => "Ohmwert von F&uuml;hler TR5/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen, getrennt von I/O-BoardKlemme.&bull;Schraubklemme an I/O-Board &uuml;berpr&uuml;fen&bull;F&uuml;hler TR5 wechseln&bull;I/O-Board wechseln"                                                                                                                                            };
	$ErrorCodeList{"0448"} = {errorcode => "A01",                                    subcode => "5427",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Sauggasf&uuml;hler TR5 kurzgeschlossen",                                                                                                                                                                                                                                                                                checklist => "F&uuml;hler TR5/Signalkabel kurzgeschlossen&bull;F&uuml;hler TR5 kurzgeschlossen&bull;Signalkabel/F&uuml;hler TR5 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                               correction => "Ohmwert von F&uuml;hler TR5/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen, getrennt von I/O-Board-Klemme&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hler-tabelle &uuml;berpr&uuml;fen/F&uuml;hler TR5 wechseln&bull;F&uuml;hler TR5 wechseln&bull;I/O-Board wechseln"                                                                                       };
	$ErrorCodeList{"0449"} = {errorcode => "A01",                                    subcode => "5429",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Heizgasf&uuml;hler TR5 Unterbrechung",                                                                                                                                                                                                                                                                                  checklist => "F&uuml;hler TR5/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu F&uuml;hler TR5 nicht angezogen&bull;F&uuml;hler TR5/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                     correction => "Ohmwert von F&uuml;hler TR5/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen, getrennt von I/O-BoardKlemme&bull;Schraubklemme an I/O-Board &uuml;berpr&uuml;fen&bull;F&uuml;hler TR5 wechseln&bull;I/O-Board wechseln"                                                                                                                                             };
	$ErrorCodeList{"0450"} = {errorcode => "A01",                                    subcode => "5430",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Sauggasf&uuml;hler TR5 kurzgeschlossen",                                                                                                                                                                                                                                                                                checklist => "F&uuml;hler TR5/Signalkabel kurzgeschlossen&bull;F&uuml;hler TR5 kurzgeschlossen&bull;Signalkabel/F&uuml;hler TR5 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                               correction => "Ohmwert von F&uuml;hler TR5/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen, getrennt von I/O-Board-Klemme&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hler-tabelle &uuml;berpr&uuml;fen/F&uuml;hler TR5 wechseln&bull;F&uuml;hler TR5 wechseln&bull;I/O-Board wechseln"                                                                                       };
	$ErrorCodeList{"0451"} = {errorcode => "A01",                                    subcode => "5432",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Druckgeber; niedriger Druck JR0 Unterbrechung",                                                                                                                                                                                                                                                                         checklist => "Spiel in Kabelbaum/Klemme 17, 19 an IO-Board&bull;I/O-Board defekt&bull;Druckgeber JR0 defekt",                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Kabelbaum und Klemmenanschl&uuml;sse 17, 19 an IO-Board &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, dass Klemme 19 mit ausgeschaltetem Druckgeber 5 V DC an Klemme 17 abgibt, wenn Spannung; < 0,5 V, IO-Board wechseln&bull;&Uuml;berpr&uuml;fen, dass Klemme 19 mit ausgeschaltetem Druckgeber 5 V DC an Klemme 17 abgibt, wenn Spannung; = 5 V, IO-Board wechseln"                        };
	$ErrorCodeList{"0452"} = {errorcode => "A01",                                    subcode => "5435",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Druckgeber niedriger Druck JR0 Unterbrechung",                                                                                                                                                                                                                                                                          checklist => "Spiel in Kabelbaum/Klemme 17, 19 an IO-Board&bull;I/O-Board defekt&bull;Druckgeber JR0 defekt",                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Kabelbaum und Klemmenanschl&uuml;sse 17, 19 an IO-Board &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, dass Klemme 19 mit ausgeschaltetem Druckgeber 5V DC an Klemme 17 abgibt, wenn Spannung; < 0,5 V, IO-Board wechseln&bull;&Uuml;berpr&uuml;fen, dass Klemme 19 mit ausgeschaltetem Druckgeber 5 V DC an Klemme 17 abgibt, wenn Spannung; = 5 V, IO-Board wechseln"                         };
	$ErrorCodeList{"0453"} = {errorcode => "A01",                                    subcode => "5438",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Druckgeber hoher Druck JR1 Unterbrechung",                                                                                                                                                                                                                                                                              checklist => "Spiel in Kabelbaum/Klemme 16, 18 an IO-Board&bull;I/O-Board defekt&bull;Druckgeber JR1 defekt",                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Kabelbaum und Klemmenanschl&uuml;sse 16, 18 an IO-Board &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, dass Klemme 19 mit ausgeschaltetem Druckgeber 5V DC an Klemme 17 abgibt, wenn Spannung; < 0,5 V, IO-Board wechseln&bull;&Uuml;berpr&uuml;fen, dass Klemme 19 mit ausgeschaltetem Druckgeber 5V DC an Klemme 17 abgibt, wenn Spannung; = 5 V, IO-Board wechseln"                          };
	$ErrorCodeList{"0454"} = {errorcode => "A01",                                    subcode => "5441",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2; Druckgeber hoher Druck JR1 Unterbrechung",                                                                                                                                                                                                                                                                             checklist => "Spiel in Kabelbaum/Klemme 16, 18 an IO-Board&bull;I/O-Board defekt&bull;Druckgeber JR1 defekt",                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Kabelbaum und Klemmenanschl&uuml;sse 16, 18 an IO-Board &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, dass Klemme 19 mit ausgeschaltetem Druckgeber 5V DC an Klemme 17 abgibt, wenn Spannung; < 0,5 V, IO-Board wechseln&bull;&Uuml;berpr&uuml;fen, dass Klemme 19 mit ausgeschaltetem Druckgeber 5V DC an Klemme 17 abgibt, wenn Spannung = 5 V,; IO-Board wechseln"                          };
	$ErrorCodeList{"0455"} = {errorcode => "A01",                                    subcode => "5445",                                               class => "WPL AR",      reason => "Alarm W&auml;rmepumpe Z1 Sammelalarm Gebl&auml;se",                                                                                                                                                                                                                                                                                                checklist => "Gebl&auml;se blockiert&bull;Gebl&auml;se gem&auml;&szlig; Test Au&szlig;eneinheit &uuml;berpr&uuml;fen&bull;I/O-Board defekt&bull;I/O-Board defekt&bull;Gebl&auml;se hat nicht 230 V Spannung&bull;Gebl&auml;se ohne 0: 10 V-Signal&bull;Gebl&auml;se defekt",                                                                                                                                                                                                                                        correction => "&Uuml;berpr&uuml;fen, dass Gebl&auml;se nicht physisch blockiert wird&bull;Test Au&szlig;eneinheit aktivieren&bull;&Uuml;berpr&uuml;fen, dass 230 V Spannung an Ausgang PL3 SSM Klemme 34 (75, N)&bull;Fehlt Spannung wie oben, I/O-Board wechseln&bull;&Uuml;berpr&uuml;fen, dass 230 V Spannung an Ausgang PL3 Klemme 32 (78, N)&bull;0: 10 V Spannung an Ausgang PL3 PWM Klemme 20 (20, 26) laut Test Au&szlig;eneinheit &uuml;berpr&uuml;fen&bull;Ist Spannung wie oben vorhanden, Gebl&auml;se wechseln"};
	$ErrorCodeList{"0456"} = {errorcode => "A01",                                    subcode => "5450",                                               class => "WPL AR",      reason => "Kondenswasserablauf in W&auml;rmepumpe Z1 blockiert",                                                                                                                                                                                                                                                                                              checklist => "Kondenswasserablauf in W&auml;rmepumpe ist blockiert&bull;Heizkabel im Kondenswasserablauf defekt, Ablauf zugefroren&bull;Heizkabel hat keine Spannung",                                                                                                                                                                                                                                                                                                                                              correction => "Ablauf &uuml;berpr&uuml;fen/reinigen&bull;Heizkabel &uuml;berpr&uuml;fen/Heizkabel wechseln&bull;&Uuml;berpr&uuml;fen, ob Ausgang EA1-Klemme 26 (79, N) bei manuellem Betrieb aus Heizkabel 230 V hat"                                                                                                                                                                                           };
	$ErrorCodeList{"0457"} = {errorcode => "A01",                                    subcode => "5452",                                               class => "WPL AR",      reason => "Interner Fehler am Inverter zum Kompressor",                                                                                                                                                                                                                                                                                                       checklist => "Inverter defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     correction => "Inverter wechseln"                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0458"} = {errorcode => "A01",                                    subcode => "5454",                                               class => "WPL AR",      reason => "Kondenswasserablauf in W&auml;rmepumpe Z2 blockiert",                                                                                                                                                                                                                                                                                              checklist => "Kondenswasserablauf in W&auml;rmepumpe ist blockiert&bull;Heizkabel im Kondenswasserablauf defekt, Ablauf zugefroren&bull;Heizkabel hat keine Spannung",                                                                                                                                                                                                                                                                                                                                              correction => "Tropfschale/Ablauf &uuml;berpr&uuml;fen/reinigen&bull;Heizkabel &uuml;berpr&uuml;fen/Heizkabel wechseln&bull;&Uuml;berpr&uuml;fen, ob Ausgang EA1-Klemme 26 (79, N) bei manuellem Betrieb aus Heizkabel 230 V hat"                                                                                                                                                                               };
	$ErrorCodeList{"0459"} = {errorcode => "A01",                                    subcode => "5456",                                               class => "WPL AR",      reason => "Interner Fehler am Inverter zum Kompressor in W&auml;rmepumpe Z2",                                                                                                                                                                                                                                                                                 checklist => "Inverter defekt",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     correction => "Inverter wechseln"                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0460"} = {errorcode => "A01",                                    subcode => "5458",                                               class => "WPL AR",      reason => "Warnung Z2 Warmwasserf&uuml;hler TW1 Unterbrechung",                                                                                                                                                                                                                                                                                               checklist => "Warmwasserf&uuml;hler TW1/Signalkabel Unterbrechung&bull;Schraubklemme an I/O-Board zu Warmwasserf&uuml;hler TW1 nicht angezogen&bull;Warmwasserf&uuml;hler TW1/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                       correction => "Ohmwert von F&uuml;hler TW1/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle; &uuml;berpr&uuml;fen; das Innenteil ist dabei von I/OBoard-Klemme getrennt&bull;Schraubklemme &uuml;berpr&uuml;fen&bull;Vorlauff&uuml;hler TW1 wechseln&bull;I/O-Board wechseln"                                                                                                                          };
	$ErrorCodeList{"0461"} = {errorcode => "A01",                                    subcode => "5459",                                               class => "WPL AR",      reason => "Warnung Z2 Warmwasserf&uuml;hler TW1 kurzgeschlossen",                                                                                                                                                                                                                                                                                             checklist => "Warmwasserf&uuml;hler TW1/Signalkabel kurzgeschlossen&bull;W&auml;rmetr&auml;ger in F&uuml;hler TW1 kurzgeschlossen&bull;Signalkabel zu W&auml;rmetr&auml;ger in F&uuml;hler TW1 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                correction => "Ohmwert von F&uuml;hler TW1/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen; das Innenteil ist dabei von; I/O-Board-Klemme getrennt&bull;Ohmwert im Vergleich zur aktuellen F&uuml;hlertabelle &uuml;berpr&uuml;fen/W&auml;rmetr&auml;ger in F&uuml;hler TW1 wechseln&bull;Signalkabel zu W&auml;rmetr&auml;ger aus F&uuml;hler TW1 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0462"} = {errorcode => "A01",                                    subcode => "5465",                                               class => "WPL AR",      reason => "Kein Spannung zu W&auml;rmepumpe Z2",                                                                                                                                                                                                                                                                                                              checklist => "Niedrige/Keine Spannung zur W&auml;rmepumpe&bull;Niedrige eingehende Netzspannung",                                                                                                                                                                                                                                                                                                                                                                                                                   correction => "Hauptsicherungen in Bezug auf schlechten Kontakt/Defekt &uuml;berpr&uuml;fen&bull;Bei wiederholten Warnungen Kontakt zum Stromanbieter aufnehmen"                                                                                                                                                                                                                                                };
	$ErrorCodeList{"0463"} = {errorcode => "A01",                                    subcode => "5500",                                               class => "WPL AR",      reason => "Fu&szlig;hodenheizungsthermostat am externen Eingang ausgel&ouml;st",                                                                                                                                                                                                                                                                              checklist => "Sicherheitsthermostat f&uuml;r Fu&szlig;bodenheizung wurde ausgel&ouml;st",                                                                                                                                                                                                                                                                                                                                                                                                                           correction => "Thermostat zur&uuml;ckstellen, eventuelle Heizkurve einstellen"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0464"} = {errorcode => "A01",                                    subcode => "5503",                                               class => "WPL AR",      reason => "Warnung Problem mit Anschluss zur Belastungsw&auml;chter",                                                                                                                                                                                                                                                                                         checklist => "Kabelbaum/Anschl&uuml;sse fehlerhaft&bull;Loser Kontakt in CANbus-Anschl&uuml;ssen an Installer-Board Tower und Belastungsw&auml;chter&bull;Unterbrechung an CANbus-Kabel zwischen Installer-Board Tower und Belastungsw&auml;chter&bull;Falscher Typ CANbus-Kabel&bull;CANbus-Kabel zusammen mit Stromversorgung der W&auml;rmepumpe verlegt",                                                                                                                                                       correction => "Kabelbaum/Anschl&uuml;sse &uuml;berpr&uuml;fen&bull;CANbus-Anschl&uuml;sse an Installer-Board Tower und Belastungsw&auml;chter &uuml;berpr&uuml;fen&bull;CANbus-Kabel zwischen Installer-Board Tower und Belastungsw&auml;chter wechseln&bull;Zum richtigen Typ von Kabel wechseln&bull;CANbusund Stromkabel mit einem Abstand von mindestens 100 mm verlegen"                                   };
	$ErrorCodeList{"0465"} = {errorcode => "A01",                                    subcode => "5504",                                               class => "WPL AR",      reason => "Alarm Unterbrechung Anschluss zum Belastungsw&auml;chter",                                                                                                                                                                                                                                                                                         checklist => "Siehe Pr&uuml;fvorgang/Ursachen f&uuml;r Ursachencode 5432",                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Siehe m&ouml;gliche Ma&szlig;nahmen f&uuml;r Ursachencode 5432"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0466"} = {errorcode => "A01",                                    subcode => "5506",                                               class => "WPL AR",      reason => "Kompressor startet nicht",                                                                                                                                                                                                                                                                                                                         checklist => "Vor&uuml;bergehende Funktionst&ouml;rung Inverter&bull;Interner Fehler im Inverter",                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Spannung zur W&auml;rmepumpe ausund wieder einschalten&bull;Inverter wechseln"                                                                                                                                                                                                                                                                                                                   };
	$ErrorCodeList{"0467"} = {errorcode => "A01",                                    subcode => "5507",                                               class => "WPL AR",      reason => "MR1 hoher Druck Alarm Alarm B in W&auml;rmepumpe Z1",                                                                                                                                                                                                                                                                                              checklist => "Pressostat au&szlig;er Funktion&bull;Unterbrechung/schlechte Verbindung im Kabelbaum zwischen MR1 und Inverter&bull;Unterbrechung/schlechte Verbindung im Kabelbaum zwischen MR1 und Inverter",                                                                                                                                                                                                                                                                                                       correction => "Aufgebrochener Kreis < 43 bar = Pressostat wechseln&bull;Kabelbaum/Anschl&uuml;sse zwischen MR1 und Inverter &uuml;berpr&uuml;fen&bull;&Uuml;berpr&uuml;fen, ob Hochdruckw&auml;chter angeschlossen ist"                                                                                                                                                                                         };
	$ErrorCodeList{"0468"} = {errorcode => "A01",                                    subcode => "5509",                                               class => "WPL AR",      reason => "MR1 hoher Druck Alarm Alarm B in W&auml;rmepumpe Z2",                                                                                                                                                                                                                                                                                              checklist => "Unterbrechung in Kabel/Pressostat MR1&bull;Pressostat au&szlig;er Funktion&bull;Unterbrechung/schlechte Verbindung im Kabelbaum zwischen MR1 und Inverter&bull;Unterbrechung/schlechte Verbindung im Kabelbaum zwischen MR1 und Inverter",                                                                                                                                                                                                                                                            correction => "An ersten Molexanschluss Ohmwert Pressostat messen/summieren, geschlossen = OK&bull;Aufgebrochener Kreis < 43bar = Pressostat wechseln&bull;Kabelbaum/Anschl&uuml;sse zwischen MR1 und Inverter &uuml;berpr&uuml;fen&bull;MR1 im Inverter verbinden, Warnung sollte verschwinden, wenn nicht, Inverter wechseln"                                                                                 };
	$ErrorCodeList{"0469"} = {errorcode => "A01",                                    subcode => "5512",                                               class => "WPL AR",      reason => "Kondensationstemperatur au&szlig;erhalb der Grenze in W&auml;rmepumpe Z1",                                                                                                                                                                                                                                                                         checklist => "Zu niedrige Verdampftemperatur im Verh&auml;ltnis zur Kondensationstemperatur",                                                                                                                                                                                                                                                                                                                                                                                                                       correction => "Warscheinlich zu niedrige Au&szlig;entemperatur. Siehe Diagramm Kapitel 4 in Installationshandbuch"                                                                                                                                                                                                                                                                                              };
	$ErrorCodeList{"0470"} = {errorcode => "A01",                                    subcode => "5514",                                               class => "WPL AR",      reason => "Warnung niedriger Druck JR0 in W&auml;rmepumpe Z1",                                                                                                                                                                                                                                                                                                checklist => "Blockierter/niedriger Luftstrom zu Luftw&auml;rmetauscher an W&auml;rmepumpe&bull;Gebl&auml;se au&szlig;er Funktion&bull;Gebl&auml;se hat nicht 230 V Spannung&bull;Gebl&auml;se ohne 0: 10 V-Signal&bull;Gebl&auml;se defekt&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                   correction => "Ausreichenden Luftstrom &uuml;ber Luftw&auml;rmetauscher sicherstellen&bull;Test Au&szlig;eneinheit aktivieren&bull;&Uuml;berpr&uuml;fen, dass 230 V Spannung an Ausgang PL3 Klemme 32 (78, N)&bull;0: 10 V Spannung an Ausgang PL3 PWM Klemme 20 (20, 26) laut Test Au&szlig;eneinheit &uuml;berpr&uuml;fen&bull;Ist Spannung wie oben vorhanden, Gebl&auml;se wechseln&bull;Fehlt Spannung wie oben, I/O-Board wechseln"};
	$ErrorCodeList{"0471"} = {errorcode => "A01",                                    subcode => "5517",                                               class => "WPL AR",      reason => "Falscher Kompressortreiber in W&auml;rmepumpe Z2 installiert",                                                                                                                                                                                                                                                                                     checklist => "Falsche Kombination aus Inverter/I/O-Board in W&auml;rmepumpe 2 installiert&bull;Bei Wechsel des Inverters, lasches Invertermodell&bull;Beim Wechsel des I/O-Boards, falsche Einstellung des Dreh-Encoders auf neuem I/OBoard",                                                                                                                                                                                                                                                                       correction => "Einstellung im Vergleich zum aktuellen elektrischen Schaltplan &uuml;berpr&uuml;fen&bull;Invertermodell &uuml;berpr&uuml;fen&bull;Einstellung des Dreh-Encoders im Vergleich zum ausgetauschten I/O-Board &uuml;berpr&uuml;fen"                                                                                                                                                                  };
	$ErrorCodeList{"0472"} = {errorcode => "A01",                                    subcode => "5519",                                               class => "WPL AR",      reason => "Zu hohe Kondensation in W&auml;rmepumpe Z2",                                                                                                                                                                                                                                                                                                       checklist => "Zu niedrige Verdampftemperatur im Verh&auml;ltnis zur Kondensationstemperatur&bull;Zu hohe Stromrate/niedriges Delta im Heizzystem",                                                                                                                                                                                                                                                                                                                                                                  correction => "Warscheinlich zu niedrige Au&szlig;entemperatur. Siehe Diagramm Kapitel 4 in Installationshandbuch&bull;Temperatur/Stromrate einstellen"                                                                                                                                                                                                                                                         };
	$ErrorCodeList{"0473"} = {errorcode => "A01",                                    subcode => "5521",                                               class => "WPL AR",      reason => "Alarm geringe Verdampfung JR0 in W&auml;rmepumpe Z2",                                                                                                                                                                                                                                                                                              checklist => "Siehe Pr&uuml;fvorgang/Ursachen f&uuml;r Ursachencode 5520",                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Siehe m&ouml;gliche Ma&szlig;nahmen f&uuml;r Ursachencode 5520"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0474"} = {errorcode => "A01",                                    subcode => "5522",                                               class => "WPL AR",      reason => "Falsche Kombination aus W&auml;rmepumpe und Innenteil",                                                                                                                                                                                                                                                                                            checklist => "Falsche Kombination aus W&auml;rmepumpe/ Innenteil&bull;Beim Wechsel des I/O-Boards,; falsche Einstellung des Dreh-Encoders auf neuem I/O-Board&bull;Beim Wechsel des Installer-Boards, falsche Einstellung des Dreh-Encoders auf neuem Installer-Board",                                                                                                                                                                                                                                             correction => "Kombination &uuml;berpr&uuml;fen&bull;Einstellung des Dreh-Encoders im Vergleich zum ausgetauschten I/O-Board &uuml;berpr&uuml;fen&bull;Einstellung des Dreh-Encoders im Vergleich zum ausgetauschten Installer-Board &uuml;berpr&uuml;fen"                                                                                                                                                      };
	$ErrorCodeList{"0475"} = {errorcode => "A01",                                    subcode => "5524",                                               class => "WPL AR",      reason => "Alarm W&auml;rmepumpe Z1 PFC &Uuml;berstrom des Kompressortreibers",                                                                                                                                                                                                                                                                               checklist => "Siehe Pr&uuml;fvorgang/Ursachen f&uuml;r Ursachencode 5523",                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "Siehe m&ouml;gliche Ma&szlig;nahmen f&uuml;r Ursachencode 5523"                                                                                                                                                                                                                                                                                                                                  };
	$ErrorCodeList{"0476"} = {errorcode => "A01",                                    subcode => "5526",                                               class => "WPL AR",      reason => "Alarm W&auml;rmepumpe Z2 defr Au&szlig;enteil der; W&auml;rmepumpe reinigen",                                                                                                                                                                                                                                                                      checklist => "Verdampfer vereist",                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  correction => "Eis vorsichtig mit Warmwasser schmelzen"                                                                                                                                                                                                                                                                                                                                                         };
	$ErrorCodeList{"0477"} = {errorcode => "A01",                                    subcode => "5528",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Kompressorf&uuml;hler TR1 Unterbrechung",                                                                                                                                                                                                                                                                               checklist => "Schraubklemme an I/O-Board zu Kompressorf&uuml;hler TR1 nicht angezogen&bull;Kompressorf&uuml;hler TR3/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion&bull;Kompressorf&uuml;hler TR1/Signalkabel kurzgeschlossen",                                                                                                                                                                                                                                                                     correction => "Schrauklemme &uuml;berpr&uuml;fen&bull;Kompressorf&uuml;hler TR1 wechseln&bull;I/O-Board wechseln&bull;Ohmwert von F&uuml;hler TR1/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen."                                                                                                     };
	$ErrorCodeList{"0478"} = {errorcode => "A01",                                    subcode => "5530",                                               class => "WPL AR",      reason => "Alarm W&auml;rmepumpe Z2 Kompressorf&uuml;hler TR1 defekt",                                                                                                                                                                                                                                                                                        checklist => "Kompressorf&uuml;hler TR1/Signalkabel Unterbrechung",                                                                                                                                                                                                                                                                                                                                                                                                                                                 correction => "Ohmwert von F&uuml;hler TR1/Signalkabel; im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen"                                                                                                                                                                                                            };
	$ErrorCodeList{"0479"} = {errorcode => "A01",                                    subcode => "5531",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Kompressorf&uuml;hler TR1 Unterbrechung",                                                                                                                                                                                                                                                                               checklist => "Schraubklemme an I/O-Board zu Kompressorf&uuml;hler TR1 nicht angezogen&bull;Kompressorf&uuml;hler TR3/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                                correction => "Schrauklemme &uuml;berpr&uuml;fen&bull;Kompressorf&uuml;hler TR1 wechseln&bull;I/O-Board wechseln"                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0480"} = {errorcode => "A01",                                    subcode => "5532",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Kompressorf&uuml;hler TR1 Kurzschluss",                                                                                                                                                                                                                                                                                 checklist => "Kompressorf&uuml;hler TR1/Signalkabel kurzgeschlossen&bull;Kompressorf&uuml;hler TR1 kurzgeschlossen&bull;Signalkabel zu Kompressorf&uuml;hler TR1 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                              correction => "Ohmwert von F&uuml;hler TR1/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen; F&uuml;hlertabelle und tats&auml;chlichen Temperatur &uuml;berpr&uuml;fen/F&uuml;hler TR1 wechseln&bull;Signalkabel zu Kompressorf&uuml;hler TR1 reparieren/wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0481"} = {errorcode => "A01",                                    subcode => "5534",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Temperaturf&uuml;hler TA4 Unterbrechung",                                                                                                                                                                                                                                                                               checklist => "Schrauklemme an I/O-Board zu F&uuml;hler TA4 nicht angezogen&bull;F&uuml;hler TA4/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                                                     correction => "Schrauklemme an I/O-Board &uuml;berpr&uuml;fen&bull;Byt F&uuml;hler/Signalkabel F&uuml;hler TA4&bull;I/O-Board wechseln"                                                                                                                                                                                                                                                                         };
	$ErrorCodeList{"0482"} = {errorcode => "A01",                                    subcode => "5535",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z2 Temperaturf&uuml;hler TA4 kurzgeschlossen",                                                                                                                                                                                                                                                                             checklist => "F&uuml;hler TA4/Signalkabel kurzgeschlossen&bull;F&uuml;hler TA4 kurzgeschlossen&bull;Signalkabel/F&uuml;hler TA4 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                               correction => "Ohmwert von F&uuml;hler TA4/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen; F&uuml;hler-tabelle und tats&auml;chlichen Temperatur &uuml;ber-pr&uuml;fen/F&uuml;hler TA4 wechseln&bull;Signalkabel/F&uuml;hler zu F&uuml;hler TA4 wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0483"} = {errorcode => "A01",                                    subcode => "5538",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Temperaturf&uuml;hler TA4 Unterbrechung",                                                                                                                                                                                                                                                                               checklist => "Schrauklemme an I/O-Board zu F&uuml;hler TA4 nicht angezogen&bull;F&uuml;hler TA4/Signalkabel Unterbrechung&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                                                     correction => "Schrauklemme an I/O-Board &uuml;berpr&uuml;fen&bull;Byt F&uuml;hler/Signalkabel F&uuml;hler TA4&bull;I/O-Board wechseln"                                                                                                                                                                                                                                                                         };
	$ErrorCodeList{"0484"} = {errorcode => "A01",                                    subcode => "5539",                                               class => "WPL AR",      reason => "Warnung W&auml;rmepumpe Z1 Temperaturf&uuml;hler TA4 kurzgeschlossen",                                                                                                                                                                                                                                                                             checklist => "F&uuml;hler TA4/Signalkabel kurzgeschlossen&bull;F&uuml;hler TA4 kurzgeschlossen&bull;Signalkabel/F&uuml;hler TA4 kurzgeschlossen&bull;I/O-Board au&szlig;er Funktion",                                                                                                                                                                                                                                                                                                                               correction => "Ohmwert von F&uuml;hler TA4/Signalkabel im Vergleich zur aktuellen F&uuml;hlertabelle und der tats&auml;chlichen Temperatur, getrennt von der I/OBoard-Klemme, &uuml;berpr&uuml;fen&bull;Ohmwert im Vergleich zur aktuellen; F&uuml;hlertabelle und tats&auml;chlichen Temperatur &uuml;berpr&uuml;fen/F&uuml;hler TA4 wechseln&bull;Signalkabel/F&uuml;hler zu F&uuml;hler TA4 wechseln&bull;I/O-Board wechseln"};
	$ErrorCodeList{"0485"} = {errorcode => "A01",                                    subcode => "5542",                                               class => "8",           reason => "Warnung Z2 Versorgungsspannung zu niedrig",                                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0486"} = {errorcode => "A01",                                    subcode => "5543",                                               class => "4",           reason => "Warnung Z1 &Uuml;berhitzungstemperatur ist &uuml;ber dem Optimum",                                                                                                                                                                                                                                                                                 checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0487"} = {errorcode => "A01",                                    subcode => "5544",                                               class => "4",           reason => "Warnung Z2 &Uuml;berhitzungstemperatur ist &uuml;ber dem Optimum",                                                                                                                                                                                                                                                                                 checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0488"} = {errorcode => "A01",                                    subcode => "5547",                                               class => "4",           reason => "Warnung Z1 Sole-Eintrittstemperatur an TB0 ist zu niedrig",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0489"} = {errorcode => "A01",                                    subcode => "5548",                                               class => "4",           reason => "Warnung Z2 Sole-Eintrittstemperatur an TB0 ist zu niedrig",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0490"} = {errorcode => "A01",                                    subcode => "5551",                                               class => "4",           reason => "Warnung Z1 Sole-Austrittstemperatur an TB1 ist zu niedrig",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0491"} = {errorcode => "A01",                                    subcode => "5552",                                               class => "4",           reason => "Warnung Z2 Sole-Austrittstemperatur an TB1 ist zu niedrig",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0492"} = {errorcode => "A01",                                    subcode => "5555",                                               class => "4",           reason => "Warnung Z1 Sole-Eintrittstemperatur an TB0 ist zu hoch",                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0493"} = {errorcode => "A01",                                    subcode => "5556",                                               class => "4",           reason => "Warnung Z2 Sole-Eintrittstemperatur an TB0 ist zu hoch",                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0494"} = {errorcode => "A01",                                    subcode => "5559",                                               class => "4",           reason => "Warnung Z1 Sole-Aus-trittstemperatur an TB1; ist zu hoch",                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0495"} = {errorcode => "A01",                                    subcode => "5560",                                               class => "4",           reason => "Warnung Z2 Sole-Aus-trittstemperatur an TB1; ist zu hoch",                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0496"} = {errorcode => "A01",                                    subcode => "5563",                                               class => "8",           reason => "Warnung Z1 Hohe Temperaturdiff. zwischen TB0 und TB1",                                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0497"} = {errorcode => "A01",                                    subcode => "5564",                                               class => "8",           reason => "Warnung Z2 Hohe Temperaturdiff. zwischen TB0 und TB1",                                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0498"} = {errorcode => "A01",                                    subcode => "5567",                                               class => "8",           reason => "Warnung Z1 St&ouml;rung Solepumpe",                                                                                                                                                                                                                                                                                                                checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0499"} = {errorcode => "A01",                                    subcode => "5568",                                               class => "8",           reason => "Warnung Z2 St&ouml;rung Solepumpe",                                                                                                                                                                                                                                                                                                                checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0500"} = {errorcode => "A01",                                    subcode => "5571",                                               class => "4",           reason => "Warnung Z1 Sole-Eintrittstemperatursensor TB0 Kurzschluss",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0501"} = {errorcode => "A01",                                    subcode => "5572",                                               class => "4",           reason => "Warnung Z2 Sole-Eintrittstemperatursensor TB0 Kurzschluss",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0502"} = {errorcode => "A01",                                    subcode => "5573",                                               class => "4",           reason => "Warnung Z1 Sole-Eintrittstemperatursensor TB0 Kabelbruch",                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0503"} = {errorcode => "A01",                                    subcode => "5574",                                               class => "4",           reason => "Warnung Z2 Sole-Eintrittstemperatursensor TB0 Kabelbruch",                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0504"} = {errorcode => "A01",                                    subcode => "5577",                                               class => "4",           reason => "Warnung Z1 Sole-Austrittstemperatursensor TB1 Kurzschluss",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0505"} = {errorcode => "A01",                                    subcode => "5578",                                               class => "4",           reason => "Warnung Z2 Sole-Austrittstemperatursensor TB1 Kurzschluss",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0506"} = {errorcode => "A01",                                    subcode => "5579",                                               class => "4",           reason => "Warnung Z1 Sole-Austrittstemperatursensor TB1 Kabelbruch",                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0507"} = {errorcode => "A01",                                    subcode => "5580",                                               class => "4",           reason => "Warnung Z2 Sole-Austrittstemperatursensor TB1 Kabelbruch",                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0508"} = {errorcode => "A01",                                    subcode => "5587",                                               class => "4",           reason => "Warnung Vorlauftemperaturf&uuml;hler/K&uuml;hlung TK1 Kabelbruch",                                                                                                                                                                                                                                                                                 checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0509"} = {errorcode => "A01",                                    subcode => "5588",                                               class => "4",           reason => "Warnung Vorlauftstemperaturf&uuml;hler/K&uuml;hlung TK1 Kurzschluss",                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0510"} = {errorcode => "A01",                                    subcode => "5590",                                               class => "4",           reason => "Warnung Eingangstemperaturf&uuml;hler/K&uuml;hlung TK2 Kabelbruch",                                                                                                                                                                                                                                                                                checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0511"} = {errorcode => "A01",                                    subcode => "5591",                                               class => "4",           reason => "Warnung Eingangstemperaturf&uuml;hler/K&uuml;hlung TK2 Kurzschluss",                                                                                                                                                                                                                                                                               checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0512"} = {errorcode => "A01",                                    subcode => "5597",                                               class => "4",           reason => "Warnung Z1 Verdampfertemp.-sensor TR4 Kurzschluss",                                                                                                                                                                                                                                                                                                checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0513"} = {errorcode => "A01",                                    subcode => "5598",                                               class => "4",           reason => "Warnung Z1 Verdampfertemp.-sensor TR4 Kabelbruch",                                                                                                                                                                                                                                                                                                 checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0514"} = {errorcode => "A01",                                    subcode => "5600",                                               class => "4",           reason => "Warnung Kompressor blockiert (IPM Fehler)",                                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0515"} = {errorcode => "A01",                                    subcode => "5602",                                               class => "4",           reason => "Warnung Stromaufnahme Inverter zu gro&szlig;",                                                                                                                                                                                                                                                                                                     checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0516"} = {errorcode => "A01",                                    subcode => "5604",                                               class => "4",           reason => "Warnung Gleichspannung au&szlig;erhalb zul&auml;ssigen Bereichs",                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0517"} = {errorcode => "A01",                                    subcode => "5606",                                               class => "4",           reason => "Warnung Hochoder Niederdruckschalter hat ausgel&ouml;st",                                                                                                                                                                                                                                                                                          checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0518"} = {errorcode => "A01",                                    subcode => "5608",                                               class => "4",           reason => "Warnung Eingangssp. zu klein (<140VAC) oder zu hoch (>300VAC)",                                                                                                                                                                                                                                                                                    checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0519"} = {errorcode => "A01",                                    subcode => "5610",                                               class => "4",           reason => "Warnung Kompressor-phasenfolge falsch angeschlossen",                                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0520"} = {errorcode => "A01",                                    subcode => "5612",                                               class => "4",           reason => "Warnung Zu hohe Stromaufnahme am IGBT, Spule pr&uuml;fen",                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0521"} = {errorcode => "A01",                                    subcode => "5614",                                               class => "4",           reason => "Warnung Gleichspannung gr&ouml;&szlig;er als 420 VDC.",                                                                                                                                                                                                                                                                                            checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0522"} = {errorcode => "A01",                                    subcode => "5616",                                               class => "4",           reason => "Warnung Stromaufnahme des Kompressors zu gro&szlig;",                                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0523"} = {errorcode => "A01",                                    subcode => "5618",                                               class => "4",           reason => "Warnung Heissgastemperatur zu hoch",                                                                                                                                                                                                                                                                                                               checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0524"} = {errorcode => "A01",                                    subcode => "5620",                                               class => "4",           reason => "Warnung CT Sensor Fehler",                                                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0525"} = {errorcode => "A01",                                    subcode => "5622",                                               class => "4",           reason => "Warnung Heissgassensor TR6 Fehler",                                                                                                                                                                                                                                                                                                                checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0526"} = {errorcode => "A01",                                    subcode => "5624",                                               class => "4",           reason => "Warnung Lufttemperatursensor  Ausseneinheit TL2 Fehler",                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0527"} = {errorcode => "A01",                                    subcode => "5626",                                               class => "4",           reason => "Warnung VerdampferMitteltemp.-sensor TR8 Fehler",                                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0528"} = {errorcode => "A01",                                    subcode => "5628",                                               class => "4",           reason => "Warnung Sauggastemperatursensor TR5 Fehler",                                                                                                                                                                                                                                                                                                       checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0529"} = {errorcode => "A01",                                    subcode => "5630",                                               class => "4",           reason => "Warnung Nennsparnnung zu hoch Thermische Belastung zu hoch",                                                                                                                                                                                                                                                                                       checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0530"} = {errorcode => "A01",                                    subcode => "5632",                                               class => "4",           reason => "Warnung Interne InverterKommunikation gest&ouml;rt",                                                                                                                                                                                                                                                                                               checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0531"} = {errorcode => "A01",                                    subcode => "5634",                                               class => "4",           reason => "Warnung Interne CAN-Kommunikation der Au&szlig;eneinheit gest&ouml;rt",                                                                                                                                                                                                                                                                            checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0532"} = {errorcode => "A01",                                    subcode => "5636",                                               class => "4",           reason => "Warnung Phasenreihenfolge falsch oder Phase fehlt",                                                                                                                                                                                                                                                                                                checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0533"} = {errorcode => "A01",                                    subcode => "5638",                                               class => "4",           reason => "Warnung Defektes EEPROM auf der Platine in Au&szlig;eneinheit",                                                                                                                                                                                                                                                                                    checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0534"} = {errorcode => "A01",                                    subcode => "5640",                                               class => "4",           reason => "Warnung Kondensationstemp. zu hoch (gr&ouml;&szlig;er 65°C)",                                                                                                                                                                                                                                                                                     checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0535"} = {errorcode => "A01",                                    subcode => "5642",                                               class => "4",           reason => "Warnung Temperatur am Frequenzumrichter zu hoch (gr&ouml;&szlig;er 85°C)",                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0536"} = {errorcode => "A01",                                    subcode => "5644",                                               class => "4",           reason => "Warnung Kondensationstemp. zu niedrig",                                                                                                                                                                                                                                                                                                            checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0537"} = {errorcode => "A01",                                    subcode => "5646",                                               class => "4",           reason => "Warnung Frequenzumrichter Temperaturf&uuml;hler Fehler",                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0538"} = {errorcode => "A01",                                    subcode => "5648",                                               class => "4",           reason => "Warnung Kompressor Anlaufstrom zu hoch",                                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0539"} = {errorcode => "A01",                                    subcode => "5650",                                               class => "4",           reason => "Warnung Z1 Kondensatoraustrittstemperaturf&uuml;hler TR7 Kabelbruch",                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0540"} = {errorcode => "A01",                                    subcode => "5651",                                               class => "4",           reason => "Warnung Z1 Kondensatoraustrittstemperaturf&uuml;hler TR7 Kurzschluss",                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0541"} = {errorcode => "A01",                                    subcode => "5653",                                               class => "4",           reason => "Warnung Z2 Kondensatoraustrittstemperaturf&uuml;hler TR7 Kabelbruch",                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0542"} = {errorcode => "A01",                                    subcode => "5654",                                               class => "4",           reason => "Warnung Z2 Kondensatoraustrittstemperaturf&uuml;hler TR7 Kurzschluss",                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0543"} = {errorcode => "A01",                                    subcode => "5656",                                               class => "4",           reason => "Warnung Z1 Kondensatoreintrittstemperaturf&uuml;hler TR3 Kabelbruch",                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0544"} = {errorcode => "A01",                                    subcode => "5657",                                               class => "4",           reason => "Warnung Z1 Kondensatoreintrittstemperaturf&uuml;hler TR3 Kurzschluss",                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0545"} = {errorcode => "A01",                                    subcode => "5659",                                               class => "4",           reason => "Warnung Z2 Kondensatoreintrittstemperaturf&uuml;hler TR3 Kabelbruch",                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0546"} = {errorcode => "A01",                                    subcode => "5660",                                               class => "4",           reason => "Warnung Z2 Kondensatoreintrittstemperaturf&uuml;hler TR3 Kurzschluss",                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0547"} = {errorcode => "A01",                                    subcode => "5662",                                               class => "4",           reason => "Warnung Z1 Drucksensor Kabelbruch oder Kurzschluss",                                                                                                                                                                                                                                                                                               checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0548"} = {errorcode => "A01",                                    subcode => "5664",                                               class => "4",           reason => "Warnung Z1 Eintritt Verdampfertemperaturf&uuml;hler TR4 Kabelbruch oder Kurzschluss",                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0549"} = {errorcode => "A01",                                    subcode => "5666",                                               class => "4",           reason => "Warnung Z1 Verdampfungsdruck zu niedrig",                                                                                                                                                                                                                                                                                                          checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0550"} = {errorcode => "A01",                                    subcode => "5668",                                               class => "4",           reason => "Warnung Z1 Kommunikation zwischen Innenund Au&szlig;eneinheit gest&ouml;rt",                                                                                                                                                                                                                                                                       checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0551"} = {errorcode => "A01",                                    subcode => "5670",                                               class => "4",           reason => "Warnung Z1 Ventilator Au&szlig;eneinheit ist blockiert",                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0552"} = {errorcode => "A01",                                    subcode => "5672",                                               class => "4",           reason => "Warnung Z2 Drucksensor Kabelbruch oder Kurzschluss",                                                                                                                                                                                                                                                                                               checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0553"} = {errorcode => "A01",                                    subcode => "5674",                                               class => "4",           reason => "Warnung Z2 Eintritt Verdampfertemperaturf&uuml;hler TR4 Kabelbruch oder Kurzschluss",                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0554"} = {errorcode => "A01",                                    subcode => "5676",                                               class => "4",           reason => "Warnung Z2 Druck zu niedrig",                                                                                                                                                                                                                                                                                                                      checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0555"} = {errorcode => "A01",                                    subcode => "5678",                                               class => "4",           reason => "Warnung Z2 Kommunikation zwischen Innenund Au&szlig;eneinheit gest&ouml;rt",                                                                                                                                                                                                                                                                       checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0556"} = {errorcode => "A01",                                    subcode => "5680",                                               class => "4",           reason => "Warnung Z2 Ventilator Au&szlig;eneinheit ist blockiert",                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0557"} = {errorcode => "A01",                                    subcode => "5700",                                               class => "4",           reason => "Warnung Z1 Vorlauftemperatursensor TC2 Fehler",                                                                                                                                                                                                                                                                                                    checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0558"} = {errorcode => "A01",                                    subcode => "5701",                                               class => "4",           reason => "Warnung Z1 Kurzschluss Vorlauftemperaturf&uuml;hler TC2",                                                                                                                                                                                                                                                                                          checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0559"} = {errorcode => "A01",                                    subcode => "5703",                                               class => "4",           reason => "Warnung Z1 R&uuml;cklauftemperatursensor TC4 Fehler",                                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0560"} = {errorcode => "A01",                                    subcode => "5704",                                               class => "4",           reason => "Warnung Z1 Kurzschluss R&uuml;cklauftemperaturf&uuml;hler TC4",                                                                                                                                                                                                                                                                                    checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0561"} = {errorcode => "A01",                                    subcode => "5706",                                               class => "4",           reason => "Warnung Z1 Einspritztemperatursensor TR2  Fehler",                                                                                                                                                                                                                                                                                                 checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0562"} = {errorcode => "A01",                                    subcode => "5707",                                               class => "4",           reason => "Warnung Z1 Kurzschluss Einspritztemperaturf&uuml;hler TR2",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0563"} = {errorcode => "A01",                                    subcode => "5708",                                               class => "4",           reason => "Warnung Z1 Einspritztemperatursensor TR2  Fehler",                                                                                                                                                                                                                                                                                                 checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0564"} = {errorcode => "A01",                                    subcode => "5709",                                               class => "4",           reason => "Warnung Z1 Unterbrechung Temperaturf&uuml;hler Luftaustritt TL3",                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0565"} = {errorcode => "A01",                                    subcode => "5710",                                               class => "4",           reason => "Warnung Z1 Kurzschluss Temperaturf&uuml;hler Luftaustritt TL3",                                                                                                                                                                                                                                                                                    checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0566"} = {errorcode => "A01",                                    subcode => "5713",                                               class => "4",           reason => "Warnung Z1 Mitteldrucksensor Fehler",                                                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0567"} = {errorcode => "A01",                                    subcode => "5717",                                               class => "4",           reason => "Warnung Z1 K&auml;ltemittelleckage. Expansionsventil arbeitet nicht richtig",                                                                                                                                                                                                                                                                      checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0568"} = {errorcode => "A01",                                    subcode => "5718",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Spannungsdiff. zw den drei Eingangs-Phasen",                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0569"} = {errorcode => "A01",                                    subcode => "5719",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Ents&auml;ttigung",                                                                                                                                                                                                                                                                                                      checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0570"} = {errorcode => "A01",                                    subcode => "5720",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Interner Rechenfehler",                                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0571"} = {errorcode => "A01",                                    subcode => "5721",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Eingangsrelais offen",                                                                                                                                                                                                                                                                                                   checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0572"} = {errorcode => "A01",                                    subcode => "5722",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Stromunterschiede zw. den drei IGBT`s",                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0573"} = {errorcode => "A01",                                    subcode => "5723",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Stromunterschiede zw. den drei PFC IGBT`s.",                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0574"} = {errorcode => "A01",                                    subcode => "5724",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Kleinspannung Bereichs&uuml;berschreitung",                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0575"} = {errorcode => "A01",                                    subcode => "5725",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Motordrehzahl zu hoch",                                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0576"} = {errorcode => "A01",                                    subcode => "5726",                                               class => "4",           reason => "Warnung Z1 Inverterfehler DC Spannung zu niedrig",                                                                                                                                                                                                                                                                                                 checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0577"} = {errorcode => "A01",                                    subcode => "5727",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Drehmoment Grenze erreicht",                                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0578"} = {errorcode => "A01",                                    subcode => "5728",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Modbus Kommunikation gest&ouml;rt",                                                                                                                                                                                                                                                                                      checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0579"} = {errorcode => "A01",                                    subcode => "5729",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Verdichter-Scroll &Uuml;bertemperatur",                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0580"} = {errorcode => "A01",                                    subcode => "5730",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Verdichter-Motor &Uuml;bertemperatur",                                                                                                                                                                                                                                                                                   checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0581"} = {errorcode => "A01",                                    subcode => "5731",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Schaltkreis &Uuml;bertemperatur",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0582"} = {errorcode => "A01",                                    subcode => "5732",                                               class => "4",           reason => "Warnung Z1 Inverterfehler IGBT`s &Uuml;bertemperatur",                                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0583"} = {errorcode => "A01",                                    subcode => "5733",                                               class => "4",           reason => "Warnung Z1 Inverterfehler PFC IGBT's &Uuml;bertemperatur",                                                                                                                                                                                                                                                                                         checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0584"} = {errorcode => "A01",                                    subcode => "5734",                                               class => "4",           reason => "Warnung Z1 Inverter Kommunikation. zw. Signalund Hauptprozessor gest&ouml;rt",                                                                                                                                                                                                                                                                     checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0585"} = {errorcode => "A01",                                    subcode => "5735",                                               class => "4",           reason => "Warnung Z1 Inverter Kommunikation. zw. Hauptund Signalprozessor gest&ouml;rt",                                                                                                                                                                                                                                                                     checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0586"} = {errorcode => "A01",                                    subcode => "5736",                                               class => "4",           reason => "Warnung Z1 Temperaturdifferenz zw. den drei Inverter IGBT`s",                                                                                                                                                                                                                                                                                      checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0587"} = {errorcode => "A01",                                    subcode => "5737",                                               class => "4",           reason => "Warnung Z1 Temperaturdifferenz zw. den drei PFC IGBT`s",                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0588"} = {errorcode => "A01",                                    subcode => "5738",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Verdichter-Scroll Temp. zu niedrig",                                                                                                                                                                                                                                                                                     checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0589"} = {errorcode => "A01",                                    subcode => "5739",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Verdichter-Motor Temp. zu niedrig",                                                                                                                                                                                                                                                                                      checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0590"} = {errorcode => "A01",                                    subcode => "5740",                                               class => "4",           reason => "Warnung Z1 Inverterfehler Schaltkreis Temp. zu niedrig",                                                                                                                                                                                                                                                                                           checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0591"} = {errorcode => "A01",                                    subcode => "5741",                                               class => "4",           reason => "Warnung Z1 Inverterfehler IGBT`s Temp. zu niedrig",                                                                                                                                                                                                                                                                                                checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0592"} = {errorcode => "A01",                                    subcode => "5742",                                               class => "4",           reason => "Warnung Z1 Inverterfehler PFC IGBT's Temp. zu niedrig",                                                                                                                                                                                                                                                                                            checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0593"} = {errorcode => "A01",                                    subcode => "5743",                                               class => "4",           reason => "Warnung Z1 Inverter Kommunikation zw. AD-Wandler und Nebenprozessor gest&ouml;rt",                                                                                                                                                                                                                                                                 checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0594"} = {errorcode => "A01",                                    subcode => "5745",                                               class => "4",           reason => "Warnung Z1 Niederdrucksensor JR0 Druck zu hoch",                                                                                                                                                                                                                                                                                                   checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0595"} = {errorcode => "A01",                                    subcode => "5746",                                               class => "4",           reason => "Warnung Z1 Niederdrucksensor JR0 Druck zu niedrig",                                                                                                                                                                                                                                                                                                checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0596"} = {errorcode => "A01",                                    subcode => "5747",                                               class => "4",           reason => "Warnung Z1 Hochdrucksensor JR1 Druck zu niedrig",                                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0597"} = {errorcode => "A01",                                    subcode => "5748",                                               class => "4",           reason => "Warnung Z2 Vorlauftemperatursensor TC2 Fehler",                                                                                                                                                                                                                                                                                                    checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0598"} = {errorcode => "A01",                                    subcode => "5749",                                               class => "4",           reason => "Warnung Z2 Kurzschluss Vorlauftemperaturf&uuml;hler TC2",                                                                                                                                                                                                                                                                                          checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0599"} = {errorcode => "A01",                                    subcode => "5751",                                               class => "4",           reason => "Warnung Z2 R&uuml;cklauftemperatursensor TC4 Fehler",                                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0600"} = {errorcode => "A01",                                    subcode => "5752",                                               class => "4",           reason => "Warnung Z2 Kurzschluss R&uuml;cklauftemperaturf&uuml;hler TC4",                                                                                                                                                                                                                                                                                    checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0601"} = {errorcode => "A01",                                    subcode => "5754",                                               class => "4",           reason => "Warnung Z2 Einspritztemperatursensor TR2 Fehler",                                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0602"} = {errorcode => "A01",                                    subcode => "5755",                                               class => "4",           reason => "Warnung Z2 Kurzschluss Einspritztemperaturf&uuml;hler TR2",                                                                                                                                                                                                                                                                                        checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0603"} = {errorcode => "A01",                                    subcode => "5757",                                               class => "4",           reason => "Warnung Z2 Unterbrechung Temperaturf&uuml;hler Luftaustritt TL3",                                                                                                                                                                                                                                                                                  checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0604"} = {errorcode => "A01",                                    subcode => "5758",                                               class => "4",           reason => "Warnung Z2 Kurzschluss Temperaturf&uuml;hler Luftaustritt TL3",                                                                                                                                                                                                                                                                                    checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0605"} = {errorcode => "A01",                                    subcode => "5761",                                               class => "4",           reason => "Warnung Z2 Mitteldrucksensor Fehler",                                                                                                                                                                                                                                                                                                              checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0606"} = {errorcode => "A01",                                    subcode => "5765",                                               class => "4",           reason => "Warnung Z2 K&auml;ltemittelleckage. Expansionsventil arbeitet nicht richtig",                                                                                                                                                                                                                                                                      checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0607"} = {errorcode => "A01",                                    subcode => "5766",                                               class => "4",           reason => "Warnung Z2 Inverterfehler Spannungsdiff. zw den drei Eingangs-Phasen",                                                                                                                                                                                                                                                                             checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                                                                                                                                                                                                                                                                                                                                                          correction => "-"                                                                                                                                                                                                                                                                                                                                                                                               };
	$ErrorCodeList{"0608"} = {errorcode => "A01",                                    subcode => "5767",                                               class => "4",           reason => "Warnung Z2 Inverterfehler Ents&auml;ttigung",                                                                                                                                                                                                                                                                                                      checklist => "siehe Serviceunterlage der W&auml;rmepumpe",                                                                                                                          