/*
 * Copyright (C) 2015 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "IDBIndexInfo.h"

#include <wtf/CrossThreadCopier.h>
#include <wtf/text/MakeString.h>

namespace WebCore {

IDBIndexInfo::IDBIndexInfo(uint64_t identifier, IDBObjectStoreIdentifier objectStoreIdentifier, const String& name, IDBKeyPath&& keyPath, bool unique, bool multiEntry)
    : m_identifier(identifier)
    , m_objectStoreIdentifier(objectStoreIdentifier)
    , m_name(name)
    , m_keyPath(WTFMove(keyPath))
    , m_unique(unique)
    , m_multiEntry(multiEntry)
{
}

IDBIndexInfo IDBIndexInfo::isolatedCopy() const &
{
    return { m_identifier, m_objectStoreIdentifier, m_name.isolatedCopy(), crossThreadCopy(m_keyPath), m_unique, m_multiEntry };
}

IDBIndexInfo IDBIndexInfo::isolatedCopy() &&
{
    return { m_identifier, m_objectStoreIdentifier, WTFMove(m_name).isolatedCopy(), crossThreadCopy(WTFMove(m_keyPath)), m_unique, m_multiEntry };
}

#if !LOG_DISABLED

String IDBIndexInfo::loggingString(int indent) const
{
    StringBuilder indentString;
    for (int i = 0; i < indent; ++i)
        indentString.append(' ');
    return makeString(indentString.toString(), "Index: "_s, m_name, " ("_s, m_identifier, ") keyPath: "_s, WebCore::loggingString(m_keyPath), '\n');
}

String IDBIndexInfo::condensedLoggingString() const
{
    return makeString("<Idx: "_s, m_name, " ("_s, m_identifier, "), OS ("_s, m_objectStoreIdentifier, ")>"_s);
}

#endif

} // namespace WebCore
