#include "CheckTask.h"
#include "Utils.h"
#include "FsTab.h"
#include "Device.h"
#include <QJsonObject>
#include <QDir>
#include <QDebug>

CheckTask::CheckTask() : AsyncTask(), m_opType(OperateType::Invalid), m_recoveryType(RecoveryType::InvalidRecoveryType)
{}

CheckTask::CheckTask(OperateType opType, RecoveryType recoveryType) : AsyncTask(),
    m_opType(opType), m_recoveryType(recoveryType)
{}

CheckTask::~CheckTask()
{}

bool CheckTask::buildArgumentsForBackup()
{
    return true;
}

bool CheckTask::buildArgumentsForRestore(const QString &)
{
    return true;
}

void CheckTask::doResult()
{
    switch (m_opType) {
        case OperateType::CheckFullSystemBackupSpace: {
            this->checkFullSystemBackupSpace();
            break;
        }
        case OperateType::CheckIncSystemBackupSpace: {
            this->checkIncSystemBackupSpace();
            break;
        }
        case OperateType::CheckGhostBackupSpace: {
            this->checkGhostBackupSpace();
            break;
        }
        case OperateType::CheckDimFileUseSpace: {
            this->checkDimFileUseSpace();
            break;
        }
        default:
            break;
    }

    m_rootUUID = "";
    m_destUUID = "";
    m_destPath = "";
}

void CheckTask::setRootUUID(const QString &rootUUID)
{
    m_rootUUID = rootUUID;
}

void CheckTask::setDestUUID(const QString &destUUID)
{
    m_destUUID = destUUID;
}

void CheckTask::setDestPath(const QString &destPath)
{
    m_destPath = destPath;
}

void CheckTask::checkFullSystemBackupSpace()
{
    if (m_recoveryType != RecoveryType::Rsync) {
        return;
    }

    if (m_rootUUID.isEmpty() || m_destUUID.isEmpty()) {
        this->reportSpace(InvalidDirectory, 0, 0, "");
        return;
    }

    quint64 expectedBackupSizeBytes = 0;
    int liveFlag = 0;
    QString errMsg;

    QStringList noExcludeDirs;
    quint64 optSizeBytes = 0;
    if (!this->getSystemBackupDirSizeBytes(m_rootUUID, m_destUUID, "/opt", noExcludeDirs, optSizeBytes, errMsg)) {
        qCritical()<<"checkFullSystemBackupSpace: calculateDirSize /opt failed ";
        this->reportSpace(UnKnow, expectedBackupSizeBytes, liveFlag, errMsg);
        return;
    }

    QStringList varExcludeDirs = {"--exclude=/var/run/*", "--exclude=/var/cache/*"};
    quint64 varSizeBytes = 0;
    if (!this->getSystemBackupDirSizeBytes(m_rootUUID, m_destUUID, "/var", varExcludeDirs, varSizeBytes, errMsg)) {
        qCritical()<<"checkFullSystemBackupSpace: calculateDirSize /var failed";
        this->reportSpace(UnKnow, expectedBackupSizeBytes, liveFlag, errMsg);
        return;
    }

    const quint64 reserveSizeBytes = static_cast<quint64> (1 * GiB);
    DevicePtr pRootDevice(new Device(m_rootUUID));
    pRootDevice->calculateDiskSpace();
    DeviceInfoPtr pRootDevInfo = pRootDevice->getDeviceInfo();
    if (nullptr == pRootDevInfo) {
        this->reportSpace(NullPointer, expectedBackupSizeBytes, liveFlag, errMsg);
        return;
    }
    quint64 backupSizeBytes = optSizeBytes + varSizeBytes + reserveSizeBytes + pRootDevInfo->usedBytes;
 //   qDebug()<<"backup="<<backupSizeBytes<<", opt="<<optSizeBytes<<", var="<<varSizeBytes;

    if (m_rootUUID == m_destUUID) {
        if (pRootDevInfo->availableBytes < backupSizeBytes) {
            qCritical()<<"root Insufficient disk space, available = "<<pRootDevInfo->availableBytes;
            this->reportSpace(InsufficientDiskSpace, backupSizeBytes, liveFlag, errMsg);
            return;
        }

        this->reportSpace(OK, expectedBackupSizeBytes, liveFlag, errMsg);
        return;
    }

    DevicePtr pBackupDevice(new Device(m_destUUID));
    pBackupDevice->calculateDiskSpace();
    DeviceInfoPtr pBackupDevInfo = pBackupDevice->getDeviceInfo();
    if (nullptr == pBackupDevInfo) {
        this->reportSpace(NullPointer, expectedBackupSizeBytes, liveFlag, errMsg);
        return;
    }
    if (pBackupDevInfo->availableBytes < backupSizeBytes) {
        qCritical()<<"backup Insufficient disk space, available = "<<pBackupDevInfo->availableBytes
                   <<", backupSizeBytes = "<<backupSizeBytes;
        this->reportSpace(InsufficientDiskSpace, backupSizeBytes, liveFlag, errMsg);
        return;
    }

    this->reportSpace(OK, expectedBackupSizeBytes, liveFlag, errMsg);
}

void CheckTask::checkIncSystemBackupSpace()
{
    if (m_recoveryType != RecoveryType::Rsync && m_recoveryType != RecoveryType::OSTree) {
        return;
    }

    if (m_destUUID.isEmpty()) {
        this->reportSpace(InvalidDirectory, 0, 0, "destUUID is empty");
        return;
    }

    quint64 expectedBackupSizeBytes = 0;
    int liveFlag = 0;
    QString errMsg;

    DevicePtr pBackupDevice(new Device(m_destUUID));
    pBackupDevice->calculateDiskSpace();
    DeviceInfoPtr pBackupDevInfo = pBackupDevice->getDeviceInfo();
    if (nullptr == pBackupDevInfo) {
        this->reportSpace(NullPointer, expectedBackupSizeBytes, liveFlag, errMsg);
        return;
    }

    // 系统增量备份至少需要1Gib剩余可用空间
    const quint64 incrementBackupSizeBytes = static_cast<quint64> (1 * GiB);
    if (pBackupDevInfo->availableBytes < incrementBackupSizeBytes) {
        qCritical()<<"increment backup Insufficient disk space, available = "<<pBackupDevInfo->availableBytes;
        this->reportSpace(InsufficientDiskSpace, incrementBackupSizeBytes, liveFlag, errMsg);
        return;
    }

    this->reportSpace(OK, expectedBackupSizeBytes, liveFlag, errMsg);
}

void CheckTask::checkGhostBackupSpace()
{
    if (m_destUUID.isEmpty() || m_destPath.isEmpty()) {
        this->reportSpace(InvalidDirectory, 0, 0, "destUUID or m_destPath is empty");
        return;
    }

    QStringList usedPartsUuids = {};
    QStringList usedPartMountPoints = {};
    QStringList destPaths = {m_destPath};
    QStringList bindPaths = {};

    // 根据fstab找出当前系统使用的所有分区
    FSTabInfoList fstabInfos = FSTab::getFSTabFromFile("/etc/fstab");
    for (FSTabInfoPtr fstabInfoItem : fstabInfos) {
        QString fstabDeviceItem = fstabInfoItem->device;
        QString fstabMountPointItem = fstabInfoItem->mountPoint;
        QString fstabOptionItem = fstabInfoItem->options;

        if (!fstabDeviceItem.isEmpty()) {
            if (fstabDeviceItem.contains("=")) {
                fstabDeviceItem = fstabDeviceItem.right(fstabDeviceItem.length() - fstabDeviceItem.indexOf("=") - 1);
                usedPartsUuids.append(fstabDeviceItem);
                usedPartMountPoints.append(fstabMountPointItem);
                if (!fstabDeviceItem.compare(m_destUUID)) {
                    QString destMountPath = fstabMountPointItem + m_destPath;
                    destMountPath.replace("//", "/");
                    if (!destPaths.contains(destMountPath)) {
                        destPaths.append(destMountPath);
                    }
                }
            }
            if (fstabOptionItem.contains("bind")) {
                bindPaths.append(fstabMountPointItem);
            }
        }
    }

    // 获取已使用的空间
    qint64 usedDevKib = getUsedSpace(bindPaths) + (GiB / KiB);

    // 获取备份目标目录的可用空间大小
    quint64 backupDevAvailableBytes = 0;
    Device usedDev;
    DeviceInfoList usedDevices = usedDev.getDeviceByLsblk();
    auto devIter = std::find_if(usedDevices.begin(), usedDevices.end(), [=](DeviceInfoPtr devItem) {
            return devItem->uuid == m_destUUID;
        });
    if ( devIter != usedDevices.end() ){
        backupDevAvailableBytes = (*devIter)->availableBytes;
    }

    // 在挂载点列表中寻找压缩比几乎不会变的文件
    quint64 totalFileSize = getTotalFileSize(usedPartMountPoints);

    // 预估ghost镜像文件大小，加上其他分区已使用空间大小，与目标分区未使用空间大小做比较
    QString errMsg;
    qint64 ghostFileSize = usedDevKib * 0.37;
    quint64 totalUsedSize = usedDevKib + totalFileSize + ghostFileSize;
    qInfo() << "totalUsedSize is : "  << totalUsedSize << "to Gib is : " << totalUsedSize / MiB
            << " usedDevKib is : "  << usedDevKib  << "to Gib is : " << usedDevKib  / MiB
            << " totalFileSize is : " << totalFileSize << "to Gib is : " << totalFileSize / MiB
            << " ghostFileSize is : " << ghostFileSize << "to Gib is : " << ghostFileSize / MiB;
    qInfo() << "backupDevAvailableBytes is : " << backupDevAvailableBytes << "to Gib is : " << backupDevAvailableBytes / GiB;

    // 目标大小比较
    totalUsedSize = totalUsedSize * KiB;
    if (totalUsedSize < backupDevAvailableBytes) {
        this->reportSpace(OK, totalUsedSize, 0, errMsg);
    } else {
        this->reportSpace(InsufficientDiskSpace, totalUsedSize, 0, errMsg);
    }
}

void CheckTask::checkDimFileUseSpace()
{
    QJsonObject jsonObj;
    jsonObj.insert("errCode", OK);
    jsonObj.insert("operateType", m_opType);
    jsonObj.insert("dimFilePath", m_destPath);

    // 获取总空间
    quint64 dimFilesTotalSize = getDimFileTotalSize(m_destPath);

    // 判断当前目录空间是否满足要求
    QJsonObject InfoObj;
    QString err = "";
    QString imgFilePath = "";
    if (Utils::getDestPartInfoByDir(m_destPath, InfoObj, err)) {
        imgFilePath = (InfoObj.value("fsavail").toVariant().toLongLong() > dimFilesTotalSize) ? m_destPath : QString();
    }

    // 如果当前选择的目录，空间不满足要求，则尝试在系统其他地方寻找空间满足要求的分区
    quint64 thresholdBytes = 200 * 1024 * 1024L;
    Device::findSuitablePartition(dimFilesTotalSize, imgFilePath, thresholdBytes);

    if (!imgFilePath.isEmpty()) {
        // imgFilePath不为空，代表，找到了空间满足要求的位置
        imgFilePath += "/dimFileRestore";
        QDir imgFilePathDir;
        imgFilePathDir.mkpath(imgFilePath);
    }

    if (imgFilePath.contains(" ")) {
        imgFilePath = QString("\"%1\"").arg(imgFilePath);
    }
    qInfo()<<"checkDimFileUseSpace, end imgFilePath = "<<imgFilePath;

    jsonObj.insert("imgFilePath", imgFilePath);
    Q_EMIT spaceCheckFinished(jsonObj);

    return;
}

bool CheckTask::getSystemBackupDirSizeBytes(const QString &rootUUID, const QString &backupDeviceUUID,
                                            const QString &dirPath, const QStringList &excludeDir,
                                            quint64 &totalBytes, QString &errMsg)
{
    errMsg = "";
    bool samePartition = false;
    if (!Utils::calculateDirSize(dirPath, excludeDir, totalBytes, errMsg, samePartition)) {
        qCritical()<<"getBackupDirSizeBytes: calc dir: "<<dirPath<<" failed, errMsg = "<<errMsg;
        return false;
    }
//    qInfo()<<"dirPath = "<<dirPath<<", totalBytes = "<<totalBytes;

    return true;
}

void CheckTask::reportSpace(ErrorCode errCode, quint64 backupSizeBytes, int liveFlag, const QString &errMsg)
{
    QJsonObject jsonObj;
    jsonObj.insert("errCode", errCode);
    jsonObj.insert("backupSizeBytes", static_cast<qint64>(backupSizeBytes));
    jsonObj.insert("liveFlag", liveFlag);
    jsonObj.insert("errMsg", errMsg);
    jsonObj.insert("uuid", m_destUUID);
    jsonObj.insert("recoveryType", m_recoveryType);
//    sleep(1); // 验证转圈圈的效果

    Q_EMIT spaceCheckFinished(jsonObj);
}

qint64 CheckTask::getUsedSpace(const QStringList &bindPaths)
{
    qint64 usedDevBytes = 0;
    QStringList spaceNeedToFilter = {"/media","/cdrom","/dev","/proc","/run","/mnt","/sys","/tmp"};
    // 使用du命令从"/"获取将要备份的文件的总大小
    QDir dir("/");
    dir.setFilter(QDir::Dirs | QDir::NoDotAndDotDot);
    dir.setSorting(QDir::Name);
    //QStringList allFolder = dir.entryList();
    QFileInfoList allFolder = dir.entryInfoList();
    for (QFileInfo dirItem : allFolder) {
        QString dirName = dirItem.fileName();
        if (spaceNeedToFilter.contains("/" + dirName)) {
            continue;
        }
        QStringList fileSizes = Utils::getCmdListReturn(QString("du /%1 | awk '{print $1}'").arg(dirName));
        if (fileSizes.size() > 0) {
            qint64 fileSize = fileSizes.last().toLongLong();
            usedDevBytes = usedDevBytes + fileSize;
        }
    }

    // 减掉多算的备份目标目录的大小
    for (QString bindPathItem : bindPaths) {
        QStringList fileSizes = Utils::getCmdListReturn(QString("du %1 | awk '{print $1}'").arg(bindPathItem));
        if (fileSizes.size() > 0) {
            qint64 fileSize = fileSizes.last().toLongLong();
            usedDevBytes = usedDevBytes - fileSize;
        }
    }

    return usedDevBytes;
}

quint64 CheckTask::getTotalFileSize(const QStringList &list)
{
    QStringList allKeepFileList = {};
    for (QString usedMountPoint : list) {
        if (!usedMountPoint.compare("none") || !usedMountPoint.compare("/")) {
            continue;
        }
        QStringList files = Utils::getCmdListReturn(QString("find %1 \\( -name \"*.iso\" -o -name \"*.zip\" -o -name \"*.uimg\" \\)").arg(usedMountPoint));
        allKeepFileList.append(files);
    }

    // 处理列表数据，20条为一组，方便后面使用du命令获取文件大小信息
    QStringList splitKeepFileList;
    QStringList splitKeepFileListItem = {};
    for (QString keepFileItem : allKeepFileList) {
        splitKeepFileListItem.append(keepFileItem);
        if (splitKeepFileListItem.size() == 10) {
            splitKeepFileList.append(splitKeepFileListItem.join(" "));
            splitKeepFileListItem.clear();
        }
    }
    if (splitKeepFileListItem.size() > 0) {
        splitKeepFileList.append(splitKeepFileListItem.join(" "));
        splitKeepFileListItem.clear();
    }

    // 计算文件列表中的所有文件的大小总和
    QStringList allFileSizeList = {};
    for (QString splitKeepFileItem : splitKeepFileList) {
        QStringList fileSizes = Utils::getCmdListReturn(QString("du %1 | awk '{print $1}'").arg(splitKeepFileItem));
        if (fileSizes.size() > 0) {
            allFileSizeList.append(fileSizes);
        }
    }

    quint64 totalFileSize = 0;
    for (QString fileSize : allFileSizeList) {
        totalFileSize = totalFileSize + fileSize.toULongLong();
    }

    return totalFileSize;
}

quint64 CheckTask::getDimFileTotalSize(const QString &dimFilePath)
{
    QDir srcDir(dimFilePath);
    QStringList fileNames = {"*.dim"};
    QFileInfoList dimFileInfos = srcDir.entryInfoList(fileNames, QDir::Files | QDir::Readable, QDir::Name);
    quint64 dimFilesTotalSize = 0;
    for (QFileInfo item : dimFileInfos) {
        QString err = "";
        QJsonObject dimFileJsonInfo;
        if (!Utils::getDimFileJsonInfo(item.absoluteFilePath(), dimFileJsonInfo, err)) {
            continue;
        }
        dimFilesTotalSize += dimFileJsonInfo.value("totalReadableDataSize").toVariant().toULongLong();
    }

    return dimFilesTotalSize;
}
