//
// Created by uos on 2022/3/17.
//

#include "AdvancedWidget.h"
#include "common/ItemFrame.h"
#include "common/HelpDialog.h"
#include "utils/Utils.h"
#include "utils/Device.h"
#include <QVBoxLayout>
#include <QSpacerItem>
#include <DFontSizeManager>
#include <QStackedLayout>
#include <QLabel>
#include <DDialog>

static const int ITEM_FRAME_HEIGHT_VERTICAL_LAYOUT = 85;
static const int ITEM_FRAME_WIDTH_VERTICAL_LAYOUT = 550;

AdvancedWidget::AdvancedWidget(DWidget *parent) : DWidget(parent)
{
    this->setAccessibleName("AdvancedWidget");
    m_bDirectionRTL = Utils::isDirectionRTL();

    // 主题变换
    m_guiHelper = Dtk::Gui::DGuiApplicationHelper::instance();
    connect(m_guiHelper, &Dtk::Gui::DGuiApplicationHelper::themeTypeChanged, this, &AdvancedWidget::onThemeChange);
    // 初始化UI
    initUI();
    initConnect();
}

void AdvancedWidget::initHelpLayout()
{
    m_helpDialog = new HelpDialog();
    m_helpDialog->setFixedSize(QSize(380,90));
    QList<HelpInfoItem> helpInfos;
    HelpInfoItem infoItem;
    infoItem.title = ::QObject::tr("Ghost");
    infoItem.subTitle = ::QObject::tr("Create a .uimg image file of the current system with one click.");
    helpInfos.append(infoItem);
    m_helpDialog->setContentText(helpInfos);
}

void AdvancedWidget::initUI()
{
    int uiLayout = this->getAdvanceUiLayoutType();
    QVBoxLayout *mainLayout = new QVBoxLayout;
    if (UI::UI_LAYOUT_ADVANCE_HORIZONTAL == uiLayout) {
        mainLayout->setContentsMargins(0, 20, 0, 10);
    } else if (UI::UI_LAYOUT_ADVANCE_VERTICAL == uiLayout) {
        mainLayout->setContentsMargins(0, 10, 0, 5);
    }

    // 初始化主标题和副标题
    m_title = new QLabel;
    m_title->setText(tr("Choose an advanced function"));
    m_title->setAccessibleName("uosAdvanced_ChooseTitle");
    m_title->setAlignment(Qt::AlignHCenter);
    DFontSizeManager::instance()->bind(m_title, DFontSizeManager::T3);
    QFont font = m_title->font();
    font.setWeight(QFont::Bold);
    m_title->setFont(font);
    m_subTitle = new QLabel;
    m_subTitle->setText("");
    m_subTitle->setAccessibleName("uosAdvanced_ChooseSubTitle");
    m_subTitle->hide();

    m_nextBtn = new DPushButton;
    m_nextBtn->setText(tr("Next"));
    m_nextBtn->setAccessibleName("uosAdvanced_Next");
    m_nextBtn->setFixedSize(200, 36);
    m_nextBtn->setEnabled(false);

    // 主界面上的组合控件
    m_itemGhostFrame = new ItemFrame(
            ::QObject::tr("Ghost"),
            ::QObject::tr("Create a .uimg image file of the current system with one click."),
            ":/resources/icons/make_ghost.svg", this);
    m_itemGhostFrame->setLayoutStype(uiLayout);

    QHBoxLayout *nextBtnLayout = new QHBoxLayout;
    if (UI::UI_LAYOUT_ADVANCE_HORIZONTAL == uiLayout) {
        // 下一步按钮的布局
        QLabel *leftLabel = new QLabel;
        leftLabel->setFixedSize(20, 20);
        m_helpLabel = new QLabel;
        m_helpLabel->setAccessibleName("uosAdvanced_HelpLabel");
        m_helpLabel->installEventFilter(this);
        Dtk::Gui::DGuiApplicationHelper::ColorType themeType =  Dtk::Gui::DGuiApplicationHelper::instance()->themeType();
        m_helpLabel->setPixmap(HelpDialog::getPixMapByThemeType(themeType));
        m_helpLabel->setFixedSize(20, 20);

        nextBtnLayout->setContentsMargins(13, 0, 13, 0);
        nextBtnLayout->addWidget(leftLabel, 0, Qt::AlignLeft);
        nextBtnLayout->addWidget(m_nextBtn, 0, Qt::AlignHCenter);
        nextBtnLayout->addWidget(m_helpLabel, 0, Qt::AlignRight);
        this->initHelpLayout();

        m_itemGhostFrame->setFixedSize(QSize(300,260));
        m_itemGhostFrame->setIconPixmap(":/resources/icons/make_ghost.svg", QSize(136, 136));
    } else if (UI::UI_LAYOUT_ADVANCE_VERTICAL == uiLayout) {
        nextBtnLayout->setContentsMargins(0, 0, 0, 0);
        nextBtnLayout->addWidget(m_nextBtn, 0, Qt::AlignHCenter);

        m_itemGhostFrame->setFixedSize(QSize(ITEM_FRAME_WIDTH_VERTICAL_LAYOUT,ITEM_FRAME_HEIGHT_VERTICAL_LAYOUT));
        m_itemGhostFrame->setIconPixmap(":/resources/icons/make_ghost.svg", QSize(56, 56));
    }
    m_itemGhostFrame->initLayout();
    m_itemGhostFrame->showDescription(true);

    m_nextButtonWidget = new QWidget;
    m_nextButtonWidget->setAccessibleName("uosAdvanced_NextButtonWidget");
    m_nextButtonWidget->setLayout(nextBtnLayout);

    mainLayout->addWidget(m_title, 0, Qt::AlignHCenter);
    mainLayout->addWidget(m_subTitle, 0, Qt::AlignCenter);
    if (UI::UI_LAYOUT_ADVANCE_HORIZONTAL == uiLayout) {
        mainLayout->addStretch();
    } else if (UI::UI_LAYOUT_ADVANCE_VERTICAL == uiLayout) {
        mainLayout->setSpacing(28);
    }

    mainLayout->addWidget(m_itemGhostFrame, 0, Qt::AlignHCenter);
    mainLayout->addStretch();
    mainLayout->addStretch();
    mainLayout->addWidget(m_nextButtonWidget);

    this->setLayout(mainLayout);
}

int AdvancedWidget::getAdvanceUiLayoutType()
{
    int defaultLayout = Utils::getUILayoutType(UI::UI_ADVANCE_MODULE);
    if (-1 == defaultLayout) {
        qCritical()<<"getAdvanceUiLayoutType failed";
    }
    return defaultLayout;
}

void AdvancedWidget::initConnect()
{
    connect(m_nextBtn, &DPushButton::clicked, this, &AdvancedWidget::onNextBtnBtnClicked);
    connect(m_itemGhostFrame, &ItemFrame::clicked, this, &AdvancedWidget::onItemIcoBtnClicked);

    connect(this, &AdvancedWidget::enter, this, [=] {
        QPoint point = m_helpLabel->mapToGlobal(QPoint());
        QPoint nextBtnPoint = m_nextBtn->mapToGlobal(QPoint());
        int x = point.x();
        int w = m_helpDialog->width();
        if (x > nextBtnPoint.x()) {
            x -= w;
        }
        m_helpDialog->move(x, point.y() - m_helpDialog->height() -10);
        m_helpDialog->raise();
        m_helpDialog->activateWindow();
        m_helpDialog->show();
    });

    connect(this, &AdvancedWidget::leave, this, [=] {
        m_helpDialog->hide();
        if (!this->isActiveWindow()) {
            this->raise();
            this->activateWindow();
        }
    });
}

void AdvancedWidget::onItemIcoBtnClicked()
{
    m_nextBtn->setEnabled(true);
}

void AdvancedWidget::onNextBtnBtnClicked()
{
    bool isEncrypted = false;
    bool isLVM = Utils::isLVM(isEncrypted);
    bool limitGhost = false;
    static QString osEditionType = Utils::getOSEditionType();
    if ("Community" == osEditionType) {
        if (isLVM) {
            limitGhost = true;
        }
    } else {
        if (isEncrypted) {
            DDialog dlg;
            dlg.setMessage(tr("Your system has used full-disk encryption, so you cannot take the Ghost backup method"));
            dlg.setIcon(QIcon::fromTheme("dialog-warning"));
            dlg.exec();
            return;
        }
    }

    if (limitGhost) {
        DDialog dlg;
        dlg.setMessage(tr("The lvm file system is used or the disk is encrypted, so you cannot create uimg files for the system at present"));
        dlg.setIcon(QIcon::fromTheme("dialog-warning"));
        dlg.exec();
        return;
    }

    // 进入ghost备份先进行鉴权处理
    if (!Utils::authorization()) {
        return;
    }

    // 更新目标存储位置
    Q_EMIT showGhostWidgetSignal();
}

void AdvancedWidget::onThemeChange(Dtk::Gui::DGuiApplicationHelper::ColorType themeType)
{
    if (nullptr != m_helpLabel) {
        m_helpLabel->setPixmap(HelpDialog::getPixMapByThemeType(themeType));
    }
}

bool AdvancedWidget::eventFilter(QObject* watched, QEvent* event)
{
    if (watched == m_helpLabel) {
        switch (event->type()) {
            case QEvent::Enter: Q_EMIT enter(); break;
            case QEvent::Leave: Q_EMIT leave(); break;
            default: break;
        }
    }

    return QWidget::eventFilter(watched, event);
}
