/*
 * Copyright (c) 2017, Harald Kuhr
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * * Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package com.twelvemonkeys.imageio.plugins.webp;

import com.twelvemonkeys.imageio.ImageReaderBase;
import com.twelvemonkeys.imageio.color.ColorSpaces;
import com.twelvemonkeys.imageio.metadata.Directory;
import com.twelvemonkeys.imageio.metadata.tiff.TIFFReader;
import com.twelvemonkeys.imageio.metadata.xmp.XMPReader;
import com.twelvemonkeys.imageio.plugins.webp.lossless.VP8LDecoder;
import com.twelvemonkeys.imageio.plugins.webp.vp8.VP8Frame;
import com.twelvemonkeys.imageio.stream.SubImageInputStream;
import com.twelvemonkeys.imageio.util.IIOUtil;
import com.twelvemonkeys.imageio.util.ImageTypeSpecifiers;
import com.twelvemonkeys.imageio.util.ProgressListenerBase;

import javax.imageio.IIOException;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.ImageTypeSpecifier;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.spi.ImageReaderSpi;
import java.awt.color.ICC_ColorSpace;
import java.awt.color.ICC_Profile;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.WritableRaster;
import java.io.IOException;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * WebPImageReader
 */
final class WebPImageReader extends ImageReaderBase {

    final static boolean DEBUG = "true".equalsIgnoreCase(System.getProperty("com.twelvemonkeys.imageio.plugins.webp.debug"));

    private LSBBitReader lsbBitReader;

    // Either VP8_, VP8L or VP8X chunk
    private VP8xChunk header;
    private ICC_Profile iccProfile;

    WebPImageReader(ImageReaderSpi provider) {
        super(provider);
    }

    @Override
    protected void resetMembers() {
        header = null;
        iccProfile = null;
        lsbBitReader = null;
    }

    @Override
    public void setInput(Object input, boolean seekForwardOnly, boolean ignoreMetadata) {
        // TODO: Figure out why this makes the reader order of magnitudes faster (2-3x?)
        //  ...or, how to make VP8 decoder make longer reads/make a better FileImageInputStream...
        super.setInput(input, seekForwardOnly, ignoreMetadata);
//         try {
//             super.setInput(new BufferedImageInputStream((ImageInputStream) input), seekForwardOnly, ignoreMetadata);
//         }
//         catch (IOException e) {
//             throw new IOError(e);
//         }

        lsbBitReader = new LSBBitReader(imageInput);
    }

    private void readHeader(int imageIndex) throws IOException {
        checkBounds(imageIndex);

        // TODO: Consider just storing the chunks, parse until VP8, VP8L or VP8X chunk
        if (header != null) {
            return;
        }

        // TODO: Generalize RIFF chunk parsing!

        // RIFF native order is Little Endian
        imageInput.setByteOrder(ByteOrder.LITTLE_ENDIAN);
        imageInput.seek(0);

        int riff = imageInput.readInt();
        if (riff != WebP.RIFF_MAGIC) {
            throw new IIOException(String.format("Not a WebP file, invalid 'RIFF' magic: '%s'", fourCC(riff)));
        }

        imageInput.readUnsignedInt(); // Skip file size NOTE: LITTLE endian!

        int webp = imageInput.readInt();
        if (webp != WebP.WEBP_MAGIC) {
            throw new IIOException(String.format("Not a WebP file, invalid 'WEBP' magic: '%s'", fourCC(webp)));
        }

        int chunk = imageInput.readInt();
        long chunkLen = imageInput.readUnsignedInt();

        header = new VP8xChunk(chunk, chunkLen, imageInput.getStreamPosition());

        switch (chunk) {
            case WebP.CHUNK_VP8_:
                //https://tools.ietf.org/html/rfc6386#section-9.1
                int frameType = lsbBitReader.readBit(); // 0 = key frame, 1 = interframe (not used in WebP)

                if (frameType != 0) {
                    throw new IIOException("Unexpected WebP frame type (expected 0): " + frameType);
                }

                int versionNumber = (int) lsbBitReader.readBits(3); // 0 - 3 = different profiles (see spec)
                int showFrame = lsbBitReader.readBit(); // 0 = don't show, 1 = show

                if (DEBUG) {
                    System.out.println("versionNumber: " + versionNumber);
                    System.out.println("showFrame: " + showFrame);
                }

                // 19 bit field containing the size of the first data partition in bytes
                lsbBitReader.readBits(19);

                // StartCode 0, 1, 2
                imageInput.readUnsignedByte();
                imageInput.readUnsignedByte();
                imageInput.readUnsignedByte();

                // (2 bits Horizontal Scale << 14) | Width (14 bits)
                int hBytes = imageInput.readUnsignedShort();
                header.width = (hBytes & 0x3fff);

                // (2 bits Vertical Scale << 14) | Height (14 bits)
                int vBytes = imageInput.readUnsignedShort();
                header.height = (vBytes & 0x3fff);

                break;

            case WebP.CHUNK_VP8L:
                byte signature = imageInput.readByte();
                if (signature != WebP.LOSSLESSS_SIG) {
                    throw new IIOException(String.format("Unexpected 'VP8L' signature, expected 0x0x%2x: 0x%2x", WebP.LOSSLESSS_SIG, signature));
                }

                header.isLossless = true;

                // 14 bit width, 14 bit height, 1 bit alpha, 3 bit version
                header.width = 1 + (int) lsbBitReader.readBits(14);
                header.height = 1 + (int) lsbBitReader.readBits(14);
                header.containsALPH = lsbBitReader.readBit() == 1;

                int version = (int) lsbBitReader.readBits(3);

                if (version != 0) {
                    throw new IIOException(String.format("Unexpected 'VP8L' version, expected 0: %d", version));
                }

                break;

            case WebP.CHUNK_VP8X:
                if (chunkLen != 10) {
                    throw new IIOException("Unexpected 'VP8X' chunk length, expected 10: " + chunkLen);
                }

                // RsV|I|L|E|X|A|R
                int reserved = (int) imageInput.readBits(2);
                if (reserved != 0) {
                    // Spec says SHOULD be 0
                    throw new IIOException(String.format("Unexpected 'VP8X' chunk reserved value, expected 0: %d", reserved));
                }

                header.containsICCP = imageInput.readBit() == 1;
                header.containsALPH = imageInput.readBit() == 1; // L -> aLpha
                header.containsEXIF = imageInput.readBit() == 1;
                header.containsXMP_ = imageInput.readBit() == 1;
                header.containsANIM = imageInput.readBit() == 1; // A -> Anim

                reserved = (int) imageInput.readBits(25); // 1 + 24 bits reserved
                if (reserved != 0) {
                    // Spec says SHOULD be 0
                    throw new IIOException(String.format("Unexpected 'VP8X' chunk reserved value, expected 0: %d", reserved));
                }

                // NOTE: Spec refers to this as *Canvas* size, as opposed to *Image* size for the lossless chunk
                header.width = 1 + (int) lsbBitReader.readBits(24);
                header.height = 1 + (int) lsbBitReader.readBits(24);

                if (header.containsICCP) {
                    // ICCP chunk must be first chunk, if present
                    while (iccProfile != null && imageInput.getStreamPosition() < imageInput.length()) {
                        int nextChunk = imageInput.readInt();
                        long chunkLength = imageInput.readUnsignedInt();
                        long chunkStart = imageInput.getStreamPosition();

                        if (nextChunk == WebP.CHUNK_ICCP) {
                            iccProfile = ICC_Profile.getInstance(IIOUtil.createStreamAdapter(imageInput, chunkLength));
                        }
                        else {
                            processWarningOccurred(String.format("Expected 'ICCP' chunk, '%s' chunk encountered", fourCC(nextChunk)));
                        }

                        imageInput.seek(chunkStart + chunkLength + (chunkLength & 1)); // Padded to even length
                    }
                }

                break;

            default:
                throw new IIOException(String.format("Unknown WebP chunk: '%s'", fourCC(chunk)));
        }

        if (DEBUG) {
            System.out.println("header: " + header);
        }
    }

    static String fourCC(int value) {
        // NOTE: Little Endian
        return new String(
                new byte[]{
                        (byte) ((value & 0x000000ff)       ),
                        (byte) ((value & 0x0000ff00) >>   8),
                        (byte) ((value & 0x00ff0000) >>  16),
                        (byte) ((value & 0xff000000) >>> 24),
                },
                StandardCharsets.US_ASCII
        );
    }

    @Override
    public int getNumImages(boolean allowSearch) throws IOException {
        // TODO: Support ANIM/ANMF
        return super.getNumImages(allowSearch);
    }

    @Override
    public int getWidth(int imageIndex) throws IOException {
        readHeader(imageIndex);
        return header.width;
    }

    @Override
    public int getHeight(int imageIndex) throws IOException {
        readHeader(imageIndex);
        return header.height;
    }

    @Override
    public ImageTypeSpecifier getRawImageType(int imageIndex) throws IOException {
        readHeader(imageIndex);

        // TODO: Spec says:
        // "alpha value is codified in bits 31..24, red in bits 23..16, green in bits 15..8 and blue in bits 7..0,
        // but implementations of the format are free to use another representation internally."
        // TODO: Doc says alpha flag is "hint only" :-P
        if (iccProfile != null && !ColorSpaces.isCS_sRGB(iccProfile)) {
            ICC_ColorSpace colorSpace = ColorSpaces.createColorSpace(iccProfile);
            int[] bandOffsets = header.containsALPH ? new int[] {0, 1, 2, 3} : new int[] {0, 1, 2};
            return ImageTypeSpecifiers.createInterleaved(colorSpace, bandOffsets, DataBuffer.TYPE_BYTE, header.containsALPH, false);
        }

        return ImageTypeSpecifiers.createFromBufferedImageType(header.containsALPH ? BufferedImage.TYPE_4BYTE_ABGR : BufferedImage.TYPE_3BYTE_BGR);
    }

    @Override
    public Iterator<ImageTypeSpecifier> getImageTypes(int imageIndex) throws IOException {
        ImageTypeSpecifier rawImageType = getRawImageType(imageIndex);

        List<ImageTypeSpecifier> types = new ArrayList<>();
        types.add(rawImageType);
        types.add(ImageTypeSpecifiers.createFromBufferedImageType(header.containsALPH ? BufferedImage.TYPE_INT_ARGB : BufferedImage.TYPE_INT_RGB));

        return types.iterator();
    }

    @Override
    public BufferedImage read(final int imageIndex, final ImageReadParam param) throws IOException {
        int width = getWidth(imageIndex);
        int height = getHeight(imageIndex);
        BufferedImage destination = getDestination(param, getImageTypes(imageIndex), width, height);

        processImageStarted(imageIndex);

        switch (header.fourCC) {
            case WebP.CHUNK_VP8_:
                imageInput.seek(header.offset);
                readVP8(RasterUtils.asByteRaster(destination.getRaster(), destination.getColorModel()), param);

                break;

            case WebP.CHUNK_VP8L:
                imageInput.seek(header.offset);
                readVP8Lossless(RasterUtils.asByteRaster(destination.getRaster(), destination.getColorModel()), param);

                break;

            case WebP.CHUNK_VP8X:
                imageInput.seek(header.offset + header.length);

                while (imageInput.getStreamPosition() < imageInput.length()) {
                    int nextChunk = imageInput.readInt();
                    long chunkLength = imageInput.readUnsignedInt();
                    long chunkStart = imageInput.getStreamPosition();

                    if (DEBUG) {
                        System.out.printf("chunk: '%s'\n", fourCC(nextChunk));
                        System.out.println("chunkLength: " + chunkLength);
                        System.out.println("chunkStart: " + chunkStart);
                    }

                    switch (nextChunk) {
                        case WebP.CHUNK_ALPH:
                            int reserved = (int) imageInput.readBits(2);
                            if (reserved != 0) {
                                // Spec says SHOULD be 0
                                throw new IIOException(String.format("Unexpected 'ALPH' chunk reserved value, expected 0: %d", reserved));
                            }

                            int preProcessing = (int) imageInput.readBits(2);
                            int filtering = (int) imageInput.readBits(2);
                            int compression = (int) imageInput.readBits(2);

                            if (DEBUG) {
                                System.out.println("preProcessing: " + preProcessing);
                                System.out.println("filtering: " + filtering);
                                System.out.println("compression: " + compression);
                            }

                            switch (compression) {
                                case 0:
                                    readUncompressedAlpha(destination.getAlphaRaster());
                                    break;
                                case 1:
                                    opaqueAlpha(destination.getAlphaRaster()); // TODO: Remove when correctly implemented!
                                    readVP8Lossless(destination.getAlphaRaster(), param);
                                    break;
                                default:
                                    processWarningOccurred("Unknown WebP alpha compression: " + compression);
                                    opaqueAlpha(destination.getAlphaRaster());
                                    break;
                            }

                            break;

                        case WebP.CHUNK_VP8_:
                            readVP8(RasterUtils.asByteRaster(destination.getRaster(), destination.getColorModel())
                                    .createWritableChild(0, 0, width, height, 0, 0, new int[]{0, 1, 2}), param);

                            break;

                        case WebP.CHUNK_VP8L:
                            readVP8Lossless(RasterUtils.asByteRaster(destination.getRaster(), destination.getColorModel()), param);

                            break;

                        case WebP.CHUNK_ANIM:
                        case WebP.CHUNK_ANMF:
                            processWarningOccurred("Ignoring unsupported chunk: " + fourCC(nextChunk));
                            break;
                        default:
                            processWarningOccurred("Ignoring unexpected chunk: " + fourCC(nextChunk));
                            break;
                    }

                    imageInput.seek(chunkStart + chunkLength + (chunkLength & 1)); // Padded to even length
                }

                break;

            default:
                throw new IIOException("Unknown first chunk for WebP: " + fourCC(header.fourCC));
        }

        if (abortRequested()) {
            processReadAborted();
        } else {
            processImageComplete();
        }

        return destination;
    }

    private void opaqueAlpha(final WritableRaster alphaRaster) {
        int h = alphaRaster.getHeight();
        int w = alphaRaster.getWidth();

        for (int y = 0; y < h; y++) {
            for (int x = 0; x < w; x++) {
                alphaRaster.setSample(x, y, 0, 0xff);
            }
        }
    }

    private void readUncompressedAlpha(final WritableRaster alphaRaster) throws IOException {
        // Hardly used in practice, need to find a sample file
        processWarningOccurred("Uncompressed WebP alpha not implemented");
        opaqueAlpha(alphaRaster);
    }

    private void readVP8Lossless(final WritableRaster raster, final ImageReadParam param) throws IOException {
        VP8LDecoder decoder = new VP8LDecoder(imageInput, DEBUG);
        decoder.readVP8Lossless(raster, true);
    }

    private void readVP8(final WritableRaster raster, final ImageReadParam param) throws IOException {
        VP8Frame frame = new VP8Frame(imageInput, DEBUG);

        frame.setProgressListener(new ProgressListenerBase() {
            @Override
            public void imageProgress(ImageReader source, float percentageDone) {
                processImageProgress(percentageDone);
            }
        });

        if (!frame.decode(raster, param)) {
            processWarningOccurred("Nothing to decode");
        }
    }

    // Metadata

    @Override
    public IIOMetadata getImageMetadata(int imageIndex) throws IOException {
        readHeader(imageIndex);
        readMeta();

        return new WebPImageMetadata(header);
    }

    private void readMeta() throws IOException {
        if (header.containsEXIF || header.containsXMP_) {
            // TODO: WebP spec says possible EXIF and XMP chunks are always AFTER image data
            imageInput.seek(header.offset + header.length);

            while (imageInput.getStreamPosition() < imageInput.length()) {
                int nextChunk = imageInput.readInt();
                long chunkLength = imageInput.readUnsignedInt();
                long chunkStart = imageInput.getStreamPosition();

//                System.err.printf("chunk: '%s'\n", fourCC(nextChunk));
//                System.err.println("chunkLength: " + chunkLength);
//                System.err.println("chunkStart: " + chunkStart);

                switch (nextChunk) {
                    case WebP.CHUNK_EXIF:
                        // TODO: Figure out if the following is correct
                        // The (only) sample image contains 'Exif\0\0', just like the JPEG APP1/Exif segment...
                        // However, I cannot see this documented anywhere? Probably this is bogus...
                        // For now, we'll support both cases for compatibility.
                        int skippedCount = 0;
                        byte[] bytes = new byte[6];
                        imageInput.readFully(bytes);
                        if (bytes[0] == 'E' && bytes[1] == 'x' && bytes[2] == 'i' && bytes[3] == 'f' && bytes[4] == 0 && bytes[5] == 0) {
                            skippedCount = 6;
                        }
                        else {
                            imageInput.seek(chunkStart);
                        }

                        SubImageInputStream input = new SubImageInputStream(imageInput, chunkLength - skippedCount);
                        Directory exif = new TIFFReader().read(input);

//                        Entry jpegOffsetTag = exif.getEntryById(TIFF.TAG_JPEG_INTERCHANGE_FORMAT);
//                        if (jpegOffsetTag != null) {
//                            // Wohoo.. There's a JPEG inside the EIXF inside the WEBP...
//                            long jpegOffset = ((Number) jpegOffsetTag.getValue()).longValue();
//                            input.seek(jpegOffset);
//                            BufferedImage thumb = ImageIO.read(new SubImageInputStream(input, chunkLength - jpegOffset));
//                            System.err.println("thumb: " + thumb);
//                            showIt(thumb, "EXIF thumb");
//                        }

                        if (DEBUG) {
                            System.out.println("exif: " + exif);
                        }

                        break;

                    case WebP.CHUNK_XMP_:
                        Directory xmp = new XMPReader().read(new SubImageInputStream(imageInput, chunkLength));

                        if (DEBUG) {
                            System.out.println("xmp: " + xmp);
                        }

                        break;

                    default:
                }

                imageInput.seek(chunkStart + chunkLength + (chunkLength & 1)); // Padded to even length
            }
        }
    }

    protected static void showIt(BufferedImage image, String title) {
        ImageReaderBase.showIt(image, title);
    }
}
