/**
 * @file module_configure_debug.c
 * @brief Module debug level configuration management.
 *
 * This module provides functions to:
 * - Query debug levels from config file by module type
 * - Check if any module has debug enabled
 * - Modify config files safely with backup & atomic rename
 * - Set debug levels for single, multiple, or all modules
 *
 * Uses GLib, cJSON, and system I/O with full internationalization support via N_().
 */

#include <dirent.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <openssl/sha.h>
#include <stdlib.h>
#include <glib.h>

#include "module_configure.h"
#include "util.h"
#include "cJSON.h"

#define MAX_KEY_LEN         1024
#define MAX_LINE_LEN        512
#define LINE_BUF_SIZE       512

typedef struct sub_module_cfg
{
  char *name;
  char *shell_cmd;
} sub_module_cfg;

typedef struct module_cfg
{
  char *name;
  char *type;
  int reboot;
  int sub_modules_num;
  sub_module_cfg **sub_modules;
} module_cfg;

// Forward declarations
static int parse_hook_json_file(const char *filepath, module_cfg *cfg);
static void free_module_cfg(void *t_pointer);
static void free_submodule_cfg(sub_module_cfg *p_cfg);

G_DEFINE_AUTOPTR_CLEANUP_FUNC(module_cfg, free_module_cfg)
G_DEFINE_AUTOPTR_CLEANUP_FUNC(sub_module_cfg, free_submodule_cfg)

extern struct set *S_modules, *S_types;
extern unsigned char config_sha256[][SHA256_DIGEST_LENGTH];
extern int config_sha256_count;

// Global hash table to store module configurations, keyed by module name
GHashTable *g_module_cfgs = NULL;
static const char *default_group = "config";

/**
 * Frees a single sub_module_cfg structure.
 * @param p_cfg Pointer to the sub_module_cfg to free (can be NULL).
 */
static void free_submodule_cfg(sub_module_cfg *p_cfg) {
    if (!p_cfg) return;
    g_free(p_cfg->name);
    g_free(p_cfg->shell_cmd);
    g_free(p_cfg);
}

/**
 * GDestroyNotify callback for freeing a module_cfg structure.
 * Used by GHashTable when removing entries.
 * @param t_pointer Pointer to module_cfg.
 */
static void free_module_cfg(void *t_pointer) {
    module_cfg *p_cfg = t_pointer;
    if (!p_cfg) return;

    g_free(p_cfg->name);
    g_free(p_cfg->type);

    if (p_cfg->sub_modules) {
        for (int i = 0; p_cfg->sub_modules[i]; ++i) {
            free_submodule_cfg(p_cfg->sub_modules[i]);
        }
        g_free(p_cfg->sub_modules);
    }
    g_free(p_cfg);
}

/**
 * Initializes the global module configuration storage by loading all .json files
 * from the specified directory.
 *
 * @param dir_path Path to the directory containing JSON configuration files.
 * @return OK on success, ERROR on failure.
 */
int init_module_cfgs(const char *dir_path) {
    DIR *pdir = NULL;
    struct dirent *pdirent = NULL;
    char filepath[PATH_MAX] = {0};

    g_return_val_if_fail(dir_path != NULL && dir_path[0] != '\0', ERROR);

    // Avoid reinitialization
    if (g_module_cfgs) {
        return OK;
    }

    pdir = opendir(dir_path);
    if (!pdir) {
        fprintf(stderr, N_("Error: Failed to open directory '%s': %s\n"), dir_path, strerror(errno));
        return ERROR;
    }

    // Create hash table with string keys and full cleanup callbacks
    g_module_cfgs = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, free_module_cfg);
    if (!g_module_cfgs) {
        fprintf(stderr, "%s\n", N_("Error: Failed to create GHashTable."));
        closedir(pdir);
        return ERROR;
    }

    while ((pdirent = readdir(pdir)) != NULL) {
        // Skip current and parent directory entries
        if (strcmp(pdirent->d_name, ".") == 0 || strcmp(pdirent->d_name, "..") == 0) {
            continue;
        }

        snprintf(filepath, sizeof(filepath), "%s/%s", dir_path, pdirent->d_name);

        struct stat sbuf;
        if (stat(filepath, &sbuf) == -1) {
            fprintf(stderr, N_("Warning: stat failed on '%s': %s\n"), filepath, strerror(errno));
            continue;
        }

        // Only process regular .json files
        if (S_ISREG(sbuf.st_mode) && strstr(pdirent->d_name, ".json") != NULL) {
            module_cfg *mdle_cfg = g_new0(module_cfg, 1);

            int ret = parse_hook_json_file(filepath, mdle_cfg);
            if (ret < 0) {
                fprintf(stderr, N_("Error: Cannot parse '%s'\n"), filepath);
                continue;
            }

            g_hash_table_insert(g_module_cfgs, g_strdup(mdle_cfg->name), mdle_cfg);
        }
    }

    closedir(pdir);
    return OK;
}

/**
 * Deinitializes the module configuration system by destroying the global hash table.
 */
void deinit_module_cfgs(void) {
    if (g_module_cfgs) {
        g_hash_table_destroy(g_module_cfgs);
        g_module_cfgs = NULL;
    }
}

/**
 * Callback function used by g_hash_table_foreach to collect module names.
 * @param key Module name (string)
 * @param value module_cfg pointer (unused)
 * @param user_data GPtrArray to collect names
 */
static void collect_module_names(gpointer key, gpointer value, gpointer user_data) {
    GPtrArray *names = (GPtrArray *)user_data;
    module_cfg *cfg = (module_cfg *)value;
    if (cfg && cfg->name) {
        g_ptr_array_add(names, g_strdup(cfg->name)); // Copy for caller ownership
    }
}

/**
 * Retrieves an array of all loaded module names.
 *
 * The returned array is NULL-terminated. Caller must free each string and the array itself.
 *
 * Example:
 *   char **list = get_module_names();
 *   for (int i = 0; list[i]; i++) {
 *       printf("%s\n", list[i]);
 *       g_free(list[i]);
 *   }
 *   g_free(list);
 *
 * @return A newly allocated NULL-terminated array of strings, or NULL if empty/no data.
 */
char **get_module_names(void) {
    if (!g_module_cfgs || g_hash_table_size(g_module_cfgs) == 0) {
        return NULL;
    }

    GPtrArray *names = g_ptr_array_new();

    g_hash_table_foreach(g_module_cfgs, collect_module_names, names);
    g_ptr_array_add(names, NULL); // Null terminate

    return (char **)g_ptr_array_free(names, FALSE);
}

/**
 * Creates a directory recursively (like mkdir -p).
 * @param path Full path to create.
 * @return OK on success, ERROR on failure.
 */
static int create_dir(const char *path) {
    if (!path || !*path) return ERROR;

    struct stat st = {0};
    if (stat(path, &st) == 0 && S_ISDIR(st.st_mode)) {
        return OK;
    }

    g_autofree char *dir = g_strdup(path);
    if (!dir) return ERROR;

    char *p = dir;
    // Skip leading slashes/spaces
    while (*p && (*p == '/' || *p == ' ' || *p == '\t')) p++;

    for (; *p; ++p) {
        if (*p == '/') {
            *p = '\0';
            if (stat(dir, &st) != 0) {
                if (mkdir(dir, 0775) != 0) {
                    fprintf(stderr, N_("Failed to create directory: %s (%s)\n"), dir, strerror(errno));
                    return ERROR;
                }
            }
            *p = '/';
        }
    }

    return OK;
}

/**
 * Copies a file from source to destination.
 * Also ensures the destination directory exists.
 *
 * @param src Source file path.
 * @param dest Destination file path.
 * @return OK on success, ERROR on failure.
 */
/**
 * Copies a file from source to destination.
 */
static int copy_file(const char *src, const char *dest) {
    g_return_val_if_fail(src != NULL && dest != NULL, ERROR);

    const int BUFFER_SIZE = 4096;
    FILE *fpsrc = NULL, *fpdest = NULL;
    char buffer[BUFFER_SIZE];
    size_t bytesRead, bytesWritten;
    int ret = OK;

    // Ensure destination directory exists
    g_autofree char* dest_dir = g_path_get_dirname(dest);
    if (create_dir(dest_dir) != OK) {
        return ERROR;
    }

    fpsrc = fopen(src, "rb");
    if (!fpsrc) {
        fprintf(stderr, N_("Error: Cannot open source file '%s': %s\n"), src, strerror(errno));
        return ERROR;
    }

    fpdest = fopen(dest, "wb");
    if (!fpdest) {
        fprintf(stderr, N_("Error: Cannot create destination file '%s': %s\n"), dest, strerror(errno));
        fclose(fpsrc);
        return ERROR;
    }

    while ((bytesRead = fread(buffer, 1, BUFFER_SIZE, fpsrc)) > 0) {
        bytesWritten = fwrite(buffer, 1, bytesRead, fpdest);
        if (bytesWritten != bytesRead) {
            ret = ERROR;
            break;
        }
    }

    fclose(fpsrc);
    fclose(fpdest);

    if (ret != OK) {
        remove(dest);
        fprintf(stderr, N_("Error: File copy failed from '%s' to '%s'\n"), src, dest);
    }

    return ret;
}

/**
 * Enables or disables journal logging by installing appropriate config.
 *
 * @param flag true to enable journal logging, false to disable.
 * @return OK on success, ERROR on failure.
 */
static int set_journal_log(bool flag) {
    const char *src = NULL, *dest = NULL, *to_remove = NULL;

    if (flag) {
        src = JOURNAL_LOG_ON_CONF;
        dest = JOURNAL_LOG_ON_INSTALLED;
        to_remove = JOURNAL_LOG_OFF_INSTALLED;
    } else {
        src = JOURNAL_LOG_OFF_CONF;
        dest = JOURNAL_LOG_OFF_INSTALLED;
        to_remove = JOURNAL_LOG_ON_INSTALLED;
    }

    // Remove conflicting config if exists
    if (access(to_remove, F_OK) == 0) {
        int ret = remove(to_remove);
        fprintf(stdout, N_("Removed '%s': %s\n"), to_remove, (ret == 0) ? N_("OK") : N_("FAIL"));
        if (ret != 0) return ERROR;
    }

    // If target already exists, nothing to do
    if (access(dest, F_OK) == 0) {
        return OK;
    }

    // Copy new config into place
    int ret = copy_file(src, dest);
    fprintf(stdout, N_("Copied '%s' to '%s': %s\n"), src, dest, (ret == OK) ? N_("OK") : N_("FAIL"));

    return ret;
}

/**
 * Retrieves the debug level for a given module type from the debug levels configuration file.
 *
 * If the configuration file does not exist, defaults to "off".
 *
 * @param module_type The module type to query (e.g., "network", "storage").
 * @param level Output parameter: allocated string containing the debug level ("on", "debug", etc.).
 *              Caller must free() it.
 * @return OK on success, ERROR on failure (e.g., parse error, OOM).
 */
int config_module_get_debug_level_by_type(const char *module_type, char **level) {
    const char *filename = MODULES_DEBUG_LEVELS_PATH;
    g_autoptr(GKeyFile) keyfile = NULL;
    GError *error = NULL;

    g_return_val_if_fail(module_type != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);
    g_return_val_if_fail(g_module_cfgs != NULL, ERROR);

    *level = NULL;

    if (strcmp(module_type,"all") && strcmp(module_type,"coredump") && !g_hash_table_contains(g_module_cfgs, module_type)) {
        fprintf(stderr, N_("Error: Module '%s' is not supported.\n"), module_type);
        return ERROR;
    }

    // If config file doesn't exist, assume debug is off
    if (access(filename, F_OK) == -1) {
        *level = g_strdup("off");
        return (*level) ? OK : ERROR;
    }

    keyfile = g_key_file_new();

    if (!g_key_file_load_from_file(keyfile, filename, G_KEY_FILE_NONE, &error)) {
        fprintf(stderr, N_("Error: Failed to load config file %s: %s\n"), filename, error->message);
        g_error_free(error);
        return ERROR;
    }

    g_autofree char *value = g_key_file_get_value(keyfile, default_group, module_type, &error);
    if (error) {
        // Key not found is not an error, we'll use default
        if (error->code != G_KEY_FILE_ERROR_KEY_NOT_FOUND) {
            fprintf(stderr, N_("Error: Failed to get debug level for %s: %s\n"), module_type, error->message);
        }
        g_error_free(error);
        *level = g_strdup("off");
    } else {
        char *trimmed_value = g_strstrip(value);
        *level = g_strdup(trimmed_value && trimmed_value[0] != '\0' ? trimmed_value : "off");
    }

    return (*level) ? OK : ERROR;
}


/**
 * Get debug levels for multiple module types and concatenate them with commas
 *
 * This function takes a comma-separated list of module types, retrieves the debug level
 * for each module using config_module_get_debug_level_by_type(), and concatenates all
 * levels into a single string separated by commas.
 *
 * @param module_types Comma-separated list of module type names (e.g., "systemd,dde-dock,network")
 * @param levels Output parameter that will contain the concatenated debug levels string
 *               The caller is responsible for freeing this memory
 *
 * @return OK on success, ERROR if any module fails or memory allocation fails
 */
int config_module_get_debug_level_by_types(const char *module_types, char **levels) {
    g_return_val_if_fail(module_types != NULL, ERROR);
    g_return_val_if_fail(levels != NULL, ERROR);

    *levels = NULL;  // Initialize output to NULL

    _cleanup_strv_free_ char** names = g_strsplit(module_types, ",", -1);
    int count = g_strv_length(names);

    if (count <= 0 || !names) {
        fprintf(stderr, N_("Error: Invalid module names: %s\n"), module_types);
        return ERROR;
    }

    GPtrArray *level_array = g_ptr_array_new_with_free_func(g_free);
    int ret = OK;

    for (int i = 0; i < count; i++) {
        char *level = NULL;
        int module_result = config_module_get_debug_level_by_type(names[i], &level);

        if (module_result != OK || level == NULL) {
            fprintf(stderr, N_("Error: Failed to get debug level for module '%s'\n"), names[i]);
            ret = ERROR;
            break;  // Stop processing on first error
        }

        g_ptr_array_add(level_array, level);
    }

    // If any module failed, clean up and return error
    if (ret != OK) {
        g_ptr_array_free(level_array, TRUE);
        return ERROR;
    }

    // Add NULL terminator to the pointer array for g_strjoinv
    g_ptr_array_add(level_array, NULL);

    // Join all levels with commas
    *levels = g_strjoinv(",", (char**)level_array->pdata);

    // Clean up the array (this will free the individual level strings)
    g_ptr_array_free(level_array, TRUE);

    // If memory allocation failed for the final string
    if (!*levels) {
        fprintf(stderr, N_("Error: Failed to concatenate debug levels\n"));
        return ERROR;
    }

    return OK;
}

/**
 * Checks whether any module in the debug levels config has level "on" or "debug".
 *
 * @param level Output: true if any module has debug enabled, false otherwise.
 * @return OK on success, ERROR on failure (e.g., file access error).
 */
int config_module_check_debug_level_has_on(bool *level) {
    const char *filename = MODULES_DEBUG_LEVELS_PATH;
    g_autoptr(GKeyFile) keyfile = NULL;
    GError *error = NULL;
    _cleanup_strv_free_ char **keys = NULL;
    gsize length = 0;
    int ret = OK;

    g_return_val_if_fail(level != NULL, ERROR);
    *level = false;

    if (access(filename, F_OK) == -1) {
        return OK; // No config → no debug
    }

    keyfile = g_key_file_new();
    if (!g_key_file_load_from_file(keyfile, filename, G_KEY_FILE_NONE, &error)) {
        fprintf(stderr, N_("Error: Failed to load config file %s: %s\n"), filename, error->message);
        g_error_free(error);
        return ERROR;
    }

    keys = g_key_file_get_keys(keyfile, default_group, &length, &error);
    if (error) {
        fprintf(stderr, N_("Error: Failed to get keys from config: %s\n"), error->message);
        g_error_free(error);
        return ERROR;
    }

    for (gsize i = 0; i < length; i++) {
        g_autofree char *value = g_key_file_get_value(keyfile, default_group, keys[i], NULL);
        if (value) {
            char *trimmed_value = g_strstrip(value);
            if (trimmed_value &&
                (g_strcmp0(trimmed_value, "on") == 0 || g_strcmp0(trimmed_value, "debug") == 0)) {
                *level = true;
                break;
            }
        }
    }

    return ret;
}

/**
 * Updates the debug level for a given module type in the config file.
 *
 * Creates the file if it doesn't exist.
 *
 * @param type Module type (e.g., "network").
 * @param level Desired level ("on", "off", "debug").
 * @return 0 on success, negative errno on failure.
 */
static int modify_debug_levels(const char *type, const char *level) {
    const char *conf_file = MODULES_DEBUG_LEVELS_PATH;
    g_autoptr(GKeyFile) keyfile = NULL;
    GError *error = NULL;
    int ret = OK;

    g_return_val_if_fail(type != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);

    // Ensure directory exists
    if (access(conf_file, F_OK) == -1) {
        if (create_dir(conf_file) != OK) {
            return ERROR;
        }
    }

    keyfile = g_key_file_new();

    // Load existing config if it exists
    if (access(conf_file, F_OK) == 0) {
        copy_file(conf_file,MODULES_DEBUG_LEVELS_FILENAME_BAK);

        if (!g_key_file_load_from_file(keyfile, conf_file,
                        G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS, &error)) {
            fprintf(stderr, N_("Warning: Failed to load existing config %s: %s\n"), conf_file, error->message);
            g_error_free(error);
            error = NULL;
        }
    }

    g_key_file_set_value(keyfile, default_group, type, level);

    gsize data_length;
    g_autofree char *data = g_key_file_to_data(keyfile, &data_length, &error);
    if (!data) {
        fprintf(stderr, N_("Error: Failed to serialize config: %s\n"), error->message);
        g_error_free(error);
        return ERROR;
    }

    if (!g_file_set_contents(conf_file, data, data_length, &error)) {
        fprintf(stderr, N_("Error: Failed to write config file %s: %s\n"), conf_file, error->message);
        g_error_free(error);
        ret = ERROR;
    }

    return ret;
}

/**
 * Check if a shell command file is allowed to execute by verifying its SHA256 checksum.
 *
 * @param file_path Path to the shell script file
 * @return true if the file exists and its SHA256 matches allowed hashes, false otherwise
 */
static bool is_shell_cmd_allowed(const char* file_path)
{
    unsigned char sha256_digest[SHA256_DIGEST_LENGTH];
    struct stat file_info;

    g_return_val_if_fail(file_path != NULL, false);

    // Check if file exists and is a regular file
    if (!(stat(file_path, &file_info) == 0 && S_ISREG(file_info.st_mode))) {
        return false;
    }

    // Calculate SHA256 digest of the file
    if (calculate_sha256(file_path, sha256_digest) < 0) {
        fprintf(stderr, N_("Warning: Failed to calculate SHA256 digest for %s\n"), file_path);
        return false;
    }

    // Compare with allowed SHA256 digests
    for (int i = 0; i < config_sha256_count; i++) {
        if (memcmp(config_sha256[i], sha256_digest, SHA256_DIGEST_LENGTH) == 0) {
            return true;
        }
    }

    return false;
}

/**
 * Execute a debug shell command with the specified debug level.
 *
 * @param filename Shell script filename (without path)
 * @param level Debug level to set
 * @return OK on success, ERROR on failure
 */
int exec_debug_shell_cmd_internal(const char *filename, const char *level) {
    char real_path[PATH_MAX] = {0};
    char real_level[PATH_MAX] = {0};
    int ret = OK;

    g_return_val_if_fail(filename != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);

    // Construct full path to shell script
    snprintf(real_path, sizeof(real_path), "%s/%s", CONFIG_SHELL_PATH, filename);
    snprintf(real_level, sizeof(real_level), "debug=%s", level);

    // Verify shell script integrity
    if (!is_shell_cmd_allowed(real_path)) {
        fprintf(stderr, N_("Error: SHA256 digest mismatch for shell file %s - file may have been modified\n"),
                real_path);
        return ERROR;
    }

    // Execute the shell script
    ret = start_process(real_path, real_level, NULL);
    if (ret != OK) {
        fprintf(stderr, N_("Error: Failed to execute %s %s (ret=%d, errno=%d)\n"),
                real_path, real_level, ret, errno);
        ret = ERROR;
    }

    return ret;
}

/**
 * Check if a package is installed on the system.
 *
 * @param package_name Name of the package to check
 * @return 1 if package is installed, 0 if not installed, ERROR on invalid input
 */
int check_package_installed(const char *package_name) {
    char cmd_arg[512];
    g_autofree char *output = NULL;

    g_return_val_if_fail(package_name != NULL && package_name[0] != '\0', ERROR);

    // Use dpkg-query to check package installation status
    snprintf(cmd_arg, sizeof(cmd_arg), "-Wf=${db:Status-Abbrev} %s", package_name);

    if (start_process("/usr/bin/dpkg-query", cmd_arg, &output) != OK) {
        return 0; // Package not installed or query failed
    }

    // Check if package status is "ii" (installed)
    return (strncmp(output, "ii", 2) == 0);
}

/**
 * Set debug level for all sub-modules of a given module configuration.
 *
 * @param mdle_cfg Module configuration structure
 * @param level Debug level to set
 * @return OK if all sub-modules configured successfully, ERROR on failure
 */
static int config_modules_set_debug_level_internal(const module_cfg *mdle_cfg, const char *level) {
    int ret = OK;
    int submodule_result = OK;

    g_return_val_if_fail(mdle_cfg != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);

    // Process all sub-modules
    for (int i = 0; mdle_cfg->sub_modules[i]; i++) {
        submodule_result = exec_debug_shell_cmd_internal(mdle_cfg->sub_modules[i]->shell_cmd, level);

        if (submodule_result != OK) {
            // Skip if package is not installed
            if (check_package_installed(mdle_cfg->sub_modules[i]->name) == 0) {
                fprintf(stderr, "Package %s is not installed, skipping\n",
                        mdle_cfg->sub_modules[i]->name);
                continue;
            }

            fprintf(stderr, "Failed to execute '%s' with level '%s'\n",
                    mdle_cfg->sub_modules[i]->shell_cmd, level);
            fprintf(stderr, N_("Error: Failed to configure %s\n"), mdle_cfg->name);

            // Update return code if this is the first error
            if (ret == OK) {
                ret = submodule_result;
            }
        }
    }

    // Update configuration file if all operations succeeded
    if (ret == OK) {
        modify_debug_levels(mdle_cfg->name, level);
    }

    fprintf(stdout, "Set %s debug level to %s: %s\n",
            mdle_cfg->name, level, (ret == OK) ? "OK" : "FAIL");

    return ret;
}

/**
 * Set debug level for all modules in the system.
 *
 * @param level Debug level to set for all modules
 * @return OK if all modules configured successfully, ERROR on failure
 */
static int config_modules_set_debug_level_all(const char *level) {
    GHashTableIter iter;
    module_cfg *mdle_cfg = NULL;
    int ret = OK;
    int module_result = OK;

    g_return_val_if_fail(g_module_cfgs != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);

    // Iterate through all modules
    g_hash_table_iter_init(&iter, g_module_cfgs);
    while (g_hash_table_iter_next(&iter, NULL, (void**)&mdle_cfg)) {
        module_result = config_modules_set_debug_level_internal(mdle_cfg, level);
        if (ret == OK) {
            ret = module_result;
        }
    }

    // Update global "all" setting if successful
    if (ret == OK) {
        modify_debug_levels("all", level);
    }

    return ret;
}

/**
 * Set debug level for all modules of a specific type.
 *
 * @param module_type Type of modules to configure, or "all" for all modules
 * @param level Debug level to set
 * @return OK on success, ERROR if no matching modules found or execution fails
 */
int config_modules_set_debug_level_by_type(const char* module_type, const char *level) {
    GHashTableIter iter;
    module_cfg *mdle_cfg = NULL;
    int ret = OK;
    int module_result = OK;
    int found = 0;

    g_return_val_if_fail(module_type != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);
    g_return_val_if_fail(g_module_cfgs != NULL, ERROR);

    // Handle "all" special case
    if (g_strcmp0(module_type, "all") == 0) {
        return config_modules_set_debug_level_all(level);
    }

    // Find and configure modules of the specified type
    g_hash_table_iter_init(&iter, g_module_cfgs);
    while (g_hash_table_iter_next(&iter, NULL, (void**)&mdle_cfg)) {
        if (g_strcmp0(mdle_cfg->type, module_type) != 0) {
            continue;
        }

        found = 1;
        module_result = config_modules_set_debug_level_internal(mdle_cfg, level);
        if (ret == OK) {
            ret = module_result;
        }
    }

    if (!found) {
        fprintf(stderr, N_("Error: No modules of type '%s' found\n"), module_type);
        return ERROR;
    }

    return ret;
}

/**
 * Set debug level for a single module by name.
 *
 * @param module_name Name of the module to configure, or "all" for all modules
 * @param level Debug level to set
 * @return OK on success, ERROR if module not found or execution fails
 */
int config_module_set_debug_level_by_module_name(const char *module_name, const char *level) {
    module_cfg *mdle_cfg = NULL;

    g_return_val_if_fail(module_name != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);
    g_return_val_if_fail(g_module_cfgs != NULL, ERROR);

    // Handle "all" special case
    if (g_strcmp0(module_name, "all") == 0) {
        return config_modules_set_debug_level_all(level);
    }

    // Look up module by name
    mdle_cfg = g_hash_table_lookup(g_module_cfgs, module_name);
    if (!mdle_cfg) {
        fprintf(stderr, N_("Error: Module '%s' not found\n"), module_name);
        return ERROR;
    }

    return config_modules_set_debug_level_internal(mdle_cfg, level);
}

/**
 * Set debug level for multiple module types specified in a comma-separated string.
 */
int config_modules_set_debug_level_by_types(const char* module_types, const char *level) {
    g_return_val_if_fail(module_types != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);

    _cleanup_strv_free_ char** types = g_strsplit(module_types, ",", -1);
    int count = g_strv_length(types);

    if (count <= 0 || !types) {
        fprintf(stderr, N_("Error: Invalid module types: %s\n"), module_types);
        return ERROR;
    }

    int ret = OK;
    for (int i = 0; i < count; i++) {
        int type_result = config_modules_set_debug_level_by_type(types[i], level);
        if (ret == OK) {
            ret = type_result;
        }
    }

    return ret;
}

/**
 * Set debug level for multiple modules specified by name in a comma-separated string.
 */
int config_module_set_debug_level_by_module_names(const char *module_names, const char *level) {
    g_return_val_if_fail(module_names != NULL, ERROR);
    g_return_val_if_fail(level != NULL, ERROR);

    _cleanup_strv_free_ char** names = g_strsplit(module_names, ",", -1);
    int count = g_strv_length(names);

    if (count <= 0 || !names) {
        fprintf(stderr, N_("Error: Invalid module names: %s\n"), module_names);
        return ERROR;
    }

    int ret = OK;
    for (int i = 0; i < count; i++) {
        int module_result = config_module_set_debug_level_by_module_name(names[i], level);
        if (ret == OK) {
            ret = module_result;
        }
    }

    return ret;
}

/**
 * Check current log configuration and adjust journal log level accordingly.
 */
int config_module_check_log(void) {
    bool debug_enabled = false;
    int ret = OK;

    ret = config_module_check_debug_level_has_on(&debug_enabled);
    if (ret != OK) {
        return ret;
    }

    return set_journal_log(debug_enabled);
}

/**
 * Get the reboot requirement for a module or all modules.
 */
int config_module_get_property_reboot(const char *module_name, int *reboot) {
    module_cfg *mdle_cfg = NULL;
    GHashTableIter iter;

    g_return_val_if_fail(module_name != NULL, ERROR);
    g_return_val_if_fail(reboot != NULL, ERROR);
    g_return_val_if_fail(g_module_cfgs != NULL, ERROR);

    *reboot = 0;

    if (g_strcmp0(module_name, "all") == 0) {
        g_hash_table_iter_init(&iter, g_module_cfgs);
        while (g_hash_table_iter_next(&iter, NULL, (void**)&mdle_cfg)) {
            if (mdle_cfg->reboot) {
                *reboot = 1;
                break;
            }
        }
    } else {
        mdle_cfg = g_hash_table_lookup(g_module_cfgs, module_name);
        if (mdle_cfg == NULL) {
            fprintf(stderr, N_("Error: Module '%s' not found\n"), module_name);
            return ERROR;
        }
        *reboot = mdle_cfg->reboot;
    }

    return OK;
}

/**
 * Check if debug package installation is allowed by verifying shell script integrity.
 */
bool check_can_install_dbg(void) {
    return is_shell_cmd_allowed(INSTALL_DBGPKG_SHELL_PATH);
}

/**
 * Install debug packages for multiple modules.
 */
int config_modules_install_dbgpkgs(const char *module_names) {
    g_return_val_if_fail(module_names != NULL, ERROR);

    if (!is_shell_cmd_allowed(INSTALL_DBGPKG_SHELL_PATH)) {
        fprintf(stderr, N_("Error: SHA256 digest mismatch for install script\n"));
        return ERROR;
    }

    _cleanup_strv_free_ char** names = g_strsplit(module_names, ",", -1);
    int count = g_strv_length(names);

    if (count <= 0 || !names) {
        fprintf(stderr, N_("Error: Invalid module names: %s\n"), module_names);
        return ERROR;
    }


    for (int i = 0; i < count; i++) {
        int ret = config_module_install_dbgpkgs_internal(names[i]);
        if (ret != OK) {
            return ret;
        }
    }

    return OK;
}

/**
 * Install debug packages for a single module.
 */
int config_module_install_dbgpkgs_internal(const char *module_name) {
    g_return_val_if_fail(module_name != NULL, ERROR);

    int ret = start_process(INSTALL_DBGPKG_SHELL_PATH, module_name, NULL);
    if (ret != OK) {
        fprintf(stderr, N_("Error: Failed to install debug packages for %s\n"), module_name);
        return ERROR;
    }

    return OK;
}

/**
 * Enable or disable system core dump functionality.
 */
int config_system_coredump(bool enable) {
    const char *state = enable ? "on" : "off";
    int ret = OK;

    if (!is_shell_cmd_allowed(CONFIG_COREDUMP_SHELL_PATH)) {
        fprintf(stderr, N_("Error: SHA256 digest mismatch for coredump script\n"));
        return ERROR;
    }

    ret = start_process(CONFIG_COREDUMP_SHELL_PATH, state, NULL);
    if (ret != OK) {
        fprintf(stderr, N_("Error: Failed to configure core dump\n"));
        return ERROR;
    }

    ret = modify_debug_levels("coredump", state);
    return ret;
}

/**
 * Parse a JSON hook file to extract module configuration.
 */

/**
 * Parse and validate a single submodule configuration from JSON.
 *
 * @param item JSON object representing a submodule
 * @param out Output parameter for allocated sub_module_cfg
 * @return OK on success, ERROR on validation or allocation failure
 */
static int parse_one_submodule(cJSON *item, sub_module_cfg **out) {
    g_return_val_if_fail(item != NULL, ERROR);
    g_return_val_if_fail(out != NULL, ERROR);

    if (!cJSON_IsObject(item)) {
        fprintf(stderr, "Error: submodule is not a JSON object\n");
        return ERROR;
    }

    cJSON *name_item = cJSON_GetObjectItemCaseSensitive(item, "name");
    cJSON *exec_item = cJSON_GetObjectItemCaseSensitive(item, "exec");

    // Validate required name field
    if (!cJSON_IsString(name_item) || !name_item->valuestring || name_item->valuestring[0] == '\0') {
        fprintf(stderr, "Error: missing or invalid 'name' in submodule\n");
        return ERROR;
    }

    // Validate required exec field
    if (!cJSON_IsString(exec_item) || !exec_item->valuestring || exec_item->valuestring[0] == '\0') {
        fprintf(stderr, "Error: missing or invalid 'exec' in submodule\n");
        return ERROR;
    }

    // Allocate and initialize submodule configuration
    g_autoptr(sub_module_cfg) sub = g_new0(sub_module_cfg, 1);
    sub->name = g_strdup(name_item->valuestring);
    sub->shell_cmd = g_strdup(exec_item->valuestring);

    // Verify string allocation succeeded
    if (!sub->name || !sub->shell_cmd) {
        fprintf(stderr, "Error: memory allocation failed for submodule\n");
        return ERROR;
    }

    *out = g_steal_pointer(&sub);
    return OK;
}

/**
 * Parse the submodules array from JSON configuration.
 *
 * @param json JSON array containing submodule configurations
 * @param out_subs Output parameter for allocated submodule array
 * @param out_num Output parameter for number of submodules
 * @return OK on success, ERROR on validation or allocation failure
 */
static int parse_submodules(cJSON *json, sub_module_cfg ***out_subs, int *out_num) {
    g_return_val_if_fail(out_subs != NULL, ERROR);
    g_return_val_if_fail(out_num != NULL, ERROR);

    if (!cJSON_IsArray(json)) {
        fprintf(stderr, "Error: 'submodules' must be a JSON array\n");
        return ERROR;
    }

    int num = cJSON_GetArraySize(json);
    if (num == 0) {
        fprintf(stderr, "Error: 'submodules' array cannot be empty\n");
        return ERROR;
    }

    // Allocate submodule array with NULL terminator
    sub_module_cfg **subs = g_new0(sub_module_cfg*, num + 1);

    // Parse each submodule in the array
    for (int i = 0; i < num; i++) {
        cJSON *item = cJSON_GetArrayItem(json, i);
        if (parse_one_submodule(item, &subs[i]) != OK) {
            // Cleanup already allocated submodules on failure
            for (int j = 0; j < i; j++) {
                free_submodule_cfg(subs[j]);
            }
            g_free(subs);
            return ERROR;
        }
    }

    *out_subs = subs;
    *out_num = num;
    return OK;
}

/**
 * Check for duplicate keys in a JSON object to detect configuration errors.
 *
 * @param object JSON object to check for duplicate keys
 * @return OK if no duplicates found, ERROR if duplicates detected
 */
static int check_no_duplicate_keys(cJSON *object) {
    if (!cJSON_IsObject(object)) {
        return OK;
    }

    g_autoptr(GHashTable) keys = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    cJSON *item = object->child;

    while (item) {
        if (item->string) {
            if (g_hash_table_contains(keys, item->string)) {
                fprintf(stderr, "Error: duplicate key '%s' in JSON configuration\n", item->string);
                return ERROR;
            }
            g_hash_table_add(keys, g_strdup(item->string));
        }
        item = item->next;
    }

    return OK;
}

/**
 * Parse a module configuration from a JSON hook file.
 *
 * @param filename Path to the JSON configuration file
 * @param cfg Output parameter for parsed module configuration
 * @return OK on success, ERROR on file or parsing failure
 */

static int parse_hook_json_file(const char *filename, module_cfg *cfg) {
    g_return_val_if_fail(filename != NULL, ERROR);
    g_return_val_if_fail(cfg != NULL, ERROR);

    memset(cfg, 0, sizeof(*cfg));

    g_autofree char *content = NULL;
    cJSON *root = NULL;
    GError *error = NULL;
    int result = ERROR;

    // Read file
    if (!g_file_get_contents(filename, &content, NULL, &error)) {
        fprintf(stderr, "Error: Failed to read file '%s': %s\n", filename, error->message);
        g_error_free(error);
        return ERROR;
    }

    // Parse JSON content
    root = cJSON_Parse(content);
    if (!root) {
        const char *error_ptr = cJSON_GetErrorPtr();
        fprintf(stderr, "Error: Failed to parse JSON in '%s'\n", filename);
        if (error_ptr) {
            fprintf(stderr, "Error location: %s\n", error_ptr);
        }
        return ERROR;
    }

    if (!cJSON_IsObject(root)) {
        fprintf(stderr, "Error: JSON root must be an object in '%s'\n", filename);
        goto cleanup;
    }

    // Check for duplicate keys
    if (check_no_duplicate_keys(root) != OK) {
        goto cleanup;
    }

    // Parse required 'name' field
    cJSON *name_item = cJSON_GetObjectItemCaseSensitive(root, "name");
    if (!cJSON_IsString(name_item) || !name_item->valuestring || name_item->valuestring[0] == '\0') {
        fprintf(stderr, "Error: Required field 'name' is missing or invalid in '%s'\n", filename);
        goto cleanup;
    }
    cfg->name = g_strdup(name_item->valuestring);

    // Parse optional 'group' field (becomes type)
    cJSON *group_item = cJSON_GetObjectItemCaseSensitive(root, "group");
    if (cJSON_IsString(group_item) && group_item->valuestring && group_item->valuestring[0] != '\0') {
        cfg->type = g_strdup(group_item->valuestring);
    }

    // Parse optional 'reboot' field
    cJSON *reboot_item = cJSON_GetObjectItemCaseSensitive(root, "reboot");
    if (cJSON_IsBool(reboot_item)) {
        cfg->reboot = cJSON_IsTrue(reboot_item) ? 1 : 0;
    } else if (cJSON_IsNumber(reboot_item)) {
        cfg->reboot = reboot_item->valueint ? 1 : 0;
    }

    // Parse required 'submodules' array
    cJSON *submodules_item = cJSON_GetObjectItemCaseSensitive(root, "submodules");
    if (!submodules_item) {
        fprintf(stderr, "Error: Required field 'submodules' is missing in '%s'\n", filename);
        goto cleanup;
    }

    result = parse_submodules(submodules_item, &cfg->sub_modules, &cfg->sub_modules_num);

cleanup:
    if (root) {
        cJSON_Delete(root);
    }
    if (result != OK) {
        free_module_cfg(cfg);
        memset(cfg, 0, sizeof(*cfg));
    }
    return result;
}

/**
 * Load key-value configuration from a single file using GKeyFile into existing hash table
 * @param config_file Configuration file path
 * @param levels Hash table to store the key-value pairs
 * @return OK on success, ERROR on failure. Returns OK even if file doesn't exist (empty hash)
 */
static int load_keyfile_config(const char *config_file, GHashTable *levels) {
    g_autoptr(GKeyFile) keyfile = NULL;
    GError *error = NULL;
    _cleanup_strv_free_ char **keys = NULL;
    gsize length = 0;

    g_return_val_if_fail(config_file != NULL, ERROR);
    g_return_val_if_fail(levels != NULL, ERROR);

    // If config file doesn't exist, return OK (empty config is valid)
    if (access(config_file, F_OK) == -1) {
        return OK;
    }

    keyfile = g_key_file_new();
    if (!g_key_file_load_from_file(keyfile, config_file, G_KEY_FILE_NONE, &error)) {
        fprintf(stderr, N_("Warning: Failed to load config file %s: %s\n"), config_file, error->message);
        g_error_free(error);
        return ERROR;
    }

    // Get all keys from the config file
    keys = g_key_file_get_keys(keyfile, default_group, &length, &error);
    if (error) {
        fprintf(stderr, N_("Warning: Failed to get keys from config file %s: %s\n"), config_file, error->message);
        g_error_free(error);
        return ERROR;
    }

    // Extract all key-value pairs
    for (gsize i = 0; i < length; i++) {
        g_autofree char *value = g_key_file_get_value(keyfile, default_group, keys[i], NULL);
        if (value) {
            char *trimmed_value = g_strstrip(value);
            if (trimmed_value && trimmed_value[0] != '\0') {
                g_hash_table_insert(levels, g_strdup(keys[i]), g_strdup(trimmed_value));
            }
        }
    }

    return OK;
}

/**
 * Load key-value configuration from a directory containing .conf files using GKeyFile
 * @param conf_dir Configuration directory path
 * @param levels Hash table to store the merged key-value pairs
 * @return OK on success, ERROR on failure. Returns OK even if directory doesn't exist (empty hash)
 */
static int load_keyfile_config_dir(const char *conf_dir, GHashTable *levels) {
    DIR *dir = NULL;
    struct dirent *entry = NULL;
    char filepath[PATH_MAX] = {0};

    g_return_val_if_fail(conf_dir != NULL, ERROR);
    g_return_val_if_fail(levels != NULL, ERROR);

    dir = opendir(conf_dir);
    if (!dir) {
        // Directory doesn't exist is not an error (no init configs)
        if (errno != ENOENT) {
            fprintf(stderr, N_("Warning: Failed to open config directory %s: %s\n"),
                    conf_dir, strerror(errno));
            return ERROR;
        }
        return OK;
    }

    while ((entry = readdir(dir)) != NULL) {
        // Skip current and parent directory entries
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0) {
            continue;
        }

        snprintf(filepath, sizeof(filepath), "%s/%s", conf_dir, entry->d_name);

        struct stat sbuf;
        if (stat(filepath, &sbuf) == -1) {
            fprintf(stderr, N_("Warning: stat failed on '%s': %s\n"), filepath, strerror(errno));
            continue;
        }

        // Only process regular .conf files
        if (S_ISREG(sbuf.st_mode) && strstr(entry->d_name, ".conf") != NULL) {
            if (load_keyfile_config(filepath, levels) != OK) {
                fprintf(stderr, N_("Warning: Failed to parse config file %s\n"), filepath);
                // Continue processing other files, don't fail immediately
            }
        }
    }

    closedir(dir);
    return OK;
}

/**
 * Apply debug levels according to priority rules
 * @param user_levels User configuration levels (highest priority)
 * @param init_levels Initial configuration levels (medium priority)
 * @return OK on success, ERROR if any module fails to apply
 */
static int apply_debug_levels(GHashTable *user_levels, GHashTable *init_levels) {
    GHashTableIter iter;
    module_cfg *mdle_cfg = NULL;
    int ret = OK;
    int module_result = OK;
    const char *final_level = "warning";  // Default level

    g_return_val_if_fail(g_module_cfgs != NULL, ERROR);
    g_return_val_if_fail(user_levels != NULL, ERROR);
    g_return_val_if_fail(init_levels != NULL, ERROR);

    // Check if user has set "all" level
    const char *global_level = g_hash_table_lookup(user_levels, "all");

    // Iterate through all modules
    g_hash_table_iter_init(&iter, g_module_cfgs);
    while (g_hash_table_iter_next(&iter, NULL, (void**)&mdle_cfg)) {
        const char *cur_level = global_level;

        // Priority 1: User configuration
        if (!cur_level) {
            const char *user_level = g_hash_table_lookup(user_levels, mdle_cfg->name);
            if (user_level) {
                cur_level = user_level;
            }
        }

        // Priority 2: Initial configuration (only if no user setting)
        if (!cur_level) {
            const char *init_level = g_hash_table_lookup(init_levels, mdle_cfg->name);
            if (init_level) {
                cur_level = init_level;
            }
        }
        // Priority 3: Default level (warning) is already set
        if (!cur_level) {
            cur_level = final_level;
        }

        // Apply the level to the module
        module_result = config_module_set_debug_level_by_module_name(mdle_cfg->name, cur_level);
        if (ret == OK) {
            ret = module_result;
        }
    }

    return ret;
}

/**
 * Reload debug levels according to the priority rules:
 * 1. User settings from /var/lib/deepin-debug-config/deepin-debug-levels.cfg
 * 2. Initial configuration from /usr/share/deepin-debug-config/deepin-debug-config.d/init-conf/
 * 3. Default level (warning) for unmatched modules
 *
 * @return OK on success, ERROR on failure
 */
int config_module_reload_debug_level(void) {
    const char *user_config = MODULES_DEBUG_LEVELS_PATH;
    const char *init_conf_dir = MODULES_INIT_CONFIG_PATH;

    GHashTable *user_levels = NULL;
    GHashTable *init_levels = NULL;
    int ret = ERROR;

    // Create hash tables for levels
    user_levels = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    init_levels = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    if (!user_levels || !init_levels) {
        fprintf(stderr, N_("Error: Failed to create hash tables\n"));
        goto cleanup;
    }

    // Step 1: Load user configuration (highest priority)
    if (load_keyfile_config(user_config, user_levels) != OK) {
        fprintf(stderr, N_("Warning: Failed to load user debug levels\n"));
        //continue
    }

    // Step 2: Load initial configuration from init-conf directory
    if (load_keyfile_config_dir(init_conf_dir, init_levels) != OK) {
        fprintf(stderr, N_("Error: Failed to load initial debug levels\n"));
        goto cleanup;
    }

    // Step 3: Apply levels according to priority rules
    ret = apply_debug_levels(user_levels, init_levels);

cleanup:
    if (user_levels) {
        g_hash_table_destroy(user_levels);
    }
    if (init_levels) {
        g_hash_table_destroy(init_levels);
    }
    return ret;
}

// The test function prints the parsed global hash.
static void print_one_module(gpointer key, gpointer value, gpointer user_data) {
    char *mod_name = (char *)key;
    module_cfg *cfg = (module_cfg *)value;

    printf("Module: %s\n", mod_name ? mod_name : "(null)");
    printf("  Type (group): %s\n", cfg->type ? cfg->type : "(null)");
    printf("  Reboot: %d\n", cfg->reboot);
    printf("  Submodules (%d):\n", cfg->sub_modules_num);

    for (int i = 0; i < cfg->sub_modules_num; i++) {
        sub_module_cfg *sub = cfg->sub_modules[i];
        if (!sub) continue;
        printf("    [%d] Name: %s, Exec: %s\n",
               i,
               sub->name ? sub->name : "(null)",
               sub->shell_cmd ? sub->shell_cmd : "(null)");
    }
    printf("\n");
}

void print_module_cfgs(void) {
    if (!g_module_cfgs) {
        printf("g_module_cfgs is NULL\n");
        return;
    }
    g_hash_table_foreach(g_module_cfgs, print_one_module, NULL);
}