//  (C) Copyright Nick Thompson 2021.
//  (C) Copyright Matt Borland 2022.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_MATH_COLOR_MAPS_HPP
#define BOOST_MATH_COLOR_MAPS_HPP
#include <algorithm> // for std::clamp
#include <array>     // for table data
#include <cmath>     // for std::floor
#include <cstdint>   // fixed width integer types

#if __has_include("lodepng.h")

#include "lodepng.h"
#include <iostream>
#include <string>
#include <vector>

namespace boost::math::tools {

// In lodepng, the vector is expected to be row major, with the top row
// specified first. Note that this is a bit confusing sometimes as it's more
// natural to let y increase moving *up*.
unsigned write_png(const std::string &filename,
                   const std::vector<std::uint8_t> &img, std::size_t width,
                   std::size_t height) {
  unsigned error = lodepng::encode(filename, img, width, height,
                                   LodePNGColorType::LCT_RGBA, 8);
  if (error) {
    std::cerr << "Error encoding png: " << lodepng_error_text(error) << "\n";
  }
  return error;
}

} // Namespace boost::math::tools
#endif // __has_include("lodepng.h")

namespace boost::math::tools {

namespace detail {

// Data taken from: https://www.kennethmoreland.com/color-advice
template <typename Real>
static constexpr std::array<std::array<Real, 3>, 256> extended_kindlmann_data_ = {{
    {0.0, 0.0, 0.0},
    {0.01780246283347332, 0.0008750907117329381, 0.01626889466306607},
    {0.03532931821571093, 0.001701802855992888, 0.03371323527689844},
    {0.05144940541667612, 0.0024840651967150905, 0.05129428731410766},
    {0.0645621163212177, 0.0032248645766847638, 0.06656239137344208},
    {0.07579594587865912, 0.0038635326859125735, 0.08040589852559868},
    {0.08560873187665013, 0.004465497712210337, 0.0932078727850277},
    {0.0942303824351094, 0.005050242022804749, 0.10546097842833466},
    {0.10126270743754426, 0.005684892741078179, 0.1185281778831713},
    {0.10670728559557496, 0.006372379780229372, 0.1327264977880264},
    {0.11057848107867363, 0.00711002225409038, 0.1481009557210271},
    {0.11345334766875653, 0.007801930722079915, 0.16375632363209905},
    {0.11546967754116774, 0.008551159105416175, 0.17919646921236213},
    {0.11670723282215431, 0.009290109551844812, 0.19453816576504604},
    {0.11719317716660224, 0.010053390003759367, 0.2096941755954832},
    {0.11698590922074067, 0.010738982214965748, 0.2248153138735493},
    {0.11610841583017484, 0.011423659103765034, 0.2397105899482299},
    {0.11462189563389044, 0.012116630983625478, 0.2543011097912221},
    {0.11260886490150057, 0.012821489721377753, 0.26850817678211986},
    {0.11014892407081325, 0.013468369062412165, 0.28238564749171374},
    {0.1073826713115642, 0.014119392392571815, 0.29573128232180523},
    {0.10439994774781663, 0.014757514745798027, 0.3086071932061514},
    {0.10102122555390916, 0.015343280218838306, 0.3215039849422961},
    {0.0973378808050489, 0.015957600087593806, 0.33424777423118057},
    {0.09343083659118528, 0.016584962806154634, 0.34680083232754255},
    {0.08930170392567455, 0.01713482653380763, 0.35927335578767955},
    {0.08525050060408929, 0.0177409908749586, 0.3713432638745381},
    {0.08133294699659654, 0.018315663017042745, 0.3831259644824107},
    {0.07770620978910117, 0.018846472161798116, 0.3945981112365364},
    {0.07453843004300356, 0.019323310547852572, 0.4057423733826484},
    {0.07234014248177378, 0.019905434286220342, 0.4162427930814269},
    {0.07070739337749535, 0.020358202603665252, 0.4265710517267966},
    {0.06795561424010746, 0.020880774373512326, 0.4372263626095861},
    {0.05495344243231267, 0.02153229240969933, 0.4511422122889977},
    {0.03272243214743511, 0.022251621841318136, 0.4663539858995716},
    {0.022542079629107543, 0.030087469482649072, 0.47217057965768244},
    {0.022458068304623265, 0.043510017517847034, 0.47110682779300844},
    {0.022335095562005403, 0.057612064160193856, 0.4678570071198978},
    {0.022116548807623634, 0.07134877081764332, 0.46268236332653717},
    {0.021827538746008197, 0.08452599623053175, 0.4558745921468776},
    {0.02152986717396973, 0.09703540427752388, 0.44774032508967526},
    {0.02094627101890751, 0.1088025513135693, 0.4387215006543877},
    {0.02045606049512309, 0.11985098869691474, 0.4289709820913745},
    {0.02004833311662032, 0.13019611448976093, 0.418764250948481},
    {0.019665629961975118, 0.13987346436551024, 0.4083506364708954},
    {0.019207577360338738, 0.14893167904386712, 0.39794947927856517},
    {0.018537140772831598, 0.15742727919356447, 0.38774823408835746},
    {0.01809424185998195, 0.16542627229987475, 0.3776911843959934},
    {0.01766141281376398, 0.1729786761988682, 0.3679328135626787},
    {0.01730286273144011, 0.18013597770041306, 0.3585022095429489},
    {0.016765513848818806, 0.18694679199155506, 0.3495097966827364},
    {0.016370126207011747, 0.19345202807035153, 0.3408690453055597},
    {0.016125995919003662, 0.19968741140572166, 0.3325877811713702},
    {0.015743835848192653, 0.20569130878531383, 0.32474294907403245},
    {0.01521780170762751, 0.21149505980303446, 0.31731795633577514},
    {0.014826523924383142, 0.21711739930665275, 0.31022963838193685},
    {0.01455583202575653, 0.22257980137551625, 0.3034714591858722},
    {0.014390286122102107, 0.2279012896050861, 0.29703484457819845},
    {0.014011918932944173, 0.23311039219433838, 0.29095765923101685},
    {0.013696545884500274, 0.23821209907670415, 0.28517017024142427},
    {0.013424574431729107, 0.24321964058367318, 0.2796599788292758},
    {0.013175868771764966, 0.2481448283322712, 0.27441390646690217},
    {0.01292963716411873, 0.2529982361555283, 0.2694181731856298},
    {0.012664284210951764, 0.25778935856659907, 0.2646585394556099},
    {0.012715975678052705, 0.26251129306552407, 0.26008624607226105},
    {0.012935277312292002, 0.26721180456135596, 0.25537236409530106},
    {0.013151828277929324, 0.2719132248379506, 0.25038669592493334},
    {0.013365797957379829, 0.2766141217472141, 0.24512715790174772},
    {0.013577263863962, 0.28131308067549815, 0.23959178395846548},
    {0.01378615567948453, 0.2860087090215386, 0.2337787381698218},
    {0.013992194922389659, 0.290699640854686, 0.22768632911662062},
    {0.014194831158161888, 0.2953845417355913, 0.22131302646406403},
    {0.014393175927633691, 0.3000621136791825, 0.21465748025951026},
    {0.014585935863149577, 0.3047311002377708, 0.20771854359069752},
    {0.014771346788489043, 0.3093902916806014, 0.2004952994228159},
    {0.015316330980203078, 0.314019615719643, 0.19303372895005055},
    {0.015485867479267465, 0.318655599270661, 0.18524675366655605},
    {0.015639979963365892, 0.32327849834847494, 0.17717467982889423},
    {0.015774461623191554, 0.32788734271858383, 0.16881795823876974},
    {0.01588440162820881, 0.33248123546361147, 0.16017753232003507},
    {0.016371319978387793, 0.3370394958269932, 0.15133715480703355},
    {0.016424021691114017, 0.34160094506458616, 0.14214299485979426},
    {0.01686035446719742, 0.3461250509254044, 0.13277234235578353},
    {0.016830214158745957, 0.35065150775079285, 0.12304027310946478},
    {0.017189890203442686, 0.3551393166240582, 0.11316410079747846},
    {0.0175057105195252, 0.35960835629733406, 0.10305638071018988},
    {0.017770106546070377, 0.36405841970084374, 0.09273371145052256},
    {0.01797534767005524, 0.3684894220401705, 0.08221895294189338},
    {0.018113811240203243, 0.37290140979305425, 0.07154410587314339},
    {0.01817830121675857, 0.37729457180004805, 0.060754926838460115},
    {0.018678925823257978, 0.3816479795981337, 0.05016037872247533},
    {0.01858851921615119, 0.38600461104105177, 0.039408863532102954},
    {0.018946886533322625, 0.39032256740422405, 0.0299653011886783},
    {0.019234421105765478, 0.3946241249528643, 0.022430553487672214},
    {0.023922664243633134, 0.39873762922211814, 0.019079643738415406},
    {0.032696329777136775, 0.40268821797442356, 0.01927104403025176},
    {0.04029295620970803, 0.40668863206181144, 0.019627058713507277},
    {0.0445051772107334, 0.4108066755028012, 0.019605221754402217},
    {0.05023033461306361, 0.4148702424589703, 0.019837552743924382},
    {0.05751407234865658, 0.4188712020471279, 0.01998429281648217},
    {0.06662959400279314, 0.42277996280066904, 0.020350820236973332},
    {0.07687976931053314, 0.42661141097115085, 0.02061535207978936},
    {0.08808647489550425, 0.4303590557627002, 0.02076563040362206},
    {0.10009395068503488, 0.4340168594615687, 0.020789592249385695},
    {0.11305049861381905, 0.43755751986676594, 0.021011622748947127},
    {0.1265170163106781, 0.440997674167719, 0.021092771828857385},
    {0.14064355039494827, 0.44431106175406876, 0.021367147770030247},
    {0.15509985051348987, 0.4475149894903065, 0.021487010699090946},
    {0.17002834364548297, 0.4505843077306851, 0.02179659002621885},
    {0.18517425344416824, 0.4535369003239919, 0.021939140218683366},
    {0.2005125857975878, 0.4563695433411364, 0.02190396156431214},
    {0.2161609253409036, 0.459058907077816, 0.022051019673724737},
    {0.23193556411255867, 0.46162347760839634, 0.022009915914684375},
    {0.24793825797637276, 0.4640413181325817, 0.022151516874270333},
    {0.26412488725558203, 0.4663117397956657, 0.022482129570858295},
    {0.2803606712938716, 0.4684536010278855, 0.02261757765237555},
    {0.296637938074376, 0.47046590244093683, 0.0225494975844844},
    {0.31302849974924285, 0.47232989838033596, 0.022671977207001336},
    {0.3294300351704986, 0.4740642435062818, 0.0225864988960952},
    {0.3459012409505257, 0.475652007794974, 0.022699138396783294},
    {0.3624157976623936, 0.4770953196310352, 0.02302034933556348},
    {0.37889809766361104, 0.47841222515883747, 0.02313701355469799},
    {0.39534430178227603, 0.4796035027123265, 0.023043880179838123},
    {0.41179003849439816, 0.48065635396111855, 0.023172253504396497},
    {0.42818080415815335, 0.4815874024816033, 0.02309359581707615},
    {0.4445420271983608, 0.48238619597444804, 0.02325212759648816},
    {0.46083197524945735, 0.48306807922963163, 0.023208795520823236},
    {0.477066527169599, 0.4836251567552959, 0.023421052378504333},
    {0.49330882182932195, 0.48404259768616237, 0.02351307077213582},
    {0.5101341569176142, 0.4841269939874597, 0.02474700862397102},
    {0.5276083798119485, 0.4838477594175364, 0.025493073746431755},
    {0.5457675808274345, 0.4831581006215279, 0.026232895897136985},
    {0.5646323060116551, 0.48201590123588106, 0.02698278973250526},
    {0.5842072626155643, 0.4803756805891137, 0.02819024072200311},
    {0.6045266103850581, 0.4781882736718959, 0.028981783530659853},
    {0.6255919000983664, 0.47539734049324306, 0.02979697562508809},
    {0.6473800379403847, 0.47195081918673226, 0.031044032153032777},
    {0.6699068794902967, 0.4677815168730453, 0.03227150878455887},
    {0.6932001837200186, 0.46280562738356607, 0.03304163325898518},
    {0.7171335868744761, 0.4569883986979126, 0.034530544840017025},
    {0.7417906038430161, 0.4502061037239105, 0.035463650965058914},
    {0.7670484685327643, 0.4424084941387379, 0.03656485419201359},
    {0.7927923527600884, 0.4335431093764269, 0.038110096816360045},
    {0.8190538618349916, 0.4234676649184017, 0.03921665123224114},
    {0.845635547462146, 0.4121565351852396, 0.040504503887090086},
    {0.8724273941843113, 0.399532161591502, 0.04177752341186975},
    {0.8993071734347423, 0.38551601321374424, 0.04288218868653492},
    {0.9259938049276442, 0.3701581656966096, 0.0443028413484576},
    {0.95240006413475, 0.3533630589382493, 0.045512688804357486},
    {0.9573302263433631, 0.356573181832402, 0.10485176035519561},
    {0.9593762651267365, 0.3625998458005728, 0.14695621074226434},
    {0.9609003469257634, 0.3691267599755898, 0.17919496865835138},
    {0.9621556849513918, 0.37591169845314204, 0.20542079832986643},
    {0.9631644672895096, 0.3829300179877212, 0.22750885439024698},
    {0.964095792386828, 0.39002737893200634, 0.24632170224175154},
    {0.9648793311938041, 0.3972605885272423, 0.26263686798960245},
    {0.9655077342802952, 0.40462643797868664, 0.2769429329997333},
    {0.9660587860054425, 0.41205105448230417, 0.2895251327064319},
    {0.9665994127853909, 0.4194734509128834, 0.3006180918171856},
    {0.9671516506858119, 0.4268421716026803, 0.31083965567414507},
    {0.9677276657441855, 0.4338269194529401, 0.3245165284653611},
    {0.968504522437997, 0.44042580568421147, 0.33966351784918897},
    {0.9692871734210263, 0.4467701596827656, 0.3565758064426104},
    {0.9702540772004374, 0.45269977328586647, 0.37534236971527246},
    {0.9712179772934669, 0.45837869586824703, 0.39562556319119124},
    {0.9722317464026985, 0.4637976195449472, 0.41693497643767774},
    {0.9732263821032568, 0.46903038930488866, 0.438958844869214},
    {0.9742462614418889, 0.4740678468975709, 0.46139896139324615},
    {0.9750871661398574, 0.4790671203997809, 0.48416548522290065},
    {0.9754147288760454, 0.4842669015553663, 0.5072296427052377},
    {0.9756180439862371, 0.48941330245201103, 0.5302646150938287},
    {0.975842338006588, 0.49442037450446213, 0.5531433026983604},
    {0.9761081874822527, 0.4992854244201668, 0.5758237590340696},
    {0.9763112164847584, 0.5040877038326032, 0.5983033566276463},
    {0.9765888685497933, 0.5087442572698033, 0.6205369331248914},
    {0.9767110237468087, 0.5134164774149236, 0.642535156682777},
    {0.9769336559396905, 0.5179413879764756, 0.6642639445578681},
    {0.97714593184672, 0.5224000733132641, 0.6857254497050739},
    {0.9773578783172437, 0.5267928368807481, 0.7069126158797037},
    {0.9775785758579868, 0.5311203562318053, 0.7278211218920855},
    {0.9778162437300041, 0.5353836351919077, 0.7484486916246914},
    {0.9779596006498089, 0.5396657970725794, 0.7687562310971295},
    {0.9782535179019477, 0.5438049537609145, 0.7888137943375886},
    {0.9784674751508416, 0.5479668227414403, 0.8085390020370261},
    {0.9786094914349676, 0.5521542299695001, 0.8279128914232778},
    {0.978802652274665, 0.5562873882925448, 0.8469838693949315},
    {0.9790515109370453, 0.5603687136966451, 0.8657552952207226},
    {0.9792467908639096, 0.5644843583828043, 0.884150643372716},
    {0.979395255578016, 0.5686380796177348, 0.902153526529961},
    {0.9796145283071062, 0.5727494207545081, 0.9198418221195682},
    {0.9797975302508571, 0.5769060493757885, 0.9371197812002375},
    {0.9799502605574898, 0.581112131921931, 0.9539723226660747},
    {0.9802283260758732, 0.5851928436561017, 0.9708243730936762},
    {0.9738084905670951, 0.5944815591203718, 0.9806690105429076},
    {0.9601472511008894, 0.609371508764095, 0.9813253841505029},
    {0.9478379001570983, 0.6230463820702153, 0.9820290537295789},
    {0.9366516898009005, 0.6357832537301149, 0.9826041912890565},
    {0.926560589527891, 0.647678912888478, 0.9831062604022219},
    {0.917620083682068, 0.6587517109196314, 0.9837253021469462},
    {0.9095356769991322, 0.6692646118840428, 0.9841128740419515},
    {0.9024501226204644, 0.6791460846603548, 0.9845858176159193},
    {0.8962465735427142, 0.6885075377321772, 0.9850533329169425},
    {0.8908928004592865, 0.6973907065167453, 0.9855589168505193},
    {0.8862829519968738, 0.7058867599470733, 0.9859932359170603},
    {0.8823207949642251, 0.7140769556818014, 0.9862329988164894},
    {0.8791184292035925, 0.7218801604700443, 0.9866187467737363},
    {0.876573928783208, 0.729373858466773, 0.9870301106239344},
    {0.8745946211951233, 0.7366283504729199, 0.9873335856528983},
    {0.8732148850190337, 0.7436116593104639, 0.9877197983463958},
    {0.8723449314581001, 0.7503888453166406, 0.9880489690947681},
    {0.871958051266914, 0.7569736347405297, 0.988343697831869},
    {0.8720269380214271, 0.7633787505197638, 0.9886267968388239},
    {0.872524018161346, 0.7696159559904953, 0.9889212756286642},
    {0.8734217880637981, 0.7756960853936921, 0.9892503149320232},
    {0.8746431257299938, 0.7816733363052604, 0.989454685042798},
    {0.8762135041761673, 0.7875116467112275, 0.9897367010924504},
    {0.878107018520635, 0.7932193523390607, 0.9901203210173019},
    {0.880203346150829, 0.798890286193674, 0.9902546300884126},
    {0.8825765528541782, 0.8044436600674107, 0.9905332036399778},
    {0.885156863735442, 0.8099282584935031, 0.9907909519806376},
    {0.8879236719089479, 0.815348848661961, 0.9910495231926625},
    {0.890857185433795, 0.8207097106633178, 0.9913312422092354},
    {0.8939377339865693, 0.8260147864401665, 0.9916599021959583},
    {0.8970530658331168, 0.8313161221171385, 0.9919275427432451},
    {0.8995688302115749, 0.8368229403489976, 0.9920969557419316},
    {0.9019890025620054, 0.8423510813517494, 0.9923144994581347},
    {0.9041189751589628, 0.8479541784289291, 0.992640232246134},
    {0.9058973525369924, 0.8536784698518803, 0.9928660652275899},
    {0.9073992963543378, 0.8594712199063077, 0.9931944239168621},
    {0.9086279741575543, 0.8653520270780298, 0.993459751831716},
    {0.9096655096337553, 0.8712669417189164, 0.9938665873784843},
    {0.9105035986931571, 0.8772607309343242, 0.9940839016190398},
    {0.9112280528710646, 0.8832787634425558, 0.994319531984062},
    {0.9118856791641718, 0.8893033549283489, 0.9945972975280256},
    {0.9125160989249034, 0.8953200253413901, 0.994937094070283},
    {0.9131521110159186, 0.9013364967475651, 0.9952011954414587},
    {0.9138294807228358, 0.907320474034156, 0.9955652276122916},
    {0.9145934489263029, 0.9132934622711121, 0.9957533433444864},
    {0.9154721624193913, 0.9192261671429544, 0.9959406615249712},
    {0.9165162261152294, 0.9251182443147156, 0.9960169412606291},
    {0.9177621627085478, 0.9309558310395318, 0.9960174468690394},
    {0.9192000090908993, 0.9367250258396427, 0.9960919139576695},
    {0.920883428013182, 0.9424235334496841, 0.9961424557894082},
    {0.922780751860871, 0.9480459375852194, 0.9963020587068238},
    {0.9250092107169314, 0.9535821762261815, 0.9963856518750184},
    {0.9275356684808912, 0.9590268903215627, 0.996529920191361},
    {0.9304408284972104, 0.9643654252860402, 0.996676965947248},
    {0.9338239050383462, 0.9695749605431102, 0.9967943472276378},
    {0.9376406019134518, 0.97465362230454, 0.9970154826822965},
    {0.9421630827040032, 0.9795433335221727, 0.9971921136372807},
    {0.9473578426139436, 0.9842358767121905, 0.9974820554788658},
    {0.9536370282349044, 0.9886296256370345, 0.9977639971129887},
    {0.961277626215249, 0.9926367016857381, 0.9981081638729355},
    {0.9706782225332987, 0.996129580837985, 0.9985982664103797},
    {0.982998706722621, 0.9987706424331216, 0.9991799080785383},
    {1.0, 1.0, 1.0},
}};

template <typename Real>
static constexpr std::array<std::array<Real, 3>, 256> kindlmann_data_ = {{
    {0.0, 0.0, 0.0},
    {0.017846074066284252, 0.0009158559874893362, 0.016056295374498146},
    {0.03572864786642702, 0.0017291229250328806, 0.03302143519907636},
    {0.05220741890989234, 0.002509857835251149, 0.04992588792612627},
    {0.06579320887318146, 0.0031951554951018895, 0.06457251900523656},
    {0.07760358429899875, 0.003724534058880596, 0.07776920342353211},
    {0.08789358790850602, 0.004294574826048965, 0.0897198393739522},
    {0.09712335630575433, 0.004842625166895046, 0.10086394984030281},
    {0.10521029512901023, 0.005373953338851922, 0.11223547956850252},
    {0.11206559338234914, 0.005946169553289608, 0.12432777397900399},
    {0.11777117178354526, 0.006573754050745371, 0.13716171084354922},
    {0.12274092212849055, 0.007190967398920267, 0.15010297798342745},
    {0.12721746956233745, 0.007765930791621213, 0.1628260628965371},
    {0.13120675065695167, 0.008361505755992952, 0.17526973313157249},
    {0.13476383339950823, 0.008965513206303781, 0.1874768588728739},
    {0.1379416850964659, 0.009551513984339633, 0.1995014270040738},
    {0.14078915018940993, 0.010080340518883863, 0.21140807652685528},
    {0.14328800070233533, 0.010654292784112113, 0.22305086172802313},
    {0.1454925697745392, 0.011223945666441406, 0.23450000042491753},
    {0.14744975665550916, 0.01172962096468942, 0.24583568409411483},
    {0.14915607230438568, 0.012265761398485226, 0.2569125532571454},
    {0.15064370028657081, 0.012825896285586958, 0.2678146545485108},
    {0.15192214754665317, 0.013340464397739584, 0.27899859238410224},
    {0.1529680602403743, 0.01383169406084123, 0.29046429953683084},
    {0.15375355050540976, 0.014469905953793126, 0.3019580272214856},
    {0.1542908116514455, 0.014991396474282258, 0.313852441550019},
    {0.15455584740256964, 0.015566553274758022, 0.3258902207027679},
    {0.15454470889937796, 0.01619279407587136, 0.3380624435826929},
    {0.15424074730776174, 0.01677345343135211, 0.3504951935746775},
    {0.15362513362574176, 0.01730005530615292, 0.3631847587056686},
    {0.15272770714157807, 0.017955417060815455, 0.3758470094405716},
    {0.15150259153104584, 0.018542834077193353, 0.3887481108848792},
    {0.14996750216169222, 0.019150121523940637, 0.4017384234617481},
    {0.1481223688773872, 0.019769579646988222, 0.4148055969839783},
    {0.14596849097254724, 0.02039268583437749, 0.42793691410477086},
    {0.14350871584895447, 0.021010135344188375, 0.4411193724446502},
    {0.14081675084553444, 0.021716621813360978, 0.4541885105212106},
    {0.13777021342781093, 0.022293516296525755, 0.4674318891638602},
    {0.13452661891393936, 0.022940593473192553, 0.48053217251642305},
    {0.13103033265010522, 0.023541432666866674, 0.49362776425328886},
    {0.12740778581316103, 0.024195126427194676, 0.5065484866366465},
    {0.12359314328868806, 0.024781571830105195, 0.5194370648731684},
    {0.11974712167592705, 0.025403149292983516, 0.5321219022221657},
    {0.1159404877051333, 0.0260531205128538, 0.5445888017818501},
    {0.11209206787148834, 0.02660656407054731, 0.5569877413493692},
    {0.10842241040440373, 0.02717266184945997, 0.5691469684334989},
    {0.1050253848722921, 0.027746789078543728, 0.5810568608426063},
    {0.101999609017976, 0.02832560680437759, 0.5927097735404816},
    {0.09922826785261996, 0.028779550542447317, 0.6042668353483279},
    {0.09722602817414985, 0.02936178919960733, 0.6153923636579981},
    {0.09528935573037353, 0.029890777438344337, 0.6264955320241609},
    {0.0930823731478326, 0.030438044833523397, 0.6376509097116411},
    {0.0781891239412737, 0.031121586300131683, 0.6525577781586464},
    {0.05857961917089497, 0.031924832133126, 0.6676204841351329},
    {0.03284057845765246, 0.03469657698669593, 0.6813738713631898},
    {0.032679442069734624, 0.04697971049610799, 0.6849034147771731},
    {0.033207560139530704, 0.05995495438213944, 0.6868241583911188},
    {0.033317187200492364, 0.07300667062971217, 0.6874759761236308},
    {0.033087445334472984, 0.08601513051125326, 0.6868754153808352},
    {0.03265381069556877, 0.09888929701373608, 0.6850522082772128},
    {0.03277467744530855, 0.11161079522124029, 0.6818870111240353},
    {0.0325022291000698, 0.1240111603824268, 0.6777553638535389},
    {0.032066045593596146, 0.13606903293962488, 0.6727110790151253},
    {0.03227600851941834, 0.14780096691985975, 0.6666596846794193},
    {0.031649331579416705, 0.15909372791133777, 0.6601264529755937},
    {0.03155801725217711, 0.17001566158899392, 0.6528643797965168},
    {0.0310838330844518, 0.18051935094604918, 0.6452318387821873},
    {0.030432973047734187, 0.19062325974122588, 0.6372784105620277},
    {0.030338619295788537, 0.20036054096647732, 0.6289128168374558},
    {0.029853960367767295, 0.20971568331952556, 0.6204629562646412},
    {0.02966850358552374, 0.2187242415010579, 0.6118301548875958},
    {0.02882374869999708, 0.2273866229818785, 0.603314553849001},
    {0.028500638040224537, 0.2357441994433061, 0.5946825020006182},
    {0.028245919974351025, 0.24380777892174738, 0.5860925629625072},
    {0.02760771294131195, 0.2515940717685919, 0.5776869157187913},
    {0.02717197158834595, 0.25912806556578544, 0.5693601411266609},
    {0.026982221792628543, 0.266427065214078, 0.5611305112918692},
    {0.026559642397133442, 0.27350794101395076, 0.5531226098917067},
    {0.02644787880084816, 0.2803874742418932, 0.5452351099365369},
    {0.02564842515734134, 0.2870852596241667, 0.5376830481507792},
    {0.02570596827933925, 0.29360869815637475, 0.5301627048805485},
    {0.025105902288682952, 0.2999809743840178, 0.5229767137199569},
    {0.024870375536254246, 0.30620709089548, 0.5159307080617925},
    {0.024495137563131944, 0.3123034508268453, 0.5091188274555094},
    {0.02397954323509251, 0.3182816255246543, 0.502536537756261},
    {0.023829708770758788, 0.32414426165027227, 0.4960985638991457},
    {0.023532604704116965, 0.32990776857516413, 0.48988582950117815},
    {0.02307993829976581, 0.3355813090448093, 0.4838916171846975},
    {0.022979595210601424, 0.3411632670690572, 0.47803935490382515},
    {0.022707635361622264, 0.3466702701862077, 0.4723976815702628},
    {0.022250414084789737, 0.352109624090965, 0.4669584702823865},
    {0.022124477998067993, 0.3574761812445124, 0.46165444016667906},
    {0.021791069977336153, 0.3627869887174417, 0.45654262929804507},
    {0.021776256319502838, 0.3680348834809229, 0.45156215250307846},
    {0.02152936248024083, 0.3732373044965933, 0.44676258718151424},
    {0.02158822978060292, 0.37838527471660577, 0.44208955676541317},
    {0.021388037427047302, 0.38349668536630105, 0.43758535324043873},
    {0.020906126379826747, 0.3885760631680783, 0.4332402802381849},
    {0.02070114401850818, 0.3936120031480679, 0.4290098739833766},
    {0.020774708005350834, 0.3986070226842801, 0.4248945065220385},
    {0.020526909471275397, 0.4035800658477305, 0.4209222889365068},
    {0.01993018897938756, 0.4085347069464005, 0.4170831873583663},
    {0.02020289046425452, 0.4134392235922892, 0.41332502021802126},
    {0.020207032023267582, 0.41834155873012324, 0.40954621812568487},
    {0.020205716119178502, 0.42324888777657366, 0.4055896035873459},
    {0.020829051716263414, 0.42814253234088623, 0.4014443452785458},
    {0.020817169253550774, 0.4330580290829061, 0.3971329152020192},
    {0.02143446555544063, 0.437958395126164, 0.3926385324661419},
    {0.02140951927285399, 0.44287991873506705, 0.38796855906767097},
    {0.021375101041108045, 0.4478037488631326, 0.3831160408679206},
    {0.021978368208421244, 0.45271034115960945, 0.3780894735135355},
    {0.021925941019470395, 0.45763704847590275, 0.372873157602702},
    {0.022518324495068114, 0.4625451606740822, 0.3674890229173533},
    {0.0224439650744986, 0.4674727059721964, 0.3619056954025697},
    {0.02302262600613305, 0.47238034113781563, 0.3561610413615427},
    {0.022921527380028887, 0.4773067430685342, 0.350207729192515},
    {0.023482745772902724, 0.4822119663418752, 0.34409988145088766},
    {0.02334902491772799, 0.48713531181669306, 0.3377738746272117},
    {0.023887996465323004, 0.4920362610259304, 0.33130046883268455},
    {0.023714510501347055, 0.4969547158806737, 0.32459934406811614},
    {0.024225153974701766, 0.5018496124079483, 0.3177583627477745},
    {0.024723266142386315, 0.5067407770624851, 0.31073519425545465},
    {0.02447812509827954, 0.5116485921195848, 0.3034698224412634},
    {0.02493648885622696, 0.5165312197695955, 0.2960768968333586},
    {0.0253764023681478, 0.5214089374064284, 0.28850111378145044},
    {0.025795626075147713, 0.5262813821767296, 0.28074244497417117},
    {0.025421474545393858, 0.5311694950761902, 0.2727216863169472},
    {0.025780309667251394, 0.53603048885089, 0.2645922829614648},
    {0.02610999696637235, 0.5408852158877796, 0.25628033494817076},
    {0.026407437583934296, 0.5457333806719771, 0.2477862504464216},
    {0.02666932808870506, 0.5505747064434545, 0.2391105880601489},
    {0.02689217252655957, 0.5554089359982296, 0.2302540772313253},
    {0.027072298940817408, 0.5602358325123109, 0.22121764273058586},
    {0.027205880727598008, 0.5650551803951163, 0.2120024343570966},
    {0.027288963146258696, 0.5698667861810437, 0.20260986337352},
    {0.028204203304642753, 0.5746482086659772, 0.19318007629151898},
    {0.028188559261174452, 0.5794437480873181, 0.18344751604265283},
    {0.0281103207218881, 0.5842311106814237, 0.17354484850307106},
    {0.028896130485080208, 0.5889876510327495, 0.16364429504261102},
    {0.028695320134244477, 0.5937583183864393, 0.1534244803870241},
    {0.029380675523176917, 0.5984978909930776, 0.14324398649586118},
    {0.029041895792470927, 0.6032517278747106, 0.13273404723048013},
    {0.029612109748158165, 0.6079743365463844, 0.12231515476812047},
    {0.030127173837479853, 0.6126885119214123, 0.1118109733754355},
    {0.029563230689198295, 0.6174173890888045, 0.10096978844729684},
    {0.029944189232757292, 0.6221151405821017, 0.09035027988372307},
    {0.030261789165931138, 0.6268049184840729, 0.07974541336580176},
    {0.030514080739125825, 0.6314869729283614, 0.06921734345076924},
    {0.030699652650745195, 0.6361616116842398, 0.05885329347452114},
    {0.0308177148230897, 0.6408292087724975, 0.04877724313606999},
    {0.03086818674766468, 0.6454902160815501, 0.03915677764354433},
    {0.03197034466002908, 0.6501217563441514, 0.031444380194868886},
    {0.046293996677654235, 0.6544720579771781, 0.031637291074855106},
    {0.058794558570863804, 0.6588271549701028, 0.0317521581310916},
    {0.06503863628200135, 0.6633055777623768, 0.03209873268943441},
    {0.07023149742711651, 0.6678120145394701, 0.03183433002082528},
    {0.0767138128414942, 0.6722807177648771, 0.032132076596637124},
    {0.08432067120858369, 0.6767139870108101, 0.032338563602709626},
    {0.09294277780722546, 0.6811085964562938, 0.032444071987208487},
    {0.10305233592409935, 0.6854378106573311, 0.03307401108271617},
    {0.11331609119127764, 0.6897459817142692, 0.03295480891791886},
    {0.12476614069574826, 0.6939830479703625, 0.033353619722170935},
    {0.13673711440841982, 0.6981699856866227, 0.03362590965526557},
    {0.1491636843492167, 0.7023042408994585, 0.033762275984369046},
    {0.16199129317074465, 0.706383361101949, 0.03375336199700755},
    {0.17553308151438088, 0.7103815097863063, 0.03426353710068628},
    {0.18933948484375165, 0.7143200103636559, 0.03462289684303036},
    {0.20338652270943824, 0.7181967154976012, 0.03482214714504671},
    {0.21765378276012448, 0.7220095660591789, 0.0348520554844732},
    {0.23212369587552023, 0.7257565893785501, 0.03470346402383256},
    {0.24702985279407214, 0.7294127796374531, 0.03507444712661609},
    {0.2620775705935113, 0.7329996606625439, 0.035262260320886224},
    {0.2772591134255436, 0.7365155484918555, 0.03525791814557077},
    {0.2927712616124238, 0.7399360689986639, 0.035780536388425606},
    {0.3083781576598969, 0.7432827601590504, 0.03610724156995657},
    {0.3240763123743059, 0.746554228453955, 0.036229314118799705},
    {0.33986207442019495, 0.7497491625691519, 0.03613819725600252},
    {0.3558907708679886, 0.7528442021403842, 0.036581966518590206},
    {0.3718311425153303, 0.7558826557986548, 0.036046421794990055},
    {0.38813047246050614, 0.7587977057098403, 0.03681458696854892},
    {0.4043371221939297, 0.7616541995826684, 0.03658736257615303},
    {0.42072356025646707, 0.7644081013195043, 0.03690635392025313},
    {0.43714739423907745, 0.7670803847862102, 0.03699298731846738},
    {0.45360699836908186, 0.769670380372701, 0.03683987811183992},
    {0.4702051248974935, 0.7721571183110766, 0.03724796925537228},
    {0.4868227556665157, 0.7745610930844775, 0.0374172194555026},
    {0.503458588002791, 0.7768819000929792, 0.03734086718256391},
    {0.5201979969654729, 0.779099877137358, 0.03784362950139135},
    {0.5368601535092656, 0.7812537776842732, 0.03726439057607977},
    {0.5536883905115004, 0.7832866691953926, 0.0381141962950549},
    {0.570436957512631, 0.7852552273306327, 0.03787015508826886},
    {0.5872524511360537, 0.787122879084239, 0.03822846891183527},
    {0.604057585524296, 0.7889082049024678, 0.03833696358570872},
    {0.6208512632987039, 0.7906113236317394, 0.03819060762615894},
    {0.6376323001265665, 0.7922323958682111, 0.03778469202199975},
    {0.654448292617523, 0.7937561555975436, 0.03800948341735075},
    {0.6712409304097, 0.7951995038068883, 0.03798214896623336},
    {0.6880499867367655, 0.7965485565944844, 0.038610111469974305},
    {0.7047891048851465, 0.7978328175913231, 0.03807498680487568},
    {0.7215350356575515, 0.7990249426739793, 0.03820456984010566},
    {0.7382756146491691, 0.8001275152547487, 0.03902092612392253},
    {0.7549478024082115, 0.8011661514812047, 0.03865607007534288},
    {0.7716059891201252, 0.8021178857956822, 0.038989380941084874},
    {0.7882203132421456, 0.8029955440735611, 0.03908165535934637},
    {0.8047902362352904, 0.8037997989252117, 0.03893065650816213},
    {0.8215523268463777, 0.8044555316790379, 0.03926557049926157},
    {0.8387398480023723, 0.8048817689898747, 0.04067065212316836},
    {0.856341544941598, 0.8050796719349672, 0.04123085284140562},
    {0.8743768083538662, 0.8050267567672601, 0.0419261583900458},
    {0.8928528884736874, 0.8047076768017004, 0.0427644722591047},
    {0.911776412947498, 0.8041062177363495, 0.04375041022498651},
    {0.9311533128855841, 0.8032052518345966, 0.044885135667359986},
    {0.9510015669358057, 0.8019861521093661, 0.04532603899543026},
    {0.962930103481011, 0.8014439331866959, 0.22169249312004036},
    {0.9683410354743099, 0.8024669332735311, 0.3358446858430373},
    {0.9719380800534977, 0.8044065772211566, 0.4114102805053895},
    {0.9746751561332506, 0.8069447701330453, 0.46819081680773716},
    {0.9768136632655356, 0.8099358552525316, 0.5140546510678895},
    {0.978649037790159, 0.8132574705132256, 0.5522569503355605},
    {0.9802036704451057, 0.8168594951941832, 0.5851451968636262},
    {0.981554091047551, 0.8206924722262127, 0.6139795814385243},
    {0.9827732625825781, 0.8247142837530934, 0.6395746026568098},
    {0.9839157453834834, 0.8288914274285599, 0.6625186328985215},
    {0.9848784092487323, 0.8332318824714189, 0.6834911738162105},
    {0.9858320115191356, 0.8376808147738173, 0.7025706724123436},
    {0.9866558657916994, 0.8422559436774792, 0.7202399867839039},
    {0.9873677995679001, 0.8469433249862409, 0.7366844886107697},
    {0.9881805379141889, 0.8516804941618812, 0.7518089864773918},
    {0.9887611047789485, 0.8565442781873747, 0.7661909960366566},
    {0.9894043443414985, 0.8614528824718757, 0.7795879261784779},
    {0.9900758649057856, 0.8664087680409848, 0.7921604870664645},
    {0.99059092673338, 0.8714545088111433, 0.8041942791071222},
    {0.9911277189421445, 0.8765391108526217, 0.8155756474332213},
    {0.9916061462381213, 0.8816790723671002, 0.8264519900822266},
    {0.9922158985563451, 0.8868207617131697, 0.836709699896381},
    {0.9925679440918894, 0.8920625188742286, 0.8467473851492474},
    {0.9931424279738416, 0.8972751802406753, 0.8561933859696547},
    {0.9935217736048007, 0.9025652348682511, 0.8654402820837326},
    {0.9938879084327678, 0.9078821573987383, 0.874370338308405},
    {0.99427342946785, 0.9132148454645382, 0.8829927351962013},
    {0.9947111506654817, 0.918552416162813, 0.8913160852881722},
    {0.995056938233056, 0.9239302650640105, 0.8994722486725081},
    {0.995512287532556, 0.9292939740928252, 0.9073439419102849},
    {0.9959250022023757, 0.9346812618630085, 0.9150611281410694},
    {0.9961274042213023, 0.9401333891345969, 0.9227518397011023},
    {0.9965143545143408, 0.9455470167979051, 0.9301752047442942},
    {0.996922153842586, 0.9509629844614244, 0.9374587708548032},
    {0.9971731694726661, 0.9564250004713418, 0.9447286698401807},
    {0.9976825719783714, 0.9618265695993463, 0.9517442599956717},
    {0.9978661370132459, 0.967313947090583, 0.958874650466398},
    {0.9983489942212443, 0.972728972574113, 0.9657569772496776},
    {0.9985297770627254, 0.9782208004653724, 0.9727591013871136},
    {0.9988366714262512, 0.9836818403108719, 0.9796396969161855},
    {0.999287652981863, 0.989107488582374, 0.9864006969015998},
    {0.9994697671436109, 0.9945977204425903, 0.9932874563643866},
    {1.0, 1.0, 1.0},
}};

template <typename Real>
static constexpr std::array<std::array<Real, 3>, 256> inferno_data_ = {{
    {0.0014619955811715805, 0.0004659913919114934, 0.013866005775115809},
    {0.0022669056023600243, 0.001269897101615975, 0.018569490325902337},
    {0.003299036110031063, 0.0022490183451722313, 0.024239243465136288},
    {0.004546896852350439, 0.0033918841632656804, 0.03090851258977682},
    {0.006006105056993791, 0.004692061241092744, 0.038558624443389096},
    {0.007675918804434457, 0.006135891503856028, 0.046835705273408614},
    {0.009561203394731096, 0.0077131160510677, 0.05514393874236257},
    {0.011662968995142865, 0.0094169153553472, 0.06345998081431048},
    {0.013994707785115502, 0.011224701548363442, 0.07186110007643282},
    {0.01656105637139426, 0.01313595377716254, 0.08028228905271992},
    {0.019372742788596516, 0.01513271743178829, 0.08876637381708319},
    {0.02244719411002579, 0.017198996785303594, 0.09732759285010725},
    {0.02579282311826555, 0.01933074318511054, 0.10592963918696552},
    {0.029432387489373633, 0.02150303695278668, 0.11462190156591848},
    {0.03338491092042817, 0.023701776954208033, 0.12339694246694158},
    {0.03766747608151851, 0.025920506649278047, 0.13223096637532075},
    {0.042253067017337005, 0.028138807463796305, 0.1411412582171954},
    {0.04691458399133113, 0.030323540520811973, 0.15016326468244964},
    {0.05164425209311529, 0.03247382729611796, 0.15925458396315},
    {0.05644871419547314, 0.03456857075460381, 0.16841357852522199},
    {0.06134044494312783, 0.03658982977294056, 0.17764292167289156},
    {0.06633085137688166, 0.038503594004715896, 0.18696190141292804},
    {0.07142826458096291, 0.04029339894981024, 0.19635286429314014},
    {0.07663702254956387, 0.04190461115902603, 0.2057992341013456},
    {0.08196142528164385, 0.043327451450508585, 0.2152881892106146},
    {0.087411214696131, 0.04455560411873583, 0.22481357132253754},
    {0.09298959953890884, 0.04558246689176038, 0.23435752319694075},
    {0.0987024276165075, 0.0464015612109181, 0.24390490595057124},
    {0.1045507907963224, 0.04700744862572622, 0.2534298623464375},
    {0.11053667232221573, 0.04739845362551404, 0.26291322780905574},
    {0.11665600122383982, 0.04757339371302694, 0.27232119603015154},
    {0.12290731198945251, 0.04753536748517837, 0.2816231724086292},
    {0.12928523040832837, 0.04729230240096718, 0.2907885083783503},
    {0.13577751583663375, 0.046855310502846824, 0.29977552118778034},
    {0.14237847430795506, 0.04624117675574093, 0.30855378680836465},
    {0.1490727384829781, 0.04546721023082908, 0.3170848511067041},
    {0.1558507266503708, 0.044558026492337664, 0.32533901695859424},
    {0.1626889760379725, 0.0435530737190721, 0.3332771435961689},
    {0.16957421944883505, 0.042488139902889556, 0.3408733080618529},
    {0.17649322158660016, 0.04140091975980204, 0.3481113785987225},
    {0.18342846019226636, 0.04032795978956832, 0.35497062509269495},
    {0.19036746731312684, 0.03930776663057157, 0.36144755914871207},
    {0.19729670678394517, 0.03839881405885271, 0.3675348910966813},
    {0.20420970925948218, 0.03763066869776516, 0.3732386865741054},
    {0.21109495319112997, 0.0370286796745015, 0.3785631007654439},
    {0.21794820252834113, 0.036613674395490264, 0.38352155568816976},
    {0.22476319518924603, 0.03640357374688182, 0.38812924842856483},
    {0.2315374529310898, 0.03640351817286241, 0.39239977934534576},
    {0.2382734320696066, 0.03661949264330557, 0.3963533531157254},
    {0.2449666980359304, 0.03705338654185691, 0.4000069520159531},
    {0.2516206637792285, 0.037703436097523746, 0.4033784221656792},
    {0.2582339375612672, 0.038569281262784215, 0.4064850812186898},
    {0.2648092188471272, 0.039645098963587894, 0.4093447670552162},
    {0.27134717156688476, 0.040920248896872104, 0.41197616680691795},
    {0.27784945989435444, 0.04235106621674233, 0.41439190152253697},
    {0.2843214014200332, 0.043931272334442044, 0.4166082251959268},
    {0.2907626953161, 0.04564206683863344, 0.41863700244674007},
    {0.2971786279866803, 0.04746832221255668, 0.42049126153840694},
    {0.3035679263767977, 0.04939409900213965, 0.42218207588627776},
    {0.3099342310464275, 0.05140486616668636, 0.4237209077370659},
    {0.3162821544923382, 0.0534881537223529, 0.42511612281358585},
    {0.3226094622720638, 0.05563191003781664, 0.42637698442915734},
    {0.32892138033063256, 0.057825220219243655, 0.4275111514018109},
    {0.33521669037969953, 0.060057968826412046, 0.42852404020939816},
    {0.34150060454122194, 0.06232329248965957, 0.42942516526482805},
    {0.34777091610802874, 0.06461403589398387, 0.43021707827304073},
    {0.35403123226470046, 0.06692277585963113, 0.43090600382653765},
    {0.3602841408353374, 0.06924510666507289, 0.4314970989990494},
    {0.366528457743285, 0.07157684449494817, 0.4319940445656597},
    {0.3727683648145053, 0.0739131776252345, 0.4324001060667602},
    {0.37900068183696356, 0.07625091481761018, 0.432719071099916},
    {0.3852285882170877, 0.07858924726823341, 0.43295510186497294},
    {0.391452904966983, 0.08092498458487726, 0.4331090849213901},
    {0.3976732251941389, 0.08325472094735673, 0.43318307823977154},
    {0.4038941281371945, 0.08557805215845263, 0.4331790858011016},
    {0.4101124478324236, 0.08789378949993748, 0.43309809489281975},
    {0.41633135152767126, 0.09020111672608602, 0.4329430750225925},
    {0.4225486702204431, 0.09249885580078815, 0.43271410026220614},
    {0.42876857492222104, 0.09478817843783055, 0.43241205420979634},
    {0.43498689244861954, 0.09706691940470738, 0.43203909514551403},
    {0.4412077984093083, 0.09933623724389631, 0.4315940236596366},
    {0.447428115165217, 0.10159497990744795, 0.43108007898556605},
    {0.453650434334324, 0.10384572179396404, 0.4304981431533577},
    {0.4598753385411611, 0.10608703735300581, 0.4298460517373555},
    {0.4660996564728312, 0.10831978115925261, 0.42912513104437605},
    {0.47232856222023284, 0.11054509253877559, 0.428334014815688},
    {0.47855787863379246, 0.11276183809637635, 0.42747510913916503},
    {0.4847897859200844, 0.11497214601360681, 0.42654796832755787},
    {0.49102210135850827, 0.11717689277568918, 0.4255520769077051},
    {0.4972564191293403, 0.11937663835804257, 0.42448819407856125},
    {0.5034933248743276, 0.12157294554931822, 0.42335603351162365},
    {0.5097296414881407, 0.12376669200308499, 0.4221561654704694},
    {0.5159675484473075, 0.12595799741934913, 0.42088698041253036},
    {0.522205863946251, 0.1281477445657914, 0.4195491273628126},
    {0.5284447718500154, 0.1303390492990618, 0.41814191774726495},
    {0.5346830866012935, 0.1325317964071948, 0.4166670792419288},
    {0.5409194040751939, 0.13472654160063324, 0.4151232492186141},
    {0.5471573097689081, 0.1369268480746088, 0.41351101955274133},
    {0.5533916265944651, 0.13913159264204145, 0.4118292045095448},
    {0.5596245324931008, 0.14134390081316123, 0.41007795018698034},
    {0.565853849035322, 0.14356464417843176, 0.408258150100951},
    {0.5720817545692755, 0.14579495531212058, 0.40636887114816594},
    {0.5783040710826255, 0.1480366969821801, 0.4044110859921461},
    {0.5845203906718561, 0.15029143569692027, 0.4023853090317509},
    {0.5907342930011564, 0.15256075186424045, 0.40029001001388526},
    {0.5969396129909775, 0.1548454880249755, 0.3981252481044335},
    {0.6031395137899174, 0.15714880993418692, 0.39589092443931145},
    {0.6093298346523712, 0.15947154274866268, 0.3935891773266438},
    {0.6155137330265519, 0.1618148719991605, 0.3912188293792153},
    {0.6216850552354126, 0.1641816008624372, 0.38878109691187235},
    {0.6278463807173074, 0.1665723257656469, 0.3862763730397491},
    {0.6339982745553374, 0.16898966363736537, 0.3837040052607365},
    {0.6401346020827624, 0.17143538357408133, 0.3810652955393134},
    {0.6462604918955885, 0.17391173185514677, 0.3783589039476966},
    {0.6523688217818822, 0.17641844614159138, 0.37558620861598563},
    {0.6584637067856558, 0.17895980652738866, 0.37274779377320805},
    {0.6645400395062868, 0.1815365142239986, 0.36984611221667457},
    {0.6705983762227489, 0.18415021667573447, 0.36687943961906233},
    {0.6766382547106133, 0.18680458958736995, 0.3638490055304848},
    {0.6826555952415246, 0.1894982847225483, 0.3607573457624624},
    {0.6886534667732338, 0.1922366729397742, 0.3576028897156723},
    {0.6946268114822389, 0.19501835963153946, 0.35438824300823807},
    {0.7005766750861048, 0.19784876490425912, 0.35111276552861453},
    {0.7065000246724039, 0.20072544237887552, 0.34777713155969964},
    {0.7123953793054851, 0.20365311312098916, 0.34438350640978904},
    {0.7182642339188836, 0.20663353472537765, 0.34093101094395056},
    {0.7241025940115757, 0.20966719575096354, 0.3374243970752363},
    {0.7299094384556731, 0.21275663751205837, 0.33386088208499215},
    {0.7356828051012506, 0.21590328739196812, 0.3302452795512392},
    {0.7414236378808644, 0.2191097508283441, 0.32657574613398255},
    {0.7471270116731434, 0.2223753888240187, 0.32285615422841446},
    {0.7527948315414559, 0.22570387532426378, 0.31908460363536006},
    {0.7584222127903781, 0.22909450162198797, 0.31526602130047027},
    {0.7640096003184447, 0.23255111985362076, 0.3113994474823908},
    {0.769556407505972, 0.236074626914262, 0.30748488142933444},
    {0.7750588038588068, 0.23966423164432504, 0.3035263167932573},
    {0.780517595641067, 0.24332476411029125, 0.29952273568573573},
    {0.7859290015238034, 0.2470533545534328, 0.29547717944438406},
    {0.7912937764432374, 0.25085391354890735, 0.29138958485956357},
    {0.7966071922843984, 0.25472548987486643, 0.28726403596779526},
    {0.8018706152376446, 0.2586710580118296, 0.28309949569184917},
    {0.8070823747046473, 0.26268963921769606, 0.27889788734079896},
    {0.8122388089759692, 0.26678319189183297, 0.2746613536242391},
    {0.8173415491080798, 0.27095180082645115, 0.27038973420543944},
    {0.8223859952913317, 0.27519433772522317, 0.2660852062616442},
    {0.8273727148541918, 0.2795149750247453, 0.26174957724469494},
    {0.8322991733910813, 0.2839104959497832, 0.25738305415913015},
    {0.8371646398804131, 0.2883820094161365, 0.252988540402642},
    {0.8419693412456319, 0.29293066888944563, 0.2485638989415601},
    {0.846708821213788, 0.29755616578042293, 0.2441133893753179},
    {0.8513844997023888, 0.3022578538064719, 0.23963574040066615},
    {0.855991993845198, 0.3070353341236893, 0.23513323409260284},
    {0.8605336484626919, 0.31189005054876767, 0.230605578997568},
    {0.865006157141607, 0.316819514079576, 0.2260550749407627},
    {0.8694086743123614, 0.3218239712798184, 0.2214825826370368},
    {0.873741308408031, 0.32690370860531126, 0.21688591478628577},
    {0.8780008409852356, 0.33205714927467816, 0.212268424608822},
    {0.8821884493306369, 0.33728491405873967, 0.20762775209164724},
    {0.8863019976673894, 0.3425833384271254, 0.20296826303399285},
    {0.8903415794934312, 0.3479551300484793, 0.19828558691030201},
    {0.894305144045262, 0.3533965381592675, 0.19358409819949085},
    {0.8981917181039129, 0.35890794053186803, 0.1888606247963219},
    {0.902003277546344, 0.36448975074745, 0.18411593340897883},
    {0.9057348680145032, 0.37013713800749826, 0.17935046113406727},
    {0.9093903996647411, 0.3758539732830311, 0.17456276670706064},
    {0.9129660070101344, 0.38163334525219944, 0.1697552941505453},
    {0.9164625104530708, 0.3874792047452722, 0.16492359727200515},
    {0.9198791350013626, 0.39338656173222386, 0.16007012382758104},
    {0.9232147698912994, 0.3993559137889013, 0.1551936703489871},
    {0.9264702498113229, 0.40538678902429115, 0.15029195385198924},
    {0.9296439014941755, 0.41147612778815296, 0.14536750158970949},
    {0.9327373523126056, 0.4176250250322833, 0.14041678247519898},
    {0.9357470217364718, 0.42382835018615, 0.13544032915276746},
    {0.9386754432346461, 0.4300892683482416, 0.13043760827101122},
    {0.9415211303729047, 0.4364025802485547, 0.12540915246812306},
    {0.9442848284617543, 0.4427688880476121, 0.12035472282994122},
    {0.9469652258626157, 0.44918881902600133, 0.11527197615350762},
    {0.9495619408937822, 0.4556571156219838, 0.11016454791032279},
    {0.9520753085089858, 0.4621760654003351, 0.10503080049999589},
    {0.9545060414475516, 0.468741350075834, 0.09987436834002841},
    {0.9568523791775677, 0.4753543175690524, 0.09469462335752518},
    {0.9591141303022052, 0.4820115908490303, 0.08949918430671361},
    {0.9612934380573407, 0.4887145750891598, 0.08428844691950368},
    {0.9633872061002898, 0.4954598384326913, 0.07907300229562086},
    {0.9653969856405892, 0.5022460991811288, 0.07385959127645336},
    {0.9673222685246364, 0.5090760925737746, 0.06865882956308128},
    {0.9691630662078553, 0.5159433430954236, 0.06348840124042007},
    {0.970919318954511, 0.5228513513717987, 0.05836666742473027},
    {0.9725901348141913, 0.529795592113789, 0.053324210911405455},
    {0.9741763574084353, 0.5367786143884726, 0.04839153015610343},
    {0.975677190645675, 0.5437958464199314, 0.043618036090846},
    {0.9770920356520277, 0.5508470769374313, 0.03905055303111867},
    {0.9784222326450961, 0.5579351064433469, 0.034930916546532825},
    {0.979666095876416, 0.5650543281731687, 0.0314092990272324},
    {0.9808242624402922, 0.5722073701062071, 0.028507883553058225},
    {0.9818951440117506, 0.5793895836320366, 0.02625012573179964},
    {0.982881280054488, 0.5866046371076197, 0.024660950197297846},
    {0.9837791795198422, 0.5938468430754209, 0.023770046474572596},
    {0.9845910909683125, 0.6011190480992957, 0.023606059102018802},
    {0.9853152007413414, 0.608420107478453, 0.02420209240341553},
    {0.9859521306674056, 0.6157473052058917, 0.025591934272695092},
    {0.9865022096405148, 0.6231033747802108, 0.027814257478890377},
    {0.9869641579064014, 0.6304825655882533, 0.030907915000313535},
    {0.9873372060008064, 0.6378886447799842, 0.034916559619452406},
    {0.987622172670732, 0.6453178289458518, 0.039886017500422775},
    {0.9878191520033638, 0.6527700129448509, 0.045580403012790684},
    {0.9879261727372323, 0.6602480965501968, 0.05175024605159847},
    {0.9879451706317489, 0.6677452744424257, 0.05832852598609512},
    {0.9878741601998459, 0.675265366380739, 0.06525752407999996},
    {0.9877141765851464, 0.6828045385569567, 0.07248872059669512},
    {0.9874641349558363, 0.690364638021527, 0.07999083430127826},
    {0.9871241697382241, 0.6979418049868507, 0.08773096280857864},
    {0.9866942187047217, 0.7055369718171162, 0.09569310480416554},
    {0.9861751485274827, 0.7131510743985874, 0.103863256330502},
    {0.9855662155788282, 0.7207792367356154, 0.11222834433058854},
    {0.9848651143109276, 0.7284253454782301, 0.12078557949263882},
    {0.9840751996931422, 0.7360845034258754, 0.12952661474987895},
    {0.9831960676449997, 0.7437566173190143, 0.13845392837069237},
    {0.9822281707577935, 0.7514397717174452, 0.14756491297187693},
    {0.9811732892122574, 0.7591319265002779, 0.15686190499859645},
    {0.9800321283245658, 0.7668350414250283, 0.16635325033390524},
    {0.9788062634501479, 0.7745421938654863, 0.1760361942373471},
    {0.9774970739304246, 0.7822563112740953, 0.18592362086777084},
    {0.976108224906952, 0.7899714619636461, 0.1960175103988243},
    {0.9746380082272822, 0.7976905802513132, 0.20633302377351131},
    {0.9730881744707035, 0.8054067300567858, 0.21687685576253862},
    {0.9714683557085738, 0.8131188812721618, 0.2276566805995705},
    {0.9697831130812914, 0.8208229971469754, 0.23868624376433029},
    {0.968041305354793, 0.8285121501431812, 0.24997101177190928},
    {0.9662430438023951, 0.836189261303023, 0.2615346777715621},
    {0.9643942451078572, 0.8438454171701422, 0.273390375539181},
    {0.9625169727369577, 0.8514745194008927, 0.2855471549893648},
    {0.9606261754860036, 0.8590666816140542, 0.29800977930778266},
    {0.9587199044763673, 0.86662276892152, 0.310821685193729},
    {0.9568341051627561, 0.8741269393072385, 0.3239742355406551},
    {0.9549973084887696, 0.8815661177531517, 0.3374737603502378},
    {0.9532150458875698, 0.8889401848892272, 0.3513697543417886},
    {0.9515462272506953, 0.8962233797142661, 0.36562618964558136},
    {0.950018012245954, 0.9034074125145412, 0.3802723264284489},
    {0.9486831530694507, 0.9104706308186431, 0.3952886725437556},
    {0.9475940275601823, 0.9173976138382822, 0.4106669431858655},
    {0.9468091087983127, 0.9241658620670481, 0.4263732130148635},
    {0.9463921563722028, 0.9307581300310862, 0.4423654634949841},
    {0.9464031311450462, 0.9371570608055302, 0.458592798889426},
    {0.9469030741916794, 0.9433453721925603, 0.4749690020361225},
    {0.9479372327167105, 0.9493162253885609, 0.49142738476056075},
    {0.9495450508917115, 0.9550605846153878, 0.5078595779865485},
    {0.9517404201103928, 0.9605853570748163, 0.5242049355302347},
    {0.9545291096426641, 0.9658937621848641, 0.5403611560472134},
    {0.9578957403425837, 0.971000188760133, 0.5562734123813695},
    {0.9618122666687167, 0.9759219036314865, 0.571925695133198},
    {0.966248777481722, 0.980675368253291, 0.5872050182876477},
    {0.9711624870815522, 0.985280023457573, 0.6021551859229256},
    {0.9765108962599102, 0.9897505184610552, 0.6167595818025277},
    {0.9822577545753769, 0.9941071290312428, 0.6310186261682112},
    {0.9883620799212208, 0.9983616470620554, 0.6449240982803861},
}};

template <typename Real>
static constexpr std::array<std::array<Real, 3>, 256> black_body_data_ = {{
    {0.0, 0.0, 0.0},
    {0.013038855104993618, 0.0037537033758315535, 0.002103027943341456},
    {0.02607771020998725, 0.007507406751663157, 0.004206055886683011},
    {0.03911656531498074, 0.01126111012749475, 0.006309083830024466},
    {0.05111281846889674, 0.0150148135033264, 0.008412111773366015},
    {0.06145201887409168, 0.018768516879157954, 0.010515139716707521},
    {0.07064326337459148, 0.02252222025498951, 0.012618167660048977},
    {0.07897806577026972, 0.026275923630821117, 0.014721195603390481},
    {0.08664361913059343, 0.03002962700665271, 0.016824223546731985},
    {0.09376835672821318, 0.03378333038248431, 0.018927251490073488},
    {0.10044479589169861, 0.03753703375831587, 0.021030279433414945},
    {0.10674212293169041, 0.041271634201045536, 0.02313330737675645},
    {0.11271363536038692, 0.04482597891806931, 0.025236335320097954},
    {0.11840139369546077, 0.048211428422088996, 0.027339363263439456},
    {0.12383926177477003, 0.051448139039433315, 0.029442391206781018},
    {0.12905496821437284, 0.05455261530914037, 0.03154541915012247},
    {0.13415881215680708, 0.05748457013282512, 0.03365241563251253},
    {0.1394282635051205, 0.06009480851170646, 0.035776007200922486},
    {0.1448680365076732, 0.06239658846641056, 0.03791741526748158},
    {0.15046463427844672, 0.06440943575351296, 0.04007701382092898},
    {0.15620567729622375, 0.06614841603004337, 0.042202264001466754},
    {0.1620642871793741, 0.06766725618508368, 0.04427088573012868},
    {0.1679681168976293, 0.06914956019046758, 0.04626676146380417},
    {0.17390879488805058, 0.07061411563078038, 0.048192562428916404},
    {0.17988507705788212, 0.07206107502164819, 0.05005305250836759},
    {0.18589582390636886, 0.073490579693259, 0.0518524382574397},
    {0.19193998883434982, 0.07490276049381833, 0.053594454613670746},
    {0.19801660795994802, 0.07629773842679596, 0.05528243440605366},
    {0.20412479122741767, 0.0776756252290656, 0.05691936524751085},
    {0.21026371462828286, 0.07903652389614257, 0.05850793649497276},
    {0.2164326133809747, 0.08038052915995664, 0.060050578312412416},
    {0.2226307759380115, 0.08170772792393452, 0.06154949439758999},
    {0.22885753870905418, 0.08301819965959711, 0.0630066895818179},
    {0.23511228140445872, 0.08431201676837646, 0.06442399324878009},
    {0.2413944229177235, 0.08558924491193284, 0.06580307931909093},
    {0.24770341767689402, 0.08684994331387272, 0.06714548339482998},
    {0.25403875240487506, 0.08809416503544282, 0.06845261754062515},
    {0.2603999432369943, 0.089321957227491, 0.06972578308625077},
    {0.26678653315130063, 0.09053336136073056, 0.07096618176381639},
    {0.2731980896731617, 0.09172841343612223, 0.07217492543577086},
    {0.2796342028209079, 0.09290714417700069, 0.07335304462466913},
    {0.2860944832637109, 0.09406957920438949, 0.07450149601935155},
    {0.29257856066667054, 0.09521573919680446, 0.07562116910289712},
    {0.29908608220134825, 0.09634564003570609, 0.07671289202394302},
    {0.3056167112027767, 0.09745929293764075, 0.07777743681356081},
    {0.31217012595638344, 0.0985567045740047, 0.07881552403396175},
    {0.31874601860034424, 0.09963787717926836, 0.0798278269321728},
    {0.32534319363742265, 0.10070307071710113, 0.08082304674985791},
    {0.3319596041219896, 0.1017528114823801, 0.08181815186765931},
    {0.3385951481217959, 0.10278707282194993, 0.08281314988265284},
    {0.34524972170116486, 0.10380582517735878, 0.08380805027785368},
    {0.35192321947598326, 0.10480903601879557, 0.08480286230027789},
    {0.35861553489350906, 0.10579666984465097, 0.08579759496848993},
    {0.36532656048179657, 0.10676868817762619, 0.08679225707984467},
    {0.3720561880720667, 0.10772504955741516, 0.08778685721743867},
    {0.3788043089969579, 0.10866570952997373, 0.08878140375678456},
    {0.38557081426725, 0.10959062063337968, 0.08977590487222206},
    {0.392355594729354, 0.11049973238028288, 0.09077036854307868},
    {0.3991585412055988, 0.1113929912369285, 0.09176480255959146},
    {0.40597954461911684, 0.11227034059873503, 0.09275921452860061},
    {0.41281849610493104, 0.1131317207623947, 0.09375361187902712},
    {0.4196752871086668, 0.11397706889445536, 0.09474800186714263},
    {0.4265498094741573, 0.11480631899633567, 0.0957423915816421},
    {0.4334419555210749, 0.11561940186571482, 0.09673678794852814},
    {0.44035161811359996, 0.11641624505422682, 0.09773119773581507},
    {0.44727869072102816, 0.11719677282138305, 0.09872562755806155},
    {0.45422306747112823, 0.11796090608463114, 0.09972008388073797},
    {0.46118464319697156, 0.11870856236545602, 0.10071457302443804},
    {0.4681633134778857, 0.11943965573140711, 0.1017091011689388},
    {0.47515897467511364, 0.12015409673393382, 0.10270367435711814},
    {0.4821715239627041, 0.1208517923418958, 0.10369829849873419},
    {0.4892008593541031, 0.12153264587059787, 0.1046929793740726},
    {0.49624687972487286, 0.12219655690619557, 0.10568772263746884},
    {0.5033094848319175, 0.12284342122529493, 0.10668253382070847},
    {0.5103885753295632, 0.12347313070955779, 0.10767741833631286},
    {0.5174840527828012, 0.12408557325511224, 0.10867238148071282},
    {0.5245958196779755, 0.12468063267654717, 0.10966742843731633},
    {0.5317237794311703, 0.12525818860524965, 0.11066256427947438},
    {0.5388678363945218, 0.12581811638183757, 0.1116577939733483},
    {0.5460278958606667, 0.12636028694240006, 0.11265312238068345},
    {0.5532038640655134, 0.12688456669826084, 0.11364855426149251},
    {0.5603956481895038, 0.12739081740893254, 0.11464409427665198},
    {0.5676031563575221, 0.12787889604792488, 0.11563974699041571},
    {0.5748262976375883, 0.12834865466103332, 0.11663551687284723},
    {0.5820649820384645, 0.12879994021670815, 0.11763140830217725},
    {0.5893191205062877, 0.1292325944480741, 0.11862742556708483},
    {0.5965886249203336, 0.12964645368613567, 0.11962357286890879},
    {0.6038734080880022, 0.13004134868367784, 0.12061985432379002},
    {0.6111733837391209, 0.13041710442930957, 0.12161627396474861},
    {0.618488466519628, 0.1307735399510897, 0.12261283574369633},
    {0.6258185719847208, 0.13111046810909502, 0.12360954353338896},
    {0.6331636165915261, 0.13142769537626633, 0.12460640112931987},
    {0.6405235176913502, 0.13172502160680363, 0.125603412251556},
    {0.6478981935215649, 0.13200223979132508, 0.12660058054652026},
    {0.6552875631971755, 0.13225913579794255, 0.1275979095887203},
    {0.6626915467021134, 0.13249548809833553, 0.1285954028824274},
    {0.6701100648802979, 0.13271106747782782, 0.12959306386330607},
    {0.6775430394264935, 0.13290563672840347, 0.13059089589999584},
    {0.6849903928770076, 0.133078950323475, 0.13158890229564854},
    {0.6924520486002461, 0.13323075407316545, 0.13258708628942104},
    {0.6990269444289148, 0.13526285713010114, 0.13312878204163103},
    {0.7029385031056067, 0.14276164252353424, 0.13230087174729266},
    {0.706849160518104, 0.15003367886528432, 0.13144472110159658},
    {0.7107589339135043, 0.1571072636984418, 0.13055959687730143},
    {0.7146678402982333, 0.16400574341929422, 0.12964472942353197},
    {0.7185758964424839, 0.17074862440458652, 0.1286993099989257},
    {0.7224831188845541, 0.1773523851612915, 0.12772248784920406},
    {0.7263895239350789, 0.18383108192179715, 0.12671336699848523},
    {0.7302951276811657, 0.19019680820853718, 0.12567100271919956},
    {0.7341999459904341, 0.19646004903982417, 0.12459439764026739},
    {0.7381039945149617, 0.20262995773846548, 0.12348249744706852},
    {0.742007288695138, 0.20871457496055673, 0.12233418611951721},
    {0.7459098437634323, 0.21472100396010463, 0.1211482806460144},
    {0.7498116747480748, 0.22065555226824157, 0.11992352514088764},
    {0.7537127964766552, 0.22652384728976455, 0.11865858428082418},
    {0.7576132235796376, 0.23233093142236372, 0.11735203596128549},
    {0.7615129704938024, 0.2380813409383808, 0.11600236305643169},
    {0.7654120514656042, 0.24377917187260056, 0.11460794414495701},
    {0.7693104805544623, 0.24942813542336495, 0.11316704303859387},
    {0.7732082716359728, 0.25503160482399895, 0.11167779691868823},
    {0.7771054384050546, 0.260592655225706, 0.11013820284777173},
    {0.7810019943790253, 0.266114097815743, 0.1085461023755086},
    {0.784897952900607, 0.27159850915024936, 0.10689916389931109},
    {0.7887933271408747, 0.2770482564911545, 0.10519486236601067},
    {0.792688130102133, 0.28246551978783785, 0.10343045580787474},
    {0.7965823746207359, 0.28785231082677337, 0.1016029580881492},
    {0.8004760733698467, 0.2932104899790604, 0.09970910708026909},
    {0.8043692388621376, 0.2985417809010483, 0.09774532731012447},
    {0.8082618834524321, 0.30384778348310415, 0.09570768583739653},
    {0.8121540193402956, 0.30912998529284585, 0.09359183981917277},
    {0.8160456585725648, 0.31438977171944593, 0.09139297375735872},
    {0.8199368130458352, 0.31962843499310634, 0.08910572383870971},
    {0.8238274945088833, 0.32484718222701414, 0.0867240859712489},
    {0.8277177145650534, 0.33004714260695417, 0.08424130301309385},
    {0.8316074846745842, 0.33522937383533186, 0.08164972514342614},
    {0.8354968161568933, 0.3403948679210186, 0.07894063513317165},
    {0.8393857201928158, 0.3455445563935293, 0.07610402711123557},
    {0.8432742078267932, 0.350679315009233, 0.07312832277504416},
    {0.8471622899690239, 0.35579996800812036, 0.07000000201860399},
    {0.8510499773975676, 0.3609072919719153, 0.06670311423184544},
    {0.8549372807604048, 0.3660020193276998, 0.06321861961083997},
    {0.8588242105774635, 0.37108484153559657, 0.05952348231757188},
    {0.8627107772425975, 0.37615641199422883, 0.05558939105319657},
    {0.8665969910255304, 0.38121734869353224, 0.0513809015925008},
    {0.870482862073761, 0.3862682366409203, 0.046852647240651586},
    {0.8743684004144292, 0.39130963008373454, 0.04194497518373403},
    {0.8782536159561499, 0.39634205454821875, 0.03662194288964867},
    {0.8821385184908072, 0.40136600871296174, 0.03117410153953512},
    {0.8860231176953159, 0.40638196613271077, 0.025643215104659372},
    {0.8899074231333507, 0.4113903768267081, 0.020028781568071516},
    {0.891065382338744, 0.4184635684621956, 0.021466471088552607},
    {0.8919871230973406, 0.4256470552949243, 0.02354603033215587},
    {0.8928908994946323, 0.4327756220188152, 0.02570095483314324},
    {0.8937766259341077, 0.4398522418561916, 0.027932126084511257},
    {0.8946442158848376, 0.4468796753940895, 0.030240425579256377},
    {0.8954935818622007, 0.4538604904909103, 0.032626734810374966},
    {0.8963246354082741, 0.4607970798827202, 0.03509193527086419},
    {0.8971372870718866, 0.4676916768037213, 0.037636908453718976},
    {0.8979314463883195, 0.4745463688860316, 0.04026253585193785},
    {0.8987070218586405, 0.48136311056322806, 0.042883774659142816},
    {0.899463920928664, 0.4881437341684682, 0.045483032991513316},
    {0.9002020499675142, 0.49488995989000545, 0.048065003458762835},
    {0.9009213142457936, 0.5016034047235632, 0.05063130309097131},
    {0.9016216179133228, 0.5082855905414284, 0.053183371551623206},
    {0.9023028639764596, 0.5149379513816515, 0.055722495847998925},
    {0.902964954274961, 0.5215618400467861, 0.05824983088146562},
    {0.9036077894583927, 0.5281585340897971, 0.060766416652818796},
    {0.9042312689620561, 0.5347292412547051, 0.06327319275680962},
    {0.9048352909824239, 0.5412751044309391, 0.06577101066369245},
    {0.9054197524520664, 0.5477972061730271, 0.0682606441818552},
    {0.9059845490140521, 0.5542965728309169, 0.07074279841591524},
    {0.9065295749958051, 0.5607741783307898, 0.07321811747290291},
    {0.9070547233823978, 0.5672309476415033, 0.07568719112093336},
    {0.9075598857892664, 0.5736677599577373, 0.07815056056678477},
    {0.9080449524343264, 0.5800854516273583, 0.08060872348871567},
    {0.9085098121094662, 0.5864848188474415, 0.083062138436825},
    {0.9089543521514049, 0.5928666201506813, 0.08551122869395478},
    {0.9093784584118849, 0.5992315787015761, 0.08795638567456415},
    {0.909782015227184, 0.6055803844196885, 0.09039797192630794},
    {0.9101649053869216, 0.6119136959454732, 0.09283632378872866},
    {0.9105270101021349, 0.6182321424625531, 0.09527175375494526},
    {0.9108682089726016, 0.6245363253889187, 0.09770455257523358},
    {0.9111883799533859, 0.630826819948256, 0.10013499113555632},
    {0.9114873993205763, 0.6371041766315243, 0.10256332213928446},
    {0.9117651416361977, 0.6433689225578914, 0.10498978161628147},
    {0.9120214797122554, 0.649621562743278, 0.10741459028014497},
    {0.912256284573901, 0.6558625812839713, 0.1098379547515301},
    {0.91246942542167, 0.6620924424620748, 0.11226006866305718},
    {0.9126607695927773, 0.6683115917789337, 0.11468111365925662},
    {0.9128301825214254, 0.6745204569221216, 0.11710126030325854},
    {0.9129775276981008, 0.6807194486710708, 0.11952066890043198},
    {0.9131026666278179, 0.6869089617459752, 0.12193949024790668},
    {0.9132054587872768, 0.6930893756041974, 0.12435786631781182},
    {0.9132857615808976, 0.6992610551880414, 0.1267759308811086},
    {0.9133434302956924, 0.7054243516274225, 0.12919381007809072},
    {0.9133783180549305, 0.7115796029006738, 0.13161162294090045},
    {0.9133902757705612, 0.7177271344564569, 0.13402948187280572},
    {0.9133791520943453, 0.7238672597994953, 0.13644749308843845},
    {0.913344793367654, 0.7300002810426419, 0.13886575701873277},
    {0.9132870435698786, 0.7361264894275755, 0.1412843686838794},
    {0.9132057442654163, 0.7422461658162536, 0.1437034180372798},
    {0.913100734549166, 0.7483595811550647, 0.1461229902831289},
    {0.9129718509904932, 0.7544669969134981, 0.14854316617002308},
    {0.9128189275755945, 0.7605686654989812, 0.15096402226269828},
    {0.9126417956482176, 0.7666648306494336, 0.15338563119383095},
    {0.9124402838486693, 0.7727557278049575, 0.1558080618976074},
    {0.9122142180510416, 0.778841584459982, 0.15823137982661514},
    {0.9119634212986102, 0.7849226204970876, 0.16065564715346278},
    {0.9116877137373088, 0.7909990485036371, 0.16308092295837506},
    {0.9113869125472361, 0.797071074072276, 0.16550726340392827},
    {0.9110608318720959, 0.803138896086265, 0.16793472189794498},
    {0.9107092827465118, 0.8092027069905707, 0.1703633492455004},
    {0.9103320730211178, 0.8152626930495434, 0.17279319379088973},
    {0.909929007285357, 0.8213190345919861, 0.17522430155033844},
    {0.9094998867878827, 0.8273719062443313, 0.17765671633615956},
    {0.9090445093544782, 0.8334214771526294, 0.1800904798730111},
    {0.9085626693033965, 0.8394679111939684, 0.18252563190683574},
    {0.9080541573580172, 0.8455113671779366, 0.18496221030703314},
    {0.9075187605567102, 0.8515519990386777, 0.18740025116234627},
    {0.9069562621598014, 0.8575899560180628, 0.18983978887092715},
    {0.9063664415535179, 0.8636253828404642, 0.19228085622498767},
    {0.9057490741507888, 0.8696584198795938, 0.19472348449042068},
    {0.9051039312887829, 0.875689203317828, 0.197167703481744},
    {0.9044307801230321, 0.8817178652984231, 0.19961354163268363},
    {0.9037293835180177, 0.8877445340709991, 0.2020610260626995},
    {0.9029994999340526, 0.8937693341306414, 0.20451018263972115},
    {0.9022408833103207, 0.8997923863509576, 0.2069610360393544},
    {0.9059888943947992, 0.9040221289848984, 0.2345476432758713},
    {0.9121353668077321, 0.9072538390291374, 0.2723959319742808},
    {0.918103004002936, 0.910496798148879, 0.30707485069273716},
    {0.9238889872583352, 0.913751436869596, 0.3395892760133458},
    {0.9294904738341414, 0.9170181766931385, 0.3705416931634215},
    {0.9349045893736176, 0.9202974302069179, 0.4003237614900562},
    {0.9401284204049521, 0.9235896011902338, 0.4292051255044417},
    {0.9451590068972918, 0.9268950847179593, 0.45737955986332374},
    {0.9499933348250024, 0.9302142672617891, 0.4849910259595919},
    {0.9546283286947478, 0.9335475267892457, 0.5121493527958956},
    {0.9590608439900784, 0.936895232860618, 0.5389401619772186},
    {0.9632876594878218, 0.9402577467240134, 0.5654314125128757},
    {0.9673054693997428, 0.9436354214086733, 0.5916778659454154},
    {0.971110875291573, 0.9470286018167143, 0.6177242212867308},
    {0.9747003777296833, 0.9504376248134289, 0.6436073706776709},
    {0.9780703676032384, 0.9538628193162827, 0.6693580571761435},
    {0.9812171170667078, 0.9573045063827349, 0.6950021159198041},
    {0.9841367700439326, 0.9607629992969943, 0.7205614186521087},
    {0.9868253322306101, 0.9642386036558243, 0.7460546029896806},
    {0.9892786605268938, 0.9677316174534963, 0.7714976428174245},
    {0.9914924518257658, 0.9712423311659882, 0.7969042996355301},
    {0.9934622310757725, 0.9747710278345115, 0.8222864834722017},
    {0.9951833385285259, 0.9783179831484546, 0.8476545442427741},
    {0.9966509160718274, 0.9818834655278131, 0.873017509008704},
    {0.9978598925382617, 0.9854677362051741, 0.8983832767211907},
    {0.9988049678662737, 0.9890710493073268, 0.9237587792366369},
    {0.999480595975958, 0.9926936519365445, 0.9491501153416039},
    {0.9998809662045313, 0.9963357842516011, 0.9745626630050933},
    {1.0, 1.0, 1.0},
}};

template <typename Real>
static constexpr std::array<std::array<Real, 3>, 256> plasma_data_ = {{
    {0.05038205347059877, 0.029801736499741757, 0.5279751010495176},
    {0.06353382706361996, 0.028424851177690835, 0.5331235351456174},
    {0.07535267397875561, 0.027204618108821313, 0.5380072654878371},
    {0.08622056271327161, 0.02612369628606181, 0.5426577546250408},
    {0.09637909234021627, 0.02516351238474626, 0.5471034027913018},
    {0.10597907050045527, 0.02430756301203486, 0.5513679345503644},
    {0.11512442337588075, 0.0235544144422598, 0.5554685209574166},
    {0.12390246837508934, 0.022876443394988937, 0.5594230869193363},
    {0.13237958264924907, 0.022256467283011946, 0.5632496670754658},
    {0.14060280627862978, 0.02168533317299063, 0.5669592160880329},
    {0.14860595337659185, 0.021152345584078584, 0.5705618203562642},
    {0.1564211065006705, 0.020649224182780333, 0.5740653310686394},
    {0.16406928001421978, 0.020169228789859318, 0.5774779563374444},
    {0.17157438178522647, 0.019704113165841947, 0.5808064352016079},
    {0.1789495767422052, 0.019250113119147653, 0.584054078695406},
    {0.18621180115469324, 0.018801112566559006, 0.5872277307558299},
    {0.19337385391747602, 0.018351995878840264, 0.5903301881492813},
    {0.20044409185351197, 0.01789999393180025, 0.593363854926767},
    {0.20743511639383128, 0.01743987460850466, 0.5963332877187512},
    {0.21434936531465415, 0.016970872739093673, 0.5992389686439623},
    {0.22119736746683188, 0.016494748620846298, 0.6020833788659872},
    {0.22798262579984302, 0.01600474729153228, 0.6048670732620841},
    {0.23471390034056855, 0.015499747535836161, 0.6075917748304479},
    {0.24139587735519374, 0.014976616724022502, 0.6102591681981376},
    {0.24803115851643903, 0.01443661887416811, 0.6128678816678474},
    {0.25462712204192733, 0.013879480363965033, 0.6154192540143766},
    {0.26118240815987454, 0.013305484446440807, 0.6179109803957406},
    {0.2677033607536392, 0.012713338432089919, 0.6203463317006325},
    {0.27419065115382085, 0.012106344331753975, 0.6227220708698477},
    {0.2806485945135714, 0.011485191704992936, 0.625038400861471},
    {0.2870758894612177, 0.010852198594472949, 0.6272951521878755},
    {0.29347719408360684, 0.010210206090003579, 0.6294899111444117},
    {0.2998551242452729, 0.009558047948332786, 0.6316242235777169},
    {0.30620943129674466, 0.008899055475505046, 0.6336939963015096},
    {0.3125433553743016, 0.008235893717416108, 0.6357002857590853},
    {0.31885566464941756, 0.007572899889250414, 0.6376400727865161},
    {0.3251505835029789, 0.0069117371894770714, 0.6395123380850828},
    {0.3314258950888549, 0.006257740755607049, 0.6413161393477004},
    {0.3376822144299248, 0.005614742031369818, 0.643048949066938},
    {0.343925123616484, 0.00498757999986155, 0.6447101941536691},
    {0.35014944433050965, 0.004378575729205644, 0.6462980198153693},
    {0.3563593495264177, 0.0037944194657679817, 0.6478102381291251},
    {0.3625526716933222, 0.0032394075990651583, 0.6492450805997739},
    {0.3687335729955321, 0.002720261374452697, 0.6506012705463909},
    {0.3748968968242403, 0.002241239865606646, 0.6518761301685362},
    {0.38104622727684795, 0.001810210773066439, 0.6530679993306958},
    {0.3871831206545137, 0.0014300761021609504, 0.6541771655986202},
    {0.3933034524768182, 0.0011100331980040906, 0.6551990534286212},
    {0.39941134218209506, 0.0008549181444151815, 0.6561331884791302},
    {0.40550267547562835, 0.0006738585958816878, 0.656977095844568},
    {0.4115805613352908, 0.0005727689276359682, 0.6577301981873359},
    {0.4176418962835305, 0.000559689848675834, 0.6583901257368883},
    {0.4236882377452387, 0.0006415963809730485, 0.6589560636788017},
    {0.4297191157868382, 0.0008265327400941048, 0.6594251390708742},
    {0.43573345912178957, 0.0011224144971593696, 0.6597970982982151},
    {0.4417323328094361, 0.0015353882778950684, 0.6600691382766763},
    {0.4477136781734527, 0.002075242343276844, 0.6602401195491348},
    {0.45367754705516167, 0.0027502600296914884, 0.6603101230367315},
    {0.45962289480536683, 0.003569082980411755, 0.6602771266757947},
    {0.4655492492356133, 0.004539882193632057, 0.6601391424818946},
    {0.4714571093971736, 0.005672943386802442, 0.6598971161567793},
    {0.4773434667747343, 0.0069747060416045946, 0.6595491544709485},
    {0.48321032105464906, 0.00845482487303384, 0.6590950901585111},
    {0.4890546814926811, 0.010121547209062692, 0.6585341518234636},
    {0.49487752945710545, 0.011984730267283644, 0.6578650488841956},
    {0.5006778931801235, 0.01404940920364397, 0.6570881342241002},
    {0.5064532641572821, 0.016327053582825775, 0.6562022327592171},
    {0.512206101996081, 0.018827299101003524, 0.6552090987935029},
    {0.517932476985869, 0.021556895188755264, 0.6541092197088938},
    {0.5236333071472989, 0.02452621873336916, 0.6529010471797861},
    {0.5293056865203295, 0.02774076220334348, 0.651586191657666},
    {0.5349525083091329, 0.031211170127490386, 0.6501649809244013},
    {0.5405698921735111, 0.034943657938542025, 0.6486401484123572},
    {0.5461562839529602, 0.03894709993862434, 0.6470103291518646},
    {0.5517150940137298, 0.04312993669146801, 0.6452770884507062},
    {0.5572424908996385, 0.04732473166532047, 0.6434432899905641},
    {0.5627382914453162, 0.05153956982770822, 0.6415090129244055},
    {0.5682006939186361, 0.05577231508140689, 0.639477236154527},
    {0.5736324841026456, 0.06002312768941918, 0.6373489241079173},
    {0.5790288922820648, 0.06429083299444488, 0.635126168256215},
    {0.5843916716197954, 0.06857462612193421, 0.6328118231142932},
    {0.5897190859433459, 0.07287330061804262, 0.6304080857996939},
    {0.5950105087389658, 0.07718497188889406, 0.627917359511397},
    {0.600266274548656, 0.08151172798904885, 0.6253419897153364},
    {0.6054847038259513, 0.08584937397306835, 0.6226862814869409},
    {0.6106674577942587, 0.09020012029057023, 0.6199508828839105},
    {0.6158118936935632, 0.09455974518910823, 0.6171401916106442},
    {0.6209196353232503, 0.09893048328312148, 0.6142567668402714},
    {0.625987078238522, 0.10330809138750463, 0.6113050903371116},
    {0.6310165297116345, 0.10769470016602048, 0.6082874219554474},
    {0.6360082571946242, 0.11208841745759027, 0.6052049784133696},
    {0.6409587158624425, 0.11648801193763364, 0.6020653233567386},
    {0.6458724305886356, 0.120894724408147, 0.5988668588802512},
    {0.6507458962880415, 0.1253053066756304, 0.5956172158180971},
    {0.65558059804725, 0.12972201477991677, 0.59231673323355},
    {0.6603740711643852, 0.13414058702631182, 0.5889711004114799},
    {0.6651285526432004, 0.1385621615837031, 0.5855824728255514},
    {0.669845240296135, 0.1429888560980906, 0.582153977816893},
    {0.6745217289587141, 0.14741542201076516, 0.5786883585907729},
    {0.6791604036855876, 0.15184511348395324, 0.5751888508715063},
    {0.6837578997178535, 0.1562746720253843, 0.5716602384468066},
    {0.6883185616652895, 0.16070636061039098, 0.5681027207214548},
    {0.6928400648216108, 0.16513791277492407, 0.5645221138655818},
    {0.6973235757805955, 0.1695694677210192, 0.5609195098635373},
    {0.7017692243107729, 0.1740021467142358, 0.5572959855617685},
    {0.7061777422578236, 0.17843369623403554, 0.5536573853799269},
    {0.7105493783435904, 0.18286537302055275, 0.5500038553740142},
    {0.7148829033672169, 0.1872959178727836, 0.5463382579948199},
    {0.7191815273421805, 0.19172659246702414, 0.5426627244615733},
    {0.7234440590934259, 0.196155133293472, 0.5389811286444972},
    {0.7276695980921538, 0.20058267652430906, 0.5352935340967259},
    {0.7318622095195731, 0.20501034405621088, 0.5316009984849964},
    {0.7360187549600071, 0.20943588386264006, 0.527908404300108},
    {0.7401433549469377, 0.2138615497973006, 0.5242158683817802},
    {0.7442319068498962, 0.2182850864322064, 0.520524273866664},
    {0.7482894956274093, 0.22270875088361386, 0.516833738969623},
    {0.7523120538611476, 0.2271302847089524, 0.5131491434259574},
    {0.7563036182318114, 0.23155182037018107, 0.509468548147597},
    {0.760264196042346, 0.23597347976619776, 0.505794013908438},
    {0.7641927663268164, 0.24039301314981837, 0.5021264170700042},
    {0.7680903336977949, 0.24481467160443043, 0.49846488562080354},
    {0.7719579098239822, 0.24923420250548467, 0.494813286735782},
    {0.7757964670137317, 0.2536558602885356, 0.49117075886112754},
    {0.7796040489503765, 0.2580753889528994, 0.487539157444613},
    {0.7833826364367662, 0.262496918855311, 0.48391855557538205},
    {0.7871331836672137, 0.2669195732674377, 0.4803070299618413},
    {0.7908547764917832, 0.2713421013551332, 0.4767064258861618},
    {0.7945493142314948, 0.2757677557136531, 0.47311690432863956},
    {0.7982159123706819, 0.2801942814938595, 0.4695382974681735},
    {0.801855440805233, 0.28462393619999926, 0.46597078035645983},
    {0.8054670443253337, 0.28905445984517764, 0.46241517054928255},
    {0.8090525637702202, 0.2934891151740604, 0.4588696581516773},
    {0.8126121722311862, 0.29792563692815727, 0.4553380455774297},
    {0.8161437857429269, 0.3023651595826911, 0.45181643298028346},
    {0.8196512962080388, 0.30680981323456863, 0.44830592266123775},
    {0.8231319147062668, 0.31125833345906073, 0.4448063071891859},
    {0.8265884164961489, 0.31571198854333155, 0.4413158011478088},
    {0.8300180400348914, 0.3201695064581668, 0.43783618296069426},
    {0.8334225331012705, 0.3246331633551905, 0.43436568094830924},
    {0.8368011616192005, 0.3291026789528604, 0.4309050603621552},
    {0.8401547945201179, 0.33357719515430856, 0.42745544005358527},
    {0.8434842793132458, 0.33805985158665625, 0.42401293964429126},
    {0.8467879171814312, 0.3425483651134692, 0.42057931701643003},
    {0.8500663933256695, 0.3470460241716122, 0.41715281999997633},
    {0.853319036312829, 0.35155053489832117, 0.4137341950926385},
    {0.8565475037861691, 0.35606419689875035, 0.4103217013252645},
    {0.8597501517507731, 0.3605857048991911, 0.4069170743064185},
    {0.8629268039532411, 0.36511621303625214, 0.40351944853543975},
    {0.8660782631128492, 0.36965787597771715, 0.4001259551388744},
    {0.8692029205728464, 0.3742093808977564, 0.3967383275867196},
    {0.8723033707464051, 0.37877204764837713, 0.39335483663056486},
    {0.8753760334959819, 0.38334454940566526, 0.38997620732417515},
    {0.8784234745450857, 0.3879302203109463, 0.3865997186779487},
    {0.8814431427224427, 0.3925267186370576, 0.38322908763780417},
    {0.8844358150168464, 0.39713621668929217, 0.3798604589983262},
    {0.8874022475255273, 0.4017598897994324, 0.3764939692996154},
    {0.8903399254603347, 0.40639538423669647, 0.3731303392842935},
    {0.8932503482591801, 0.41104606226907536, 0.3697678512068009},
    {0.8961310320776956, 0.41570955292043654, 0.3664072199362221},
    {0.8989844448210237, 0.42039023641999307, 0.36304673325600695},
    {0.9018071346530614, 0.4250847229961323, 0.35968810073229274},
    {0.9046008288037846, 0.4297942088907302, 0.35632947125103215},
    {0.9073652324299479, 0.4345218955954952, 0.3529699824197478},
    {0.9100979328220618, 0.43926537724749737, 0.3496103521951318},
    {0.9128003257007559, 0.4440270702284773, 0.3462508642176817},
    {0.9154710325568927, 0.4488045475293247, 0.3428902329848131},
    {0.9181094141599885, 0.45360124699433185, 0.3395287458827024},
    {0.9207141279319739, 0.45841471975818476, 0.3361661136226204},
    {0.9232868464124966, 0.463248191094171, 0.3328014851212701},
    {0.9258252179695905, 0.46810089493647095, 0.3294349948733575},
    {0.928328943461609, 0.4729723618415593, 0.32606736582761625},
    {0.9307983028156818, 0.4778650729009944, 0.32269687622031573},
    {0.9332320356689454, 0.48277753475654456, 0.31932524626654357},
    {0.935630382313249, 0.4877102534916354, 0.3159517572279356},
    {0.9379901229720649, 0.4926647102078156, 0.3125751263224191},
    {0.94031286905006, 0.4976391653394365, 0.30919749958566495},
    {0.9425982043994484, 0.5026368889510107, 0.3058160067766645},
    {0.9448439581861301, 0.507655339193432, 0.30243337976158996},
    {0.9470512797767, 0.5126970710073548, 0.2990488875873464},
    {0.9492170420289342, 0.5177605158051324, 0.29566225939396684},
    {0.9513443493064214, 0.5228482560427825, 0.2922747680564989},
    {0.95342812024145, 0.5279576952925963, 0.2888831386174737},
    {0.9554704126542967, 0.5330914441820193, 0.28548964857337034},
    {0.9574691920662094, 0.5382478782970336, 0.2820960181641041},
    {0.9594239777856378, 0.5434283103460965, 0.27870139293719676},
    {0.9613362570153808, 0.5486340651043815, 0.27530489863581714},
    {0.9632030520719662, 0.5538624913974601, 0.2719092715909324},
    {0.9650243153126318, 0.5591162551393607, 0.26851277921609085},
    {0.9667981202444841, 0.5643936755766422, 0.2651181499385089},
    {0.9685263667936735, 0.5696984485228503, 0.26172065963235575},
    {0.9702051814086846, 0.5750258633596333, 0.258325028790098},
    {0.9718350028152128, 0.5803792760776375, 0.25493140399836905},
    {0.973416234677147, 0.5857590553651805, 0.2515399093896209},
    {0.9749470666071361, 0.5911624621207934, 0.24815128177724166},
    {0.9764282806289234, 0.5965932508399702, 0.24476679063210272},
    {0.9778561235005537, 0.6020486514176712, 0.24138715955646184},
    {0.9792333190737449, 0.6075304496592373, 0.23801267291982608},
    {0.980556172868528, 0.6130368443483086, 0.23464603815019064},
    {0.9818260338342485, 0.618569237111758, 0.2312874100772107},
    {0.983041213251059, 0.6241290418881347, 0.2279369200964918},
    {0.984199086129537, 0.6297154282611285, 0.22459528753260782},
    {0.9853012455753073, 0.6353282428938619, 0.22126480373791835},
    {0.9863451305526932, 0.6409666230896872, 0.2179481656192832},
    {0.987332269632511, 0.6466314474784756, 0.21464768988345995},
    {0.988260166889663, 0.6523228213835303, 0.2113640451754461},
    {0.9891280720682684, 0.6580401935630623, 0.20810040712251393},
    {0.989935193134965, 0.6637850245048573, 0.2048589308368297},
    {0.9906811112647872, 0.6695553903969398, 0.20164228467841383},
    {0.9913652104916543, 0.675353231166615, 0.19845282041957812},
    {0.9919851419644327, 0.681176590770927, 0.19529516473738912},
    {0.9925412186450403, 0.6870284413732891, 0.19216971523394297},
    {0.9930321637354537, 0.6929047946873668, 0.18908404856516375},
    {0.9934561180556553, 0.6988071461016482, 0.18604138565745748},
    {0.9938141746253392, 0.7047390032802133, 0.18304294167742508},
    {0.9941031429791255, 0.7106953484789053, 0.18009726563511483},
    {0.9943241754734297, 0.7166792156038955, 0.17720784326567807},
    {0.9944741584952311, 0.7226885545120336, 0.17438115112347216},
    {0.9945531662213939, 0.7287264314705092, 0.17162175513054803},
    {0.994561164165776, 0.7347887640856848, 0.16893804446424532},
    {0.9944951727911301, 0.7408770946176795, 0.16633533149747243},
    {0.9943551582286289, 0.7469929781194636, 0.16382095249343898},
    {0.9941411819033531, 0.7531343027240105, 0.16140421771177754},
    {0.993851140881807, 0.7593021960529095, 0.1590918763664673},
    {0.9934821808433413, 0.7654965143680154, 0.15689111532223957},
    {0.9930331122322037, 0.7717184174350065, 0.15480781781660563},
    {0.9925051688188334, 0.7779647295700995, 0.15285502755098562},
    {0.9918972375664695, 0.7842360395514291, 0.15104222448313137},
    {0.9912091443124736, 0.7905349489016323, 0.1493769620239061},
    {0.9904392291943233, 0.7968562534486284, 0.1478701260432203},
    {0.9895871071514554, 0.8032031721875696, 0.14652892189857109},
    {0.9886482096752364, 0.8095764705517057, 0.1453570476879498},
    {0.9876210577245915, 0.815976398662919, 0.14436290762709728},
    {0.986509178197983, 0.8223986911764639, 0.14355699253114637},
    {0.9853139963633676, 0.8288446279885695, 0.14294492163758302},
    {0.9840311334696324, 0.8353129155540111, 0.14252796680830504},
    {0.9826532853628255, 0.8418092005650518, 0.14230298881977124},
    {0.9811900755223993, 0.8483271434249618, 0.14227897296255487},
    {0.9796442447131871, 0.854863423552606, 0.1424529510156046},
    {0.9779950043835023, 0.8614303744688768, 0.14280800468858543},
    {0.9762651924699953, 0.8680136489692144, 0.14335094248589855},
    {0.9744429204121595, 0.8746206082101065, 0.14406105658536314},
    {0.972530126666216, 0.8812478776950604, 0.14492296106549138},
    {0.9705333468892211, 0.8878931459431306, 0.14591884881454822},
    {0.968443047163475, 0.8945621097233081, 0.14701399877638566},
    {0.9662712851816472, 0.901246373209881, 0.14817986820470014},
    {0.964020955903533, 0.9079483440427187, 0.1493700336843369},
    {0.9616812107840874, 0.9146696029935507, 0.1505199051295305},
    {0.959275857396244, 0.9214055790441742, 0.15156603392174148},
    {0.9568081246010715, 0.9281498350816461, 0.1524089373934798},
    {0.9542874028057909, 0.934905090703852, 0.15292087802256293},
    {0.9517260321586368, 0.9416690683960072, 0.15292489914850274},
    {0.9491513146628231, 0.9484323225943193, 0.1521779775289114},
    {0.9466019493393495, 0.9551882982848324, 0.15032766414862223},
    {0.9441522165792432, 0.9619135560223769, 0.14686103617160504},
    {0.9418959241387963, 0.9685885113553053, 0.14095488309848805},
    {0.9400151278782742, 0.9751557856205376, 0.131325887773911},
}};

template <typename Real>
static constexpr std::array<std::array<Real, 3>, 256> smooth_cool_warm_data_ = {
        {{0.22999950386952345, 0.2989989340493756, 0.754000138575591},
         {0.23451750918602265, 0.30586471825124395, 0.760211287847582},
         {0.23905139222321087, 0.31271835359723077, 0.7663613706951183},
         {0.2436017365913441, 0.3195596165920886, 0.7724494083708939},
         {0.24816908043486074, 0.3263882334026712, 0.7784744332927407},
         {0.25275391815771386, 0.33320388462766504, 0.7844354891865984},
         {0.2573567020073366, 0.34000620954660093, 0.7903316312278389},
         {0.2619778435347082, 0.34679480991553485, 0.7961619261808826},
         {0.26661771494573344, 0.3535692533668808, 0.8019254525370478},
         {0.271276650357161, 0.3603290764626063, 0.8076213006505808},
         {0.27595494696856954, 0.3670737874430388, 0.813248572872804},
         {0.28065286616048707, 0.37380286870769464, 0.8188063836843368},
         {0.28537063452740463, 0.3805157790595848, 0.8242938598253321},
         {0.29010844485334814, 0.3872119557402635, 0.829710140423685},
         {0.29486645703670583, 0.3938908162793105, 0.8350543771211604},
         {0.2996447989701334, 0.4005517601788923, 0.8403257341974003},
         {0.30444356738064665, 0.4071941704514421, 0.8455233886917599},
         {0.3092628286343541, 0.41381741502624314, 0.8506465305229377},
         {0.31410261950970153, 0.42042084803879193, 0.8556943626063482},
         {0.3189629479426216, 0.4270038110151302, 0.8606661009692109},
         {0.32384379374653405, 0.43356563396190284, 0.8655609748633052},
         {0.32874510930975565, 0.4401056363716498, 0.8703782268753609},
         {0.33366682027256384, 0.4466231281517378, 0.8751171130350492},
         {0.33860882618583127, 0.45311741048440196, 0.8797769029205332},
         {0.3435710011529169, 0.45958777662452927, 0.8843568797615591},
         {0.3485531944562718, 0.46603351264108767, 0.8888563405400349},
         {0.3535552311699886, 0.47245389810747423, 0.8932745960880898},
         {0.3585769127594044, 0.47884820674548473, 0.8976109711835649},
         {0.3636180176686533, 0.4852157070271205, 0.9018648046429221},
         {0.3686783018969878, 0.49155566273800605, 0.9060354494115308},
         {0.37375749956453747, 0.49786733350580864, 0.9101222726513168},
         {0.37885532346808715, 0.5041499752967064, 0.9141246558257436},
         {0.38397146562736917, 0.5104028408826495, 0.918041994782103},
         {0.38910559782229387, 0.5166251802818846, 0.9218736998310882},
         {0.39425737212145806, 0.5228162411749818, 0.9256191958236355},
         {0.39942642140225487, 0.5289752692983762, 0.9292779222250065},
         {0.4046123598628117, 0.5351015088172548, 0.9328493331860931},
         {0.4098147835259849, 0.541194202679442, 0.9363328976119261},
         {0.41503327073558766, 0.5472525929517891, 0.9397280992273693},
         {0.4202673826449946, 0.5532759211404236, 0.943034436639982},
         {0.42551666369825586, 0.5592634284961118, 0.9462514234000324},
         {0.43078064210382544, 0.5652143563058468, 0.9493785880576479},
         {0.43605883030099185, 0.5711279461717078, 0.9524154742170866},
         {0.44135072541910086, 0.5770034402779125, 0.9553616405881157},
         {0.4466558097296202, 0.5828400816469308, 0.9582166610344868},
         {0.4519735510911263, 0.5886371143854353, 0.960980124619491},
         {0.457303403387259, 0.5943937839208078, 0.9636516356485881},
         {0.46264480695769356, 0.6001093372288535, 0.9662308137090968},
         {0.4679971890221843, 0.6057830230533238, 0.968717293706932},
         {0.4733599640977317, 0.6114140921177984, 0.9711107259003909},
         {0.4787325344089037, 0.6170017973304325, 0.9734107759309701},
         {0.48411429029138214, 0.6225453939820256, 0.9756171248512094},
         {0.48950461058876166, 0.6280441399378467, 0.9777294691495587},
         {0.49490286304267245, 0.6334972958235978, 0.9797475207722576},
         {0.5003084046762621, 0.6389041252058836, 0.9816710071422271},
         {0.5057205821711058, 0.644263894767512, 0.983499671174965},
         {0.511138732237593, 0.6495758744779339, 0.9852332712914448},
         {0.5165621819788595, 0.654839337759102, 0.9868715814280132},
         {0.5219902492483212, 0.6600535616470082, 0.9884143910432847},
         {0.5274222430008886, 0.6652178269491371, 0.9898615051220324},
         {0.5328574636379119, 0.670331418398056, 0.9912127441760742},
         {0.5382952033459485, 0.6753936248013475, 0.9924679442421503},
         {0.5437347464294153, 0.6804037391880692, 0.9936269568768029},
         {0.5491753696372063, 0.685361058951912, 0.9946896491482432},
         {0.5546163424833603, 0.690264885991224, 0.9956559036252249},
         {0.5600569275618472, 0.695114526846042, 0.9965256183629146},
         {0.5654963808555731, 0.6999092928322662, 0.9972987068857656},
         {0.5709339520396719, 0.7046485001731109, 0.9979750981673984},
         {0.5763688847791901, 0.7093314701279408, 0.9985547366074979},
         {0.5818004170212294, 0.7139575291186055, 0.9990375820057199},
         {0.5872277812816606, 0.7185260088533676, 0.9994236095326229},
         {0.5926502049264796, 0.7230362464485207, 0.9997128096976274},
         {0.5980669104479099, 0.7274875845477786, 0.9999051883140077},
         {0.603477115735333, 0.7318793714395154, 1.0},
         {0.6088800343411463, 0.7362109611719294, 0.9999995804425049},
         {0.6142748757416401, 0.7404817136661965, 0.9999016817439826},
         {0.6196608455929782, 0.7446909948276748, 0.9997071369848932},
         {0.6250371459823868, 0.7488381766552155, 0.9994160278693497},
         {0.6304029756746347, 0.7529226373486367, 0.9990284511333963},
         {0.6357575303538953, 0.7569437614144018, 0.9985445184894549},
         {0.6411000028610946, 0.7609009397695506, 0.9979643565678705},
         {0.6464295834268168, 0.7647935698439208, 0.9972881068555687},
         {0.6517454598998733, 0.7686210556806996, 0.9965159256318358},
         {0.6570468179716188, 0.7723828080353353, 0.9956479839012286},
         {0.6623328413960967, 0.7760782444728409, 0.9946844673236305},
         {0.6676027122061123, 0.7797067894635203, 0.9936255761414589},
         {0.6728556109253114, 0.7832678744771355, 0.9924715251040422},
         {0.6780907167763519, 0.7867609380755451, 0.9912225433891731},
         {0.683307207885253, 0.7901854260038265, 0.9898788745218482},
         {0.6885042614820046, 0.7935407912799094, 0.988440776290216},
         {0.6936810540975126, 0.7968264942827246, 0.9869085206587308},
         {0.6988367617569751, 0.8000420028388932, 0.9852823936785383},
         {0.7039705601697447, 0.8031867923079613, 0.983562695395095},
         {0.7090816249157784, 0.8062603456661956, 0.9817497397530396},
         {0.714169131628733, 0.8092621535889422, 0.9798438544983238},
         {0.7192322561757896, 0.8121917145315617, 0.9778453810776183},
         {0.7242701748342745, 0.8150485348089448, 0.9757546745350029},
         {0.7292820644651583, 0.8178321286736122, 0.9735721034059553},
         {0.7342671026834836, 0.8205420183923986, 0.971298049608644},
         {0.7392244680258144, 0.8231777343217284, 0.9689329083325428},
         {0.7441533401147525, 0.8257388149814784, 0.9664770879243741},
         {0.7490528998206014, 0.8282248071274302, 0.9639310097713906},
         {0.7539223294202332, 0.8306352658223072, 0.9612951081820061},
         {0.7587608127532254, 0.8329697545053927, 0.9585698302637836},
         {0.7635675353753275, 0.8352278450607313, 0.9557556357987903},
         {0.7683416847093205, 0.8374091178838983, 0.9528529971163268},
         {0.7730824501933197, 0.8395131619473374, 0.9498623989630331},
         {0.7777890234265885, 0.8415395748642601, 0.9467843383703861},
         {0.7824605983129111, 0.8434879629510917, 0.9436193245195849},
         {0.7870963712015859, 0.8453579412884608, 0.9403678786038298},
         {0.7916955410260809, 0.8471491337807205, 0.9370305336880028},
         {0.7962573094404225, 0.8488611732139871, 0.933607834565744},
         {0.8007808809533457, 0.8504937013126886, 0.9301003376139276},
         {0.8052654630602748, 0.8520463687946024, 0.9265086106445327},
         {0.8097102663731715, 0.8535188354243787, 0.9228332327539112},
         {0.8141145047483038, 0.854910770065524, 0.9190747941694408},
         {0.8184773954119764, 0.8562218507308311, 0.9152338960935597},
         {0.822798159084272, 0.857451764631246, 0.9113111505451725},
         {0.827076020100851, 0.8586002082231403, 0.9073071801984174},
         {0.8313102065328385, 0.8596668872539794, 0.9032226182187731},
         {0.8354999503048602, 0.8606515168063638, 0.8990581080965018},
         {0.8396444873112505, 0.8615538213404197, 0.8948143034773921},
         {0.8437430575304858, 0.8623735347345196, 0.8904918679907933},
         {0.8477949051378751, 0.8631104003243038, 0.8860914750749006},
         {0.8517992786165454, 0.8637641709399845, 0.8816138077992733},
         {0.8557554308667662, 0.8643346089419015, 0.8770595586845422},
         {0.8596626193136427, 0.8648214862543027, 0.8724294295192715},
         {0.8635201060132168, 0.8652245843973202, 0.8677241311739308},
         {0.8676431881102519, 0.8645140349150363, 0.8626161485985507},
         {0.8719973001889246, 0.8626912788780988, 0.8571344116931553},
         {0.8762551679574422, 0.8607866996599632, 0.8516174449794455},
         {0.8804169678453379, 0.8588005566711197, 0.8460661683343939},
         {0.8844828646854237, 0.8567331182372422, 0.8404815005112565},
         {0.8884530127026752, 0.8545846614985785, 0.834864358985154},
         {0.892327556446231, 0.8523554723065947, 0.829215659800157},
         {0.8961066316684723, 0.8500458451177988, 0.8235363174178995},
         {0.8997903661548088, 0.8476560828846528, 0.8178272445677435},
         {0.9033788805075311, 0.8451864969434798, 0.812089352098517},
         {0.9068722888867912, 0.8426374068992736, 0.8063235488318476},
         {0.910270699711566, 0.8400091405073019, 0.8005307414171138},
         {0.9135742163232042, 0.8373020335514, 0.7947118341880348},
         {0.9167829376139871, 0.8345164297188294, 0.7888677290209213},
         {0.9198969586229171, 0.8316526804715908, 0.7829993251946056},
         {0.9229163711008167, 0.8287111449140491, 0.7771075192520753},
         {0.9258412640466325, 0.8256921896567365, 0.7711932048638259},
         {0.9286717242167226, 0.822596188676188, 0.7652572726929551},
         {0.9314078366087624, 0.8194235231706464, 0.7593006102620192},
         {0.9340496849217929, 0.8161745814114749, 0.7533241018216675},
         {0.9365973519938281, 0.8128497585900935, 0.7473286282210768},
         {0.9390509202183217, 0.8094494566602553, 0.7413150667802011},
         {0.9414104719407324, 0.8059740841754534, 0.7352842911638605},
         {0.9436760898363059, 0.8024240561212462, 0.7292371712576793},
         {0.9458478572701444, 0.7987997937422701, 0.7231745730458999},
         {0.9479258586405416, 0.7951017243636899, 0.717097358491085},
         {0.9499101797065052, 0.7913302812068249, 0.7110063854157246},
         {0.951800907900325, 0.7874859031986702, 0.7049025073857709},
         {0.953598132625986, 0.7835690347750057, 0.6987865735961123},
         {0.9553019455441706, 0.7795801256767763, 0.6926594287580068},
         {0.9569124408445594, 0.7755196307393911, 0.6865219129884912},
         {0.9584297155060683, 0.7713880096745724, 0.6803748617017831},
         {0.9598538695456499, 0.7671857268443542, 0.6742191055026909},
         {0.9611850062562218, 0.7629132510268043, 0.6680554700820523},
         {0.9624232324342575, 0.758571055173009, 0.6618847761142137},
         {0.9635686585975486, 0.7541596161548229, 0.6557078391565709},
         {0.9646213991936067, 0.7496794145028565, 0.6495254695511868},
         {0.9655815727991436, 0.7451309341341271, 0.6433384723285004},
         {0.9664493023110522, 0.7405146620687538, 0.6371476471131512},
         {0.9672247151292687, 0.7358310881350345, 0.6309537880319284},
         {0.9679079433318883, 0.7310807046621858, 0.6247576836238656},
         {0.9684991238428787, 0.7262640061599712, 0.6185601167525033},
         {0.9689983985927076, 0.7213814889843815, 0.612361864520328},
         {0.9694059146721955, 0.7164336509884588, 0.6061636981854146},
         {0.9697218244798756, 0.7114209911572899, 0.5999663830802852},
         {0.9699462858631324, 0.7063440092261024, 0.5937706785330095},
         {0.9700794622533694, 0.7012032052803152, 0.5875773377905578},
         {0.9701215227954475, 0.6959990793362977, 0.5813871079444363},
         {0.9700726424716136, 0.6907321309014792, 0.5752007298586196},
         {0.9699330022201408, 0.6854028585123331, 0.5690189380998024},
         {0.9697027890488664, 0.6800117592486405, 0.5628424608699958},
         {0.9693821961438281, 0.6745593282222776, 0.556672019941486},
         {0.9689714229731695, 0.6690460580386323, 0.5505083305941867},
         {0.968470675386481, 0.663472438228566, 0.5443521015554007},
         {0.9678801657097379, 0.65783895464866, 0.5382040349420275},
         {0.9672001128359847, 0.6521460888472544, 0.5320648262052324},
         {0.9664307423119013, 0.6463943173935754, 0.5259351640776159},
         {0.9655722864203883, 0.6405841111669635, 0.5198157305229032},
         {0.964624984259297, 0.6347159346029446, 0.5137072006881956},
         {0.9635890818164178, 0.6287902448925465, 0.507610242858806},
         {0.9624648320408441, 0.6228074911309212, 0.5015255184157178},
         {0.9612524949108144, 0.6167681134109175, 0.4954536817957061},
         {0.9599523374981328, 0.6106725418568123, 0.4893953804541488},
         {0.9585646340292623, 0.6045211955929051, 0.4833512548305818},
         {0.9570896659431806, 0.5983144816411149, 0.4773219383170267},
         {0.9555277219460815, 0.5920527937410947, 0.47130805722914443},
         {0.9538790980630029, 0.5857365110856726, 0.4653102307802585},
         {0.9521440976864516, 0.5793659969636203, 0.4593290710582975},
         {0.9503230316221086, 0.57294159730086, 0.4533651830057083},
         {0.948416218131665, 0.5664636390902013, 0.4474191644024001},
         {0.9464239829728659, 0.5599324286985411, 0.44149160585177294},
         {0.9443466594368136, 0.5533482500391552, 0.4355830907698924},
         {0.9421845883825966, 0.546711362595224, 0.42969419537788606},
         {0.9399381182692833, 0.5400219992790316, 0.4238254886976179},
         {0.9376076051853501, 0.533280364109342, 0.4179775325507263},
         {0.9351934128755736, 0.5264866296872404, 0.4121508815610997},
         {0.9326959127654463, 0.5196409344481697, 0.40634608316087356},
         {0.9301154839831494, 0.5127433796649743, 0.40056367760003836},
         {0.9274525133791317, 0.5057940261733677, 0.39480419795975347},
         {0.9247073955433249, 0.49879289078734274, 0.38906817016946776},
         {0.9218805328200373, 0.4917399423674949, 0.38335611302795025},
         {0.9189723353205592, 0.4846350974999652, 0.37766853822834884},
         {0.9159832209335118, 0.47747821573754495, 0.37200595038739454},
         {0.9129136153329698, 0.47026909434728564, 0.3663688470788784},
         {0.90976395198439, 0.46300746250050845, 0.36075771887154084},
         {0.9065346721483711, 0.4556929748311352, 0.35517304937151345},
         {0.9032262248822704, 0.44832520427650463, 0.34961531526947287},
         {0.8998390670397057, 0.44090363410085304, 0.34408498639266283},
         {0.896373663267962, 0.43342764898501834, 0.3385825257619646},
         {0.8928304860033279, 0.42589652504602493, 0.33310838965420186},
         {0.8892100154643842, 0.41830941862630455, 0.32766302766986616},
         {0.8855127396432578, 0.4106653536635276, 0.32224688280648867},
         {0.8817391542948727, 0.4029632074170497, 0.31686039153786477},
         {0.8778897629242048, 0.3952016942845104, 0.3115039838993836},
         {0.8739650767715597, 0.3873793473900178, 0.30617808357969667},
         {0.8699656147959004, 0.37949449756134346, 0.30088310801901375},
         {0.8658919036562271, 0.37154524923423027, 0.2956194685142945},
         {0.8617444776910301, 0.3635294527231879, 0.29038757033164536},
         {0.8575238788958367, 0.35544467217440223, 0.2851878128262474},
         {0.8532306568988527, 0.3472881483602095, 0.2800205895701496},
         {0.8488653689347272, 0.33905675527607415, 0.27488628848829755},
         {0.844428579816439, 0.33074694924673165, 0.2697852920031808},
         {0.8399208619053304, 0.3223547089197083, 0.26471797718851403},
         {0.835342795079293, 0.3138754640964424, 0.2596847159323806},
         {0.8306949666991194, 0.30530401078823033, 0.25468587511031116},
         {0.825977971573034, 0.2966344091359757, 0.24972181676877747},
         {0.8211924119194148, 0.28785985982733997, 0.2447928983196327},
         {0.8163388973277163, 0.2789725532777859, 0.23989947274604145},
         {0.8114180447176049, 0.2699634839588782, 0.2350418888204895},
         {0.8064304782963232, 0.2608222196264688, 0.230220491335487},
         {0.8013768295142913, 0.25153661146923556, 0.2254356213476219},
         {0.7962577370189533, 0.24209242581295431, 0.22068761643565016},
         {0.7910738466068946, 0.2324728700992567, 0.21597681097335308},
         {0.785825811174225, 0.22265797397565007, 0.2113035364179236},
         {0.7805142906652605, 0.21262376808227257, 0.20666812161469034},
         {0.7751399520194979, 0.20234117434687338, 0.2020708931190197},
         {0.7697034691169152, 0.19177447487857652, 0.19751217553628297},
         {0.7642055227215985, 0.18087914808905534, 0.19299229188080813},
         {0.7586468004237221, 0.16959872367321424, 0.18851156395477772},
         {0.7530279965798936, 0.15786005774647732, 0.18407031274806854},
         {0.7473498122518853, 0.1455659466237694, 0.17966885886006054},
         {0.7416129551437725, 0.13258300379739485, 0.1753075229444704},
         {0.7358181395374971, 0.11872050682883606, 0.17098662617829208},
         {0.729966086226886, 0.1036904371731237, 0.16670649075593325},
         {0.7240575224501452, 0.08702339888581809, 0.1624674404096517},
         {0.7180931818208612, 0.06786174152286231, 0.15826980095738155},
         {0.712073804257538, 0.04430319728532283, 0.15411390087901955},
         {0.7060001359117047, 0.015991824033980695, 0.15000007192220008}}};

template <typename Real>
static constexpr std::array<std::array<Real, 3>, 256> viridis_data_ = {
    {{0.2670039853213788, 0.0048725657145795975, 0.32941506855247793},
     {0.26850981914385313, 0.009602990407952114, 0.33542640725404194},
     {0.2699440291511295, 0.014623657659867702, 0.34137927634304566},
     {0.271304877824855, 0.01994002237673082, 0.3472686291318146},
     {0.27259406260318486, 0.0255617891189931, 0.3530934750332732},
     {0.27380892621369024, 0.03149509118386225, 0.3588528426972369},
     {0.27495208578509805, 0.03775096330071057, 0.36454366367333946},
     {0.27602196419599767, 0.04416532238909473, 0.3701640471152313},
     {0.2770178542655486, 0.050341751402647274, 0.37571443702683854},
     {0.2779409905222335, 0.05632269668695442, 0.381191238867294},
     {0.2787908952957126, 0.06214312525024266, 0.386591645366443},
     {0.27956600607518295, 0.06783500475980075, 0.39191741935399227},
     {0.2802669260610985, 0.07341543578584622, 0.3971628433351474},
     {0.2808940110629919, 0.07890627296909186, 0.402329588370124},
     {0.2814459462121492, 0.08431870698834085, 0.4074140304183279},
     {0.2819238934697361, 0.08966415929405766, 0.4124144814203868},
     {0.2823269547340802, 0.09495395414045926, 0.4173312035070467},
     {0.28265591676374746, 0.10019440706631136, 0.42215967285462885},
     {0.2829099523467409, 0.10539218382981966, 0.42690236327520725},
     {0.2830909293924702, 0.11055163703080462, 0.43155385240026206},
     {0.2831969393871863, 0.11567940038180258, 0.4361155100499537},
     {0.2832289314889045, 0.1207758539544326, 0.4405840195472734},
     {0.2831869362255922, 0.1258463172454443, 0.4449595399107695},
     {0.28307192228564804, 0.13089406218358995, 0.4492411716459423},
     {0.28288394120635924, 0.1359185257084681, 0.45342671165618736},
     {0.2826229022373855, 0.1409252633679265, 0.4575173087262126},
     {0.28228993571972105, 0.14591072658175047, 0.46150987016675354},
     {0.28188687220607433, 0.1508804582163323, 0.46540543189260436},
     {0.2814119198278991, 0.15583292129399035, 0.4692010151483717},
     {0.28086783263625503, 0.16077064757597467, 0.4728995412212838},
     {0.28025489346354565, 0.16569211148611415, 0.47649814467710355},
     {0.2795739667261627, 0.17059758179728512, 0.47999676019204823},
     {0.2788258570423396, 0.1754892977307652, 0.48339725816565043},
     {0.27801194338413965, 0.1803657682817326, 0.4866968956818299},
     {0.27713381181214125, 0.18522747944269774, 0.4898983578428951},
     {0.27619391034977797, 0.19007295088871226, 0.49300101709660177},
     {0.2751907580349635, 0.19490465707076896, 0.49600544367769356},
     {0.2741278682015853, 0.19972012989770083, 0.4989111236465034},
     {0.2730059901819072, 0.20451860824383267, 0.501720814888259},
     {0.2718278175684958, 0.2093023055918103, 0.5044342141093779},
     {0.270594950010942, 0.21406778548878086, 0.5070519265124999},
     {0.269307759397765, 0.21881747729569145, 0.5095772914792083},
     {0.26796790172934454, 0.22354795925559695, 0.5120080244602125},
     {0.26657969502130613, 0.22826164495517987, 0.5143493563955112},
     {0.2651448458327315, 0.23295512949133207, 0.5165991088272563},
     {0.2636630077044143, 0.23762961892442647, 0.518761871180001},
     {0.2621377837546321, 0.2422852962306279, 0.5208371781925083},
     {0.2605709530063988, 0.2469207884353261, 0.5228279594551629},
     {0.2589647163025411, 0.25153645885437026, 0.5247362360674637},
     {0.25732189181335147, 0.2561289544795441, 0.526563035336139},
     {0.2556446444203948, 0.2607026175814763, 0.5283122832934375},
     {0.25393482507335086, 0.26525311670734747, 0.529983099667603},
     {0.2521940157752079, 0.2697816202890161, 0.5315789243388298},
     {0.25042475493562577, 0.27428927505771267, 0.5331031508732055},
     {0.24862894949587117, 0.27877378236392025, 0.5345559912288645},
     {0.2468106816724398, 0.2832364292997213, 0.535941192655048},
     {0.24497187865778625, 0.28767394082010833, 0.5372600476888723},
     {0.24311260591427697, 0.2920915797036902, 0.5385162258554254},
     {0.24123680455606578, 0.29648409536913767, 0.5397090946733651},
     {0.23934601183850307, 0.30085361498147906, 0.5408439702969619},
     {0.23744072960970264, 0.3052012458833173, 0.54192113151443},
     {0.23552593740524197, 0.30952576959302097, 0.5429440191062838},
     {0.23360265375847195, 0.3138273924379346, 0.5439141607822711},
     {0.2316738606254108, 0.31810492063632195, 0.5448340596245334},
     {0.22973857739190007, 0.3223605353157074, 0.545706183707169},
     {0.22780178259701628, 0.3265930678980134, 0.5465320928009385},
     {0.2258629950192772, 0.3308036038834664, 0.547314007363377},
     {0.22392470580707102, 0.33499321123505194, 0.5480531185864527},
     {0.22198891605799553, 0.33915975136273824, 0.5487520417750932},
     {0.22005663012512805, 0.34330635098841344, 0.5494131386548431},
     {0.2181298368119725, 0.34743089526543186, 0.550038069892983},
     {0.21620955549289145, 0.3515344874402382, 0.5506271540883877},
     {0.21429775792403594, 0.3556180358438001, 0.5511840927098516},
     {0.21239496605173308, 0.3596815871313281, 0.5517100354247325},
     {0.21050268170035974, 0.36372617292766, 0.5522061103879761},
     {0.20862288591525807, 0.36775072789850616, 0.5526750588505742},
     {0.20675560837319001, 0.3717573067826811, 0.5531171238472018},
     {0.20490280700807212, 0.3757448655138847, 0.5535330780419422},
     {0.203062536964648, 0.3797154379631177, 0.5539251341179698},
     {0.20123872979412472, 0.3836690001875795, 0.554294093380729},
     {0.1994294675547129, 0.3876065665572156, 0.5546421417067892},
     {0.19763565582691228, 0.39152713200801975, 0.5549691052245624},
     {0.1958598486981541, 0.39543169955970453, 0.5552760714931869},
     {0.19409958576782552, 0.3993222611485653, 0.5555651140662646},
     {0.19235677239756216, 0.40319783169922974, 0.5558360841919235},
     {0.19063051802725675, 0.4070603881536437, 0.5560891205440711},
     {0.18892269821185473, 0.41090896152858886, 0.5563260943031525},
     {0.18723045233808386, 0.4147455132535933, 0.5565471249886993},
     {0.185555627191317, 0.4185690891632772, 0.5567531019269415},
     {0.18389780714307888, 0.42238166634427127, 0.5569440809497231},
     {0.18225556044836397, 0.4261832146898191, 0.5571201071543374},
     {0.1806287343914301, 0.4299737941881375, 0.5572820892587651},
     {0.17901849569766765, 0.43375533862493676, 0.5574301104997077},
     {0.1774226636398102, 0.4375259202510252, 0.55756509546256},
     {0.17584043270381128, 0.4412894613100537, 0.5576851118285253},
     {0.17427359528182015, 0.4450430447202011, 0.557792099701029},
     {0.17271876393003963, 0.4487896287788335, 0.557885089501858},
     {0.1711755310919842, 0.45252916766885326, 0.5579651016381476},
     {0.1696456943850158, 0.4562607533360476, 0.5580300944043132},
     {0.1681254681463012, 0.4599872896301423, 0.5580821016030234},
     {0.1666166265008438, 0.46370687659642523, 0.5581190973303202},
     {0.16511640601237285, 0.46742241082026353, 0.5581410993753088},
     {0.1636245598287687, 0.47113199888460483, 0.5581480982786068},
     {0.1621417205544945, 0.4748365872638394, 0.5581400991890099},
     {0.16066449675721917, 0.4785391201733818, 0.5581150965608792},
     {0.15919365349458042, 0.48223570943504873, 0.5580731010879704},
     {0.15772843426155927, 0.48593124098759627, 0.5580130923782937},
     {0.1562695869579672, 0.4896228308128307, 0.5579361006261909},
     {0.1548143719173188, 0.49331236133238937, 0.5578400853105705},
     {0.1533635209362011, 0.4969989517366848, 0.5577240977120556},
     {0.1519176771981809, 0.5006835421001327, 0.5575871128989297},
     {0.15047545758720676, 0.504368072136516, 0.557430091404037},
     {0.14903861048813108, 0.5080496628380102, 0.5572501110986087},
     {0.1476063950561745, 0.5117321922659829, 0.5570490817856095},
     {0.14617954401090907, 0.5154117832267491, 0.5568231063967104},
     {0.14475833306803976, 0.519092312358787, 0.5565720684276931},
     {0.14334247788386512, 0.522771903366683, 0.5562950984952888},
     {0.141934629618203, 0.5264514940838813, 0.5559911321954153},
     {0.14053541463564725, 0.5301310233517843, 0.5556590862704173},
     {0.13914656229528236, 0.5338106140906136, 0.555298125858835},
     {0.13776935391441048, 0.5374911432800976, 0.5549060695018647},
     {0.13640749570156585, 0.5411717339632349, 0.5544831157469011},
     {0.13506529616957136, 0.544852263189304, 0.554029048083696},
     {0.13374243075510836, 0.5485338538220298, 0.5535411013024801},
     {0.13244357030675505, 0.5522144442800521, 0.553018159322021},
     {0.13117137121614486, 0.5558979736087772, 0.5524590815222541},
     {0.12993250302214393, 0.5595805638443141, 0.5518641468320762},
     {0.12872831884644725, 0.5632640933331003, 0.5512290558865701},
     {0.12756743939839627, 0.5669476835219266, 0.5505561295873137},
     {0.12645227472350737, 0.5706322130377723, 0.5498410244005342},
     {0.12539338154311577, 0.5743168031831126, 0.5490861069146677},
     {0.12439424184336756, 0.5780013325949876, 0.5482869865182912},
     {0.12346233479357416, 0.5816859227477941, 0.5474450777703374},
     {0.122605426766972, 0.5853695126930177, 0.5465571752923358},
     {0.12183030409088419, 0.5890540421334565, 0.5456230413386914},
     {0.1211473750007544, 0.592737632144662, 0.5446411489781364},
     {0.12056429075653713, 0.5964211612480832, 0.5436109978665574},
     {0.12009133699819924, 0.6001027515502533, 0.5425301161500574},
     {0.11973729926029653, 0.6037842799987541, 0.541399947187814},
     {0.11951131929191963, 0.6074628706490315, 0.5402180758379009},
     {0.11942233180714437, 0.6111394611583431, 0.5389822116295199},
     {0.11948233048502255, 0.6148159892453784, 0.537692026866569},
     {0.11969830860262576, 0.6184885802617517, 0.5363471743826108},
     {0.12008036996652452, 0.6221601072840744, 0.5349459699054088},
     {0.12063730989641286, 0.6258266990315783, 0.5334881295553462},
     {0.1213794417235072, 0.6294912247800063, 0.5319729045347256},
     {0.12231134108889466, 0.6331518172152074, 0.5303980764654408},
     {0.1234432274542607, 0.6368074097895444, 0.5287632561537092},
     {0.12477941321977204, 0.6404599344231255, 0.5270680133779303},
     {0.12632525332983902, 0.6441055281645606, 0.5253112061650272},
     {0.1280865204366354, 0.6477480507159987, 0.5234909414076758},
     {0.1300663115014182, 0.6513826457355586, 0.5216081476931036},
     {0.13226766345372454, 0.6550131658699597, 0.5196608602452112},
     {0.1346914034616326, 0.6586347623899736, 0.5176490803131216},
     {0.13733813187439356, 0.6622503590720668, 0.5155713087695435},
     {0.1402095413205375, 0.6658578775169739, 0.5134270017152724},
     {0.14330221816701194, 0.6694574760216492, 0.511215244517751},
     {0.14661571262857995, 0.6730489911445044, 0.5089359136232384},
     {0.15014733747892414, 0.6766295918037593, 0.5065891708280766},
     {0.1538939151679172, 0.6802021032660456, 0.5041718156197705},
     {0.15785048833835377, 0.6837637060778221, 0.5016860876044946},
     {0.162015053737826, 0.6873143096440719, 0.49912936913057365},
     {0.1663826794355118, 0.6908548182506177, 0.49650199274870527},
     {0.17094719748923695, 0.6943824243247052, 0.4938032891165472},
     {0.1757068993715085, 0.6978989283401605, 0.4910328874274477},
     {0.18065236935056878, 0.7014005373289202, 0.4881891991816308},
     {0.18578314370608157, 0.704890036301888, 0.48527277191686385},
     {0.19108956697744361, 0.7083646483099804, 0.48228409897291785},
     {0.19656998786638302, 0.7118252614850998, 0.4792214366155811},
     {0.20221879696964223, 0.7152707565867668, 0.47608398702197946},
     {0.2080291781284243, 0.7186993731402823, 0.47287333975819085},
     {0.21400005157818647, 0.7221128620832928, 0.4695878639271211},
     {0.22012339076765558, 0.7255074824139881, 0.46622623255251056},
     {0.22639732501345455, 0.7288869647642494, 0.46278873032462237},
     {0.2328146240063111, 0.7322455890831692, 0.45927711461163473},
     {0.2393729244291451, 0.7355862151598742, 0.4556885106741981},
     {0.2460698827335157, 0.7389086924367883, 0.452023984943217},
     {0.25289815079190964, 0.742209322598094, 0.44828439600718756},
     {0.2598571632843528, 0.745490792270686, 0.4444668436999196},
     {0.2669403953805219, 0.7487494270612788, 0.44057327076578856},
     {0.27414945925613216, 0.7519868887012355, 0.43660069156818965},
     {0.28147665665716626, 0.755201528324013, 0.43255213457796365},
     {0.2889217691195198, 0.7583929812556258, 0.42842552870164097},
     {0.2964789382360529, 0.7615596257155389, 0.4242229866985808},
     {0.3041471124530943, 0.7647022727239619, 0.41994345744752104},
     {0.31192524014449496, 0.7678207187194435, 0.41558582710182396},
     {0.31980838304287407, 0.7709123713051562, 0.41115231354707876},
     {0.32779655496333804, 0.7739788075712202, 0.40663965647349865},
     {0.33588466791449195, 0.7770164660401522, 0.4020491587042435},
     {0.34407488193371133, 0.7800278920656099, 0.3973804752308849},
     {0.3523599689997323, 0.7830095562872507, 0.3926359923281875},
     {0.36074006346245197, 0.7859622236060678, 0.3878145232212208},
     {0.36921428939455786, 0.7888866412701451, 0.38291381435701694},
     {0.37777835651210034, 0.7917793150973379, 0.37793936040098913},
     {0.3864336209519174, 0.7946427215203885, 0.3728856256306759},
     {0.3951736619954228, 0.797473402038798, 0.3677571865876491},
     {0.4040019632351159, 0.8002737966998706, 0.3625514261814766},
     {0.4129129804482464, 0.8030394840252982, 0.3572690015831872},
     {0.4219070059826357, 0.805772174900056, 0.3519105924567475},
     {0.4309833172715677, 0.8084715598896897, 0.3464758054878499},
     {0.4401363189373519, 0.8111362580724669, 0.34096741055434293},
     {0.44936866381374757, 0.8137666304213753, 0.3353835991203212},
     {0.458673642833963, 0.8163613360822743, 0.32972721765157326},
     {0.46805401938353625, 0.8189196953983692, 0.3239973827222524},
     {0.47750397699915853, 0.8214424087022711, 0.3181950138984179},
     {0.48702494316955264, 0.8239271261653897, 0.31232166294182256},
     {0.49661532811633474, 0.8263744743966721, 0.306376800619062},
     {0.5062702739767431, 0.8287841998369972, 0.3003624619571904},
     {0.5159926874598648, 0.8311565342261135, 0.2942785778807102},
     {0.5257756141542576, 0.8334892678013092, 0.28812725033290043},
     {0.5356220541188685, 0.8357835879688815, 0.28190734645112897},
     {0.5455239631257167, 0.838037329833802, 0.2756260283833135},
     {0.5554828787549823, 0.8402520764412206, 0.2692817313403921},
     {0.5654983251339641, 0.8424283847332169, 0.2628767994071899},
     {0.5755622254006186, 0.8445641394621258, 0.25641551104548627},
     {0.5856786942007215, 0.8466594333420596, 0.24989656346356887},
     {0.5958385796727564, 0.8487151964088427, 0.2433292811877542},
     {0.6060460684908495, 0.8507314758177834, 0.2367113216835447},
     {0.616292940650758, 0.8527072471488548, 0.23005204277336477},
     {0.6265778183939344, 0.8546430238400886, 0.2233537871857682},
     {0.6369023105271391, 0.8565402909227283, 0.2166198017460239},
     {0.6472561782044223, 0.8583980753432965, 0.2098615478515251},
     {0.6576426853779066, 0.8602173283504794, 0.20308155987152815},
     {0.6680535433907611, 0.8619971207604976, 0.19629330158139313},
     {0.6784900623681449, 0.8637403601361504, 0.18950231977185272},
     {0.6889439127328617, 0.86544616002535, 0.18272505128011118},
     {0.6994137684819729, 0.8671149653844072, 0.17597180349415806},
     {0.7098982863987701, 0.868749193048019, 0.1692568079838667},
     {0.7203901380970731, 0.8703480050687866, 0.16260354455700873},
     {0.7308896616716263, 0.8719142204262896, 0.1560285800997179},
     {0.7413875101130836, 0.873447039008859, 0.1495612865358065},
     {0.7518850354669301, 0.8749492431662675, 0.14322737498076443},
     {0.7623728828501631, 0.8764220675983936, 0.13706403823168214},
     {0.7728534061279669, 0.8778662616531304, 0.13110820913488913},
     {0.7833152551224711, 0.8792830913055214, 0.1254048217206159},
     {0.7937591112086629, 0.880675925144139, 0.12000541978485552},
     {0.8041826246822775, 0.8820441108502998, 0.11496466377053949},
     {0.8145754845039437, 0.8833909489919768, 0.11034716773962891},
     {0.8249409891695173, 0.8847181273467387, 0.10621658195896774},
     {0.8352698545299048, 0.8860269688856748, 0.1026459672214125},
     {0.8455623467619914, 0.8873201416511192, 0.09970160643955417},
     {0.8558102197403156, 0.8885989857373076, 0.09745185636949635},
     {0.8660121015321746, 0.8898658327014707, 0.09595302142849352},
     {0.876168577861264, 0.8911230006650279, 0.09524987622820644},
     {0.8862704693847823, 0.8923718488516762, 0.09537386433165168},
     {0.8963209276849421, 0.8936140145545759, 0.09633501924437288},
     {0.9063108305457804, 0.8948528630958348, 0.0981248266153556},
     {0.916243267627844, 0.8960890285051739, 0.10071727474383468},
     {0.9261061834911258, 0.8973278764153646, 0.10407091253184719},
     {0.9359031089867296, 0.8985677257631529, 0.10813047743488122},
     {0.9456365256783655, 0.899812890310841, 0.112838121119233},
     {0.9552994651064829, 0.9010627378634819, 0.11812754907477127},
     {0.9648948576822979, 0.9023209051300205, 0.12394140017411451},
     {0.9744168120052771, 0.9035877501583347, 0.13021471316475502},
     {0.9838691793408701, 0.9048649215326077, 0.13689772464133854},
     {0.9932481489335602, 0.9061547634208059, 0.14393594366968385}}};

template <typename Real>
inline std::array<Real, 3>
color_map_(Real scalar, std::array<std::array<Real, 3>, 256> const &table) {
  static_assert(std::is_floating_point<Real>::value,
                "Color tables are only implemented in floating point "
                "arithmetic. If you require bytes please submit an issue or "
                "pull request");

  scalar = std::clamp(scalar, static_cast<Real>(0), static_cast<Real>(1));

  if (scalar == static_cast<Real>(1)) {
    return table.back();
  }

  Real s = (table.size() - 1) * scalar;
  Real ii = std::floor(s);
  Real t = s - ii;
  std::size_t i = static_cast<std::size_t>(ii);
  auto const &rgb0 = table[i];
  auto const &rgb1 = table[i + 1];
  return {(1 - t) * rgb0[0] + t * rgb1[0], (1 - t) * rgb0[1] + t * rgb1[1],
          (1 - t) * rgb0[2] + t * rgb1[2]};
}
} // namespace detail

template <typename Real = float> std::array<Real, 3> viridis(Real x) {
  return detail::color_map_<Real>(x, detail::viridis_data_<Real>);
}

template <typename Real = float> std::array<Real, 3> smooth_cool_warm(Real x) {
  return detail::color_map_<Real>(x, detail::smooth_cool_warm_data_<Real>);
}

template <typename Real = float> std::array<Real, 3> plasma(Real x) {
  return detail::color_map_<Real>(x, detail::plasma_data_<Real>);
}

template <typename Real = float> std::array<Real, 3> black_body(Real x) {
  return detail::color_map_<Real>(x, detail::black_body_data_<Real>);
}

template <typename Real = float> std::array<Real, 3> inferno(Real x) {
  return detail::color_map_<Real>(x, detail::inferno_data_<Real>);
}

template <typename Real = float> std::array<Real, 3> kindlmann(Real x) {
  return detail::color_map_<Real>(x, detail::kindlmann_data_<Real>);
}

template <typename Real = float>
std::array<Real, 3> extended_kindlmann(Real x) {
  return detail::color_map_<Real>(x, detail::extended_kindlmann_data_<Real>);
}

template <typename Real>
std::array<std::uint8_t, 4> to_8bit_rgba(const std::array<Real, 3> &v) {
  using std::sqrt;
  std::array<std::uint8_t, 4> pixel;
  for (auto i = 0; i < 3; ++i) {
    // Apply gamma correction here:
    Real u = sqrt(v[i]);
    pixel[i] = 255 * std::clamp(u, static_cast<Real>(0), static_cast<Real>(1));
  }

  pixel[3] = 255;
  return pixel;
}

} // Namespace boost::math::tools

#endif // BOOST_MATH_COLOR_MAPS_HPP
