/* XRACER (C) 1999-2000 Richard W.M. Jones <rich@annexia.org> and other AUTHORS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: xracer-file-browser.h,v 1.1 2000/01/15 16:14:35 rich Exp $
 */

#ifndef __xracer_file_browser_h__
#define __xracer_file_browser_h__

#include "xracer-PATH_MAX.h"
#include "xracer-NAME_MAX.h"

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

struct xrFileBrowserCallbacks
{
  /* This is called to filter particular directory entries. Return true
   * if you want that directory entry to show up. Setting this to NULL
   * will show every file in the directory.
   */
  int (*filter) (const char *dirname, const char *filename,
		 const struct stat *stat);

  /* When the user calls xrFileBrowserDisplay, this function is called
   * once per file in the directory (it is not called for filtered
   * files, however). The INDEX argument ranges from 0 to the number
   * of files - 1. This function should return 0, or -1 if an error
   * occurs. If it returns -1, then the xrFileBrowserDisplay function
   * will immediately stop and also return -1.
   */
  int (*display) (int index, const char *dirname, const char *filename,
		  const struct stat *stat, void *args);
};

struct xrFileBrowserDirectoryEntry
{
  char *filename;
  struct stat stat;
  int show;
};

struct xrFileBrowserDirectoryContents
{
  int nr_entries;
  struct xrFileBrowserDirectoryEntry *entries;
};

struct xrFileBrowserObject
{
  /* The current directory which we are browsing. */
  char cwd[PATH_MAX];

  /* It would be really inefficient to reread the directory every time
   * we called display(), so instead stash the contents of the current
   * directory here.
   */
  struct xrFileBrowserDirectoryContents contents;

  /* Callbacks structure, supplied by the user. */
  const struct xrFileBrowserCallbacks *callbacks;
};

/* Program-level file browser initializations. */
extern void xrFileBrowserInit (void);

/* Create a file browser object for the given directory. The pathname
 * given may be either relative or absolute. If relative, it is
 * taken relative to the current process's working directory.
 */
extern void *xrFileBrowserCreate (const char *pathname,
			       const struct xrFileBrowserCallbacks *callbacks);

/* Display each filename in current directory. */
extern int xrFileBrowserDisplay (void *file_browser, void *args);

/* Change directory. The path given may be relative or absolute. If
 * relative, it is taken relative to the currently held directory.
 */
extern int xrFileBrowserChangeDirectory (void *file_browser,
					 const char *pathname);

/* Reload directory. */
extern int xrFileBrowserReload (void *file_browser);

/* Change the callbacks. */
extern void xrFileBrowserSetCallbacks (void *file_browser,
			       const struct xrFileBrowserCallbacks *callbacks);

/* Return number of entries in current directory (after filtering). */
extern int xrFileBrowserGetNrEntries (void *file_browser);

/* Return the current directory. */
extern const char *xrFileBrowserGetDirectory (void *file_browser);

/* Return details of the nth item (after filtering). */
extern void xrFileBrowserGetFileInfo (void *file_browser, int index,
				      const char **filename_rtn,
				      const struct stat **stat_rtn);

/* Dispose of file browser object. */
extern void xrFileBrowserDelete (void *file_browser);

#endif /* __xracer_file_browser_h__ */
