@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-job-list: List all ECaccess Jobs
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use Term::ReadKey;

my %opt = ( version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $jobId = $ARGV[0];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

if ( not($jobId) ) {

	# Get the list of jobs
	my $jobs = $controlChannel->getJobList($token);

	# Display the information for each job
	foreach $job ( $jobs->valueof('//getJobListResponse/return') ) {
		printf "%-10s %-10s %-10s %-6s %-15s %-15s", $job->{jobId}, $job->{queueName}, $job->{status}, $job->{tryDone} . "/" . $job->{tryCount},
		  $job->{scheduledDate}, '(' . ( $job->{eventIds} ? $job->{eventIds} : '-' ) . ')';
		print " (" . $job->{name} . ")" if ( $job->{name} );
		print "\n";
	}
}
else {

	# Get the detail for the specified jobId
	my $job = $controlChannel->getJob( $token, $jobId )->valueof('//getJobResponse/return');
	print "     Job-Id: " . $job->{jobId} . "\n";
	print "   Job Name: " . $job->{name} . "\n" if ( $job->{name} );
	print "      Queue: " . $job->{queueName} . "\n";
	print "       Host: " . $job->{hostName} . "\n" if ( $job->{hostName} );
	print "   Schedule: " . $job->{scheduledDate} . "\n";
	print " Expiration: " . $job->{expirationDate} . "\n";
	print "  Try Count: " . $job->{tryCount} . "\n";
	print "     Status: " . $job->{status} . "\n";
	print "  Event-Ids: " . $job->{eventIds} . "\n" if ( $job->{eventIds} );

	if ( $job->{status} eq 'DONE' ) {
		print "Stdout Size: " . $job->{outputFileSize} . "\n" if ( $job->{outputFileSize} != '-1' );
		print "Stderr Size: " . $job->{errorFileSize} . "\n"  if ( $job->{errorFileSize} != '-1' );
		print " Stdin Size: " . $job->{inputFileSize} . "\n"  if ( $job->{inputFileSize} != '-1' );
	}

	print "    Comment: " . $job->{comment} . "\n" if ( $job->{comment} );
}

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-job-list - List all ECaccess Jobs

=head1 SYNOPSIS

B<ecaccess-job-list -version|-help|-manual>

B<ecaccess-job-list [-debug]> B<[>I<job-id>B<]>

=head1 DESCRIPTION

List all your ECaccess Jobs. When a I<job-id> is specified
then the details for this job are displayed.

The Job statuses can have the following values:

=over 8

=item B<INIT>

Jobs are being initialised

=item B<STDBY>

Jobs are waiting for an event

=item B<EXEC>

Jobs are running

=item B<WAIT>

Jobs have been queued to the scheduler (e.g. LoadLeveler)

=item B<RETR>

Jobs will be resubmitted

=item B<STOP>

Jobs have NOT completed (error)

=item B<DONE>

Jobs have successfully completed

=back

=head1 ARGUMENTS

=over 8

=item I<job-id> (optional)

The identifier of the ECaccess Job to retrieve the details.

=back

=head1 OPTIONS

=over 8

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-job-list> I<124356>

Give the informations concerning the ECaccess Job with identifier I<124356>.

B<ecaccess-job-list>

List all your ECaccess Jobs in the spool.

=head1 SEE ALSO

B<ecaccess-queue-list>, B<ecaccess-job-delete>, B<ecaccess-job-get>, B<ecaccess-job-restart>,
B<ecaccess-job-submit> and B<ecaccess>.

=cut

__END__
:endofperl
