#ifndef TC_API_H
#define TC_API_H

#include <stdint.h>
#include <stdbool.h>
#include <stdlib.h>
#include "tc_type.h"

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************
描述：初始化可信计算环境，根据传入类型，与可信设备建立连接，
返回连接信息句柄。

    @device ：可信设备路径 或 代理连接名称 
         例 ：/dev/tpm0或tabrmd
    @handle ：可信设备服务句柄

返回值: 0, 成功；其他为失败。
 ******************************************************/
TC_RC TC_Start (
    const uint8_t           *device,          //设备路径
    TC_HANDLE               *handle);         //信息句柄

/******************************************************
描述： 释放在可信计算过程中分配的可信计算资源，释放句柄，
断开连接。

    @handle ：可信设备服务句柄

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_End (
    TC_HANDLE               *handle);         //信息句柄

/******************************************************
描述： 密钥或对象创建，创建主键，作为后续创建的密钥/对象的父节点。
注：任意可信设备中创建的密钥/对象都需寄生在主键的保护中。

    @handle ：可信设备服务句柄
    @alg_hash ：摘要算法
    @hierarchy ：Hierarchy 类型
    @hierarchy_auth_msg ：Hierarchy 认证信息
    @alg_primary ：主密钥算法
    @primary_auth_msg ：主密钥授权认证信息
    @primary_index ：返回主密钥索引

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_CreatePrimary (
    TC_HANDLE              handle,
    const TC_ALG           alg_hash,
    const uint32_t         hierarchy,
    const TC_BUFFER       *hierarchy_auth_msg,
    const TC_ALG           alg_primary,
    const TC_BUFFER       *primary_auth_msg,
    uint32_t              *primary_index);

/******************************************************
描述： 创建对象/密钥，创建的密钥对象详细信息将封存在 信息句柄中，
用户不可见；可以通过返回的对象索引编号使用创建的对象/密钥。

    @handle ：可信设备服务句柄
    @alg_hash ：摘要算法
    @primary_index ：主密钥索引
    @primary_auth_msg ：主密钥授权认证信息
    @alg_key ：子密钥算法
    @key_auth_msg ：子密钥授权认证信息
    @key_index ：返回子密钥索引

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_Create (
    TC_HANDLE              handle,
    const TC_ALG           alg_hash,
    const uint32_t         primary_index,
    const TC_BUFFER       *primary_auth_msg,
    const TC_ALG           alg_key,
    const TC_BUFFER       *key_auth_msg,
    uint32_t              *key_index);

/******************************************************
描述： 将对象/密钥的索引编号导入可信设备中，返回在可信设备中的索
引编号地址；后续通过使用在可信设备中的索引编号地址即可使用可
信设备的功能接口。

    @handle ：可信设备服务句柄
    @key_index ：密钥索引
    @parent_auth_msg ：父键授权验证信息

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_Load (
    TC_HANDLE              handle,
    const uint32_t         key_index,
    const TC_BUFFER       *parent_auth_msg);

/******************************************************
描述： 临时对象持久化或驱逐持久性对象，将已导入可信设备中密钥
对象的索引编号地址设置为持久存储在可信设备中，并且设置在
指定 hierarchy 下保护将设置为持久存储在可信设备中的密钥
对象取消持久存储。

    @handle ：可信设备服务句柄
    @enable ：使能/失能
    @persist_index ：持久索引
    @key_index ：密钥索引
    @hierarchy: Hierarchy 类型
    @hierarchy_auth_msg ：Hierarchy 认证信息

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_EvictControl (
    TC_HANDLE              handle,
    const bool             enable,
    const uint32_t         persist_index,
    const uint32_t         key_index,
    const uint32_t         hierarchy,
    const TC_BUFFER       *hierarchy_auth_msg);

/******************************************************
描述： 数据加密计算

    @handle ：可信设备服务句柄
    @key_index ：密钥索引
    @key_auth_msg ：授权验证信息
    @alg_encrypt: 加密算法
    @plain_text ：明文信息
    @ciphter_text ：返回密文信息

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_Encrypt (
    TC_HANDLE              handle,
    const uint32_t         key_index,
    const TC_BUFFER       *key_auth_msg,
    const TC_ALG           alg_encrypt,
    const TC_BUFFER       *plain_text,
    TC_BUFFER             *ciphter_text);

/******************************************************
描述： 数据解密计算

    @handle ：可信设备服务句柄
    @key_index ：密钥索引
    @key_auth_msg ：密钥对象授权秘密验证信息
    @alg_decrypt: 解密算法
    @ciphter_text ：密文信息
    @plain_text ：返回明文信息

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_Decrypt (
    TC_HANDLE              handle,
    const uint32_t         key_index,
    const TC_BUFFER       *key_auth_msg,
    const TC_ALG           alg_decrypt,
    const TC_BUFFER       *ciphter_text,
    TC_BUFFER             *plain_text);

/******************************************************
描述： 根据给定摘要算法计算数据摘要

    @handle ：可信设备服务句柄
    @alg_hash: 摘要算法
    @plain_text ：明文信息
    @digest_msg ：返回摘要结果

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_Hash (
    TC_HANDLE              handle,
    const TC_ALG           alg_hash,
    const TC_BUFFER       *plain_text,
    TC_BUFFER             *digest_msg);

/******************************************************
描述：给定签名数据及相关算法，对其进行签名计算

    @handle ：可信设备服务句柄
    @key_index ：密钥索引
    @key_auth_msg ：授权验证信息
    @alg_sign: 签名算法
    @alg_hash: 摘要算法
    @plain_text ：待签名信息
    @sign_text ：返回签名结果

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_Sign (
    TC_HANDLE              handle,
    const uint32_t         key_index,
    const TC_BUFFER       *key_auth_msg,
    const TC_ALG           alg_sign,
    const TC_ALG           alg_hash,
    const TC_BUFFER       *plain_text,
    TC_BUFFER             *sign_text);

/******************************************************
描述： 验证签名

    @handle ：可信设备服务句柄
    @key_index ：密钥索引
    @key_auth_msg ：授权验证信息
    @alg_sign: 签名算法
    @alg_hash: 摘要算法
    @plain_text ：待验证信息
    @sign_text ：签名信息
    @verify: 返回验证结果。1，验证成功；0，验证失败

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_VerifySignature (
    TC_HANDLE              handle,
    const uint32_t         key_index,
    const TC_BUFFER       *key_auth_msg,
    const TC_ALG           alg_sign,
    const TC_ALG           alg_hash,
    const TC_BUFFER       *plain_text,
    const TC_BUFFER       *sign_text,
    unsigned int          *verify);

/******************************************************
描述： 定义 NV

    @handle ：可信设备服务句柄
    @nv_index ：NV 索引
    @hierarchy: Hierarchy 类型
    @size: 定义空间大小
    @hierarchy_auth_msg ：Hierarchy 认证信息

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_NVDefine (
    TC_HANDLE              handle,
    const uint32_t         nv_index,
    const uint32_t         hierarchy,
    const uint32_t         size,
    const TC_BUFFER       *hierarchy_auth_msg);

/******************************************************
描述： 释放 NV

    @handle ：可信设备服务句柄
    @nv_index ：NV 索引
    @hierarchy: Hierarchy 类型
    @hierarchy_auth_msg ：Hierarchy 认证信息

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_NVRelease (
    TC_HANDLE              handle,
    const uint32_t         nv_index,
    const uint32_t         hierarchy,
    const TC_BUFFER       *hierarchy_auth_msg);

/******************************************************
描述： 读取 NV

    @handle ：可信设备服务句柄
    @nv_index ：NV 索引
    @hierarchy: Hierarchy 类型
    @hierarchy_auth_msg ：Hierarchy 认证信息
    @offset_read ：读取数据大小
    @size_read ：读取数据初始位置（偏移量）
    @nv_data_read ：返回读取数据

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_NVRead (
    TC_HANDLE              handle,
    const uint32_t         nv_index,
    const uint32_t         hierarchy,
    const TC_BUFFER       *hierarchy_auth_msg,
    const uint32_t         offset_read,
    const uint32_t         size_read,
    TC_BUFFER             *nv_data_read);

/******************************************************
描述： 写入 NV

    @handle ：可信设备服务句柄
    @nv_index ：NV 索引
    @hierarchy: Hierarchy 类型
    @hierarchy_auth_msg ：Hierarchy 认证信息
    @offset_written ：写入数据初始位置（偏移量）
    @nv_data_written ：写入数据内容

返回值: 0, 成功， 其他为失败。
******************************************************/
TC_RC TC_NVWrite (
    TC_HANDLE              handle,
    const uint32_t         nv_index,
    const uint32_t         hierarchy,
    const TC_BUFFER       *hierarchy_auth_msg,
    const uint32_t         offset_written,
    const TC_BUFFER       *nv_data_written);

#ifdef __cplusplus
}
#endif

#endif /* TC_API_H */