#!/usr/bin/env zsh

emulate zsh -o extended_glob -o no_function_argzero -o no_unset

##
# Compares options in `rg --help` output to options in zsh completion function

get_comp_args() {
    # Technically there are many options that the completion system sets that
    # our function may rely on, but we'll trust that we've got it mostly right
    setopt local_options unset

    # Our completion function recognises a special variable which tells it to
    # dump the _arguments specs and then just return. But do this in a sub-shell
    # anyway to avoid any weirdness
    ( _RG_COMPLETE_LIST_ARGS=1 source $1 )
}

main() {
    local diff
    local  rg="${0:a:h}/../target/${TARGET:-}/release/rg"
    local _rg="${0:a:h}/../complete/_rg"
    local -a help_args comp_args

    [[ -e $rg ]] || rg=${rg/%\/release\/rg/\/debug\/rg}

     rg=${rg:a}
    _rg=${_rg:a}

    [[ -e $rg ]] || {
        print -r >&2 "File not found: $rg"
        return 1
    }
    [[ -e $_rg ]] || {
        print -r >&2 "File not found: $_rg"
        return 1
    }

    print -rl - 'Comparing options:' "-$rg" "+$_rg"

    # 'Parse' options out of the `--help` output. To prevent false positives we
    # only look at lines where the first non-white-space character is `-`, or
    # where a long option starting with certain letters (see `_rg`) is found.
    # Occasionally we may have to handle some manually, however
    help_args=( ${(f)"$(
        $rg --help |
        $rg -i -- '^\s+--?[a-z0-9]|--[imnp]' |
        $rg -ior '$1' -- $'[\t /\"\'`.,](-[a-z0-9]|--[a-z0-9-]+)\\b' |
        $rg -v -- --print0 | # False positives
        sort -u
    )"} )

    # 'Parse' options out of the completion function
    comp_args=( ${(f)"$( get_comp_args $_rg )"} )

    # Note that we currently exclude hidden (!...) options; matching these
    # properly against the `--help` output could be irritating
    comp_args=( ${comp_args#\(*\)}    ) # Strip excluded options
    comp_args=( ${comp_args#\*}       ) # Strip repetition indicator
    comp_args=( ${comp_args%%-[:[]*}  ) # Strip everything after -optname-
    comp_args=( ${comp_args%%[:+=[]*} ) # Strip everything after other optspecs
    comp_args=( ${comp_args##[^-]*}   ) # Remove non-options
    comp_args=( ${(f)"$( print -rl - $comp_args | sort -u )"} )

    (( $#help_args )) || {
        print -r >&2 'Failed to get help_args'
        return 1
    }
    (( $#comp_args )) || {
        print -r >&2 'Failed to get comp_args'
        return 1
    }

    diff="$(
        if diff --help 2>&1 | grep -qF -- '--label'; then
            diff -U2 \
                --label '`rg --help`' \
                --label '`_rg`' \
                =( print -rl - $help_args ) =( print -rl - $comp_args )
        else
            diff -U2 \
                -L '`rg --help`' \
                -L '`_rg`' \
                =( print -rl - $help_args ) =( print -rl - $comp_args )
        fi
    )"

    (( $#diff )) && {
        printf >&2 '%s\n' 'zsh completion options differ from `--help` options:'
        printf >&2 '%s\n' $diff
        return 1
    }
    printf 'OK\n'
    return 0
}

main "$@"
