// Copyright 2018 Kyle Mayes
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

extern crate libloading;

use std::env;
use std::fs::{File};
use std::io::{Read, Seek, SeekFrom};
use std::path::{Path, PathBuf};

use self::libloading::{Library};

use super::common;

/// Returns the ELF class from the ELF header in the supplied file.
fn parse_elf_header(path: &Path) -> Result<u8, String> {
    let mut file = File::open(path).map_err(|e| e.to_string())?;
    let mut elf = [0; 5];
    file.read_exact(&mut elf).map_err(|e| e.to_string())?;
    if elf[..4] == [127, 69, 76, 70] {
        Ok(elf[4])
    } else {
        Err("invalid ELF header".into())
    }
}

/// Returns the magic number from the PE header in the supplied file.
fn parse_pe_header(path: &Path) -> Result<u16, String> {
    let mut file = File::open(path).map_err(|e| e.to_string())?;
    let mut pe = [0; 4];

    // Determine the header offset.
    file.seek(SeekFrom::Start(0x3C)).map_err(|e| e.to_string())?;
    file.read_exact(&mut pe).map_err(|e| e.to_string())?;
    let offset =
        i32::from(pe[0]) +
        (i32::from(pe[1]) << 8) +
        (i32::from(pe[2]) << 16) +
        (i32::from(pe[3]) << 24);

    // Determine the validity of the header.
    file.seek(SeekFrom::Start(offset as u64)).map_err(|e| e.to_string())?;
    file.read_exact(&mut pe).map_err(|e| e.to_string())?;
    if pe != [80, 69, 0, 0] {
        return Err("invalid PE header".into());
    }

    // Find the magic number.
    file.seek(SeekFrom::Current(20)).map_err(|e| e.to_string())?;
    file.read_exact(&mut pe).map_err(|e| e.to_string())?;
    Ok(u16::from(pe[0]) + (u16::from(pe[1]) << 8))
}

/// Validates the header for the supplied `libclang` shared library.
fn validate_header(path: &Path) -> Result<(), String> {
    if cfg!(any(target_os="freebsd", target_os="linux")) {
        let class = parse_elf_header(path)?;
        if cfg!(target_pointer_width="32") && class != 1 {
            return Err("invalid ELF class (64-bit)".into());
        }
        if cfg!(target_pointer_width="64") && class != 2 {
            return Err("invalid ELF class (32-bit)".into());
        }
        Ok(())
    } else if cfg!(target_os="windows") {
        let magic = parse_pe_header(path)?;
        if cfg!(target_pointer_width="32") && magic != 267 {
            return Err("invalid DLL (64-bit)".into());
        }
        if cfg!(target_pointer_width="64") && magic != 523 {
            return Err("invalid DLL (32-bit)".into());
        }
        Ok(())
    } else {
        Ok(())
    }
}

/// Determines the version of the supplied `libclang` shared library by loading functions only
/// available on certain versions.
fn determine_version(path: &Path) -> Result<Vec<u32>, String> {
    let library = Library::new(&path).map_err(|e| {
        format!(
            "the `libclang` shared library at {} could not be opened: {}",
            path.display(),
            e,
        )
    })?;

    macro_rules! test {
        ($fn:expr, $version:expr) => {
            if library.get::<unsafe extern fn()>($fn).is_ok() {
                return Ok($version);
            }
        };
    }

    unsafe {
        test!(b"clang_File_tryGetRealPathName", vec![7, 0]);
        test!(b"clang_CXIndex_setInvocationEmissionPathOption", vec![6, 0]);
        test!(b"clang_Cursor_isExternalSymbol", vec![5, 0]);
        test!(b"clang_EvalResult_getAsLongLong", vec![4, 0]);
        test!(b"clang_CXXConstructor_isConvertingConstructor", vec![3, 9]);
        test!(b"clang_CXXField_isMutable", vec![3, 8]);
        test!(b"clang_Cursor_getOffsetOfField", vec![3, 7]);
        test!(b"clang_Cursor_getStorageClass", vec![3, 6]);
        test!(b"clang_Type_getNumTemplateArguments", vec![3, 5]);
    }

    Ok(vec![])
}

/// Returns the paths to, the filenames, and the versions of the `libclang` shared libraries.
fn search_libclang_directories(runtime: bool) -> Result<Vec<(PathBuf, String, Vec<u32>)>, String> {
    let mut files = vec![format!("{}clang{}", env::consts::DLL_PREFIX, env::consts::DLL_SUFFIX)];

    if cfg!(target_os="linux") {
        // Some Linux distributions don't create a `libclang.so` symlink, so we need to
        // look for versioned files (e.g., `libclang-3.9.so`).
        files.push("libclang-*.so".into());

        // Some Linux distributions don't create a `libclang.so` symlink and don't have
        // versioned files as described above, so we need to look for suffix versioned
        // files (e.g., `libclang.so.1`). However, `ld` cannot link to these files, so
        // this will only be included when linking at runtime.
        if runtime {
            files.push("libclang.so.*".into());
        }
    }

    if cfg!(any(target_os="openbsd", target_os="freebsd", target_os="netbsd")) {
        // Some BSD distributions don't create a `libclang.so` symlink either, but use
        // a different naming scheme for versioned files (e.g., `libclang.so.7.0`).
        files.push("libclang.so.*".into());
    }

    if cfg!(target_os="windows") {
        // The official LLVM build uses `libclang.dll` on Windows instead of `clang.dll`. However,
        // unofficial builds such as MinGW use `clang.dll`.
        files.push("libclang.dll".into());
    }

    // Validate the `libclang` shared libraries and collect the versions.
    let mut valid = vec![];
    let mut invalid = vec![];
    for (directory, filename) in common::search_libclang_directories(&files, "LIBCLANG_PATH") {
        let path = directory.join(&filename);
        match validate_header(&path).and_then(|_| determine_version(&path)) {
            Ok(version) => valid.push((directory, filename, version)),
            Err(message) => invalid.push(format!("({}: {})", path.display(), message)),
        }
    }

    if !valid.is_empty() {
        return Ok(valid);
    }

    let message = format!(
        "couldn't find any valid shared libraries matching: [{}], set the `LIBCLANG_PATH` \
        environment variable to a path where one of these files can be found (invalid: [{}])",
        files.iter().map(|f| format!("'{}'", f)).collect::<Vec<_>>().join(", "),
        invalid.join(", "),
    );

    Err(message)
}

/// Returns the requested minimum version.
fn get_minimum() -> &'static [u32] {
    macro_rules! test {
        ($feature:expr, $version:expr) => {
            if cfg!(feature=$feature) {
                return $version;
            }
        };
    }

    test!("clang_7_0", &[7, 0]);
    test!("clang_6_0", &[6, 0]);
    test!("clang_5_0", &[5, 0]);
    test!("clang_4_0", &[4, 0]);
    test!("clang_3_9", &[3, 9]);
    test!("clang_3_8", &[3, 8]);
    test!("clang_3_7", &[3, 7]);
    test!("clang_3_6", &[3, 6]);
    &[3, 5]
}

/// Returns the directory and filename of the "best" available `libclang` shared library.
pub fn find(runtime: bool) -> Result<(PathBuf, String), String> {
    let candidates = search_libclang_directories(runtime)?;
    let (path, filename, version) = candidates.iter()
        .max_by_key(|f| &f.2)
        .cloned()
        .expect("unreachable");

    // Assert that the selected version is at least as high as the requested version.
    let minimum = get_minimum();
    if cfg!(feature="assert-minimum") && &version[..] < minimum {
        return Err(format!(
            "couldn't find any valid shared libraries with a minimum version of {:?} \
            (invalid: [{}])",
            minimum,
            candidates.iter()
                .map(|c| format!("({}: {:?})", c.0.join(&c.1).display(), c.2))
                .collect::<Vec<_>>()
                .join(", "),
        ));
    }

    Ok((path, filename))
}

/// Find and link to `libclang` dynamically.
#[cfg(not(feature="runtime"))]
pub fn link() {
    use std::fs;

    let (directory, filename) = find(false).unwrap();
    println!("cargo:rustc-link-search={}", directory.display());

    if cfg!(all(target_os="windows", target_env="msvc")) {
        // Find the `libclang` stub static library required for the MSVC toolchain.
        let lib = if !directory.ends_with("bin") {
            directory.to_owned()
        } else {
            directory.parent().unwrap().join("lib")
        };

        if lib.join("libclang.lib").exists() {
            println!("cargo:rustc-link-search={}", lib.display());
        } else if lib.join("libclang.dll.a").exists() {
            // MSYS and MinGW use `libclang.dll.a` instead of `libclang.lib`. It is linkable with
            // the MSVC linker, but Rust doesn't recognize the `.a` suffix, so we need to copy it
            // with a different name.
            //
            // FIXME: Maybe we can just hardlink or symlink it?
            let out = env::var("OUT_DIR").unwrap();
            fs::copy(lib.join("libclang.dll.a"), Path::new(&out).join("libclang.lib")).unwrap();
            println!("cargo:rustc-link-search=native={}", out);
        } else {
            panic!(
                "using '{}', so 'libclang.lib' or 'libclang.dll.a' must be available in {}",
                filename,
                lib.display(),
            );
        }

        println!("cargo:rustc-link-lib=dylib=libclang");
    } else {
        let name = filename.trim_left_matches("lib");

        // Strip extensions and trailing version numbers (e.g., the `.so.7.0` in `libclang.so.7.0`).
        let name = match name.find(".dylib").or(name.find(".so")) {
            Some(index) => &name[0..index],
            None => &name,
        };

        println!("cargo:rustc-link-lib=dylib={}", name);
    }
}
