///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef.h"
using namespace rheolef;
using namespace std;
struct f1 {
  Float operator() (const point& x) const { return x[0]+2*x[1]+3*x[2]; }
};
struct f2 {
  point operator() (const point& x) const { return point(x[0]+2*x[1]+3*x[2], 5*x[1]+x[2], 3*x[2]); }
};
int main(int argc, char**argv) {
  environment rheolef(argc, argv);
  geo omega(argv[1]);
  string approx = argv[2];
  Float tol = (argc > 3) ? atof(argv[3]) : 1e-10;
  Float err = 0;
  // ------------------------- 
  // 1) grad(uh)
  // ------------------------- 
  {
    space Xh (omega, approx);
    size_t k = Xh.degree();
    string grad_approx = "P" + itos(k-1) + "d";
    space Th (omega, grad_approx, "vector");
    trial u (Xh); test v (Xh);
    trial p (Th); test q (Th);
    field uh = interpolate (Xh, f1());
    integrate_option fopt;
    fopt.invert = true;
    form inv_mass = integrate (dot(p,q), fopt);
    form mgrad = integrate (dot(grad(u),q));
    field grad_uh = inv_mass*(mgrad*uh);
  
    // 1.1) grad(uh) in integrate()
    quadrature_option qopt;
    qopt.set_order(5);
    Float err1 = integrate (omega, norm(grad(uh)-grad_uh), qopt);
    derr << "err1 = " << err1 << endl;
    err = std::max (err, err1);
  
    // 1.2) grad(uh) in interpolate()
warning_macro ("main[1.2.a]...");
    field grad_uh2 = interpolate (Th, grad(uh));
warning_macro ("main[1.2.b]...");
    Float err2 = field(grad_uh - grad_uh2).max_abs();
warning_macro ("main[1.2.c]...");
    derr << "err2 = " << err2 << endl;
    err = std::max (err, err2);
  }
  // ------------------------- 
  // 2) D(uh)
  // ------------------------- 
  {
    space Xh (omega, approx, "vector");
    size_t k = Xh.degree();
    string grad_approx = "P" + itos(k-1) + "d";
    space Th (omega, grad_approx, "tensor");
    trial u (Xh);     test v (Xh);
    trial sigma (Th); test tau (Th);
    field uh = interpolate (Xh, f2());
    integrate_option fopt;
    fopt.invert = true;
    form inv_mass = integrate (ddot(sigma,tau), fopt);
    form mD = integrate (ddot(D(u),tau));
    field D_uh = inv_mass*(mD*uh);

    // 2.1) D(uh) in integrate()
    quadrature_option qopt;
    qopt.set_order(5);
    Float err3 = integrate (omega, norm(D(uh)-D_uh), qopt);
    derr << "err3 = " << err3 << endl;
    err = std::max (err, err3);
  
    // 2.2) D(uh) in interpolate()
    field D_uh2 = interpolate (Th, D(uh));
    Float err4 = field(D_uh - D_uh2).max_abs();
    derr << "err4 = " << err4 << endl;
    err = std::max (err, err4);
  }
  // ------------------------- 
  // 3) div(uh)
  // ------------------------- 
  {
    space Xh (omega, approx, "vector");
    size_t k = Xh.degree();
    string grad_approx = "P" + itos(k-1) + "d";
    space Th (omega, grad_approx);
    trial u (Xh); test v (Xh);
    trial p (Th); test q (Th);
    field uh = interpolate (Xh, f2());
    integrate_option fopt;
    fopt.invert = true;
    form inv_mass = integrate (p*q, fopt);
    form m_div = integrate (div(u)*q);
    field div_uh = inv_mass*(m_div*uh);

    // 3.1) in integrate()
    quadrature_option qopt;
    qopt.set_order(5);
    Float err5 = integrate (omega, abs(div(uh)-div_uh), qopt);
    derr << "err5 = " << err5 << endl;
    err = std::max (err, err5);
  
    // 3.2) in interpolate()
    field div_uh2 = interpolate (Th, div(uh));
    Float err6 = field(div_uh - div_uh2).max_abs();
    derr << "err6 = " << err6 << endl;
    err = std::max (err, err6);
  }
  return (err < tol) ? 0 : 1;
}
