///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "basis_fem_Pk_lagrange.h"
#include "rheolef/rheostream.h"
#include "equispaced.icc"
#include "warburton.icc"
#include "fekete.icc"
#include "Pk_get_local_idof_on_side.icc"

namespace rheolef {

using namespace std;

// =========================================================================
// basis members
// =========================================================================
template<class T>
basis_fem_Pk_lagrange<T>::~basis_fem_Pk_lagrange()
{
}
template<class T>
basis_fem_Pk_lagrange<T>::basis_fem_Pk_lagrange (
  std::string              name, 
  const basis_option& sopt) 
  : basis_rep<T> (name,sopt),
    _raw_basis()
{
  size_type degree = 0;
  if ((name.length()) > 0 && (name[0] == 'P')) {
    // TODO: check also that name fits "Pk" where is an k integer
    degree = atoi(name.c_str()+1);
  } else if (name.length() > 0) { // missing 'P' !
    error_macro ("invalid polynomial name `"<<name<<"' for the Pk polynomial set");
  } else {
    // empty name : default cstor
    degree = 0;
  }
  string R = "?";
  switch (base::_sopt.get_raw_polynomial()) {
    case basis_option::monomial:  R = "M"; break;
    case basis_option::bernstein: R = "B"; break;
    case basis_option::dubiner:   R = "D"; break;
    default: error_macro ("unsupported polynomial: "<<sopt.get_raw_polynomial_name());
  }
  _raw_basis = basis_raw_basic<T> (R+itos(degree));
}
template<class T>
bool
basis_fem_Pk_lagrange<T>::is_nodal() const
{
  return true;
}
template<class T>
void
basis_fem_Pk_lagrange<T>::_initialize (reference_element hat_K) const
{
trace_macro ("_initialize \""<<base::name()<<"\"");
  size_type k = degree();
  size_type variant = hat_K.variant();

  // first_idof, first_inod by variant and dim
  base::_first_idof [variant].fill (0);
  if (k == 0) {
    base::_first_idof [variant][hat_K.dimension()+1] = 1;
  } else {
    switch (variant) {
      case reference_element::p:
        base::_first_idof [variant][1] = 1;
        break;
      case reference_element::e:
        base::_first_idof [variant][1] = 2;
        base::_first_idof [variant][2] = k+1;
        break;
      case reference_element::t:
        base::_first_idof [variant][1] = 3;
        base::_first_idof [variant][2] = 3*k;
        base::_first_idof [variant][3] = (k+1)*(k+2)/2;
        break;
      case reference_element::q:
        base::_first_idof [variant][1] = 4;
        base::_first_idof [variant][2] = 4*k;
        base::_first_idof [variant][3] = (k+1)*(k+1);
        break;
      case reference_element::T:
        base::_first_idof [variant][1] = 4;
        base::_first_idof [variant][2] = 6*(k-1) + 4;
        base::_first_idof [variant][3] = 4*(k-1)*(k-2)/2 + base::_first_idof [variant][2];
        base::_first_idof [variant][4] = (k+1)*(k+2)*(k+3)/6;
        break;
      case reference_element::P:
        base::_first_idof [variant][1] = 6;
        base::_first_idof [variant][2] = 9*(k-1) + 6;
        base::_first_idof [variant][3] = 2*(k-1)*(k-2)/2 + 3*(k-1)*(k-1) + base::_first_idof [variant][2];
        base::_first_idof [variant][4] = (k+1)*(k+1)*(k+2)/2;
        break;
      case reference_element::H:
        base::_first_idof [variant][1] = 8;
        base::_first_idof [variant][2] = 12*(k-1) + 8;
        base::_first_idof [variant][3] = 6*(k-1)*(k-1) +  base::_first_idof [variant][2];
        base::_first_idof [variant][4] = (k+1)*(k+1)*(k+1);
        break;
      default: error_macro ("unexpected element variant `"<<variant<<"'");
    }
  }
  base::_first_inod [variant] = base::_first_idof [variant];

#ifdef TO_CLEAN
  // idof & inod pointers by dimension and subgeo
  // in basis.h :
  static const ndim_max = 4; // d in [0:3]
  static const nsid_max = 6;
  mutable std::array<
            std::array<
              std::array<size_type,nsid_max+1>
             ,ndim_max+1>
           ,reference_element::max_variant>        _first_inod_by_subgeo, _first_idof_by_subgeo;

  // here:
  std::array<std::array<size_type,nsid_max+1>,ndim_max+1>& first_idof_by_subgeo = base::_first_idof_by_subgeo[variant];
  std::array<std::array<size_type,nsid_max+1>,ndim_max+1>& first_inod_by_subgeo = base::_first_inod_by_subgeo[variant];
  first_idof_by_subgeo [0].fill(0);
  first_inod_by_subgeo [0].fill(0);
  size_type last_idof = 0;
  for (size_type dim = 0; dim <= d; ++dim) {
    first_idof_by_subgeo [dim+1].fill(last_idof);
    first_inod_by_subgeo [dim+1].fill(last_idof);
    for (size_type loc_isid = 0, loc_nsid = hat_K.n_subgeo(dim); loc_isid < loc_nsid; ++loc_isid) {
      size_type loc_nsidvert = hat_K.subgeo_size (dim, loc_isid);
      size_type sid_variant = reference_element::variant (loc_nsidvert, dim); 
      size_type sid_ndof = base::_first_idof [sid_variant][dim+1] - base::_first_idof [sid_variant][dim];
      last_idof += sid_ndof;
      first_idof_by_subgeo [dim+1][loc_isid+1] = last_idof;
      first_inod_by_subgeo [dim+1][loc_isid+1] = last_idof;
    }
  }
#endif // TO_CLEAN
  // nodes:
  switch (base::_sopt.get_node()) {
    case basis_option::equispaced:
          pointset_lagrange_equispaced (hat_K, k, base::_hat_node[variant]);
          break;
    case basis_option::warburton:
          pointset_lagrange_warburton  (hat_K, k, base::_hat_node[variant]); break;
    case basis_option::fekete:
          pointset_lagrange_fekete     (hat_K, k, base::_hat_node[variant]); break;
    default: error_macro ("unsupported node set: "<<base::_sopt.get_node_name());
  }
  // vdm:
  _raw_basis.eval (hat_K, base::_hat_node[variant], base::_vdm[variant]);
  size_type loc_ndof = base::size(hat_K);
  base::_inv_vdm[variant].resize (loc_ndof,loc_ndof);
  arma::Mat<T> identity (loc_ndof,loc_ndof);
  identity.eye();
  bool inv_ok = solve (base::_inv_vdm[variant], base::_vdm[variant], identity);
  check_macro (inv_ok, "unisolvence failed for "
	<< base::name() <<"(" << hat_K.name() << ") basis");
}
// evaluation of all basis functions at hat_x:
template<class T>
void
basis_fem_Pk_lagrange<T>::eval (
  reference_element                 hat_K,
  const point_basic<T>&             hat_x,
  arma::Col<T>&                     value) const
{
  base::_initialize_guard (hat_K);
  _raw_basis.eval (hat_K, hat_x, base::_raw_value[hat_K.variant()]);
  value = trans(base::_inv_vdm[hat_K.variant()])*base::_raw_value[hat_K.variant()];
}
// evaluate the gradient:
template<class T>
void
basis_fem_Pk_lagrange<T>::grad_eval (
  reference_element           hat_K,
  const point_basic<T>&       hat_x,
  std::vector<point_basic<T> >& value) const 
{
  base::_initialize_guard (hat_K);
  std::vector<point_basic<T> >& raw_v_value = base::_raw_v_value [hat_K.variant()];
  arma::Mat<T>&                 inv_vdm     = base::_inv_vdm     [hat_K.variant()];
  _raw_basis.grad_eval (hat_K, hat_x, raw_v_value);
  size_type loc_ndof = raw_v_value.size(); 
  value.resize (loc_ndof);
  // value := trans(inv_vdm)*raw_value
  for (size_type loc_idof = 0; loc_idof < loc_ndof; ++loc_idof) {
    value [loc_idof] = point_basic<T>(0,0,0);
    for (size_type loc_jdof = 0; loc_jdof < loc_ndof; ++loc_jdof) {
      value [loc_idof] += inv_vdm (loc_jdof,loc_idof)*raw_v_value[loc_jdof];
    }
  }
}
// extract local dof-indexes on a side
template<class T>
void
basis_fem_Pk_lagrange<T>::get_local_idof_on_side (
  reference_element            hat_K,
  const side_information_type& sid,
  arma::Col<size_type>&        loc_idof) const
{
  details::Pk_get_local_idof_on_side (hat_K, sid, degree(), loc_idof);
}
// dofs for a scalar-valued function
template<class T>
void
basis_fem_Pk_lagrange<T>::_compute_dofs (
  reference_element     hat_K,
  const arma::Col<T>&   f_xnod, 
        arma::Col<T>&   dof) const
{
  error_macro ("_compute_dofs: should not be called in nodal basis");
  dof = f_xnod; // avoid a physical copy: already done in compute_dof(f) with is_nodal() 
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T)                                             	\
template class basis_fem_Pk_lagrange<T>;

_RHEOLEF_instanciation(Float)

}// namespace rheolef
