/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2011-2016 OpenFOAM Foundation
     \\/     M anipulation  | Copyright (C) 2016-2018 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#ifndef HashSet_C
#define HashSet_C

#include "HashSet.H"
#include "FixedList.H"

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

template<class Key, class Hash>
template<class InputIter>
inline Foam::label Foam::HashSet<Key, Hash>::assignMany
(
    const label nItems,
    InputIter first,
    InputIter last
)
{
    if (!this->capacity())
    {
        // Zero-sized from a previous transfer()?
        this->resize(2*nItems);
    }
    else
    {
        this->clear();
    }

    return insert(first, last);
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class Key, class Hash>
template<unsigned Size>
Foam::HashSet<Key, Hash>::HashSet(const FixedList<Key, Size>& list)
:
    parent_type(2*list.size())
{
    for (const auto& k : list)
    {
        this->insert(k);
    }
}


template<class Key, class Hash>
Foam::HashSet<Key, Hash>::HashSet(const UList<Key>& list)
:
    parent_type(2*list.size())
{
    for (const auto& k : list)
    {
        this->insert(k);
    }
}


template<class Key, class Hash>
Foam::HashSet<Key, Hash>::HashSet(const UIndirectList<Key>& list)
:
    parent_type(2*list.size())
{
    for (const auto& k : list)
    {
        this->insert(k);
    }
}


template<class Key, class Hash>
Foam::HashSet<Key, Hash>::HashSet(std::initializer_list<Key> list)
:
    parent_type(2*list.size())
{
    for (const auto& k : list)
    {
        this->insert(k);
    }
}


template<class Key, class Hash>
template<class AnyType, class AnyHash>
Foam::HashSet<Key, Hash>::HashSet
(
    const HashTable<AnyType, Key, AnyHash>& tbl
)
:
    parent_type(tbl.capacity())
{
    for (auto iter = tbl.cbegin(); iter != tbl.cend(); ++iter)
    {
        this->insert(iter.key());
    }
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Key, class Hash>
template<class InputIter>
inline Foam::label Foam::HashSet<Key, Hash>::insert
(
    InputIter first,
    InputIter last
)
{
    label changed = 0;
    for (; first != last; ++first)
    {
        if (insert(*first))
        {
            ++changed;
        }
    }
    return changed;
}


template<class Key, class Hash>
inline Foam::label Foam::HashSet<Key, Hash>::insert
(
    std::initializer_list<Key> list
)
{
    return insert(list.begin(), list.end());
}


template<class Key, class Hash>
template<unsigned Size>
inline Foam::label Foam::HashSet<Key, Hash>::insert
(
    const FixedList<Key, Size>& list
)
{
    return insert(list.begin(), list.end());
}


template<class Key, class Hash>
inline Foam::label Foam::HashSet<Key, Hash>::insert
(
    const UList<Key>& list
)
{
    return insert(list.begin(), list.end());
}


template<class Key, class Hash>
inline  Foam::label Foam::HashSet<Key, Hash>::insert
(
    const UIndirectList<Key>& list
)
{
    return insert(list.begin(), list.end());
}


template<class Key, class Hash>
template<class InputIter>
inline Foam::label Foam::HashSet<Key, Hash>::unset
(
    InputIter first,
    InputIter last
)
{
    return this->parent_type::erase(first, last);
}


template<class Key, class Hash>
inline Foam::label Foam::HashSet<Key, Hash>::unset
(
    std::initializer_list<Key> list
)
{
    return unset(list.begin(), list.end());
}


template<class Key, class Hash>
template<unsigned Size>
inline Foam::label Foam::HashSet<Key, Hash>::unset
(
    const FixedList<Key, Size>& list
)
{
    return unset(list.begin(), list.end());
}


template<class Key, class Hash>
inline Foam::label Foam::HashSet<Key, Hash>::unset
(
    const UList<Key>& list
)
{
    return unset(list.begin(), list.end());
}


template<class Key, class Hash>
inline Foam::label Foam::HashSet<Key, Hash>::unset
(
    const UIndirectList<Key>& list
)
{
    return unset(list.begin(), list.end());
}


// * * * * * * * * * * * * * * * Member Operators  * * * * * * * * * * * * * //

template<class Key, class Hash>
inline bool Foam::HashSet<Key, Hash>::operator()(const Key& key) const
{
    return this->found(key);
}


template<class Key, class Hash>
inline bool Foam::HashSet<Key, Hash>::operator[](const Key& key) const
{
    return this->found(key);
}


template<class Key, class Hash>
template<unsigned Size>
void Foam::HashSet<Key, Hash>::operator=(const FixedList<Key, Size>& rhs)
{
    assignMany(rhs.size(), rhs.begin(), rhs.end());
}


template<class Key, class Hash>
void Foam::HashSet<Key, Hash>::operator=(const UList<Key>& rhs)
{
    assignMany(rhs.size(), rhs.begin(), rhs.end());
}


template<class Key, class Hash>
void Foam::HashSet<Key, Hash>::operator=(std::initializer_list<Key> rhs)
{
    assignMany(rhs.size(), rhs.begin(), rhs.end());
}


template<class Key, class Hash>
bool Foam::HashSet<Key, Hash>::operator==(const HashSet<Key, Hash>& rhs) const
{
    // Sizes (number of keys) must match
    if (this->size() != rhs.size())
    {
        return false;
    }

    for (const_iterator iter = rhs.cbegin(); iter != rhs.cend(); ++iter)
    {
        if (!this->found(iter.key()))
        {
            return false;
        }
    }

    return true;
}


template<class Key, class Hash>
bool Foam::HashSet<Key, Hash>::operator!=(const HashSet<Key, Hash>& rhs) const
{
    return !operator==(rhs);
}


template<class Key, class Hash>
Foam::HashSet<Key, Hash>&
Foam::HashSet<Key, Hash>::operator|=(const HashSet<Key, Hash>& rhs)
{
    // Add rhs elements into lhs
    for (const_iterator iter = rhs.cbegin(); iter != rhs.cend(); ++iter)
    {
        this->insert(iter.key());
    }

    return *this;
}


template<class Key, class Hash>
inline Foam::HashSet<Key, Hash>&
Foam::HashSet<Key, Hash>::operator&=(const HashSet<Key, Hash>& rhs)
{
    this->parent_type::retain(rhs);
    return *this;
}


template<class Key, class Hash>
Foam::HashSet<Key, Hash>&
Foam::HashSet<Key, Hash>::operator^=(const HashSet<Key, Hash>& rhs)
{
    // Add missed rhs elements, remove duplicate elements
    for (const_iterator iter = rhs.cbegin(); iter != rhs.cend(); ++iter)
    {
        if (this->found(iter.key()))
        {
            this->erase(iter.key());
        }
        else
        {
            this->insert(iter.key());
        }
    }

    return *this;
}


template<class Key, class Hash>
inline Foam::HashSet<Key, Hash>&
Foam::HashSet<Key, Hash>::operator-=(const HashSet<Key, Hash>& rhs)
{
    this->parent_type::erase(rhs);

    return *this;
}


// * * * * * * * * * * * * * * * IOstream Operators  * * * * * * * * * * * * //

template<class Key, class Hash>
Foam::Ostream& Foam::operator<<(Ostream& os, const HashSet<Key, Hash>& tbl)
{
    return tbl.writeList(os, 10);  // 10=consistent with UList
}


/* * * * * * * * * * * * * * * * Global Operators  * * * * * * * * * * * * * */

template<class Key, class Hash>
Foam::HashSet<Key, Hash> Foam::operator|
(
    const HashSet<Key, Hash>& hash1,
    const HashSet<Key, Hash>& hash2
)
{
    HashSet<Key, Hash> out(hash1);
    out |= hash2;
    return out;
}


template<class Key, class Hash>
Foam::HashSet<Key, Hash> Foam::operator&
(
    const HashSet<Key, Hash>& hash1,
    const HashSet<Key, Hash>& hash2
)
{
    HashSet<Key, Hash> out(hash1);
    out &= hash2;
    return out;
}


template<class Key, class Hash>
Foam::HashSet<Key, Hash> Foam::operator^
(
    const HashSet<Key, Hash>& hash1,
    const HashSet<Key, Hash>& hash2
)
{
    HashSet<Key, Hash> out(hash1);
    out ^= hash2;
    return out;
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

template<class Key, class Hash>
inline typename Foam::HashSet<Key, Hash>::iterator
Foam::HashSet<Key, Hash>::begin()
{
    return HashTableCore::iterator_begin<iterator>
    (
        static_cast<parent_type&>(*this)
    );
}


template<class Key, class Hash>
inline typename Foam::HashSet<Key, Hash>::const_iterator
Foam::HashSet<Key, Hash>::begin() const
{
    return HashTableCore::iterator_cbegin<const_iterator>
    (
        static_cast<const parent_type&>(*this)
    );
}


template<class Key, class Hash>
inline typename Foam::HashSet<Key, Hash>::const_iterator
Foam::HashSet<Key, Hash>::cbegin() const
{
    return HashTableCore::iterator_cbegin<const_iterator>
    (
        static_cast<const parent_type&>(*this)
    );
}


template<class Key, class Hash>
inline const typename Foam::HashSet<Key, Hash>::iterator&
Foam::HashSet<Key, Hash>::end()
{
    return HashTableCore::iterator_end<iterator>();
}


template<class Key, class Hash>
inline const typename Foam::HashSet<Key, Hash>::const_iterator&
Foam::HashSet<Key, Hash>::end() const
{
    return HashTableCore::iterator_cend<const_iterator>();
}


template<class Key, class Hash>
inline const typename Foam::HashSet<Key, Hash>::const_iterator&
Foam::HashSet<Key, Hash>::cend() const
{
    return HashTableCore::iterator_cend<const_iterator>();
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
