// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/browsing_data/chrome_browsing_data_lifetime_manager.h"

#include <array>
#include <memory>

#include "base/files/file_path.h"
#include "base/guid.h"
#include "base/json/json_reader.h"
#include "base/path_service.h"
#include "base/run_loop.h"
#include "base/test/bind.h"
#include "base/test/scoped_feature_list.h"
#include "base/test/test_mock_time_task_runner.h"
#include "base/time/time.h"
#include "build/build_config.h"
#include "chrome/browser/autofill/autofill_uitest_util.h"
#include "chrome/browser/autofill/personal_data_manager_factory.h"
#include "chrome/browser/browsing_data/browsing_data_remover_browsertest_base.h"
#include "chrome/browser/browsing_data/chrome_browsing_data_lifetime_manager_factory.h"
#include "chrome/browser/browsing_data/chrome_browsing_data_remover_delegate.h"
#include "chrome/browser/browsing_data/counters/site_data_counting_helper.h"
#include "chrome/browser/content_settings/host_content_settings_map_factory.h"
#include "chrome/browser/history/history_service_factory.h"
#include "chrome/browser/net/system_network_context_manager.h"
#include "chrome/test/base/testing_profile.h"
#include "components/autofill/core/browser/autofill_test_utils.h"
#include "components/autofill/core/browser/data_model/autofill_profile.h"
#include "components/autofill/core/browser/personal_data_manager.h"
#include "components/browsing_data/core/features.h"
#include "components/browsing_data/core/pref_names.h"
#include "components/content_settings/core/browser/host_content_settings_map.h"
#include "components/history/core/browser/history_service.h"
#include "components/history/core/browser/history_types.h"
#include "components/keyed_service/core/service_access_type.h"
#include "components/prefs/testing_pref_service.h"
#include "components/sync/base/pref_names.h"
#include "components/sync/driver/sync_driver_switches.h"
#include "content/public/browser/browser_context.h"
#include "content/public/browser/browsing_data_filter_builder.h"
#include "content/public/browser/browsing_data_remover.h"
#include "content/public/browser/browsing_data_remover_delegate.h"
#include "content/public/browser/download_manager.h"
#include "content/public/browser/storage_partition.h"
#include "content/public/common/content_paths.h"
#include "content/public/test/browser_task_environment.h"
#include "content/public/test/browser_test.h"
#include "content/public/test/browsing_data_remover_test_util.h"
#include "content/public/test/download_test_observer.h"
#include "net/dns/mock_host_resolver.h"
#include "net/test/embedded_test_server/embedded_test_server.h"
#include "storage/browser/quota/special_storage_policy.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/abseil-cpp/absl/types/optional.h"
#include "url/gurl.h"

#if !BUILDFLAG(IS_ANDROID)
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/browser_list.h"
#include "chrome/test/base/ui_test_utils.h"
#endif

namespace {

enum class BrowserType { Default, Incognito };

constexpr std::array<const char*, 7> kSiteDataTypes{
    "Cookie", "LocalStorage",  "SessionStorage", "IndexedDb",
    "WebSql", "ServiceWorker", "CacheStorage"};

}  // namespace

class ChromeBrowsingDataLifetimeManagerTest
    : public BrowsingDataRemoverBrowserTestBase {
 protected:
  ChromeBrowsingDataLifetimeManagerTest() {
    InitFeatureList(
        {browsing_data::features::kEnableBrowsingDataLifetimeManager});
  }

  ~ChromeBrowsingDataLifetimeManagerTest() override = default;

  void SetUpOnMainThread() override {
    BrowsingDataRemoverBrowserTestBase::SetUpOnMainThread();
    GetProfile()->GetPrefs()->Set(syncer::prefs::kSyncManaged,
                                  base::Value(true));
  }

  void ApplyBrowsingDataLifetimeDeletion(base::StringPiece pref) {
    auto* browsing_data_lifetime_manager =
        ChromeBrowsingDataLifetimeManagerFactory::GetForProfile(GetProfile());
    browsing_data_lifetime_manager->SetEndTimeForTesting(base::Time::Max());
    content::BrowsingDataRemover* remover =
        GetProfile()->GetBrowsingDataRemover();
    content::BrowsingDataRemoverCompletionObserver completion_observer(remover);
    browsing_data_lifetime_manager->SetBrowsingDataRemoverObserverForTesting(
        &completion_observer);
    // The pref needs to be cleared so that the browsing data deletion is
    // triggered even if the same pref value is set twice in a row.
    GetProfile()->GetPrefs()->ClearPref(
        browsing_data::prefs::kBrowsingDataLifetime);
    GetProfile()->GetPrefs()->Set(browsing_data::prefs::kBrowsingDataLifetime,
                                  *base::JSONReader::Read(pref));

    completion_observer.BlockUntilCompletion();
  }

  void SetupSiteData(content::WebContents* web_contents) {
    for (const char* data_type : kSiteDataTypes) {
      SetDataForType(data_type, web_contents);
      EXPECT_TRUE(HasDataForType(data_type, web_contents));
    }
  }

  void CheckSiteData(content::WebContents* web_contents, bool has_site_data) {
    for (const char* data_type : kSiteDataTypes) {
      EXPECT_EQ(HasDataForType(data_type), has_site_data) << data_type;
    }
  }
};

class ChromeBrowsingDataLifetimeManagerScheduledRemovalTest
    : public ChromeBrowsingDataLifetimeManagerTest,
      public testing::WithParamInterface<BrowserType> {
 protected:
  ChromeBrowsingDataLifetimeManagerScheduledRemovalTest() = default;
  ~ChromeBrowsingDataLifetimeManagerScheduledRemovalTest() override = default;

  void SetUpOnMainThread() override {
    ChromeBrowsingDataLifetimeManagerTest::SetUpOnMainThread();
#if !BUILDFLAG(IS_ANDROID)
    if (GetParam() == BrowserType::Incognito)
      UseIncognitoBrowser();
#endif
    GetProfile()->GetPrefs()->Set(syncer::prefs::kSyncManaged,
                                  base::Value(true));
  }
};

IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       PrefChange) {
  static constexpr char kCookiesPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":
      ["cookies_and_other_site_data"]}])";
  static constexpr char kCachePref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":
      ["cached_images_and_files"]}])";

  GURL url = embedded_test_server()->GetURL("/browsing_data/site_data.html");
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));

  // Add cookie.
  SetDataForType("Cookie");
  EXPECT_TRUE(HasDataForType("Cookie"));

  // Expect that cookies are deleted.
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(),
                                     GURL(url::kAboutBlankURL)));
  ApplyBrowsingDataLifetimeDeletion(kCookiesPref);
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));
  EXPECT_FALSE(HasDataForType("Cookie"));

  url = embedded_test_server()->GetURL("/cachetime");

  EXPECT_EQ(net::OK, content::LoadBasicRequest(network_context(), url));

  // Check that the cache has been populated by revisiting these pages with the
  // server stopped.
  ASSERT_TRUE(embedded_test_server()->ShutdownAndWaitUntilComplete());
  EXPECT_EQ(net::OK, content::LoadBasicRequest(network_context(), url));

  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(),
                                     GURL(url::kAboutBlankURL)));
  ApplyBrowsingDataLifetimeDeletion(kCachePref);
  EXPECT_NE(net::OK, content::LoadBasicRequest(network_context(), url));
}

#if !BUILDFLAG(IS_ANDROID)
// TODO(crbug/1179729): Enable this test for android once we figure out if it
// is possible to delete download history on Android while the browser is
// running.
IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       Download) {
  static constexpr char kPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":["download_history"]}])";
  DownloadAnItem();
  VerifyDownloadCount(1u);
  ApplyBrowsingDataLifetimeDeletion(kPref);
  // The download is not deleted since the page where it happened is still
  // opened.
  VerifyDownloadCount(1u);

  // Navigate away.
  GURL url = embedded_test_server()->GetURL("/browsing_data/site_data.html");
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));
  VerifyDownloadCount(1u);

  // The download should now be deleted since the page where it happened is not
  // active.
  ApplyBrowsingDataLifetimeDeletion(kPref);
  VerifyDownloadCount(0u);
}
#endif

IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       History) {
  // No history saved in incognito mode.
  if (IsIncognito())
    return;
  static constexpr char kPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":["browsing_history"]}])";

  GURL url = embedded_test_server()->GetURL("/browsing_data/site_data.html");
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));

  SetDataForType("History");
  EXPECT_TRUE(HasDataForType("History"));

  ApplyBrowsingDataLifetimeDeletion(kPref);
  EXPECT_FALSE(HasDataForType("History"));
}

IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       ContentSettings) {
  static constexpr char kPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":["site_settings"]}])";

  auto* map = HostContentSettingsMapFactory::GetForProfile(GetProfile());
  map->SetContentSettingDefaultScope(GURL("http://host1.com:1"), GURL(),
                                     ContentSettingsType::COOKIES,
                                     CONTENT_SETTING_BLOCK);

  ApplyBrowsingDataLifetimeDeletion(kPref);

  ContentSettingsForOneType host_settings;
  map->GetSettingsForOneType(ContentSettingsType::COOKIES, &host_settings);
  for (const auto& host_setting : host_settings) {
    if (host_setting.source == "webui_allowlist")
      continue;
    EXPECT_EQ(ContentSettingsPattern::Wildcard(), host_setting.primary_pattern);
    EXPECT_EQ(CONTENT_SETTING_ALLOW, host_setting.GetContentSetting());
  }
}

IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       SiteData) {
  static constexpr char kPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":
      ["cookies_and_other_site_data"]}])";

  GURL url = embedded_test_server()->GetURL("/browsing_data/site_data.html");
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));
  SetupSiteData(GetActiveWebContents());
  ApplyBrowsingDataLifetimeDeletion(kPref);

#if !BUILDFLAG(IS_ANDROID)
  // The site data is not deleted since the page where it happened is still
  // opened.
  CheckSiteData(GetActiveWebContents(), /*has_site_data=*/true);

  // Navigate away.
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(),
                                     GURL(url::kAboutBlankURL)));

  // The site should now be deleted since the page where it happened is not
  // active.
  ApplyBrowsingDataLifetimeDeletion(kPref);
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));
#else
  ApplyBrowsingDataLifetimeDeletion(kPref);
#endif
  CheckSiteData(GetActiveWebContents(), /*has_site_data=*/false);
}

IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       Cache) {
  static constexpr char kPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":
      ["cached_images_and_files"]}])";

  GURL url = embedded_test_server()->GetURL("/cachetime");

  EXPECT_EQ(net::OK, content::LoadBasicRequest(network_context(), url));

  // Check that the cache has been populated by revisiting these pages with the
  // server stopped.
  ASSERT_TRUE(embedded_test_server()->ShutdownAndWaitUntilComplete());
  EXPECT_EQ(net::OK, content::LoadBasicRequest(network_context(), url));

  ApplyBrowsingDataLifetimeDeletion(kPref);
  EXPECT_NE(net::OK, content::LoadBasicRequest(network_context(), url));
}

#if !BUILDFLAG(IS_ANDROID)
IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       KeepsOtherTabData) {
  if (IsIncognito())
    return;

  static constexpr char kPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":
      ["cookies_and_other_site_data"]}])";

  GURL url = embedded_test_server()->GetURL("/browsing_data/site_data.html");
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));

  auto* first_tab = GetActiveWebContents();
  ui_test_utils::NavigateToURLWithDisposition(
      browser(), url, WindowOpenDisposition::NEW_FOREGROUND_TAB,
      ui_test_utils::BROWSER_TEST_WAIT_FOR_LOAD_STOP);
  auto* second_tab = GetActiveWebContents();
  DCHECK_NE(first_tab, second_tab);

  SetupSiteData(first_tab);
  SetupSiteData(second_tab);

  ApplyBrowsingDataLifetimeDeletion(kPref);

  // The site data is not deleted since the page where it happened is still
  // opened.
  CheckSiteData(first_tab, /*has_site_data=*/true);
  CheckSiteData(second_tab, /*has_site_data=*/true);

  // Navigate away first tab.
  ASSERT_TRUE(content::NavigateToURL(first_tab, GURL(url::kAboutBlankURL)));

  // The site data is not deleted since the domain of the data is still in use.
  ApplyBrowsingDataLifetimeDeletion(kPref);
  ASSERT_TRUE(content::NavigateToURL(first_tab, url));
  CheckSiteData(first_tab, /*has_site_data=*/true);
  CheckSiteData(second_tab, /*has_site_data=*/true);

  // Navigate away second tab.
  ASSERT_TRUE(content::NavigateToURL(second_tab, GURL(url::kAboutBlankURL)));

  // The site data is not deleted since the domain of the data is still in use.
  ApplyBrowsingDataLifetimeDeletion(kPref);
  ASSERT_TRUE(content::NavigateToURL(second_tab, url));
  CheckSiteData(first_tab, /*has_site_data=*/true);
  CheckSiteData(second_tab, /*has_site_data=*/true);

  // Navigate away both tabs.
  ASSERT_TRUE(content::NavigateToURL(first_tab, GURL(url::kAboutBlankURL)));
  ASSERT_TRUE(content::NavigateToURL(second_tab, GURL(url::kAboutBlankURL)));

  // The site data is not deleted since the domain of the data is still in use.
  ApplyBrowsingDataLifetimeDeletion(kPref);
  ASSERT_TRUE(content::NavigateToURL(first_tab, url));
  ASSERT_TRUE(content::NavigateToURL(second_tab, url));
  CheckSiteData(first_tab, /*has_site_data=*/false);
  CheckSiteData(second_tab, /*has_site_data=*/false);
}

IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       KeepsOtherWindowData) {
  if (IsIncognito())
    return;

  static constexpr char kPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":
      ["cookies_and_other_site_data"]}])";

  GURL url = embedded_test_server()->GetURL("/browsing_data/site_data.html");
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));

  SetupSiteData(GetActiveWebContents());

  ApplyBrowsingDataLifetimeDeletion(kPref);

  // The site data is not deleted since the page where it happened is still
  // opened.
  CheckSiteData(GetActiveWebContents(), /*has_site_data=*/true);

  // Open current url in new tab.
  ui_test_utils::NavigateToURLWithDisposition(
      browser(), url, WindowOpenDisposition::NEW_WINDOW,
      ui_test_utils::BROWSER_TEST_WAIT_FOR_LOAD_STOP);

  EXPECT_EQ(BrowserList::GetInstance()->size(), 2u);
  content::WebContents* new_tab = nullptr;
  for (auto* b : *BrowserList::GetInstance()) {
    if (b != browser())
      new_tab = b->tab_strip_model()->GetActiveWebContents();
  }

  ASSERT_TRUE(new_tab);
  ASSERT_NE(new_tab, GetActiveWebContents());

  // Navigate away current tab.
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(),
                                     GURL(url::kAboutBlankURL)));

  // The site data is not deleted since the page's domain is opened in another
  // tab.
  ApplyBrowsingDataLifetimeDeletion(kPref);
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));
  CheckSiteData(GetActiveWebContents(), /*has_site_data=*/true);

  // Navigate away both tabs.
  ASSERT_TRUE(content::NavigateToURL(new_tab, GURL(url::kAboutBlankURL)));
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(),
                                     GURL(url::kAboutBlankURL)));

  ApplyBrowsingDataLifetimeDeletion(kPref);
  ASSERT_TRUE(content::NavigateToURL(GetActiveWebContents(), url));
  CheckSiteData(GetActiveWebContents(), /*has_site_data=*/false);
}

// Disabled because "autofill::AddTestProfile" times out when sync is disabled.
IN_PROC_BROWSER_TEST_P(ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                       DISABLED_Autofill) {
  // No autofill data saved in incognito mode.
  if (IsIncognito())
    return;
  static constexpr char kPref[] =
      R"([{"time_to_live_in_hours": 1, "data_types":["autofill"]}])";

  autofill::AutofillProfile profile("01234567-89ab-cdef-fedc-ba9876543210",
                                    autofill::test::kEmptyOrigin);
  autofill::test::SetProfileInfo(
      &profile, "Marion", "Mitchell", "Morrison", "johnwayne@me.xyz", "Fox",
      "123 Zoo St.", "unit 5", "Hollywood", "CA", "91601", "US", "12345678910");
  autofill::AddTestProfile(GetProfile(), profile);
  auto* personal_data_manager =
      autofill::PersonalDataManagerFactory::GetForProfile(GetProfile());
  EXPECT_EQ(
      profile.Compare(*personal_data_manager->GetProfileByGUID(profile.guid())),
      0);

  ApplyBrowsingDataLifetimeDeletion(kPref);

  EXPECT_EQ(nullptr, personal_data_manager->GetProfileByGUID(profile.guid()));
}
#endif

INSTANTIATE_TEST_SUITE_P(All,
                         ChromeBrowsingDataLifetimeManagerScheduledRemovalTest,
                         ::testing::Values(BrowserType::Default,
                                           BrowserType::Incognito));

class ChromeBrowsingDataLifetimeManagerShutdownTest
    : public ChromeBrowsingDataLifetimeManagerTest {
 protected:
  ChromeBrowsingDataLifetimeManagerShutdownTest() = default;
  ~ChromeBrowsingDataLifetimeManagerShutdownTest() override = default;

  history::HistoryService* history_service() {
    return HistoryServiceFactory::GetForProfile(
        GetProfile(), ServiceAccessType::EXPLICIT_ACCESS);
  }

  void VerifyHistorySize(size_t expected_size) {
    history::QueryResults history_query_results;
    base::RunLoop run_loop;
    base::CancelableTaskTracker tracker;
    history_service()->QueryHistory(
        std::u16string(), history::QueryOptions(),
        base::BindLambdaForTesting([&](history::QueryResults results) {
          history_query_results = std::move(results);
          run_loop.QuitClosure().Run();
        }),
        &tracker);
    run_loop.Run();
    EXPECT_EQ(history_query_results.size(), expected_size);
  }
};

IN_PROC_BROWSER_TEST_F(ChromeBrowsingDataLifetimeManagerShutdownTest,
                       PRE_PRE_BrowserShutdown) {
  // browsing_history
  history_service()->AddPage(GURL("https://www.website.com"),
                             base::Time::FromDoubleT(1000),
                             history::VisitSource::SOURCE_BROWSED);
  VerifyHistorySize(1u);

  // download_history
  DownloadAnItem();
  VerifyDownloadCount(1u);

  // site_settings
  auto* map = HostContentSettingsMapFactory::GetForProfile(GetProfile());
  map->SetContentSettingDefaultScope(GURL("http://host1.com:1"), GURL(),
                                     ContentSettingsType::COOKIES,
                                     CONTENT_SETTING_BLOCK);

  ContentSettingsForOneType host_settings;
  bool has_pref_setting = false;
  map->GetSettingsForOneType(ContentSettingsType::COOKIES, &host_settings);
  for (const auto& host_setting : host_settings) {
    if (host_setting.source == "webui_allowlist")
      continue;
    if (host_setting.source == "preference") {
      has_pref_setting = true;
      EXPECT_EQ(ContentSettingsPattern::FromURL(GURL("http://host1.com:1")),
                host_setting.primary_pattern);
      EXPECT_EQ(CONTENT_SETTING_BLOCK, host_setting.GetContentSetting());
    }
  }
  EXPECT_TRUE(has_pref_setting);

  // Ensure nothing gets deleted when the browser closes.
  static constexpr char kPref[] = R"([])";
  GetProfile()->GetPrefs()->Set(
      browsing_data::prefs::kClearBrowsingDataOnExitList,
      *base::JSONReader::Read(kPref));
  base::RunLoop().RunUntilIdle();
}

IN_PROC_BROWSER_TEST_F(ChromeBrowsingDataLifetimeManagerShutdownTest,
                       PRE_BrowserShutdown) {
  // browsing_history
  VerifyHistorySize(1u);

  // download_history
  VerifyDownloadCount(1u);

  // site_settings
  auto* map = HostContentSettingsMapFactory::GetForProfile(GetProfile());
  ContentSettingsForOneType host_settings;
  bool has_pref_setting = false;
  map->GetSettingsForOneType(ContentSettingsType::COOKIES, &host_settings);
  for (const auto& host_setting : host_settings) {
    if (host_setting.source == "webui_allowlist")
      continue;
    if (host_setting.source == "preference") {
      has_pref_setting = true;
      EXPECT_EQ(ContentSettingsPattern::FromURL(GURL("http://host1.com:1")),
                host_setting.primary_pattern);
      EXPECT_EQ(CONTENT_SETTING_BLOCK, host_setting.GetContentSetting());
    }
  }
  EXPECT_TRUE(has_pref_setting);

  // Ensure data gets deleted when the browser closes.
  static constexpr char kPref[] =
      R"(["browsing_history", "download_history", "cookies_and_other_site_data",
      "cached_images_and_files", "password_signin", "autofill", "site_settings",
      "hosted_app_data"])";
  GetProfile()->GetPrefs()->Set(
      browsing_data::prefs::kClearBrowsingDataOnExitList,
      *base::JSONReader::Read(kPref));
  base::RunLoop().RunUntilIdle();
}

IN_PROC_BROWSER_TEST_F(ChromeBrowsingDataLifetimeManagerShutdownTest,
                       BrowserShutdown) {
  // browsing_history
  VerifyHistorySize(0u);

  // download_history
  VerifyDownloadCount(0u);

  // site_settings
  auto* map = HostContentSettingsMapFactory::GetForProfile(GetProfile());

  ContentSettingsForOneType host_settings;
  map->GetSettingsForOneType(ContentSettingsType::COOKIES, &host_settings);
  for (const auto& host_setting : host_settings) {
    if (host_setting.source == "webui_allowlist")
      continue;
    EXPECT_EQ(ContentSettingsPattern::Wildcard(), host_setting.primary_pattern);
    EXPECT_EQ(CONTENT_SETTING_ALLOW, host_setting.GetContentSetting());
  }
}
