// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ash/crosapi/migration_progress_tracker.h"

#include "base/cxx17_backports.h"
#include "base/logging.h"

namespace ash {
MigrationProgressTrackerImpl::MigrationProgressTrackerImpl(
    const ProgressCallback& callback)
    : progress_(0),
      size_copied_(0),
      total_size_to_copy_(-1),
      progress_callback_(callback) {}

MigrationProgressTrackerImpl::~MigrationProgressTrackerImpl() = default;

void MigrationProgressTrackerImpl::UpdateProgress(int64_t size) {
  DCHECK_NE(total_size_to_copy_, -1);

  if (total_size_to_copy_ == 0) {
    LOG(WARNING) << "total_size_to_copy_ is zero.";
    progress_callback_.Run(0);
    return;
  }

  size_copied_ += size;

  int new_progress = static_cast<int>(size_copied_ * 100 / total_size_to_copy_);
  new_progress = base::clamp(new_progress, 0, 100);

  if (progress_ < new_progress) {
    progress_ = new_progress;
    progress_callback_.Run(new_progress);
  }
}

void MigrationProgressTrackerImpl::SetTotalSizeToCopy(int64_t size) {
  // Ensure that the method has not been called.
  DCHECK_EQ(total_size_to_copy_, -1);

  total_size_to_copy_ = size;
}
}  // namespace ash
