// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_THEMES_THEME_SYNCABLE_SERVICE_H_
#define CHROME_BROWSER_THEMES_THEME_SYNCABLE_SERVICE_H_

#include <memory>

#include "base/gtest_prod_util.h"
#include "base/macros.h"
#include "base/threading/thread_checker.h"
#include "components/sync/model/sync_change.h"
#include "components/sync/model/sync_data.h"
#include "components/sync/model/sync_error.h"
#include "components/sync/model/sync_error_factory.h"
#include "components/sync/model/syncable_service.h"

class Profile;
class ThemeService;
class ThemeSyncableServiceTest;

namespace sync_pb {
class ThemeSpecifics;
}

class ThemeSyncableService : public syncer::SyncableService {
 public:
  ThemeSyncableService(Profile* profile,  // Same profile used by theme_service.
                       ThemeService* theme_service);
  ~ThemeSyncableService() override;

  static syncer::ModelType model_type() { return syncer::THEMES; }

  // Called by ThemeService when user changes theme.
  void OnThemeChange();

  // syncer::SyncableService implementation.
  void WaitUntilReadyToSync(base::OnceClosure done) override;
  syncer::SyncMergeResult MergeDataAndStartSyncing(
      syncer::ModelType type,
      const syncer::SyncDataList& initial_sync_data,
      std::unique_ptr<syncer::SyncChangeProcessor> sync_processor,
      std::unique_ptr<syncer::SyncErrorFactory> error_handler) override;
  void StopSyncing(syncer::ModelType type) override;
  syncer::SyncDataList GetAllSyncData(syncer::ModelType type) const override;
  syncer::SyncError ProcessSyncChanges(
      const base::Location& from_here,
      const syncer::SyncChangeList& change_list) override;

  // Client tag and tile of theme node in sync.
  static const char kCurrentThemeClientTag[];
  static const char kCurrentThemeNodeTitle[];

 private:
  static bool AreThemeSpecificsEqual(
      const sync_pb::ThemeSpecifics& a,
      const sync_pb::ThemeSpecifics& b,
      bool is_system_theme_distinct_from_default_theme);

  // Returns whether extensions or autogenerated themes are used.
  static bool HasNonDefaultTheme(
      const sync_pb::ThemeSpecifics& theme_specifics);

  // Set theme from theme specifics in |sync_data| using
  // SetCurrentThemeFromThemeSpecifics() if it's different from |current_specs|.
  void MaybeSetTheme(const sync_pb::ThemeSpecifics& current_specs,
                     const syncer::SyncData& sync_data);

  void SetCurrentThemeFromThemeSpecifics(
      const sync_pb::ThemeSpecifics& theme_specifics);

  // If the current theme is syncable, fills in the passed |theme_specifics|
  // structure based on the currently applied theme and returns |true|.
  // Otherwise returns |false|.
  bool GetThemeSpecificsFromCurrentTheme(
      sync_pb::ThemeSpecifics* theme_specifics) const;

  // Updates theme specifics in sync to |theme_specifics|.
  syncer::SyncError ProcessNewTheme(
      syncer::SyncChange::SyncChangeType change_type,
      const sync_pb::ThemeSpecifics& theme_specifics);

  Profile* const profile_;
  ThemeService* const theme_service_;

  std::unique_ptr<syncer::SyncChangeProcessor> sync_processor_;
  std::unique_ptr<syncer::SyncErrorFactory> sync_error_handler_;

  // Persist use_system_theme_by_default for platforms that use it, even if
  // we're not on one.
  bool use_system_theme_by_default_;

  base::ThreadChecker thread_checker_;

  FRIEND_TEST_ALL_PREFIXES(ThemeSyncableServiceTest, AreThemeSpecificsEqual);

  DISALLOW_COPY_AND_ASSIGN(ThemeSyncableService);
};

#endif  // CHROME_BROWSER_THEMES_THEME_SYNCABLE_SERVICE_H_
