# Copyright 2014 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

import re

from core import perf_benchmark

from telemetry import benchmark
from telemetry import story
from telemetry.timeline import chrome_trace_category_filter
from telemetry.timeline import chrome_trace_config
from telemetry.web_perf import timeline_based_measurement

import page_sets


# TODO(rnephew): Revisit the re-enabled benchmarks on Wed, Aug 8 2017.

# Regex to filter out a few names of statistics supported by
# Histogram.getStatisticScalar(), see:
#   https://github.com/catapult-project/catapult/blob/d4179a05/tracing/tracing/value/histogram.html#L645  pylint: disable=line-too-long
_IGNORED_STATS_RE = re.compile(
    r'(?<!dump)(?<!process)_(std|count|max|min|sum|pct_\d{4}(_\d+)?)$')


class _MediaBenchmark(perf_benchmark.PerfBenchmark):
  """Base class for TBMv2-based media benchmarks (MediaDesktop and
  MediaMobile)."""

  def CreateStorySet(self, options):
    return page_sets.MediaCasesStorySet(measure_memory=True)

  def CreateCoreTimelineBasedMeasurementOptions(self):
    category_filter = chrome_trace_category_filter.ChromeTraceCategoryFilter()

    # 'toplevel' category provides CPU time slices used by # cpuTimeMetric.
    category_filter.AddIncludedCategory('toplevel')

    # Collect media related events required by mediaMetric.
    category_filter.AddIncludedCategory('media')

    # Collect memory data.
    category_filter.AddDisabledByDefault('disabled-by-default-memory-infra')

    options = timeline_based_measurement.Options(category_filter)
    options.config.enable_android_graphics_memtrack = True

    # Setting an empty memory dump config disables periodic dumps.
    options.config.chrome_trace_config.SetMemoryDumpConfig(
        chrome_trace_config.MemoryDumpConfig())

    options.SetTimelineBasedMetrics(['mediaMetric', 'cpuTimeMetric',
                                     'memoryMetric'])
    return options

  @classmethod
  def ShouldAddValue(cls, name, from_first_story_run):
    del from_first_story_run # unused
    # TODO(crbug.com/610962): Remove this stopgap when the perf dashboard
    # is able to cope with the data load generated by TBMv2 metrics.
    return not _IGNORED_STATS_RE.search(name)


@benchmark.Info(emails=['johnchen@chromium.org', 'crouleau@chromium.org'],
                 component='Internals>Media')
class MediaDesktop(_MediaBenchmark):
  """Obtains media performance for key user scenarios on desktop."""
  SUPPORTED_PLATFORMS = [story.expectations.ALL_DESKTOP]

  @classmethod
  def Name(cls):
    return 'media.desktop'


# If any story is failing on svelte, please only disable on svelte.
@benchmark.Info(emails=['johnchen@chromium.org', 'crouleau@chromium.org'],
                 component='Internals>Media')
class MediaMobile(_MediaBenchmark):
  """Obtains media performance for key user scenarios on mobile devices."""

  tag = 'android'
  options = {'story_tag_filter_exclude': 'is_4k,is_50fps'}
  SUPPORTED_PLATFORMS = [story.expectations.ANDROID_NOT_WEBVIEW]

  @classmethod
  def Name(cls):
    return 'media.mobile'

  def SetExtraBrowserOptions(self, options):
    # By default, Chrome on Android does not allow autoplay
    # of media: it requires a user gesture event to start a video.
    # The following option works around that.
    options.AppendExtraBrowserArgs(
        ['--autoplay-policy=no-user-gesture-required'])
