// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/*
 * strings.m.js is generated when we enable it via UseStringsJs() in webUI
 * controller. When loading it, it will populate data such as localized strings
 * into |loadTimeData| that is imported below.
 */
import '../../strings.m.js';

import {loadTimeData} from 'chrome://resources/js/load_time_data.js';

import {Flag} from '../flag.js';
import {I18nString} from '../i18n_string.js';

/**
 * Returns the device board name.
 */
export function getBoard(): string {
  return loadTimeData.getString('board_name');
}

/**
 * Returns the device type.
 */
export function getDeviceType(): string {
  return loadTimeData.getString('device_type');
}

/**
 * Returns the I18N message generated by given |name| and |substitutions|.
 */
export function getI18nMessage(
    name: I18nString, ...substitutions: Array<number|string>): string {
  return loadTimeData.getStringF(name, ...substitutions);
}

/**
 * @return Returns 'ltr'/'rtl' for left-to-right/right-to-left system UI
 *     language.
 */
export function getTextDirection(): string {
  return loadTimeData.getString('textdirection');
}

/**
 * Returns the boolean value of the chrome flag.
 */
export function getChromeFlag(flag: Flag): boolean {
  return loadTimeData.getBoolean(flag);
}

/**
 * Returns the boolean value of whether the image is test image.
 */
export function getIsTestImage(): boolean {
  return loadTimeData.getBoolean('is_test_image');
}


/**
 * Returns the browser version string.
 */
export function getBrowserVersion(): string {
  return loadTimeData.getString('browser_version');
}

/**
 * Returns if CCA is running on local development server.
 */
export function isLocalDev(): boolean {
  return getBoard() === 'local-dev';
}
