// Copyright 2013 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/process/process_metrics.h"

#include <limits.h>
#include <mach/task.h>
#include <mach/vm_region.h>
#include <malloc/malloc.h>
#include <stddef.h>

#include "base/check_op.h"
#include "base/memory/ptr_util.h"
#include "base/numerics/safe_conversions.h"
#include "build/blink_buildflags.h"
#include "process_metrics_apple_internal.h"

namespace base {

ProcessMetrics::ProcessMetrics(ProcessHandle process) {
  process_metrics_helper_ =
      std::make_unique<ProcessMetricsAppleInternal>(process);
}

// static
std::unique_ptr<ProcessMetrics> ProcessMetrics::CreateProcessMetrics(
    ProcessHandle process) {
  return WrapUnique(new ProcessMetrics(process));
}

TimeDelta ProcessMetrics::GetCumulativeCPUUsage() {
  return process_metrics_helper_->GetCumulativeCPUUsage();
}

// TODO(https://crbug.com/1439714): This is a duplicated implementation of
// process_metrics_mac.cc. Need to find a way to share the implementation with
// the iOS port.
int ProcessMetrics::GetPackageIdleWakeupsPerSecond() {
  task_power_info power_info_data = process_metrics_helper_->GetTaskPowerInfo();

  // The task_power_info struct contains two wakeup counters:
  // task_interrupt_wakeups and task_platform_idle_wakeups.
  // task_interrupt_wakeups is the total number of wakeups generated by the
  // process, and is the number that Activity Monitor reports.
  // task_platform_idle_wakeups is a subset of task_interrupt_wakeups that
  // tallies the number of times the processor was taken out of its low-power
  // idle state to handle a wakeup. task_platform_idle_wakeups therefore result
  // in a greater power increase than the other interrupts which occur while the
  // CPU is already working, and reducing them has a greater overall impact on
  // power usage. See the powermetrics man page for more info.
  return CalculatePackageIdleWakeupsPerSecond(
      power_info_data.task_platform_idle_wakeups);
}

// TODO(https://crbug.com/1439714): This is a duplicated implementation of
// process_metrics_mac.cc. Need to find a way to share the implementation with
// the iOS port.
int ProcessMetrics::GetIdleWakeupsPerSecond() {
  task_power_info power_info_data = process_metrics_helper_->GetTaskPowerInfo();

  return CalculateIdleWakeupsPerSecond(power_info_data.task_interrupt_wakeups);
}

// The blink code path pulls in process_metrics_posix.cc which
// is used for the following implementations.
#if !BUILDFLAG(USE_BLINK)

size_t GetMaxFds() {
  static const rlim_t kSystemDefaultMaxFds = 256;
  rlim_t max_fds;
  struct rlimit nofile;
  if (getrlimit(RLIMIT_NOFILE, &nofile)) {
    // Error case: Take a best guess.
    max_fds = kSystemDefaultMaxFds;
  } else {
    max_fds = nofile.rlim_cur;
  }

  if (max_fds > INT_MAX)
    max_fds = INT_MAX;

  return static_cast<size_t>(max_fds);
}

void IncreaseFdLimitTo(unsigned int max_descriptors) {
  // Unimplemented.
}

#endif  // !BUILDFLAG(USE_BLINK)

}  // namespace base
