/*====================================================================*
 -  Copyright (C) 2001 Leptonica.  All rights reserved.
 -  This software is distributed in the hope that it will be
 -  useful, but with NO WARRANTY OF ANY KIND.
 -  No author or distributor accepts responsibility to anyone for the
 -  consequences of using this software, or for whether it serves any
 -  particular purpose or works at all, unless he or she says so in
 -  writing.  Everyone is granted permission to copy, modify and
 -  redistribute this source code, for commercial or non-commercial
 -  purposes, with the following restrictions: (1) the origin of this
 -  source code must not be misrepresented; (2) modified versions must
 -  be plainly marked as such; and (3) this notice may not be removed
 -  or altered from any source or modified source distribution.
 *====================================================================*/



/*
 *  sel1.c
 *
 *      Basic ops on Sels and Selas
 *
 *         Create/destroy/copy:
 *            SELA      *selaCreate()
 *            void       selaDestroy()
 *            SEL       *selCreate()
 *            void       selDestroy()
 *            SEL       *selCopy()
 *            SEL       *selCreateBrick()
 *
 *         Helper proc:
 *            l_int32  **create2dIntArray()
 *
 *         Extension of sela:
 *            SELA      *selaAddSel()
 *            l_int32    selaExtendArray()
 *
 *         Accessors:
 *            l_int32    selaGetCount()
 *            SEL       *selaGetSel()
 *            char      *selGetName()
 *            l_int32    selaFindSelByName()
 *            l_int32    selGetElement()
 *            l_int32    selSetElement()
 *
 *         Max extent vals for erosion and hmt:
 *            l_int32    selFindMaxTranslations()
 *
 *         Write/read & visualization:
 *            l_int32    selaWrite()
 *            l_int32    selaWriteStream()
 *            l_int32    selWriteStream()
 *            l_int32    selaRead()
 *            l_int32    selaReadStream()
 *            l_int32    selReadStream()
 *
 *
 *     Usage note:
 *        Consistency, being the hobgoblin of small minds,
 *        is adhered to here in the dimensioning and accessing of sels.
 *        Everything is done in standard matrix (row, column) order.
 *        We have 2 functions that make sels:
 *             selCreate(), with input (h, w, [name])
 *             selCreateBrick(), with input (h, w, cy, cx, val)
 *        When we set specific elements in a sel, we likewise use
 *        (row, col) ordering:
 *             selSetElement(), with input (row, col, type)
 */



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "allheaders.h"

static const l_int32  INITIAL_PTR_ARRAYSIZE = 50;  /* n'import quoi */
static const l_int32  MANY_SELS = 1000;
static const l_int32  BUFSIZE = 256;


/*------------------------------------------------------------------------*
 *                      Create / Destroy / Copy                           *
 *------------------------------------------------------------------------*/
/*!
 *  selaCreate()
 *
 *      Input:  n (initial number of sel ptrs; use 0 for default)
 *      Return: sela, or null on error
 */
SELA *
selaCreate(l_int32  n)
{
SELA  *sela;

    PROCNAME("selaCreate");

    if (n <= 0)
	n = INITIAL_PTR_ARRAYSIZE;
    if (n > MANY_SELS)
	L_WARNING_INT("%d sels", procName, n);

    if ((sela = (SELA *)CALLOC(1, sizeof(SELA))) == NULL)
	return (SELA *)ERROR_PTR("sela not made", procName, NULL);

    sela->nalloc = n;
    sela->n = 0;

	/* make array of se ptrs */
    if ((sela->sel = (SEL **)CALLOC(n, sizeof(SEL *))) == NULL)
	return (SELA *)ERROR_PTR("sel ptrs not made", procName, NULL);

    return sela;
}


/*!
 *  selaDestroy()
 *
 *      Input:  &sela (<to be nulled>)
 *      Return: void
 */
void
selaDestroy(SELA  **psela)
{
SELA    *sela;
l_int32  i;

    if (!psela) return;
    if ((sela = *psela) == NULL)
	return;

    for (i = 0; i < sela->n; i++)
	selDestroy(&sela->sel[i]);
    FREE((void *)sela->sel);

    FREE((void *)sela);
    *psela = NULL;
    return;
}


/*!
 *  selCreate()
 *
 *      Input:  height, width
 *              name (<optional> sel name)
 *      Return: sel, or null on error
 *
 *  Note: selCreate() initializes all values to 0.
 *        After this call, (cy,cx) and nonzero data values must be
 *        assigned.  If a text name is not assigned here, it will
 *        be needed later when the sel is put into a sela.
 */
SEL *
selCreate(l_int32      height,
	  l_int32      width,
	  const char  *name)
{
SEL  *sel;

    PROCNAME("selCreate");

    if ((sel = (SEL *)CALLOC(1, sizeof(SEL))) == NULL)
	return (SEL *)ERROR_PTR("sel not made", procName, NULL);
    if (name)
	sel->name = stringNew(name);
    sel->sy = height;
    sel->sx = width;
    if ((sel->data = create2dIntArray(height, width)) == NULL)
	return (SEL *)ERROR_PTR("data not allocated", procName, NULL);

    return sel;
}


/*!
 *  selDestroy()
 *
 *      Input:  &sel (<to be nulled>)
 *      Return: void
 */
void
selDestroy(SEL  **psel)
{
l_int32  i;
SEL     *sel;

    PROCNAME("selDestroy");

    if (psel == NULL)  {
	L_WARNING("ptr address is NULL!", procName);
	return;
    }
    if ((sel = *psel) == NULL)
	return;

    for (i = 0; i < sel->sy; i++)
	FREE((void *)sel->data[i]);
    FREE((void *)sel->data);

    if (sel->name)
	FREE((void *)sel->name);

    FREE((void *)sel);

    *psel = NULL;
    return;
}


/*!
 *  selCopy() 
 *
 *      Input:  sel
 *      Return: a copy of the sel, or null on error
 */
SEL *
selCopy(SEL  *sel)
{
l_int32  i, j;
SEL     *csel;

    PROCNAME("selCopy");

    if (!sel)
	return (SEL *)ERROR_PTR("sel not defined", procName, NULL);

    if ((csel = (SEL *)CALLOC(1, sizeof(SEL))) == NULL)
	return (SEL *)ERROR_PTR("csel not made", procName, NULL);

    csel->sy = sel->sy;
    csel->sx = sel->sx;
    csel->cy = sel->cy;
    csel->cx = sel->cx;

    if ((csel->data = create2dIntArray(sel->sy, sel->sx)) == NULL)
	return (SEL *)ERROR_PTR("sel data not made", procName, NULL);

    for (i = 0; i < sel->sy; i++)
	for (j = 0; j < sel->sx; j++)
	    csel->data[i][j] = sel->data[i][j];

    csel->name = stringNew(sel->name);

    return csel;
}



/*!
 *  selCreateBrick()
 *
 *      Input:  height, width
 *              cy, cx  (center, relative to UL corner at 0,0)
 *              type  (SEL_HIT, SEL_MISS, or SEL_DONT_CARE)
 *      Return: sel, or null on error
 *
 *  Action: a "brick" is a rectangular array of either hits,
 *          misses, or don't-cares.
 */
SEL *
selCreateBrick(l_int32  h,
	       l_int32  w,
	       l_int32  cy,
	       l_int32  cx,
	       l_int32  type)
{
l_int32  i, j;
SEL     *sel;

    PROCNAME("selCreateBrick");

    if (h <= 0 || w <= 0)
	return (SEL *)ERROR_PTR("h and w must both be > 0", procName, NULL);
    if (type != SEL_HIT && type != SEL_MISS && type != SEL_DONT_CARE)
	return (SEL *)ERROR_PTR("invalid sel element type", procName, NULL);

    if ((sel = selCreate(h, w, NULL)) == NULL)
	return (SEL *)ERROR_PTR("sel not made", procName, NULL);
    sel->cy = cy;
    sel->cx = cx;
    for (i = 0; i < h; i++)
	for (j = 0; j < w; j++)
	    sel->data[i][j] = type;

    return sel;
}


/*!
 *  create2dIntArray()
 *
 *      Input:  sy (rows == height)
 *              sx (columns == width)
 *      Return: doubly indexed array (i.e., an array of sy row pointers,
 *              each of which points to an array of sx ints)
 *
 *  Note: the array[sy][sx] is indexed in standard
 *        "matrix notation" with the row index first.
 */
l_int32 **
create2dIntArray(l_int32  sy,
                 l_int32  sx)
{
l_int32    i;
l_int32  **array;

    PROCNAME("create2dIntArray");

    if ((array = (l_int32 **)CALLOC(sy, sizeof(l_int32 *))) == NULL)
	return (l_int32 **)ERROR_PTR("ptr array not made", procName, NULL);

    for (i = 0; i < sy; i++) {
	if ((array[i] = (l_int32 *)CALLOC(sx, sizeof(l_int32))) == NULL)
	    return (l_int32 **)ERROR_PTR("array not made", procName, NULL);
    }

    return array;
}



/*------------------------------------------------------------------------*
 *                           Extension of sela                            *
 *------------------------------------------------------------------------*/
/*!
 *  selaAddSel()
 *
 *      Input:  sela
 *              sel to be added
 *              selname (ignored if already defined in sel;
 *                       req'd in sel when added to a sela)
 *              copyflag (for sel: 0 inserts, 1 copies)
 *      Return: 0 if OK; 1 on error
 *
 *  Action:  Adds sel to arrays, making a copy if flagged.
 *           Copies the name to the sel if necessary.
 *           Increments the sel count.
 */
l_int32
selaAddSel(SELA        *sela,
	   SEL         *sel,
	   const char  *selname,
	   l_int32      copyflag)
{
l_int32  n;
SEL     *csel;

    PROCNAME("selaAddSel");

    if (!sela)
	return ERROR_INT("sela not defined", procName, 1);
    if (!sel)
	return ERROR_INT("sel not defined", procName, 1);
    if (!sel->name && !selname)
	return ERROR_INT("added sel must have name", procName, 1);

    if (copyflag == TRUE) {
	if ((csel = selCopy(sel)) == NULL)
	    return ERROR_INT("csel not made", procName, 1);
    }
    else   /* copyflag is false; insert directly */
	csel = sel;

    if (csel->name == NULL)
	csel->name = stringNew(selname);

    n = selaGetCount(sela);
    if (n >= sela->nalloc)
	selaExtendArray(sela);
    sela->sel[n] = csel;
    sela->n++;

    return 0;
}
    

/*!
 *  selaExtendArray()
 *
 *      Input:  sela
 *      Return: 0 if OK; 1 on error
 *
 *  Action: doubles the ptr array; copies the old ptr addresses
 *          into the new array; updates ptr array size
 */
l_int32
selaExtendArray(SELA  *sela)
{
    PROCNAME("selaExtendArray");

    if (!sela)
        return ERROR_INT("sela not defined", procName, 1);
    
    if ((sela->sel = (SEL **)reallocNew((void **)&sela->sel,
                              sizeof(l_intptr_t) * sela->nalloc,
                              2 * sizeof(l_intptr_t) * sela->nalloc)) == NULL)
	    return ERROR_INT("new ptr array not returned", procName, 1);

    sela->nalloc = 2 * sela->nalloc;
    return 0;
}



/*----------------------------------------------------------------------*
 *                               Accessors                              *
 *----------------------------------------------------------------------*/
/*!
 *  selaGetCount()
 *
 *      Input:  sela
 *      Return: count, or 0 on error
 */
l_int32
selaGetCount(SELA  *sela)
{
    PROCNAME("selaGetCount");

    if (!sela)
	return ERROR_INT("sela not defined", procName, 0);

    return sela->n;
}


/*!
 *  selaGetSel()
 *
 *      Input:  sela
 *              index of sel to be retrieved (not copied)
 *      Return: sel, or null on error
 */
SEL *
selaGetSel(SELA    *sela,
	   l_int32  i)
{
    PROCNAME("selaGetSel");

    if (!sela)
	return (SEL *)ERROR_PTR("sela not defined", procName, NULL);

    if (i < 0 || i >= sela->n)
	return (SEL *)ERROR_PTR("invalid index", procName, NULL);

    return sela->sel[i];
}


/*!
 *  selGetName()
 *
 *      Input:  sel
 *      Return: sel name (not copied), or null if no name or on error
 */
char *
selGetName(SEL  *sel)
{
    PROCNAME("selGetName");

    if (!sel)
	return (char *)ERROR_PTR("sel not defined", procName, NULL);

    return sel->name;
}


/*!
 *  selaFindSelByName()
 *
 *      Input:  sela
 *              sel name
 *              &index (<optional, return>)
 *              &sel  (<optional, return> sel (not a copy))
 *      Return: 0 if OK; 1 on error
 */
l_int32
selaFindSelByName(SELA        *sela, 
	          const char  *name,
		  l_int32     *pindex,
		  SEL        **psel)
{
l_int32  i, n;
char    *sname;
SEL     *sel;

    PROCNAME("selaFindSelByName");

    if (pindex) *pindex = -1;
    if (psel) *psel = NULL;

    if (!sela)
	return ERROR_INT("sela not defined", procName, 1);

    n = selaGetCount(sela);
    for (i = 0; i < n; i++)
    {
	if ((sel = selaGetSel(sela, i)) == NULL) {
	    L_WARNING("missing sel", procName);
	    continue;
	}
	    
	sname = selGetName(sel);
	if (sname && (!strcmp(name, sname))) {
            if (pindex)
                *pindex = i;
	    if (psel)
                *psel = sel;
	    return 0;
	}
    }
    
    return 1;
}


/*!
 *  selGetElement()
 *
 *      Input:  sel
 *              row
 *              col
 *              &type  (<return> SEL_HIT, SEL_MISS, SEL_DONT_CARE)
 *      Return: 0 if OK; 1 on error
 */
l_int32
selGetElement(SEL      *sel,
              l_int32   row,
	      l_int32   col,
	      l_int32  *ptype)
{
    PROCNAME("selGetElement");

    if (!sel)
	return ERROR_INT("sel not defined", procName, 1);
    if (row < 0 || row >= sel->sy)
	return ERROR_INT("sel row out of bounds", procName, 1);
    if (col < 0 || col >= sel->sx)
	return ERROR_INT("sel col out of bounds", procName, 1);

    *ptype = sel->data[row][col];
    return 0;
}


/*!
 *  selSetElement()
 *
 *      Input:  sel
 *              row
 *              col
 *              type  (SEL_HIT, SEL_MISS, SEL_DONT_CARE)
 *      Return: 0 if OK; 1 on error
 */
l_int32
selSetElement(SEL     *sel,
              l_int32  row,
	      l_int32  col,
	      l_int32  type)
{
    PROCNAME("selSetElement");

    if (!sel)
	return ERROR_INT("sel not defined", procName, 1);
    if (type != SEL_HIT && type != SEL_MISS && type != SEL_DONT_CARE)
	return ERROR_INT("invalid sel element type", procName, 1);
    if (row < 0 || row >= sel->sy)
	return ERROR_INT("sel row out of bounds", procName, 1);
    if (col < 0 || col >= sel->sx)
	return ERROR_INT("sel col out of bounds", procName, 1);

    sel->data[row][col] = type;
    return 0;
}


/*----------------------------------------------------------------------*
 *                Max translations for erosion and hmt                  *
 *----------------------------------------------------------------------*/
/*!
 *  selFindMaxTranslations()
 *
 *      Input:  sel
 *              &xp, &yp, &xn, &yn  (<return> max shifts)
 *      Return: 0 if OK; 1 on error
 *
 *  Note: these are the maximum shifts for the erosion operation.
 *        For example, when j < cx, the shift of the image
 *        is +x to the cx.  This is a positive xp shift.
 */
l_int32
selFindMaxTranslations(SEL      *sel,
                       l_int32  *pxp,
                       l_int32  *pyp,
                       l_int32  *pxn,
                       l_int32  *pyn)
{
l_int32  i, j, maxxp, maxyp, maxxn, maxyn;

    PROCNAME("selaFindMaxTranslations");

    *pxp = *pyp = *pxn = *pyn = 0;

    if (!sel)
	return ERROR_INT("sel not defined", procName, 1);

    maxxp = maxyp = maxxn = maxyn = 0;
    for (i = 0; i < sel->sy; i++) {
	for (j = 0; j < sel->sx; j++) {
	    if (sel->data[i][j] == 1) {
		maxxp = L_MAX(maxxp, sel->cx - j);
		maxyp = L_MAX(maxyp, sel->cy - i);
		maxxn = L_MAX(maxxn, j - sel->cx);
		maxyn = L_MAX(maxyn, i - sel->cy);
	    }
	}
    }

    *pxp = maxxp;
    *pyp = maxyp;
    *pxn = maxxn;
    *pyn = maxyn;

    return 0;
}



/*----------------------------------------------------------------------*
 *                    I/O and visualizing Sela & Sel                    *
 *----------------------------------------------------------------------*/
/*!
 *  selaWrite()
 *
 *      Input:  filename
 *              sela
 *      Return: 0 if OK, 1 on error
 */
l_int32
selaWrite(const char  *fname,
	  SELA        *sela)
{
FILE  *fp;

    PROCNAME("selaWrite");

    if (!fname)
	return ERROR_INT("fname not defined", procName, 1);
    if (!sela)
	return ERROR_INT("sela not defined", procName, 1);

    if ((fp = fopen(fname, "wb")) == NULL)
	return ERROR_INT("stream not opened", procName, 1);
    selaWriteStream(fp, sela);
    fclose(fp);

    return 0;
}


/*!
 *  selaWriteStream()
 *
 *      Input:  stream
 *              sela
 *      Return: 0 if OK, 1 on error
 */
l_int32
selaWriteStream(FILE  *fp,
	        SELA  *sela)
{
l_int32  i, n;
SEL     *sel;

    PROCNAME("selaWriteStream");

    if (!fp)
	return ERROR_INT("stream not defined", procName, 1);
    if (!sela)
	return ERROR_INT("sela not defined", procName, 1);

    n = selaGetCount(sela);
    fprintf(fp, "sel array: number of sels = %d\n\n", n);
    for (i = 0; i < n; i++)
    {
	if ((sel = selaGetSel(sela, i)) == NULL)
	    continue;
	selWriteStream(fp, sel);
    }
    return 0;
}


/*!
 *  selWriteStream()
 *
 *      Input:  stream
 *              sel
 *      Return: 0 if OK, 1 on error
 */
l_int32
selWriteStream(FILE  *fp,
	       SEL   *sel)
{
l_int32  i, j;

    PROCNAME("selWriteStream");

    if (!fp)
	return ERROR_INT("stream not defined", procName, 1);
    if (!sel)
	return ERROR_INT("sel not defined", procName, 1);

    fprintf(fp, "  ------  %s  ------\n", selGetName(sel));
    fprintf(fp, "  sy = %d, sx = %d, cy = %d, cx = %d\n",
	    sel->sy, sel->sx, sel->cy, sel->cx);
    for (i = 0; i < sel->sy; i++) {
	fprintf(fp, "    ");
	for (j = 0; j < sel->sx; j++)
	    fprintf(fp, "%d", sel->data[i][j]);
	fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    return 0;
}


/*!
 *  selaRead()
 *
 *      Input:  filename
 *      Return: sela, or null on error
 */
SELA  *
selaRead(const char  *fname)
{
FILE  *fp;
SELA  *sela;

    PROCNAME("selaRead");

    if (!fname)
	return (SELA *)ERROR_PTR("fname not defined", procName, NULL);

    if ((fp = fopen(fname, "rb")) == NULL)
	return (SELA *)ERROR_PTR("stream not opened", procName, NULL);
    if ((sela = selaReadStream(fp)) == NULL)
	return (SELA *)ERROR_PTR("sela not returned", procName, NULL);
    fclose(fp);

    return sela;
}


/*!
 *  selaReadStream()
 *
 *      Input:  stream
 *      Return: sela, or null on error
 */
SELA  *
selaReadStream(FILE  *fp)
{
l_int32  i, n;
SEL     *sel;
SELA    *sela;

    PROCNAME("selaReadStream");

    if (!fp)
	return (SELA *)ERROR_PTR("stream not defined", procName, NULL);

    if (fscanf(fp, "sel array: number of sels = %d\n\n", &n) != 1)
	return (SELA *)ERROR_PTR("not a sela", procName, NULL);

    if ((sela = selaCreate(n)) == NULL)
	return (SELA *)ERROR_PTR("sela not made", procName, NULL);
    sela->nalloc = n;

    for (i = 0; i < n; i++)
    {
	if ((sel = selReadStream(fp)) == NULL)
	    return (SELA *)ERROR_PTR("sel not made", procName, NULL);
	selaAddSel(sela, sel, NULL, 0);
    }

    return sela;
}


/*!
 *  selReadStream()
 *
 *      Input:  stream
 *      Return: sel, or null on error
 */
SEL  *
selReadStream(FILE  *fp)
{
char    *selname;
char     linebuf[BUFSIZE];
l_int32  sy, sx, cy, cx, i, j;
SEL     *sel;

    PROCNAME("selReadStream");

    if (!fp)
	return (SEL *)ERROR_PTR("stream not defined", procName, NULL);

    fgets(linebuf, BUFSIZE, fp);
    selname = stringNew(linebuf);
    sscanf(linebuf, "  ------  %s  ------", selname);

    if (fscanf(fp, "  sy = %d, sx = %d, cy = %d, cx = %d\n",
	    &sy, &sx, &cy, &cx) != 4)
	return (SEL *)ERROR_PTR("dimensions not read", procName, NULL);

    if ((sel = selCreate(sy, sx, selname)) == NULL)
	return (SEL *)ERROR_PTR("sel not made", procName, NULL);
    sel->cy = cy;
    sel->cx = cx;

    for (i = 0; i < sy; i++) {
	fscanf(fp, "    ");
	for (j = 0; j < sx; j++)
	    fscanf(fp, "%1d", &sel->data[i][j]);
	fscanf(fp, "\n");
    }
    fscanf(fp, "\n");

    FREE((void *)selname);
    return sel;
}

